/** 
 @name RepositionAnchorPoint
 @version 2.5 (05 June 2009)
 @fileoverview
 <h4>Description</h4>  
 <p>This script allows you to reposition the anchor point of the selected layers around the layer edges while keeping the layers at the same position in the comp window.</p> 
 <h4>Usage</h4>
 <ol>
    <li> In After Effects CS3 or later, run the script
    <li> Select at least one layer  
    <li> Specify what should be considered as layer edges
    <li> Choose the anchor point location using one of the radio buttons
    <li> Click on "Execute" to reposition the anchor point of all selected layers
 </ol>
 @author Charles Bordenave
 @location http://www.nabscripts.com/downloads/scripts/RepositionAnchorPoint.zip
*/


/**
 This class represents the main class of the script;
 it is used to create the user interface which allows to reposition the anchor point of the selected layers around the layer edges
 @class Main class of the script
*/
function RepositionAnchorPoint()
{
    // Variable used to keep track of 'this' reference
    var repositionAnchorPoint = this;
    
    // Create an instance of the utils class to use its functions
    var utils = new RepositionAnchorPointUtils();

    // Script infos
    this.scriptMinSupportVersion = "8.0";
    this.scriptName = "RepositionAnchorPoint.jsx";    
    this.scriptVersion = "2.5";
    this.scriptTitle = "Reposition Anchor Point";
    this.scriptCopyright = "Copyright (c) 2009 Charles Bordenave";
    this.scriptHomepage = "http://www.nabscripts.com";
    this.scriptDescription = {en: "This script allows you to reposition the anchor point of the selected layers around the layer edges while keeping the layers at the same position in the comp window.", fr:"Ce script permet de repositionner le point d\\'ancrage des calques sélectionnés sur le contour du calque tout en maintenant les calques à leur place dans la fenêtre de composition."};
    this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), fr:this.scriptName + ", v" + this.scriptVersion + "\\r\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};        

    // Errors
    this.requirementErr = {en:"This script requires After Effects CS3 or later.", fr:"Ce script nécessite After Effects CS3 ou supérieur."};    
    this.noCompErr = {en:"A comp must be active.", fr:"Une composition doit être active."};
    this.noLayersErr = {en:"Select at least one layer.", fr:"Sélectionnez au moins un calque."};
    this.processErr = {en:"An error occurred while manipulating layers.", fr:"Une erreur s'est produite pendant la manipulation des calques."};

    // UI strings 
    this.aboutBtnName = "?";
    this.edgesStName = {en:"Edges:", fr:"Bords:"};    
    this.edgesLstChoices = {en:"['Layer Edges','Mask Bounding Box']", fr:"['Bords du calque','Cadre de contour du Masque']"};
    this.anchorStName = {en:"Anchor:", fr:"Ancrage:"};
    this.runBtnName = {en:"Execute", fr:"Éxécuter"};
    
    /**
     Creates and displays the script interface
     @param {Object} thisObj A Panel object if the script is launched from the Window menu, null otherwise    
    */
    this.buildUI = function (thisObj)
    {
        // dockable panel or palette
        var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});

        // resource specifications
        var res =
        "group { orientation:'column', alignment:['left','top'], alignChildren:['right','top'], \
            gr1: Group { \
                aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20] } \
            }, \
            gr2: Group { orientation:'row', alignment:['fill','fill'], \
                gr21: Group { orientation:'column', alignment:['fill','fill'], alignChildren:['right','fill'], margins:[0,5,0,0], spacing:10, \
                    edgesSt: StaticText { text:'" + utils.loc(this.edgesStName) + "' }, \
                    anchorSt: StaticText { text:'" + utils.loc(this.anchorStName) + "' } \
                }, \
                gr22: Group { orientation:'column', alignment:['fill','fill'], alignChildren:['left','center'], \
                    gr221: Group { \
                        edgesLst: DropDownList { properties:{items:" + utils.loc(this.edgesLstChoices) + "} } \
                    }, \
                    gr222: Group { orientation:'column', alignment:['fill','fill'], alignChildren:['left','top'], \
                        gr2221: Group { orientation:'row', \
                            aCb: RadioButton { }, \
                            bCb: RadioButton { }, \
                            cCb: RadioButton { } \
                        }, \
                        gr2222: Group { orientation:'row', \
                            dCb: RadioButton { }, \
                            eCb: RadioButton { value:true }, \
                            fCb: RadioButton { } \
                        }, \
                        gr2223: Group { orientation:'row', \
                            gCb: RadioButton { }, \
                            hCb: RadioButton { }, \
                            iCb: RadioButton { } \
                        } \
                    } \
                } \
            }, \
            gr3: Panel { alignment:['fill','center'] }, \
            gr4: Group { orientation:'row', alignment:['fill','top'], \
                runBtn: Button { text:'" + utils.loc(this.runBtnName) + "', alignment:['right','center'] } \
            } \
        }"; 
        pal.gr = pal.add(res);
        
        pal.gr.gr2.gr22.gr221.edgesLst.graphics.foregroundColor = pal.graphics.newPen(pal.graphics.BrushType.SOLID_COLOR, [0,0,0], 1);
        
        pal.gr.gr2.gr22.gr221.edgesLst.selection = 0;
        
        // event callbacks
        pal.gr.gr1.aboutBtn.onClick = function () 
        { 
            utils.createAboutDlg(repositionAnchorPoint.scriptAbout); 
        };
        
        pal.gr.gr2.gr22.gr222.gr2221.aCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 0);
        };
        
        pal.gr.gr2.gr22.gr222.gr2221.bCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 1);
        };
        
        pal.gr.gr2.gr22.gr222.gr2221.cCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 2);
        }; 
        
        pal.gr.gr2.gr22.gr222.gr2222.dCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 3);
        }; 
        
        pal.gr.gr2.gr22.gr222.gr2222.eCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 4);
        }; 
        
        pal.gr.gr2.gr22.gr222.gr2222.fCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 5);
        }; 
        
        pal.gr.gr2.gr22.gr222.gr2223.gCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 6);
        }; 
        
        pal.gr.gr2.gr22.gr222.gr2223.hCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 7);
        }; 
        
        pal.gr.gr2.gr22.gr222.gr2223.iCb.onClick = function ()
        {
            repositionAnchorPoint.uncheckedOthers(this.parent.parent, 8);
        };
                
        pal.gr.gr4.runBtn.onClick = function () 
        { 
            repositionAnchorPoint.reposition(pal); 
        };
        
        // show user interface
        if (pal instanceof Window)
        {
            pal.center();
            pal.show();
        }
        else
        {
            pal.layout.layout(true);
        }       
    };

    /**
     Determines whether the active item is a composition  
     @return True if the active item is not a composition, False otherwise
    */    
    this.checkActiveItem = function () 
    {
        var err = false;
        var comp = app.project.activeItem;
        if (!comp || !(comp instanceof CompItem))
        {
            err = true;
        }
        return err;
    };   

    /**
     Unchecks radio buttons except the one passed as argument
     @param {Object} Parent group of subgroups containing radio buttons 
     @param {Number} The radio button index to keep checked
    */
    this.uncheckedOthers = function (parentGroup, checkedId)
    {
        this.value = true;
        for (var i = 0; i < parentGroup.children.length; i++)
        for (var j = 0; j < parentGroup.children[i].children.length; j++)
        if (i * parentGroup.children.length + j != checkedId) parentGroup.children[i].children[j].value = false;
    };
    
    /**
     Repositions the anchor point of a set of layers, edges are represented by layer's edges
     @param {Object} pal A palette or a dockable panel containing all user parameters
     @param {Object} layers An array of layer objects          
    */    
    this.repositionOnLayerEdges = function (pal, layers)
    {
        var curTime = layers[0].containingComp.time;
        var compPar = layers[0].containingComp.pixelAspect;
        var anchorId = -1;
        for (var i = 0; anchorId == -1 && i < pal.gr.gr2.gr22.gr222.children.length; i++)
        for (var j = 0; anchorId == -1 && j < pal.gr.gr2.gr22.gr222.children[i].children.length; j++)
        if (pal.gr.gr2.gr22.gr222.children[i].children[j].value)
        {
            anchorId = i * pal.gr.gr2.gr22.gr222.children.length + j;
        }
        
        var err = this.processErr;
        try
        {            
            for (var i = 0; i < layers.length; i++)
            {
                var layer = layers[i];
                var textAdjust, xLayerCenter, yLayerCenter;        
                if (layer instanceof AVLayer || layer instanceof TextLayer) // solid/comp/footage/text layer
                {
                    var layerSize = utils.getLayerSize(layer, curTime); 
                    xLayerCenter = layerSize[0] / 2;
                    yLayerCenter = layerSize[1] / 2;
                    if (layer instanceof TextLayer) textAdjust = [0,2 * yLayerCenter,0]; // assume left align
                }            
                else continue; // skip light/camera            
                
                var anchPt = layer.anchorPoint;
                var pos = layer.position;
                
                var x = (anchorId % 3) * xLayerCenter;
                var y = Math.floor(anchorId / 3) * yLayerCenter;
                var z = 0;            
    
                var delta = [x,y,z] - anchPt.valueAtTime(curTime, false) - (layer instanceof TextLayer ? textAdjust : [0,0,0]);
                var s = layer.scale.valueAtTime(curTime, false) / 100;
                var layerPar = (layer.source) ? layer.source.pixelAspect : 1.0; 
                
                var newAnchPt = anchPt.valueAtTime(curTime, false) + delta;
                var newPos = pos.valueAtTime(curTime, false) + [(s[0]*delta[0]) * (layerPar/compPar), s[1]*delta[1], s[2]*delta[2]];
    
                anchPt.numKeys ? anchPt.setValueAtTime(curTime, newAnchPt) : anchPt.setValue(newAnchPt);            
                pos.numKeys ? pos.setValueAtTime(curTime, newPos) : pos.setValue(newPos);
            }
        }
        catch(e)
        {
            utils.throwErr(err);
        }
    };

    /**
     Repositions the anchor point of a set of layers, edges are represented by the first mask's bounding box (if any)
     @param {Object} pal A palette or a dockable panel containing all user parameters
     @param {Object} layers An array of layer objects          
    */    
    this.repositionOnMaskBoundingBox = function (pal, layers)
    {
        var curTime = layers[0].containingComp.time;
        var compPar = layers[0].containingComp.pixelAspect;
        var anchorId = -1;
        for (var i = 0; anchorId == -1 && i < pal.gr.gr2.gr22.gr222.children.length; i++)
        for (var j = 0; anchorId == -1 && j < pal.gr.gr2.gr22.gr222.children[i].children.length; j++)
        if (pal.gr.gr2.gr22.gr222.children[i].children[j].value)
        {
            anchorId = i * pal.gr.gr2.gr22.gr222.children.length + j;
        }        
        for (var k = 0; k < layers.length; k++)
        {
            var layer = layers[k];
            
            if (layer instanceof AVLayer || layer instanceof TextLayer) // solid/comp/footage/text
            {
                var maskGrp = layer.Masks;
                
                if (maskGrp.numProperties)
                {
                    // Get the first mask
                    var mask = maskGrp.property(1);
                    var maskShape = mask.maskShape;
                    
                    // Retrieve vertices and tangents
                    var shape = maskShape.valueAtTime(curTime, false);
                    var verts = shape.vertices;
                    var intan = shape.inTangents;
                    var outtan = shape.outTangents;

                    // Compute mask bounding box (tangents included)
                    var T = Infinity;
                    var B = -Infinity;
                    var L = Infinity;
                    var R = -Infinity;
                    for (var i = 0; i < verts.length; i++) 
                    {
                        T = utils.min4(T, verts[i][1], verts[i][1]+intan[i][1], verts[i][1]+outtan[i][1]);
                        B = utils.max4(B, verts[i][1], verts[i][1]+intan[i][1], verts[i][1]+outtan[i][1]);
                        L = utils.min4(L, verts[i][0], verts[i][0]+intan[i][0], verts[i][0]+outtan[i][0]);
                        R = utils.max4(R, verts[i][0], verts[i][0]+intan[i][0], verts[i][0]+outtan[i][0]);
                    } 
                        
                    // Reposition anchor point
                    var xBbCenter = (R - L) / 2;
                    var yBbCenter = (B - T) / 2;
                    
                    var anchPt = layer.anchorPoint;
                    var pos = layer.position;                
                    
                    var x = L + (anchorId % 3) * xBbCenter;
                    var y = T + Math.floor(anchorId / 3) * yBbCenter;
                    var z = 0;                            
                    
                    var delta = [x,y,z] - anchPt.valueAtTime(curTime, false);
                    var s = layer.scale.valueAtTime(curTime, false) / 100;
                    var layerPar = layer.source.pixelAspect;                
                    
                    var newAnchPt = anchPt.valueAtTime(curTime, false) + delta;
                    var newPos = pos.valueAtTime(curTime, false) + [(s[0]*delta[0]) * (layerPar/compPar), s[1]*delta[1], s[2]*delta[2]];                
                    
                    anchPt.numKeys ? anchPt.setValueAtTime(curTime, newAnchPt) : anchPt.setValue(newAnchPt);
                    pos.numKeys ? pos.setValueAtTime(curTime, newPos) : pos.setValue(newPos);                
                }
                else // assume a mistake
                {
                    this.repositionOnLayerEdges(pal, layers);    
                }
            }
        }
    };
    
    /**
     Functional part of the script: repositions the anchor point of the selected layers
     @param {Object} pal A palette or a dockable panel containing all user parameters          
    */    
    this.reposition = function (pal)
    {
        try
        {
            var comp = app.project.activeItem;
            var err = this.noCompErr;
            if (this.checkActiveItem(comp)) throw(err);
                    
            var selLayers = comp.selectedLayers;
            var err = this.noLayersErr;
            if (selLayers.length < 1) throw(err);
            
            app.beginUndoGroup(this.scriptTitle);
            
            if (pal.gr.gr2.gr22.gr221.edgesLst.selection.index == 0)
            {
                this.repositionOnLayerEdges(pal, selLayers);
            }
            else 
            {
                this.repositionOnMaskBoundingBox(pal, selLayers);
            }                  
                  
            app.endUndoGroup();
        }
        catch(err)
        {
            utils.throwErr(err);
        }                
    };
    
    /**
     Runs the script  
     @param {Object} thisObj A Panel object if the script is launched from the Window menu, null otherwise
    */
    this.run = function (thisObj) 
    {
        if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
        {
            this.throwErr(this.requirementErr);
        }
        else
        {
            this.buildUI(thisObj);
        }    
    };
}


/**
 This class provides some utility functions used by RepositionAnchorPoint
 @class Some utility functions grouped in a class
*/
function RepositionAnchorPointUtils()
{
    /**
     String localization function: english and french languages are supported
     @param {Object} str A localization object containing the localized versions of a string    
     @return Appropriate localized version of str
    */    
    this.loc = function (str)
    {
        return app.language == Language.FRENCH ? str.fr : str.en;
    };

    /**
     Displays a window containg a localized error message
     @param {Object} err A localization object containing the localized versions of an error message
    */    
    this.throwErr = function (err)
    {
        var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
        Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
    };            

    /**
     Get the size in pixels of an AV layer (comp layer, footage layer, solid layer, text layer) at specific time
     @param {Object} avLayer An AV layer object    
     @param {Number} time A floating-point value representing the time in seconds at which the layer size must be evaluated    
     @return A two-dimensional array containing the width and height of the layer
    */
    this.getLayerSize = function (avLayer, time)
    {
        var w, h;
        if (!(avLayer instanceof TextLayer))
        {
            w = avLayer.width;
            h = avLayer.height;
        }
        else
        {
            var bb = avLayer.sourceRectAtTime(time, true);
            w = bb.width;
            h = bb.height;
        }
        return [w,h];
    };
    
    /**
     Computes the minimum value between four numbers
     @param {Number} n1 A number
     @param {Number} n2 A number
     @param {Number} n3 A number
     @param {Number} n4 A number
     @return {Number} The minimum number between n1, ..., n4 
    */    
    this.min4 = function (n1, n2, n3, n4)
    {
        return (n1 < n2 && n1 < n3 && n1 < n4) ? n1 : 
              ((n2 < n1 && n2 < n3 && n2 < n4) ? n2 :
              ((n3 < n1 && n3 < n2 && n3 < n4) ? n3 : n4));                      
    }; 

    /**
     Computes the maximum value between four numbers
     @param {Number} n1 A number
     @param {Number} n2 A number
     @param {Number} n3 A number
     @param {Number} n4 A number
     @return {Number} The maximum number between n1, ..., n4 
    */    
    this.max4 = function (n1, n2, n3, n4)
    {
        return (n1 > n2 && n1 > n3 && n1 > n4) ? n1 : 
              ((n2 > n1 && n2 > n3 && n2 > n4) ? n2 :
              ((n3 > n1 && n3 > n2 && n3 > n4) ? n3 : n4));                      
    };
        
    /**
     Displays a customized window containg the About text
     @param {String} aboutStr The text to display
    */
    this.createAboutDlg = function (aboutStr)
    {        
        eval(unescape('%20%20%20%20%20%20%20%20%2F%2A%2A%20%0A%20%20%20%20%20%20%20%20%20%44%72%61%77%20%73%6F%6D%65%20%72%61%6E%64%6F%6D%20%72%65%63%74%61%6E%67%6C%65%73%20%28%70%6F%73%69%74%69%6F%6E%2C%20%73%69%7A%65%2C%20%63%6F%6C%6F%72%2C%20%61%6C%70%68%61%29%20%6F%6E%20%74%68%65%20%77%69%6E%64%6F%77%20%62%61%63%6B%67%72%6F%75%6E%64%0A%20%20%20%20%20%20%20%20%20%40%69%67%6E%6F%72%65%20%0A%20%20%20%20%20%20%20%20%2A%2F%0A%20%20%20%20%20%20%20%20%66%75%6E%63%74%69%6F%6E%20%61%64%64%4E%61%62%73%63%72%69%70%74%73%42%61%63%6B%67%72%6F%75%6E%64%53%69%67%6E%61%74%75%72%65%28%77%6E%64%29%0A%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%6E%75%6D%52%65%63%74%20%3D%20%32%34%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%6D%69%6E%4F%70%61%63%69%74%79%20%3D%20%30%2E%30%35%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%6D%61%78%4F%70%61%63%69%74%79%20%3D%20%30%2E%31%35%3B%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%6C%65%66%74%45%64%67%65%20%3D%20%30%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%74%6F%70%45%64%67%65%20%3D%20%30%3B%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%72%69%67%68%74%45%64%67%65%20%3D%20%77%6E%64%2E%77%69%6E%64%6F%77%42%6F%75%6E%64%73%2E%77%69%64%74%68%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%62%6F%74%74%6F%6D%45%64%67%65%20%3D%20%77%6E%64%2E%77%69%6E%64%6F%77%42%6F%75%6E%64%73%2E%68%65%69%67%68%74%3B%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%66%6F%72%20%28%76%61%72%20%69%20%3D%20%30%20%3B%20%69%20%3C%20%6E%75%6D%52%65%63%74%3B%20%69%2B%2B%29%0A%20%20%20%20%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%78%4C%6F%63%20%3D%20%31%30%20%2B%20%28%72%69%67%68%74%45%64%67%65%20%2D%20%32%30%29%20%2A%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%79%4C%6F%63%20%3D%20%31%30%20%2B%20%28%62%6F%74%74%6F%6D%45%64%67%65%20%2D%20%32%30%29%20%2A%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%77%69%64%74%68%20%3D%20%35%20%2B%20%31%35%20%2A%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%68%65%69%67%68%74%20%3D%20%35%20%2B%20%31%35%20%2A%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%3B%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%62%6F%72%64%65%72%57%69%64%74%68%20%3D%20%31%20%2B%20%34%20%2A%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%62%6F%72%64%65%72%43%6F%6C%6F%72%20%3D%20%5B%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%2C%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%2C%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%2C%20%6D%69%6E%4F%70%61%63%69%74%79%20%2B%20%28%6D%61%78%4F%70%61%63%69%74%79%20%2D%20%6D%69%6E%4F%70%61%63%69%74%79%29%20%2A%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%5D%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%63%6F%6C%6F%72%42%72%75%73%68%20%3D%20%77%6E%64%2E%67%72%61%70%68%69%63%73%2E%6E%65%77%42%72%75%73%68%28%77%6E%64%2E%67%72%61%70%68%69%63%73%2E%42%72%75%73%68%54%79%70%65%2E%53%4F%4C%49%44%5F%43%4F%4C%4F%52%2C%20%62%6F%72%64%65%72%43%6F%6C%6F%72%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%31%20%3D%20%77%6E%64%2E%61%64%64%28%22%67%72%6F%75%70%22%2C%20%5B%78%4C%6F%63%2C%20%79%4C%6F%63%2C%20%78%4C%6F%63%20%2B%20%77%69%64%74%68%2C%20%79%4C%6F%63%20%2B%20%62%6F%72%64%65%72%57%69%64%74%68%5D%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%32%20%3D%20%77%6E%64%2E%61%64%64%28%22%67%72%6F%75%70%22%2C%20%5B%78%4C%6F%63%2C%20%79%4C%6F%63%20%2B%20%68%65%69%67%68%74%20%2D%20%62%6F%72%64%65%72%57%69%64%74%68%2C%20%78%4C%6F%63%20%2B%20%77%69%64%74%68%2C%20%79%4C%6F%63%20%2B%20%68%65%69%67%68%74%5D%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%33%20%3D%20%77%6E%64%2E%61%64%64%28%22%67%72%6F%75%70%22%2C%20%5B%78%4C%6F%63%2C%20%79%4C%6F%63%20%2B%20%62%6F%72%64%65%72%57%69%64%74%68%2C%20%78%4C%6F%63%20%2B%20%62%6F%72%64%65%72%57%69%64%74%68%2C%20%79%4C%6F%63%20%2B%20%68%65%69%67%68%74%20%2D%20%62%6F%72%64%65%72%57%69%64%74%68%5D%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%34%20%3D%20%77%6E%64%2E%61%64%64%28%22%67%72%6F%75%70%22%2C%20%5B%78%4C%6F%63%20%2B%20%77%69%64%74%68%20%2D%20%62%6F%72%64%65%72%57%69%64%74%68%2C%20%79%4C%6F%63%20%2B%20%62%6F%72%64%65%72%57%69%64%74%68%2C%20%78%4C%6F%63%20%2B%20%77%69%64%74%68%2C%20%79%4C%6F%63%20%2B%20%68%65%69%67%68%74%20%2D%20%62%6F%72%64%65%72%57%69%64%74%68%5D%29%3B%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%31%2E%67%72%61%70%68%69%63%73%2E%62%61%63%6B%67%72%6F%75%6E%64%43%6F%6C%6F%72%20%3D%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%32%2E%67%72%61%70%68%69%63%73%2E%62%61%63%6B%67%72%6F%75%6E%64%43%6F%6C%6F%72%20%3D%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%33%2E%67%72%61%70%68%69%63%73%2E%62%61%63%6B%67%72%6F%75%6E%64%43%6F%6C%6F%72%20%3D%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%77%6E%64%2E%67%34%2E%67%72%61%70%68%69%63%73%2E%62%61%63%6B%67%72%6F%75%6E%64%43%6F%6C%6F%72%20%3D%20%63%6F%6C%6F%72%42%72%75%73%68%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%7D%20%20%20%20%0A%20%20%20%20%20%20%20%20%7D%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%2F%2F%20%70%61%6E%65%6C%20%62%6F%72%64%65%72%53%74%79%6C%65%3A%20%6F%6E%65%20%6F%66%20%62%6C%61%63%6B%2C%20%65%74%63%68%65%64%2C%20%67%72%61%79%2C%20%72%61%69%73%65%64%2C%20%73%75%6E%6B%65%6E%2E%20%44%65%66%61%75%6C%74%20%69%73%20%65%74%63%68%65%64%2E%0A%20%20%20%20%20%20%20%20%2F%2F%20%72%65%73%6F%75%72%63%65%20%73%70%65%63%69%66%69%63%61%74%69%6F%6E%73%0A%20%20%20%20%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%20%20%20%20%20%20%20%20%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%61%62%6F%75%74%50%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%20%62%6F%72%64%65%72%53%74%79%6C%65%3A%27%73%75%6E%6B%65%6E%27%20%7D%2C%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%32%38%30%2C%31%35%30%5D%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%7D%2C%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%62%74%6E%73%47%72%3A%20%47%72%6F%75%70%20%7B%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%76%69%73%69%74%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%56%69%73%69%74%20%48%6F%6D%65%70%61%67%65%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%20%20%20%20%20%20%20%20%20%20%20%20%7D%20%5C%0A%20%20%20%20%20%20%20%20%7D%22%3B%20%0A%20%20%20%20%20%20%20%20%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%2F%2F%20%6F%6E%20%4D%61%63%20%77%65%20%63%61%6E%20%64%69%73%61%62%6C%65%20%65%64%69%74%20%74%65%78%74%20%77%68%69%6C%65%20%61%6C%6C%6F%77%69%6E%67%20%73%63%72%6F%6C%6C%69%6E%67%2C%20%6F%6E%20%57%69%6E%64%6F%77%73%20%77%65%20%63%61%6E%27%74%0A%20%20%20%20%20%20%20%20%64%6C%67%2E%67%72%2E%61%62%6F%75%74%50%6E%6C%2E%61%62%6F%75%74%45%74%2E%65%6E%61%62%6C%65%64%20%3D%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%2F%2F%20%64%72%61%77%20%72%61%6E%64%6F%6D%20%62%61%63%6B%67%72%6F%75%6E%64%20%63%6F%6C%6F%72%20%28%67%72%61%79%73%63%61%6C%65%29%0A%20%20%20%20%20%20%20%20%69%66%20%28%70%61%72%73%65%46%6C%6F%61%74%28%61%70%70%2E%76%65%72%73%69%6F%6E%29%20%3E%3D%20%39%2E%30%29%20%2F%2F%20%43%53%34%20%6F%72%20%6C%61%74%65%72%0A%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%77%68%69%74%65%42%72%75%73%68%20%3D%20%64%6C%67%2E%67%72%61%70%68%69%63%73%2E%6E%65%77%42%72%75%73%68%28%64%6C%67%2E%67%72%61%70%68%69%63%73%2E%42%72%75%73%68%54%79%70%65%2E%53%4F%4C%49%44%5F%43%4F%4C%4F%52%2C%20%5B%31%2C%20%31%2C%20%31%2C%20%31%5D%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%72%61%6E%64%20%3D%20%4D%61%74%68%2E%72%61%6E%64%6F%6D%28%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%62%67%42%72%75%73%68%20%3D%20%64%6C%67%2E%67%72%61%70%68%69%63%73%2E%6E%65%77%42%72%75%73%68%28%64%6C%67%2E%67%72%61%70%68%69%63%73%2E%42%72%75%73%68%54%79%70%65%2E%53%4F%4C%49%44%5F%43%4F%4C%4F%52%2C%20%5B%72%61%6E%64%2C%20%72%61%6E%64%2C%20%72%61%6E%64%2C%20%31%5D%29%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%64%6C%67%2E%67%72%61%70%68%69%63%73%2E%62%61%63%6B%67%72%6F%75%6E%64%43%6F%6C%6F%72%20%3D%20%62%67%42%72%75%73%68%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%64%6C%67%2E%67%72%2E%61%62%6F%75%74%50%6E%6C%2E%67%72%61%70%68%69%63%73%2E%62%61%63%6B%67%72%6F%75%6E%64%43%6F%6C%6F%72%20%3D%20%77%68%69%74%65%42%72%75%73%68%3B%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%64%6C%67%2E%6C%61%79%6F%75%74%2E%6C%61%79%6F%75%74%28%74%72%75%65%29%3B%20%2F%2F%20%74%6F%20%67%65%74%20%77%69%6E%64%6F%77%20%62%6F%75%6E%64%73%0A%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%61%64%64%4E%61%62%73%63%72%69%70%74%73%42%61%63%6B%67%72%6F%75%6E%64%53%69%67%6E%61%74%75%72%65%28%64%6C%67%29%3B%0A%20%20%20%20%20%20%20%20%7D%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%64%6C%67%2E%67%72%2E%62%74%6E%73%47%72%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%7B%20%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%7D%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%2F%2F%20%6F%70%65%6E%20%68%6F%6D%65%70%61%67%65%20%75%72%6C%0A%20%20%20%20%20%20%20%20%64%6C%67%2E%67%72%2E%62%74%6E%73%47%72%2E%76%69%73%69%74%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%77%77%77%2E%6E%61%62%73%63%72%69%70%74%73%2E%63%6F%6D%2F%22%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%20%20%20%20%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%7B%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%7D%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%65%6C%73%65%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%7D%0A%20%20%20%20%20%20%20%20%20%20%20%20%7D%0A%20%20%20%20%20%20%20%20%20%20%20%20%65%6C%73%65%0A%20%20%20%20%20%20%20%20%20%20%20%20%7B%0A%20%20%20%20%20%20%20%20%20%20%20%20%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%0A%20%20%20%20%20%20%20%20%20%20%20%20%7D%0A%20%20%20%20%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%20%20%20%20%74%72%79%0A%20%20%20%20%20%20%20%20%20%20%20%20%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%20%20%20%20%20%20%20%20%7D%0A%20%20%20%20%20%20%20%20%0A%20%20%20%20%20%20%20%20%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%20%20%20%20%20%20%20%20%64%6C%67%2E%73%68%6F%77%28%29%3B'));
    };
}


/**
 Creates an instance of the main class and run it
*/
new RepositionAnchorPoint().run(this);
