/*! @azure/msal-browser v2.24.0 2022-05-02 */
'use strict';
import { __extends, __awaiter, __generator, __assign } from '../_virtual/_tslib.js';
import { OIDC_DEFAULT_SCOPES, AuthError, UrlString, ThrottlingUtils, ProtocolUtils, PerformanceEvents, Constants } from '@azure/msal-common';
import { StandardInteractionClient } from './StandardInteractionClient.js';
import { PopupUtils } from '../utils/PopupUtils.js';
import { EventType } from '../event/EventType.js';
import { InteractionType, ApiId } from '../utils/BrowserConstants.js';
import { PopupHandler } from '../interaction_handler/PopupHandler.js';
import { BrowserUtils } from '../utils/BrowserUtils.js';
import { NativeInteractionClient } from './NativeInteractionClient.js';
import { NativeMessageHandler } from '../broker/nativeBroker/NativeMessageHandler.js';
import { BrowserAuthError } from '../error/BrowserAuthError.js';

/*
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License.
 */
var PopupClient = /** @class */ (function (_super) {
    __extends(PopupClient, _super);
    function PopupClient() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Acquires tokens by opening a popup window to the /authorize endpoint of the authority
     * @param request
     */
    PopupClient.prototype.acquireToken = function (request) {
        try {
            var popupName = PopupUtils.generatePopupName(this.config.auth.clientId, request.scopes || OIDC_DEFAULT_SCOPES, request.authority || this.config.auth.authority, this.correlationId);
            var popupWindowAttributes = request.popupWindowAttributes || {};
            // asyncPopups flag is true. Acquires token without first opening popup. Popup will be opened later asynchronously.
            if (this.config.system.asyncPopups) {
                this.logger.verbose("asyncPopups set to true, acquiring token");
                // Passes on popup position and dimensions if in request
                return this.acquireTokenPopupAsync(request, popupName, popupWindowAttributes);
            }
            else {
                // asyncPopups flag is set to false. Opens popup before acquiring token.
                this.logger.verbose("asyncPopup set to false, opening popup before acquiring token");
                var popup = PopupUtils.openSizedPopup("about:blank", popupName, popupWindowAttributes, this.logger);
                return this.acquireTokenPopupAsync(request, popupName, popupWindowAttributes, popup);
            }
        }
        catch (e) {
            return Promise.reject(e);
        }
    };
    /**
     * Clears local cache for the current user then opens a popup window prompting the user to sign-out of the server
     * @param logoutRequest
     */
    PopupClient.prototype.logout = function (logoutRequest) {
        try {
            this.logger.verbose("logoutPopup called");
            var validLogoutRequest = this.initializeLogoutRequest(logoutRequest);
            var popupName = PopupUtils.generateLogoutPopupName(this.config.auth.clientId, validLogoutRequest);
            var authority = logoutRequest && logoutRequest.authority;
            var mainWindowRedirectUri = logoutRequest && logoutRequest.mainWindowRedirectUri;
            var popupWindowAttributes = (logoutRequest === null || logoutRequest === void 0 ? void 0 : logoutRequest.popupWindowAttributes) || {};
            // asyncPopups flag is true. Acquires token without first opening popup. Popup will be opened later asynchronously.
            if (this.config.system.asyncPopups) {
                this.logger.verbose("asyncPopups set to true");
                // Passes on popup position and dimensions if in request
                return this.logoutPopupAsync(validLogoutRequest, popupName, popupWindowAttributes, authority, undefined, mainWindowRedirectUri);
            }
            else {
                // asyncPopups flag is set to false. Opens popup before logging out.
                this.logger.verbose("asyncPopup set to false, opening popup");
                var popup = PopupUtils.openSizedPopup("about:blank", popupName, popupWindowAttributes, this.logger);
                return this.logoutPopupAsync(validLogoutRequest, popupName, popupWindowAttributes, authority, popup, mainWindowRedirectUri);
            }
        }
        catch (e) {
            // Since this function is synchronous we need to reject
            return Promise.reject(e);
        }
    };
    /**
     * Helper which obtains an access_token for your API via opening a popup window in the user's browser
     * @param validRequest
     * @param popupName
     * @param popup
     * @param popupWindowAttributes
     *
     * @returns A promise that is fulfilled when this function has completed, or rejected if an error was raised.
     */
    PopupClient.prototype.acquireTokenPopupAsync = function (request, popupName, popupWindowAttributes, popup) {
        return __awaiter(this, void 0, void 0, function () {
            var serverTelemetryManager, validRequest, authCodeRequest, authClient, isNativeBroker, fetchNativeAccountIdMeasurement, navigateUrl, interactionHandler, popupParameters, popupWindow, hash, serverParams, state_1, nativeInteractionClient, userRequestState, result, e_1;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.logger.verbose("acquireTokenPopupAsync called");
                        serverTelemetryManager = this.initializeServerTelemetryManager(ApiId.acquireTokenPopup);
                        return [4 /*yield*/, this.initializeAuthorizationRequest(request, InteractionType.Popup)];
                    case 1:
                        validRequest = _a.sent();
                        this.browserStorage.updateCacheEntries(validRequest.state, validRequest.nonce, validRequest.authority, validRequest.loginHint || Constants.EMPTY_STRING, validRequest.account || null);
                        _a.label = 2;
                    case 2:
                        _a.trys.push([2, 8, , 9]);
                        return [4 /*yield*/, this.initializeAuthorizationCodeRequest(validRequest)];
                    case 3:
                        authCodeRequest = _a.sent();
                        return [4 /*yield*/, this.createAuthCodeClient(serverTelemetryManager, validRequest.authority, validRequest.azureCloudOptions)];
                    case 4:
                        authClient = _a.sent();
                        this.logger.verbose("Auth code client created");
                        isNativeBroker = NativeMessageHandler.isNativeAvailable(this.config, this.logger, this.nativeMessageHandler, request.authenticationScheme);
                        fetchNativeAccountIdMeasurement = void 0;
                        if (isNativeBroker) {
                            fetchNativeAccountIdMeasurement = this.performanceClient.startMeasurement(PerformanceEvents.FetchAccountIdWithNativeBroker, request.correlationId);
                        }
                        return [4 /*yield*/, authClient.getAuthCodeUrl(__assign(__assign({}, validRequest), { nativeBroker: isNativeBroker }))];
                    case 5:
                        navigateUrl = _a.sent();
                        interactionHandler = new PopupHandler(authClient, this.browserStorage, authCodeRequest, this.logger);
                        popupParameters = {
                            popup: popup,
                            popupName: popupName,
                            popupWindowAttributes: popupWindowAttributes
                        };
                        popupWindow = interactionHandler.initiateAuthRequest(navigateUrl, popupParameters);
                        this.eventHandler.emitEvent(EventType.POPUP_OPENED, InteractionType.Popup, { popupWindow: popupWindow }, null);
                        return [4 /*yield*/, interactionHandler.monitorPopupForHash(popupWindow)];
                    case 6:
                        hash = _a.sent();
                        serverParams = UrlString.getDeserializedHash(hash);
                        state_1 = this.validateAndExtractStateFromHash(serverParams, InteractionType.Popup, validRequest.correlationId);
                        // Remove throttle if it exists
                        ThrottlingUtils.removeThrottle(this.browserStorage, this.config.auth.clientId, authCodeRequest);
                        if (serverParams.accountId) {
                            this.logger.verbose("Account id found in hash, calling WAM for token");
                            // end measurement for server call with native brokering enabled
                            if (fetchNativeAccountIdMeasurement) {
                                fetchNativeAccountIdMeasurement.endMeasurement({
                                    success: true,
                                    isNativeBroker: true
                                });
                            }
                            if (!this.nativeMessageHandler) {
                                throw BrowserAuthError.createNativeConnectionNotEstablishedError();
                            }
                            nativeInteractionClient = new NativeInteractionClient(this.config, this.browserStorage, this.browserCrypto, this.logger, this.eventHandler, this.navigationClient, ApiId.acquireTokenPopup, this.performanceClient, this.nativeMessageHandler, serverParams.accountId, validRequest.correlationId);
                            userRequestState = ProtocolUtils.parseRequestState(this.browserCrypto, state_1).userRequestState;
                            return [2 /*return*/, nativeInteractionClient.acquireToken(__assign(__assign({}, validRequest), { state: userRequestState, prompt: undefined // Server should handle the prompt, ideally native broker can do this part silently
                                 })).finally(function () {
                                    _this.browserStorage.cleanRequestByState(state_1);
                                })];
                        }
                        return [4 /*yield*/, interactionHandler.handleCodeResponseFromHash(hash, state_1, authClient.authority, this.networkClient)];
                    case 7:
                        result = _a.sent();
                        return [2 /*return*/, result];
                    case 8:
                        e_1 = _a.sent();
                        if (popup) {
                            // Close the synchronous popup if an error is thrown before the window unload event is registered
                            popup.close();
                        }
                        if (e_1 instanceof AuthError) {
                            e_1.setCorrelationId(this.correlationId);
                        }
                        serverTelemetryManager.cacheFailedRequest(e_1);
                        this.browserStorage.cleanRequestByState(validRequest.state);
                        throw e_1;
                    case 9: return [2 /*return*/];
                }
            });
        });
    };
    /**
     *
     * @param validRequest
     * @param popupName
     * @param requestAuthority
     * @param popup
     * @param mainWindowRedirectUri
     * @param popupWindowAttributes
     */
    PopupClient.prototype.logoutPopupAsync = function (validRequest, popupName, popupWindowAttributes, requestAuthority, popup, mainWindowRedirectUri) {
        return __awaiter(this, void 0, void 0, function () {
            var serverTelemetryManager, authClient, logoutUri, popupUtils, popupWindow, e_2, navigationOptions, absoluteUrl, e_3;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.logger.verbose("logoutPopupAsync called");
                        this.eventHandler.emitEvent(EventType.LOGOUT_START, InteractionType.Popup, validRequest);
                        serverTelemetryManager = this.initializeServerTelemetryManager(ApiId.logoutPopup);
                        _a.label = 1;
                    case 1:
                        _a.trys.push([1, 8, , 9]);
                        // Clear cache on logout
                        return [4 /*yield*/, this.clearCacheOnLogout(validRequest.account)];
                    case 2:
                        // Clear cache on logout
                        _a.sent();
                        return [4 /*yield*/, this.createAuthCodeClient(serverTelemetryManager, requestAuthority)];
                    case 3:
                        authClient = _a.sent();
                        this.logger.verbose("Auth code client created");
                        logoutUri = authClient.getLogoutUri(validRequest);
                        this.eventHandler.emitEvent(EventType.LOGOUT_SUCCESS, InteractionType.Popup, validRequest);
                        popupUtils = new PopupUtils(this.browserStorage, this.logger);
                        popupWindow = popupUtils.openPopup(logoutUri, { popupName: popupName, popupWindowAttributes: popupWindowAttributes, popup: popup });
                        this.eventHandler.emitEvent(EventType.POPUP_OPENED, InteractionType.Popup, { popupWindow: popupWindow }, null);
                        _a.label = 4;
                    case 4:
                        _a.trys.push([4, 6, , 7]);
                        // Don't care if this throws an error (User Cancelled)
                        return [4 /*yield*/, popupUtils.monitorPopupForSameOrigin(popupWindow)];
                    case 5:
                        // Don't care if this throws an error (User Cancelled)
                        _a.sent();
                        this.logger.verbose("Popup successfully redirected to postLogoutRedirectUri");
                        return [3 /*break*/, 7];
                    case 6:
                        e_2 = _a.sent();
                        this.logger.verbose("Error occurred while monitoring popup for same origin. Session on server may remain active. Error: " + e_2);
                        return [3 /*break*/, 7];
                    case 7:
                        popupUtils.cleanPopup(popupWindow);
                        if (mainWindowRedirectUri) {
                            navigationOptions = {
                                apiId: ApiId.logoutPopup,
                                timeout: this.config.system.redirectNavigationTimeout,
                                noHistory: false
                            };
                            absoluteUrl = UrlString.getAbsoluteUrl(mainWindowRedirectUri, BrowserUtils.getCurrentUri());
                            this.logger.verbose("Redirecting main window to url specified in the request");
                            this.logger.verbosePii("Redirecting main window to: " + absoluteUrl);
                            this.navigationClient.navigateInternal(absoluteUrl, navigationOptions);
                        }
                        else {
                            this.logger.verbose("No main window navigation requested");
                        }
                        return [3 /*break*/, 9];
                    case 8:
                        e_3 = _a.sent();
                        if (popup) {
                            // Close the synchronous popup if an error is thrown before the window unload event is registered
                            popup.close();
                        }
                        if (e_3 instanceof AuthError) {
                            e_3.setCorrelationId(this.correlationId);
                        }
                        this.browserStorage.setInteractionInProgress(false);
                        this.eventHandler.emitEvent(EventType.LOGOUT_FAILURE, InteractionType.Popup, null, e_3);
                        this.eventHandler.emitEvent(EventType.LOGOUT_END, InteractionType.Popup);
                        serverTelemetryManager.cacheFailedRequest(e_3);
                        throw e_3;
                    case 9:
                        this.eventHandler.emitEvent(EventType.LOGOUT_END, InteractionType.Popup);
                        return [2 /*return*/];
                }
            });
        });
    };
    return PopupClient;
}(StandardInteractionClient));

export { PopupClient };
//# sourceMappingURL=PopupClient.js.map
