// cspell:ignore arraybuffer
import { __awaiter } from "tslib";
import { parse as parseProtobuf } from './parse';
import { ProtobufJsonParserBrowser } from './protobuf-json-parser.browser';
import { getFilePathFromURL } from './utils';
function revokeObjectURLs(files) {
    files.forEach((file) => {
        URL.revokeObjectURL(file.url);
    });
}
function request(url, options) {
    return __awaiter(this, void 0, void 0, function* () {
        return new Promise((resolve, reject) => {
            const XMLHttpRequestClass = options.xhrAgentClass || XMLHttpRequest;
            const xhr = new XMLHttpRequestClass();
            xhr.open('GET', url, true);
            xhr.onreadystatechange = () => {
                if (xhr.readyState === 4) {
                    resolve(xhr.responseText);
                }
            };
            xhr.onerror = (err) => {
                reject(err);
            };
            xhr.send();
        });
    });
}
function isRemoteFile(rootFile) {
    return typeof rootFile === 'string';
}
function maybeReplaceXhr(rootFile, options) {
    if (!isRemoteFile(rootFile) || !options.xhrAgentClass) {
        return () => { };
    }
    const util = require('protobufjs/src/util');
    const originalXhr = util.fetch.xhr;
    /**
     * 覆盖 protobufjs 中的 xhr 方法
     * https://github.com/protobufjs/protobuf.js/blob/protobufjs-v7.2.3/lib/fetch/index.js#L78
     */
    util.fetch.xhr = (filename, opts, callback) => {
        const xhr = new options.xhrAgentClass();
        xhr.onreadystatechange = function fetchOnReadyStateChange() {
            if (xhr.readyState !== 4) {
                return undefined;
            }
            if (xhr.status !== 0 && xhr.status !== 200) {
                return callback(Error('status ' + xhr.status));
            }
            /**
             * packages/app-renderer/src/lib/ApiTestRunner/chrome/agent.ts 中实现的 AgentXHR
             * 的 responseText 字段有可能是 Promise 类型，这里做兼容
             */
            Promise.resolve(xhr.responseText).then(responseText => {
                if (opts.binary) {
                    let buffer = xhr.response;
                    if (!buffer) {
                        buffer = [];
                        for (let i = 0; i < responseText.length; ++i) {
                            buffer.push(responseText.charCodeAt(i) & 255);
                        }
                    }
                    return callback(null, typeof Uint8Array !== 'undefined' ? new Uint8Array(buffer) : buffer);
                }
                return callback(null, responseText);
            });
        };
        if (opts.binary) {
            // ref: https://developer.mozilla.org/en-US/docs/Web/API/XMLHttpRequest/Sending_and_Receiving_Binary_Data#Receiving_binary_data_in_older_browsers
            if ('overrideMimeType' in xhr) {
                xhr.overrideMimeType('text/plain; charset=x-user-defined');
            }
            xhr.responseType = 'arraybuffer';
        }
        xhr.open('GET', filename);
        xhr.send();
    };
    return () => {
        util.fetch.xhr = originalXhr;
    };
}
export function parse(rootFile, options = {}) {
    var _a;
    return __awaiter(this, void 0, void 0, function* () {
        let root;
        const fileMap = {};
        if (isRemoteFile(rootFile)) {
            root = {
                name: '',
                relativePath: getFilePathFromURL(rootFile, rootFile),
                url: rootFile,
            };
        }
        else {
            const rootFileURL = URL.createObjectURL(rootFile);
            fileMap[rootFileURL] = rootFile;
            root = {
                name: rootFile.name,
                relativePath: rootFile.name,
                url: rootFileURL,
            };
        }
        const files = [];
        const dependentFiles = (_a = options.includes) !== null && _a !== void 0 ? _a : [];
        for (let file of dependentFiles) {
            if (file.webkitRelativePath === undefined) {
                throw Error('Cannot found field "webkitRelativePath" in File.');
            }
            const fileURL = URL.createObjectURL(file);
            files.push({
                name: file.name,
                relativePath: file.webkitRelativePath.replace(/^[^\/]+\//, ''),
                url: fileURL,
            });
            fileMap[fileURL] = file;
        }
        const restore = maybeReplaceXhr(rootFile, options);
        try {
            const result = yield parseProtobuf({
                rootFile: root,
                dependencies: files,
                protobufJsonParser: ProtobufJsonParserBrowser,
                appName: options.appName,
                getFileContent(url) {
                    return __awaiter(this, void 0, void 0, function* () {
                        if (url.startsWith('http://') || url.startsWith('https://')) {
                            const content = yield request(url, options);
                            return content;
                        }
                        const file = fileMap[url];
                        if (file === undefined) {
                            throw Error(`File (${url}) not found.`);
                        }
                        return yield file.text();
                    });
                },
            });
            return result;
        }
        finally {
            restore();
            revokeObjectURLs([root, ...files]);
        }
    });
}
