import { parse } from 'protocol-buffers-schema';
import { __awaiter } from 'tslib';
import primitive from './types';
function normalizeFilePath(filePath) {
  const normalizedFilePath = filePath.replace(/^\.\//, '');
  return normalizedFilePath;
}
class Compiler {
  constructor(rootFile, dependedFiles, getFileContent) {
    this.rootFile = rootFile;
    this.dependedFiles = dependedFiles;
    this.getFileContent = getFileContent;
    // private readonly inBrowserMode: boolean
    // private fileMap = new Map<FileInfo, string>()
    this.visitedFiles = {};
    this.messages = {};
    this.enums = {};
    this.root = {
      definitions: {},
    };
    // const rootFileURL = rootFile.url.trim()
    // this.inBrowserMode = rootFileURL.startsWith('blob:') || rootFileURL.startsWith('http://') || rootFileURL.startsWith('https://')
  }
  open(file) {
    return __awaiter(this, void 0, void 0, function* () {
      console.log('open', file);
      const content = (yield this.getFileContent(file.url)).toString();
      var schema = parse(content);
      this.visit(schema, schema.package || '');
      // const sourceFilePath = this.fileMap.get(file)
      for (const path of schema.imports) {
        const file = this.findFile(path);
        // console.log('findFile', path, sourceFilePath, file)
        if (file) {
          // this.fileMap.set(file, path)
          yield this.open(file);
        }
      }
      return schema;
    });
  }
  findFile(filePath) {
    if (this.visitedFiles[filePath]) {
      console.warn(`Circular reference: ${filePath}`);
      return ''; // 防止循环引用
    }
    const file = this.dependedFiles.find((file) => {
      const normalizedFilePath = normalizeFilePath(filePath);
      return file.relativePath === normalizedFilePath;
      // // browser
      // if (this.inBrowserMode) {
      //   return file.relativePath === normalizedFilePath
      // }
      // console.log('--------- file path', filePath, this.includeDirs, file)
      // // node
      // if (!Array.isArray(this.includeDirs)) {
      //   return file.url.endsWith(normalizedFilePath)
      // }
      // for (let dir of this.includeDirs) {
      //   const dirPath = typeof dir === 'string' ? dir : dir.url
      //   const normalizedDirPath = dirPath.replace(/^\.\//, '')
      //   if (file.url.endsWith(`${normalizedDirPath}/${normalizedFilePath}`)) {
      //     return true
      //   }
      // }
      // return false
    });
    if (file) {
      this.visitedFiles[filePath] = true;
    }
    return file;
  }
  /**
   * Visits a schema in the tree, and assigns messages and enums to the lookup tables.
   */
  visit(schema, prefix) {
    if (schema.enums) {
      schema.enums.forEach((e) => {
        e.id = prefix + (prefix ? '.' : '') + (e.id || e.name);
        this.enums[e.id] = e;
      });
    }
    if (schema.messages) {
      schema.messages.forEach((m) => {
        m.id = prefix + (prefix ? '.' : '') + (m.id || m.name);
        this.messages[m.id] = m;
        this.visit(m, m.id);
      });
    }
  }
  /**
   * Top level compile method. If a type name is provided,
   * compiles just that type and its dependencies. Otherwise,
   * compiles all types in the files.
   */
  compile(type) {
    return __awaiter(this, void 0, void 0, function* () {
      if (type) {
        this.resolve(type, '');
      } else {
        yield this.open(this.rootFile);
        Object.values(this.messages).forEach((message) => {
          var _a;
          this.resolve((_a = message.id) !== null && _a !== void 0 ? _a : '', '');
        });
        Object.values(this.enums).forEach((e) => {
          var _a;
          this.resolve((_a = e.id) !== null && _a !== void 0 ? _a : '', '');
        });
      }
      console.log('----------- messages', this.messages);
      return this.root;
    });
  }
  /**
   * Resolves a type name at the given path in the schema tree.
   * Returns a compiled JSON schema.
   */
  resolve(type, from) {
    if (primitive[type]) {
      return primitive[type];
    }
    var lookup = from.split('.');
    for (var i = lookup.length; i >= 0; i--) {
      var id = lookup.slice(0, i).concat(type).join('.');
      // If already defined, reuse
      if (this.root.definitions[id]) {
        return { $ref: '#/definitions/' + id };
      }
      // Compile the message or enum
      if (this.messages[id]) {
        const res = this.compileMessage(this.messages[id]);
        this.root.definitions[id] = res;
        return { $ref: '#/definitions/' + id };
      }
      if (this.enums[id]) {
        return this.compileEnum(this.enums[id]);
      }
    }
    console.warn('Could not resolve ' + type);
    return null;
  }
  /**
   * Compiles a protobuf enum to JSON schema
   */
  compileEnum(enumType) {
    const res = {
      title: enumType.name,
      type: 'string',
      enum: Object.keys(enumType.values),
    };
    return res;
  }
  /**
   * Compiles a protobuf message to JSON schema
   */
  compileMessage(message) {
    var _a;
    const res = {
      title: message.name,
      type: 'object',
      properties: {},
      required: [],
    };
    message.fields.forEach((field) => {
      var _a, _b, _c, _d;
      if (field.map) {
        const fromType = primitive[field.map.from].type;
        if (!['string', 'integer', 'number'].includes(fromType)) {
          console.warn(
            `Can only use "string", "integer" or "number" as map keys at ${message.id}.${field.name}. Got "${fromType}."`,
          );
          return;
        }
        res.properties[field.name] = {
          type: 'object',
          additionalProperties: this.resolve(
            field.map.to,
            (_a = message.id) !== null && _a !== void 0 ? _a : '',
          ),
        };
      } else {
        if (field.repeated) {
          res.properties[field.name] = {
            type: 'array',
            items: this.resolve(field.type, (_b = message.id) !== null && _b !== void 0 ? _b : ''),
          };
        } else {
          const val = this.resolve(
            field.type,
            (_c = message.id) !== null && _c !== void 0 ? _c : '',
          );
          if (val) {
            res.properties[field.name] = val;
          }
        }
      }
      if (field.required) {
        res.required = (_d = res.required) !== null && _d !== void 0 ? _d : [];
        res.required.push(field.name);
      }
    });
    if (((_a = res.required) === null || _a === void 0 ? void 0 : _a.length) === 0) {
      delete res.required;
    }
    return res;
  }
}
export function compile(rootFile, dependentFiles, getFileContent) {
  return __awaiter(this, void 0, void 0, function* () {
    var compiler = new Compiler(rootFile, dependentFiles, getFileContent);
    return yield compiler.compile();
  });
}
