import { __awaiter } from "tslib";
import { JsonSchemaParser } from './json-schema/parser';
import { correctURL, getFileName, isEnum, isMessage, isNamespace, isService, resolveFullName, } from './utils';
export var MethodKind;
(function (MethodKind) {
    MethodKind["UNARY"] = "UNARY";
    MethodKind["SERVER_STREAMING"] = "SERVER_STREAMING";
    MethodKind["CLIENT_STREAMING"] = "CLIENT_STREAMING";
    MethodKind["BIDIRECTIONAL_STREAMING"] = "BIDIRECTIONAL_STREAMING";
})(MethodKind || (MethodKind = {}));
function parseNamespace(obj, ctx) {
    if (!isNamespace(obj)) {
        return;
    }
    obj.nestedArray.forEach((value) => {
        parseNamespace(value, ctx);
        parseService(value, ctx);
    });
}
function isRootProtoPackage(ctx, packageName) {
    return ctx.services.some((service) => {
        var _a;
        return ((_a = service.parent) === null || _a === void 0 ? void 0 : _a.name) === packageName;
    });
}
function isRootFile(ctx, fileName) {
    return ctx.rootFile.url === correctURL(fileName !== null && fileName !== void 0 ? fileName : '');
}
function parseService(obj, ctx) {
    var _a;
    if (!isService(obj)) {
        return;
    }
    if (obj.filename && !isRootFile(ctx, obj.filename) && !isRootProtoPackage(ctx, (_a = obj.parent) === null || _a === void 0 ? void 0 : _a.name)) {
        // 使用 json 文件导入时，protobuf.ReflectionObject 没有文件名，无法判断是否为根文件，因此全部导入。
        // 否则，只导入根 proto 文件（或与根文件相同 package 的文件）中定义的 service
        return;
    }
    ctx.services.push(obj);
    obj.methodsArray.forEach((method) => {
        parseMethod(method, ctx);
    });
}
function parseMethod(method, ctx) {
    method.resolve();
    ctx.methods.push(method);
    if (method.resolvedRequestType) {
        parseMessage(method.resolvedRequestType, ctx);
    }
    if (method.resolvedResponseType) {
        parseMessage(method.resolvedResponseType, ctx);
    }
}
function parseMessage(obj, ctx) {
    obj.resolve();
    if (!isMessage(obj) && !isEnum(obj)) {
        return;
    }
    if ((isMessage(obj) || isEnum(obj)) && ctx.messages.includes(obj)) {
        return;
    }
    ctx.messages.push(obj);
    if (isMessage(obj)) {
        obj.fieldsArray.forEach((field) => {
            field.resolve();
            if (field.resolvedType) {
                parseMessage(field.resolvedType, ctx);
            }
        });
    }
}
function resolveMethodKind(method) {
    if (method.requestStream && method.responseStream) {
        return MethodKind.BIDIRECTIONAL_STREAMING;
    }
    if (method.requestStream) {
        return MethodKind.CLIENT_STREAMING;
    }
    if (method.responseStream) {
        return MethodKind.SERVER_STREAMING;
    }
    return MethodKind.UNARY;
}
function resolveMethodFullPath(method) {
    // .abc.def.ghi -> /abc.def/ghi
    return method.fullName.replace(/^\.(.*)\.([^.]+)$/, '/$1/$2');
}
function toSpecificationFormat(ctx, appName) {
    var _a, _b, _c;
    return __awaiter(this, void 0, void 0, function* () {
        const rpcServices = ctx.services.map((service) => {
            const rpcMethods = service.methodsArray.map((method) => {
                return {
                    name: method.name,
                    kind: resolveMethodKind(method),
                    path: resolveMethodFullPath(method),
                    commonParameters: [],
                    requestPath: method.resolvedRequestType
                        ? resolveFullName(method.resolvedRequestType.fullName)
                        : '',
                    responsePath: method.resolvedResponseType
                        ? resolveFullName(method.resolvedResponseType.fullName)
                        : '',
                    serverUrl: '',
                    requestId: 0,
                    responseId: 0,
                    moduleId: 0,
                    serviceId: 0,
                };
            });
            return {
                moduleId: 0,
                name: resolveFullName(service.fullName),
                rpcMethods,
            };
        });
        const jsonSchemaParser = new JsonSchemaParser(ctx.messages, appName);
        const jsonSchemaMap = jsonSchemaParser.parse();
        const rpcMessages = ctx.messages.map((message) => {
            const path = resolveFullName(message.fullName);
            const jsonSchema = jsonSchemaMap[path] || {};
            return {
                name: message.name,
                path: resolveFullName(message.fullName),
                definitionType: 'PROTO',
                content: JSON.stringify(message.toJSON()),
                jsonSchema,
                moduleId: 0,
            };
        });
        const rpcModule = {
            name: resolveFullName((_c = (_b = (_a = ctx.services[0]) === null || _a === void 0 ? void 0 : _a.parent) === null || _b === void 0 ? void 0 : _b.fullName) !== null && _c !== void 0 ? _c : '') ||
                `filename:${getFileName(ctx.rootFile.relativePath)}`,
            type: 'GRPC',
            dataSourceId: 0,
            rpcMessages,
            rpcServices,
        };
        const resources = yield Promise.all(ctx.dataResources.map((dataResource) => __awaiter(this, void 0, void 0, function* () {
            const content = (yield ctx.getFileContent(dataResource.file.url)).toString();
            if (dataResource.file.url === ctx.rootFile.url) {
                ctx.isJsonProto = content.trim().startsWith('{');
            }
            return {
                relativePath: dataResource.file.relativePath,
                content,
                isRoot: dataResource.isRoot,
                dataSourceId: 0,
            };
        })));
        const isBlobFile = ctx.rootFile.url.startsWith('blob:');
        const result = {
            name: '',
            type: ctx.rootFile.url.startsWith('http://') || ctx.rootFile.url.startsWith('https://')
                ? 'URL'
                : 'FILE',
            format: ctx.isJsonProto ? 'json' : 'proto',
            location: isBlobFile ? '' : ctx.rootFile.url,
            dependentLocation: isBlobFile
                ? []
                : ctx.dependencies.filter((item) => typeof item === 'string'),
            dataResources: resources,
            rpcModule,
        };
        return result;
    });
}
export function parseProtobufSpec(options) {
    return __awaiter(this, void 0, void 0, function* () {
        const { root } = options;
        const ctx = Object.assign(Object.assign({}, options), { services: [], methods: [], messages: [], isJsonProto: false });
        parseNamespace(root, ctx);
        const result = yield toSpecificationFormat(ctx, options.appName);
        return result;
    });
}
