import { __awaiter } from "tslib";
import fs from 'fs';
import path from 'path';
import protobuf from 'protobufjs';
import { ProtobufJsonParserBase } from './protobuf-json-parser-base';
function getExistFilePath(prefix, relativePath) {
    const filePath = path.resolve(prefix, relativePath);
    try {
        if (fs.statSync(filePath).isFile()) {
            return filePath;
        }
    }
    catch (err) {
        // ignore
    }
    return null;
}
export class ProtobufJsonParserNode extends ProtobufJsonParserBase {
    useFile(originalPath, targetPath) {
        this.ctx.usedFiles.push({
            file: {
                name: path.basename(originalPath),
                relativePath: originalPath,
                url: targetPath,
            },
            isRoot: false,
        });
    }
    parse() {
        return __awaiter(this, void 0, void 0, function* () {
            let root = new protobuf.Root();
            let err = null;
            root.resolvePath = (source, target) => {
                if (path.isAbsolute(target)) {
                    return target;
                }
                const normalizedTarget = target.trim().replace(/^\.\//, '');
                if (this.rootFile.url === target) {
                    // 根文件
                    return target;
                }
                if (!Array.isArray(this.dependencies)) {
                    return normalizedTarget;
                }
                let filePath = this.resolveFileInDependencies(normalizedTarget, this.dependencies);
                if (!filePath) {
                    const defaultDir = path.dirname(this.rootFile.url);
                    filePath = this.resolveFileInDependencies(normalizedTarget, [defaultDir]);
                    if (filePath) {
                        this.dependencies.push(defaultDir);
                    }
                }
                if (filePath) {
                    return filePath;
                }
                // 依赖文件中如果依赖了一个不存在的文件，这里直接 throw error 是不会使解析停止的。
                // 这会造成错误提示不准确（本来是文件找不到，却报了某个 Message、Enum 未找到）
                // 所以把错误记录到全局，在解析完成后判断是否需要抛出错误。
                err = Error(`File (${target}) not found.`);
                throw err;
            };
            root = root.loadSync(this.rootFile.url, { keepCase: true });
            if (err !== null) {
                throw err;
            }
            return {
                root,
                dataResources: this.ctx.usedFiles,
            };
        });
    }
    resolveFileInDependencies(target, dependencies) {
        for (let prefix of dependencies) {
            if (typeof prefix !== 'string') {
                continue;
            }
            const filePath = getExistFilePath(prefix, target);
            if (filePath) {
                this.useFile(target, filePath);
                return filePath;
            }
        }
        return null;
    }
}
