"use strict";
// cspell:ignore oneofs, oneof, apifox
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonSchemaParser = void 0;
const tslib_1 = require("tslib");
const utils_1 = require("../utils");
const types_1 = tslib_1.__importDefault(require("./types"));
function isMapField(field) {
    return field.map;
}
class JsonSchemaParser {
    constructor(messages, appName = '') {
        this.messages = messages;
        this._messageMap = {};
        this._visited = [];
        this._appName = appName.toLowerCase();
        this._messageMap = messages.reduce((result, message) => {
            result[(0, utils_1.resolveFullName)(message.fullName)] = message;
            return result;
        }, {});
    }
    parse() {
        const result = {};
        for (const message of this.messages) {
            result[(0, utils_1.resolveFullName)(message.fullName)] = this._toJsonSchema(message, result);
        }
        return result;
    }
    _toJsonSchema(message, cache) {
        var _a, _b;
        if (cache[(0, utils_1.resolveFullName)(message.fullName)]) {
            return cache[(0, utils_1.resolveFullName)(message.fullName)];
        }
        const res = {
            title: message.name,
            type: 'object',
            properties: {},
            required: [],
            [`x-${this._appName}-orders`]: [],
        };
        if ((0, utils_1.isEnum)(message)) {
            return {
                title: message.name,
                type: 'string',
                enum: Object.keys(message.values),
            };
        }
        for (const field of Object.values(message.fields)) {
            if (isMapField(field)) {
                if (!['string', 'integer', 'number'].includes(field.keyType)) {
                    console.warn(`Can only use "string", "integer" or "number" as map keys at ${message.name}.${field.name}. Got "${field.keyType}."`);
                    break;
                }
                res.properties[field.name] = {
                    type: 'object',
                    additionalProperties: this._resolveField(getFileType(field), cache),
                };
            }
            else {
                if (field.repeated) {
                    res.properties[field.name] = {
                        type: 'array',
                        items: this._resolveField(getFileType(field), cache),
                    };
                }
                else {
                    const val = this._resolveField(getFileType(field), cache);
                    if (val) {
                        res.properties[field.name] = val;
                    }
                }
            }
            if (field.required) {
                res.required = (_a = res.required) !== null && _a !== void 0 ? _a : [];
                res.required.push(field.name);
            }
            if (!res[`x-${this._appName}-orders`]) {
                res[`x-${this._appName}-orders`] = [];
            }
            res[`x-${this._appName}-orders`].push(field.name);
        }
        if (((_b = res.required) === null || _b === void 0 ? void 0 : _b.length) === 0) {
            delete res.required;
        }
        if (!message.oneofs) {
            return res;
        }
        for (const val of Object.values(message.oneofs)) {
            if (!res.allOf) {
                res.allOf = [];
            }
            let oneOfs = [];
            val.oneof.forEach((key) => {
                oneOfs.push({
                    required: [key],
                });
            });
            res.allOf.push({
                oneOf: oneOfs,
            });
        }
        return res;
    }
    _resolveField(type, cache) {
        if (types_1.default[type]) {
            return types_1.default[type];
        }
        if (cache[type]) {
            return { $ref: `#/definitions/${type}` };
        }
        if (this._messageMap[type]) {
            if (!this._visited.includes(type)) {
                this._visited.push(type);
                this._toJsonSchema(this._messageMap[type], cache);
            }
            return { $ref: `#/definitions/${type}` };
        }
        return null;
    }
}
exports.JsonSchemaParser = JsonSchemaParser;
function getFileType(field) {
    if (field.resolvedType) {
        return (0, utils_1.resolveFullName)(field.resolvedType.fullName);
    }
    return field.type;
}
