"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DubboAdapter = exports.ZOOKEEPER_DUBBO_DEFAULT_ROOT_NODE = void 0;
var tslib_1 = require("tslib");
var types_1 = require("../types");
var url_parse_1 = tslib_1.__importDefault(require("url-parse"));
var service_profile_1 = require("../service-profile");
var dubbo_metadata_parser_1 = require("../service-profile/dubbo-metadata-parser");
exports.ZOOKEEPER_DUBBO_DEFAULT_ROOT_NODE = 'dubbo';
function stringFilter(item) {
    return typeof item === 'string';
}
function nacosResultFilter(item) {
    return typeof item === 'object' &&
        item !== null &&
        typeof item.ip === 'string' &&
        ['number', 'string'].includes(typeof item.port);
}
function edasServiceProviderResultFilter(item) {
    return typeof item === 'object' &&
        item !== null &&
        typeof item.Ip === 'string' &&
        typeof item.Port === 'string' &&
        typeof item.SerializeType === 'string' &&
        typeof item.Timeout === 'string';
}
var DubboAdapter = /** @class */ (function () {
    function DubboAdapter(options) {
        this.options = options;
    }
    Object.defineProperty(DubboAdapter.prototype, "logger", {
        get: function () {
            var _a;
            return (_a = this.options.logger) !== null && _a !== void 0 ? _a : console;
        },
        enumerable: false,
        configurable: true
    });
    DubboAdapter.prototype._getServiceOptions = function () {
        var serviceOptions = this.options.service;
        if (serviceOptions.serviceType !== types_1.ServiceType.DUBBO) {
            throw new Error("serviceType (".concat(serviceOptions.serviceType, ") is not \"").concat(types_1.ServiceType.DUBBO, "\""));
        }
        return serviceOptions;
    };
    DubboAdapter.prototype.getServiceProviderPath = function (key) {
        var _a, _b, _c;
        var serviceOptions = this._getServiceOptions();
        var registerOptions = this.options.register;
        var registerType = registerOptions.registerType;
        switch (registerType) {
            case types_1.RegisterType.ZOOKEEPER:
                return "/".concat((_a = registerOptions.group) !== null && _a !== void 0 ? _a : exports.ZOOKEEPER_DUBBO_DEFAULT_ROOT_NODE, "/").concat(key, "/providers");
            case types_1.RegisterType.NACOS:
                return "providers:".concat(key, ":").concat((_b = serviceOptions.version) !== null && _b !== void 0 ? _b : '', ":").concat((_c = serviceOptions.group) !== null && _c !== void 0 ? _c : '');
            default:
                throw new Error("registerType (".concat(registerType, ") is not supported"));
        }
    };
    DubboAdapter.prototype.resultFilterForZooKeeper = function (result) {
        var serviceOptions = this._getServiceOptions();
        return result.filter(stringFilter).reduce(function (result, item) {
            var _a, _b, _c, _d;
            var url = (0, url_parse_1.default)(decodeURIComponent(item).trim(), true);
            if (((_a = serviceOptions.group) !== null && _a !== void 0 ? _a : '') !== ((_b = url.query['group']) !== null && _b !== void 0 ? _b : '')) {
                return result;
            }
            if (((_c = serviceOptions.version) !== null && _c !== void 0 ? _c : '') !== ((_d = url.query['version']) !== null && _d !== void 0 ? _d : '')) {
                return result;
            }
            result.push({
                protocol: url.protocol.replace(':', ''),
                hostname: url.hostname,
                port: url.port,
            });
            return result;
        }, []);
    };
    DubboAdapter.prototype.resultFilterForNacos = function (result) {
        return result.filter(nacosResultFilter).reduce(function (result, item) {
            result.push({
                protocol: item.metadata.protocol,
                hostname: item.ip,
                port: item.port.toString(),
            });
            return result;
        }, []);
    };
    DubboAdapter.prototype.resultFilterForEdas = function (result, options) {
        return result.filter(edasServiceProviderResultFilter).reduce(function (result, item) {
            result.push({
                protocol: options.service.serviceType.toLowerCase(),
                hostname: item.Ip,
                port: item.Port
            });
            return result;
        }, []);
    };
    DubboAdapter.prototype.processServiceProviderResult = function (result) {
        if (result.length === 0) {
            return [];
        }
        var registerOptions = this.options.register;
        var registerType = registerOptions.registerType;
        switch (registerType) {
            case types_1.RegisterType.ZOOKEEPER:
                return this.resultFilterForZooKeeper(result);
            case types_1.RegisterType.NACOS:
                return this.resultFilterForNacos(result);
            case types_1.RegisterType.EDAS:
                return this.resultFilterForEdas(result, this.options);
            default:
                this.logger.warn("[Adapter] registerType (".concat(registerType, ") is not supported"));
                return [];
        }
    };
    DubboAdapter.prototype.parseMetadataList = function (appName, metadataList) {
        var parser = new service_profile_1.ServiceProfile(appName, this.options.register.registerType.toUpperCase());
        parser.parseDubboMetadata(metadataList);
        return parser.toJSON();
    };
    DubboAdapter.prototype.getAppInfoList = function (metadataList) {
        return dubbo_metadata_parser_1.DubboMetadataParser.getAppInfoList(metadataList);
    };
    return DubboAdapter;
}());
exports.DubboAdapter = DubboAdapter;
