"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessageLoader = exports.getGenericTypes = void 0;
var tslib_1 = require("tslib");
var types_1 = require("./types");
var jsonpath_1 = tslib_1.__importDefault(require("jsonpath"));
var DEFINITION_TYPE = 'TYPE';
var GENERIC_TYPE_PATTERN = /^([\w.]+)<([\w., ]+)>$/;
var JSON_SCHEMA_TYPES = ['string', 'number', 'integer', 'boolean', 'object', 'array', 'null'];
var TEMP_KEY_PREFIX = '__';
function isEmptySchema(schema) {
    var _a, _b;
    return !schema.$ref &&
        !((_a = schema.enums) === null || _a === void 0 ? void 0 : _a.length) &&
        !((_b = schema.items) === null || _b === void 0 ? void 0 : _b.length) &&
        !Object.keys(schema.properties || {}).length;
}
function getGenericTypes(type) {
    var result = GENERIC_TYPE_PATTERN.exec(type);
    if (!result) {
        return null;
    }
    return tslib_1.__spreadArray([result[1]], result[2].split(',').map(function (t) { return t.trim(); }), true);
}
exports.getGenericTypes = getGenericTypes;
function getTypeFromSchema(schema, fallback) {
    var _a;
    if (fallback === void 0) { fallback = ''; }
    return (_a = schema.type) !== null && _a !== void 0 ? _a : fallback;
}
var MessageLoader = /** @class */ (function () {
    function MessageLoader(_schemas) {
        if (_schemas === void 0) { _schemas = []; }
        this._schemas = _schemas;
        this.dataSchemaMap = {};
        this.messageMap = {};
    }
    Object.defineProperty(MessageLoader.prototype, "schemas", {
        set: function (schemas) {
            this._schemas = schemas;
        },
        enumerable: false,
        configurable: true
    });
    MessageLoader.prototype._loadArrayType = function (type, path) {
        if (path === void 0) { path = []; }
        if (!type.endsWith('[]')) {
            throw Error('Array type must ends with "[]".');
        }
        var itemType = type.slice(0, -2);
        return {
            type: 'array',
            items: this._loadType(itemType, path),
        };
    };
    MessageLoader.prototype._loadGenericType = function (type, path) {
        var _this = this;
        if (path === void 0) { path = []; }
        var matchResult = getGenericTypes(type);
        if (!matchResult) {
            throw Error("Type (".concat(type, ") is not generic type."));
        }
        var genericType = matchResult[0], subTypes = matchResult.slice(1);
        return types_1.resolveGenericType.apply(void 0, tslib_1.__spreadArray([genericType], subTypes.map(function (t) { return _this._loadType(t, path); }), false));
    };
    MessageLoader.prototype._getMessages = function (moduleName) {
        var messages = this.messageMap[moduleName];
        if (!messages) {
            messages = [];
            this.messageMap[moduleName] = messages;
        }
        return messages;
    };
    MessageLoader.prototype._getSchemaAndType = function (type) {
        var schema;
        var typeStr;
        if (typeof type === 'string') {
            typeStr = type;
            var _schema = this._schemas.find(function (item) { return item.type === type; });
            if (!_schema) {
                return { schema: null, type: type };
            }
            schema = _schema;
        }
        else {
            typeStr = getTypeFromSchema(type);
            schema = type;
        }
        return { schema: schema, type: typeStr };
    };
    MessageLoader.prototype._handleEnumType = function (ctx) {
        if (Array.isArray(ctx.schema.enums) && ctx.schema.enums.length) {
            ctx.schema = {
                type: 'object',
                properties: {
                    name: {
                        type: 'string',
                        enums: ctx.schema.enums,
                    },
                },
            };
            ctx.done = true;
        }
    };
    MessageLoader.prototype._handleRecursiveReference = function (ctx) {
        if (ctx.type && this.dataSchemaMap[ctx.type]) {
            ctx.schema = {
                $ref: "#/definitions/".concat(ctx.type),
            };
            ctx.done = true;
            return;
        }
        if (ctx.type && ctx.path.includes(ctx.type)) {
            if (isEmptySchema(ctx.schema)) {
                var cachedSchema = ctx.schemaMap.get(ctx.type);
                if (cachedSchema) {
                    ctx.schemaMap.set(ctx.type, Object.assign({}, cachedSchema));
                    Object.assign(ctx.schema, cachedSchema);
                    cachedSchema.$ref = "#/definitions/".concat(ctx.type);
                }
            }
            var newSchema = {
                items: ctx.schema.items,
                properties: ctx.schema.properties,
                type: ctx.schema.type,
            };
            this.dataSchemaMap[ctx.type] = {
                name: ctx.type,
                path: ctx.type,
                definitionType: DEFINITION_TYPE,
                content: ctx.type,
                jsonSchema: newSchema,
                folderId: 0,
            };
            ctx.schema.$ref = "#/definitions/".concat(ctx.type);
            ctx.schema = {
                $ref: ctx.schema.$ref,
            };
            ctx.done = true;
        }
    };
    MessageLoader.prototype._handleReference = function (ctx) {
        if (ctx.schema.$ref) {
            var schemaFounded = jsonpath_1.default.query({ types: this._schemas }, ctx.schema.$ref)[0];
            if (JSON_SCHEMA_TYPES.includes(schemaFounded.type)) {
                ctx.schema = schemaFounded;
            }
            else {
                ctx.schema = this._loadType(schemaFounded.type, ctx.path);
            }
            ctx.done = true;
        }
    };
    MessageLoader.prototype._handleArrayType = function (ctx) {
        var _this = this;
        if (Array.isArray(ctx.schema.items)) {
            ctx.schema.items.forEach(function (item) {
                if (typeof item === 'boolean' || item.$ref) {
                    return;
                }
                _this._schemas.push(item);
            });
        }
    };
    MessageLoader.prototype._handleGeneralSchema = function (ctx) {
        var _this = this;
        var _a;
        var propertyKeys = Object.keys((_a = ctx.schema.properties) !== null && _a !== void 0 ? _a : {});
        if (!propertyKeys.length) {
            ctx.schema = this._loadType(getTypeFromSchema(ctx.schema, 'java.lang.Object'), ctx.path);
            ctx.done = true;
            return;
        }
        ctx.schemaMap.set(ctx.type, ctx.schema);
        propertyKeys.forEach(function (property) {
            var _a;
            var propertyValue = (_a = ctx.schema.properties) === null || _a === void 0 ? void 0 : _a[property];
            if (!propertyValue || typeof propertyValue !== 'object') {
                return;
            }
            _this._schemas.push(propertyValue);
            ctx.schema.properties[property] = _this._loadTypeFromSchema(propertyValue, tslib_1.__spreadArray(tslib_1.__spreadArray([], ctx.path, true), [ctx.type], false), ctx.schemaMap);
        });
        ctx.done = true;
    };
    MessageLoader.prototype._handleResolvedState = function (ctx) {
        if (ctx.schema["".concat(TEMP_KEY_PREFIX, "resolved")] === true) {
            ctx.done = true;
            return;
        }
        ;
        ctx.schema["".concat(TEMP_KEY_PREFIX, "resolved")] = true;
        return false;
    };
    MessageLoader.prototype._applyProcessors = function (ctx, processors) {
        var _this = this;
        processors.reduce(function (ctx, processor) {
            if (ctx.done) {
                return ctx;
            }
            processor.call(_this, ctx);
            return ctx;
        }, ctx);
    };
    MessageLoader.prototype._loadTypeFromSchema = function (type, path, schemaMap) {
        if (path === void 0) { path = []; }
        if (schemaMap === void 0) { schemaMap = new Map(); }
        var _a = this._getSchemaAndType(type), schema = _a.schema, typeStr = _a.type;
        if (!schema) {
            return types_1.defaultType;
        }
        var ctx = { schema: schema, schemaMap: schemaMap, path: path, type: typeStr, done: false };
        var processors = [
            this._handleEnumType,
            this._handleRecursiveReference,
            this._handleResolvedState,
            this._handleReference,
            this._handleArrayType,
            this._handleGeneralSchema,
        ];
        this._applyProcessors(ctx, processors);
        return ctx.schema;
    };
    MessageLoader.prototype._loadType = function (type, path, schemaMap) {
        var _a;
        if (path === void 0) { path = []; }
        if (schemaMap === void 0) { schemaMap = new Map(); }
        if (type.endsWith('[]')) {
            return this._loadArrayType(type, path);
        }
        if (GENERIC_TYPE_PATTERN.test(type)) {
            return this._loadGenericType(type, path);
        }
        return (_a = types_1.baseTypes[type]) !== null && _a !== void 0 ? _a : this._loadTypeFromSchema(type, path, schemaMap);
    };
    MessageLoader.prototype._pushMessages = function (moduleName, message) {
        var messages = this._getMessages(moduleName);
        if (!messages.some(function (item) { return item.path === message.path; })) {
            messages.push(message);
        }
    };
    MessageLoader.prototype._loadParameterType = function (messageId, method, moduleName) {
        var _this = this;
        this._pushMessages(moduleName, {
            name: 'request',
            path: "/".concat(messageId.requestId, "/request"),
            definitionType: DEFINITION_TYPE,
            content: JSON.stringify(method.parameterTypes),
            jsonSchema: method.parameterTypes.map(function (item) { return _this._loadType(item); }),
            moduleId: 0,
        });
    };
    MessageLoader.prototype._loadReturnType = function (messageId, method, moduleName) {
        this._pushMessages(moduleName, {
            name: 'response',
            path: "/".concat(messageId.responseId, "/response"),
            definitionType: DEFINITION_TYPE,
            content: method.returnType,
            jsonSchema: this._loadType(method.returnType),
            moduleId: 0,
        });
    };
    MessageLoader.prototype.trimJSONSchemaProperties = function (obj) {
        var _this = this;
        if (typeof obj !== 'object' || !obj) {
            return;
        }
        if (Array.isArray(obj)) {
            obj.forEach(function (item) { return _this.trimJSONSchemaProperties(item); });
            return;
        }
        if ('$ref' in obj) {
            Object.keys(obj).filter(function (key) { return key !== '$ref'; }).forEach(function (key) {
                delete obj[key];
            });
            return;
        }
        if ('type' in obj &&
            typeof obj.type === 'string' &&
            !JSON_SCHEMA_TYPES.includes(obj.type)) {
            obj.type = 'object';
        }
        Object.keys(obj).filter(function (key) { return key.startsWith(TEMP_KEY_PREFIX) || key === 'typeBuilderName'; }).forEach(function (key) {
            delete obj[key];
        });
        if ('items' in obj && Array.isArray(obj.items) && !obj.items.length) {
            delete obj.items;
        }
        if ('enums' in obj) {
            if (Array.isArray(obj.enums) && obj.enums.length) {
                (obj.enum = obj.enums);
            }
            delete obj.enums;
        }
        Object.keys(obj).forEach(function (key) {
            _this.trimJSONSchemaProperties(obj[key]);
        });
    };
    MessageLoader.prototype.load = function (messageId, method, moduleName) {
        this._loadReturnType(messageId, method, moduleName);
        this._loadParameterType(messageId, method, moduleName);
    };
    MessageLoader.prototype.getDataSchemas = function () {
        return Object.values(this.dataSchemaMap);
    };
    MessageLoader.prototype.getMessages = function (moduleName) {
        var _a;
        return Object.values((_a = this.messageMap[moduleName]) !== null && _a !== void 0 ? _a : {});
    };
    MessageLoader.prototype.javaTypeToJsonSchema = function (javaType) {
        return this._loadType(javaType);
    };
    return MessageLoader;
}());
exports.MessageLoader = MessageLoader;
