"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DubboMetadataParser = void 0;
var tslib_1 = require("tslib");
var md5_1 = tslib_1.__importDefault(require("md5"));
var dubbo_1 = require("../message-loader/dubbo");
var types_1 = require("./types");
function isStringArray(arr) {
    return arr.every(function (item) { return typeof item === 'string'; });
}
function splitName(serviceName) {
    var i = serviceName.lastIndexOf('.');
    return {
        moduleName: serviceName.slice(0, i),
        serviceName: serviceName.slice(i + 1),
    };
}
function isProvider(metadata) {
    var _a;
    return ((_a = metadata.parameters) === null || _a === void 0 ? void 0 : _a.side) === 'provider';
}
var DubboMetadataParser = /** @class */ (function () {
    function DubboMetadataParser(_name) {
        this._name = _name;
        this._rpcModuleType = 'DUBBO';
        this._dataSchemas = [];
        this._rpcModuleMap = new Map();
        this._messageLoader = new dubbo_1.MessageLoader();
    }
    DubboMetadataParser.getAppInfoList = function (metadataList) {
        if (!isStringArray(metadataList)) {
            return metadataList.map(function (item) { return ({
                appId: item.AppId,
                appName: item.Name,
            }); });
        }
        var result = [];
        var _loop_1 = function (metadataStr) {
            try {
                var metadata_1 = JSON.parse(metadataStr);
                if (!isProvider(metadata_1)) {
                    return "continue";
                }
                if (!result.some(function (item) { return item.appName === metadata_1.parameters.application; })) {
                    result.push({
                        appId: metadata_1.parameters.application,
                        appName: metadata_1.parameters.application,
                    });
                }
            }
            catch (error) {
                console.log('error: ', error);
            }
        };
        for (var _i = 0, metadataList_1 = metadataList; _i < metadataList_1.length; _i++) {
            var metadataStr = metadataList_1[_i];
            _loop_1(metadataStr);
        }
        return result;
    };
    DubboMetadataParser.prototype._getRPCModule = function (moduleName) {
        var rpcModule = this._rpcModuleMap.get(moduleName);
        if (rpcModule) {
            return rpcModule;
        }
        var newRPCModule = {
            name: moduleName,
            type: this._rpcModuleType,
            dataSourceId: 0,
            rpcMessages: [],
            rpcServices: [],
        };
        this._rpcModuleMap.set(moduleName, newRPCModule);
        return newRPCModule;
    };
    DubboMetadataParser.prototype._loadMethod = function (ctx, method, servicePath) {
        var methodPath = "".concat(servicePath, "/").concat(method.name);
        var requestPathId = (0, md5_1.default)(JSON.stringify(method.parameterTypes).replace(/\s*/g, ''));
        var responsePathId = (0, md5_1.default)(JSON.stringify(method.returnType).replace(/\s*/g, ''));
        var messageId = {
            requestId: "".concat(methodPath, "/").concat(requestPathId),
            responseId: "".concat(methodPath, "/").concat(responsePathId),
        };
        var m = {
            name: method.name,
            kind: types_1.MethodKind.UNARY,
            path: "/".concat(methodPath),
            commonParameters: [],
            requestBody: {
                messages: method.parameterTypes.map(function (item) { return ({
                    type: item,
                    message: '',
                }); }),
            },
            requestPath: "/".concat(messageId.requestId, "/request"),
            responsePath: "/".concat(messageId.responseId, "/response"),
            requestId: 0,
            responseId: 0,
            moduleId: 0,
            serviceId: 0,
            serverUrl: '',
        };
        ctx.methodIdMap.set(method, messageId);
        return m;
    };
    DubboMetadataParser.prototype._getServiceName = function (fullName) {
        var i = fullName.lastIndexOf('.');
        return fullName.slice(i + 1);
    };
    DubboMetadataParser.prototype._loadMessagesOfMethod = function (ctx, method, moduleName) {
        var id = ctx.methodIdMap.get(method);
        if (!id) {
            throw Error('Method id not found');
        }
        ctx.messageLoader.load(id, method, moduleName);
    };
    DubboMetadataParser.prototype._loadMessagesOfService = function (ctx, methods, moduleName) {
        for (var _i = 0, methods_1 = methods; _i < methods_1.length; _i++) {
            var method = methods_1[_i];
            this._loadMessagesOfMethod(ctx, method, moduleName);
        }
    };
    DubboMetadataParser.prototype._loadService = function (ctx, service, methods) {
        var _this = this;
        var _a, _b;
        var serviceCache = ctx.serviceMap.get(service.name);
        if (!serviceCache) {
            var rpcMethods = methods.map(function (method) { return _this._loadMethod(ctx, method, service.name); });
            serviceCache = {
                moduleId: 0,
                name: this._getServiceName(service.name),
                additional: {
                    dubboMetadata: [],
                },
                rpcMethods: rpcMethods,
            };
            ctx.serviceMap.set(service.name, serviceCache);
            this._loadMessagesOfService(ctx, methods, ctx.moduleName);
        }
        if (service.group || service.version) {
            serviceCache.additional.dubboMetadata.push({
                group: (_a = service.group) !== null && _a !== void 0 ? _a : '',
                version: (_b = service.version) !== null && _b !== void 0 ? _b : '',
            });
        }
        return serviceCache;
    };
    DubboMetadataParser.prototype._parseEdasMetadata = function (ctx, metadataList) {
        for (var _i = 0, metadataList_2 = metadataList; _i < metadataList_2.length; _i++) {
            var metadata = metadataList_2[_i];
            try {
                var moduleName = splitName(metadata.service.ServiceName).moduleName;
                ctx.moduleName = moduleName;
                var rpcModule = this._getRPCModule(moduleName);
                var methods = metadata.methods.map(function (item) {
                    var parameterTypes = [];
                    try {
                        parameterTypes = JSON.parse(item.ParameterTypes);
                    }
                    catch (err) {
                        // ignore
                    }
                    return {
                        name: item.Name,
                        parameterTypes: parameterTypes,
                        returnType: item.ReturnType,
                    };
                });
                var service = {
                    name: metadata.service.ServiceName,
                    version: metadata.service.Version,
                    group: metadata.service.Group,
                };
                var rpcService = this._loadService(ctx, service, methods);
                if (!rpcModule.rpcServices.includes(rpcService)) {
                    rpcModule.rpcServices.push(rpcService);
                }
            }
            catch (error) {
                console.log('error: ', error);
                continue;
            }
        }
        this._dataSchemas = ctx.messageLoader.getDataSchemas();
    };
    DubboMetadataParser.prototype._parseMetadataString = function (ctx, metadataList) {
        for (var _i = 0, metadataList_3 = metadataList; _i < metadataList_3.length; _i++) {
            var metadataStr = metadataList_3[_i];
            try {
                var metadata = JSON.parse(metadataStr);
                if (!isProvider(metadata)) {
                    continue;
                }
                if (metadata.parameters.application !== this._name) {
                    continue;
                }
                var moduleName = splitName(metadata.canonicalName).moduleName;
                ctx.moduleName = moduleName;
                ctx.messageLoader.schemas = metadata.types;
                var rpcModule = this._getRPCModule(moduleName);
                var methods = metadata.methods;
                var service = {
                    name: metadata.canonicalName,
                    version: metadata.parameters.version,
                    group: metadata.parameters.group,
                };
                var rpcService = this._loadService(ctx, service, methods);
                if (!rpcModule.rpcServices.includes(rpcService)) {
                    rpcModule.rpcServices.push(rpcService);
                }
            }
            catch (error) {
                console.log('error: ', error);
                continue;
            }
        }
        this._dataSchemas = ctx.messageLoader.getDataSchemas();
    };
    DubboMetadataParser.prototype.parse = function (metadataList) {
        var ctx = {
            methodIdMap: new Map(),
            serviceMap: new Map(),
            messageLoader: this._messageLoader,
            moduleName: '',
        };
        if (isStringArray(metadataList)) {
            this._parseMetadataString(ctx, metadataList);
        }
        else {
            this._parseEdasMetadata(ctx, metadataList);
        }
        return {
            dataSchemas: this._dataSchemas,
            rpcModuleMap: this._rpcModuleMap,
            messageLoader: this._messageLoader,
        };
    };
    return DubboMetadataParser;
}());
exports.DubboMetadataParser = DubboMetadataParser;
