"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AeadAes256CbcHmac256EncryptionKey = exports.generateKeySalt = exports.deriveKey = exports.keySize = void 0;

var _crypto = require("crypto");

var _symmetricKey = _interopRequireDefault(require("./symmetric-key"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const keySize = 256;
exports.keySize = keySize;
const keySizeInBytes = keySize / 8;

const deriveKey = (rootKey, salt) => {
  const hmac = (0, _crypto.createHmac)('sha256', rootKey);
  hmac.update(Buffer.from(salt, 'utf16le'));
  return hmac.digest();
};

exports.deriveKey = deriveKey;

const generateKeySalt = (keyType, algorithmName, keySize) => `Microsoft SQL Server cell ${keyType} key ` + `with encryption algorithm:${algorithmName} and key length:${keySize}`;

exports.generateKeySalt = generateKeySalt;

class AeadAes256CbcHmac256EncryptionKey extends _symmetricKey.default {
  constructor(rootKey, algorithmName) {
    super(rootKey);
    this.algorithmName = void 0;
    this.encryptionKeySaltFormat = void 0;
    this.macKeySaltFormat = void 0;
    this.ivKeySaltFormat = void 0;
    this.encryptionKey = void 0;
    this.macKey = void 0;
    this.ivKey = void 0;
    this.algorithmName = algorithmName;
    this.encryptionKeySaltFormat = generateKeySalt('encryption', this.algorithmName, keySize);
    this.macKeySaltFormat = generateKeySalt('MAC', this.algorithmName, keySize);
    this.ivKeySaltFormat = generateKeySalt('IV', this.algorithmName, keySize);

    if (rootKey.length !== keySizeInBytes) {
      throw new Error(`The column encryption key has been successfully decrypted but it's length: ${rootKey.length} does not match the length: ${keySizeInBytes} for algorithm "${this.algorithmName}". Verify the encrypted value of the column encryption key in the database.`);
    }

    try {
      const encKeyBuff = deriveKey(rootKey, this.encryptionKeySaltFormat);
      this.encryptionKey = new _symmetricKey.default(encKeyBuff);
      const macKeyBuff = deriveKey(rootKey, this.macKeySaltFormat);
      this.macKey = new _symmetricKey.default(macKeyBuff);
      const ivKeyBuff = deriveKey(rootKey, this.ivKeySaltFormat);
      this.ivKey = new _symmetricKey.default(ivKeyBuff);
    } catch (error) {
      throw new Error(`Key extraction failed : ${error.message}.`);
    }
  }

  getEncryptionKey() {
    return this.encryptionKey.rootKey;
  }

  getMacKey() {
    return this.macKey.rootKey;
  }

  getIvKey() {
    return this.ivKey.rootKey;
  }

}

exports.AeadAes256CbcHmac256EncryptionKey = AeadAes256CbcHmac256EncryptionKey;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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