"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getParameterEncryptionMetadata = void 0;

var _types = require("./types");

var _cekEntry = require("./cek-entry");

var _keyCrypto = require("./key-crypto");

var _dataType = require("../data-type");

var _request = _interopRequireDefault(require("../request"));

var _rpcrequestPayload = _interopRequireDefault(require("../rpcrequest-payload"));

var _packet = require("../packet");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const getParameterEncryptionMetadata = (connection, request, callback) => {
  if (request.cryptoMetadataLoaded === true) {
    return callback();
  }

  const metadataRequest = new _request.default('sp_describe_parameter_encryption', error => {
    if (error) {
      return callback(error);
    }

    const decryptSymmetricKeyPromises = [];
    const cekList = [];
    let paramCount = 0;

    for (const columns of resultRows) {
      try {
        const isFirstRecordSet = columns.some(col => (col && col.metadata && col.metadata.colName) === 'database_id');

        if (isFirstRecordSet === true) {
          const currentOrdinal = columns[_types.DescribeParameterEncryptionResultSet1.KeyOrdinal].value;
          let cekEntry;

          if (!cekList[currentOrdinal]) {
            cekEntry = new _cekEntry.CEKEntry(currentOrdinal);
            cekList[cekEntry.ordinal] = cekEntry;
          } else {
            cekEntry = cekList[currentOrdinal];
          }

          cekEntry.add(columns[_types.DescribeParameterEncryptionResultSet1.EncryptedKey].value, columns[_types.DescribeParameterEncryptionResultSet1.DbId].value, columns[_types.DescribeParameterEncryptionResultSet1.KeyId].value, columns[_types.DescribeParameterEncryptionResultSet1.KeyVersion].value, columns[_types.DescribeParameterEncryptionResultSet1.KeyMdVersion].value, columns[_types.DescribeParameterEncryptionResultSet1.KeyPath].value, columns[_types.DescribeParameterEncryptionResultSet1.ProviderName].value, columns[_types.DescribeParameterEncryptionResultSet1.KeyEncryptionAlgorithm].value);
        } else {
          paramCount++;
          const paramName = columns[_types.DescribeParameterEncryptionResultSet2.ParameterName].value;
          const paramIndex = request.parameters.findIndex(param => paramName === `@${param.name}`);
          const cekOrdinal = columns[_types.DescribeParameterEncryptionResultSet2.ColumnEncryptionKeyOrdinal].value;
          const cekEntry = cekList[cekOrdinal];

          if (cekEntry && cekList.length < cekOrdinal) {
            return callback(new Error(`Internal error. The referenced column encryption key ordinal "${cekOrdinal}" is missing in the encryption metadata returned by sp_describe_parameter_encryption. Max ordinal is "${cekList.length}".`));
          }

          const encType = columns[_types.DescribeParameterEncryptionResultSet2.ColumnEncrytionType].value;

          if (_types.SQLServerEncryptionType.PlainText !== encType) {
            request.parameters[paramIndex].cryptoMetadata = {
              cekEntry: cekEntry,
              ordinal: cekOrdinal,
              cipherAlgorithmId: columns[_types.DescribeParameterEncryptionResultSet2.ColumnEncryptionAlgorithm].value,
              encryptionType: encType,
              normalizationRuleVersion: Buffer.from([columns[_types.DescribeParameterEncryptionResultSet2.NormalizationRuleVersion].value])
            };
            decryptSymmetricKeyPromises.push((0, _keyCrypto.decryptSymmetricKey)(request.parameters[paramIndex].cryptoMetadata, connection.config.options));
          } else if (request.parameters[paramIndex].forceEncrypt === true) {
            return callback(new Error(`Cannot execute statement or procedure ${request.sqlTextOrProcedure} because Force Encryption was set as true for parameter ${paramIndex + 1} and the database expects this parameter to be sent as plaintext. This may be due to a configuration error.`));
          }
        }
      } catch {
        return callback(new Error(`Internal error. Unable to parse parameter encryption metadata in statement or procedure "${request.sqlTextOrProcedure}"`));
      }
    }

    if (paramCount !== request.parameters.length) {
      return callback(new Error(`Internal error. Metadata for some parameters in statement or procedure "${request.sqlTextOrProcedure}" is missing in the resultset returned by sp_describe_parameter_encryption.`));
    }

    return Promise.all(decryptSymmetricKeyPromises).then(() => {
      request.cryptoMetadataLoaded = true;
      process.nextTick(callback);
    }, error => {
      process.nextTick(callback, error);
    });
  });
  metadataRequest.addParameter('tsql', _dataType.typeByName.NVarChar, request.sqlTextOrProcedure);

  if (request.parameters.length) {
    metadataRequest.addParameter('params', _dataType.typeByName.NVarChar, metadataRequest.makeParamsParameter(request.parameters));
  }

  const resultRows = [];
  metadataRequest.on('row', columns => {
    resultRows.push(columns);
  });
  connection.makeRequest(metadataRequest, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default(metadataRequest.sqlTextOrProcedure, metadataRequest.parameters, connection.currentTransactionDescriptor(), connection.config.options, connection.databaseCollation));
};

exports.getParameterEncryptionMetadata = getParameterEncryptionMetadata;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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