"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.decryptSymmetricKey = exports.decryptWithKey = exports.encryptWithKey = exports.validateAndGetEncryptionAlgorithmName = void 0;

var _symmetricKeyCache = require("./symmetric-key-cache");

var _aeadAes256CbcHmacAlgorithm = require("./aead-aes-256-cbc-hmac-algorithm");

var _aeadAes256CbcHmacEncryptionKey = require("./aead-aes-256-cbc-hmac-encryption-key");

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const validateAndGetEncryptionAlgorithmName = (cipherAlgorithmId, cipherAlgorithmName) => {
  if (cipherAlgorithmId !== 2) {
    throw new Error('Custom cipher algorithm not supported.');
  }

  return _aeadAes256CbcHmacAlgorithm.algorithmName;
};

exports.validateAndGetEncryptionAlgorithmName = validateAndGetEncryptionAlgorithmName;

const encryptWithKey = async (plaintext, md, options) => {
  if (!options.trustedServerNameAE) {
    throw new Error('Server name should not be null in EncryptWithKey');
  }

  if (!md.cipherAlgorithm) {
    await decryptSymmetricKey(md, options);
  }

  if (!md.cipherAlgorithm) {
    throw new Error('Cipher Algorithm should not be null in EncryptWithKey');
  }

  const cipherText = md.cipherAlgorithm.encryptData(plaintext);

  if (!cipherText) {
    throw new Error('Internal error. Ciphertext value cannot be null.');
  }

  return cipherText;
};

exports.encryptWithKey = encryptWithKey;

const decryptWithKey = (cipherText, md, options) => {
  if (!options.trustedServerNameAE) {
    throw new Error('Server name should not be null in DecryptWithKey');
  } // if (!md.cipherAlgorithm) {
  //   await decryptSymmetricKey(md, options);
  // }


  if (!md.cipherAlgorithm) {
    throw new Error('Cipher Algorithm should not be null in DecryptWithKey');
  }

  const plainText = md.cipherAlgorithm.decryptData(cipherText);

  if (!plainText) {
    throw new Error('Internal error. Plaintext value cannot be null.');
  }

  return plainText;
};

exports.decryptWithKey = decryptWithKey;

const decryptSymmetricKey = async (md, options) => {
  if (!md) {
    throw new Error('md should not be null in DecryptSymmetricKey.');
  }

  if (!md.cekEntry) {
    throw new Error('md.EncryptionInfo should not be null in DecryptSymmetricKey.');
  }

  if (!md.cekEntry.columnEncryptionKeyValues) {
    throw new Error('md.EncryptionInfo.ColumnEncryptionKeyValues should not be null in DecryptSymmetricKey.');
  }

  let symKey;
  let encryptionKeyInfoChosen;
  const CEKValues = md.cekEntry.columnEncryptionKeyValues;
  let lastError;

  for (const CEKValue of CEKValues) {
    try {
      symKey = await (0, _symmetricKeyCache.getKey)(CEKValue, options);

      if (symKey) {
        encryptionKeyInfoChosen = CEKValue;
        break;
      }
    } catch (error) {
      lastError = error;
    }
  }

  if (!symKey) {
    if (lastError) {
      throw lastError;
    } else {
      throw new Error('Exception while decryption of encrypted column encryption key.');
    }
  }

  const algorithmName = validateAndGetEncryptionAlgorithmName(md.cipherAlgorithmId, md.cipherAlgorithmName);
  const cipherAlgorithm = new _aeadAes256CbcHmacAlgorithm.AeadAes256CbcHmac256Algorithm(new _aeadAes256CbcHmacEncryptionKey.AeadAes256CbcHmac256EncryptionKey(symKey.rootKey, algorithmName), md.encryptionType);
  md.cipherAlgorithm = cipherAlgorithm;
  md.encryptionKeyInfo = encryptionKeyInfoChosen;
};

exports.decryptSymmetricKey = decryptSymmetricKey;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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