"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ColumnEncryptionAzureKeyVaultProvider = void 0;

var _identity = require("@azure/identity");

var _keyvaultKeys = require("@azure/keyvault-keys");

var _crypto = require("crypto");

var _url = require("url");

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
class ColumnEncryptionAzureKeyVaultProvider {
  constructor(clientId, clientKey, tenantId) {
    this.name = void 0;
    this.url = void 0;
    this.rsaEncryptionAlgorithmWithOAEPForAKV = void 0;
    this.firstVersion = void 0;
    this.credentials = void 0;
    this.azureKeyVaultDomainName = void 0;
    this.keyClient = void 0;
    this.name = 'AZURE_KEY_VAULT';
    this.azureKeyVaultDomainName = 'vault.azure.net';
    this.rsaEncryptionAlgorithmWithOAEPForAKV = 'RSA-OAEP';
    this.firstVersion = Buffer.from([0x01]);
    this.credentials = new _identity.ClientSecretCredential(tenantId, clientId, clientKey);
  }

  async decryptColumnEncryptionKey(masterKeyPath, encryptionAlgorithm, encryptedColumnEncryptionKey) {
    if (!encryptedColumnEncryptionKey) {
      throw new Error('Internal error. Encrypted column encryption key cannot be null.');
    }

    if (encryptedColumnEncryptionKey.length === 0) {
      throw new Error('Internal error. Empty encrypted column encryption key specified.');
    }

    encryptionAlgorithm = this.validateEncryptionAlgorithm(encryptionAlgorithm);
    const masterKey = await this.getMasterKey(masterKeyPath);
    const keySizeInBytes = this.getAKVKeySize(masterKey);
    const cryptoClient = this.createCryptoClient(masterKey);

    if (encryptedColumnEncryptionKey[0] !== this.firstVersion[0]) {
      throw new Error(`Specified encrypted column encryption key contains an invalid encryption algorithm version ${Buffer.from([encryptedColumnEncryptionKey[0]]).toString('hex')}. Expected version is ${Buffer.from([this.firstVersion[0]]).toString('hex')}.`);
    }

    let currentIndex = this.firstVersion.length;
    const keyPathLength = encryptedColumnEncryptionKey.readInt16LE(currentIndex);
    currentIndex += 2;
    const cipherTextLength = encryptedColumnEncryptionKey.readInt16LE(currentIndex);
    currentIndex += 2;
    currentIndex += keyPathLength;

    if (cipherTextLength !== keySizeInBytes) {
      throw new Error(`The specified encrypted column encryption key's ciphertext length: ${cipherTextLength} does not match the ciphertext length: ${keySizeInBytes} when using column master key (Azure Key Vault key) in ${masterKeyPath}. The encrypted column encryption key may be corrupt, or the specified Azure Key Vault key path may be incorrect.`);
    }

    const signatureLength = encryptedColumnEncryptionKey.length - currentIndex - cipherTextLength;

    if (signatureLength !== keySizeInBytes) {
      throw new Error(`The specified encrypted column encryption key's signature length: ${signatureLength} does not match the signature length: ${keySizeInBytes} when using column master key (Azure Key Vault key) in ${masterKeyPath}. The encrypted column encryption key may be corrupt, or the specified Azure Key Vault key path may be incorrect.`);
    }

    const cipherText = Buffer.alloc(cipherTextLength);
    encryptedColumnEncryptionKey.copy(cipherText, 0, currentIndex, currentIndex + cipherTextLength);
    currentIndex += cipherTextLength;
    const signature = Buffer.alloc(signatureLength);
    encryptedColumnEncryptionKey.copy(signature, 0, currentIndex, currentIndex + signatureLength);
    const hash = Buffer.alloc(encryptedColumnEncryptionKey.length - signature.length);
    encryptedColumnEncryptionKey.copy(hash, 0, 0, encryptedColumnEncryptionKey.length - signature.length);
    const messageDigest = (0, _crypto.createHash)('sha256');
    messageDigest.update(hash);
    const dataToVerify = messageDigest.digest();

    if (!dataToVerify) {
      throw new Error('Hash should not be null while decrypting encrypted column encryption key.');
    }

    const verifyKey = await cryptoClient.verify('RS256', dataToVerify, signature);

    if (!verifyKey.result) {
      throw new Error(`The specified encrypted column encryption key signature does not match the signature computed with the column master key (Asymmetric key in Azure Key Vault) in ${masterKeyPath}. The encrypted column encryption key may be corrupt, or the specified path may be incorrect.`);
    }

    const decryptedCEK = await this.azureKeyVaultUnWrap(cryptoClient, encryptionAlgorithm, cipherText);
    return decryptedCEK;
  }

  async encryptColumnEncryptionKey(masterKeyPath, encryptionAlgorithm, columnEncryptionKey) {
    if (!columnEncryptionKey) {
      throw new Error('Column encryption key cannot be null.');
    }

    if (columnEncryptionKey.length === 0) {
      throw new Error('Empty column encryption key specified.');
    }

    encryptionAlgorithm = this.validateEncryptionAlgorithm(encryptionAlgorithm);
    const masterKey = await this.getMasterKey(masterKeyPath);
    const keySizeInBytes = this.getAKVKeySize(masterKey);
    const cryptoClient = this.createCryptoClient(masterKey);
    const version = Buffer.from([this.firstVersion[0]]);
    const masterKeyPathBytes = Buffer.from(masterKeyPath.toLowerCase(), 'utf8');
    const keyPathLength = Buffer.alloc(2);
    keyPathLength[0] = masterKeyPathBytes.length & 0xff;
    keyPathLength[1] = masterKeyPathBytes.length >> 8 & 0xff;
    const cipherText = await this.azureKeyVaultWrap(cryptoClient, encryptionAlgorithm, columnEncryptionKey);
    const cipherTextLength = Buffer.alloc(2);
    cipherTextLength[0] = cipherText.length & 0xff;
    cipherTextLength[1] = cipherText.length >> 8 & 0xff;

    if (cipherText.length !== keySizeInBytes) {
      throw new Error('CipherText length does not match the RSA key size.');
    }

    const dataToHash = Buffer.alloc(version.length + keyPathLength.length + cipherTextLength.length + masterKeyPathBytes.length + cipherText.length);
    let destinationPosition = version.length;
    version.copy(dataToHash, 0, 0, version.length);
    keyPathLength.copy(dataToHash, destinationPosition, 0, keyPathLength.length);
    destinationPosition += keyPathLength.length;
    cipherTextLength.copy(dataToHash, destinationPosition, 0, cipherTextLength.length);
    destinationPosition += cipherTextLength.length;
    masterKeyPathBytes.copy(dataToHash, destinationPosition, 0, masterKeyPathBytes.length);
    destinationPosition += masterKeyPathBytes.length;
    cipherText.copy(dataToHash, destinationPosition, 0, cipherText.length);
    const messageDigest = (0, _crypto.createHash)('sha256');
    messageDigest.update(dataToHash);
    const dataToSign = messageDigest.digest();
    const signedHash = await this.azureKeyVaultSignedHashedData(cryptoClient, dataToSign);

    if (signedHash.length !== keySizeInBytes) {
      throw new Error('Signed hash length does not match the RSA key size.');
    }

    const verifyKey = await cryptoClient.verify('RS256', dataToSign, signedHash);

    if (!verifyKey.result) {
      throw new Error('Invalid signature of the encrypted column encryption key computed.');
    }

    const encryptedColumnEncryptionKeyLength = version.length + cipherTextLength.length + keyPathLength.length + cipherText.length + masterKeyPathBytes.length + signedHash.length;
    const encryptedColumnEncryptionKey = Buffer.alloc(encryptedColumnEncryptionKeyLength);
    let currentIndex = 0;
    version.copy(encryptedColumnEncryptionKey, currentIndex, 0, version.length);
    currentIndex += version.length;
    keyPathLength.copy(encryptedColumnEncryptionKey, currentIndex, 0, keyPathLength.length);
    currentIndex += keyPathLength.length;
    cipherTextLength.copy(encryptedColumnEncryptionKey, currentIndex, 0, cipherTextLength.length);
    currentIndex += cipherTextLength.length;
    masterKeyPathBytes.copy(encryptedColumnEncryptionKey, currentIndex, 0, masterKeyPathBytes.length);
    currentIndex += masterKeyPathBytes.length;
    cipherText.copy(encryptedColumnEncryptionKey, currentIndex, 0, cipherText.length);
    currentIndex += cipherText.length;
    signedHash.copy(encryptedColumnEncryptionKey, currentIndex, 0, signedHash.length);
    return encryptedColumnEncryptionKey;
  }

  async getMasterKey(masterKeyPath) {
    if (!masterKeyPath) {
      throw new Error('Master key path cannot be null or undefined');
    }

    const keyParts = this.parsePath(masterKeyPath);
    this.createKeyClient(keyParts.vaultUrl);
    return this.keyClient.getKey(keyParts.name, keyParts.version ? {
      version: keyParts.version
    } : {});
  }

  createKeyClient(keyVaultUrl) {
    if (!keyVaultUrl) {
      throw new Error('Cannot create key client with null or undefined keyVaultUrl');
    }

    if (!this.keyClient) {
      this.url = keyVaultUrl;
      this.keyClient = new _keyvaultKeys.KeyClient(keyVaultUrl, this.credentials);
    }
  }

  createCryptoClient(masterKey) {
    if (!masterKey) {
      throw new Error('Cannot create CryptographyClient with null or undefined masterKey');
    }

    return new _keyvaultKeys.CryptographyClient(masterKey, this.credentials);
  }

  parsePath(masterKeyPath) {
    if (!masterKeyPath || masterKeyPath.trim() === '') {
      throw new Error('Azure Key Vault key path cannot be null.');
    }

    let baseUri;

    try {
      baseUri = (0, _url.parse)(masterKeyPath, true, true);
    } catch {
      throw new Error(`Invalid keys identifier: ${masterKeyPath}. Not a valid URI`);
    }

    if (!baseUri.hostname || !baseUri.hostname.toLowerCase().endsWith(this.azureKeyVaultDomainName)) {
      throw new Error(`Invalid Azure Key Vault key path specified: ${masterKeyPath}.`);
    } // Path is of the form '/collection/name[/version]'


    const segments = (baseUri.pathname || '').split('/');

    if (segments.length !== 3 && segments.length !== 4) {
      throw new Error(`Invalid keys identifier: ${masterKeyPath}. Bad number of segments: ${segments.length}`);
    }

    if ('keys' !== segments[1]) {
      throw new Error(`Invalid keys identifier: ${masterKeyPath}. segment [1] should be "keys", found "${segments[1]}"`);
    }

    const vaultUrl = `${baseUri.protocol}//${baseUri.host}`;
    const name = segments[2];
    const version = segments.length === 4 ? segments[3] : undefined;
    return {
      vaultUrl,
      name,
      version
    };
  }

  async azureKeyVaultSignedHashedData(cryptoClient, dataToSign) {
    if (!cryptoClient) {
      throw new Error('Azure KVS Crypto Client is not defined.');
    }

    const signedData = await cryptoClient.sign('RS256', dataToSign);
    return Buffer.from(signedData.result);
  }

  async azureKeyVaultWrap(cryptoClient, encryptionAlgorithm, columnEncryptionKey) {
    if (!cryptoClient) {
      throw new Error('Azure KVS Crypto Client is not defined.');
    }

    if (!columnEncryptionKey) {
      throw new Error('Column encryption key cannot be null.');
    }

    const wrappedKey = await cryptoClient.wrapKey(encryptionAlgorithm, columnEncryptionKey);
    return Buffer.from(wrappedKey.result);
  }

  async azureKeyVaultUnWrap(cryptoClient, encryptionAlgorithm, encryptedColumnEncryptionKey) {
    if (!cryptoClient) {
      throw new Error('Azure KVS Crypto Client is not defined.');
    }

    if (!encryptionAlgorithm) {
      throw new Error('Encryption Algorithm cannot be null or undefined');
    }

    if (!encryptedColumnEncryptionKey) {
      throw new Error('Encrypted column encryption key cannot be null.');
    }

    if (encryptedColumnEncryptionKey.length === 0) {
      throw new Error('Encrypted Column Encryption Key length should not be zero.');
    }

    const unwrappedKey = await cryptoClient.unwrapKey(encryptionAlgorithm, encryptedColumnEncryptionKey);
    return Buffer.from(unwrappedKey.result);
  }

  getAKVKeySize(retrievedKey) {
    if (!retrievedKey) {
      throw new Error('Retrieved key cannot be null or undefined');
    }

    const key = retrievedKey.key;

    if (!key) {
      throw new Error(`Key does not exist ${retrievedKey.name}`);
    }

    const kty = key && key.kty && key.kty.toString().toUpperCase();

    if (!kty || 'RSA'.localeCompare(kty, 'en') !== 0) {
      throw new Error(`Cannot use a non-RSA key: ${kty}.`);
    }

    const keyLength = key && key.n && key.n.length;
    return keyLength || 0;
  }

  validateEncryptionAlgorithm(encryptionAlgorithm) {
    if (!encryptionAlgorithm) {
      throw new Error('Key encryption algorithm cannot be null.');
    }

    if ('RSA_OAEP'.localeCompare(encryptionAlgorithm.toUpperCase(), 'en') === 0) {
      encryptionAlgorithm = 'RSA-OAEP';
    }

    if (this.rsaEncryptionAlgorithmWithOAEPForAKV.localeCompare(encryptionAlgorithm.trim().toUpperCase(), 'en') !== 0) {
      throw new Error(`Invalid key encryption algorithm specified: ${encryptionAlgorithm}. Expected value: ${this.rsaEncryptionAlgorithmWithOAEPForAKV}.`);
    }

    return encryptionAlgorithm;
  }

}

exports.ColumnEncryptionAzureKeyVaultProvider = ColumnEncryptionAzureKeyVaultProvider;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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