"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _events = require("events");

var _writableTrackingBuffer = _interopRequireDefault(require("./tracking-buffer/writable-tracking-buffer"));

var _stream = require("stream");

var _token = require("./token/token");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * @private
 */
const FLAGS = {
  nullable: 1 << 0,
  caseSen: 1 << 1,
  updateableReadWrite: 1 << 2,
  updateableUnknown: 1 << 3,
  identity: 1 << 4,
  computed: 1 << 5,
  // introduced in TDS 7.2
  fixedLenCLRType: 1 << 8,
  // introduced in TDS 7.2
  sparseColumnSet: 1 << 10,
  // introduced in TDS 7.3.B
  hidden: 1 << 13,
  // introduced in TDS 7.2
  key: 1 << 14,
  // introduced in TDS 7.2
  nullableUnknown: 1 << 15 // introduced in TDS 7.2

};
/**
 * @private
 */

const DONE_STATUS = {
  FINAL: 0x00,
  MORE: 0x1,
  ERROR: 0x2,
  INXACT: 0x4,
  COUNT: 0x10,
  ATTN: 0x20,
  SRVERROR: 0x100
};
/**
 * @private
 */

const rowTokenBuffer = Buffer.from([_token.TYPE.ROW]);
const textPointerAndTimestampBuffer = Buffer.from([// TextPointer length
0x10, // TextPointer
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // Timestamp
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00]);
const textPointerNullBuffer = Buffer.from([0x00]); // A transform that converts rows to packets.

class RowTransform extends _stream.Transform {
  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */
  constructor(bulkLoad) {
    super({
      writableObjectMode: true
    });
    this.columnMetadataWritten = void 0;
    this.bulkLoad = void 0;
    this.mainOptions = void 0;
    this.columns = void 0;
    this.bulkLoad = bulkLoad;
    this.mainOptions = bulkLoad.options;
    this.columns = bulkLoad.columns;
    this.columnMetadataWritten = false;
  }
  /**
   * @private
   */


  _transform(row, _encoding, callback) {
    if (!this.columnMetadataWritten) {
      this.push(this.bulkLoad.getColMetaData());
      this.columnMetadataWritten = true;
    }

    this.push(rowTokenBuffer);

    for (let i = 0; i < this.columns.length; i++) {
      const c = this.columns[i];
      let value = Array.isArray(row) ? row[i] : row[c.objName];

      if (!this.bulkLoad.firstRowWritten) {
        try {
          value = c.type.validate(value, c.collation);
        } catch (error) {
          return callback(error);
        }
      }

      const parameter = {
        length: c.length,
        scale: c.scale,
        precision: c.precision,
        value: value
      };

      if (c.type.name === 'Text' || c.type.name === 'Image' || c.type.name === 'NText') {
        if (value == null) {
          this.push(textPointerNullBuffer);
          continue;
        }

        this.push(textPointerAndTimestampBuffer);
      }

      this.push(c.type.generateParameterLength(parameter, this.mainOptions));

      for (const chunk of c.type.generateParameterData(parameter, this.mainOptions)) {
        this.push(chunk);
      }
    }

    process.nextTick(callback);
  }
  /**
   * @private
   */


  _flush(callback) {
    this.push(this.bulkLoad.createDoneToken());
    process.nextTick(callback);
  }

}
/**
 * A BulkLoad instance is used to perform a bulk insert.
 *
 * Use [[Connection.newBulkLoad]] to create a new instance, and [[Connection.execBulkLoad]] to execute it.
 *
 * Example of BulkLoad Usages:
 *
 * ```js
 * // optional BulkLoad options
 * const options = { keepNulls: true };
 *
 * // instantiate - provide the table where you'll be inserting to, options and a callback
 * const bulkLoad = connection.newBulkLoad('MyTable', options, (error, rowCount) => {
 *   console.log('inserted %d rows', rowCount);
 * });
 *
 * // setup your columns - always indicate whether the column is nullable
 * bulkLoad.addColumn('myInt', TYPES.Int, { nullable: false });
 * bulkLoad.addColumn('myString', TYPES.NVarChar, { length: 50, nullable: true });
 *
 * // execute
 * connection.execBulkLoad(bulkLoad, [
 *   { myInt: 7, myString: 'hello' },
 *   { myInt: 23, myString: 'world' }
 * ]);
 * ```
 */


class BulkLoad extends _events.EventEmitter {
  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */
  constructor(table, collation, connectionOptions, {
    checkConstraints = false,
    fireTriggers = false,
    keepNulls = false,
    lockTable = false,
    order = {}
  }, callback) {
    if (typeof checkConstraints !== 'boolean') {
      throw new TypeError('The "options.checkConstraints" property must be of type boolean.');
    }

    if (typeof fireTriggers !== 'boolean') {
      throw new TypeError('The "options.fireTriggers" property must be of type boolean.');
    }

    if (typeof keepNulls !== 'boolean') {
      throw new TypeError('The "options.keepNulls" property must be of type boolean.');
    }

    if (typeof lockTable !== 'boolean') {
      throw new TypeError('The "options.lockTable" property must be of type boolean.');
    }

    if (typeof order !== 'object' || order === null) {
      throw new TypeError('The "options.order" property must be of type object.');
    }

    for (const [column, direction] of Object.entries(order)) {
      if (direction !== 'ASC' && direction !== 'DESC') {
        throw new TypeError('The value of the "' + column + '" key in the "options.order" object must be either "ASC" or "DESC".');
      }
    }

    super();
    this.error = void 0;
    this.canceled = void 0;
    this.executionStarted = void 0;
    this.streamingMode = void 0;
    this.table = void 0;
    this.timeout = void 0;
    this.options = void 0;
    this.callback = void 0;
    this.columns = void 0;
    this.columnsByName = void 0;
    this.firstRowWritten = void 0;
    this.rowToPacketTransform = void 0;
    this.bulkOptions = void 0;
    this.connection = void 0;
    this.rows = void 0;
    this.rst = void 0;
    this.rowCount = void 0;
    this.collation = void 0;
    this.error = undefined;
    this.canceled = false;
    this.executionStarted = false;
    this.collation = collation;
    this.table = table;
    this.options = connectionOptions;
    this.callback = callback;
    this.columns = [];
    this.columnsByName = {};
    this.firstRowWritten = false;
    this.streamingMode = false;
    this.rowToPacketTransform = new RowTransform(this); // eslint-disable-line no-use-before-define

    this.bulkOptions = {
      checkConstraints,
      fireTriggers,
      keepNulls,
      lockTable,
      order
    };
  }
  /**
   * Adds a column to the bulk load.
   *
   * The column definitions should match the table you are trying to insert into.
   * Attempting to call addColumn after the first row has been added will throw an exception.
   *
   * ```js
   * bulkLoad.addColumn('MyIntColumn', TYPES.Int, { nullable: false });
   * ```
   *
   * @param name The name of the column.
   * @param type One of the supported `data types`.
   * @param __namedParameters Additional column type information. At a minimum, `nullable` must be set to true or false.
   * @param length For VarChar, NVarChar, VarBinary. Use length as `Infinity` for VarChar(max), NVarChar(max) and VarBinary(max).
   * @param nullable Indicates whether the column accepts NULL values.
   * @param objName If the name of the column is different from the name of the property found on `rowObj` arguments passed to [[addRow]] or [[Connection.execBulkLoad]], then you can use this option to specify the property name.
   * @param precision For Numeric, Decimal.
   * @param scale For Numeric, Decimal, Time, DateTime2, DateTimeOffset.
  */


  addColumn(name, type, {
    output = false,
    length,
    precision,
    scale,
    objName = name,
    nullable = true
  }) {
    if (this.firstRowWritten) {
      throw new Error('Columns cannot be added to bulk insert after the first row has been written.');
    }

    if (this.executionStarted) {
      throw new Error('Columns cannot be added to bulk insert after execution has started.');
    }

    const column = {
      type: type,
      name: name,
      value: null,
      output: output,
      length: length,
      precision: precision,
      scale: scale,
      objName: objName,
      nullable: nullable,
      collation: this.collation
    };

    if ((type.id & 0x30) === 0x20) {
      if (column.length == null && type.resolveLength) {
        column.length = type.resolveLength(column);
      }
    }

    if (type.resolvePrecision && column.precision == null) {
      column.precision = type.resolvePrecision(column);
    }

    if (type.resolveScale && column.scale == null) {
      column.scale = type.resolveScale(column);
    }

    this.columns.push(column);
    this.columnsByName[name] = column;
  }
  /**
   * @private
   */


  getOptionsSql() {
    const addOptions = [];

    if (this.bulkOptions.checkConstraints) {
      addOptions.push('CHECK_CONSTRAINTS');
    }

    if (this.bulkOptions.fireTriggers) {
      addOptions.push('FIRE_TRIGGERS');
    }

    if (this.bulkOptions.keepNulls) {
      addOptions.push('KEEP_NULLS');
    }

    if (this.bulkOptions.lockTable) {
      addOptions.push('TABLOCK');
    }

    if (this.bulkOptions.order) {
      const orderColumns = [];

      for (const [column, direction] of Object.entries(this.bulkOptions.order)) {
        orderColumns.push(`${column} ${direction}`);
      }

      if (orderColumns.length) {
        addOptions.push(`ORDER (${orderColumns.join(', ')})`);
      }
    }

    if (addOptions.length > 0) {
      return ` WITH (${addOptions.join(',')})`;
    } else {
      return '';
    }
  }
  /**
   * @private
   */


  getBulkInsertSql() {
    let sql = 'insert bulk ' + this.table + '(';

    for (let i = 0, len = this.columns.length; i < len; i++) {
      const c = this.columns[i];

      if (i !== 0) {
        sql += ', ';
      }

      sql += '[' + c.name + '] ' + c.type.declaration(c);
    }

    sql += ')';
    sql += this.getOptionsSql();
    return sql;
  }
  /**
   * This is simply a helper utility function which returns a `CREATE TABLE SQL` statement based on the columns added to the bulkLoad object.
   * This may be particularly handy when you want to insert into a temporary table (a table which starts with `#`).
   *
   * ```js
   * var sql = bulkLoad.getTableCreationSql();
   * ```
   *
   * A side note on bulk inserting into temporary tables: if you want to access a local temporary table after executing the bulk load,
   * you'll need to use the same connection and execute your requests using [[Connection.execSqlBatch]] instead of [[Connection.execSql]]
   */


  getTableCreationSql() {
    let sql = 'CREATE TABLE ' + this.table + '(\n';

    for (let i = 0, len = this.columns.length; i < len; i++) {
      const c = this.columns[i];

      if (i !== 0) {
        sql += ',\n';
      }

      sql += '[' + c.name + '] ' + c.type.declaration(c);

      if (c.nullable !== undefined) {
        sql += ' ' + (c.nullable ? 'NULL' : 'NOT NULL');
      }
    }

    sql += '\n)';
    return sql;
  }
  /**
   * @private
   */


  getColMetaData() {
    const tBuf = new _writableTrackingBuffer.default(100, null, true); // TokenType

    tBuf.writeUInt8(_token.TYPE.COLMETADATA); // Count

    tBuf.writeUInt16LE(this.columns.length);

    for (let j = 0, len = this.columns.length; j < len; j++) {
      const c = this.columns[j]; // UserType

      if (this.options.tdsVersion < '7_2') {
        tBuf.writeUInt16LE(0);
      } else {
        tBuf.writeUInt32LE(0);
      } // Flags


      let flags = FLAGS.updateableReadWrite;

      if (c.nullable) {
        flags |= FLAGS.nullable;
      } else if (c.nullable === undefined && this.options.tdsVersion >= '7_2') {
        flags |= FLAGS.nullableUnknown;
      }

      tBuf.writeUInt16LE(flags); // TYPE_INFO

      tBuf.writeBuffer(c.type.generateTypeInfo(c, this.options)); // TableName

      if (c.type.hasTableName) {
        tBuf.writeUsVarchar(this.table, 'ucs2');
      } // ColName


      tBuf.writeBVarchar(c.name, 'ucs2');
    }

    return tBuf.data;
  }
  /**
   * Sets a timeout for this bulk load.
   *
   * ```js
   * bulkLoad.setTimeout(timeout);
   * ```
   *
   * @param timeout The number of milliseconds before the bulk load is considered failed, or 0 for no timeout.
   *   When no timeout is set for the bulk load, the [[ConnectionOptions.requestTimeout]] of the Connection is used.
   */


  setTimeout(timeout) {
    this.timeout = timeout;
  }
  /**
   * @private
   */


  createDoneToken() {
    // It might be nice to make DoneToken a class if anything needs to create them, but for now, just do it here
    const tBuf = new _writableTrackingBuffer.default(this.options.tdsVersion < '7_2' ? 9 : 13);
    tBuf.writeUInt8(_token.TYPE.DONE);
    const status = DONE_STATUS.FINAL;
    tBuf.writeUInt16LE(status);
    tBuf.writeUInt16LE(0); // CurCmd (TDS ignores this)

    tBuf.writeUInt32LE(0); // row count - doesn't really matter

    if (this.options.tdsVersion >= '7_2') {
      tBuf.writeUInt32LE(0); // row count is 64 bits in >= TDS 7.2
    }

    return tBuf.data;
  }
  /**
   * @private
   */


  cancel() {
    if (this.canceled) {
      return;
    }

    this.canceled = true;
    this.emit('cancel');
  }

}

var _default = BulkLoad;
exports.default = _default;
module.exports = BulkLoad;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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