"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _crypto = _interopRequireDefault(require("crypto"));

var _os = _interopRequireDefault(require("os"));

var _constants = _interopRequireDefault(require("constants"));

var _tls = require("tls");

var _stream = require("stream");

var _identity = require("@azure/identity");

var _bulkLoad = _interopRequireDefault(require("./bulk-load"));

var _debug = _interopRequireDefault(require("./debug"));

var _events = require("events");

var _instanceLookup = require("./instance-lookup");

var _transientErrorLookup = require("./transient-error-lookup");

var _packet = require("./packet");

var _preloginPayload = _interopRequireDefault(require("./prelogin-payload"));

var _login7Payload = _interopRequireDefault(require("./login7-payload"));

var _ntlmPayload = _interopRequireDefault(require("./ntlm-payload"));

var _request = _interopRequireDefault(require("./request"));

var _rpcrequestPayload = _interopRequireDefault(require("./rpcrequest-payload"));

var _sqlbatchPayload = _interopRequireDefault(require("./sqlbatch-payload"));

var _messageIo = _interopRequireDefault(require("./message-io"));

var _tokenStreamParser = require("./token/token-stream-parser");

var _transaction = require("./transaction");

var _errors = require("./errors");

var _connector = require("./connector");

var _library = require("./library");

var _tdsVersions = require("./tds-versions");

var _message = _interopRequireDefault(require("./message"));

var _ntlm = require("./ntlm");

var _nodeAbortController = require("node-abort-controller");

var _dataType = require("./data-type");

var _bulkLoadPayload = require("./bulk-load-payload");

var _esAggregateError = _interopRequireDefault(require("es-aggregate-error"));

var _package = require("../package.json");

var _url = require("url");

var _handler = require("./token/handler");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let trustServerWarningEmitted = false;

const emitTrustServerCertificateWarning = () => {
  if (!trustServerWarningEmitted) {
    trustServerWarningEmitted = true;
    process.emitWarning('`config.options.trustServerCertificate` will default to false in the future. To silence this message, specify a value explicitly in the config options');
  }
};

let domainRenameToTenantIdWarningEmitted = false;

const emitDomainRenameToTenantIdWarning = () => {
  if (!domainRenameToTenantIdWarningEmitted) {
    domainRenameToTenantIdWarningEmitted = true;
    process.emitWarning('`When using authentication type `azure-active-directory-password`,' + ' config.authentication.options.domain` will be renamed to config.authentications.options.tenantId`' + ' in the future. Rename `domain` to `tenantId` to silence this message.');
  }
};

/**
 * @private
 */
const KEEP_ALIVE_INITIAL_DELAY = 30 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CLIENT_REQUEST_TIMEOUT = 15 * 1000;
/**
 * @private
 */

const DEFAULT_CANCEL_TIMEOUT = 5 * 1000;
/**
 * @private
 */

const DEFAULT_CONNECT_RETRY_INTERVAL = 500;
/**
 * @private
 */

const DEFAULT_PACKET_SIZE = 4 * 1024;
/**
 * @private
 */

const DEFAULT_TEXTSIZE = 2147483647;
/**
 * @private
 */

const DEFAULT_DATEFIRST = 7;
/**
 * @private
 */

const DEFAULT_PORT = 1433;
/**
 * @private
 */

const DEFAULT_TDS_VERSION = '7_4';
/**
 * @private
 */

const DEFAULT_LANGUAGE = 'us_english';
/**
 * @private
 */

const DEFAULT_DATEFORMAT = 'mdy';

/**
 * @private
 */
const CLEANUP_TYPE = {
  NORMAL: 0,
  REDIRECT: 1,
  RETRY: 2
};

/**
 * A [[Connection]] instance represents a single connection to a database server.
 *
 * ```js
 * var Connection = require('tedious').Connection;
 * var config = {
 *  "authentication": {
 *    ...,
 *    "options": {...}
 *  },
 *  "options": {...}
 * };
 * var connection = new Connection(config);
 * ```
 *
 * Only one request at a time may be executed on a connection. Once a [[Request]]
 * has been initiated (with [[Connection.callProcedure]], [[Connection.execSql]],
 * or [[Connection.execSqlBatch]]), another should not be initiated until the
 * [[Request]]'s completion callback is called.
 */
class Connection extends _events.EventEmitter {
  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * @private
   */

  /**
   * Note: be aware of the different options field:
   * 1. config.authentication.options
   * 2. config.options
   *
   * ```js
   * const { Connection } = require('tedious');
   *
   * const config = {
   *  "authentication": {
   *    ...,
   *    "options": {...}
   *  },
   *  "options": {...}
   * };
   *
   * const connection = new Connection(config);
   * ```
   *
   * @param config
   */
  constructor(config) {
    super();
    this.fedAuthRequired = void 0;
    this.config = void 0;
    this.secureContext = void 0;
    this.inTransaction = void 0;
    this.transactionDescriptors = void 0;
    this.transactionDepth = void 0;
    this.isSqlBatch = void 0;
    this.curTransientRetryCount = void 0;
    this.transientErrorLookup = void 0;
    this.closed = void 0;
    this.loginError = void 0;
    this.debug = void 0;
    this.ntlmpacket = void 0;
    this.ntlmpacketBuffer = void 0;
    this.routingData = void 0;
    this.messageIo = void 0;
    this.state = void 0;
    this.resetConnectionOnNextRequest = void 0;
    this.request = void 0;
    this.procReturnStatusValue = void 0;
    this.socket = void 0;
    this.messageBuffer = void 0;
    this.connectTimer = void 0;
    this.cancelTimer = void 0;
    this.requestTimer = void 0;
    this.retryTimer = void 0;
    this._cancelAfterRequestSent = void 0;
    this.databaseCollation = void 0;

    if (typeof config !== 'object' || config === null) {
      throw new TypeError('The "config" argument is required and must be of type Object.');
    }

    if (typeof config.server !== 'string') {
      throw new TypeError('The "config.server" property is required and must be of type string.');
    }

    this.fedAuthRequired = false;
    let authentication;

    if (config.authentication !== undefined) {
      if (typeof config.authentication !== 'object' || config.authentication === null) {
        throw new TypeError('The "config.authentication" property must be of type Object.');
      }

      const type = config.authentication.type;
      const options = config.authentication.options === undefined ? {} : config.authentication.options;

      if (typeof type !== 'string') {
        throw new TypeError('The "config.authentication.type" property must be of type string.');
      }

      if (type !== 'default' && type !== 'ntlm' && type !== 'azure-active-directory-password' && type !== 'azure-active-directory-access-token' && type !== 'azure-active-directory-msi-vm' && type !== 'azure-active-directory-msi-app-service' && type !== 'azure-active-directory-service-principal-secret' && type !== 'azure-active-directory-default') {
        throw new TypeError('The "type" property must one of "default", "ntlm", "azure-active-directory-password", "azure-active-directory-access-token", "azure-active-directory-default", "azure-active-directory-msi-vm" or "azure-active-directory-msi-app-service" or "azure-active-directory-service-principal-secret".');
      }

      if (typeof options !== 'object' || options === null) {
        throw new TypeError('The "config.authentication.options" property must be of type object.');
      }

      if (type === 'ntlm') {
        if (typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        }

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'ntlm',
          options: {
            userName: options.userName,
            password: options.password,
            domain: options.domain && options.domain.toUpperCase()
          }
        };
      } else if (type === 'azure-active-directory-password') {
        var _options$tenantId, _options$clientId;

        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        } else if (options.clientId === undefined) {
          emitAzureADPasswordClientIdDeprecationWarning();
        }

        if (options.domain !== undefined && typeof options.domain !== 'string') {
          throw new TypeError('The "config.authentication.options.domain" property must be of type string.');
        } else if (options.domain !== undefined) {
          emitDomainRenameToTenantIdWarning();
        }

        if (options.tenantId !== undefined && typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-password',
          options: {
            userName: options.userName,
            password: options.password,
            tenantId: (_options$tenantId = options.tenantId) !== null && _options$tenantId !== void 0 ? _options$tenantId : options.domain,
            clientId: (_options$clientId = options.clientId) !== null && _options$clientId !== void 0 ? _options$clientId : '7f98cb04-cd1e-40df-9140-3bf7e2cea4db'
          }
        };
      } else if (type === 'azure-active-directory-access-token') {
        if (typeof options.token !== 'string') {
          throw new TypeError('The "config.authentication.options.token" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-access-token',
          options: {
            token: options.token
          }
        };
      } else if (type === 'azure-active-directory-msi-vm') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-vm',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-default') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-default',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-msi-app-service') {
        if (options.clientId !== undefined && typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-msi-app-service',
          options: {
            clientId: options.clientId
          }
        };
      } else if (type === 'azure-active-directory-service-principal-secret') {
        if (typeof options.clientId !== 'string') {
          throw new TypeError('The "config.authentication.options.clientId" property must be of type string.');
        }

        if (typeof options.clientSecret !== 'string') {
          throw new TypeError('The "config.authentication.options.clientSecret" property must be of type string.');
        }

        if (typeof options.tenantId !== 'string') {
          throw new TypeError('The "config.authentication.options.tenantId" property must be of type string.');
        }

        authentication = {
          type: 'azure-active-directory-service-principal-secret',
          options: {
            clientId: options.clientId,
            clientSecret: options.clientSecret,
            tenantId: options.tenantId
          }
        };
      } else {
        if (options.userName !== undefined && typeof options.userName !== 'string') {
          throw new TypeError('The "config.authentication.options.userName" property must be of type string.');
        }

        if (options.password !== undefined && typeof options.password !== 'string') {
          throw new TypeError('The "config.authentication.options.password" property must be of type string.');
        }

        authentication = {
          type: 'default',
          options: {
            userName: options.userName,
            password: options.password
          }
        };
      }
    } else {
      authentication = {
        type: 'default',
        options: {
          userName: undefined,
          password: undefined
        }
      };
    }

    this.config = {
      server: config.server,
      authentication: authentication,
      options: {
        abortTransactionOnError: false,
        appName: undefined,
        camelCaseColumns: false,
        cancelTimeout: DEFAULT_CANCEL_TIMEOUT,
        columnEncryptionKeyCacheTTL: 2 * 60 * 60 * 1000,
        // Units: miliseconds
        columnEncryptionSetting: false,
        columnNameReplacer: undefined,
        connectionRetryInterval: DEFAULT_CONNECT_RETRY_INTERVAL,
        connectTimeout: DEFAULT_CONNECT_TIMEOUT,
        connectionIsolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        cryptoCredentialsDetails: {},
        database: undefined,
        datefirst: DEFAULT_DATEFIRST,
        dateFormat: DEFAULT_DATEFORMAT,
        debug: {
          data: false,
          packet: false,
          payload: false,
          token: false
        },
        enableAnsiNull: true,
        enableAnsiNullDefault: true,
        enableAnsiPadding: true,
        enableAnsiWarnings: true,
        enableArithAbort: true,
        enableConcatNullYieldsNull: true,
        enableCursorCloseOnCommit: null,
        enableImplicitTransactions: false,
        enableNumericRoundabort: false,
        enableQuotedIdentifier: true,
        encrypt: true,
        fallbackToDefaultDb: false,
        encryptionKeyStoreProviders: undefined,
        instanceName: undefined,
        isolationLevel: _transaction.ISOLATION_LEVEL.READ_COMMITTED,
        language: DEFAULT_LANGUAGE,
        localAddress: undefined,
        maxRetriesOnTransientErrors: 3,
        multiSubnetFailover: false,
        packetSize: DEFAULT_PACKET_SIZE,
        port: DEFAULT_PORT,
        readOnlyIntent: false,
        requestTimeout: DEFAULT_CLIENT_REQUEST_TIMEOUT,
        rowCollectionOnDone: false,
        rowCollectionOnRequestCompletion: false,
        serverName: undefined,
        serverSupportsColumnEncryption: false,
        tdsVersion: DEFAULT_TDS_VERSION,
        textsize: DEFAULT_TEXTSIZE,
        trustedServerNameAE: undefined,
        trustServerCertificate: true,
        useColumnNames: false,
        useUTC: true,
        workstationId: undefined,
        lowerCaseGuids: false
      }
    };

    if (config.options) {
      if (config.options.port && config.options.instanceName) {
        throw new Error('Port and instanceName are mutually exclusive, but ' + config.options.port + ' and ' + config.options.instanceName + ' provided');
      }

      if (config.options.abortTransactionOnError !== undefined) {
        if (typeof config.options.abortTransactionOnError !== 'boolean' && config.options.abortTransactionOnError !== null) {
          throw new TypeError('The "config.options.abortTransactionOnError" property must be of type string or null.');
        }

        this.config.options.abortTransactionOnError = config.options.abortTransactionOnError;
      }

      if (config.options.appName !== undefined) {
        if (typeof config.options.appName !== 'string') {
          throw new TypeError('The "config.options.appName" property must be of type string.');
        }

        this.config.options.appName = config.options.appName;
      }

      if (config.options.camelCaseColumns !== undefined) {
        if (typeof config.options.camelCaseColumns !== 'boolean') {
          throw new TypeError('The "config.options.camelCaseColumns" property must be of type boolean.');
        }

        this.config.options.camelCaseColumns = config.options.camelCaseColumns;
      }

      if (config.options.cancelTimeout !== undefined) {
        if (typeof config.options.cancelTimeout !== 'number') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type number.');
        }

        this.config.options.cancelTimeout = config.options.cancelTimeout;
      }

      if (config.options.columnNameReplacer) {
        if (typeof config.options.columnNameReplacer !== 'function') {
          throw new TypeError('The "config.options.cancelTimeout" property must be of type function.');
        }

        this.config.options.columnNameReplacer = config.options.columnNameReplacer;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.connectionIsolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.connectionIsolationLevel, 'config.options.connectionIsolationLevel');
        this.config.options.connectionIsolationLevel = config.options.connectionIsolationLevel;
      }

      if (config.options.connectTimeout !== undefined) {
        if (typeof config.options.connectTimeout !== 'number') {
          throw new TypeError('The "config.options.connectTimeout" property must be of type number.');
        }

        this.config.options.connectTimeout = config.options.connectTimeout;
      }

      if (config.options.cryptoCredentialsDetails !== undefined) {
        if (typeof config.options.cryptoCredentialsDetails !== 'object' || config.options.cryptoCredentialsDetails === null) {
          throw new TypeError('The "config.options.cryptoCredentialsDetails" property must be of type Object.');
        }

        this.config.options.cryptoCredentialsDetails = config.options.cryptoCredentialsDetails;
      }

      if (config.options.database !== undefined) {
        if (typeof config.options.database !== 'string') {
          throw new TypeError('The "config.options.database" property must be of type string.');
        }

        this.config.options.database = config.options.database;
      }

      if (config.options.datefirst !== undefined) {
        if (typeof config.options.datefirst !== 'number' && config.options.datefirst !== null) {
          throw new TypeError('The "config.options.datefirst" property must be of type number.');
        }

        if (config.options.datefirst !== null && (config.options.datefirst < 1 || config.options.datefirst > 7)) {
          throw new RangeError('The "config.options.datefirst" property must be >= 1 and <= 7');
        }

        this.config.options.datefirst = config.options.datefirst;
      }

      if (config.options.dateFormat !== undefined) {
        if (typeof config.options.dateFormat !== 'string' && config.options.dateFormat !== null) {
          throw new TypeError('The "config.options.dateFormat" property must be of type string or null.');
        }

        this.config.options.dateFormat = config.options.dateFormat;
      }

      if (config.options.debug) {
        if (config.options.debug.data !== undefined) {
          if (typeof config.options.debug.data !== 'boolean') {
            throw new TypeError('The "config.options.debug.data" property must be of type boolean.');
          }

          this.config.options.debug.data = config.options.debug.data;
        }

        if (config.options.debug.packet !== undefined) {
          if (typeof config.options.debug.packet !== 'boolean') {
            throw new TypeError('The "config.options.debug.packet" property must be of type boolean.');
          }

          this.config.options.debug.packet = config.options.debug.packet;
        }

        if (config.options.debug.payload !== undefined) {
          if (typeof config.options.debug.payload !== 'boolean') {
            throw new TypeError('The "config.options.debug.payload" property must be of type boolean.');
          }

          this.config.options.debug.payload = config.options.debug.payload;
        }

        if (config.options.debug.token !== undefined) {
          if (typeof config.options.debug.token !== 'boolean') {
            throw new TypeError('The "config.options.debug.token" property must be of type boolean.');
          }

          this.config.options.debug.token = config.options.debug.token;
        }
      }

      if (config.options.enableAnsiNull !== undefined) {
        if (typeof config.options.enableAnsiNull !== 'boolean' && config.options.enableAnsiNull !== null) {
          throw new TypeError('The "config.options.enableAnsiNull" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNull = config.options.enableAnsiNull;
      }

      if (config.options.enableAnsiNullDefault !== undefined) {
        if (typeof config.options.enableAnsiNullDefault !== 'boolean' && config.options.enableAnsiNullDefault !== null) {
          throw new TypeError('The "config.options.enableAnsiNullDefault" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiNullDefault = config.options.enableAnsiNullDefault;
      }

      if (config.options.enableAnsiPadding !== undefined) {
        if (typeof config.options.enableAnsiPadding !== 'boolean' && config.options.enableAnsiPadding !== null) {
          throw new TypeError('The "config.options.enableAnsiPadding" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiPadding = config.options.enableAnsiPadding;
      }

      if (config.options.enableAnsiWarnings !== undefined) {
        if (typeof config.options.enableAnsiWarnings !== 'boolean' && config.options.enableAnsiWarnings !== null) {
          throw new TypeError('The "config.options.enableAnsiWarnings" property must be of type boolean or null.');
        }

        this.config.options.enableAnsiWarnings = config.options.enableAnsiWarnings;
      }

      if (config.options.enableArithAbort !== undefined) {
        if (typeof config.options.enableArithAbort !== 'boolean' && config.options.enableArithAbort !== null) {
          throw new TypeError('The "config.options.enableArithAbort" property must be of type boolean or null.');
        }

        this.config.options.enableArithAbort = config.options.enableArithAbort;
      }

      if (config.options.enableConcatNullYieldsNull !== undefined) {
        if (typeof config.options.enableConcatNullYieldsNull !== 'boolean' && config.options.enableConcatNullYieldsNull !== null) {
          throw new TypeError('The "config.options.enableConcatNullYieldsNull" property must be of type boolean or null.');
        }

        this.config.options.enableConcatNullYieldsNull = config.options.enableConcatNullYieldsNull;
      }

      if (config.options.enableCursorCloseOnCommit !== undefined) {
        if (typeof config.options.enableCursorCloseOnCommit !== 'boolean' && config.options.enableCursorCloseOnCommit !== null) {
          throw new TypeError('The "config.options.enableCursorCloseOnCommit" property must be of type boolean or null.');
        }

        this.config.options.enableCursorCloseOnCommit = config.options.enableCursorCloseOnCommit;
      }

      if (config.options.enableImplicitTransactions !== undefined) {
        if (typeof config.options.enableImplicitTransactions !== 'boolean' && config.options.enableImplicitTransactions !== null) {
          throw new TypeError('The "config.options.enableImplicitTransactions" property must be of type boolean or null.');
        }

        this.config.options.enableImplicitTransactions = config.options.enableImplicitTransactions;
      }

      if (config.options.enableNumericRoundabort !== undefined) {
        if (typeof config.options.enableNumericRoundabort !== 'boolean' && config.options.enableNumericRoundabort !== null) {
          throw new TypeError('The "config.options.enableNumericRoundabort" property must be of type boolean or null.');
        }

        this.config.options.enableNumericRoundabort = config.options.enableNumericRoundabort;
      }

      if (config.options.enableQuotedIdentifier !== undefined) {
        if (typeof config.options.enableQuotedIdentifier !== 'boolean' && config.options.enableQuotedIdentifier !== null) {
          throw new TypeError('The "config.options.enableQuotedIdentifier" property must be of type boolean or null.');
        }

        this.config.options.enableQuotedIdentifier = config.options.enableQuotedIdentifier;
      }

      if (config.options.encrypt !== undefined) {
        if (typeof config.options.encrypt !== 'boolean') {
          throw new TypeError('The "config.options.encrypt" property must be of type boolean.');
        }

        this.config.options.encrypt = config.options.encrypt;
      }

      if (config.options.fallbackToDefaultDb !== undefined) {
        if (typeof config.options.fallbackToDefaultDb !== 'boolean') {
          throw new TypeError('The "config.options.fallbackToDefaultDb" property must be of type boolean.');
        }

        this.config.options.fallbackToDefaultDb = config.options.fallbackToDefaultDb;
      }

      if (config.options.instanceName !== undefined) {
        if (typeof config.options.instanceName !== 'string') {
          throw new TypeError('The "config.options.instanceName" property must be of type string.');
        }

        this.config.options.instanceName = config.options.instanceName;
        this.config.options.port = undefined;
      }

      if (config.options.isolationLevel !== undefined) {
        (0, _transaction.assertValidIsolationLevel)(config.options.isolationLevel, 'config.options.isolationLevel');
        this.config.options.isolationLevel = config.options.isolationLevel;
      }

      if (config.options.language !== undefined) {
        if (typeof config.options.language !== 'string' && config.options.language !== null) {
          throw new TypeError('The "config.options.language" property must be of type string or null.');
        }

        this.config.options.language = config.options.language;
      }

      if (config.options.localAddress !== undefined) {
        if (typeof config.options.localAddress !== 'string') {
          throw new TypeError('The "config.options.localAddress" property must be of type string.');
        }

        this.config.options.localAddress = config.options.localAddress;
      }

      if (config.options.multiSubnetFailover !== undefined) {
        if (typeof config.options.multiSubnetFailover !== 'boolean') {
          throw new TypeError('The "config.options.multiSubnetFailover" property must be of type boolean.');
        }

        this.config.options.multiSubnetFailover = config.options.multiSubnetFailover;
      }

      if (config.options.packetSize !== undefined) {
        if (typeof config.options.packetSize !== 'number') {
          throw new TypeError('The "config.options.packetSize" property must be of type number.');
        }

        this.config.options.packetSize = config.options.packetSize;
      }

      if (config.options.port !== undefined) {
        if (typeof config.options.port !== 'number') {
          throw new TypeError('The "config.options.port" property must be of type number.');
        }

        if (config.options.port <= 0 || config.options.port >= 65536) {
          throw new RangeError('The "config.options.port" property must be > 0 and < 65536');
        }

        this.config.options.port = config.options.port;
        this.config.options.instanceName = undefined;
      }

      if (config.options.readOnlyIntent !== undefined) {
        if (typeof config.options.readOnlyIntent !== 'boolean') {
          throw new TypeError('The "config.options.readOnlyIntent" property must be of type boolean.');
        }

        this.config.options.readOnlyIntent = config.options.readOnlyIntent;
      }

      if (config.options.requestTimeout !== undefined) {
        if (typeof config.options.requestTimeout !== 'number') {
          throw new TypeError('The "config.options.requestTimeout" property must be of type number.');
        }

        this.config.options.requestTimeout = config.options.requestTimeout;
      }

      if (config.options.maxRetriesOnTransientErrors !== undefined) {
        if (typeof config.options.maxRetriesOnTransientErrors !== 'number') {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be of type number.');
        }

        if (config.options.maxRetriesOnTransientErrors < 0) {
          throw new TypeError('The "config.options.maxRetriesOnTransientErrors" property must be equal or greater than 0.');
        }

        this.config.options.maxRetriesOnTransientErrors = config.options.maxRetriesOnTransientErrors;
      }

      if (config.options.connectionRetryInterval !== undefined) {
        if (typeof config.options.connectionRetryInterval !== 'number') {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be of type number.');
        }

        if (config.options.connectionRetryInterval <= 0) {
          throw new TypeError('The "config.options.connectionRetryInterval" property must be greater than 0.');
        }

        this.config.options.connectionRetryInterval = config.options.connectionRetryInterval;
      }

      if (config.options.rowCollectionOnDone !== undefined) {
        if (typeof config.options.rowCollectionOnDone !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnDone" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnDone = config.options.rowCollectionOnDone;
      }

      if (config.options.rowCollectionOnRequestCompletion !== undefined) {
        if (typeof config.options.rowCollectionOnRequestCompletion !== 'boolean') {
          throw new TypeError('The "config.options.rowCollectionOnRequestCompletion" property must be of type boolean.');
        }

        this.config.options.rowCollectionOnRequestCompletion = config.options.rowCollectionOnRequestCompletion;
      }

      if (config.options.tdsVersion !== undefined) {
        if (typeof config.options.tdsVersion !== 'string') {
          throw new TypeError('The "config.options.tdsVersion" property must be of type string.');
        }

        this.config.options.tdsVersion = config.options.tdsVersion;
      }

      if (config.options.textsize !== undefined) {
        if (typeof config.options.textsize !== 'number' && config.options.textsize !== null) {
          throw new TypeError('The "config.options.textsize" property must be of type number or null.');
        }

        if (config.options.textsize > 2147483647) {
          throw new TypeError('The "config.options.textsize" can\'t be greater than 2147483647.');
        } else if (config.options.textsize < -1) {
          throw new TypeError('The "config.options.textsize" can\'t be smaller than -1.');
        }

        this.config.options.textsize = config.options.textsize | 0;
      }

      if (config.options.trustServerCertificate !== undefined) {
        if (typeof config.options.trustServerCertificate !== 'boolean') {
          throw new TypeError('The "config.options.trustServerCertificate" property must be of type boolean.');
        }

        this.config.options.trustServerCertificate = config.options.trustServerCertificate;
      } else {
        emitTrustServerCertificateWarning();
      }

      if (config.options.useColumnNames !== undefined) {
        if (typeof config.options.useColumnNames !== 'boolean') {
          throw new TypeError('The "config.options.useColumnNames" property must be of type boolean.');
        }

        this.config.options.useColumnNames = config.options.useColumnNames;
      }

      if (config.options.useUTC !== undefined) {
        if (typeof config.options.useUTC !== 'boolean') {
          throw new TypeError('The "config.options.useUTC" property must be of type boolean.');
        }

        this.config.options.useUTC = config.options.useUTC;
      }

      if (config.options.workstationId !== undefined) {
        if (typeof config.options.workstationId !== 'string') {
          throw new TypeError('The "config.options.workstationId" property must be of type string.');
        }

        this.config.options.workstationId = config.options.workstationId;
      }

      if (config.options.lowerCaseGuids !== undefined) {
        if (typeof config.options.lowerCaseGuids !== 'boolean') {
          throw new TypeError('The "config.options.lowerCaseGuids" property must be of type boolean.');
        }

        this.config.options.lowerCaseGuids = config.options.lowerCaseGuids;
      }
    }

    let credentialsDetails = this.config.options.cryptoCredentialsDetails;

    if (credentialsDetails.secureOptions === undefined) {
      // If the caller has not specified their own `secureOptions`,
      // we set `SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS` here.
      // Older SQL Server instances running on older Windows versions have
      // trouble with the BEAST workaround in OpenSSL.
      // As BEAST is a browser specific exploit, we can just disable this option here.
      credentialsDetails = Object.create(credentialsDetails, {
        secureOptions: {
          value: _constants.default.SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS
        }
      });
    }

    this.secureContext = (0, _tls.createSecureContext)(credentialsDetails);
    this.debug = this.createDebug();
    this.inTransaction = false;
    this.transactionDescriptors = [Buffer.from([0, 0, 0, 0, 0, 0, 0, 0])]; // 'beginTransaction', 'commitTransaction' and 'rollbackTransaction'
    // events are utilized to maintain inTransaction property state which in
    // turn is used in managing transactions. These events are only fired for
    // TDS version 7.2 and beyond. The properties below are used to emulate
    // equivalent behavior for TDS versions before 7.2.

    this.transactionDepth = 0;
    this.isSqlBatch = false;
    this.closed = false;
    this.messageBuffer = Buffer.alloc(0);
    this.curTransientRetryCount = 0;
    this.transientErrorLookup = new _transientErrorLookup.TransientErrorLookup();
    this.state = this.STATE.INITIALIZED;

    this._cancelAfterRequestSent = () => {
      this.messageIo.sendMessage(_packet.TYPE.ATTENTION);
      this.createCancelTimer();
    };
  }

  connect(connectListener) {
    if (this.state !== this.STATE.INITIALIZED) {
      throw new _errors.ConnectionError('`.connect` can not be called on a Connection in `' + this.state.name + '` state.');
    }

    if (connectListener) {
      const onConnect = err => {
        this.removeListener('error', onError);
        connectListener(err);
      };

      const onError = err => {
        this.removeListener('connect', onConnect);
        connectListener(err);
      };

      this.once('connect', onConnect);
      this.once('error', onError);
    }

    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * The server has reported that the charset has changed.
   */


  on(event, listener) {
    return super.on(event, listener);
  }
  /**
   * @private
   */


  emit(event, ...args) {
    return super.emit(event, ...args);
  }
  /**
   * Closes the connection to the database.
   *
   * The [[Event_end]] will be emitted once the connection has been closed.
   */


  close() {
    this.transitionTo(this.STATE.FINAL);
  }
  /**
   * @private
   */


  initialiseConnection() {
    const signal = this.createConnectTimer();

    if (this.config.options.port) {
      return this.connectOnPort(this.config.options.port, this.config.options.multiSubnetFailover, signal);
    } else {
      return new _instanceLookup.InstanceLookup().instanceLookup({
        server: this.config.server,
        instanceName: this.config.options.instanceName,
        timeout: this.config.options.connectTimeout,
        signal: signal
      }, (err, port) => {
        if (err) {
          if (err.name === 'AbortError') {
            return;
          }

          this.emit('connect', new _errors.ConnectionError(err.message, 'EINSTLOOKUP'));
        } else {
          this.connectOnPort(port, this.config.options.multiSubnetFailover, signal);
        }
      });
    }
  }
  /**
   * @private
   */


  cleanupConnection(cleanupType) {
    if (!this.closed) {
      this.clearConnectTimer();
      this.clearRequestTimer();
      this.clearRetryTimer();
      this.closeConnection();

      if (cleanupType === CLEANUP_TYPE.REDIRECT) {
        this.emit('rerouting');
      } else if (cleanupType !== CLEANUP_TYPE.RETRY) {
        process.nextTick(() => {
          this.emit('end');
        });
      }

      const request = this.request;

      if (request) {
        const err = new _errors.RequestError('Connection closed before request completed.', 'ECLOSE');
        request.callback(err);
        this.request = undefined;
      }

      this.closed = true;
      this.loginError = undefined;
    }
  }
  /**
   * @private
   */


  createDebug() {
    const debug = new _debug.default(this.config.options.debug);
    debug.on('debug', message => {
      this.emit('debug', message);
    });
    return debug;
  }
  /**
   * @private
   */


  createTokenStreamParser(message, handler) {
    return new _tokenStreamParser.Parser(message, this.debug, handler, this.config.options);
  }

  connectOnPort(port, multiSubnetFailover, signal) {
    const connectOpts = {
      host: this.routingData ? this.routingData.server : this.config.server,
      port: this.routingData ? this.routingData.port : port,
      localAddress: this.config.options.localAddress
    };
    new _connector.Connector(connectOpts, signal, multiSubnetFailover).execute((err, socket) => {
      if (err) {
        if (err.name === 'AbortError') {
          return;
        }

        return this.socketError(err);
      }

      socket = socket;
      socket.on('error', error => {
        this.socketError(error);
      });
      socket.on('close', () => {
        this.socketClose();
      });
      socket.on('end', () => {
        this.socketEnd();
      });
      socket.setKeepAlive(true, KEEP_ALIVE_INITIAL_DELAY);
      this.messageIo = new _messageIo.default(socket, this.config.options.packetSize, this.debug);
      this.messageIo.on('secure', cleartext => {
        this.emit('secure', cleartext);
      });
      this.socket = socket;
      this.closed = false;
      this.debug.log('connected to ' + this.config.server + ':' + this.config.options.port);
      this.sendPreLogin();
      this.transitionTo(this.STATE.SENT_PRELOGIN);
    });
  }
  /**
   * @private
   */


  closeConnection() {
    if (this.socket) {
      this.socket.destroy();
    }
  }
  /**
   * @private
   */


  createConnectTimer() {
    const controller = new _nodeAbortController.AbortController();
    this.connectTimer = setTimeout(() => {
      controller.abort();
      this.connectTimeout();
    }, this.config.options.connectTimeout);
    return controller.signal;
  }
  /**
   * @private
   */


  createCancelTimer() {
    this.clearCancelTimer();
    const timeout = this.config.options.cancelTimeout;

    if (timeout > 0) {
      this.cancelTimer = setTimeout(() => {
        this.cancelTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRequestTimer() {
    this.clearRequestTimer(); // release old timer, just to be safe

    const request = this.request;
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;

    if (timeout) {
      this.requestTimer = setTimeout(() => {
        this.requestTimeout();
      }, timeout);
    }
  }
  /**
   * @private
   */


  createRetryTimer() {
    this.clearRetryTimer();
    this.retryTimer = setTimeout(() => {
      this.retryTimeout();
    }, this.config.options.connectionRetryInterval);
  }
  /**
   * @private
   */


  connectTimeout() {
    const message = `Failed to connect to ${this.config.server}${this.config.options.port ? `:${this.config.options.port}` : `\\${this.config.options.instanceName}`} in ${this.config.options.connectTimeout}ms`;
    this.debug.log(message);
    this.emit('connect', new _errors.ConnectionError(message, 'ETIMEOUT'));
    this.connectTimer = undefined;
    this.dispatchEvent('connectTimeout');
  }
  /**
   * @private
   */


  cancelTimeout() {
    const message = `Failed to cancel request in ${this.config.options.cancelTimeout}ms`;
    this.debug.log(message);
    this.dispatchEvent('socketError', new _errors.ConnectionError(message, 'ETIMEOUT'));
  }
  /**
   * @private
   */


  requestTimeout() {
    this.requestTimer = undefined;
    const request = this.request;
    request.cancel();
    const timeout = request.timeout !== undefined ? request.timeout : this.config.options.requestTimeout;
    const message = 'Timeout: Request failed to complete in ' + timeout + 'ms';
    request.error = new _errors.RequestError(message, 'ETIMEOUT');
  }
  /**
   * @private
   */


  retryTimeout() {
    this.retryTimer = undefined;
    this.emit('retry');
    this.transitionTo(this.STATE.CONNECTING);
  }
  /**
   * @private
   */


  clearConnectTimer() {
    if (this.connectTimer) {
      clearTimeout(this.connectTimer);
      this.connectTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearCancelTimer() {
    if (this.cancelTimer) {
      clearTimeout(this.cancelTimer);
      this.cancelTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRequestTimer() {
    if (this.requestTimer) {
      clearTimeout(this.requestTimer);
      this.requestTimer = undefined;
    }
  }
  /**
   * @private
   */


  clearRetryTimer() {
    if (this.retryTimer) {
      clearTimeout(this.retryTimer);
      this.retryTimer = undefined;
    }
  }
  /**
   * @private
   */


  transitionTo(newState) {
    if (this.state === newState) {
      this.debug.log('State is already ' + newState.name);
      return;
    }

    if (this.state && this.state.exit) {
      this.state.exit.call(this, newState);
    }

    this.debug.log('State change: ' + (this.state ? this.state.name : 'undefined') + ' -> ' + newState.name);
    this.state = newState;

    if (this.state.enter) {
      this.state.enter.apply(this);
    }
  }
  /**
   * @private
   */


  getEventHandler(eventName) {
    const handler = this.state.events[eventName];

    if (!handler) {
      throw new Error(`No event '${eventName}' in state '${this.state.name}'`);
    }

    return handler;
  }
  /**
   * @private
   */


  dispatchEvent(eventName, ...args) {
    const handler = this.state.events[eventName];

    if (handler) {
      handler.apply(this, args);
    } else {
      this.emit('error', new Error(`No event '${eventName}' in state '${this.state.name}'`));
      this.close();
    }
  }
  /**
   * @private
   */


  socketError(error) {
    if (this.state === this.STATE.CONNECTING || this.state === this.STATE.SENT_TLSSSLNEGOTIATION) {
      const message = `Failed to connect to ${this.config.server}:${this.config.options.port} - ${error.message}`;
      this.debug.log(message);
      this.emit('connect', new _errors.ConnectionError(message, 'ESOCKET'));
    } else {
      const message = `Connection lost - ${error.message}`;
      this.debug.log(message);
      this.emit('error', new _errors.ConnectionError(message, 'ESOCKET'));
    }

    this.dispatchEvent('socketError', error);
  }
  /**
   * @private
   */


  socketEnd() {
    this.debug.log('socket ended');

    if (this.state !== this.STATE.FINAL) {
      const error = new Error('socket hang up');
      error.code = 'ECONNRESET';
      this.socketError(error);
    }
  }
  /**
   * @private
   */


  socketClose() {
    this.debug.log('connection to ' + this.config.server + ':' + this.config.options.port + ' closed');

    if (this.state === this.STATE.REROUTING) {
      this.debug.log('Rerouting to ' + this.routingData.server + ':' + this.routingData.port);
      this.dispatchEvent('reconnect');
    } else if (this.state === this.STATE.TRANSIENT_FAILURE_RETRY) {
      const server = this.routingData ? this.routingData.server : this.config.server;
      const port = this.routingData ? this.routingData.port : this.config.options.port;
      this.debug.log('Retry after transient failure connecting to ' + server + ':' + port);
      this.dispatchEvent('retry');
    } else {
      this.transitionTo(this.STATE.FINAL);
    }
  }
  /**
   * @private
   */


  sendPreLogin() {
    var _$exec;

    const [, major, minor, build] = (_$exec = /^(\d+)\.(\d+)\.(\d+)/.exec(_package.version)) !== null && _$exec !== void 0 ? _$exec : ['0.0.0', '0', '0', '0'];
    const payload = new _preloginPayload.default({
      encrypt: this.config.options.encrypt,
      version: {
        major: Number(major),
        minor: Number(minor),
        build: Number(build),
        subbuild: 0
      }
    });
    this.messageIo.sendMessage(_packet.TYPE.PRELOGIN, payload.data);
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  emptyMessageBuffer() {
    this.messageBuffer = Buffer.alloc(0);
  }
  /**
   * @private
   */


  addToMessageBuffer(data) {
    this.messageBuffer = Buffer.concat([this.messageBuffer, data]);
  }
  /**
   * @private
   */


  sendLogin7Packet() {
    const payload = new _login7Payload.default({
      tdsVersion: _tdsVersions.versions[this.config.options.tdsVersion],
      packetSize: this.config.options.packetSize,
      clientProgVer: 0,
      clientPid: process.pid,
      connectionId: 0,
      clientTimeZone: new Date().getTimezoneOffset(),
      clientLcid: 0x00000409
    });
    const {
      authentication
    } = this.config;

    switch (authentication.type) {
      case 'azure-active-directory-password':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'default'
        };
        break;

      case 'azure-active-directory-access-token':
        payload.fedAuth = {
          type: 'SECURITYTOKEN',
          echo: this.fedAuthRequired,
          fedAuthToken: authentication.options.token
        };
        break;

      case 'azure-active-directory-msi-vm':
      case 'azure-active-directory-default':
      case 'azure-active-directory-msi-app-service':
      case 'azure-active-directory-service-principal-secret':
        payload.fedAuth = {
          type: 'ADAL',
          echo: this.fedAuthRequired,
          workflow: 'integrated'
        };
        break;

      case 'ntlm':
        payload.sspi = (0, _ntlm.createNTLMRequest)({
          domain: authentication.options.domain
        });
        break;

      default:
        payload.userName = authentication.options.userName;
        payload.password = authentication.options.password;
    }

    payload.hostname = this.config.options.workstationId || _os.default.hostname();
    payload.serverName = this.routingData ? this.routingData.server : this.config.server;
    payload.appName = this.config.options.appName || 'Tedious';
    payload.libraryName = _library.name;
    payload.language = this.config.options.language;
    payload.database = this.config.options.database;
    payload.clientId = Buffer.from([1, 2, 3, 4, 5, 6]);
    payload.readOnlyIntent = this.config.options.readOnlyIntent;
    payload.initDbFatal = !this.config.options.fallbackToDefaultDb;
    this.routingData = undefined;
    this.messageIo.sendMessage(_packet.TYPE.LOGIN7, payload.toBuffer());
    this.debug.payload(function () {
      return payload.toString('  ');
    });
  }
  /**
   * @private
   */


  sendFedAuthTokenMessage(token) {
    const accessTokenLen = Buffer.byteLength(token, 'ucs2');
    const data = Buffer.alloc(8 + accessTokenLen);
    let offset = 0;
    offset = data.writeUInt32LE(accessTokenLen + 4, offset);
    offset = data.writeUInt32LE(accessTokenLen, offset);
    data.write(token, offset, 'ucs2');
    this.messageIo.sendMessage(_packet.TYPE.FEDAUTH_TOKEN, data); // sent the fedAuth token message, the rest is similar to standard login 7

    this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
  }
  /**
   * @private
   */


  sendInitialSql() {
    const payload = new _sqlbatchPayload.default(this.getInitialSql(), this.currentTransactionDescriptor(), this.config.options);
    const message = new _message.default({
      type: _packet.TYPE.SQL_BATCH
    });
    this.messageIo.outgoingMessageStream.write(message);

    _stream.Readable.from(payload).pipe(message);
  }
  /**
   * @private
   */


  getInitialSql() {
    const options = [];

    if (this.config.options.enableAnsiNull === true) {
      options.push('set ansi_nulls on');
    } else if (this.config.options.enableAnsiNull === false) {
      options.push('set ansi_nulls off');
    }

    if (this.config.options.enableAnsiNullDefault === true) {
      options.push('set ansi_null_dflt_on on');
    } else if (this.config.options.enableAnsiNullDefault === false) {
      options.push('set ansi_null_dflt_on off');
    }

    if (this.config.options.enableAnsiPadding === true) {
      options.push('set ansi_padding on');
    } else if (this.config.options.enableAnsiPadding === false) {
      options.push('set ansi_padding off');
    }

    if (this.config.options.enableAnsiWarnings === true) {
      options.push('set ansi_warnings on');
    } else if (this.config.options.enableAnsiWarnings === false) {
      options.push('set ansi_warnings off');
    }

    if (this.config.options.enableArithAbort === true) {
      options.push('set arithabort on');
    } else if (this.config.options.enableArithAbort === false) {
      options.push('set arithabort off');
    }

    if (this.config.options.enableConcatNullYieldsNull === true) {
      options.push('set concat_null_yields_null on');
    } else if (this.config.options.enableConcatNullYieldsNull === false) {
      options.push('set concat_null_yields_null off');
    }

    if (this.config.options.enableCursorCloseOnCommit === true) {
      options.push('set cursor_close_on_commit on');
    } else if (this.config.options.enableCursorCloseOnCommit === false) {
      options.push('set cursor_close_on_commit off');
    }

    if (this.config.options.datefirst !== null) {
      options.push(`set datefirst ${this.config.options.datefirst}`);
    }

    if (this.config.options.dateFormat !== null) {
      options.push(`set dateformat ${this.config.options.dateFormat}`);
    }

    if (this.config.options.enableImplicitTransactions === true) {
      options.push('set implicit_transactions on');
    } else if (this.config.options.enableImplicitTransactions === false) {
      options.push('set implicit_transactions off');
    }

    if (this.config.options.language !== null) {
      options.push(`set language ${this.config.options.language}`);
    }

    if (this.config.options.enableNumericRoundabort === true) {
      options.push('set numeric_roundabort on');
    } else if (this.config.options.enableNumericRoundabort === false) {
      options.push('set numeric_roundabort off');
    }

    if (this.config.options.enableQuotedIdentifier === true) {
      options.push('set quoted_identifier on');
    } else if (this.config.options.enableQuotedIdentifier === false) {
      options.push('set quoted_identifier off');
    }

    if (this.config.options.textsize !== null) {
      options.push(`set textsize ${this.config.options.textsize}`);
    }

    if (this.config.options.connectionIsolationLevel !== null) {
      options.push(`set transaction isolation level ${this.getIsolationLevelText(this.config.options.connectionIsolationLevel)}`);
    }

    if (this.config.options.abortTransactionOnError === true) {
      options.push('set xact_abort on');
    } else if (this.config.options.abortTransactionOnError === false) {
      options.push('set xact_abort off');
    }

    return options.join('\n');
  }
  /**
   * @private
   */


  processedInitialSql() {
    this.clearConnectTimer();
    this.emit('connect');
  }
  /**
   * Execute the SQL batch represented by [[Request]].
   * There is no param support, and unlike [[Request.execSql]],
   * it is not likely that SQL Server will reuse the execution plan it generates for the SQL.
   *
   * In almost all cases, [[Request.execSql]] will be a better choice.
   *
   * @param request A [[Request]] object representing the request.
   */


  execSqlBatch(request) {
    this.makeRequest(request, _packet.TYPE.SQL_BATCH, new _sqlbatchPayload.default(request.sqlTextOrProcedure, this.currentTransactionDescriptor(), this.config.options));
  }
  /**
   *  Execute the SQL represented by [[Request]].
   *
   * As `sp_executesql` is used to execute the SQL, if the same SQL is executed multiples times
   * using this function, the SQL Server query optimizer is likely to reuse the execution plan it generates
   * for the first execution. This may also result in SQL server treating the request like a stored procedure
   * which can result in the [[Event_doneInProc]] or [[Event_doneProc]] events being emitted instead of the
   * [[Event_done]] event you might expect. Using [[execSqlBatch]] will prevent this from occurring but may have a negative performance impact.
   *
   * Beware of the way that scoping rules apply, and how they may [affect local temp tables](http://weblogs.sqlteam.com/mladenp/archive/2006/11/03/17197.aspx)
   * If you're running in to scoping issues, then [[execSqlBatch]] may be a better choice.
   * See also [issue #24](https://github.com/pekim/tedious/issues/24)
   *
   * @param request A [[Request]] object representing the request.
   */


  execSql(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'statement',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    if (request.parameters.length) {
      parameters.push({
        type: _dataType.TYPES.NVarChar,
        name: 'params',
        value: request.makeParamsParameter(request.parameters),
        output: false,
        length: undefined,
        precision: undefined,
        scale: undefined
      });
      parameters.push(...request.parameters);
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_executesql', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Creates a new BulkLoad instance.
   *
   * @param table The name of the table to bulk-insert into.
   * @param options A set of bulk load options.
   */


  newBulkLoad(table, callbackOrOptions, callback) {
    let options;

    if (callback === undefined) {
      callback = callbackOrOptions;
      options = {};
    } else {
      options = callbackOrOptions;
    }

    if (typeof options !== 'object') {
      throw new TypeError('"options" argument must be an object');
    }

    return new _bulkLoad.default(table, this.databaseCollation, this.config.options, options, callback);
  }
  /**
   * Execute a [[BulkLoad]].
   *
   * ```js
   * // We want to perform a bulk load into a table with the following format:
   * // CREATE TABLE employees (first_name nvarchar(255), last_name nvarchar(255), day_of_birth date);
   *
   * const bulkLoad = connection.newBulkLoad('employees', (err, rowCount) => {
   *   // ...
   * });
   *
   * // First, we need to specify the columns that we want to write to,
   * // and their definitions. These definitions must match the actual table,
   * // otherwise the bulk load will fail.
   * bulkLoad.addColumn('first_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('last_name', TYPES.NVarchar, { nullable: false });
   * bulkLoad.addColumn('date_of_birth', TYPES.Date, { nullable: false });
   *
   * // Execute a bulk load with a predefined list of rows.
   * //
   * // Note that these rows are held in memory until the
   * // bulk load was performed, so if you need to write a large
   * // number of rows (e.g. by reading from a CSV file),
   * // passing an `AsyncIterable` is advisable to keep memory usage low.
   * connection.execBulkLoad(bulkLoad, [
   *   { 'first_name': 'Steve', 'last_name': 'Jobs', 'day_of_birth': new Date('02-24-1955') },
   *   { 'first_name': 'Bill', 'last_name': 'Gates', 'day_of_birth': new Date('10-28-1955') }
   * ]);
   * ```
   *
   * @param bulkLoad A previously created [[BulkLoad]].
   * @param rows A [[Iterable]] or [[AsyncIterable]] that contains the rows that should be bulk loaded.
   */


  execBulkLoad(bulkLoad, rows) {
    bulkLoad.executionStarted = true;

    if (rows) {
      if (bulkLoad.streamingMode) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that was put in streaming mode.");
      }

      if (bulkLoad.firstRowWritten) {
        throw new Error("Connection.execBulkLoad can't be called with a BulkLoad that already has rows written to it.");
      }

      const rowStream = _stream.Readable.from(rows); // Destroy the packet transform if an error happens in the row stream,
      // e.g. if an error is thrown from within a generator or stream.


      rowStream.on('error', err => {
        bulkLoad.rowToPacketTransform.destroy(err);
      }); // Destroy the row stream if an error happens in the packet transform,
      // e.g. if the bulk load is cancelled.

      bulkLoad.rowToPacketTransform.on('error', err => {
        rowStream.destroy(err);
      });
      rowStream.pipe(bulkLoad.rowToPacketTransform);
    } else if (!bulkLoad.streamingMode) {
      // If the bulkload was not put into streaming mode by the user,
      // we end the rowToPacketTransform here for them.
      //
      // If it was put into streaming mode, it's the user's responsibility
      // to end the stream.
      bulkLoad.rowToPacketTransform.end();
    }

    const onCancel = () => {
      request.cancel();
    };

    const payload = new _bulkLoadPayload.BulkLoadPayload(bulkLoad);
    const request = new _request.default(bulkLoad.getBulkInsertSql(), error => {
      bulkLoad.removeListener('cancel', onCancel);

      if (error) {
        if (error.code === 'UNKNOWN') {
          error.message += ' This is likely because the schema of the BulkLoad does not match the schema of the table you are attempting to insert into.';
        }

        bulkLoad.error = error;
        bulkLoad.callback(error);
        return;
      }

      this.makeRequest(bulkLoad, _packet.TYPE.BULK_LOAD, payload);
    });
    bulkLoad.once('cancel', onCancel);
    this.execSqlBatch(request);
  }
  /**
   * Prepare the SQL represented by the request.
   *
   * The request can then be used in subsequent calls to
   * [[execute]] and [[unprepare]]
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type. Parameter values are ignored.
   */


  prepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      value: undefined,
      output: true,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'params',
      value: request.parameters.length ? request.makeParamsParameter(request.parameters) : null,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    parameters.push({
      type: _dataType.TYPES.NVarChar,
      name: 'stmt',
      value: request.sqlTextOrProcedure,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    request.preparing = true; // TODO: We need to clean up this event handler, otherwise this leaks memory

    request.on('returnValue', (name, value) => {
      if (name === 'handle') {
        request.handle = value;
      } else {
        request.error = new _errors.RequestError(`Tedious > Unexpected output parameter ${name} from sp_prepare`);
      }
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_prepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Release the SQL Server resources associated with a previously prepared request.
   *
   * @param request A [[Request]] object representing the request.
   *   Parameters only require a name and type.
   *   Parameter values are ignored.
   */


  unprepare(request) {
    const parameters = [];
    parameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });
    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_unprepare', parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Execute previously prepared SQL, using the supplied parameters.
   *
   * @param request A previously prepared [[Request]].
   * @param parameters  An object whose names correspond to the names of
   *   parameters that were added to the [[Request]] before it was prepared.
   *   The object's values are passed as the parameters' values when the
   *   request is executed.
   */


  execute(request, parameters) {
    const executeParameters = [];
    executeParameters.push({
      type: _dataType.TYPES.Int,
      name: 'handle',
      // TODO: Abort if `request.handle` is not set
      value: request.handle,
      output: false,
      length: undefined,
      precision: undefined,
      scale: undefined
    });

    try {
      for (let i = 0, len = request.parameters.length; i < len; i++) {
        const parameter = request.parameters[i];
        executeParameters.push({ ...parameter,
          value: parameter.type.validate(parameters ? parameters[parameter.name] : null, this.databaseCollation)
        });
      }
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default('sp_execute', executeParameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Call a stored procedure represented by [[Request]].
   *
   * @param request A [[Request]] object representing the request.
   */


  callProcedure(request) {
    try {
      request.validateParameters(this.databaseCollation);
    } catch (error) {
      request.error = error;
      process.nextTick(() => {
        this.debug.log(error.message);
        request.callback(error);
      });
      return;
    }

    this.makeRequest(request, _packet.TYPE.RPC_REQUEST, new _rpcrequestPayload.default(request.sqlTextOrProcedure, request.parameters, this.currentTransactionDescriptor(), this.config.options, this.databaseCollation));
  }
  /**
   * Start a transaction.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`
   *   is present.
   * @param isolationLevel The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  beginTransaction(callback, name = '', isolationLevel = this.config.options.isolationLevel) {
    (0, _transaction.assertValidIsolationLevel)(isolationLevel, 'isolationLevel');
    const transaction = new _transaction.Transaction(name, isolationLevel);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SET TRANSACTION ISOLATION LEVEL ' + transaction.isolationLevelToTSQL() + ';BEGIN TRAN ' + transaction.name, err => {
        this.transactionDepth++;

        if (this.transactionDepth === 1) {
          this.inTransaction = true;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, err => {
      return callback(err, this.currentTransactionDescriptor());
    });
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.beginPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Commit a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string. Required when `isolationLevel`is present.
   */


  commitTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('COMMIT TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.commitPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Rollback a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  rollbackTransaction(callback, name = '') {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('ROLLBACK TRAN ' + transaction.name, err => {
        this.transactionDepth--;

        if (this.transactionDepth === 0) {
          this.inTransaction = false;
        }

        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.rollbackPayload(this.currentTransactionDescriptor()));
  }
  /**
   * Set a savepoint within a transaction.
   *
   * There should be an active transaction - that is, [[beginTransaction]]
   * should have been previously called.
   *
   * @param callback
   * @param name A string representing a name to associate with the transaction.\
   *   Optional, and defaults to an empty string.
   *   Required when `isolationLevel` is present.
   */


  saveTransaction(callback, name) {
    const transaction = new _transaction.Transaction(name);

    if (this.config.options.tdsVersion < '7_2') {
      return this.execSqlBatch(new _request.default('SAVE TRAN ' + transaction.name, err => {
        this.transactionDepth++;
        callback(err);
      }));
    }

    const request = new _request.default(undefined, callback);
    return this.makeRequest(request, _packet.TYPE.TRANSACTION_MANAGER, transaction.savePayload(this.currentTransactionDescriptor()));
  }
  /**
   * Run the given callback after starting a transaction, and commit or
   * rollback the transaction afterwards.
   *
   * This is a helper that employs [[beginTransaction]], [[commitTransaction]],
   * [[rollbackTransaction]], and [[saveTransaction]] to greatly simplify the
   * use of database transactions and automatically handle transaction nesting.
   *
   * @param cb
   * @param isolationLevel
   *   The isolation level that the transaction is to be run with.
   *
   *   The isolation levels are available from `require('tedious').ISOLATION_LEVEL`.
   *   * `READ_UNCOMMITTED`
   *   * `READ_COMMITTED`
   *   * `REPEATABLE_READ`
   *   * `SERIALIZABLE`
   *   * `SNAPSHOT`
   *
   *   Optional, and defaults to the Connection's isolation level.
   */


  transaction(cb, isolationLevel) {
    if (typeof cb !== 'function') {
      throw new TypeError('`cb` must be a function');
    }

    const useSavepoint = this.inTransaction;

    const name = '_tedious_' + _crypto.default.randomBytes(10).toString('hex');

    const txDone = (err, done, ...args) => {
      if (err) {
        if (this.inTransaction && this.state === this.STATE.LOGGED_IN) {
          this.rollbackTransaction(txErr => {
            done(txErr || err, ...args);
          }, name);
        } else {
          done(err, ...args);
        }
      } else if (useSavepoint) {
        if (this.config.options.tdsVersion < '7_2') {
          this.transactionDepth--;
        }

        done(null, ...args);
      } else {
        this.commitTransaction(txErr => {
          done(txErr, ...args);
        }, name);
      }
    };

    if (useSavepoint) {
      return this.saveTransaction(err => {
        if (err) {
          return cb(err);
        }

        if (isolationLevel) {
          return this.execSqlBatch(new _request.default('SET transaction isolation level ' + this.getIsolationLevelText(isolationLevel), err => {
            return cb(err, txDone);
          }));
        } else {
          return cb(null, txDone);
        }
      }, name);
    } else {
      return this.beginTransaction(err => {
        if (err) {
          return cb(err);
        }

        return cb(null, txDone);
      }, name, isolationLevel);
    }
  }
  /**
   * @private
   */


  makeRequest(request, packetType, payload) {
    if (this.state !== this.STATE.LOGGED_IN) {
      const message = 'Requests can only be made in the ' + this.STATE.LOGGED_IN.name + ' state, not the ' + this.state.name + ' state';
      this.debug.log(message);
      request.callback(new _errors.RequestError(message, 'EINVALIDSTATE'));
    } else if (request.canceled) {
      process.nextTick(() => {
        request.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
      });
    } else {
      if (packetType === _packet.TYPE.SQL_BATCH) {
        this.isSqlBatch = true;
      } else {
        this.isSqlBatch = false;
      }

      this.request = request;
      request.connection = this;
      request.rowCount = 0;
      request.rows = [];
      request.rst = [];

      const onCancel = () => {
        payloadStream.unpipe(message);
        payloadStream.destroy(new _errors.RequestError('Canceled.', 'ECANCEL')); // set the ignore bit and end the message.

        message.ignore = true;
        message.end();

        if (request instanceof _request.default && request.paused) {
          // resume the request if it was paused so we can read the remaining tokens
          request.resume();
        }
      };

      request.once('cancel', onCancel);
      this.createRequestTimer();
      const message = new _message.default({
        type: packetType,
        resetConnection: this.resetConnectionOnNextRequest
      });
      this.messageIo.outgoingMessageStream.write(message);
      this.transitionTo(this.STATE.SENT_CLIENT_REQUEST);
      message.once('finish', () => {
        request.removeListener('cancel', onCancel);
        request.once('cancel', this._cancelAfterRequestSent);
        this.resetConnectionOnNextRequest = false;
        this.debug.payload(function () {
          return payload.toString('  ');
        });
      });

      const payloadStream = _stream.Readable.from(payload);

      payloadStream.once('error', error => {
        var _request$error;

        payloadStream.unpipe(message); // Only set a request error if no error was set yet.

        (_request$error = request.error) !== null && _request$error !== void 0 ? _request$error : request.error = error;
        message.ignore = true;
        message.end();
      });
      payloadStream.pipe(message);
    }
  }
  /**
   * Cancel currently executed request.
   */


  cancel() {
    if (!this.request) {
      return false;
    }

    if (this.request.canceled) {
      return false;
    }

    this.request.cancel();
    return true;
  }
  /**
   * Reset the connection to its initial state.
   * Can be useful for connection pool implementations.
   *
   * @param callback
   */


  reset(callback) {
    const request = new _request.default(this.getInitialSql(), err => {
      if (this.config.options.tdsVersion < '7_2') {
        this.inTransaction = false;
      }

      callback(err);
    });
    this.resetConnectionOnNextRequest = true;
    this.execSqlBatch(request);
  }
  /**
   * @private
   */


  currentTransactionDescriptor() {
    return this.transactionDescriptors[this.transactionDescriptors.length - 1];
  }
  /**
   * @private
   */


  getIsolationLevelText(isolationLevel) {
    switch (isolationLevel) {
      case _transaction.ISOLATION_LEVEL.READ_UNCOMMITTED:
        return 'read uncommitted';

      case _transaction.ISOLATION_LEVEL.REPEATABLE_READ:
        return 'repeatable read';

      case _transaction.ISOLATION_LEVEL.SERIALIZABLE:
        return 'serializable';

      case _transaction.ISOLATION_LEVEL.SNAPSHOT:
        return 'snapshot';

      default:
        return 'read committed';
    }
  }

}

let azureADPasswordClientIdDeprecationWarningEmitted = false;

function emitAzureADPasswordClientIdDeprecationWarning() {
  if (azureADPasswordClientIdDeprecationWarningEmitted) {
    return;
  }

  azureADPasswordClientIdDeprecationWarningEmitted = true;
  process.emitWarning('When using the `azure-active-directory-password` authentication method, please provide a value for the `clientId` option. ' + 'This option will be required in a future release.', 'DeprecationWarning', Connection.prototype.on);
}

function isTransientError(error) {
  if (error instanceof _esAggregateError.default) {
    error = error.errors[0];
  }

  return error instanceof _errors.ConnectionError && !!error.isTransient;
}

var _default = Connection;
exports.default = _default;
module.exports = Connection;
Connection.prototype.STATE = {
  INITIALIZED: {
    name: 'Initialized',
    events: {}
  },
  CONNECTING: {
    name: 'Connecting',
    enter: function () {
      this.initialiseConnection();
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_PRELOGIN: {
    name: 'SentPrelogin',
    enter: function () {
      this.emptyMessageBuffer();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        message.on('data', data => {
          this.addToMessageBuffer(data);
        });
        message.once('end', () => {
          const preloginPayload = new _preloginPayload.default(this.messageBuffer);
          this.debug.payload(function () {
            return preloginPayload.toString('  ');
          });

          if (preloginPayload.fedAuthRequired === 1) {
            this.fedAuthRequired = true;
          }

          if (preloginPayload.encryptionString === 'ON' || preloginPayload.encryptionString === 'REQ') {
            var _this$routingData$ser, _this$routingData;

            if (!this.config.options.encrypt) {
              this.emit('connect', new _errors.ConnectionError("Server requires encryption, set 'encrypt' config option to true.", 'EENCRYPT'));
              return this.close();
            }

            this.messageIo.startTls(this.secureContext, (_this$routingData$ser = (_this$routingData = this.routingData) === null || _this$routingData === void 0 ? void 0 : _this$routingData.server) !== null && _this$routingData$ser !== void 0 ? _this$routingData$ser : this.config.server, this.config.options.trustServerCertificate);
            this.transitionTo(this.STATE.SENT_TLSSSLNEGOTIATION);
          } else {
            this.sendLogin7Packet();
            const {
              authentication
            } = this.config;

            if (authentication.type === 'ntlm') {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
            } else {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
            }
          }
        });
      }
    }
  },
  REROUTING: {
    name: 'ReRouting',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.REDIRECT);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      reconnect: function () {
        this.transitionTo(this.STATE.CONNECTING);
      }
    }
  },
  TRANSIENT_FAILURE_RETRY: {
    name: 'TRANSIENT_FAILURE_RETRY',
    enter: function () {
      this.curTransientRetryCount++;
      this.cleanupConnection(CLEANUP_TYPE.RETRY);
    },
    events: {
      message: function () {},
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      retry: function () {
        this.createRetryTimer();
      }
    }
  },
  SENT_TLSSSLNEGOTIATION: {
    name: 'SentTLSSSLNegotiation',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        message.on('data', data => {
          this.messageIo.tlsHandshakeData(data);
        });
        message.once('end', () => {
          if (this.messageIo.tlsNegotiationComplete) {
            this.sendLogin7Packet();
            const {
              authentication
            } = this.config;

            if (authentication.type === 'azure-active-directory-password' || authentication.type === 'azure-active-directory-msi-vm' || authentication.type === 'azure-active-directory-msi-app-service' || authentication.type === 'azure-active-directory-service-principal-secret' || authentication.type === 'azure-active-directory-default') {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_FEDAUTH);
            } else if (authentication.type === 'ntlm') {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_NTLM);
            } else {
              this.transitionTo(this.STATE.SENT_LOGIN7_WITH_STANDARD_LOGIN);
            }
          } else {
            this.messageIo.readMessage().then(message => {
              this.dispatchEvent('message', message);
            }, err => {
              this.socketError(err);
            });
          }
        });
      }
    }
  },
  SENT_LOGIN7_WITH_STANDARD_LOGIN: {
    name: 'SentLogin7WithStandardLogin',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              this.transitionTo(this.STATE.REROUTING);
            } else {
              this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }
          } else if (this.loginError) {
            if (isTransientError(this.loginError)) {
              this.debug.log('Initiating retry on transient error');
              this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            this.transitionTo(this.STATE.FINAL);
          }
        });
      }
    }
  },
  SENT_LOGIN7_WITH_NTLM: {
    name: 'SentLogin7WithNTLMLogin',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              this.transitionTo(this.STATE.REROUTING);
            } else {
              this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }
          } else if (this.ntlmpacket) {
            const authentication = this.config.authentication;
            const payload = new _ntlmPayload.default({
              domain: authentication.options.domain,
              userName: authentication.options.userName,
              password: authentication.options.password,
              ntlmpacket: this.ntlmpacket
            });
            this.messageIo.sendMessage(_packet.TYPE.NTLMAUTH_PKT, payload.data);
            this.debug.payload(function () {
              return payload.toString('  ');
            });
            this.ntlmpacket = undefined;
            this.messageIo.readMessage().then(message => {
              this.dispatchEvent('message', message);
            }, err => {
              this.socketError(err);
            });
          } else if (this.loginError) {
            if (isTransientError(this.loginError)) {
              this.debug.log('Initiating retry on transient error');
              this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            this.transitionTo(this.STATE.FINAL);
          }
        });
      }
    }
  },
  SENT_LOGIN7_WITH_FEDAUTH: {
    name: 'SentLogin7Withfedauth',
    enter: function () {
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const handler = new _handler.Login7TokenHandler(this);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          if (handler.loginAckReceived) {
            if (handler.routingData) {
              this.routingData = handler.routingData;
              this.transitionTo(this.STATE.REROUTING);
            } else {
              this.transitionTo(this.STATE.LOGGED_IN_SENDING_INITIAL_SQL);
            }

            return;
          }

          const fedAuthInfoToken = handler.fedAuthInfoToken;

          if (fedAuthInfoToken && fedAuthInfoToken.stsurl && fedAuthInfoToken.spn) {
            const authentication = this.config.authentication;
            const tokenScope = new _url.URL('/.default', fedAuthInfoToken.spn).toString();

            const getToken = callback => {
              const getTokenFromCredentials = credentials => {
                credentials.getToken(tokenScope).then(tokenResponse => {
                  callback(null, tokenResponse === null || tokenResponse === void 0 ? void 0 : tokenResponse.token);
                }, callback);
              };

              if (authentication.type === 'azure-active-directory-password') {
                var _authentication$optio;

                const credentials = new _identity.UsernamePasswordCredential((_authentication$optio = authentication.options.tenantId) !== null && _authentication$optio !== void 0 ? _authentication$optio : 'common', authentication.options.clientId, authentication.options.userName, authentication.options.password);
                getTokenFromCredentials(credentials);
              } else if (authentication.type === 'azure-active-directory-msi-vm' || authentication.type === 'azure-active-directory-msi-app-service') {
                const msiArgs = authentication.options.clientId ? [authentication.options.clientId, {}] : [{}];
                const credentials = new _identity.ManagedIdentityCredential(...msiArgs);
                getTokenFromCredentials(credentials);
              } else if (authentication.type === 'azure-active-directory-default') {
                const args = authentication.options.clientId ? {
                  managedIdentityClientId: authentication.options.clientId
                } : {};
                const credentials = new _identity.DefaultAzureCredential(args);
                getTokenFromCredentials(credentials);
              } else if (authentication.type === 'azure-active-directory-service-principal-secret') {
                const credentials = new _identity.ClientSecretCredential(authentication.options.tenantId, authentication.options.clientId, authentication.options.clientSecret);
                getTokenFromCredentials(credentials);
              }
            };

            getToken((err, token) => {
              if (err) {
                this.loginError = new _esAggregateError.default([new _errors.ConnectionError('Security token could not be authenticated or authorized.', 'EFEDAUTH'), err]);
                this.emit('connect', this.loginError);
                this.transitionTo(this.STATE.FINAL);
                return;
              }

              this.sendFedAuthTokenMessage(token);
            });
          } else if (this.loginError) {
            if (isTransientError(this.loginError)) {
              this.debug.log('Initiating retry on transient error');
              this.transitionTo(this.STATE.TRANSIENT_FAILURE_RETRY);
            } else {
              this.emit('connect', this.loginError);
              this.transitionTo(this.STATE.FINAL);
            }
          } else {
            this.emit('connect', new _errors.ConnectionError('Login failed.', 'ELOGIN'));
            this.transitionTo(this.STATE.FINAL);
          }
        });
      }
    }
  },
  LOGGED_IN_SENDING_INITIAL_SQL: {
    name: 'LoggedInSendingInitialSql',
    enter: function () {
      this.sendInitialSql();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function socketError() {
        this.transitionTo(this.STATE.FINAL);
      },
      connectTimeout: function () {
        this.transitionTo(this.STATE.FINAL);
      },
      message: function (message) {
        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.InitialSqlTokenHandler(this));
        tokenStreamParser.once('end', () => {
          this.transitionTo(this.STATE.LOGGED_IN);
          this.processedInitialSql();
        });
      }
    }
  },
  LOGGED_IN: {
    name: 'LoggedIn',
    events: {
      socketError: function () {
        this.transitionTo(this.STATE.FINAL);
      }
    }
  },
  SENT_CLIENT_REQUEST: {
    name: 'SentClientRequest',
    enter: function () {
      this.emptyMessageBuffer();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    exit: function (nextState) {
      this.clearRequestTimer();
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      },
      message: function (message) {
        var _this$request, _this$request3, _this$request10;

        // request timer is stopped on first data package
        this.clearRequestTimer();
        const tokenStreamParser = this.createTokenStreamParser(message, new _handler.RequestTokenHandler(this, this.request)); // If the request was canceled and we have a `cancelTimer`
        // defined, we send a attention message after the
        // request message was fully sent off.
        //
        // We already started consuming the current message
        // (but all the token handlers should be no-ops), and
        // need to ensure the next message is handled by the
        // `SENT_ATTENTION` state.

        if ((_this$request = this.request) !== null && _this$request !== void 0 && _this$request.canceled && this.cancelTimer) {
          return this.transitionTo(this.STATE.SENT_ATTENTION);
        }

        const onResume = () => {
          tokenStreamParser.resume();
        };

        const onPause = () => {
          var _this$request2;

          tokenStreamParser.pause();
          (_this$request2 = this.request) === null || _this$request2 === void 0 ? void 0 : _this$request2.once('resume', onResume);
        };

        (_this$request3 = this.request) === null || _this$request3 === void 0 ? void 0 : _this$request3.on('pause', onPause);

        if (this.request instanceof _request.default && this.request.paused) {
          onPause();
        }

        const onCancel = () => {
          var _this$request4, _this$request5;

          tokenStreamParser.removeListener('end', onEndOfMessage);

          if (this.request instanceof _request.default && this.request.paused) {
            // resume the request if it was paused so we can read the remaining tokens
            this.request.resume();
          }

          (_this$request4 = this.request) === null || _this$request4 === void 0 ? void 0 : _this$request4.removeListener('pause', onPause);
          (_this$request5 = this.request) === null || _this$request5 === void 0 ? void 0 : _this$request5.removeListener('resume', onResume); // The `_cancelAfterRequestSent` callback will have sent a
          // attention message, so now we need to also switch to
          // the `SENT_ATTENTION` state to make sure the attention ack
          // message is processed correctly.

          this.transitionTo(this.STATE.SENT_ATTENTION);
        };

        const onEndOfMessage = () => {
          var _this$request6, _this$request7, _this$request8, _this$request9;

          (_this$request6 = this.request) === null || _this$request6 === void 0 ? void 0 : _this$request6.removeListener('cancel', this._cancelAfterRequestSent);
          (_this$request7 = this.request) === null || _this$request7 === void 0 ? void 0 : _this$request7.removeListener('cancel', onCancel);
          (_this$request8 = this.request) === null || _this$request8 === void 0 ? void 0 : _this$request8.removeListener('pause', onPause);
          (_this$request9 = this.request) === null || _this$request9 === void 0 ? void 0 : _this$request9.removeListener('resume', onResume);
          this.transitionTo(this.STATE.LOGGED_IN);
          const sqlRequest = this.request;
          this.request = undefined;

          if (this.config.options.tdsVersion < '7_2' && sqlRequest.error && this.isSqlBatch) {
            this.inTransaction = false;
          }

          sqlRequest.callback(sqlRequest.error, sqlRequest.rowCount, sqlRequest.rows);
        };

        tokenStreamParser.once('end', onEndOfMessage);
        (_this$request10 = this.request) === null || _this$request10 === void 0 ? void 0 : _this$request10.once('cancel', onCancel);
      }
    }
  },
  SENT_ATTENTION: {
    name: 'SentAttention',
    enter: function () {
      this.emptyMessageBuffer();
      this.messageIo.readMessage().then(message => {
        this.dispatchEvent('message', message);
      }, err => {
        this.socketError(err);
      });
    },
    events: {
      socketError: function (err) {
        const sqlRequest = this.request;
        this.request = undefined;
        this.transitionTo(this.STATE.FINAL);
        sqlRequest.callback(err);
      },
      message: function (message) {
        const handler = new _handler.AttentionTokenHandler(this, this.request);
        const tokenStreamParser = this.createTokenStreamParser(message, handler);
        tokenStreamParser.once('end', () => {
          // 3.2.5.7 Sent Attention State
          // Discard any data contained in the response, until we receive the attention response
          if (handler.attentionReceived) {
            this.clearCancelTimer();
            const sqlRequest = this.request;
            this.request = undefined;
            this.transitionTo(this.STATE.LOGGED_IN);

            if (sqlRequest.error && sqlRequest.error instanceof _errors.RequestError && sqlRequest.error.code === 'ETIMEOUT') {
              sqlRequest.callback(sqlRequest.error);
            } else {
              sqlRequest.callback(new _errors.RequestError('Canceled.', 'ECANCEL'));
            }
          }
        });
      }
    }
  },
  FINAL: {
    name: 'Final',
    enter: function () {
      this.cleanupConnection(CLEANUP_TYPE.NORMAL);
    },
    events: {
      connectTimeout: function () {// Do nothing, as the timer should be cleaned up.
      },
      message: function () {// Do nothing
      },
      socketError: function () {// Do nothing
      }
    }
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uL3NyYy9jb25uZWN0aW9uLnRzIl0sIm5hbWVzIjpbInRydXN0U2VydmVyV2FybmluZ0VtaXR0ZWQiLCJlbWl0VHJ1c3RTZXJ2ZXJDZXJ0aWZpY2F0ZVdhcm5pbmciLCJwcm9jZXNzIiwiZW1pdFdhcm5pbmciLCJkb21haW5SZW5hbWVUb1RlbmFudElkV2FybmluZ0VtaXR0ZWQiLCJlbWl0RG9tYWluUmVuYW1lVG9UZW5hbnRJZFdhcm5pbmciLCJLRUVQX0FMSVZFX0lOSVRJQUxfREVMQVkiLCJERUZBVUxUX0NPTk5FQ1RfVElNRU9VVCIsIkRFRkFVTFRfQ0xJRU5UX1JFUVVFU1RfVElNRU9VVCIsIkRFRkFVTFRfQ0FOQ0VMX1RJTUVPVVQiLCJERUZBVUxUX0NPTk5FQ1RfUkVUUllfSU5URVJWQUwiLCJERUZBVUxUX1BBQ0tFVF9TSVpFIiwiREVGQVVMVF9URVhUU0laRSIsIkRFRkFVTFRfREFURUZJUlNUIiwiREVGQVVMVF9QT1JUIiwiREVGQVVMVF9URFNfVkVSU0lPTiIsIkRFRkFVTFRfTEFOR1VBR0UiLCJERUZBVUxUX0RBVEVGT1JNQVQiLCJDTEVBTlVQX1RZUEUiLCJOT1JNQUwiLCJSRURJUkVDVCIsIlJFVFJZIiwiQ29ubmVjdGlvbiIsIkV2ZW50RW1pdHRlciIsImNvbnN0cnVjdG9yIiwiY29uZmlnIiwiZmVkQXV0aFJlcXVpcmVkIiwic2VjdXJlQ29udGV4dCIsImluVHJhbnNhY3Rpb24iLCJ0cmFuc2FjdGlvbkRlc2NyaXB0b3JzIiwidHJhbnNhY3Rpb25EZXB0aCIsImlzU3FsQmF0Y2giLCJjdXJUcmFuc2llbnRSZXRyeUNvdW50IiwidHJhbnNpZW50RXJyb3JMb29rdXAiLCJjbG9zZWQiLCJsb2dpbkVycm9yIiwiZGVidWciLCJudGxtcGFja2V0IiwibnRsbXBhY2tldEJ1ZmZlciIsInJvdXRpbmdEYXRhIiwibWVzc2FnZUlvIiwic3RhdGUiLCJyZXNldENvbm5lY3Rpb25Pbk5leHRSZXF1ZXN0IiwicmVxdWVzdCIsInByb2NSZXR1cm5TdGF0dXNWYWx1ZSIsInNvY2tldCIsIm1lc3NhZ2VCdWZmZXIiLCJjb25uZWN0VGltZXIiLCJjYW5jZWxUaW1lciIsInJlcXVlc3RUaW1lciIsInJldHJ5VGltZXIiLCJfY2FuY2VsQWZ0ZXJSZXF1ZXN0U2VudCIsImRhdGFiYXNlQ29sbGF0aW9uIiwiVHlwZUVycm9yIiwic2VydmVyIiwiYXV0aGVudGljYXRpb24iLCJ1bmRlZmluZWQiLCJ0eXBlIiwib3B0aW9ucyIsImRvbWFpbiIsInVzZXJOYW1lIiwicGFzc3dvcmQiLCJ0b1VwcGVyQ2FzZSIsImNsaWVudElkIiwiZW1pdEF6dXJlQURQYXNzd29yZENsaWVudElkRGVwcmVjYXRpb25XYXJuaW5nIiwidGVuYW50SWQiLCJ0b2tlbiIsImNsaWVudFNlY3JldCIsImFib3J0VHJhbnNhY3Rpb25PbkVycm9yIiwiYXBwTmFtZSIsImNhbWVsQ2FzZUNvbHVtbnMiLCJjYW5jZWxUaW1lb3V0IiwiY29sdW1uRW5jcnlwdGlvbktleUNhY2hlVFRMIiwiY29sdW1uRW5jcnlwdGlvblNldHRpbmciLCJjb2x1bW5OYW1lUmVwbGFjZXIiLCJjb25uZWN0aW9uUmV0cnlJbnRlcnZhbCIsImNvbm5lY3RUaW1lb3V0IiwiY29ubmVjdGlvbklzb2xhdGlvbkxldmVsIiwiSVNPTEFUSU9OX0xFVkVMIiwiUkVBRF9DT01NSVRURUQiLCJjcnlwdG9DcmVkZW50aWFsc0RldGFpbHMiLCJkYXRhYmFzZSIsImRhdGVmaXJzdCIsImRhdGVGb3JtYXQiLCJkYXRhIiwicGFja2V0IiwicGF5bG9hZCIsImVuYWJsZUFuc2lOdWxsIiwiZW5hYmxlQW5zaU51bGxEZWZhdWx0IiwiZW5hYmxlQW5zaVBhZGRpbmciLCJlbmFibGVBbnNpV2FybmluZ3MiLCJlbmFibGVBcml0aEFib3J0IiwiZW5hYmxlQ29uY2F0TnVsbFlpZWxkc051bGwiLCJlbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0IiwiZW5hYmxlSW1wbGljaXRUcmFuc2FjdGlvbnMiLCJlbmFibGVOdW1lcmljUm91bmRhYm9ydCIsImVuYWJsZVF1b3RlZElkZW50aWZpZXIiLCJlbmNyeXB0IiwiZmFsbGJhY2tUb0RlZmF1bHREYiIsImVuY3J5cHRpb25LZXlTdG9yZVByb3ZpZGVycyIsImluc3RhbmNlTmFtZSIsImlzb2xhdGlvbkxldmVsIiwibGFuZ3VhZ2UiLCJsb2NhbEFkZHJlc3MiLCJtYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnMiLCJtdWx0aVN1Ym5ldEZhaWxvdmVyIiwicGFja2V0U2l6ZSIsInBvcnQiLCJyZWFkT25seUludGVudCIsInJlcXVlc3RUaW1lb3V0Iiwicm93Q29sbGVjdGlvbk9uRG9uZSIsInJvd0NvbGxlY3Rpb25PblJlcXVlc3RDb21wbGV0aW9uIiwic2VydmVyTmFtZSIsInNlcnZlclN1cHBvcnRzQ29sdW1uRW5jcnlwdGlvbiIsInRkc1ZlcnNpb24iLCJ0ZXh0c2l6ZSIsInRydXN0ZWRTZXJ2ZXJOYW1lQUUiLCJ0cnVzdFNlcnZlckNlcnRpZmljYXRlIiwidXNlQ29sdW1uTmFtZXMiLCJ1c2VVVEMiLCJ3b3Jrc3RhdGlvbklkIiwibG93ZXJDYXNlR3VpZHMiLCJFcnJvciIsIlJhbmdlRXJyb3IiLCJjcmVkZW50aWFsc0RldGFpbHMiLCJzZWN1cmVPcHRpb25zIiwiT2JqZWN0IiwiY3JlYXRlIiwidmFsdWUiLCJjb25zdGFudHMiLCJTU0xfT1BfRE9OVF9JTlNFUlRfRU1QVFlfRlJBR01FTlRTIiwiY3JlYXRlRGVidWciLCJCdWZmZXIiLCJmcm9tIiwiYWxsb2MiLCJUcmFuc2llbnRFcnJvckxvb2t1cCIsIlNUQVRFIiwiSU5JVElBTElaRUQiLCJzZW5kTWVzc2FnZSIsIlRZUEUiLCJBVFRFTlRJT04iLCJjcmVhdGVDYW5jZWxUaW1lciIsImNvbm5lY3QiLCJjb25uZWN0TGlzdGVuZXIiLCJDb25uZWN0aW9uRXJyb3IiLCJuYW1lIiwib25Db25uZWN0IiwiZXJyIiwicmVtb3ZlTGlzdGVuZXIiLCJvbkVycm9yIiwib25jZSIsInRyYW5zaXRpb25UbyIsIkNPTk5FQ1RJTkciLCJvbiIsImV2ZW50IiwibGlzdGVuZXIiLCJlbWl0IiwiYXJncyIsImNsb3NlIiwiRklOQUwiLCJpbml0aWFsaXNlQ29ubmVjdGlvbiIsInNpZ25hbCIsImNyZWF0ZUNvbm5lY3RUaW1lciIsImNvbm5lY3RPblBvcnQiLCJJbnN0YW5jZUxvb2t1cCIsImluc3RhbmNlTG9va3VwIiwidGltZW91dCIsIm1lc3NhZ2UiLCJjbGVhbnVwQ29ubmVjdGlvbiIsImNsZWFudXBUeXBlIiwiY2xlYXJDb25uZWN0VGltZXIiLCJjbGVhclJlcXVlc3RUaW1lciIsImNsZWFyUmV0cnlUaW1lciIsImNsb3NlQ29ubmVjdGlvbiIsIm5leHRUaWNrIiwiUmVxdWVzdEVycm9yIiwiY2FsbGJhY2siLCJEZWJ1ZyIsImNyZWF0ZVRva2VuU3RyZWFtUGFyc2VyIiwiaGFuZGxlciIsIlRva2VuU3RyZWFtUGFyc2VyIiwiY29ubmVjdE9wdHMiLCJob3N0IiwiQ29ubmVjdG9yIiwiZXhlY3V0ZSIsInNvY2tldEVycm9yIiwiZXJyb3IiLCJzb2NrZXRDbG9zZSIsInNvY2tldEVuZCIsInNldEtlZXBBbGl2ZSIsIk1lc3NhZ2VJTyIsImNsZWFydGV4dCIsImxvZyIsInNlbmRQcmVMb2dpbiIsIlNFTlRfUFJFTE9HSU4iLCJkZXN0cm95IiwiY29udHJvbGxlciIsIkFib3J0Q29udHJvbGxlciIsInNldFRpbWVvdXQiLCJhYm9ydCIsImNsZWFyQ2FuY2VsVGltZXIiLCJjcmVhdGVSZXF1ZXN0VGltZXIiLCJjcmVhdGVSZXRyeVRpbWVyIiwicmV0cnlUaW1lb3V0IiwiZGlzcGF0Y2hFdmVudCIsImNhbmNlbCIsImNsZWFyVGltZW91dCIsIm5ld1N0YXRlIiwiZXhpdCIsImNhbGwiLCJlbnRlciIsImFwcGx5IiwiZ2V0RXZlbnRIYW5kbGVyIiwiZXZlbnROYW1lIiwiZXZlbnRzIiwiU0VOVF9UTFNTU0xORUdPVElBVElPTiIsImNvZGUiLCJSRVJPVVRJTkciLCJUUkFOU0lFTlRfRkFJTFVSRV9SRVRSWSIsIm1ham9yIiwibWlub3IiLCJidWlsZCIsImV4ZWMiLCJ2ZXJzaW9uIiwiUHJlbG9naW5QYXlsb2FkIiwiTnVtYmVyIiwic3ViYnVpbGQiLCJQUkVMT0dJTiIsInRvU3RyaW5nIiwiZW1wdHlNZXNzYWdlQnVmZmVyIiwiYWRkVG9NZXNzYWdlQnVmZmVyIiwiY29uY2F0Iiwic2VuZExvZ2luN1BhY2tldCIsIkxvZ2luN1BheWxvYWQiLCJ2ZXJzaW9ucyIsImNsaWVudFByb2dWZXIiLCJjbGllbnRQaWQiLCJwaWQiLCJjb25uZWN0aW9uSWQiLCJjbGllbnRUaW1lWm9uZSIsIkRhdGUiLCJnZXRUaW1lem9uZU9mZnNldCIsImNsaWVudExjaWQiLCJmZWRBdXRoIiwiZWNobyIsIndvcmtmbG93IiwiZmVkQXV0aFRva2VuIiwic3NwaSIsImhvc3RuYW1lIiwib3MiLCJsaWJyYXJ5TmFtZSIsImluaXREYkZhdGFsIiwiTE9HSU43IiwidG9CdWZmZXIiLCJzZW5kRmVkQXV0aFRva2VuTWVzc2FnZSIsImFjY2Vzc1Rva2VuTGVuIiwiYnl0ZUxlbmd0aCIsIm9mZnNldCIsIndyaXRlVUludDMyTEUiLCJ3cml0ZSIsIkZFREFVVEhfVE9LRU4iLCJTRU5UX0xPR0lON19XSVRIX1NUQU5EQVJEX0xPR0lOIiwic2VuZEluaXRpYWxTcWwiLCJTcWxCYXRjaFBheWxvYWQiLCJnZXRJbml0aWFsU3FsIiwiY3VycmVudFRyYW5zYWN0aW9uRGVzY3JpcHRvciIsIk1lc3NhZ2UiLCJTUUxfQkFUQ0giLCJvdXRnb2luZ01lc3NhZ2VTdHJlYW0iLCJSZWFkYWJsZSIsInBpcGUiLCJwdXNoIiwiZ2V0SXNvbGF0aW9uTGV2ZWxUZXh0Iiwiam9pbiIsInByb2Nlc3NlZEluaXRpYWxTcWwiLCJleGVjU3FsQmF0Y2giLCJtYWtlUmVxdWVzdCIsInNxbFRleHRPclByb2NlZHVyZSIsImV4ZWNTcWwiLCJ2YWxpZGF0ZVBhcmFtZXRlcnMiLCJwYXJhbWV0ZXJzIiwiVFlQRVMiLCJOVmFyQ2hhciIsIm91dHB1dCIsImxlbmd0aCIsInByZWNpc2lvbiIsInNjYWxlIiwibWFrZVBhcmFtc1BhcmFtZXRlciIsIlJQQ19SRVFVRVNUIiwiUnBjUmVxdWVzdFBheWxvYWQiLCJuZXdCdWxrTG9hZCIsInRhYmxlIiwiY2FsbGJhY2tPck9wdGlvbnMiLCJCdWxrTG9hZCIsImV4ZWNCdWxrTG9hZCIsImJ1bGtMb2FkIiwicm93cyIsImV4ZWN1dGlvblN0YXJ0ZWQiLCJzdHJlYW1pbmdNb2RlIiwiZmlyc3RSb3dXcml0dGVuIiwicm93U3RyZWFtIiwicm93VG9QYWNrZXRUcmFuc2Zvcm0iLCJlbmQiLCJvbkNhbmNlbCIsIkJ1bGtMb2FkUGF5bG9hZCIsIlJlcXVlc3QiLCJnZXRCdWxrSW5zZXJ0U3FsIiwiQlVMS19MT0FEIiwicHJlcGFyZSIsIkludCIsInByZXBhcmluZyIsImhhbmRsZSIsInVucHJlcGFyZSIsImV4ZWN1dGVQYXJhbWV0ZXJzIiwiaSIsImxlbiIsInBhcmFtZXRlciIsInZhbGlkYXRlIiwiY2FsbFByb2NlZHVyZSIsImJlZ2luVHJhbnNhY3Rpb24iLCJ0cmFuc2FjdGlvbiIsIlRyYW5zYWN0aW9uIiwiaXNvbGF0aW9uTGV2ZWxUb1RTUUwiLCJUUkFOU0FDVElPTl9NQU5BR0VSIiwiYmVnaW5QYXlsb2FkIiwiY29tbWl0VHJhbnNhY3Rpb24iLCJjb21taXRQYXlsb2FkIiwicm9sbGJhY2tUcmFuc2FjdGlvbiIsInJvbGxiYWNrUGF5bG9hZCIsInNhdmVUcmFuc2FjdGlvbiIsInNhdmVQYXlsb2FkIiwiY2IiLCJ1c2VTYXZlcG9pbnQiLCJjcnlwdG8iLCJyYW5kb21CeXRlcyIsInR4RG9uZSIsImRvbmUiLCJMT0dHRURfSU4iLCJ0eEVyciIsInBhY2tldFR5cGUiLCJjYW5jZWxlZCIsImNvbm5lY3Rpb24iLCJyb3dDb3VudCIsInJzdCIsInBheWxvYWRTdHJlYW0iLCJ1bnBpcGUiLCJpZ25vcmUiLCJwYXVzZWQiLCJyZXN1bWUiLCJyZXNldENvbm5lY3Rpb24iLCJTRU5UX0NMSUVOVF9SRVFVRVNUIiwicmVzZXQiLCJSRUFEX1VOQ09NTUlUVEVEIiwiUkVQRUFUQUJMRV9SRUFEIiwiU0VSSUFMSVpBQkxFIiwiU05BUFNIT1QiLCJhenVyZUFEUGFzc3dvcmRDbGllbnRJZERlcHJlY2F0aW9uV2FybmluZ0VtaXR0ZWQiLCJwcm90b3R5cGUiLCJpc1RyYW5zaWVudEVycm9yIiwiQWdncmVnYXRlRXJyb3IiLCJlcnJvcnMiLCJpc1RyYW5zaWVudCIsIm1vZHVsZSIsImV4cG9ydHMiLCJyZWFkTWVzc2FnZSIsInRoZW4iLCJwcmVsb2dpblBheWxvYWQiLCJlbmNyeXB0aW9uU3RyaW5nIiwic3RhcnRUbHMiLCJTRU5UX0xPR0lON19XSVRIX05UTE0iLCJyZWNvbm5lY3QiLCJyZXRyeSIsInRsc0hhbmRzaGFrZURhdGEiLCJ0bHNOZWdvdGlhdGlvbkNvbXBsZXRlIiwiU0VOVF9MT0dJTjdfV0lUSF9GRURBVVRIIiwiTG9naW43VG9rZW5IYW5kbGVyIiwidG9rZW5TdHJlYW1QYXJzZXIiLCJsb2dpbkFja1JlY2VpdmVkIiwiTE9HR0VEX0lOX1NFTkRJTkdfSU5JVElBTF9TUUwiLCJOVExNUmVzcG9uc2VQYXlsb2FkIiwiTlRMTUFVVEhfUEtUIiwiZmVkQXV0aEluZm9Ub2tlbiIsInN0c3VybCIsInNwbiIsInRva2VuU2NvcGUiLCJVUkwiLCJnZXRUb2tlbiIsImdldFRva2VuRnJvbUNyZWRlbnRpYWxzIiwiY3JlZGVudGlhbHMiLCJ0b2tlblJlc3BvbnNlIiwiVXNlcm5hbWVQYXNzd29yZENyZWRlbnRpYWwiLCJtc2lBcmdzIiwiTWFuYWdlZElkZW50aXR5Q3JlZGVudGlhbCIsIm1hbmFnZWRJZGVudGl0eUNsaWVudElkIiwiRGVmYXVsdEF6dXJlQ3JlZGVudGlhbCIsIkNsaWVudFNlY3JldENyZWRlbnRpYWwiLCJJbml0aWFsU3FsVG9rZW5IYW5kbGVyIiwibmV4dFN0YXRlIiwic3FsUmVxdWVzdCIsIlJlcXVlc3RUb2tlbkhhbmRsZXIiLCJTRU5UX0FUVEVOVElPTiIsIm9uUmVzdW1lIiwib25QYXVzZSIsInBhdXNlIiwib25FbmRPZk1lc3NhZ2UiLCJBdHRlbnRpb25Ub2tlbkhhbmRsZXIiLCJhdHRlbnRpb25SZWNlaXZlZCJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOztBQUNBOztBQUdBOztBQUNBOztBQUVBOztBQUVBOztBQVFBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUNBOztBQUVBOztBQUdBOztBQUNBOztBQUNBOztBQUdBOztBQUNBOztBQUNBOztBQUNBOzs7O0FBRUEsSUFBSUEseUJBQXlCLEdBQUcsS0FBaEM7O0FBRUEsTUFBTUMsaUNBQWlDLEdBQUcsTUFBTTtBQUM5QyxNQUFJLENBQUNELHlCQUFMLEVBQWdDO0FBQzlCQSxJQUFBQSx5QkFBeUIsR0FBRyxJQUE1QjtBQUNBRSxJQUFBQSxPQUFPLENBQUNDLFdBQVIsQ0FBb0Isd0pBQXBCO0FBQ0Q7QUFDRixDQUxEOztBQU9BLElBQUlDLG9DQUFvQyxHQUFHLEtBQTNDOztBQUNBLE1BQU1DLGlDQUFpQyxHQUFHLE1BQU07QUFDOUMsTUFBSSxDQUFDRCxvQ0FBTCxFQUEyQztBQUN6Q0EsSUFBQUEsb0NBQW9DLEdBQUcsSUFBdkM7QUFDQUYsSUFBQUEsT0FBTyxDQUFDQyxXQUFSLENBQW9CLHVFQUNwQixvR0FEb0IsR0FFcEIsd0VBRkE7QUFHRDtBQUNGLENBUEQ7O0FBMkdBO0FBQ0E7QUFDQTtBQUNBLE1BQU1HLHdCQUF3QixHQUFHLEtBQUssSUFBdEM7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMsdUJBQXVCLEdBQUcsS0FBSyxJQUFyQztBQUNBO0FBQ0E7QUFDQTs7QUFDQSxNQUFNQyw4QkFBOEIsR0FBRyxLQUFLLElBQTVDO0FBQ0E7QUFDQTtBQUNBOztBQUNBLE1BQU1DLHNCQUFzQixHQUFHLElBQUksSUFBbkM7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMsOEJBQThCLEdBQUcsR0FBdkM7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMsbUJBQW1CLEdBQUcsSUFBSSxJQUFoQztBQUNBO0FBQ0E7QUFDQTs7QUFDQSxNQUFNQyxnQkFBZ0IsR0FBRyxVQUF6QjtBQUNBO0FBQ0E7QUFDQTs7QUFDQSxNQUFNQyxpQkFBaUIsR0FBRyxDQUExQjtBQUNBO0FBQ0E7QUFDQTs7QUFDQSxNQUFNQyxZQUFZLEdBQUcsSUFBckI7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMsbUJBQW1CLEdBQUcsS0FBNUI7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMsZ0JBQWdCLEdBQUcsWUFBekI7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMsa0JBQWtCLEdBQUcsS0FBM0I7O0FBd29CQTtBQUNBO0FBQ0E7QUFDQSxNQUFNQyxZQUFZLEdBQUc7QUFDbkJDLEVBQUFBLE1BQU0sRUFBRSxDQURXO0FBRW5CQyxFQUFBQSxRQUFRLEVBQUUsQ0FGUztBQUduQkMsRUFBQUEsS0FBSyxFQUFFO0FBSFksQ0FBckI7O0FBV0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBLE1BQU1DLFVBQU4sU0FBeUJDLG9CQUF6QixDQUFzQztBQUNwQztBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQXVCRTtBQUNGO0FBQ0E7O0FBR0U7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBR0U7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQUdFO0FBQ0Y7QUFDQTs7QUFFRTtBQUNGO0FBQ0E7O0FBRUU7QUFDRjtBQUNBOztBQUVFO0FBQ0Y7QUFDQTs7QUFHRTtBQUNGO0FBQ0E7O0FBR0U7QUFDRjtBQUNBOztBQUdFO0FBQ0Y7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNFQyxFQUFBQSxXQUFXLENBQUNDLE1BQUQsRUFBa0M7QUFDM0M7QUFEMkMsU0E3SjdDQyxlQTZKNkM7QUFBQSxTQXpKN0NELE1BeUo2QztBQUFBLFNBcko3Q0UsYUFxSjZDO0FBQUEsU0FqSjdDQyxhQWlKNkM7QUFBQSxTQTdJN0NDLHNCQTZJNkM7QUFBQSxTQXpJN0NDLGdCQXlJNkM7QUFBQSxTQXJJN0NDLFVBcUk2QztBQUFBLFNBakk3Q0Msc0JBaUk2QztBQUFBLFNBN0g3Q0Msb0JBNkg2QztBQUFBLFNBekg3Q0MsTUF5SDZDO0FBQUEsU0FySDdDQyxVQXFINkM7QUFBQSxTQWpIN0NDLEtBaUg2QztBQUFBLFNBN0c3Q0MsVUE2RzZDO0FBQUEsU0F6RzdDQyxnQkF5RzZDO0FBQUEsU0FoRjdDQyxXQWdGNkM7QUFBQSxTQTNFN0NDLFNBMkU2QztBQUFBLFNBdkU3Q0MsS0F1RTZDO0FBQUEsU0FuRTdDQyw0QkFtRTZDO0FBQUEsU0E5RDdDQyxPQThENkM7QUFBQSxTQTFEN0NDLHFCQTBENkM7QUFBQSxTQXREN0NDLE1Bc0Q2QztBQUFBLFNBbEQ3Q0MsYUFrRDZDO0FBQUEsU0E3QzdDQyxZQTZDNkM7QUFBQSxTQXpDN0NDLFdBeUM2QztBQUFBLFNBckM3Q0MsWUFxQzZDO0FBQUEsU0FqQzdDQyxVQWlDNkM7QUFBQSxTQTVCN0NDLHVCQTRCNkM7QUFBQSxTQXZCN0NDLGlCQXVCNkM7O0FBRzNDLFFBQUksT0FBTzNCLE1BQVAsS0FBa0IsUUFBbEIsSUFBOEJBLE1BQU0sS0FBSyxJQUE3QyxFQUFtRDtBQUNqRCxZQUFNLElBQUk0QixTQUFKLENBQWMsK0RBQWQsQ0FBTjtBQUNEOztBQUVELFFBQUksT0FBTzVCLE1BQU0sQ0FBQzZCLE1BQWQsS0FBeUIsUUFBN0IsRUFBdUM7QUFDckMsWUFBTSxJQUFJRCxTQUFKLENBQWMsc0VBQWQsQ0FBTjtBQUNEOztBQUVELFNBQUszQixlQUFMLEdBQXVCLEtBQXZCO0FBRUEsUUFBSTZCLGNBQUo7O0FBQ0EsUUFBSTlCLE1BQU0sQ0FBQzhCLGNBQVAsS0FBMEJDLFNBQTlCLEVBQXlDO0FBQ3ZDLFVBQUksT0FBTy9CLE1BQU0sQ0FBQzhCLGNBQWQsS0FBaUMsUUFBakMsSUFBNkM5QixNQUFNLENBQUM4QixjQUFQLEtBQTBCLElBQTNFLEVBQWlGO0FBQy9FLGNBQU0sSUFBSUYsU0FBSixDQUFjLDhEQUFkLENBQU47QUFDRDs7QUFFRCxZQUFNSSxJQUFJLEdBQUdoQyxNQUFNLENBQUM4QixjQUFQLENBQXNCRSxJQUFuQztBQUNBLFlBQU1DLE9BQU8sR0FBR2pDLE1BQU0sQ0FBQzhCLGNBQVAsQ0FBc0JHLE9BQXRCLEtBQWtDRixTQUFsQyxHQUE4QyxFQUE5QyxHQUFtRC9CLE1BQU0sQ0FBQzhCLGNBQVAsQ0FBc0JHLE9BQXpGOztBQUVBLFVBQUksT0FBT0QsSUFBUCxLQUFnQixRQUFwQixFQUE4QjtBQUM1QixjQUFNLElBQUlKLFNBQUosQ0FBYyxtRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsVUFBSUksSUFBSSxLQUFLLFNBQVQsSUFBc0JBLElBQUksS0FBSyxNQUEvQixJQUF5Q0EsSUFBSSxLQUFLLGlDQUFsRCxJQUF1RkEsSUFBSSxLQUFLLHFDQUFoRyxJQUF5SUEsSUFBSSxLQUFLLCtCQUFsSixJQUFxTEEsSUFBSSxLQUFLLHdDQUE5TCxJQUEwT0EsSUFBSSxLQUFLLGlEQUFuUCxJQUF3U0EsSUFBSSxLQUFLLGdDQUFyVCxFQUF1VjtBQUNyVixjQUFNLElBQUlKLFNBQUosQ0FBYyxrU0FBZCxDQUFOO0FBQ0Q7O0FBRUQsVUFBSSxPQUFPSyxPQUFQLEtBQW1CLFFBQW5CLElBQStCQSxPQUFPLEtBQUssSUFBL0MsRUFBcUQ7QUFDbkQsY0FBTSxJQUFJTCxTQUFKLENBQWMsc0VBQWQsQ0FBTjtBQUNEOztBQUVELFVBQUlJLElBQUksS0FBSyxNQUFiLEVBQXFCO0FBQ25CLFlBQUksT0FBT0MsT0FBTyxDQUFDQyxNQUFmLEtBQTBCLFFBQTlCLEVBQXdDO0FBQ3RDLGdCQUFNLElBQUlOLFNBQUosQ0FBYyw2RUFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSUssT0FBTyxDQUFDRSxRQUFSLEtBQXFCSixTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNFLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVAsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFRCxZQUFJSyxPQUFPLENBQUNHLFFBQVIsS0FBcUJMLFNBQXJCLElBQWtDLE9BQU9FLE9BQU8sQ0FBQ0csUUFBZixLQUE0QixRQUFsRSxFQUE0RTtBQUMxRSxnQkFBTSxJQUFJUixTQUFKLENBQWMsK0VBQWQsQ0FBTjtBQUNEOztBQUVERSxRQUFBQSxjQUFjLEdBQUc7QUFDZkUsVUFBQUEsSUFBSSxFQUFFLE1BRFM7QUFFZkMsVUFBQUEsT0FBTyxFQUFFO0FBQ1BFLFlBQUFBLFFBQVEsRUFBRUYsT0FBTyxDQUFDRSxRQURYO0FBRVBDLFlBQUFBLFFBQVEsRUFBRUgsT0FBTyxDQUFDRyxRQUZYO0FBR1BGLFlBQUFBLE1BQU0sRUFBRUQsT0FBTyxDQUFDQyxNQUFSLElBQWtCRCxPQUFPLENBQUNDLE1BQVIsQ0FBZUcsV0FBZjtBQUhuQjtBQUZNLFNBQWpCO0FBUUQsT0FyQkQsTUFxQk8sSUFBSUwsSUFBSSxLQUFLLGlDQUFiLEVBQWdEO0FBQUE7O0FBQ3JELFlBQUlDLE9BQU8sQ0FBQ0UsUUFBUixLQUFxQkosU0FBckIsSUFBa0MsT0FBT0UsT0FBTyxDQUFDRSxRQUFmLEtBQTRCLFFBQWxFLEVBQTRFO0FBQzFFLGdCQUFNLElBQUlQLFNBQUosQ0FBYywrRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSUssT0FBTyxDQUFDRyxRQUFSLEtBQXFCTCxTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNHLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVIsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFRCxZQUFJSyxPQUFPLENBQUNLLFFBQVIsS0FBcUJQLFNBQXJCLElBQWtDLE9BQU9FLE9BQU8sQ0FBQ0ssUUFBZixLQUE0QixRQUFsRSxFQUE0RTtBQUMxRSxnQkFBTSxJQUFJVixTQUFKLENBQWMsK0VBQWQsQ0FBTjtBQUNELFNBRkQsTUFFTyxJQUFJSyxPQUFPLENBQUNLLFFBQVIsS0FBcUJQLFNBQXpCLEVBQW9DO0FBQ3pDUSxVQUFBQSw2Q0FBNkM7QUFDOUM7O0FBRUQsWUFBSU4sT0FBTyxDQUFDQyxNQUFSLEtBQW1CSCxTQUFuQixJQUFnQyxPQUFPRSxPQUFPLENBQUNDLE1BQWYsS0FBMEIsUUFBOUQsRUFBd0U7QUFDdEUsZ0JBQU0sSUFBSU4sU0FBSixDQUFjLDZFQUFkLENBQU47QUFDRCxTQUZELE1BRU8sSUFBSUssT0FBTyxDQUFDQyxNQUFSLEtBQW1CSCxTQUF2QixFQUFrQztBQUN2Q25ELFVBQUFBLGlDQUFpQztBQUNsQzs7QUFFRCxZQUFJcUQsT0FBTyxDQUFDTyxRQUFSLEtBQXFCVCxTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNPLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVosU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFREUsUUFBQUEsY0FBYyxHQUFHO0FBQ2ZFLFVBQUFBLElBQUksRUFBRSxpQ0FEUztBQUVmQyxVQUFBQSxPQUFPLEVBQUU7QUFDUEUsWUFBQUEsUUFBUSxFQUFFRixPQUFPLENBQUNFLFFBRFg7QUFFUEMsWUFBQUEsUUFBUSxFQUFFSCxPQUFPLENBQUNHLFFBRlg7QUFHUEksWUFBQUEsUUFBUSx1QkFBRVAsT0FBTyxDQUFDTyxRQUFWLGlFQUFzQlAsT0FBTyxDQUFDQyxNQUgvQjtBQUlQSSxZQUFBQSxRQUFRLHVCQUFFTCxPQUFPLENBQUNLLFFBQVYsaUVBQXNCO0FBSnZCO0FBRk0sU0FBakI7QUFTRCxPQWxDTSxNQWtDQSxJQUFJTixJQUFJLEtBQUsscUNBQWIsRUFBb0Q7QUFDekQsWUFBSSxPQUFPQyxPQUFPLENBQUNRLEtBQWYsS0FBeUIsUUFBN0IsRUFBdUM7QUFDckMsZ0JBQU0sSUFBSWIsU0FBSixDQUFjLDRFQUFkLENBQU47QUFDRDs7QUFFREUsUUFBQUEsY0FBYyxHQUFHO0FBQ2ZFLFVBQUFBLElBQUksRUFBRSxxQ0FEUztBQUVmQyxVQUFBQSxPQUFPLEVBQUU7QUFDUFEsWUFBQUEsS0FBSyxFQUFFUixPQUFPLENBQUNRO0FBRFI7QUFGTSxTQUFqQjtBQU1ELE9BWE0sTUFXQSxJQUFJVCxJQUFJLEtBQUssK0JBQWIsRUFBOEM7QUFDbkQsWUFBSUMsT0FBTyxDQUFDSyxRQUFSLEtBQXFCUCxTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNLLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVYsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFREUsUUFBQUEsY0FBYyxHQUFHO0FBQ2ZFLFVBQUFBLElBQUksRUFBRSwrQkFEUztBQUVmQyxVQUFBQSxPQUFPLEVBQUU7QUFDUEssWUFBQUEsUUFBUSxFQUFFTCxPQUFPLENBQUNLO0FBRFg7QUFGTSxTQUFqQjtBQU1ELE9BWE0sTUFXQSxJQUFJTixJQUFJLEtBQUssZ0NBQWIsRUFBK0M7QUFDcEQsWUFBSUMsT0FBTyxDQUFDSyxRQUFSLEtBQXFCUCxTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNLLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVYsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFDREUsUUFBQUEsY0FBYyxHQUFHO0FBQ2ZFLFVBQUFBLElBQUksRUFBRSxnQ0FEUztBQUVmQyxVQUFBQSxPQUFPLEVBQUU7QUFDUEssWUFBQUEsUUFBUSxFQUFFTCxPQUFPLENBQUNLO0FBRFg7QUFGTSxTQUFqQjtBQU1ELE9BVk0sTUFVQSxJQUFJTixJQUFJLEtBQUssd0NBQWIsRUFBdUQ7QUFDNUQsWUFBSUMsT0FBTyxDQUFDSyxRQUFSLEtBQXFCUCxTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNLLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVYsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFREUsUUFBQUEsY0FBYyxHQUFHO0FBQ2ZFLFVBQUFBLElBQUksRUFBRSx3Q0FEUztBQUVmQyxVQUFBQSxPQUFPLEVBQUU7QUFDUEssWUFBQUEsUUFBUSxFQUFFTCxPQUFPLENBQUNLO0FBRFg7QUFGTSxTQUFqQjtBQU1ELE9BWE0sTUFXQSxJQUFJTixJQUFJLEtBQUssaURBQWIsRUFBZ0U7QUFDckUsWUFBSSxPQUFPQyxPQUFPLENBQUNLLFFBQWYsS0FBNEIsUUFBaEMsRUFBMEM7QUFDeEMsZ0JBQU0sSUFBSVYsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFRCxZQUFJLE9BQU9LLE9BQU8sQ0FBQ1MsWUFBZixLQUFnQyxRQUFwQyxFQUE4QztBQUM1QyxnQkFBTSxJQUFJZCxTQUFKLENBQWMsbUZBQWQsQ0FBTjtBQUNEOztBQUVELFlBQUksT0FBT0ssT0FBTyxDQUFDTyxRQUFmLEtBQTRCLFFBQWhDLEVBQTBDO0FBQ3hDLGdCQUFNLElBQUlaLFNBQUosQ0FBYywrRUFBZCxDQUFOO0FBQ0Q7O0FBRURFLFFBQUFBLGNBQWMsR0FBRztBQUNmRSxVQUFBQSxJQUFJLEVBQUUsaURBRFM7QUFFZkMsVUFBQUEsT0FBTyxFQUFFO0FBQ1BLLFlBQUFBLFFBQVEsRUFBRUwsT0FBTyxDQUFDSyxRQURYO0FBRVBJLFlBQUFBLFlBQVksRUFBRVQsT0FBTyxDQUFDUyxZQUZmO0FBR1BGLFlBQUFBLFFBQVEsRUFBRVAsT0FBTyxDQUFDTztBQUhYO0FBRk0sU0FBakI7QUFRRCxPQXJCTSxNQXFCQTtBQUNMLFlBQUlQLE9BQU8sQ0FBQ0UsUUFBUixLQUFxQkosU0FBckIsSUFBa0MsT0FBT0UsT0FBTyxDQUFDRSxRQUFmLEtBQTRCLFFBQWxFLEVBQTRFO0FBQzFFLGdCQUFNLElBQUlQLFNBQUosQ0FBYywrRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSUssT0FBTyxDQUFDRyxRQUFSLEtBQXFCTCxTQUFyQixJQUFrQyxPQUFPRSxPQUFPLENBQUNHLFFBQWYsS0FBNEIsUUFBbEUsRUFBNEU7QUFDMUUsZ0JBQU0sSUFBSVIsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFREUsUUFBQUEsY0FBYyxHQUFHO0FBQ2ZFLFVBQUFBLElBQUksRUFBRSxTQURTO0FBRWZDLFVBQUFBLE9BQU8sRUFBRTtBQUNQRSxZQUFBQSxRQUFRLEVBQUVGLE9BQU8sQ0FBQ0UsUUFEWDtBQUVQQyxZQUFBQSxRQUFRLEVBQUVILE9BQU8sQ0FBQ0c7QUFGWDtBQUZNLFNBQWpCO0FBT0Q7QUFDRixLQTVKRCxNQTRKTztBQUNMTixNQUFBQSxjQUFjLEdBQUc7QUFDZkUsUUFBQUEsSUFBSSxFQUFFLFNBRFM7QUFFZkMsUUFBQUEsT0FBTyxFQUFFO0FBQ1BFLFVBQUFBLFFBQVEsRUFBRUosU0FESDtBQUVQSyxVQUFBQSxRQUFRLEVBQUVMO0FBRkg7QUFGTSxPQUFqQjtBQU9EOztBQUVELFNBQUsvQixNQUFMLEdBQWM7QUFDWjZCLE1BQUFBLE1BQU0sRUFBRTdCLE1BQU0sQ0FBQzZCLE1BREg7QUFFWkMsTUFBQUEsY0FBYyxFQUFFQSxjQUZKO0FBR1pHLE1BQUFBLE9BQU8sRUFBRTtBQUNQVSxRQUFBQSx1QkFBdUIsRUFBRSxLQURsQjtBQUVQQyxRQUFBQSxPQUFPLEVBQUViLFNBRkY7QUFHUGMsUUFBQUEsZ0JBQWdCLEVBQUUsS0FIWDtBQUlQQyxRQUFBQSxhQUFhLEVBQUU5RCxzQkFKUjtBQUtQK0QsUUFBQUEsMkJBQTJCLEVBQUUsSUFBSSxFQUFKLEdBQVMsRUFBVCxHQUFjLElBTHBDO0FBSzJDO0FBQ2xEQyxRQUFBQSx1QkFBdUIsRUFBRSxLQU5sQjtBQU9QQyxRQUFBQSxrQkFBa0IsRUFBRWxCLFNBUGI7QUFRUG1CLFFBQUFBLHVCQUF1QixFQUFFakUsOEJBUmxCO0FBU1BrRSxRQUFBQSxjQUFjLEVBQUVyRSx1QkFUVDtBQVVQc0UsUUFBQUEsd0JBQXdCLEVBQUVDLDZCQUFnQkMsY0FWbkM7QUFXUEMsUUFBQUEsd0JBQXdCLEVBQUUsRUFYbkI7QUFZUEMsUUFBQUEsUUFBUSxFQUFFekIsU0FaSDtBQWFQMEIsUUFBQUEsU0FBUyxFQUFFckUsaUJBYko7QUFjUHNFLFFBQUFBLFVBQVUsRUFBRWxFLGtCQWRMO0FBZVBtQixRQUFBQSxLQUFLLEVBQUU7QUFDTGdELFVBQUFBLElBQUksRUFBRSxLQUREO0FBRUxDLFVBQUFBLE1BQU0sRUFBRSxLQUZIO0FBR0xDLFVBQUFBLE9BQU8sRUFBRSxLQUhKO0FBSUxwQixVQUFBQSxLQUFLLEVBQUU7QUFKRixTQWZBO0FBcUJQcUIsUUFBQUEsY0FBYyxFQUFFLElBckJUO0FBc0JQQyxRQUFBQSxxQkFBcUIsRUFBRSxJQXRCaEI7QUF1QlBDLFFBQUFBLGlCQUFpQixFQUFFLElBdkJaO0FBd0JQQyxRQUFBQSxrQkFBa0IsRUFBRSxJQXhCYjtBQXlCUEMsUUFBQUEsZ0JBQWdCLEVBQUUsSUF6Qlg7QUEwQlBDLFFBQUFBLDBCQUEwQixFQUFFLElBMUJyQjtBQTJCUEMsUUFBQUEseUJBQXlCLEVBQUUsSUEzQnBCO0FBNEJQQyxRQUFBQSwwQkFBMEIsRUFBRSxLQTVCckI7QUE2QlBDLFFBQUFBLHVCQUF1QixFQUFFLEtBN0JsQjtBQThCUEMsUUFBQUEsc0JBQXNCLEVBQUUsSUE5QmpCO0FBK0JQQyxRQUFBQSxPQUFPLEVBQUUsSUEvQkY7QUFnQ1BDLFFBQUFBLG1CQUFtQixFQUFFLEtBaENkO0FBaUNQQyxRQUFBQSwyQkFBMkIsRUFBRTNDLFNBakN0QjtBQWtDUDRDLFFBQUFBLFlBQVksRUFBRTVDLFNBbENQO0FBbUNQNkMsUUFBQUEsY0FBYyxFQUFFdkIsNkJBQWdCQyxjQW5DekI7QUFvQ1B1QixRQUFBQSxRQUFRLEVBQUV0RixnQkFwQ0g7QUFxQ1B1RixRQUFBQSxZQUFZLEVBQUUvQyxTQXJDUDtBQXNDUGdELFFBQUFBLDJCQUEyQixFQUFFLENBdEN0QjtBQXVDUEMsUUFBQUEsbUJBQW1CLEVBQUUsS0F2Q2Q7QUF3Q1BDLFFBQUFBLFVBQVUsRUFBRS9GLG1CQXhDTDtBQXlDUGdHLFFBQUFBLElBQUksRUFBRTdGLFlBekNDO0FBMENQOEYsUUFBQUEsY0FBYyxFQUFFLEtBMUNUO0FBMkNQQyxRQUFBQSxjQUFjLEVBQUVyRyw4QkEzQ1Q7QUE0Q1BzRyxRQUFBQSxtQkFBbUIsRUFBRSxLQTVDZDtBQTZDUEMsUUFBQUEsZ0NBQWdDLEVBQUUsS0E3QzNCO0FBOENQQyxRQUFBQSxVQUFVLEVBQUV4RCxTQTlDTDtBQStDUHlELFFBQUFBLDhCQUE4QixFQUFFLEtBL0N6QjtBQWdEUEMsUUFBQUEsVUFBVSxFQUFFbkcsbUJBaERMO0FBaURQb0csUUFBQUEsUUFBUSxFQUFFdkcsZ0JBakRIO0FBa0RQd0csUUFBQUEsbUJBQW1CLEVBQUU1RCxTQWxEZDtBQW1EUDZELFFBQUFBLHNCQUFzQixFQUFFLElBbkRqQjtBQW9EUEMsUUFBQUEsY0FBYyxFQUFFLEtBcERUO0FBcURQQyxRQUFBQSxNQUFNLEVBQUUsSUFyREQ7QUFzRFBDLFFBQUFBLGFBQWEsRUFBRWhFLFNBdERSO0FBdURQaUUsUUFBQUEsY0FBYyxFQUFFO0FBdkRUO0FBSEcsS0FBZDs7QUE4REEsUUFBSWhHLE1BQU0sQ0FBQ2lDLE9BQVgsRUFBb0I7QUFDbEIsVUFBSWpDLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWlELElBQWYsSUFBdUJsRixNQUFNLENBQUNpQyxPQUFQLENBQWUwQyxZQUExQyxFQUF3RDtBQUN0RCxjQUFNLElBQUlzQixLQUFKLENBQVUsdURBQXVEakcsTUFBTSxDQUFDaUMsT0FBUCxDQUFlaUQsSUFBdEUsR0FBNkUsT0FBN0UsR0FBdUZsRixNQUFNLENBQUNpQyxPQUFQLENBQWUwQyxZQUF0RyxHQUFxSCxXQUEvSCxDQUFOO0FBQ0Q7O0FBRUQsVUFBSTNFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZVUsdUJBQWYsS0FBMkNaLFNBQS9DLEVBQTBEO0FBQ3hELFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZVUsdUJBQXRCLEtBQWtELFNBQWxELElBQStEM0MsTUFBTSxDQUFDaUMsT0FBUCxDQUFlVSx1QkFBZixLQUEyQyxJQUE5RyxFQUFvSDtBQUNsSCxnQkFBTSxJQUFJZixTQUFKLENBQWMsdUZBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CVSx1QkFBcEIsR0FBOEMzQyxNQUFNLENBQUNpQyxPQUFQLENBQWVVLHVCQUE3RDtBQUNEOztBQUVELFVBQUkzQyxNQUFNLENBQUNpQyxPQUFQLENBQWVXLE9BQWYsS0FBMkJiLFNBQS9CLEVBQTBDO0FBQ3hDLFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZVcsT0FBdEIsS0FBa0MsUUFBdEMsRUFBZ0Q7QUFDOUMsZ0JBQU0sSUFBSWhCLFNBQUosQ0FBYywrREFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JXLE9BQXBCLEdBQThCNUMsTUFBTSxDQUFDaUMsT0FBUCxDQUFlVyxPQUE3QztBQUNEOztBQUVELFVBQUk1QyxNQUFNLENBQUNpQyxPQUFQLENBQWVZLGdCQUFmLEtBQW9DZCxTQUF4QyxFQUFtRDtBQUNqRCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVZLGdCQUF0QixLQUEyQyxTQUEvQyxFQUEwRDtBQUN4RCxnQkFBTSxJQUFJakIsU0FBSixDQUFjLHlFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQlksZ0JBQXBCLEdBQXVDN0MsTUFBTSxDQUFDaUMsT0FBUCxDQUFlWSxnQkFBdEQ7QUFDRDs7QUFFRCxVQUFJN0MsTUFBTSxDQUFDaUMsT0FBUCxDQUFlYSxhQUFmLEtBQWlDZixTQUFyQyxFQUFnRDtBQUM5QyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVhLGFBQXRCLEtBQXdDLFFBQTVDLEVBQXNEO0FBQ3BELGdCQUFNLElBQUlsQixTQUFKLENBQWMscUVBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CYSxhQUFwQixHQUFvQzlDLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWEsYUFBbkQ7QUFDRDs7QUFFRCxVQUFJOUMsTUFBTSxDQUFDaUMsT0FBUCxDQUFlZ0Isa0JBQW5CLEVBQXVDO0FBQ3JDLFlBQUksT0FBT2pELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWdCLGtCQUF0QixLQUE2QyxVQUFqRCxFQUE2RDtBQUMzRCxnQkFBTSxJQUFJckIsU0FBSixDQUFjLHVFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmdCLGtCQUFwQixHQUF5Q2pELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWdCLGtCQUF4RDtBQUNEOztBQUVELFVBQUlqRCxNQUFNLENBQUNpQyxPQUFQLENBQWVrQixjQUFmLEtBQWtDcEIsU0FBdEMsRUFBaUQ7QUFDL0MsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFla0IsY0FBdEIsS0FBeUMsUUFBN0MsRUFBdUQ7QUFDckQsZ0JBQU0sSUFBSXZCLFNBQUosQ0FBYyxzRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JrQixjQUFwQixHQUFxQ25ELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWtCLGNBQXBEO0FBQ0Q7O0FBRUQsVUFBSW5ELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW1CLHdCQUFmLEtBQTRDckIsU0FBaEQsRUFBMkQ7QUFDekQsb0RBQTBCL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlbUIsd0JBQXpDLEVBQW1FLHlDQUFuRTtBQUVBLGFBQUtwRCxNQUFMLENBQVlpQyxPQUFaLENBQW9CbUIsd0JBQXBCLEdBQStDcEQsTUFBTSxDQUFDaUMsT0FBUCxDQUFlbUIsd0JBQTlEO0FBQ0Q7O0FBRUQsVUFBSXBELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWtCLGNBQWYsS0FBa0NwQixTQUF0QyxFQUFpRDtBQUMvQyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVrQixjQUF0QixLQUF5QyxRQUE3QyxFQUF1RDtBQUNyRCxnQkFBTSxJQUFJdkIsU0FBSixDQUFjLHNFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmtCLGNBQXBCLEdBQXFDbkQsTUFBTSxDQUFDaUMsT0FBUCxDQUFla0IsY0FBcEQ7QUFDRDs7QUFFRCxVQUFJbkQsTUFBTSxDQUFDaUMsT0FBUCxDQUFlc0Isd0JBQWYsS0FBNEN4QixTQUFoRCxFQUEyRDtBQUN6RCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVzQix3QkFBdEIsS0FBbUQsUUFBbkQsSUFBK0R2RCxNQUFNLENBQUNpQyxPQUFQLENBQWVzQix3QkFBZixLQUE0QyxJQUEvRyxFQUFxSDtBQUNuSCxnQkFBTSxJQUFJM0IsU0FBSixDQUFjLGdGQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQnNCLHdCQUFwQixHQUErQ3ZELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXNCLHdCQUE5RDtBQUNEOztBQUVELFVBQUl2RCxNQUFNLENBQUNpQyxPQUFQLENBQWV1QixRQUFmLEtBQTRCekIsU0FBaEMsRUFBMkM7QUFDekMsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFldUIsUUFBdEIsS0FBbUMsUUFBdkMsRUFBaUQ7QUFDL0MsZ0JBQU0sSUFBSTVCLFNBQUosQ0FBYyxnRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J1QixRQUFwQixHQUErQnhELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXVCLFFBQTlDO0FBQ0Q7O0FBRUQsVUFBSXhELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXdCLFNBQWYsS0FBNkIxQixTQUFqQyxFQUE0QztBQUMxQyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWV3QixTQUF0QixLQUFvQyxRQUFwQyxJQUFnRHpELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXdCLFNBQWYsS0FBNkIsSUFBakYsRUFBdUY7QUFDckYsZ0JBQU0sSUFBSTdCLFNBQUosQ0FBYyxpRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSTVCLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXdCLFNBQWYsS0FBNkIsSUFBN0IsS0FBc0N6RCxNQUFNLENBQUNpQyxPQUFQLENBQWV3QixTQUFmLEdBQTJCLENBQTNCLElBQWdDekQsTUFBTSxDQUFDaUMsT0FBUCxDQUFld0IsU0FBZixHQUEyQixDQUFqRyxDQUFKLEVBQXlHO0FBQ3ZHLGdCQUFNLElBQUl5QyxVQUFKLENBQWUsK0RBQWYsQ0FBTjtBQUNEOztBQUVELGFBQUtsRyxNQUFMLENBQVlpQyxPQUFaLENBQW9Cd0IsU0FBcEIsR0FBZ0N6RCxNQUFNLENBQUNpQyxPQUFQLENBQWV3QixTQUEvQztBQUNEOztBQUVELFVBQUl6RCxNQUFNLENBQUNpQyxPQUFQLENBQWV5QixVQUFmLEtBQThCM0IsU0FBbEMsRUFBNkM7QUFDM0MsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFleUIsVUFBdEIsS0FBcUMsUUFBckMsSUFBaUQxRCxNQUFNLENBQUNpQyxPQUFQLENBQWV5QixVQUFmLEtBQThCLElBQW5GLEVBQXlGO0FBQ3ZGLGdCQUFNLElBQUk5QixTQUFKLENBQWMsMEVBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CeUIsVUFBcEIsR0FBaUMxRCxNQUFNLENBQUNpQyxPQUFQLENBQWV5QixVQUFoRDtBQUNEOztBQUVELFVBQUkxRCxNQUFNLENBQUNpQyxPQUFQLENBQWV0QixLQUFuQixFQUEwQjtBQUN4QixZQUFJWCxNQUFNLENBQUNpQyxPQUFQLENBQWV0QixLQUFmLENBQXFCZ0QsSUFBckIsS0FBOEI1QixTQUFsQyxFQUE2QztBQUMzQyxjQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWV0QixLQUFmLENBQXFCZ0QsSUFBNUIsS0FBcUMsU0FBekMsRUFBb0Q7QUFDbEQsa0JBQU0sSUFBSS9CLFNBQUosQ0FBYyxtRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsZUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J0QixLQUFwQixDQUEwQmdELElBQTFCLEdBQWlDM0QsTUFBTSxDQUFDaUMsT0FBUCxDQUFldEIsS0FBZixDQUFxQmdELElBQXREO0FBQ0Q7O0FBRUQsWUFBSTNELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXRCLEtBQWYsQ0FBcUJpRCxNQUFyQixLQUFnQzdCLFNBQXBDLEVBQStDO0FBQzdDLGNBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXRCLEtBQWYsQ0FBcUJpRCxNQUE1QixLQUF1QyxTQUEzQyxFQUFzRDtBQUNwRCxrQkFBTSxJQUFJaEMsU0FBSixDQUFjLHFFQUFkLENBQU47QUFDRDs7QUFFRCxlQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQnRCLEtBQXBCLENBQTBCaUQsTUFBMUIsR0FBbUM1RCxNQUFNLENBQUNpQyxPQUFQLENBQWV0QixLQUFmLENBQXFCaUQsTUFBeEQ7QUFDRDs7QUFFRCxZQUFJNUQsTUFBTSxDQUFDaUMsT0FBUCxDQUFldEIsS0FBZixDQUFxQmtELE9BQXJCLEtBQWlDOUIsU0FBckMsRUFBZ0Q7QUFDOUMsY0FBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFldEIsS0FBZixDQUFxQmtELE9BQTVCLEtBQXdDLFNBQTVDLEVBQXVEO0FBQ3JELGtCQUFNLElBQUlqQyxTQUFKLENBQWMsc0VBQWQsQ0FBTjtBQUNEOztBQUVELGVBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CdEIsS0FBcEIsQ0FBMEJrRCxPQUExQixHQUFvQzdELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXRCLEtBQWYsQ0FBcUJrRCxPQUF6RDtBQUNEOztBQUVELFlBQUk3RCxNQUFNLENBQUNpQyxPQUFQLENBQWV0QixLQUFmLENBQXFCOEIsS0FBckIsS0FBK0JWLFNBQW5DLEVBQThDO0FBQzVDLGNBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXRCLEtBQWYsQ0FBcUI4QixLQUE1QixLQUFzQyxTQUExQyxFQUFxRDtBQUNuRCxrQkFBTSxJQUFJYixTQUFKLENBQWMsb0VBQWQsQ0FBTjtBQUNEOztBQUVELGVBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CdEIsS0FBcEIsQ0FBMEI4QixLQUExQixHQUFrQ3pDLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXRCLEtBQWYsQ0FBcUI4QixLQUF2RDtBQUNEO0FBQ0Y7O0FBRUQsVUFBSXpDLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTZCLGNBQWYsS0FBa0MvQixTQUF0QyxFQUFpRDtBQUMvQyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWU2QixjQUF0QixLQUF5QyxTQUF6QyxJQUFzRDlELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTZCLGNBQWYsS0FBa0MsSUFBNUYsRUFBa0c7QUFDaEcsZ0JBQU0sSUFBSWxDLFNBQUosQ0FBYywrRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0I2QixjQUFwQixHQUFxQzlELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTZCLGNBQXBEO0FBQ0Q7O0FBRUQsVUFBSTlELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZThCLHFCQUFmLEtBQXlDaEMsU0FBN0MsRUFBd0Q7QUFDdEQsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlOEIscUJBQXRCLEtBQWdELFNBQWhELElBQTZEL0QsTUFBTSxDQUFDaUMsT0FBUCxDQUFlOEIscUJBQWYsS0FBeUMsSUFBMUcsRUFBZ0g7QUFDOUcsZ0JBQU0sSUFBSW5DLFNBQUosQ0FBYyxzRkFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0I4QixxQkFBcEIsR0FBNEMvRCxNQUFNLENBQUNpQyxPQUFQLENBQWU4QixxQkFBM0Q7QUFDRDs7QUFFRCxVQUFJL0QsTUFBTSxDQUFDaUMsT0FBUCxDQUFlK0IsaUJBQWYsS0FBcUNqQyxTQUF6QyxFQUFvRDtBQUNsRCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWUrQixpQkFBdEIsS0FBNEMsU0FBNUMsSUFBeURoRSxNQUFNLENBQUNpQyxPQUFQLENBQWUrQixpQkFBZixLQUFxQyxJQUFsRyxFQUF3RztBQUN0RyxnQkFBTSxJQUFJcEMsU0FBSixDQUFjLGtGQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQitCLGlCQUFwQixHQUF3Q2hFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZStCLGlCQUF2RDtBQUNEOztBQUVELFVBQUloRSxNQUFNLENBQUNpQyxPQUFQLENBQWVnQyxrQkFBZixLQUFzQ2xDLFNBQTFDLEVBQXFEO0FBQ25ELFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWdDLGtCQUF0QixLQUE2QyxTQUE3QyxJQUEwRGpFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWdDLGtCQUFmLEtBQXNDLElBQXBHLEVBQTBHO0FBQ3hHLGdCQUFNLElBQUlyQyxTQUFKLENBQWMsbUZBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CZ0Msa0JBQXBCLEdBQXlDakUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlZ0Msa0JBQXhEO0FBQ0Q7O0FBRUQsVUFBSWpFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWlDLGdCQUFmLEtBQW9DbkMsU0FBeEMsRUFBbUQ7QUFDakQsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlaUMsZ0JBQXRCLEtBQTJDLFNBQTNDLElBQXdEbEUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlaUMsZ0JBQWYsS0FBb0MsSUFBaEcsRUFBc0c7QUFDcEcsZ0JBQU0sSUFBSXRDLFNBQUosQ0FBYyxpRkFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JpQyxnQkFBcEIsR0FBdUNsRSxNQUFNLENBQUNpQyxPQUFQLENBQWVpQyxnQkFBdEQ7QUFDRDs7QUFFRCxVQUFJbEUsTUFBTSxDQUFDaUMsT0FBUCxDQUFla0MsMEJBQWYsS0FBOENwQyxTQUFsRCxFQUE2RDtBQUMzRCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVrQywwQkFBdEIsS0FBcUQsU0FBckQsSUFBa0VuRSxNQUFNLENBQUNpQyxPQUFQLENBQWVrQywwQkFBZixLQUE4QyxJQUFwSCxFQUEwSDtBQUN4SCxnQkFBTSxJQUFJdkMsU0FBSixDQUFjLDJGQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmtDLDBCQUFwQixHQUFpRG5FLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWtDLDBCQUFoRTtBQUNEOztBQUVELFVBQUluRSxNQUFNLENBQUNpQyxPQUFQLENBQWVtQyx5QkFBZixLQUE2Q3JDLFNBQWpELEVBQTREO0FBQzFELFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW1DLHlCQUF0QixLQUFvRCxTQUFwRCxJQUFpRXBFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW1DLHlCQUFmLEtBQTZDLElBQWxILEVBQXdIO0FBQ3RILGdCQUFNLElBQUl4QyxTQUFKLENBQWMsMEZBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CbUMseUJBQXBCLEdBQWdEcEUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlbUMseUJBQS9EO0FBQ0Q7O0FBRUQsVUFBSXBFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW9DLDBCQUFmLEtBQThDdEMsU0FBbEQsRUFBNkQ7QUFDM0QsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlb0MsMEJBQXRCLEtBQXFELFNBQXJELElBQWtFckUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlb0MsMEJBQWYsS0FBOEMsSUFBcEgsRUFBMEg7QUFDeEgsZ0JBQU0sSUFBSXpDLFNBQUosQ0FBYywyRkFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JvQywwQkFBcEIsR0FBaURyRSxNQUFNLENBQUNpQyxPQUFQLENBQWVvQywwQkFBaEU7QUFDRDs7QUFFRCxVQUFJckUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlcUMsdUJBQWYsS0FBMkN2QyxTQUEvQyxFQUEwRDtBQUN4RCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVxQyx1QkFBdEIsS0FBa0QsU0FBbEQsSUFBK0R0RSxNQUFNLENBQUNpQyxPQUFQLENBQWVxQyx1QkFBZixLQUEyQyxJQUE5RyxFQUFvSDtBQUNsSCxnQkFBTSxJQUFJMUMsU0FBSixDQUFjLHdGQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQnFDLHVCQUFwQixHQUE4Q3RFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXFDLHVCQUE3RDtBQUNEOztBQUVELFVBQUl0RSxNQUFNLENBQUNpQyxPQUFQLENBQWVzQyxzQkFBZixLQUEwQ3hDLFNBQTlDLEVBQXlEO0FBQ3ZELFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXNDLHNCQUF0QixLQUFpRCxTQUFqRCxJQUE4RHZFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXNDLHNCQUFmLEtBQTBDLElBQTVHLEVBQWtIO0FBQ2hILGdCQUFNLElBQUkzQyxTQUFKLENBQWMsdUZBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9Cc0Msc0JBQXBCLEdBQTZDdkUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlc0Msc0JBQTVEO0FBQ0Q7O0FBRUQsVUFBSXZFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXVDLE9BQWYsS0FBMkJ6QyxTQUEvQixFQUEwQztBQUN4QyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWV1QyxPQUF0QixLQUFrQyxTQUF0QyxFQUFpRDtBQUMvQyxnQkFBTSxJQUFJNUMsU0FBSixDQUFjLGdFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQnVDLE9BQXBCLEdBQThCeEUsTUFBTSxDQUFDaUMsT0FBUCxDQUFldUMsT0FBN0M7QUFDRDs7QUFFRCxVQUFJeEUsTUFBTSxDQUFDaUMsT0FBUCxDQUFld0MsbUJBQWYsS0FBdUMxQyxTQUEzQyxFQUFzRDtBQUNwRCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWV3QyxtQkFBdEIsS0FBOEMsU0FBbEQsRUFBNkQ7QUFDM0QsZ0JBQU0sSUFBSTdDLFNBQUosQ0FBYyw0RUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3QyxtQkFBcEIsR0FBMEN6RSxNQUFNLENBQUNpQyxPQUFQLENBQWV3QyxtQkFBekQ7QUFDRDs7QUFFRCxVQUFJekUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlMEMsWUFBZixLQUFnQzVDLFNBQXBDLEVBQStDO0FBQzdDLFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTBDLFlBQXRCLEtBQXVDLFFBQTNDLEVBQXFEO0FBQ25ELGdCQUFNLElBQUkvQyxTQUFKLENBQWMsb0VBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CMEMsWUFBcEIsR0FBbUMzRSxNQUFNLENBQUNpQyxPQUFQLENBQWUwQyxZQUFsRDtBQUNBLGFBQUszRSxNQUFMLENBQVlpQyxPQUFaLENBQW9CaUQsSUFBcEIsR0FBMkJuRCxTQUEzQjtBQUNEOztBQUVELFVBQUkvQixNQUFNLENBQUNpQyxPQUFQLENBQWUyQyxjQUFmLEtBQWtDN0MsU0FBdEMsRUFBaUQ7QUFDL0Msb0RBQTBCL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlMkMsY0FBekMsRUFBeUQsK0JBQXpEO0FBRUEsYUFBSzVFLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0IyQyxjQUFwQixHQUFxQzVFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTJDLGNBQXBEO0FBQ0Q7O0FBRUQsVUFBSTVFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTRDLFFBQWYsS0FBNEI5QyxTQUFoQyxFQUEyQztBQUN6QyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWU0QyxRQUF0QixLQUFtQyxRQUFuQyxJQUErQzdFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTRDLFFBQWYsS0FBNEIsSUFBL0UsRUFBcUY7QUFDbkYsZ0JBQU0sSUFBSWpELFNBQUosQ0FBYyx3RUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0I0QyxRQUFwQixHQUErQjdFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTRDLFFBQTlDO0FBQ0Q7O0FBRUQsVUFBSTdFLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTZDLFlBQWYsS0FBZ0MvQyxTQUFwQyxFQUErQztBQUM3QyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWU2QyxZQUF0QixLQUF1QyxRQUEzQyxFQUFxRDtBQUNuRCxnQkFBTSxJQUFJbEQsU0FBSixDQUFjLG9FQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjZDLFlBQXBCLEdBQW1DOUUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlNkMsWUFBbEQ7QUFDRDs7QUFFRCxVQUFJOUUsTUFBTSxDQUFDaUMsT0FBUCxDQUFlK0MsbUJBQWYsS0FBdUNqRCxTQUEzQyxFQUFzRDtBQUNwRCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWUrQyxtQkFBdEIsS0FBOEMsU0FBbEQsRUFBNkQ7QUFDM0QsZ0JBQU0sSUFBSXBELFNBQUosQ0FBYyw0RUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0IrQyxtQkFBcEIsR0FBMENoRixNQUFNLENBQUNpQyxPQUFQLENBQWUrQyxtQkFBekQ7QUFDRDs7QUFFRCxVQUFJaEYsTUFBTSxDQUFDaUMsT0FBUCxDQUFlZ0QsVUFBZixLQUE4QmxELFNBQWxDLEVBQTZDO0FBQzNDLFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWdELFVBQXRCLEtBQXFDLFFBQXpDLEVBQW1EO0FBQ2pELGdCQUFNLElBQUlyRCxTQUFKLENBQWMsa0VBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CZ0QsVUFBcEIsR0FBaUNqRixNQUFNLENBQUNpQyxPQUFQLENBQWVnRCxVQUFoRDtBQUNEOztBQUVELFVBQUlqRixNQUFNLENBQUNpQyxPQUFQLENBQWVpRCxJQUFmLEtBQXdCbkQsU0FBNUIsRUFBdUM7QUFDckMsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlaUQsSUFBdEIsS0FBK0IsUUFBbkMsRUFBNkM7QUFDM0MsZ0JBQU0sSUFBSXRELFNBQUosQ0FBYyw0REFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSTVCLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWlELElBQWYsSUFBdUIsQ0FBdkIsSUFBNEJsRixNQUFNLENBQUNpQyxPQUFQLENBQWVpRCxJQUFmLElBQXVCLEtBQXZELEVBQThEO0FBQzVELGdCQUFNLElBQUlnQixVQUFKLENBQWUsNERBQWYsQ0FBTjtBQUNEOztBQUVELGFBQUtsRyxNQUFMLENBQVlpQyxPQUFaLENBQW9CaUQsSUFBcEIsR0FBMkJsRixNQUFNLENBQUNpQyxPQUFQLENBQWVpRCxJQUExQztBQUNBLGFBQUtsRixNQUFMLENBQVlpQyxPQUFaLENBQW9CMEMsWUFBcEIsR0FBbUM1QyxTQUFuQztBQUNEOztBQUVELFVBQUkvQixNQUFNLENBQUNpQyxPQUFQLENBQWVrRCxjQUFmLEtBQWtDcEQsU0FBdEMsRUFBaUQ7QUFDL0MsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFla0QsY0FBdEIsS0FBeUMsU0FBN0MsRUFBd0Q7QUFDdEQsZ0JBQU0sSUFBSXZELFNBQUosQ0FBYyx1RUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JrRCxjQUFwQixHQUFxQ25GLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWtELGNBQXBEO0FBQ0Q7O0FBRUQsVUFBSW5GLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW1ELGNBQWYsS0FBa0NyRCxTQUF0QyxFQUFpRDtBQUMvQyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWVtRCxjQUF0QixLQUF5QyxRQUE3QyxFQUF1RDtBQUNyRCxnQkFBTSxJQUFJeEQsU0FBSixDQUFjLHNFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQm1ELGNBQXBCLEdBQXFDcEYsTUFBTSxDQUFDaUMsT0FBUCxDQUFlbUQsY0FBcEQ7QUFDRDs7QUFFRCxVQUFJcEYsTUFBTSxDQUFDaUMsT0FBUCxDQUFlOEMsMkJBQWYsS0FBK0NoRCxTQUFuRCxFQUE4RDtBQUM1RCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWU4QywyQkFBdEIsS0FBc0QsUUFBMUQsRUFBb0U7QUFDbEUsZ0JBQU0sSUFBSW5ELFNBQUosQ0FBYyxtRkFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSTVCLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZThDLDJCQUFmLEdBQTZDLENBQWpELEVBQW9EO0FBQ2xELGdCQUFNLElBQUluRCxTQUFKLENBQWMsNEZBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9COEMsMkJBQXBCLEdBQWtEL0UsTUFBTSxDQUFDaUMsT0FBUCxDQUFlOEMsMkJBQWpFO0FBQ0Q7O0FBRUQsVUFBSS9FLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWlCLHVCQUFmLEtBQTJDbkIsU0FBL0MsRUFBMEQ7QUFDeEQsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlaUIsdUJBQXRCLEtBQWtELFFBQXRELEVBQWdFO0FBQzlELGdCQUFNLElBQUl0QixTQUFKLENBQWMsK0VBQWQsQ0FBTjtBQUNEOztBQUVELFlBQUk1QixNQUFNLENBQUNpQyxPQUFQLENBQWVpQix1QkFBZixJQUEwQyxDQUE5QyxFQUFpRDtBQUMvQyxnQkFBTSxJQUFJdEIsU0FBSixDQUFjLCtFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlCLHVCQUFwQixHQUE4Q2xELE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZWlCLHVCQUE3RDtBQUNEOztBQUVELFVBQUlsRCxNQUFNLENBQUNpQyxPQUFQLENBQWVvRCxtQkFBZixLQUF1Q3RELFNBQTNDLEVBQXNEO0FBQ3BELFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW9ELG1CQUF0QixLQUE4QyxTQUFsRCxFQUE2RDtBQUMzRCxnQkFBTSxJQUFJekQsU0FBSixDQUFjLDRFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQm9ELG1CQUFwQixHQUEwQ3JGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZW9ELG1CQUF6RDtBQUNEOztBQUVELFVBQUlyRixNQUFNLENBQUNpQyxPQUFQLENBQWVxRCxnQ0FBZixLQUFvRHZELFNBQXhELEVBQW1FO0FBQ2pFLFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXFELGdDQUF0QixLQUEyRCxTQUEvRCxFQUEwRTtBQUN4RSxnQkFBTSxJQUFJMUQsU0FBSixDQUFjLHlGQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQnFELGdDQUFwQixHQUF1RHRGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXFELGdDQUF0RTtBQUNEOztBQUVELFVBQUl0RixNQUFNLENBQUNpQyxPQUFQLENBQWV3RCxVQUFmLEtBQThCMUQsU0FBbEMsRUFBNkM7QUFDM0MsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFld0QsVUFBdEIsS0FBcUMsUUFBekMsRUFBbUQ7QUFDakQsZ0JBQU0sSUFBSTdELFNBQUosQ0FBYyxrRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQ3pGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXdELFVBQWhEO0FBQ0Q7O0FBRUQsVUFBSXpGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXlELFFBQWYsS0FBNEIzRCxTQUFoQyxFQUEyQztBQUN6QyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWV5RCxRQUF0QixLQUFtQyxRQUFuQyxJQUErQzFGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXlELFFBQWYsS0FBNEIsSUFBL0UsRUFBcUY7QUFDbkYsZ0JBQU0sSUFBSTlELFNBQUosQ0FBYyx3RUFBZCxDQUFOO0FBQ0Q7O0FBRUQsWUFBSTVCLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXlELFFBQWYsR0FBMEIsVUFBOUIsRUFBMEM7QUFDeEMsZ0JBQU0sSUFBSTlELFNBQUosQ0FBYyxrRUFBZCxDQUFOO0FBQ0QsU0FGRCxNQUVPLElBQUk1QixNQUFNLENBQUNpQyxPQUFQLENBQWV5RCxRQUFmLEdBQTBCLENBQUMsQ0FBL0IsRUFBa0M7QUFDdkMsZ0JBQU0sSUFBSTlELFNBQUosQ0FBYywwREFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J5RCxRQUFwQixHQUErQjFGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZXlELFFBQWYsR0FBMEIsQ0FBekQ7QUFDRDs7QUFFRCxVQUFJMUYsTUFBTSxDQUFDaUMsT0FBUCxDQUFlMkQsc0JBQWYsS0FBMEM3RCxTQUE5QyxFQUF5RDtBQUN2RCxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWUyRCxzQkFBdEIsS0FBaUQsU0FBckQsRUFBZ0U7QUFDOUQsZ0JBQU0sSUFBSWhFLFNBQUosQ0FBYywrRUFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0IyRCxzQkFBcEIsR0FBNkM1RixNQUFNLENBQUNpQyxPQUFQLENBQWUyRCxzQkFBNUQ7QUFDRCxPQU5ELE1BTU87QUFDTHBILFFBQUFBLGlDQUFpQztBQUNsQzs7QUFFRCxVQUFJd0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlNEQsY0FBZixLQUFrQzlELFNBQXRDLEVBQWlEO0FBQy9DLFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTRELGNBQXRCLEtBQXlDLFNBQTdDLEVBQXdEO0FBQ3RELGdCQUFNLElBQUlqRSxTQUFKLENBQWMsdUVBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CNEQsY0FBcEIsR0FBcUM3RixNQUFNLENBQUNpQyxPQUFQLENBQWU0RCxjQUFwRDtBQUNEOztBQUVELFVBQUk3RixNQUFNLENBQUNpQyxPQUFQLENBQWU2RCxNQUFmLEtBQTBCL0QsU0FBOUIsRUFBeUM7QUFDdkMsWUFBSSxPQUFPL0IsTUFBTSxDQUFDaUMsT0FBUCxDQUFlNkQsTUFBdEIsS0FBaUMsU0FBckMsRUFBZ0Q7QUFDOUMsZ0JBQU0sSUFBSWxFLFNBQUosQ0FBYywrREFBZCxDQUFOO0FBQ0Q7O0FBRUQsYUFBSzVCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0I2RCxNQUFwQixHQUE2QjlGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZTZELE1BQTVDO0FBQ0Q7O0FBRUQsVUFBSTlGLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZThELGFBQWYsS0FBaUNoRSxTQUFyQyxFQUFnRDtBQUM5QyxZQUFJLE9BQU8vQixNQUFNLENBQUNpQyxPQUFQLENBQWU4RCxhQUF0QixLQUF3QyxRQUE1QyxFQUFzRDtBQUNwRCxnQkFBTSxJQUFJbkUsU0FBSixDQUFjLHFFQUFkLENBQU47QUFDRDs7QUFFRCxhQUFLNUIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjhELGFBQXBCLEdBQW9DL0YsTUFBTSxDQUFDaUMsT0FBUCxDQUFlOEQsYUFBbkQ7QUFDRDs7QUFFRCxVQUFJL0YsTUFBTSxDQUFDaUMsT0FBUCxDQUFlK0QsY0FBZixLQUFrQ2pFLFNBQXRDLEVBQWlEO0FBQy9DLFlBQUksT0FBTy9CLE1BQU0sQ0FBQ2lDLE9BQVAsQ0FBZStELGNBQXRCLEtBQXlDLFNBQTdDLEVBQXdEO0FBQ3RELGdCQUFNLElBQUlwRSxTQUFKLENBQWMsdUVBQWQsQ0FBTjtBQUNEOztBQUVELGFBQUs1QixNQUFMLENBQVlpQyxPQUFaLENBQW9CK0QsY0FBcEIsR0FBcUNoRyxNQUFNLENBQUNpQyxPQUFQLENBQWUrRCxjQUFwRDtBQUNEO0FBQ0Y7O0FBRUQsUUFBSUcsa0JBQWtCLEdBQUcsS0FBS25HLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JzQix3QkFBN0M7O0FBQ0EsUUFBSTRDLGtCQUFrQixDQUFDQyxhQUFuQixLQUFxQ3JFLFNBQXpDLEVBQW9EO0FBQ2xEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQW9FLE1BQUFBLGtCQUFrQixHQUFHRSxNQUFNLENBQUNDLE1BQVAsQ0FBY0gsa0JBQWQsRUFBa0M7QUFDckRDLFFBQUFBLGFBQWEsRUFBRTtBQUNiRyxVQUFBQSxLQUFLLEVBQUVDLG1CQUFVQztBQURKO0FBRHNDLE9BQWxDLENBQXJCO0FBS0Q7O0FBRUQsU0FBS3ZHLGFBQUwsR0FBcUIsOEJBQW9CaUcsa0JBQXBCLENBQXJCO0FBRUEsU0FBS3hGLEtBQUwsR0FBYSxLQUFLK0YsV0FBTCxFQUFiO0FBQ0EsU0FBS3ZHLGFBQUwsR0FBcUIsS0FBckI7QUFDQSxTQUFLQyxzQkFBTCxHQUE4QixDQUFDdUcsTUFBTSxDQUFDQyxJQUFQLENBQVksQ0FBQyxDQUFELEVBQUksQ0FBSixFQUFPLENBQVAsRUFBVSxDQUFWLEVBQWEsQ0FBYixFQUFnQixDQUFoQixFQUFtQixDQUFuQixFQUFzQixDQUF0QixDQUFaLENBQUQsQ0FBOUIsQ0FscUIyQyxDQW9xQjNDO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsU0FBS3ZHLGdCQUFMLEdBQXdCLENBQXhCO0FBQ0EsU0FBS0MsVUFBTCxHQUFrQixLQUFsQjtBQUNBLFNBQUtHLE1BQUwsR0FBYyxLQUFkO0FBQ0EsU0FBS1ksYUFBTCxHQUFxQnNGLE1BQU0sQ0FBQ0UsS0FBUCxDQUFhLENBQWIsQ0FBckI7QUFFQSxTQUFLdEcsc0JBQUwsR0FBOEIsQ0FBOUI7QUFDQSxTQUFLQyxvQkFBTCxHQUE0QixJQUFJc0csMENBQUosRUFBNUI7QUFFQSxTQUFLOUYsS0FBTCxHQUFhLEtBQUsrRixLQUFMLENBQVdDLFdBQXhCOztBQUVBLFNBQUt0Rix1QkFBTCxHQUErQixNQUFNO0FBQ25DLFdBQUtYLFNBQUwsQ0FBZWtHLFdBQWYsQ0FBMkJDLGFBQUtDLFNBQWhDO0FBQ0EsV0FBS0MsaUJBQUw7QUFDRCxLQUhEO0FBSUQ7O0FBRURDLEVBQUFBLE9BQU8sQ0FBQ0MsZUFBRCxFQUEwQztBQUMvQyxRQUFJLEtBQUt0RyxLQUFMLEtBQWUsS0FBSytGLEtBQUwsQ0FBV0MsV0FBOUIsRUFBMkM7QUFDekMsWUFBTSxJQUFJTyx1QkFBSixDQUFvQixzREFBc0QsS0FBS3ZHLEtBQUwsQ0FBV3dHLElBQWpFLEdBQXdFLFVBQTVGLENBQU47QUFDRDs7QUFFRCxRQUFJRixlQUFKLEVBQXFCO0FBQ25CLFlBQU1HLFNBQVMsR0FBSUMsR0FBRCxJQUFpQjtBQUNqQyxhQUFLQyxjQUFMLENBQW9CLE9BQXBCLEVBQTZCQyxPQUE3QjtBQUNBTixRQUFBQSxlQUFlLENBQUNJLEdBQUQsQ0FBZjtBQUNELE9BSEQ7O0FBS0EsWUFBTUUsT0FBTyxHQUFJRixHQUFELElBQWdCO0FBQzlCLGFBQUtDLGNBQUwsQ0FBb0IsU0FBcEIsRUFBK0JGLFNBQS9CO0FBQ0FILFFBQUFBLGVBQWUsQ0FBQ0ksR0FBRCxDQUFmO0FBQ0QsT0FIRDs7QUFLQSxXQUFLRyxJQUFMLENBQVUsU0FBVixFQUFxQkosU0FBckI7QUFDQSxXQUFLSSxJQUFMLENBQVUsT0FBVixFQUFtQkQsT0FBbkI7QUFDRDs7QUFFRCxTQUFLRSxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2dCLFVBQTdCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQWdFRUMsRUFBQUEsRUFBRSxDQUFDQyxLQUFELEVBQXlCQyxRQUF6QixFQUE2RDtBQUM3RCxXQUFPLE1BQU1GLEVBQU4sQ0FBU0MsS0FBVCxFQUFnQkMsUUFBaEIsQ0FBUDtBQUNEO0FBRUQ7QUFDRjtBQUNBOzs7QUF1REVDLEVBQUFBLElBQUksQ0FBQ0YsS0FBRCxFQUF5QixHQUFHRyxJQUE1QixFQUF5QztBQUMzQyxXQUFPLE1BQU1ELElBQU4sQ0FBV0YsS0FBWCxFQUFrQixHQUFHRyxJQUFyQixDQUFQO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7QUFDQTtBQUNBOzs7QUFDRUMsRUFBQUEsS0FBSyxHQUFHO0FBQ04sU0FBS1AsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNEO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRUMsRUFBQUEsb0JBQW9CLEdBQUc7QUFDckIsVUFBTUMsTUFBTSxHQUFHLEtBQUtDLGtCQUFMLEVBQWY7O0FBRUEsUUFBSSxLQUFLekksTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlELElBQXhCLEVBQThCO0FBQzVCLGFBQU8sS0FBS3dELGFBQUwsQ0FBbUIsS0FBSzFJLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JpRCxJQUF2QyxFQUE2QyxLQUFLbEYsTUFBTCxDQUFZaUMsT0FBWixDQUFvQitDLG1CQUFqRSxFQUFzRndELE1BQXRGLENBQVA7QUFDRCxLQUZELE1BRU87QUFDTCxhQUFPLElBQUlHLDhCQUFKLEdBQXFCQyxjQUFyQixDQUFvQztBQUN6Qy9HLFFBQUFBLE1BQU0sRUFBRSxLQUFLN0IsTUFBTCxDQUFZNkIsTUFEcUI7QUFFekM4QyxRQUFBQSxZQUFZLEVBQUUsS0FBSzNFLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0IwQyxZQUZPO0FBR3pDa0UsUUFBQUEsT0FBTyxFQUFFLEtBQUs3SSxNQUFMLENBQVlpQyxPQUFaLENBQW9Ca0IsY0FIWTtBQUl6Q3FGLFFBQUFBLE1BQU0sRUFBRUE7QUFKaUMsT0FBcEMsRUFLSixDQUFDZCxHQUFELEVBQU14QyxJQUFOLEtBQWU7QUFDaEIsWUFBSXdDLEdBQUosRUFBUztBQUNQLGNBQUlBLEdBQUcsQ0FBQ0YsSUFBSixLQUFhLFlBQWpCLEVBQStCO0FBQzdCO0FBQ0Q7O0FBRUQsZUFBS1csSUFBTCxDQUFVLFNBQVYsRUFBcUIsSUFBSVosdUJBQUosQ0FBb0JHLEdBQUcsQ0FBQ29CLE9BQXhCLEVBQWlDLGFBQWpDLENBQXJCO0FBQ0QsU0FORCxNQU1PO0FBQ0wsZUFBS0osYUFBTCxDQUFtQnhELElBQW5CLEVBQTBCLEtBQUtsRixNQUFMLENBQVlpQyxPQUFaLENBQW9CK0MsbUJBQTlDLEVBQW1Fd0QsTUFBbkU7QUFDRDtBQUNGLE9BZk0sQ0FBUDtBQWdCRDtBQUNGO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRU8sRUFBQUEsaUJBQWlCLENBQUNDLFdBQUQsRUFBOEQ7QUFDN0UsUUFBSSxDQUFDLEtBQUt2SSxNQUFWLEVBQWtCO0FBQ2hCLFdBQUt3SSxpQkFBTDtBQUNBLFdBQUtDLGlCQUFMO0FBQ0EsV0FBS0MsZUFBTDtBQUNBLFdBQUtDLGVBQUw7O0FBQ0EsVUFBSUosV0FBVyxLQUFLdkosWUFBWSxDQUFDRSxRQUFqQyxFQUEyQztBQUN6QyxhQUFLd0ksSUFBTCxDQUFVLFdBQVY7QUFDRCxPQUZELE1BRU8sSUFBSWEsV0FBVyxLQUFLdkosWUFBWSxDQUFDRyxLQUFqQyxFQUF3QztBQUM3Q25CLFFBQUFBLE9BQU8sQ0FBQzRLLFFBQVIsQ0FBaUIsTUFBTTtBQUNyQixlQUFLbEIsSUFBTCxDQUFVLEtBQVY7QUFDRCxTQUZEO0FBR0Q7O0FBRUQsWUFBTWpILE9BQU8sR0FBRyxLQUFLQSxPQUFyQjs7QUFDQSxVQUFJQSxPQUFKLEVBQWE7QUFDWCxjQUFNd0csR0FBRyxHQUFHLElBQUk0QixvQkFBSixDQUFpQiw2Q0FBakIsRUFBZ0UsUUFBaEUsQ0FBWjtBQUNBcEksUUFBQUEsT0FBTyxDQUFDcUksUUFBUixDQUFpQjdCLEdBQWpCO0FBQ0EsYUFBS3hHLE9BQUwsR0FBZWEsU0FBZjtBQUNEOztBQUVELFdBQUt0QixNQUFMLEdBQWMsSUFBZDtBQUNBLFdBQUtDLFVBQUwsR0FBa0JxQixTQUFsQjtBQUNEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFMkUsRUFBQUEsV0FBVyxHQUFHO0FBQ1osVUFBTS9GLEtBQUssR0FBRyxJQUFJNkksY0FBSixDQUFVLEtBQUt4SixNQUFMLENBQVlpQyxPQUFaLENBQW9CdEIsS0FBOUIsQ0FBZDtBQUNBQSxJQUFBQSxLQUFLLENBQUNxSCxFQUFOLENBQVMsT0FBVCxFQUFtQmMsT0FBRCxJQUFhO0FBQzdCLFdBQUtYLElBQUwsQ0FBVSxPQUFWLEVBQW1CVyxPQUFuQjtBQUNELEtBRkQ7QUFHQSxXQUFPbkksS0FBUDtBQUNEO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRThJLEVBQUFBLHVCQUF1QixDQUFDWCxPQUFELEVBQW1CWSxPQUFuQixFQUEwQztBQUMvRCxXQUFPLElBQUlDLHlCQUFKLENBQXNCYixPQUF0QixFQUErQixLQUFLbkksS0FBcEMsRUFBMkMrSSxPQUEzQyxFQUFvRCxLQUFLMUosTUFBTCxDQUFZaUMsT0FBaEUsQ0FBUDtBQUNEOztBQUVEeUcsRUFBQUEsYUFBYSxDQUFDeEQsSUFBRCxFQUFlRixtQkFBZixFQUE2Q3dELE1BQTdDLEVBQWtFO0FBQzdFLFVBQU1vQixXQUFXLEdBQUc7QUFDbEJDLE1BQUFBLElBQUksRUFBRSxLQUFLL0ksV0FBTCxHQUFtQixLQUFLQSxXQUFMLENBQWlCZSxNQUFwQyxHQUE2QyxLQUFLN0IsTUFBTCxDQUFZNkIsTUFEN0M7QUFFbEJxRCxNQUFBQSxJQUFJLEVBQUUsS0FBS3BFLFdBQUwsR0FBbUIsS0FBS0EsV0FBTCxDQUFpQm9FLElBQXBDLEdBQTJDQSxJQUYvQjtBQUdsQkosTUFBQUEsWUFBWSxFQUFFLEtBQUs5RSxNQUFMLENBQVlpQyxPQUFaLENBQW9CNkM7QUFIaEIsS0FBcEI7QUFNQSxRQUFJZ0Ysb0JBQUosQ0FBY0YsV0FBZCxFQUEyQnBCLE1BQTNCLEVBQW1DeEQsbUJBQW5DLEVBQXdEK0UsT0FBeEQsQ0FBZ0UsQ0FBQ3JDLEdBQUQsRUFBTXRHLE1BQU4sS0FBaUI7QUFDL0UsVUFBSXNHLEdBQUosRUFBUztBQUNQLFlBQUlBLEdBQUcsQ0FBQ0YsSUFBSixLQUFhLFlBQWpCLEVBQStCO0FBQzdCO0FBQ0Q7O0FBRUQsZUFBTyxLQUFLd0MsV0FBTCxDQUFpQnRDLEdBQWpCLENBQVA7QUFDRDs7QUFFRHRHLE1BQUFBLE1BQU0sR0FBR0EsTUFBVDtBQUNBQSxNQUFBQSxNQUFNLENBQUM0RyxFQUFQLENBQVUsT0FBVixFQUFvQmlDLEtBQUQsSUFBVztBQUFFLGFBQUtELFdBQUwsQ0FBaUJDLEtBQWpCO0FBQTBCLE9BQTFEO0FBQ0E3SSxNQUFBQSxNQUFNLENBQUM0RyxFQUFQLENBQVUsT0FBVixFQUFtQixNQUFNO0FBQUUsYUFBS2tDLFdBQUw7QUFBcUIsT0FBaEQ7QUFDQTlJLE1BQUFBLE1BQU0sQ0FBQzRHLEVBQVAsQ0FBVSxLQUFWLEVBQWlCLE1BQU07QUFBRSxhQUFLbUMsU0FBTDtBQUFtQixPQUE1QztBQUNBL0ksTUFBQUEsTUFBTSxDQUFDZ0osWUFBUCxDQUFvQixJQUFwQixFQUEwQnZMLHdCQUExQjtBQUVBLFdBQUtrQyxTQUFMLEdBQWlCLElBQUlzSixrQkFBSixDQUFjakosTUFBZCxFQUFzQixLQUFLcEIsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmdELFVBQTFDLEVBQXNELEtBQUt0RSxLQUEzRCxDQUFqQjtBQUNBLFdBQUtJLFNBQUwsQ0FBZWlILEVBQWYsQ0FBa0IsUUFBbEIsRUFBNkJzQyxTQUFELElBQWU7QUFBRSxhQUFLbkMsSUFBTCxDQUFVLFFBQVYsRUFBb0JtQyxTQUFwQjtBQUFpQyxPQUE5RTtBQUVBLFdBQUtsSixNQUFMLEdBQWNBLE1BQWQ7QUFFQSxXQUFLWCxNQUFMLEdBQWMsS0FBZDtBQUNBLFdBQUtFLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZSxrQkFBa0IsS0FBS3ZLLE1BQUwsQ0FBWTZCLE1BQTlCLEdBQXVDLEdBQXZDLEdBQTZDLEtBQUs3QixNQUFMLENBQVlpQyxPQUFaLENBQW9CaUQsSUFBaEY7QUFFQSxXQUFLc0YsWUFBTDtBQUNBLFdBQUsxQyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBVzBELGFBQTdCO0FBQ0QsS0F6QkQ7QUEwQkQ7QUFFRDtBQUNGO0FBQ0E7OztBQUNFckIsRUFBQUEsZUFBZSxHQUFHO0FBQ2hCLFFBQUksS0FBS2hJLE1BQVQsRUFBaUI7QUFDZixXQUFLQSxNQUFMLENBQVlzSixPQUFaO0FBQ0Q7QUFDRjtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VqQyxFQUFBQSxrQkFBa0IsR0FBRztBQUNuQixVQUFNa0MsVUFBVSxHQUFHLElBQUlDLG9DQUFKLEVBQW5CO0FBQ0EsU0FBS3RKLFlBQUwsR0FBb0J1SixVQUFVLENBQUMsTUFBTTtBQUNuQ0YsTUFBQUEsVUFBVSxDQUFDRyxLQUFYO0FBQ0EsV0FBSzNILGNBQUw7QUFDRCxLQUg2QixFQUczQixLQUFLbkQsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmtCLGNBSE8sQ0FBOUI7QUFJQSxXQUFPd0gsVUFBVSxDQUFDbkMsTUFBbEI7QUFDRDtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VwQixFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixTQUFLMkQsZ0JBQUw7QUFDQSxVQUFNbEMsT0FBTyxHQUFHLEtBQUs3SSxNQUFMLENBQVlpQyxPQUFaLENBQW9CYSxhQUFwQzs7QUFDQSxRQUFJK0YsT0FBTyxHQUFHLENBQWQsRUFBaUI7QUFDZixXQUFLdEgsV0FBTCxHQUFtQnNKLFVBQVUsQ0FBQyxNQUFNO0FBQ2xDLGFBQUsvSCxhQUFMO0FBQ0QsT0FGNEIsRUFFMUIrRixPQUYwQixDQUE3QjtBQUdEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFbUMsRUFBQUEsa0JBQWtCLEdBQUc7QUFDbkIsU0FBSzlCLGlCQUFMLEdBRG1CLENBQ087O0FBQzFCLFVBQU1oSSxPQUFPLEdBQUcsS0FBS0EsT0FBckI7QUFDQSxVQUFNMkgsT0FBTyxHQUFJM0gsT0FBTyxDQUFDMkgsT0FBUixLQUFvQjlHLFNBQXJCLEdBQWtDYixPQUFPLENBQUMySCxPQUExQyxHQUFvRCxLQUFLN0ksTUFBTCxDQUFZaUMsT0FBWixDQUFvQm1ELGNBQXhGOztBQUNBLFFBQUl5RCxPQUFKLEVBQWE7QUFDWCxXQUFLckgsWUFBTCxHQUFvQnFKLFVBQVUsQ0FBQyxNQUFNO0FBQ25DLGFBQUt6RixjQUFMO0FBQ0QsT0FGNkIsRUFFM0J5RCxPQUYyQixDQUE5QjtBQUdEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFb0MsRUFBQUEsZ0JBQWdCLEdBQUc7QUFDakIsU0FBSzlCLGVBQUw7QUFDQSxTQUFLMUgsVUFBTCxHQUFrQm9KLFVBQVUsQ0FBQyxNQUFNO0FBQ2pDLFdBQUtLLFlBQUw7QUFDRCxLQUYyQixFQUV6QixLQUFLbEwsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlCLHVCQUZLLENBQTVCO0FBR0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFQyxFQUFBQSxjQUFjLEdBQUc7QUFDZixVQUFNMkYsT0FBTyxHQUFJLHdCQUF1QixLQUFLOUksTUFBTCxDQUFZNkIsTUFBTyxHQUFFLEtBQUs3QixNQUFMLENBQVlpQyxPQUFaLENBQW9CaUQsSUFBcEIsR0FBNEIsSUFBRyxLQUFLbEYsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlELElBQUssRUFBeEQsR0FBNkQsS0FBSSxLQUFLbEYsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjBDLFlBQWEsRUFBRSxPQUFNLEtBQUszRSxNQUFMLENBQVlpQyxPQUFaLENBQW9Ca0IsY0FBZSxJQUExTTtBQUNBLFNBQUt4QyxLQUFMLENBQVc0SixHQUFYLENBQWV6QixPQUFmO0FBQ0EsU0FBS1gsSUFBTCxDQUFVLFNBQVYsRUFBcUIsSUFBSVosdUJBQUosQ0FBb0J1QixPQUFwQixFQUE2QixVQUE3QixDQUFyQjtBQUNBLFNBQUt4SCxZQUFMLEdBQW9CUyxTQUFwQjtBQUNBLFNBQUtvSixhQUFMLENBQW1CLGdCQUFuQjtBQUNEO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRXJJLEVBQUFBLGFBQWEsR0FBRztBQUNkLFVBQU1nRyxPQUFPLEdBQUksK0JBQThCLEtBQUs5SSxNQUFMLENBQVlpQyxPQUFaLENBQW9CYSxhQUFjLElBQWpGO0FBQ0EsU0FBS25DLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZXpCLE9BQWY7QUFDQSxTQUFLcUMsYUFBTCxDQUFtQixhQUFuQixFQUFrQyxJQUFJNUQsdUJBQUosQ0FBb0J1QixPQUFwQixFQUE2QixVQUE3QixDQUFsQztBQUNEO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRTFELEVBQUFBLGNBQWMsR0FBRztBQUNmLFNBQUs1RCxZQUFMLEdBQW9CTyxTQUFwQjtBQUNBLFVBQU1iLE9BQU8sR0FBRyxLQUFLQSxPQUFyQjtBQUNBQSxJQUFBQSxPQUFPLENBQUNrSyxNQUFSO0FBQ0EsVUFBTXZDLE9BQU8sR0FBSTNILE9BQU8sQ0FBQzJILE9BQVIsS0FBb0I5RyxTQUFyQixHQUFrQ2IsT0FBTyxDQUFDMkgsT0FBMUMsR0FBb0QsS0FBSzdJLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JtRCxjQUF4RjtBQUNBLFVBQU0wRCxPQUFPLEdBQUcsNENBQTRDRCxPQUE1QyxHQUFzRCxJQUF0RTtBQUNBM0gsSUFBQUEsT0FBTyxDQUFDK0ksS0FBUixHQUFnQixJQUFJWCxvQkFBSixDQUFpQlIsT0FBakIsRUFBMEIsVUFBMUIsQ0FBaEI7QUFDRDtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VvQyxFQUFBQSxZQUFZLEdBQUc7QUFDYixTQUFLekosVUFBTCxHQUFrQk0sU0FBbEI7QUFDQSxTQUFLb0csSUFBTCxDQUFVLE9BQVY7QUFDQSxTQUFLTCxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2dCLFVBQTdCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFa0IsRUFBQUEsaUJBQWlCLEdBQUc7QUFDbEIsUUFBSSxLQUFLM0gsWUFBVCxFQUF1QjtBQUNyQitKLE1BQUFBLFlBQVksQ0FBQyxLQUFLL0osWUFBTixDQUFaO0FBQ0EsV0FBS0EsWUFBTCxHQUFvQlMsU0FBcEI7QUFDRDtBQUNGO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRWdKLEVBQUFBLGdCQUFnQixHQUFHO0FBQ2pCLFFBQUksS0FBS3hKLFdBQVQsRUFBc0I7QUFDcEI4SixNQUFBQSxZQUFZLENBQUMsS0FBSzlKLFdBQU4sQ0FBWjtBQUNBLFdBQUtBLFdBQUwsR0FBbUJRLFNBQW5CO0FBQ0Q7QUFDRjtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VtSCxFQUFBQSxpQkFBaUIsR0FBRztBQUNsQixRQUFJLEtBQUsxSCxZQUFULEVBQXVCO0FBQ3JCNkosTUFBQUEsWUFBWSxDQUFDLEtBQUs3SixZQUFOLENBQVo7QUFDQSxXQUFLQSxZQUFMLEdBQW9CTyxTQUFwQjtBQUNEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFb0gsRUFBQUEsZUFBZSxHQUFHO0FBQ2hCLFFBQUksS0FBSzFILFVBQVQsRUFBcUI7QUFDbkI0SixNQUFBQSxZQUFZLENBQUMsS0FBSzVKLFVBQU4sQ0FBWjtBQUNBLFdBQUtBLFVBQUwsR0FBa0JNLFNBQWxCO0FBQ0Q7QUFDRjtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0UrRixFQUFBQSxZQUFZLENBQUN3RCxRQUFELEVBQWtCO0FBQzVCLFFBQUksS0FBS3RLLEtBQUwsS0FBZXNLLFFBQW5CLEVBQTZCO0FBQzNCLFdBQUszSyxLQUFMLENBQVc0SixHQUFYLENBQWUsc0JBQXNCZSxRQUFRLENBQUM5RCxJQUE5QztBQUNBO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLeEcsS0FBTCxJQUFjLEtBQUtBLEtBQUwsQ0FBV3VLLElBQTdCLEVBQW1DO0FBQ2pDLFdBQUt2SyxLQUFMLENBQVd1SyxJQUFYLENBQWdCQyxJQUFoQixDQUFxQixJQUFyQixFQUEyQkYsUUFBM0I7QUFDRDs7QUFFRCxTQUFLM0ssS0FBTCxDQUFXNEosR0FBWCxDQUFlLG9CQUFvQixLQUFLdkosS0FBTCxHQUFhLEtBQUtBLEtBQUwsQ0FBV3dHLElBQXhCLEdBQStCLFdBQW5ELElBQWtFLE1BQWxFLEdBQTJFOEQsUUFBUSxDQUFDOUQsSUFBbkc7QUFDQSxTQUFLeEcsS0FBTCxHQUFhc0ssUUFBYjs7QUFFQSxRQUFJLEtBQUt0SyxLQUFMLENBQVd5SyxLQUFmLEVBQXNCO0FBQ3BCLFdBQUt6SyxLQUFMLENBQVd5SyxLQUFYLENBQWlCQyxLQUFqQixDQUF1QixJQUF2QjtBQUNEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFQyxFQUFBQSxlQUFlLENBQWtDQyxTQUFsQyxFQUFpRjtBQUM5RixVQUFNbEMsT0FBTyxHQUFHLEtBQUsxSSxLQUFMLENBQVc2SyxNQUFYLENBQWtCRCxTQUFsQixDQUFoQjs7QUFFQSxRQUFJLENBQUNsQyxPQUFMLEVBQWM7QUFDWixZQUFNLElBQUl6RCxLQUFKLENBQVcsYUFBWTJGLFNBQVUsZUFBYyxLQUFLNUssS0FBTCxDQUFXd0csSUFBSyxHQUEvRCxDQUFOO0FBQ0Q7O0FBRUQsV0FBT2tDLE9BQVA7QUFDRDtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0V5QixFQUFBQSxhQUFhLENBQWtDUyxTQUFsQyxFQUFnRCxHQUFHeEQsSUFBbkQsRUFBc0c7QUFDakgsVUFBTXNCLE9BQU8sR0FBRyxLQUFLMUksS0FBTCxDQUFXNkssTUFBWCxDQUFrQkQsU0FBbEIsQ0FBaEI7O0FBQ0EsUUFBSWxDLE9BQUosRUFBYTtBQUNYQSxNQUFBQSxPQUFPLENBQUNnQyxLQUFSLENBQWMsSUFBZCxFQUFvQnRELElBQXBCO0FBQ0QsS0FGRCxNQUVPO0FBQ0wsV0FBS0QsSUFBTCxDQUFVLE9BQVYsRUFBbUIsSUFBSWxDLEtBQUosQ0FBVyxhQUFZMkYsU0FBVSxlQUFjLEtBQUs1SyxLQUFMLENBQVd3RyxJQUFLLEdBQS9ELENBQW5CO0FBQ0EsV0FBS2EsS0FBTDtBQUNEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFMkIsRUFBQUEsV0FBVyxDQUFDQyxLQUFELEVBQWU7QUFDeEIsUUFBSSxLQUFLakosS0FBTCxLQUFlLEtBQUsrRixLQUFMLENBQVdnQixVQUExQixJQUF3QyxLQUFLL0csS0FBTCxLQUFlLEtBQUsrRixLQUFMLENBQVcrRSxzQkFBdEUsRUFBOEY7QUFDNUYsWUFBTWhELE9BQU8sR0FBSSx3QkFBdUIsS0FBSzlJLE1BQUwsQ0FBWTZCLE1BQU8sSUFBRyxLQUFLN0IsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlELElBQUssTUFBSytFLEtBQUssQ0FBQ25CLE9BQVEsRUFBMUc7QUFDQSxXQUFLbkksS0FBTCxDQUFXNEosR0FBWCxDQUFlekIsT0FBZjtBQUNBLFdBQUtYLElBQUwsQ0FBVSxTQUFWLEVBQXFCLElBQUlaLHVCQUFKLENBQW9CdUIsT0FBcEIsRUFBNkIsU0FBN0IsQ0FBckI7QUFDRCxLQUpELE1BSU87QUFDTCxZQUFNQSxPQUFPLEdBQUkscUJBQW9CbUIsS0FBSyxDQUFDbkIsT0FBUSxFQUFuRDtBQUNBLFdBQUtuSSxLQUFMLENBQVc0SixHQUFYLENBQWV6QixPQUFmO0FBQ0EsV0FBS1gsSUFBTCxDQUFVLE9BQVYsRUFBbUIsSUFBSVosdUJBQUosQ0FBb0J1QixPQUFwQixFQUE2QixTQUE3QixDQUFuQjtBQUNEOztBQUNELFNBQUtxQyxhQUFMLENBQW1CLGFBQW5CLEVBQWtDbEIsS0FBbEM7QUFDRDtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VFLEVBQUFBLFNBQVMsR0FBRztBQUNWLFNBQUt4SixLQUFMLENBQVc0SixHQUFYLENBQWUsY0FBZjs7QUFDQSxRQUFJLEtBQUt2SixLQUFMLEtBQWUsS0FBSytGLEtBQUwsQ0FBV3VCLEtBQTlCLEVBQXFDO0FBQ25DLFlBQU0yQixLQUFvQixHQUFHLElBQUloRSxLQUFKLENBQVUsZ0JBQVYsQ0FBN0I7QUFDQWdFLE1BQUFBLEtBQUssQ0FBQzhCLElBQU4sR0FBYSxZQUFiO0FBQ0EsV0FBSy9CLFdBQUwsQ0FBaUJDLEtBQWpCO0FBQ0Q7QUFDRjtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VDLEVBQUFBLFdBQVcsR0FBRztBQUNaLFNBQUt2SixLQUFMLENBQVc0SixHQUFYLENBQWUsbUJBQW1CLEtBQUt2SyxNQUFMLENBQVk2QixNQUEvQixHQUF3QyxHQUF4QyxHQUE4QyxLQUFLN0IsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlELElBQWxFLEdBQXlFLFNBQXhGOztBQUNBLFFBQUksS0FBS2xFLEtBQUwsS0FBZSxLQUFLK0YsS0FBTCxDQUFXaUYsU0FBOUIsRUFBeUM7QUFDdkMsV0FBS3JMLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZSxrQkFBa0IsS0FBS3pKLFdBQUwsQ0FBa0JlLE1BQXBDLEdBQTZDLEdBQTdDLEdBQW1ELEtBQUtmLFdBQUwsQ0FBa0JvRSxJQUFwRjtBQUVBLFdBQUtpRyxhQUFMLENBQW1CLFdBQW5CO0FBQ0QsS0FKRCxNQUlPLElBQUksS0FBS25LLEtBQUwsS0FBZSxLQUFLK0YsS0FBTCxDQUFXa0YsdUJBQTlCLEVBQXVEO0FBQzVELFlBQU1wSyxNQUFNLEdBQUcsS0FBS2YsV0FBTCxHQUFtQixLQUFLQSxXQUFMLENBQWlCZSxNQUFwQyxHQUE2QyxLQUFLN0IsTUFBTCxDQUFZNkIsTUFBeEU7QUFDQSxZQUFNcUQsSUFBSSxHQUFHLEtBQUtwRSxXQUFMLEdBQW1CLEtBQUtBLFdBQUwsQ0FBaUJvRSxJQUFwQyxHQUEyQyxLQUFLbEYsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlELElBQTVFO0FBQ0EsV0FBS3ZFLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZSxpREFBaUQxSSxNQUFqRCxHQUEwRCxHQUExRCxHQUFnRXFELElBQS9FO0FBRUEsV0FBS2lHLGFBQUwsQ0FBbUIsT0FBbkI7QUFDRCxLQU5NLE1BTUE7QUFDTCxXQUFLckQsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFa0MsRUFBQUEsWUFBWSxHQUFHO0FBQUE7O0FBQ2IsVUFBTSxHQUFJMEIsS0FBSixFQUFXQyxLQUFYLEVBQWtCQyxLQUFsQixjQUE0Qix1QkFBdUJDLElBQXZCLENBQTRCQyxnQkFBNUIsQ0FBNUIsMkNBQW9FLENBQUUsT0FBRixFQUFXLEdBQVgsRUFBZ0IsR0FBaEIsRUFBcUIsR0FBckIsQ0FBMUU7QUFFQSxVQUFNekksT0FBTyxHQUFHLElBQUkwSSx3QkFBSixDQUFvQjtBQUNsQy9ILE1BQUFBLE9BQU8sRUFBRSxLQUFLeEUsTUFBTCxDQUFZaUMsT0FBWixDQUFvQnVDLE9BREs7QUFFbEM4SCxNQUFBQSxPQUFPLEVBQUU7QUFBRUosUUFBQUEsS0FBSyxFQUFFTSxNQUFNLENBQUNOLEtBQUQsQ0FBZjtBQUF3QkMsUUFBQUEsS0FBSyxFQUFFSyxNQUFNLENBQUNMLEtBQUQsQ0FBckM7QUFBOENDLFFBQUFBLEtBQUssRUFBRUksTUFBTSxDQUFDSixLQUFELENBQTNEO0FBQW9FSyxRQUFBQSxRQUFRLEVBQUU7QUFBOUU7QUFGeUIsS0FBcEIsQ0FBaEI7QUFLQSxTQUFLMUwsU0FBTCxDQUFla0csV0FBZixDQUEyQkMsYUFBS3dGLFFBQWhDLEVBQTBDN0ksT0FBTyxDQUFDRixJQUFsRDtBQUNBLFNBQUtoRCxLQUFMLENBQVdrRCxPQUFYLENBQW1CLFlBQVc7QUFDNUIsYUFBT0EsT0FBTyxDQUFDOEksUUFBUixDQUFpQixJQUFqQixDQUFQO0FBQ0QsS0FGRDtBQUdEO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRUMsRUFBQUEsa0JBQWtCLEdBQUc7QUFDbkIsU0FBS3ZMLGFBQUwsR0FBcUJzRixNQUFNLENBQUNFLEtBQVAsQ0FBYSxDQUFiLENBQXJCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFZ0csRUFBQUEsa0JBQWtCLENBQUNsSixJQUFELEVBQWU7QUFDL0IsU0FBS3RDLGFBQUwsR0FBcUJzRixNQUFNLENBQUNtRyxNQUFQLENBQWMsQ0FBQyxLQUFLekwsYUFBTixFQUFxQnNDLElBQXJCLENBQWQsQ0FBckI7QUFDRDtBQUVEO0FBQ0Y7QUFDQTs7O0FBQ0VvSixFQUFBQSxnQkFBZ0IsR0FBRztBQUNqQixVQUFNbEosT0FBTyxHQUFHLElBQUltSixzQkFBSixDQUFrQjtBQUNoQ3ZILE1BQUFBLFVBQVUsRUFBRXdILHNCQUFTLEtBQUtqTixNQUFMLENBQVlpQyxPQUFaLENBQW9Cd0QsVUFBN0IsQ0FEb0I7QUFFaENSLE1BQUFBLFVBQVUsRUFBRSxLQUFLakYsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmdELFVBRkE7QUFHaENpSSxNQUFBQSxhQUFhLEVBQUUsQ0FIaUI7QUFJaENDLE1BQUFBLFNBQVMsRUFBRTFPLE9BQU8sQ0FBQzJPLEdBSmE7QUFLaENDLE1BQUFBLFlBQVksRUFBRSxDQUxrQjtBQU1oQ0MsTUFBQUEsY0FBYyxFQUFFLElBQUlDLElBQUosR0FBV0MsaUJBQVgsRUFOZ0I7QUFPaENDLE1BQUFBLFVBQVUsRUFBRTtBQVBvQixLQUFsQixDQUFoQjtBQVVBLFVBQU07QUFBRTNMLE1BQUFBO0FBQUYsUUFBcUIsS0FBSzlCLE1BQWhDOztBQUNBLFlBQVE4QixjQUFjLENBQUNFLElBQXZCO0FBQ0UsV0FBSyxpQ0FBTDtBQUNFNkIsUUFBQUEsT0FBTyxDQUFDNkosT0FBUixHQUFrQjtBQUNoQjFMLFVBQUFBLElBQUksRUFBRSxNQURVO0FBRWhCMkwsVUFBQUEsSUFBSSxFQUFFLEtBQUsxTixlQUZLO0FBR2hCMk4sVUFBQUEsUUFBUSxFQUFFO0FBSE0sU0FBbEI7QUFLQTs7QUFFRixXQUFLLHFDQUFMO0FBQ0UvSixRQUFBQSxPQUFPLENBQUM2SixPQUFSLEdBQWtCO0FBQ2hCMUwsVUFBQUEsSUFBSSxFQUFFLGVBRFU7QUFFaEIyTCxVQUFBQSxJQUFJLEVBQUUsS0FBSzFOLGVBRks7QUFHaEI0TixVQUFBQSxZQUFZLEVBQUUvTCxjQUFjLENBQUNHLE9BQWYsQ0FBdUJRO0FBSHJCLFNBQWxCO0FBS0E7O0FBRUYsV0FBSywrQkFBTDtBQUNBLFdBQUssZ0NBQUw7QUFDQSxXQUFLLHdDQUFMO0FBQ0EsV0FBSyxpREFBTDtBQUNFb0IsUUFBQUEsT0FBTyxDQUFDNkosT0FBUixHQUFrQjtBQUNoQjFMLFVBQUFBLElBQUksRUFBRSxNQURVO0FBRWhCMkwsVUFBQUEsSUFBSSxFQUFFLEtBQUsxTixlQUZLO0FBR2hCMk4sVUFBQUEsUUFBUSxFQUFFO0FBSE0sU0FBbEI7QUFLQTs7QUFFRixXQUFLLE1BQUw7QUFDRS9KLFFBQUFBLE9BQU8sQ0FBQ2lLLElBQVIsR0FBZSw2QkFBa0I7QUFBRTVMLFVBQUFBLE1BQU0sRUFBRUosY0FBYyxDQUFDRyxPQUFmLENBQXVCQztBQUFqQyxTQUFsQixDQUFmO0FBQ0E7O0FBRUY7QUFDRTJCLFFBQUFBLE9BQU8sQ0FBQzFCLFFBQVIsR0FBbUJMLGNBQWMsQ0FBQ0csT0FBZixDQUF1QkUsUUFBMUM7QUFDQTBCLFFBQUFBLE9BQU8sQ0FBQ3pCLFFBQVIsR0FBbUJOLGNBQWMsQ0FBQ0csT0FBZixDQUF1QkcsUUFBMUM7QUFsQ0o7O0FBcUNBeUIsSUFBQUEsT0FBTyxDQUFDa0ssUUFBUixHQUFtQixLQUFLL04sTUFBTCxDQUFZaUMsT0FBWixDQUFvQjhELGFBQXBCLElBQXFDaUksWUFBR0QsUUFBSCxFQUF4RDtBQUNBbEssSUFBQUEsT0FBTyxDQUFDMEIsVUFBUixHQUFxQixLQUFLekUsV0FBTCxHQUFtQixLQUFLQSxXQUFMLENBQWlCZSxNQUFwQyxHQUE2QyxLQUFLN0IsTUFBTCxDQUFZNkIsTUFBOUU7QUFDQWdDLElBQUFBLE9BQU8sQ0FBQ2pCLE9BQVIsR0FBa0IsS0FBSzVDLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JXLE9BQXBCLElBQStCLFNBQWpEO0FBQ0FpQixJQUFBQSxPQUFPLENBQUNvSyxXQUFSLEdBQXNCQSxhQUF0QjtBQUNBcEssSUFBQUEsT0FBTyxDQUFDZ0IsUUFBUixHQUFtQixLQUFLN0UsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjRDLFFBQXZDO0FBQ0FoQixJQUFBQSxPQUFPLENBQUNMLFFBQVIsR0FBbUIsS0FBS3hELE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J1QixRQUF2QztBQUNBSyxJQUFBQSxPQUFPLENBQUN2QixRQUFSLEdBQW1CcUUsTUFBTSxDQUFDQyxJQUFQLENBQVksQ0FBQyxDQUFELEVBQUksQ0FBSixFQUFPLENBQVAsRUFBVSxDQUFWLEVBQWEsQ0FBYixFQUFnQixDQUFoQixDQUFaLENBQW5CO0FBRUEvQyxJQUFBQSxPQUFPLENBQUNzQixjQUFSLEdBQXlCLEtBQUtuRixNQUFMLENBQVlpQyxPQUFaLENBQW9Ca0QsY0FBN0M7QUFDQXRCLElBQUFBLE9BQU8sQ0FBQ3FLLFdBQVIsR0FBc0IsQ0FBQyxLQUFLbE8sTUFBTCxDQUFZaUMsT0FBWixDQUFvQndDLG1CQUEzQztBQUVBLFNBQUszRCxXQUFMLEdBQW1CaUIsU0FBbkI7QUFDQSxTQUFLaEIsU0FBTCxDQUFla0csV0FBZixDQUEyQkMsYUFBS2lILE1BQWhDLEVBQXdDdEssT0FBTyxDQUFDdUssUUFBUixFQUF4QztBQUVBLFNBQUt6TixLQUFMLENBQVdrRCxPQUFYLENBQW1CLFlBQVc7QUFDNUIsYUFBT0EsT0FBTyxDQUFDOEksUUFBUixDQUFpQixJQUFqQixDQUFQO0FBQ0QsS0FGRDtBQUdEO0FBRUQ7QUFDRjtBQUNBOzs7QUFDRTBCLEVBQUFBLHVCQUF1QixDQUFDNUwsS0FBRCxFQUFnQjtBQUNyQyxVQUFNNkwsY0FBYyxHQUFHM0gsTUFBTSxDQUFDNEgsVUFBUCxDQUFrQjlMLEtBQWxCLEVBQXlCLE1BQXpCLENBQXZCO0FBQ0EsVUFBTWtCLElBQUksR0FBR2dELE1BQU0sQ0FBQ0UsS0FBUCxDQUFhLElBQUl5SCxjQUFqQixDQUFiO0FBQ0EsUUFBSUUsTUFBTSxHQUFHLENBQWI7QUFDQUEsSUFBQUEsTUFBTSxHQUFHN0ssSUFBSSxDQUFDOEssYUFBTCxDQUFtQkgsY0FBYyxHQUFHLENBQXBDLEVBQXVDRSxNQUF2QyxDQUFUO0FBQ0FBLElBQUFBLE1BQU0sR0FBRzdLLElBQUksQ0FBQzhLLGFBQUwsQ0FBbUJILGNBQW5CLEVBQW1DRSxNQUFuQyxDQUFUO0FBQ0E3SyxJQUFBQSxJQUFJLENBQUMrSyxLQUFMLENBQVdqTSxLQUFYLEVBQWtCK0wsTUFBbEIsRUFBMEIsTUFBMUI7QUFDQSxTQUFLek4sU0FBTCxDQUFla0csV0FBZixDQUEyQkMsYUFBS3lILGFBQWhDLEVBQStDaEwsSUFBL0MsRUFQcUMsQ0FRckM7O0FBQ0EsU0FBS21FLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXNkgsK0JBQTdCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFQyxFQUFBQSxjQUFjLEdBQUc7QUFDZixVQUFNaEwsT0FBTyxHQUFHLElBQUlpTCx3QkFBSixDQUFvQixLQUFLQyxhQUFMLEVBQXBCLEVBQTBDLEtBQUtDLDRCQUFMLEVBQTFDLEVBQStFLEtBQUtoUCxNQUFMLENBQVlpQyxPQUEzRixDQUFoQjtBQUVBLFVBQU02RyxPQUFPLEdBQUcsSUFBSW1HLGdCQUFKLENBQVk7QUFBRWpOLE1BQUFBLElBQUksRUFBRWtGLGFBQUtnSTtBQUFiLEtBQVosQ0FBaEI7QUFDQSxTQUFLbk8sU0FBTCxDQUFlb08scUJBQWYsQ0FBcUNULEtBQXJDLENBQTJDNUYsT0FBM0M7O0FBQ0FzRyxxQkFBU3hJLElBQVQsQ0FBYy9DLE9BQWQsRUFBdUJ3TCxJQUF2QixDQUE0QnZHLE9BQTVCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFaUcsRUFBQUEsYUFBYSxHQUFHO0FBQ2QsVUFBTTlNLE9BQU8sR0FBRyxFQUFoQjs7QUFFQSxRQUFJLEtBQUtqQyxNQUFMLENBQVlpQyxPQUFaLENBQW9CNkIsY0FBcEIsS0FBdUMsSUFBM0MsRUFBaUQ7QUFDL0M3QixNQUFBQSxPQUFPLENBQUNxTixJQUFSLENBQWEsbUJBQWI7QUFDRCxLQUZELE1BRU8sSUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjZCLGNBQXBCLEtBQXVDLEtBQTNDLEVBQWtEO0FBQ3ZEN0IsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLG9CQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjhCLHFCQUFwQixLQUE4QyxJQUFsRCxFQUF3RDtBQUN0RDlCLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSwwQkFBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9COEIscUJBQXBCLEtBQThDLEtBQWxELEVBQXlEO0FBQzlEOUIsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLDJCQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQitCLGlCQUFwQixLQUEwQyxJQUE5QyxFQUFvRDtBQUNsRC9CLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSxxQkFBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CK0IsaUJBQXBCLEtBQTBDLEtBQTlDLEVBQXFEO0FBQzFEL0IsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLHNCQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmdDLGtCQUFwQixLQUEyQyxJQUEvQyxFQUFxRDtBQUNuRGhDLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSxzQkFBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CZ0Msa0JBQXBCLEtBQTJDLEtBQS9DLEVBQXNEO0FBQzNEaEMsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLHVCQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmlDLGdCQUFwQixLQUF5QyxJQUE3QyxFQUFtRDtBQUNqRGpDLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSxtQkFBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CaUMsZ0JBQXBCLEtBQXlDLEtBQTdDLEVBQW9EO0FBQ3pEakMsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLG9CQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQmtDLDBCQUFwQixLQUFtRCxJQUF2RCxFQUE2RDtBQUMzRGxDLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSxnQ0FBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9Ca0MsMEJBQXBCLEtBQW1ELEtBQXZELEVBQThEO0FBQ25FbEMsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLGlDQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQm1DLHlCQUFwQixLQUFrRCxJQUF0RCxFQUE0RDtBQUMxRG5DLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSwrQkFBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CbUMseUJBQXBCLEtBQWtELEtBQXRELEVBQTZEO0FBQ2xFbkMsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLGdDQUFiO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQndCLFNBQXBCLEtBQWtDLElBQXRDLEVBQTRDO0FBQzFDeEIsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFjLGlCQUFnQixLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQndCLFNBQVUsRUFBNUQ7QUFDRDs7QUFFRCxRQUFJLEtBQUt6RCxNQUFMLENBQVlpQyxPQUFaLENBQW9CeUIsVUFBcEIsS0FBbUMsSUFBdkMsRUFBNkM7QUFDM0N6QixNQUFBQSxPQUFPLENBQUNxTixJQUFSLENBQWMsa0JBQWlCLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CeUIsVUFBVyxFQUE5RDtBQUNEOztBQUVELFFBQUksS0FBSzFELE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JvQywwQkFBcEIsS0FBbUQsSUFBdkQsRUFBNkQ7QUFDM0RwQyxNQUFBQSxPQUFPLENBQUNxTixJQUFSLENBQWEsOEJBQWI7QUFDRCxLQUZELE1BRU8sSUFBSSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQm9DLDBCQUFwQixLQUFtRCxLQUF2RCxFQUE4RDtBQUNuRXBDLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSwrQkFBYjtBQUNEOztBQUVELFFBQUksS0FBS3RQLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0I0QyxRQUFwQixLQUFpQyxJQUFyQyxFQUEyQztBQUN6QzVDLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYyxnQkFBZSxLQUFLdFAsTUFBTCxDQUFZaUMsT0FBWixDQUFvQjRDLFFBQVMsRUFBMUQ7QUFDRDs7QUFFRCxRQUFJLEtBQUs3RSxNQUFMLENBQVlpQyxPQUFaLENBQW9CcUMsdUJBQXBCLEtBQWdELElBQXBELEVBQTBEO0FBQ3hEckMsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLDJCQUFiO0FBQ0QsS0FGRCxNQUVPLElBQUksS0FBS3RQLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JxQyx1QkFBcEIsS0FBZ0QsS0FBcEQsRUFBMkQ7QUFDaEVyQyxNQUFBQSxPQUFPLENBQUNxTixJQUFSLENBQWEsNEJBQWI7QUFDRDs7QUFFRCxRQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9Cc0Msc0JBQXBCLEtBQStDLElBQW5ELEVBQXlEO0FBQ3ZEdEMsTUFBQUEsT0FBTyxDQUFDcU4sSUFBUixDQUFhLDBCQUFiO0FBQ0QsS0FGRCxNQUVPLElBQUksS0FBS3RQLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JzQyxzQkFBcEIsS0FBK0MsS0FBbkQsRUFBMEQ7QUFDL0R0QyxNQUFBQSxPQUFPLENBQUNxTixJQUFSLENBQWEsMkJBQWI7QUFDRDs7QUFFRCxRQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CeUQsUUFBcEIsS0FBaUMsSUFBckMsRUFBMkM7QUFDekN6RCxNQUFBQSxPQUFPLENBQUNxTixJQUFSLENBQWMsZ0JBQWUsS0FBS3RQLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J5RCxRQUFTLEVBQTFEO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLMUYsTUFBTCxDQUFZaUMsT0FBWixDQUFvQm1CLHdCQUFwQixLQUFpRCxJQUFyRCxFQUEyRDtBQUN6RG5CLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYyxtQ0FBa0MsS0FBS0MscUJBQUwsQ0FBMkIsS0FBS3ZQLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0JtQix3QkFBL0MsQ0FBeUUsRUFBekg7QUFDRDs7QUFFRCxRQUFJLEtBQUtwRCxNQUFMLENBQVlpQyxPQUFaLENBQW9CVSx1QkFBcEIsS0FBZ0QsSUFBcEQsRUFBMEQ7QUFDeERWLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSxtQkFBYjtBQUNELEtBRkQsTUFFTyxJQUFJLEtBQUt0UCxNQUFMLENBQVlpQyxPQUFaLENBQW9CVSx1QkFBcEIsS0FBZ0QsS0FBcEQsRUFBMkQ7QUFDaEVWLE1BQUFBLE9BQU8sQ0FBQ3FOLElBQVIsQ0FBYSxvQkFBYjtBQUNEOztBQUVELFdBQU9yTixPQUFPLENBQUN1TixJQUFSLENBQWEsSUFBYixDQUFQO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFQyxFQUFBQSxtQkFBbUIsR0FBRztBQUNwQixTQUFLeEcsaUJBQUw7QUFDQSxTQUFLZCxJQUFMLENBQVUsU0FBVjtBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFDRXVILEVBQUFBLFlBQVksQ0FBQ3hPLE9BQUQsRUFBbUI7QUFDN0IsU0FBS3lPLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUtnSSxTQUEvQixFQUEwQyxJQUFJSix3QkFBSixDQUFvQjVOLE9BQU8sQ0FBQzBPLGtCQUE1QixFQUFpRCxLQUFLWiw0QkFBTCxFQUFqRCxFQUFzRixLQUFLaFAsTUFBTCxDQUFZaUMsT0FBbEcsQ0FBMUM7QUFDRDtBQUVEO0FBQ0Y7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0U0TixFQUFBQSxPQUFPLENBQUMzTyxPQUFELEVBQW1CO0FBQ3hCLFFBQUk7QUFDRkEsTUFBQUEsT0FBTyxDQUFDNE8sa0JBQVIsQ0FBMkIsS0FBS25PLGlCQUFoQztBQUNELEtBRkQsQ0FFRSxPQUFPc0ksS0FBUCxFQUFtQjtBQUNuQi9JLE1BQUFBLE9BQU8sQ0FBQytJLEtBQVIsR0FBZ0JBLEtBQWhCO0FBRUF4TCxNQUFBQSxPQUFPLENBQUM0SyxRQUFSLENBQWlCLE1BQU07QUFDckIsYUFBSzFJLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZU4sS0FBSyxDQUFDbkIsT0FBckI7QUFDQTVILFFBQUFBLE9BQU8sQ0FBQ3FJLFFBQVIsQ0FBaUJVLEtBQWpCO0FBQ0QsT0FIRDtBQUtBO0FBQ0Q7O0FBRUQsVUFBTThGLFVBQXVCLEdBQUcsRUFBaEM7QUFFQUEsSUFBQUEsVUFBVSxDQUFDVCxJQUFYLENBQWdCO0FBQ2R0TixNQUFBQSxJQUFJLEVBQUVnTyxnQkFBTUMsUUFERTtBQUVkekksTUFBQUEsSUFBSSxFQUFFLFdBRlE7QUFHZGpCLE1BQUFBLEtBQUssRUFBRXJGLE9BQU8sQ0FBQzBPLGtCQUhEO0FBSWRNLE1BQUFBLE1BQU0sRUFBRSxLQUpNO0FBS2RDLE1BQUFBLE1BQU0sRUFBRXBPLFNBTE07QUFNZHFPLE1BQUFBLFNBQVMsRUFBRXJPLFNBTkc7QUFPZHNPLE1BQUFBLEtBQUssRUFBRXRPO0FBUE8sS0FBaEI7O0FBVUEsUUFBSWIsT0FBTyxDQUFDNk8sVUFBUixDQUFtQkksTUFBdkIsRUFBK0I7QUFDN0JKLE1BQUFBLFVBQVUsQ0FBQ1QsSUFBWCxDQUFnQjtBQUNkdE4sUUFBQUEsSUFBSSxFQUFFZ08sZ0JBQU1DLFFBREU7QUFFZHpJLFFBQUFBLElBQUksRUFBRSxRQUZRO0FBR2RqQixRQUFBQSxLQUFLLEVBQUVyRixPQUFPLENBQUNvUCxtQkFBUixDQUE0QnBQLE9BQU8sQ0FBQzZPLFVBQXBDLENBSE87QUFJZEcsUUFBQUEsTUFBTSxFQUFFLEtBSk07QUFLZEMsUUFBQUEsTUFBTSxFQUFFcE8sU0FMTTtBQU1kcU8sUUFBQUEsU0FBUyxFQUFFck8sU0FORztBQU9kc08sUUFBQUEsS0FBSyxFQUFFdE87QUFQTyxPQUFoQjtBQVVBZ08sTUFBQUEsVUFBVSxDQUFDVCxJQUFYLENBQWdCLEdBQUdwTyxPQUFPLENBQUM2TyxVQUEzQjtBQUNEOztBQUVELFNBQUtKLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUtxSixXQUEvQixFQUE0QyxJQUFJQywwQkFBSixDQUFzQixlQUF0QixFQUF1Q1QsVUFBdkMsRUFBbUQsS0FBS2YsNEJBQUwsRUFBbkQsRUFBd0YsS0FBS2hQLE1BQUwsQ0FBWWlDLE9BQXBHLEVBQTZHLEtBQUtOLGlCQUFsSCxDQUE1QztBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFHRThPLEVBQUFBLFdBQVcsQ0FBQ0MsS0FBRCxFQUFnQkMsaUJBQWhCLEVBQXVFcEgsUUFBdkUsRUFBb0c7QUFDN0csUUFBSXRILE9BQUo7O0FBRUEsUUFBSXNILFFBQVEsS0FBS3hILFNBQWpCLEVBQTRCO0FBQzFCd0gsTUFBQUEsUUFBUSxHQUFHb0gsaUJBQVg7QUFDQTFPLE1BQUFBLE9BQU8sR0FBRyxFQUFWO0FBQ0QsS0FIRCxNQUdPO0FBQ0xBLE1BQUFBLE9BQU8sR0FBRzBPLGlCQUFWO0FBQ0Q7O0FBRUQsUUFBSSxPQUFPMU8sT0FBUCxLQUFtQixRQUF2QixFQUFpQztBQUMvQixZQUFNLElBQUlMLFNBQUosQ0FBYyxzQ0FBZCxDQUFOO0FBQ0Q7O0FBQ0QsV0FBTyxJQUFJZ1AsaUJBQUosQ0FBYUYsS0FBYixFQUFvQixLQUFLL08saUJBQXpCLEVBQTRDLEtBQUszQixNQUFMLENBQVlpQyxPQUF4RCxFQUFpRUEsT0FBakUsRUFBMEVzSCxRQUExRSxDQUFQO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7OztBQUdFc0gsRUFBQUEsWUFBWSxDQUFDQyxRQUFELEVBQXFCQyxJQUFyQixFQUFvSjtBQUM5SkQsSUFBQUEsUUFBUSxDQUFDRSxnQkFBVCxHQUE0QixJQUE1Qjs7QUFFQSxRQUFJRCxJQUFKLEVBQVU7QUFDUixVQUFJRCxRQUFRLENBQUNHLGFBQWIsRUFBNEI7QUFDMUIsY0FBTSxJQUFJaEwsS0FBSixDQUFVLHlGQUFWLENBQU47QUFDRDs7QUFFRCxVQUFJNkssUUFBUSxDQUFDSSxlQUFiLEVBQThCO0FBQzVCLGNBQU0sSUFBSWpMLEtBQUosQ0FBVSw4RkFBVixDQUFOO0FBQ0Q7O0FBRUQsWUFBTWtMLFNBQVMsR0FBRy9CLGlCQUFTeEksSUFBVCxDQUFjbUssSUFBZCxDQUFsQixDQVRRLENBV1I7QUFDQTs7O0FBQ0FJLE1BQUFBLFNBQVMsQ0FBQ25KLEVBQVYsQ0FBYSxPQUFiLEVBQXVCTixHQUFELElBQVM7QUFDN0JvSixRQUFBQSxRQUFRLENBQUNNLG9CQUFULENBQThCMUcsT0FBOUIsQ0FBc0NoRCxHQUF0QztBQUNELE9BRkQsRUFiUSxDQWlCUjtBQUNBOztBQUNBb0osTUFBQUEsUUFBUSxDQUFDTSxvQkFBVCxDQUE4QnBKLEVBQTlCLENBQWlDLE9BQWpDLEVBQTJDTixHQUFELElBQVM7QUFDakR5SixRQUFBQSxTQUFTLENBQUN6RyxPQUFWLENBQWtCaEQsR0FBbEI7QUFDRCxPQUZEO0FBSUF5SixNQUFBQSxTQUFTLENBQUM5QixJQUFWLENBQWV5QixRQUFRLENBQUNNLG9CQUF4QjtBQUNELEtBeEJELE1Bd0JPLElBQUksQ0FBQ04sUUFBUSxDQUFDRyxhQUFkLEVBQTZCO0FBQ2xDO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQUgsTUFBQUEsUUFBUSxDQUFDTSxvQkFBVCxDQUE4QkMsR0FBOUI7QUFDRDs7QUFFRCxVQUFNQyxRQUFRLEdBQUcsTUFBTTtBQUNyQnBRLE1BQUFBLE9BQU8sQ0FBQ2tLLE1BQVI7QUFDRCxLQUZEOztBQUlBLFVBQU12SCxPQUFPLEdBQUcsSUFBSTBOLGdDQUFKLENBQW9CVCxRQUFwQixDQUFoQjtBQUVBLFVBQU01UCxPQUFPLEdBQUcsSUFBSXNRLGdCQUFKLENBQVlWLFFBQVEsQ0FBQ1csZ0JBQVQsRUFBWixFQUEwQ3hILEtBQUQsSUFBMkQ7QUFDbEg2RyxNQUFBQSxRQUFRLENBQUNuSixjQUFULENBQXdCLFFBQXhCLEVBQWtDMkosUUFBbEM7O0FBRUEsVUFBSXJILEtBQUosRUFBVztBQUNULFlBQUlBLEtBQUssQ0FBQzhCLElBQU4sS0FBZSxTQUFuQixFQUE4QjtBQUM1QjlCLFVBQUFBLEtBQUssQ0FBQ25CLE9BQU4sSUFBaUIsOEhBQWpCO0FBQ0Q7O0FBQ0RnSSxRQUFBQSxRQUFRLENBQUM3RyxLQUFULEdBQWlCQSxLQUFqQjtBQUNBNkcsUUFBQUEsUUFBUSxDQUFDdkgsUUFBVCxDQUFrQlUsS0FBbEI7QUFDQTtBQUNEOztBQUVELFdBQUswRixXQUFMLENBQWlCbUIsUUFBakIsRUFBMkI1SixhQUFLd0ssU0FBaEMsRUFBMkM3TixPQUEzQztBQUNELEtBYmUsQ0FBaEI7QUFlQWlOLElBQUFBLFFBQVEsQ0FBQ2pKLElBQVQsQ0FBYyxRQUFkLEVBQXdCeUosUUFBeEI7QUFFQSxTQUFLNUIsWUFBTCxDQUFrQnhPLE9BQWxCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7OztBQUNFeVEsRUFBQUEsT0FBTyxDQUFDelEsT0FBRCxFQUFtQjtBQUN4QixVQUFNNk8sVUFBdUIsR0FBRyxFQUFoQztBQUVBQSxJQUFBQSxVQUFVLENBQUNULElBQVgsQ0FBZ0I7QUFDZHROLE1BQUFBLElBQUksRUFBRWdPLGdCQUFNNEIsR0FERTtBQUVkcEssTUFBQUEsSUFBSSxFQUFFLFFBRlE7QUFHZGpCLE1BQUFBLEtBQUssRUFBRXhFLFNBSE87QUFJZG1PLE1BQUFBLE1BQU0sRUFBRSxJQUpNO0FBS2RDLE1BQUFBLE1BQU0sRUFBRXBPLFNBTE07QUFNZHFPLE1BQUFBLFNBQVMsRUFBRXJPLFNBTkc7QUFPZHNPLE1BQUFBLEtBQUssRUFBRXRPO0FBUE8sS0FBaEI7QUFVQWdPLElBQUFBLFVBQVUsQ0FBQ1QsSUFBWCxDQUFnQjtBQUNkdE4sTUFBQUEsSUFBSSxFQUFFZ08sZ0JBQU1DLFFBREU7QUFFZHpJLE1BQUFBLElBQUksRUFBRSxRQUZRO0FBR2RqQixNQUFBQSxLQUFLLEVBQUVyRixPQUFPLENBQUM2TyxVQUFSLENBQW1CSSxNQUFuQixHQUE0QmpQLE9BQU8sQ0FBQ29QLG1CQUFSLENBQTRCcFAsT0FBTyxDQUFDNk8sVUFBcEMsQ0FBNUIsR0FBOEUsSUFIdkU7QUFJZEcsTUFBQUEsTUFBTSxFQUFFLEtBSk07QUFLZEMsTUFBQUEsTUFBTSxFQUFFcE8sU0FMTTtBQU1kcU8sTUFBQUEsU0FBUyxFQUFFck8sU0FORztBQU9kc08sTUFBQUEsS0FBSyxFQUFFdE87QUFQTyxLQUFoQjtBQVVBZ08sSUFBQUEsVUFBVSxDQUFDVCxJQUFYLENBQWdCO0FBQ2R0TixNQUFBQSxJQUFJLEVBQUVnTyxnQkFBTUMsUUFERTtBQUVkekksTUFBQUEsSUFBSSxFQUFFLE1BRlE7QUFHZGpCLE1BQUFBLEtBQUssRUFBRXJGLE9BQU8sQ0FBQzBPLGtCQUhEO0FBSWRNLE1BQUFBLE1BQU0sRUFBRSxLQUpNO0FBS2RDLE1BQUFBLE1BQU0sRUFBRXBPLFNBTE07QUFNZHFPLE1BQUFBLFNBQVMsRUFBRXJPLFNBTkc7QUFPZHNPLE1BQUFBLEtBQUssRUFBRXRPO0FBUE8sS0FBaEI7QUFVQWIsSUFBQUEsT0FBTyxDQUFDMlEsU0FBUixHQUFvQixJQUFwQixDQWpDd0IsQ0FrQ3hCOztBQUNBM1EsSUFBQUEsT0FBTyxDQUFDOEcsRUFBUixDQUFXLGFBQVgsRUFBMEIsQ0FBQ1IsSUFBRCxFQUFlakIsS0FBZixLQUE4QjtBQUN0RCxVQUFJaUIsSUFBSSxLQUFLLFFBQWIsRUFBdUI7QUFDckJ0RyxRQUFBQSxPQUFPLENBQUM0USxNQUFSLEdBQWlCdkwsS0FBakI7QUFDRCxPQUZELE1BRU87QUFDTHJGLFFBQUFBLE9BQU8sQ0FBQytJLEtBQVIsR0FBZ0IsSUFBSVgsb0JBQUosQ0FBa0IseUNBQXdDOUIsSUFBSyxrQkFBL0QsQ0FBaEI7QUFDRDtBQUNGLEtBTkQ7QUFRQSxTQUFLbUksV0FBTCxDQUFpQnpPLE9BQWpCLEVBQTBCZ0csYUFBS3FKLFdBQS9CLEVBQTRDLElBQUlDLDBCQUFKLENBQXNCLFlBQXRCLEVBQW9DVCxVQUFwQyxFQUFnRCxLQUFLZiw0QkFBTCxFQUFoRCxFQUFxRixLQUFLaFAsTUFBTCxDQUFZaUMsT0FBakcsRUFBMEcsS0FBS04saUJBQS9HLENBQTVDO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0VvUSxFQUFBQSxTQUFTLENBQUM3USxPQUFELEVBQW1CO0FBQzFCLFVBQU02TyxVQUF1QixHQUFHLEVBQWhDO0FBRUFBLElBQUFBLFVBQVUsQ0FBQ1QsSUFBWCxDQUFnQjtBQUNkdE4sTUFBQUEsSUFBSSxFQUFFZ08sZ0JBQU00QixHQURFO0FBRWRwSyxNQUFBQSxJQUFJLEVBQUUsUUFGUTtBQUdkO0FBQ0FqQixNQUFBQSxLQUFLLEVBQUVyRixPQUFPLENBQUM0USxNQUpEO0FBS2Q1QixNQUFBQSxNQUFNLEVBQUUsS0FMTTtBQU1kQyxNQUFBQSxNQUFNLEVBQUVwTyxTQU5NO0FBT2RxTyxNQUFBQSxTQUFTLEVBQUVyTyxTQVBHO0FBUWRzTyxNQUFBQSxLQUFLLEVBQUV0TztBQVJPLEtBQWhCO0FBV0EsU0FBSzROLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUtxSixXQUEvQixFQUE0QyxJQUFJQywwQkFBSixDQUFzQixjQUF0QixFQUFzQ1QsVUFBdEMsRUFBa0QsS0FBS2YsNEJBQUwsRUFBbEQsRUFBdUYsS0FBS2hQLE1BQUwsQ0FBWWlDLE9BQW5HLEVBQTRHLEtBQUtOLGlCQUFqSCxDQUE1QztBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFDRW9JLEVBQUFBLE9BQU8sQ0FBQzdJLE9BQUQsRUFBbUI2TyxVQUFuQixFQUE0RDtBQUNqRSxVQUFNaUMsaUJBQThCLEdBQUcsRUFBdkM7QUFFQUEsSUFBQUEsaUJBQWlCLENBQUMxQyxJQUFsQixDQUF1QjtBQUNyQnROLE1BQUFBLElBQUksRUFBRWdPLGdCQUFNNEIsR0FEUztBQUVyQnBLLE1BQUFBLElBQUksRUFBRSxRQUZlO0FBR3JCO0FBQ0FqQixNQUFBQSxLQUFLLEVBQUVyRixPQUFPLENBQUM0USxNQUpNO0FBS3JCNUIsTUFBQUEsTUFBTSxFQUFFLEtBTGE7QUFNckJDLE1BQUFBLE1BQU0sRUFBRXBPLFNBTmE7QUFPckJxTyxNQUFBQSxTQUFTLEVBQUVyTyxTQVBVO0FBUXJCc08sTUFBQUEsS0FBSyxFQUFFdE87QUFSYyxLQUF2Qjs7QUFXQSxRQUFJO0FBQ0YsV0FBSyxJQUFJa1EsQ0FBQyxHQUFHLENBQVIsRUFBV0MsR0FBRyxHQUFHaFIsT0FBTyxDQUFDNk8sVUFBUixDQUFtQkksTUFBekMsRUFBaUQ4QixDQUFDLEdBQUdDLEdBQXJELEVBQTBERCxDQUFDLEVBQTNELEVBQStEO0FBQzdELGNBQU1FLFNBQVMsR0FBR2pSLE9BQU8sQ0FBQzZPLFVBQVIsQ0FBbUJrQyxDQUFuQixDQUFsQjtBQUVBRCxRQUFBQSxpQkFBaUIsQ0FBQzFDLElBQWxCLENBQXVCLEVBQ3JCLEdBQUc2QyxTQURrQjtBQUVyQjVMLFVBQUFBLEtBQUssRUFBRTRMLFNBQVMsQ0FBQ25RLElBQVYsQ0FBZW9RLFFBQWYsQ0FBd0JyQyxVQUFVLEdBQUdBLFVBQVUsQ0FBQ29DLFNBQVMsQ0FBQzNLLElBQVgsQ0FBYixHQUFnQyxJQUFsRSxFQUF3RSxLQUFLN0YsaUJBQTdFO0FBRmMsU0FBdkI7QUFJRDtBQUNGLEtBVEQsQ0FTRSxPQUFPc0ksS0FBUCxFQUFtQjtBQUNuQi9JLE1BQUFBLE9BQU8sQ0FBQytJLEtBQVIsR0FBZ0JBLEtBQWhCO0FBRUF4TCxNQUFBQSxPQUFPLENBQUM0SyxRQUFSLENBQWlCLE1BQU07QUFDckIsYUFBSzFJLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZU4sS0FBSyxDQUFDbkIsT0FBckI7QUFDQTVILFFBQUFBLE9BQU8sQ0FBQ3FJLFFBQVIsQ0FBaUJVLEtBQWpCO0FBQ0QsT0FIRDtBQUtBO0FBQ0Q7O0FBRUQsU0FBSzBGLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUtxSixXQUEvQixFQUE0QyxJQUFJQywwQkFBSixDQUFzQixZQUF0QixFQUFvQ3dCLGlCQUFwQyxFQUF1RCxLQUFLaEQsNEJBQUwsRUFBdkQsRUFBNEYsS0FBS2hQLE1BQUwsQ0FBWWlDLE9BQXhHLEVBQWlILEtBQUtOLGlCQUF0SCxDQUE1QztBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0UwUSxFQUFBQSxhQUFhLENBQUNuUixPQUFELEVBQW1CO0FBQzlCLFFBQUk7QUFDRkEsTUFBQUEsT0FBTyxDQUFDNE8sa0JBQVIsQ0FBMkIsS0FBS25PLGlCQUFoQztBQUNELEtBRkQsQ0FFRSxPQUFPc0ksS0FBUCxFQUFtQjtBQUNuQi9JLE1BQUFBLE9BQU8sQ0FBQytJLEtBQVIsR0FBZ0JBLEtBQWhCO0FBRUF4TCxNQUFBQSxPQUFPLENBQUM0SyxRQUFSLENBQWlCLE1BQU07QUFDckIsYUFBSzFJLEtBQUwsQ0FBVzRKLEdBQVgsQ0FBZU4sS0FBSyxDQUFDbkIsT0FBckI7QUFDQTVILFFBQUFBLE9BQU8sQ0FBQ3FJLFFBQVIsQ0FBaUJVLEtBQWpCO0FBQ0QsT0FIRDtBQUtBO0FBQ0Q7O0FBRUQsU0FBSzBGLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUtxSixXQUEvQixFQUE0QyxJQUFJQywwQkFBSixDQUFzQnRQLE9BQU8sQ0FBQzBPLGtCQUE5QixFQUFtRDFPLE9BQU8sQ0FBQzZPLFVBQTNELEVBQXVFLEtBQUtmLDRCQUFMLEVBQXZFLEVBQTRHLEtBQUtoUCxNQUFMLENBQVlpQyxPQUF4SCxFQUFpSSxLQUFLTixpQkFBdEksQ0FBNUM7QUFDRDtBQUVEO0FBQ0Y7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0UyUSxFQUFBQSxnQkFBZ0IsQ0FBQy9JLFFBQUQsRUFBcUMvQixJQUFJLEdBQUcsRUFBNUMsRUFBZ0Q1QyxjQUFjLEdBQUcsS0FBSzVFLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0IyQyxjQUFyRixFQUFxRztBQUNuSCxnREFBMEJBLGNBQTFCLEVBQTBDLGdCQUExQztBQUVBLFVBQU0yTixXQUFXLEdBQUcsSUFBSUMsd0JBQUosQ0FBZ0JoTCxJQUFoQixFQUFzQjVDLGNBQXRCLENBQXBCOztBQUVBLFFBQUksS0FBSzVFLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQyxLQUFyQyxFQUE0QztBQUMxQyxhQUFPLEtBQUtpSyxZQUFMLENBQWtCLElBQUk4QixnQkFBSixDQUFZLHFDQUFzQ2UsV0FBVyxDQUFDRSxvQkFBWixFQUF0QyxHQUE0RSxjQUE1RSxHQUE2RkYsV0FBVyxDQUFDL0ssSUFBckgsRUFBNEhFLEdBQUQsSUFBUztBQUMzSixhQUFLckgsZ0JBQUw7O0FBQ0EsWUFBSSxLQUFLQSxnQkFBTCxLQUEwQixDQUE5QixFQUFpQztBQUMvQixlQUFLRixhQUFMLEdBQXFCLElBQXJCO0FBQ0Q7O0FBQ0RvSixRQUFBQSxRQUFRLENBQUM3QixHQUFELENBQVI7QUFDRCxPQU53QixDQUFsQixDQUFQO0FBT0Q7O0FBRUQsVUFBTXhHLE9BQU8sR0FBRyxJQUFJc1EsZ0JBQUosQ0FBWXpQLFNBQVosRUFBd0IyRixHQUFELElBQVM7QUFDOUMsYUFBTzZCLFFBQVEsQ0FBQzdCLEdBQUQsRUFBTSxLQUFLc0gsNEJBQUwsRUFBTixDQUFmO0FBQ0QsS0FGZSxDQUFoQjtBQUdBLFdBQU8sS0FBS1csV0FBTCxDQUFpQnpPLE9BQWpCLEVBQTBCZ0csYUFBS3dMLG1CQUEvQixFQUFvREgsV0FBVyxDQUFDSSxZQUFaLENBQXlCLEtBQUszRCw0QkFBTCxFQUF6QixDQUFwRCxDQUFQO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0U0RCxFQUFBQSxpQkFBaUIsQ0FBQ3JKLFFBQUQsRUFBc0MvQixJQUFJLEdBQUcsRUFBN0MsRUFBaUQ7QUFDaEUsVUFBTStLLFdBQVcsR0FBRyxJQUFJQyx3QkFBSixDQUFnQmhMLElBQWhCLENBQXBCOztBQUNBLFFBQUksS0FBS3hILE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQyxLQUFyQyxFQUE0QztBQUMxQyxhQUFPLEtBQUtpSyxZQUFMLENBQWtCLElBQUk4QixnQkFBSixDQUFZLGlCQUFpQmUsV0FBVyxDQUFDL0ssSUFBekMsRUFBZ0RFLEdBQUQsSUFBUztBQUMvRSxhQUFLckgsZ0JBQUw7O0FBQ0EsWUFBSSxLQUFLQSxnQkFBTCxLQUEwQixDQUE5QixFQUFpQztBQUMvQixlQUFLRixhQUFMLEdBQXFCLEtBQXJCO0FBQ0Q7O0FBRURvSixRQUFBQSxRQUFRLENBQUM3QixHQUFELENBQVI7QUFDRCxPQVB3QixDQUFsQixDQUFQO0FBUUQ7O0FBQ0QsVUFBTXhHLE9BQU8sR0FBRyxJQUFJc1EsZ0JBQUosQ0FBWXpQLFNBQVosRUFBdUJ3SCxRQUF2QixDQUFoQjtBQUNBLFdBQU8sS0FBS29HLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUt3TCxtQkFBL0IsRUFBb0RILFdBQVcsQ0FBQ00sYUFBWixDQUEwQixLQUFLN0QsNEJBQUwsRUFBMUIsQ0FBcEQsQ0FBUDtBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0U4RCxFQUFBQSxtQkFBbUIsQ0FBQ3ZKLFFBQUQsRUFBd0MvQixJQUFJLEdBQUcsRUFBL0MsRUFBbUQ7QUFDcEUsVUFBTStLLFdBQVcsR0FBRyxJQUFJQyx3QkFBSixDQUFnQmhMLElBQWhCLENBQXBCOztBQUNBLFFBQUksS0FBS3hILE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQyxLQUFyQyxFQUE0QztBQUMxQyxhQUFPLEtBQUtpSyxZQUFMLENBQWtCLElBQUk4QixnQkFBSixDQUFZLG1CQUFtQmUsV0FBVyxDQUFDL0ssSUFBM0MsRUFBa0RFLEdBQUQsSUFBUztBQUNqRixhQUFLckgsZ0JBQUw7O0FBQ0EsWUFBSSxLQUFLQSxnQkFBTCxLQUEwQixDQUE5QixFQUFpQztBQUMvQixlQUFLRixhQUFMLEdBQXFCLEtBQXJCO0FBQ0Q7O0FBQ0RvSixRQUFBQSxRQUFRLENBQUM3QixHQUFELENBQVI7QUFDRCxPQU53QixDQUFsQixDQUFQO0FBT0Q7O0FBQ0QsVUFBTXhHLE9BQU8sR0FBRyxJQUFJc1EsZ0JBQUosQ0FBWXpQLFNBQVosRUFBdUJ3SCxRQUF2QixDQUFoQjtBQUNBLFdBQU8sS0FBS29HLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUt3TCxtQkFBL0IsRUFBb0RILFdBQVcsQ0FBQ1EsZUFBWixDQUE0QixLQUFLL0QsNEJBQUwsRUFBNUIsQ0FBcEQsQ0FBUDtBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0VnRSxFQUFBQSxlQUFlLENBQUN6SixRQUFELEVBQW9DL0IsSUFBcEMsRUFBa0Q7QUFDL0QsVUFBTStLLFdBQVcsR0FBRyxJQUFJQyx3QkFBSixDQUFnQmhMLElBQWhCLENBQXBCOztBQUNBLFFBQUksS0FBS3hILE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQyxLQUFyQyxFQUE0QztBQUMxQyxhQUFPLEtBQUtpSyxZQUFMLENBQWtCLElBQUk4QixnQkFBSixDQUFZLGVBQWVlLFdBQVcsQ0FBQy9LLElBQXZDLEVBQThDRSxHQUFELElBQVM7QUFDN0UsYUFBS3JILGdCQUFMO0FBQ0FrSixRQUFBQSxRQUFRLENBQUM3QixHQUFELENBQVI7QUFDRCxPQUh3QixDQUFsQixDQUFQO0FBSUQ7O0FBQ0QsVUFBTXhHLE9BQU8sR0FBRyxJQUFJc1EsZ0JBQUosQ0FBWXpQLFNBQVosRUFBdUJ3SCxRQUF2QixDQUFoQjtBQUNBLFdBQU8sS0FBS29HLFdBQUwsQ0FBaUJ6TyxPQUFqQixFQUEwQmdHLGFBQUt3TCxtQkFBL0IsRUFBb0RILFdBQVcsQ0FBQ1UsV0FBWixDQUF3QixLQUFLakUsNEJBQUwsRUFBeEIsQ0FBcEQsQ0FBUDtBQUNEO0FBRUQ7QUFDRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFDRXVELEVBQUFBLFdBQVcsQ0FBQ1csRUFBRCxFQUE0S3RPLGNBQTVLLEVBQW1QO0FBQzVQLFFBQUksT0FBT3NPLEVBQVAsS0FBYyxVQUFsQixFQUE4QjtBQUM1QixZQUFNLElBQUl0UixTQUFKLENBQWMseUJBQWQsQ0FBTjtBQUNEOztBQUVELFVBQU11UixZQUFZLEdBQUcsS0FBS2hULGFBQTFCOztBQUNBLFVBQU1xSCxJQUFJLEdBQUcsY0FBZTRMLGdCQUFPQyxXQUFQLENBQW1CLEVBQW5CLEVBQXVCMUcsUUFBdkIsQ0FBZ0MsS0FBaEMsQ0FBNUI7O0FBQ0EsVUFBTTJHLE1BQTJILEdBQUcsQ0FBQzVMLEdBQUQsRUFBTTZMLElBQU4sRUFBWSxHQUFHbkwsSUFBZixLQUF3QjtBQUMxSixVQUFJVixHQUFKLEVBQVM7QUFDUCxZQUFJLEtBQUt2SCxhQUFMLElBQXNCLEtBQUthLEtBQUwsS0FBZSxLQUFLK0YsS0FBTCxDQUFXeU0sU0FBcEQsRUFBK0Q7QUFDN0QsZUFBS1YsbUJBQUwsQ0FBMEJXLEtBQUQsSUFBVztBQUNsQ0YsWUFBQUEsSUFBSSxDQUFDRSxLQUFLLElBQUkvTCxHQUFWLEVBQWUsR0FBR1UsSUFBbEIsQ0FBSjtBQUNELFdBRkQsRUFFR1osSUFGSDtBQUdELFNBSkQsTUFJTztBQUNMK0wsVUFBQUEsSUFBSSxDQUFDN0wsR0FBRCxFQUFNLEdBQUdVLElBQVQsQ0FBSjtBQUNEO0FBQ0YsT0FSRCxNQVFPLElBQUkrSyxZQUFKLEVBQWtCO0FBQ3ZCLFlBQUksS0FBS25ULE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQyxLQUFyQyxFQUE0QztBQUMxQyxlQUFLcEYsZ0JBQUw7QUFDRDs7QUFDRGtULFFBQUFBLElBQUksQ0FBQyxJQUFELEVBQU8sR0FBR25MLElBQVYsQ0FBSjtBQUNELE9BTE0sTUFLQTtBQUNMLGFBQUt3SyxpQkFBTCxDQUF3QmEsS0FBRCxJQUFXO0FBQ2hDRixVQUFBQSxJQUFJLENBQUNFLEtBQUQsRUFBUSxHQUFHckwsSUFBWCxDQUFKO0FBQ0QsU0FGRCxFQUVHWixJQUZIO0FBR0Q7QUFDRixLQW5CRDs7QUFxQkEsUUFBSTJMLFlBQUosRUFBa0I7QUFDaEIsYUFBTyxLQUFLSCxlQUFMLENBQXNCdEwsR0FBRCxJQUFTO0FBQ25DLFlBQUlBLEdBQUosRUFBUztBQUNQLGlCQUFPd0wsRUFBRSxDQUFDeEwsR0FBRCxDQUFUO0FBQ0Q7O0FBRUQsWUFBSTlDLGNBQUosRUFBb0I7QUFDbEIsaUJBQU8sS0FBSzhLLFlBQUwsQ0FBa0IsSUFBSThCLGdCQUFKLENBQVkscUNBQXFDLEtBQUtqQyxxQkFBTCxDQUEyQjNLLGNBQTNCLENBQWpELEVBQThGOEMsR0FBRCxJQUFTO0FBQzdILG1CQUFPd0wsRUFBRSxDQUFDeEwsR0FBRCxFQUFNNEwsTUFBTixDQUFUO0FBQ0QsV0FGd0IsQ0FBbEIsQ0FBUDtBQUdELFNBSkQsTUFJTztBQUNMLGlCQUFPSixFQUFFLENBQUMsSUFBRCxFQUFPSSxNQUFQLENBQVQ7QUFDRDtBQUNGLE9BWk0sRUFZSjlMLElBWkksQ0FBUDtBQWFELEtBZEQsTUFjTztBQUNMLGFBQU8sS0FBSzhLLGdCQUFMLENBQXVCNUssR0FBRCxJQUFTO0FBQ3BDLFlBQUlBLEdBQUosRUFBUztBQUNQLGlCQUFPd0wsRUFBRSxDQUFDeEwsR0FBRCxDQUFUO0FBQ0Q7O0FBRUQsZUFBT3dMLEVBQUUsQ0FBQyxJQUFELEVBQU9JLE1BQVAsQ0FBVDtBQUNELE9BTk0sRUFNSjlMLElBTkksRUFNRTVDLGNBTkYsQ0FBUDtBQU9EO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFK0ssRUFBQUEsV0FBVyxDQUFDek8sT0FBRCxFQUE4QndTLFVBQTlCLEVBQWtEN1AsT0FBbEQsRUFBbUo7QUFDNUosUUFBSSxLQUFLN0MsS0FBTCxLQUFlLEtBQUsrRixLQUFMLENBQVd5TSxTQUE5QixFQUF5QztBQUN2QyxZQUFNMUssT0FBTyxHQUFHLHNDQUFzQyxLQUFLL0IsS0FBTCxDQUFXeU0sU0FBWCxDQUFxQmhNLElBQTNELEdBQWtFLGtCQUFsRSxHQUF1RixLQUFLeEcsS0FBTCxDQUFXd0csSUFBbEcsR0FBeUcsUUFBekg7QUFDQSxXQUFLN0csS0FBTCxDQUFXNEosR0FBWCxDQUFlekIsT0FBZjtBQUNBNUgsTUFBQUEsT0FBTyxDQUFDcUksUUFBUixDQUFpQixJQUFJRCxvQkFBSixDQUFpQlIsT0FBakIsRUFBMEIsZUFBMUIsQ0FBakI7QUFDRCxLQUpELE1BSU8sSUFBSTVILE9BQU8sQ0FBQ3lTLFFBQVosRUFBc0I7QUFDM0JsVixNQUFBQSxPQUFPLENBQUM0SyxRQUFSLENBQWlCLE1BQU07QUFDckJuSSxRQUFBQSxPQUFPLENBQUNxSSxRQUFSLENBQWlCLElBQUlELG9CQUFKLENBQWlCLFdBQWpCLEVBQThCLFNBQTlCLENBQWpCO0FBQ0QsT0FGRDtBQUdELEtBSk0sTUFJQTtBQUNMLFVBQUlvSyxVQUFVLEtBQUt4TSxhQUFLZ0ksU0FBeEIsRUFBbUM7QUFDakMsYUFBSzVPLFVBQUwsR0FBa0IsSUFBbEI7QUFDRCxPQUZELE1BRU87QUFDTCxhQUFLQSxVQUFMLEdBQWtCLEtBQWxCO0FBQ0Q7O0FBRUQsV0FBS1ksT0FBTCxHQUFlQSxPQUFmO0FBQ0FBLE1BQUFBLE9BQU8sQ0FBQzBTLFVBQVIsR0FBc0IsSUFBdEI7QUFDQTFTLE1BQUFBLE9BQU8sQ0FBQzJTLFFBQVIsR0FBb0IsQ0FBcEI7QUFDQTNTLE1BQUFBLE9BQU8sQ0FBQzZQLElBQVIsR0FBZ0IsRUFBaEI7QUFDQTdQLE1BQUFBLE9BQU8sQ0FBQzRTLEdBQVIsR0FBZSxFQUFmOztBQUVBLFlBQU14QyxRQUFRLEdBQUcsTUFBTTtBQUNyQnlDLFFBQUFBLGFBQWEsQ0FBQ0MsTUFBZCxDQUFxQmxMLE9BQXJCO0FBQ0FpTCxRQUFBQSxhQUFhLENBQUNySixPQUFkLENBQXNCLElBQUlwQixvQkFBSixDQUFpQixXQUFqQixFQUE4QixTQUE5QixDQUF0QixFQUZxQixDQUlyQjs7QUFDQVIsUUFBQUEsT0FBTyxDQUFDbUwsTUFBUixHQUFpQixJQUFqQjtBQUNBbkwsUUFBQUEsT0FBTyxDQUFDdUksR0FBUjs7QUFFQSxZQUFJblEsT0FBTyxZQUFZc1EsZ0JBQW5CLElBQThCdFEsT0FBTyxDQUFDZ1QsTUFBMUMsRUFBa0Q7QUFDaEQ7QUFDQWhULFVBQUFBLE9BQU8sQ0FBQ2lULE1BQVI7QUFDRDtBQUNGLE9BWkQ7O0FBY0FqVCxNQUFBQSxPQUFPLENBQUMyRyxJQUFSLENBQWEsUUFBYixFQUF1QnlKLFFBQXZCO0FBRUEsV0FBS3RHLGtCQUFMO0FBRUEsWUFBTWxDLE9BQU8sR0FBRyxJQUFJbUcsZ0JBQUosQ0FBWTtBQUFFak4sUUFBQUEsSUFBSSxFQUFFMFIsVUFBUjtBQUFvQlUsUUFBQUEsZUFBZSxFQUFFLEtBQUtuVDtBQUExQyxPQUFaLENBQWhCO0FBQ0EsV0FBS0YsU0FBTCxDQUFlb08scUJBQWYsQ0FBcUNULEtBQXJDLENBQTJDNUYsT0FBM0M7QUFDQSxXQUFLaEIsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVdzTixtQkFBN0I7QUFFQXZMLE1BQUFBLE9BQU8sQ0FBQ2pCLElBQVIsQ0FBYSxRQUFiLEVBQXVCLE1BQU07QUFDM0IzRyxRQUFBQSxPQUFPLENBQUN5RyxjQUFSLENBQXVCLFFBQXZCLEVBQWlDMkosUUFBakM7QUFDQXBRLFFBQUFBLE9BQU8sQ0FBQzJHLElBQVIsQ0FBYSxRQUFiLEVBQXVCLEtBQUtuRyx1QkFBNUI7QUFFQSxhQUFLVCw0QkFBTCxHQUFvQyxLQUFwQztBQUNBLGFBQUtOLEtBQUwsQ0FBV2tELE9BQVgsQ0FBbUIsWUFBVztBQUM1QixpQkFBT0EsT0FBTyxDQUFFOEksUUFBVCxDQUFrQixJQUFsQixDQUFQO0FBQ0QsU0FGRDtBQUdELE9BUkQ7O0FBVUEsWUFBTW9ILGFBQWEsR0FBRzNFLGlCQUFTeEksSUFBVCxDQUFjL0MsT0FBZCxDQUF0Qjs7QUFDQWtRLE1BQUFBLGFBQWEsQ0FBQ2xNLElBQWQsQ0FBbUIsT0FBbkIsRUFBNkJvQyxLQUFELElBQVc7QUFBQTs7QUFDckM4SixRQUFBQSxhQUFhLENBQUNDLE1BQWQsQ0FBcUJsTCxPQUFyQixFQURxQyxDQUdyQzs7QUFDQSwwQkFBQTVILE9BQU8sQ0FBQytJLEtBQVIsMkRBQUEvSSxPQUFPLENBQUMrSSxLQUFSLEdBQWtCQSxLQUFsQjtBQUVBbkIsUUFBQUEsT0FBTyxDQUFDbUwsTUFBUixHQUFpQixJQUFqQjtBQUNBbkwsUUFBQUEsT0FBTyxDQUFDdUksR0FBUjtBQUNELE9BUkQ7QUFTQTBDLE1BQUFBLGFBQWEsQ0FBQzFFLElBQWQsQ0FBbUJ2RyxPQUFuQjtBQUNEO0FBQ0Y7QUFFRDtBQUNGO0FBQ0E7OztBQUNFc0MsRUFBQUEsTUFBTSxHQUFHO0FBQ1AsUUFBSSxDQUFDLEtBQUtsSyxPQUFWLEVBQW1CO0FBQ2pCLGFBQU8sS0FBUDtBQUNEOztBQUVELFFBQUksS0FBS0EsT0FBTCxDQUFheVMsUUFBakIsRUFBMkI7QUFDekIsYUFBTyxLQUFQO0FBQ0Q7O0FBRUQsU0FBS3pTLE9BQUwsQ0FBYWtLLE1BQWI7QUFDQSxXQUFPLElBQVA7QUFDRDtBQUVEO0FBQ0Y7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ0VrSixFQUFBQSxLQUFLLENBQUMvSyxRQUFELEVBQTBCO0FBQzdCLFVBQU1ySSxPQUFPLEdBQUcsSUFBSXNRLGdCQUFKLENBQVksS0FBS3pDLGFBQUwsRUFBWixFQUFtQ3JILEdBQUQsSUFBUztBQUN6RCxVQUFJLEtBQUsxSCxNQUFMLENBQVlpQyxPQUFaLENBQW9Cd0QsVUFBcEIsR0FBaUMsS0FBckMsRUFBNEM7QUFDMUMsYUFBS3RGLGFBQUwsR0FBcUIsS0FBckI7QUFDRDs7QUFDRG9KLE1BQUFBLFFBQVEsQ0FBQzdCLEdBQUQsQ0FBUjtBQUNELEtBTGUsQ0FBaEI7QUFNQSxTQUFLekcsNEJBQUwsR0FBb0MsSUFBcEM7QUFDQSxTQUFLeU8sWUFBTCxDQUFrQnhPLE9BQWxCO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFOE4sRUFBQUEsNEJBQTRCLEdBQUc7QUFDN0IsV0FBTyxLQUFLNU8sc0JBQUwsQ0FBNEIsS0FBS0Esc0JBQUwsQ0FBNEIrUCxNQUE1QixHQUFxQyxDQUFqRSxDQUFQO0FBQ0Q7QUFFRDtBQUNGO0FBQ0E7OztBQUNFWixFQUFBQSxxQkFBcUIsQ0FBQzNLLGNBQUQsRUFBdUU7QUFDMUYsWUFBUUEsY0FBUjtBQUNFLFdBQUt2Qiw2QkFBZ0JrUixnQkFBckI7QUFDRSxlQUFPLGtCQUFQOztBQUNGLFdBQUtsUiw2QkFBZ0JtUixlQUFyQjtBQUNFLGVBQU8saUJBQVA7O0FBQ0YsV0FBS25SLDZCQUFnQm9SLFlBQXJCO0FBQ0UsZUFBTyxjQUFQOztBQUNGLFdBQUtwUiw2QkFBZ0JxUixRQUFyQjtBQUNFLGVBQU8sVUFBUDs7QUFDRjtBQUNFLGVBQU8sZ0JBQVA7QUFWSjtBQVlEOztBQTFzRW1DOztBQTZzRXRDLElBQUlDLGdEQUFnRCxHQUFHLEtBQXZEOztBQUNBLFNBQVNwUyw2Q0FBVCxHQUF5RDtBQUN2RCxNQUFJb1MsZ0RBQUosRUFBc0Q7QUFDcEQ7QUFDRDs7QUFFREEsRUFBQUEsZ0RBQWdELEdBQUcsSUFBbkQ7QUFFQWxXLEVBQUFBLE9BQU8sQ0FBQ0MsV0FBUixDQUNFLCtIQUNBLG1EQUZGLEVBR0Usb0JBSEYsRUFJRW1CLFVBQVUsQ0FBQytVLFNBQVgsQ0FBcUI1TSxFQUp2QjtBQU1EOztBQUVELFNBQVM2TSxnQkFBVCxDQUEwQjVLLEtBQTFCLEVBQTRFO0FBQzFFLE1BQUlBLEtBQUssWUFBWTZLLHlCQUFyQixFQUFxQztBQUNuQzdLLElBQUFBLEtBQUssR0FBR0EsS0FBSyxDQUFDOEssTUFBTixDQUFhLENBQWIsQ0FBUjtBQUNEOztBQUNELFNBQVE5SyxLQUFLLFlBQVkxQyx1QkFBbEIsSUFBc0MsQ0FBQyxDQUFDMEMsS0FBSyxDQUFDK0ssV0FBckQ7QUFDRDs7ZUFFY25WLFU7O0FBQ2ZvVixNQUFNLENBQUNDLE9BQVAsR0FBaUJyVixVQUFqQjtBQUVBQSxVQUFVLENBQUMrVSxTQUFYLENBQXFCN04sS0FBckIsR0FBNkI7QUFDM0JDLEVBQUFBLFdBQVcsRUFBRTtBQUNYUSxJQUFBQSxJQUFJLEVBQUUsYUFESztBQUVYcUUsSUFBQUEsTUFBTSxFQUFFO0FBRkcsR0FEYztBQUszQjlELEVBQUFBLFVBQVUsRUFBRTtBQUNWUCxJQUFBQSxJQUFJLEVBQUUsWUFESTtBQUVWaUUsSUFBQUEsS0FBSyxFQUFFLFlBQVc7QUFDaEIsV0FBS2xELG9CQUFMO0FBQ0QsS0FKUztBQUtWc0QsSUFBQUEsTUFBTSxFQUFFO0FBQ043QixNQUFBQSxXQUFXLEVBQUUsWUFBVztBQUN0QixhQUFLbEMsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BSEs7QUFJTm5GLE1BQUFBLGNBQWMsRUFBRSxZQUFXO0FBQ3pCLGFBQUsyRSxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0Q7QUFOSztBQUxFLEdBTGU7QUFtQjNCbUMsRUFBQUEsYUFBYSxFQUFFO0FBQ2JqRCxJQUFBQSxJQUFJLEVBQUUsY0FETztBQUViaUUsSUFBQUEsS0FBSyxFQUFFLFlBQVc7QUFDaEIsV0FBS21CLGtCQUFMO0FBRUEsV0FBSzdMLFNBQUwsQ0FBZW9VLFdBQWYsR0FBNkJDLElBQTdCLENBQW1DdE0sT0FBRCxJQUFhO0FBQzdDLGFBQUtxQyxhQUFMLENBQW1CLFNBQW5CLEVBQThCckMsT0FBOUI7QUFDRCxPQUZELEVBRUlwQixHQUFELElBQVM7QUFDVixhQUFLc0MsV0FBTCxDQUFpQnRDLEdBQWpCO0FBQ0QsT0FKRDtBQUtELEtBVlk7QUFXYm1FLElBQUFBLE1BQU0sRUFBRTtBQUNON0IsTUFBQUEsV0FBVyxFQUFFLFlBQVc7QUFDdEIsYUFBS2xDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQUhLO0FBSU5uRixNQUFBQSxjQUFjLEVBQUUsWUFBVztBQUN6QixhQUFLMkUsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BTks7QUFPTlEsTUFBQUEsT0FBTyxFQUFFLFVBQVNBLE9BQVQsRUFBa0I7QUFDekJBLFFBQUFBLE9BQU8sQ0FBQ2QsRUFBUixDQUFXLE1BQVgsRUFBb0JyRSxJQUFELElBQVU7QUFDM0IsZUFBS2tKLGtCQUFMLENBQXdCbEosSUFBeEI7QUFDRCxTQUZEO0FBSUFtRixRQUFBQSxPQUFPLENBQUNqQixJQUFSLENBQWEsS0FBYixFQUFvQixNQUFNO0FBQ3hCLGdCQUFNd04sZUFBZSxHQUFHLElBQUk5SSx3QkFBSixDQUFvQixLQUFLbEwsYUFBekIsQ0FBeEI7QUFDQSxlQUFLVixLQUFMLENBQVdrRCxPQUFYLENBQW1CLFlBQVc7QUFDNUIsbUJBQU93UixlQUFlLENBQUMxSSxRQUFoQixDQUF5QixJQUF6QixDQUFQO0FBQ0QsV0FGRDs7QUFJQSxjQUFJMEksZUFBZSxDQUFDcFYsZUFBaEIsS0FBb0MsQ0FBeEMsRUFBMkM7QUFDekMsaUJBQUtBLGVBQUwsR0FBdUIsSUFBdkI7QUFDRDs7QUFFRCxjQUFJb1YsZUFBZSxDQUFDQyxnQkFBaEIsS0FBcUMsSUFBckMsSUFBNkNELGVBQWUsQ0FBQ0MsZ0JBQWhCLEtBQXFDLEtBQXRGLEVBQTZGO0FBQUE7O0FBQzNGLGdCQUFJLENBQUMsS0FBS3RWLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J1QyxPQUF6QixFQUFrQztBQUNoQyxtQkFBSzJELElBQUwsQ0FBVSxTQUFWLEVBQXFCLElBQUlaLHVCQUFKLENBQW9CLGtFQUFwQixFQUF3RixVQUF4RixDQUFyQjtBQUNBLHFCQUFPLEtBQUtjLEtBQUwsRUFBUDtBQUNEOztBQUVELGlCQUFLdEgsU0FBTCxDQUFld1UsUUFBZixDQUF3QixLQUFLclYsYUFBN0IsZ0RBQTRDLEtBQUtZLFdBQWpELHNEQUE0QyxrQkFBa0JlLE1BQTlELHlFQUF3RSxLQUFLN0IsTUFBTCxDQUFZNkIsTUFBcEYsRUFBNEYsS0FBSzdCLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0IyRCxzQkFBaEg7QUFDQSxpQkFBS2tDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXK0Usc0JBQTdCO0FBQ0QsV0FSRCxNQVFPO0FBQ0wsaUJBQUtpQixnQkFBTDtBQUVBLGtCQUFNO0FBQUVqTCxjQUFBQTtBQUFGLGdCQUFxQixLQUFLOUIsTUFBaEM7O0FBQ0EsZ0JBQUk4QixjQUFjLENBQUNFLElBQWYsS0FBd0IsTUFBNUIsRUFBb0M7QUFDbEMsbUJBQUs4RixZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3lPLHFCQUE3QjtBQUNELGFBRkQsTUFFTztBQUNMLG1CQUFLMU4sWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVc2SCwrQkFBN0I7QUFDRDtBQUNGO0FBQ0YsU0E1QkQ7QUE2QkQ7QUF6Q0s7QUFYSyxHQW5CWTtBQTBFM0I1QyxFQUFBQSxTQUFTLEVBQUU7QUFDVHhFLElBQUFBLElBQUksRUFBRSxXQURHO0FBRVRpRSxJQUFBQSxLQUFLLEVBQUUsWUFBVztBQUNoQixXQUFLMUMsaUJBQUwsQ0FBdUJ0SixZQUFZLENBQUNFLFFBQXBDO0FBQ0QsS0FKUTtBQUtUa00sSUFBQUEsTUFBTSxFQUFFO0FBQ04vQyxNQUFBQSxPQUFPLEVBQUUsWUFBVyxDQUNuQixDQUZLO0FBR05rQixNQUFBQSxXQUFXLEVBQUUsWUFBVztBQUN0QixhQUFLbEMsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BTEs7QUFNTm5GLE1BQUFBLGNBQWMsRUFBRSxZQUFXO0FBQ3pCLGFBQUsyRSxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0QsT0FSSztBQVNObU4sTUFBQUEsU0FBUyxFQUFFLFlBQVc7QUFDcEIsYUFBSzNOLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXZ0IsVUFBN0I7QUFDRDtBQVhLO0FBTEMsR0ExRWdCO0FBNkYzQmtFLEVBQUFBLHVCQUF1QixFQUFFO0FBQ3ZCekUsSUFBQUEsSUFBSSxFQUFFLHlCQURpQjtBQUV2QmlFLElBQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLFdBQUtsTCxzQkFBTDtBQUNBLFdBQUt3SSxpQkFBTCxDQUF1QnRKLFlBQVksQ0FBQ0csS0FBcEM7QUFDRCxLQUxzQjtBQU12QmlNLElBQUFBLE1BQU0sRUFBRTtBQUNOL0MsTUFBQUEsT0FBTyxFQUFFLFlBQVcsQ0FDbkIsQ0FGSztBQUdOa0IsTUFBQUEsV0FBVyxFQUFFLFlBQVc7QUFDdEIsYUFBS2xDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQUxLO0FBTU5uRixNQUFBQSxjQUFjLEVBQUUsWUFBVztBQUN6QixhQUFLMkUsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BUks7QUFTTm9OLE1BQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLGFBQUt6SyxnQkFBTDtBQUNEO0FBWEs7QUFOZSxHQTdGRTtBQWlIM0JhLEVBQUFBLHNCQUFzQixFQUFFO0FBQ3RCdEUsSUFBQUEsSUFBSSxFQUFFLHVCQURnQjtBQUV0QmlFLElBQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLFdBQUsxSyxTQUFMLENBQWVvVSxXQUFmLEdBQTZCQyxJQUE3QixDQUFtQ3RNLE9BQUQsSUFBYTtBQUM3QyxhQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsT0FGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsYUFBS3NDLFdBQUwsQ0FBaUJ0QyxHQUFqQjtBQUNELE9BSkQ7QUFLRCxLQVJxQjtBQVN0Qm1FLElBQUFBLE1BQU0sRUFBRTtBQUNON0IsTUFBQUEsV0FBVyxFQUFFLFlBQVc7QUFDdEIsYUFBS2xDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQUhLO0FBSU5uRixNQUFBQSxjQUFjLEVBQUUsWUFBVztBQUN6QixhQUFLMkUsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BTks7QUFPTlEsTUFBQUEsT0FBTyxFQUFFLFVBQVNBLE9BQVQsRUFBa0I7QUFDekJBLFFBQUFBLE9BQU8sQ0FBQ2QsRUFBUixDQUFXLE1BQVgsRUFBb0JyRSxJQUFELElBQVU7QUFDM0IsZUFBSzVDLFNBQUwsQ0FBZTRVLGdCQUFmLENBQWdDaFMsSUFBaEM7QUFDRCxTQUZEO0FBSUFtRixRQUFBQSxPQUFPLENBQUNqQixJQUFSLENBQWEsS0FBYixFQUFvQixNQUFNO0FBQ3hCLGNBQUksS0FBSzlHLFNBQUwsQ0FBZTZVLHNCQUFuQixFQUEyQztBQUN6QyxpQkFBSzdJLGdCQUFMO0FBRUEsa0JBQU07QUFBRWpMLGNBQUFBO0FBQUYsZ0JBQXFCLEtBQUs5QixNQUFoQzs7QUFDQSxnQkFBSThCLGNBQWMsQ0FBQ0UsSUFBZixLQUF3QixpQ0FBeEIsSUFBNkRGLGNBQWMsQ0FBQ0UsSUFBZixLQUF3QiwrQkFBckYsSUFBd0hGLGNBQWMsQ0FBQ0UsSUFBZixLQUF3Qix3Q0FBaEosSUFBNExGLGNBQWMsQ0FBQ0UsSUFBZixLQUF3QixpREFBcE4sSUFBeVFGLGNBQWMsQ0FBQ0UsSUFBZixLQUF3QixnQ0FBclMsRUFBdVU7QUFDclUsbUJBQUs4RixZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBVzhPLHdCQUE3QjtBQUNELGFBRkQsTUFFTyxJQUFJL1QsY0FBYyxDQUFDRSxJQUFmLEtBQXdCLE1BQTVCLEVBQW9DO0FBQ3pDLG1CQUFLOEYsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd5TyxxQkFBN0I7QUFDRCxhQUZNLE1BRUE7QUFDTCxtQkFBSzFOLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXNkgsK0JBQTdCO0FBQ0Q7QUFDRixXQVhELE1BV087QUFDTCxpQkFBSzdOLFNBQUwsQ0FBZW9VLFdBQWYsR0FBNkJDLElBQTdCLENBQW1DdE0sT0FBRCxJQUFhO0FBQzdDLG1CQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsYUFGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsbUJBQUtzQyxXQUFMLENBQWlCdEMsR0FBakI7QUFDRCxhQUpEO0FBS0Q7QUFDRixTQW5CRDtBQW9CRDtBQWhDSztBQVRjLEdBakhHO0FBNkozQmtILEVBQUFBLCtCQUErQixFQUFFO0FBQy9CcEgsSUFBQUEsSUFBSSxFQUFFLDZCQUR5QjtBQUUvQmlFLElBQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLFdBQUsxSyxTQUFMLENBQWVvVSxXQUFmLEdBQTZCQyxJQUE3QixDQUFtQ3RNLE9BQUQsSUFBYTtBQUM3QyxhQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsT0FGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsYUFBS3NDLFdBQUwsQ0FBaUJ0QyxHQUFqQjtBQUNELE9BSkQ7QUFLRCxLQVI4QjtBQVMvQm1FLElBQUFBLE1BQU0sRUFBRTtBQUNON0IsTUFBQUEsV0FBVyxFQUFFLFlBQVc7QUFDdEIsYUFBS2xDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQUhLO0FBSU5uRixNQUFBQSxjQUFjLEVBQUUsWUFBVztBQUN6QixhQUFLMkUsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BTks7QUFPTlEsTUFBQUEsT0FBTyxFQUFFLFVBQVNBLE9BQVQsRUFBa0I7QUFDekIsY0FBTVksT0FBTyxHQUFHLElBQUlvTSwyQkFBSixDQUF1QixJQUF2QixDQUFoQjtBQUNBLGNBQU1DLGlCQUFpQixHQUFHLEtBQUt0TSx1QkFBTCxDQUE2QlgsT0FBN0IsRUFBc0NZLE9BQXRDLENBQTFCO0FBRUFxTSxRQUFBQSxpQkFBaUIsQ0FBQ2xPLElBQWxCLENBQXVCLEtBQXZCLEVBQThCLE1BQU07QUFDbEMsY0FBSTZCLE9BQU8sQ0FBQ3NNLGdCQUFaLEVBQThCO0FBQzVCLGdCQUFJdE0sT0FBTyxDQUFDNUksV0FBWixFQUF5QjtBQUN2QixtQkFBS0EsV0FBTCxHQUFtQjRJLE9BQU8sQ0FBQzVJLFdBQTNCO0FBQ0EsbUJBQUtnSCxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2lGLFNBQTdCO0FBQ0QsYUFIRCxNQUdPO0FBQ0wsbUJBQUtsRSxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2tQLDZCQUE3QjtBQUNEO0FBQ0YsV0FQRCxNQU9PLElBQUksS0FBS3ZWLFVBQVQsRUFBcUI7QUFDMUIsZ0JBQUltVSxnQkFBZ0IsQ0FBQyxLQUFLblUsVUFBTixDQUFwQixFQUF1QztBQUNyQyxtQkFBS0MsS0FBTCxDQUFXNEosR0FBWCxDQUFlLHFDQUFmO0FBQ0EsbUJBQUt6QyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2tGLHVCQUE3QjtBQUNELGFBSEQsTUFHTztBQUNMLG1CQUFLOUQsSUFBTCxDQUFVLFNBQVYsRUFBcUIsS0FBS3pILFVBQTFCO0FBQ0EsbUJBQUtvSCxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0Q7QUFDRixXQVJNLE1BUUE7QUFDTCxpQkFBS0gsSUFBTCxDQUFVLFNBQVYsRUFBcUIsSUFBSVosdUJBQUosQ0FBb0IsZUFBcEIsRUFBcUMsUUFBckMsQ0FBckI7QUFDQSxpQkFBS08sWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNEO0FBQ0YsU0FwQkQ7QUFxQkQ7QUFoQ0s7QUFUdUIsR0E3Sk47QUF5TTNCa04sRUFBQUEscUJBQXFCLEVBQUU7QUFDckJoTyxJQUFBQSxJQUFJLEVBQUUseUJBRGU7QUFFckJpRSxJQUFBQSxLQUFLLEVBQUUsWUFBVztBQUNoQixXQUFLMUssU0FBTCxDQUFlb1UsV0FBZixHQUE2QkMsSUFBN0IsQ0FBbUN0TSxPQUFELElBQWE7QUFDN0MsYUFBS3FDLGFBQUwsQ0FBbUIsU0FBbkIsRUFBOEJyQyxPQUE5QjtBQUNELE9BRkQsRUFFSXBCLEdBQUQsSUFBUztBQUNWLGFBQUtzQyxXQUFMLENBQWlCdEMsR0FBakI7QUFDRCxPQUpEO0FBS0QsS0FSb0I7QUFTckJtRSxJQUFBQSxNQUFNLEVBQUU7QUFDTjdCLE1BQUFBLFdBQVcsRUFBRSxZQUFXO0FBQ3RCLGFBQUtsQyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0QsT0FISztBQUlObkYsTUFBQUEsY0FBYyxFQUFFLFlBQVc7QUFDekIsYUFBSzJFLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQU5LO0FBT05RLE1BQUFBLE9BQU8sRUFBRSxVQUFTQSxPQUFULEVBQWtCO0FBQ3pCLGNBQU1ZLE9BQU8sR0FBRyxJQUFJb00sMkJBQUosQ0FBdUIsSUFBdkIsQ0FBaEI7QUFDQSxjQUFNQyxpQkFBaUIsR0FBRyxLQUFLdE0sdUJBQUwsQ0FBNkJYLE9BQTdCLEVBQXNDWSxPQUF0QyxDQUExQjtBQUVBcU0sUUFBQUEsaUJBQWlCLENBQUNsTyxJQUFsQixDQUF1QixLQUF2QixFQUE4QixNQUFNO0FBQ2xDLGNBQUk2QixPQUFPLENBQUNzTSxnQkFBWixFQUE4QjtBQUM1QixnQkFBSXRNLE9BQU8sQ0FBQzVJLFdBQVosRUFBeUI7QUFDdkIsbUJBQUtBLFdBQUwsR0FBbUI0SSxPQUFPLENBQUM1SSxXQUEzQjtBQUNBLG1CQUFLZ0gsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVdpRixTQUE3QjtBQUNELGFBSEQsTUFHTztBQUNMLG1CQUFLbEUsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVdrUCw2QkFBN0I7QUFDRDtBQUNGLFdBUEQsTUFPTyxJQUFJLEtBQUtyVixVQUFULEVBQXFCO0FBQzFCLGtCQUFNa0IsY0FBYyxHQUFHLEtBQUs5QixNQUFMLENBQVk4QixjQUFuQztBQUVBLGtCQUFNK0IsT0FBTyxHQUFHLElBQUlxUyxvQkFBSixDQUF3QjtBQUN0Q2hVLGNBQUFBLE1BQU0sRUFBRUosY0FBYyxDQUFDRyxPQUFmLENBQXVCQyxNQURPO0FBRXRDQyxjQUFBQSxRQUFRLEVBQUVMLGNBQWMsQ0FBQ0csT0FBZixDQUF1QkUsUUFGSztBQUd0Q0MsY0FBQUEsUUFBUSxFQUFFTixjQUFjLENBQUNHLE9BQWYsQ0FBdUJHLFFBSEs7QUFJdEN4QixjQUFBQSxVQUFVLEVBQUUsS0FBS0E7QUFKcUIsYUFBeEIsQ0FBaEI7QUFPQSxpQkFBS0csU0FBTCxDQUFla0csV0FBZixDQUEyQkMsYUFBS2lQLFlBQWhDLEVBQThDdFMsT0FBTyxDQUFDRixJQUF0RDtBQUNBLGlCQUFLaEQsS0FBTCxDQUFXa0QsT0FBWCxDQUFtQixZQUFXO0FBQzVCLHFCQUFPQSxPQUFPLENBQUM4SSxRQUFSLENBQWlCLElBQWpCLENBQVA7QUFDRCxhQUZEO0FBSUEsaUJBQUsvTCxVQUFMLEdBQWtCbUIsU0FBbEI7QUFFQSxpQkFBS2hCLFNBQUwsQ0FBZW9VLFdBQWYsR0FBNkJDLElBQTdCLENBQW1DdE0sT0FBRCxJQUFhO0FBQzdDLG1CQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsYUFGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsbUJBQUtzQyxXQUFMLENBQWlCdEMsR0FBakI7QUFDRCxhQUpEO0FBS0QsV0F0Qk0sTUFzQkEsSUFBSSxLQUFLaEgsVUFBVCxFQUFxQjtBQUMxQixnQkFBSW1VLGdCQUFnQixDQUFDLEtBQUtuVSxVQUFOLENBQXBCLEVBQXVDO0FBQ3JDLG1CQUFLQyxLQUFMLENBQVc0SixHQUFYLENBQWUscUNBQWY7QUFDQSxtQkFBS3pDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXa0YsdUJBQTdCO0FBQ0QsYUFIRCxNQUdPO0FBQ0wsbUJBQUs5RCxJQUFMLENBQVUsU0FBVixFQUFxQixLQUFLekgsVUFBMUI7QUFDQSxtQkFBS29ILFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRDtBQUNGLFdBUk0sTUFRQTtBQUNMLGlCQUFLSCxJQUFMLENBQVUsU0FBVixFQUFxQixJQUFJWix1QkFBSixDQUFvQixlQUFwQixFQUFxQyxRQUFyQyxDQUFyQjtBQUNBLGlCQUFLTyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0Q7QUFDRixTQTFDRDtBQTJDRDtBQXRESztBQVRhLEdBek1JO0FBMlEzQnVOLEVBQUFBLHdCQUF3QixFQUFFO0FBQ3hCck8sSUFBQUEsSUFBSSxFQUFFLHVCQURrQjtBQUV4QmlFLElBQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLFdBQUsxSyxTQUFMLENBQWVvVSxXQUFmLEdBQTZCQyxJQUE3QixDQUFtQ3RNLE9BQUQsSUFBYTtBQUM3QyxhQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsT0FGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsYUFBS3NDLFdBQUwsQ0FBaUJ0QyxHQUFqQjtBQUNELE9BSkQ7QUFLRCxLQVJ1QjtBQVN4Qm1FLElBQUFBLE1BQU0sRUFBRTtBQUNON0IsTUFBQUEsV0FBVyxFQUFFLFlBQVc7QUFDdEIsYUFBS2xDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQUhLO0FBSU5uRixNQUFBQSxjQUFjLEVBQUUsWUFBVztBQUN6QixhQUFLMkUsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNELE9BTks7QUFPTlEsTUFBQUEsT0FBTyxFQUFFLFVBQVNBLE9BQVQsRUFBa0I7QUFDekIsY0FBTVksT0FBTyxHQUFHLElBQUlvTSwyQkFBSixDQUF1QixJQUF2QixDQUFoQjtBQUNBLGNBQU1DLGlCQUFpQixHQUFHLEtBQUt0TSx1QkFBTCxDQUE2QlgsT0FBN0IsRUFBc0NZLE9BQXRDLENBQTFCO0FBRUFxTSxRQUFBQSxpQkFBaUIsQ0FBQ2xPLElBQWxCLENBQXVCLEtBQXZCLEVBQThCLE1BQU07QUFDbEMsY0FBSTZCLE9BQU8sQ0FBQ3NNLGdCQUFaLEVBQThCO0FBQzVCLGdCQUFJdE0sT0FBTyxDQUFDNUksV0FBWixFQUF5QjtBQUN2QixtQkFBS0EsV0FBTCxHQUFtQjRJLE9BQU8sQ0FBQzVJLFdBQTNCO0FBQ0EsbUJBQUtnSCxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2lGLFNBQTdCO0FBQ0QsYUFIRCxNQUdPO0FBQ0wsbUJBQUtsRSxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2tQLDZCQUE3QjtBQUNEOztBQUVEO0FBQ0Q7O0FBRUQsZ0JBQU1HLGdCQUFnQixHQUFHMU0sT0FBTyxDQUFDME0sZ0JBQWpDOztBQUVBLGNBQUlBLGdCQUFnQixJQUFJQSxnQkFBZ0IsQ0FBQ0MsTUFBckMsSUFBK0NELGdCQUFnQixDQUFDRSxHQUFwRSxFQUF5RTtBQUN2RSxrQkFBTXhVLGNBQWMsR0FBRyxLQUFLOUIsTUFBTCxDQUFZOEIsY0FBbkM7QUFDQSxrQkFBTXlVLFVBQVUsR0FBRyxJQUFJQyxRQUFKLENBQVEsV0FBUixFQUFxQkosZ0JBQWdCLENBQUNFLEdBQXRDLEVBQTJDM0osUUFBM0MsRUFBbkI7O0FBRUEsa0JBQU04SixRQUFRLEdBQUlsTixRQUFELElBQTZEO0FBQzVFLG9CQUFNbU4sdUJBQXVCLEdBQUlDLFdBQUQsSUFBa0M7QUFDaEVBLGdCQUFBQSxXQUFXLENBQUNGLFFBQVosQ0FBcUJGLFVBQXJCLEVBQWlDbkIsSUFBakMsQ0FBdUN3QixhQUFELElBQW1CO0FBQ3ZEck4sa0JBQUFBLFFBQVEsQ0FBQyxJQUFELEVBQU9xTixhQUFQLGFBQU9BLGFBQVAsdUJBQU9BLGFBQWEsQ0FBRW5VLEtBQXRCLENBQVI7QUFDRCxpQkFGRCxFQUVHOEcsUUFGSDtBQUdELGVBSkQ7O0FBTUEsa0JBQUl6SCxjQUFjLENBQUNFLElBQWYsS0FBd0IsaUNBQTVCLEVBQStEO0FBQUE7O0FBQzdELHNCQUFNMlUsV0FBVyxHQUFHLElBQUlFLG9DQUFKLDBCQUNsQi9VLGNBQWMsQ0FBQ0csT0FBZixDQUF1Qk8sUUFETCx5RUFDaUIsUUFEakIsRUFFbEJWLGNBQWMsQ0FBQ0csT0FBZixDQUF1QkssUUFGTCxFQUdsQlIsY0FBYyxDQUFDRyxPQUFmLENBQXVCRSxRQUhMLEVBSWxCTCxjQUFjLENBQUNHLE9BQWYsQ0FBdUJHLFFBSkwsQ0FBcEI7QUFPQXNVLGdCQUFBQSx1QkFBdUIsQ0FBQ0MsV0FBRCxDQUF2QjtBQUNELGVBVEQsTUFTTyxJQUFJN1UsY0FBYyxDQUFDRSxJQUFmLEtBQXdCLCtCQUF4QixJQUEyREYsY0FBYyxDQUFDRSxJQUFmLEtBQXdCLHdDQUF2RixFQUFpSTtBQUN0SSxzQkFBTThVLE9BQU8sR0FBR2hWLGNBQWMsQ0FBQ0csT0FBZixDQUF1QkssUUFBdkIsR0FBa0MsQ0FBRVIsY0FBYyxDQUFDRyxPQUFmLENBQXVCSyxRQUF6QixFQUFtQyxFQUFuQyxDQUFsQyxHQUE0RSxDQUFFLEVBQUYsQ0FBNUY7QUFDQSxzQkFBTXFVLFdBQVcsR0FBRyxJQUFJSSxtQ0FBSixDQUE4QixHQUFHRCxPQUFqQyxDQUFwQjtBQUVBSixnQkFBQUEsdUJBQXVCLENBQUNDLFdBQUQsQ0FBdkI7QUFDRCxlQUxNLE1BS0EsSUFBSTdVLGNBQWMsQ0FBQ0UsSUFBZixLQUF3QixnQ0FBNUIsRUFBOEQ7QUFDbkUsc0JBQU1vRyxJQUFJLEdBQUd0RyxjQUFjLENBQUNHLE9BQWYsQ0FBdUJLLFFBQXZCLEdBQWtDO0FBQUUwVSxrQkFBQUEsdUJBQXVCLEVBQUVsVixjQUFjLENBQUNHLE9BQWYsQ0FBdUJLO0FBQWxELGlCQUFsQyxHQUFpRyxFQUE5RztBQUNBLHNCQUFNcVUsV0FBVyxHQUFHLElBQUlNLGdDQUFKLENBQTJCN08sSUFBM0IsQ0FBcEI7QUFFQXNPLGdCQUFBQSx1QkFBdUIsQ0FBQ0MsV0FBRCxDQUF2QjtBQUNELGVBTE0sTUFLQSxJQUFJN1UsY0FBYyxDQUFDRSxJQUFmLEtBQXdCLGlEQUE1QixFQUErRTtBQUNwRixzQkFBTTJVLFdBQVcsR0FBRyxJQUFJTyxnQ0FBSixDQUNsQnBWLGNBQWMsQ0FBQ0csT0FBZixDQUF1Qk8sUUFETCxFQUVsQlYsY0FBYyxDQUFDRyxPQUFmLENBQXVCSyxRQUZMLEVBR2xCUixjQUFjLENBQUNHLE9BQWYsQ0FBdUJTLFlBSEwsQ0FBcEI7QUFNQWdVLGdCQUFBQSx1QkFBdUIsQ0FBQ0MsV0FBRCxDQUF2QjtBQUNEO0FBQ0YsYUFuQ0Q7O0FBcUNBRixZQUFBQSxRQUFRLENBQUMsQ0FBQy9PLEdBQUQsRUFBTWpGLEtBQU4sS0FBZ0I7QUFDdkIsa0JBQUlpRixHQUFKLEVBQVM7QUFDUCxxQkFBS2hILFVBQUwsR0FBa0IsSUFBSW9VLHlCQUFKLENBQ2hCLENBQUMsSUFBSXZOLHVCQUFKLENBQW9CLDBEQUFwQixFQUFnRixVQUFoRixDQUFELEVBQThGRyxHQUE5RixDQURnQixDQUFsQjtBQUVBLHFCQUFLUyxJQUFMLENBQVUsU0FBVixFQUFxQixLQUFLekgsVUFBMUI7QUFDQSxxQkFBS29ILFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDQTtBQUNEOztBQUVELG1CQUFLK0YsdUJBQUwsQ0FBNkI1TCxLQUE3QjtBQUNELGFBVk8sQ0FBUjtBQVdELFdBcERELE1Bb0RPLElBQUksS0FBSy9CLFVBQVQsRUFBcUI7QUFDMUIsZ0JBQUltVSxnQkFBZ0IsQ0FBQyxLQUFLblUsVUFBTixDQUFwQixFQUF1QztBQUNyQyxtQkFBS0MsS0FBTCxDQUFXNEosR0FBWCxDQUFlLHFDQUFmO0FBQ0EsbUJBQUt6QyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV2tGLHVCQUE3QjtBQUNELGFBSEQsTUFHTztBQUNMLG1CQUFLOUQsSUFBTCxDQUFVLFNBQVYsRUFBcUIsS0FBS3pILFVBQTFCO0FBQ0EsbUJBQUtvSCxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0Q7QUFDRixXQVJNLE1BUUE7QUFDTCxpQkFBS0gsSUFBTCxDQUFVLFNBQVYsRUFBcUIsSUFBSVosdUJBQUosQ0FBb0IsZUFBcEIsRUFBcUMsUUFBckMsQ0FBckI7QUFDQSxpQkFBS08sWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUNEO0FBQ0YsU0E5RUQ7QUErRUQ7QUExRks7QUFUZ0IsR0EzUUM7QUFpWDNCMk4sRUFBQUEsNkJBQTZCLEVBQUU7QUFDN0J6TyxJQUFBQSxJQUFJLEVBQUUsMkJBRHVCO0FBRTdCaUUsSUFBQUEsS0FBSyxFQUFFLFlBQVc7QUFDaEIsV0FBS29ELGNBQUw7QUFFQSxXQUFLOU4sU0FBTCxDQUFlb1UsV0FBZixHQUE2QkMsSUFBN0IsQ0FBbUN0TSxPQUFELElBQWE7QUFDN0MsYUFBS3FDLGFBQUwsQ0FBbUIsU0FBbkIsRUFBOEJyQyxPQUE5QjtBQUNELE9BRkQsRUFFSXBCLEdBQUQsSUFBUztBQUNWLGFBQUtzQyxXQUFMLENBQWlCdEMsR0FBakI7QUFDRCxPQUpEO0FBS0QsS0FWNEI7QUFXN0JtRSxJQUFBQSxNQUFNLEVBQUU7QUFDTjdCLE1BQUFBLFdBQVcsRUFBRSxTQUFTQSxXQUFULEdBQXVCO0FBQ2xDLGFBQUtsQyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0QsT0FISztBQUlObkYsTUFBQUEsY0FBYyxFQUFFLFlBQVc7QUFDekIsYUFBSzJFLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFDRCxPQU5LO0FBT05RLE1BQUFBLE9BQU8sRUFBRSxVQUFTQSxPQUFULEVBQWtCO0FBQ3pCLGNBQU1pTixpQkFBaUIsR0FBRyxLQUFLdE0sdUJBQUwsQ0FBNkJYLE9BQTdCLEVBQXNDLElBQUlxTywrQkFBSixDQUEyQixJQUEzQixDQUF0QyxDQUExQjtBQUVBcEIsUUFBQUEsaUJBQWlCLENBQUNsTyxJQUFsQixDQUF1QixLQUF2QixFQUE4QixNQUFNO0FBQ2xDLGVBQUtDLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXeU0sU0FBN0I7QUFDQSxlQUFLL0QsbUJBQUw7QUFDRCxTQUhEO0FBSUQ7QUFkSztBQVhxQixHQWpYSjtBQTZZM0IrRCxFQUFBQSxTQUFTLEVBQUU7QUFDVGhNLElBQUFBLElBQUksRUFBRSxVQURHO0FBRVRxRSxJQUFBQSxNQUFNLEVBQUU7QUFDTjdCLE1BQUFBLFdBQVcsRUFBRSxZQUFXO0FBQ3RCLGFBQUtsQyxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3VCLEtBQTdCO0FBQ0Q7QUFISztBQUZDLEdBN1lnQjtBQXFaM0IrTCxFQUFBQSxtQkFBbUIsRUFBRTtBQUNuQjdNLElBQUFBLElBQUksRUFBRSxtQkFEYTtBQUVuQmlFLElBQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLFdBQUttQixrQkFBTDtBQUVBLFdBQUs3TCxTQUFMLENBQWVvVSxXQUFmLEdBQTZCQyxJQUE3QixDQUFtQ3RNLE9BQUQsSUFBYTtBQUM3QyxhQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsT0FGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsYUFBS3NDLFdBQUwsQ0FBaUJ0QyxHQUFqQjtBQUNELE9BSkQ7QUFLRCxLQVZrQjtBQVduQjZELElBQUFBLElBQUksRUFBRSxVQUFTNkwsU0FBVCxFQUFvQjtBQUN4QixXQUFLbE8saUJBQUw7QUFDRCxLQWJrQjtBQWNuQjJDLElBQUFBLE1BQU0sRUFBRTtBQUNON0IsTUFBQUEsV0FBVyxFQUFFLFVBQVN0QyxHQUFULEVBQWM7QUFDekIsY0FBTTJQLFVBQVUsR0FBRyxLQUFLblcsT0FBeEI7QUFDQSxhQUFLQSxPQUFMLEdBQWVhLFNBQWY7QUFDQSxhQUFLK0YsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd1QixLQUE3QjtBQUVBK08sUUFBQUEsVUFBVSxDQUFDOU4sUUFBWCxDQUFvQjdCLEdBQXBCO0FBQ0QsT0FQSztBQVFOb0IsTUFBQUEsT0FBTyxFQUFFLFVBQVNBLE9BQVQsRUFBa0I7QUFBQTs7QUFDekI7QUFDQSxhQUFLSSxpQkFBTDtBQUVBLGNBQU02TSxpQkFBaUIsR0FBRyxLQUFLdE0sdUJBQUwsQ0FBNkJYLE9BQTdCLEVBQXNDLElBQUl3Tyw0QkFBSixDQUF3QixJQUF4QixFQUE4QixLQUFLcFcsT0FBbkMsQ0FBdEMsQ0FBMUIsQ0FKeUIsQ0FNekI7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFDQSxZQUFJLHNCQUFLQSxPQUFMLHdEQUFjeVMsUUFBZCxJQUEwQixLQUFLcFMsV0FBbkMsRUFBZ0Q7QUFDOUMsaUJBQU8sS0FBS3VHLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXd1EsY0FBN0IsQ0FBUDtBQUNEOztBQUVELGNBQU1DLFFBQVEsR0FBRyxNQUFNO0FBQ3JCekIsVUFBQUEsaUJBQWlCLENBQUM1QixNQUFsQjtBQUNELFNBRkQ7O0FBR0EsY0FBTXNELE9BQU8sR0FBRyxNQUFNO0FBQUE7O0FBQ3BCMUIsVUFBQUEsaUJBQWlCLENBQUMyQixLQUFsQjtBQUVBLGlDQUFLeFcsT0FBTCxrRUFBYzJHLElBQWQsQ0FBbUIsUUFBbkIsRUFBNkIyUCxRQUE3QjtBQUNELFNBSkQ7O0FBTUEsK0JBQUt0VyxPQUFMLGtFQUFjOEcsRUFBZCxDQUFpQixPQUFqQixFQUEwQnlQLE9BQTFCOztBQUVBLFlBQUksS0FBS3ZXLE9BQUwsWUFBd0JzUSxnQkFBeEIsSUFBbUMsS0FBS3RRLE9BQUwsQ0FBYWdULE1BQXBELEVBQTREO0FBQzFEdUQsVUFBQUEsT0FBTztBQUNSOztBQUVELGNBQU1uRyxRQUFRLEdBQUcsTUFBTTtBQUFBOztBQUNyQnlFLFVBQUFBLGlCQUFpQixDQUFDcE8sY0FBbEIsQ0FBaUMsS0FBakMsRUFBd0NnUSxjQUF4Qzs7QUFFQSxjQUFJLEtBQUt6VyxPQUFMLFlBQXdCc1EsZ0JBQXhCLElBQW1DLEtBQUt0USxPQUFMLENBQWFnVCxNQUFwRCxFQUE0RDtBQUMxRDtBQUNBLGlCQUFLaFQsT0FBTCxDQUFhaVQsTUFBYjtBQUNEOztBQUVELGlDQUFLalQsT0FBTCxrRUFBY3lHLGNBQWQsQ0FBNkIsT0FBN0IsRUFBc0M4UCxPQUF0QztBQUNBLGlDQUFLdlcsT0FBTCxrRUFBY3lHLGNBQWQsQ0FBNkIsUUFBN0IsRUFBdUM2UCxRQUF2QyxFQVRxQixDQVdyQjtBQUNBO0FBQ0E7QUFDQTs7QUFDQSxlQUFLMVAsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd3USxjQUE3QjtBQUNELFNBaEJEOztBQWtCQSxjQUFNSSxjQUFjLEdBQUcsTUFBTTtBQUFBOztBQUMzQixpQ0FBS3pXLE9BQUwsa0VBQWN5RyxjQUFkLENBQTZCLFFBQTdCLEVBQXVDLEtBQUtqRyx1QkFBNUM7QUFDQSxpQ0FBS1IsT0FBTCxrRUFBY3lHLGNBQWQsQ0FBNkIsUUFBN0IsRUFBdUMySixRQUF2QztBQUNBLGlDQUFLcFEsT0FBTCxrRUFBY3lHLGNBQWQsQ0FBNkIsT0FBN0IsRUFBc0M4UCxPQUF0QztBQUNBLGlDQUFLdlcsT0FBTCxrRUFBY3lHLGNBQWQsQ0FBNkIsUUFBN0IsRUFBdUM2UCxRQUF2QztBQUVBLGVBQUsxUCxZQUFMLENBQWtCLEtBQUtmLEtBQUwsQ0FBV3lNLFNBQTdCO0FBQ0EsZ0JBQU02RCxVQUFVLEdBQUcsS0FBS25XLE9BQXhCO0FBQ0EsZUFBS0EsT0FBTCxHQUFlYSxTQUFmOztBQUNBLGNBQUksS0FBSy9CLE1BQUwsQ0FBWWlDLE9BQVosQ0FBb0J3RCxVQUFwQixHQUFpQyxLQUFqQyxJQUEwQzRSLFVBQVUsQ0FBQ3BOLEtBQXJELElBQThELEtBQUszSixVQUF2RSxFQUFtRjtBQUNqRixpQkFBS0gsYUFBTCxHQUFxQixLQUFyQjtBQUNEOztBQUNEa1gsVUFBQUEsVUFBVSxDQUFDOU4sUUFBWCxDQUFvQjhOLFVBQVUsQ0FBQ3BOLEtBQS9CLEVBQXNDb04sVUFBVSxDQUFDeEQsUUFBakQsRUFBMkR3RCxVQUFVLENBQUN0RyxJQUF0RTtBQUNELFNBYkQ7O0FBZUFnRixRQUFBQSxpQkFBaUIsQ0FBQ2xPLElBQWxCLENBQXVCLEtBQXZCLEVBQThCOFAsY0FBOUI7QUFDQSxnQ0FBS3pXLE9BQUwsb0VBQWMyRyxJQUFkLENBQW1CLFFBQW5CLEVBQTZCeUosUUFBN0I7QUFDRDtBQTVFSztBQWRXLEdBclpNO0FBa2YzQmlHLEVBQUFBLGNBQWMsRUFBRTtBQUNkL1AsSUFBQUEsSUFBSSxFQUFFLGVBRFE7QUFFZGlFLElBQUFBLEtBQUssRUFBRSxZQUFXO0FBQ2hCLFdBQUttQixrQkFBTDtBQUVBLFdBQUs3TCxTQUFMLENBQWVvVSxXQUFmLEdBQTZCQyxJQUE3QixDQUFtQ3RNLE9BQUQsSUFBYTtBQUM3QyxhQUFLcUMsYUFBTCxDQUFtQixTQUFuQixFQUE4QnJDLE9BQTlCO0FBQ0QsT0FGRCxFQUVJcEIsR0FBRCxJQUFTO0FBQ1YsYUFBS3NDLFdBQUwsQ0FBaUJ0QyxHQUFqQjtBQUNELE9BSkQ7QUFLRCxLQVZhO0FBV2RtRSxJQUFBQSxNQUFNLEVBQUU7QUFDTjdCLE1BQUFBLFdBQVcsRUFBRSxVQUFTdEMsR0FBVCxFQUFjO0FBQ3pCLGNBQU0yUCxVQUFVLEdBQUcsS0FBS25XLE9BQXhCO0FBQ0EsYUFBS0EsT0FBTCxHQUFlYSxTQUFmO0FBRUEsYUFBSytGLFlBQUwsQ0FBa0IsS0FBS2YsS0FBTCxDQUFXdUIsS0FBN0I7QUFFQStPLFFBQUFBLFVBQVUsQ0FBQzlOLFFBQVgsQ0FBb0I3QixHQUFwQjtBQUNELE9BUks7QUFTTm9CLE1BQUFBLE9BQU8sRUFBRSxVQUFTQSxPQUFULEVBQWtCO0FBQ3pCLGNBQU1ZLE9BQU8sR0FBRyxJQUFJa08sOEJBQUosQ0FBMEIsSUFBMUIsRUFBZ0MsS0FBSzFXLE9BQXJDLENBQWhCO0FBQ0EsY0FBTTZVLGlCQUFpQixHQUFHLEtBQUt0TSx1QkFBTCxDQUE2QlgsT0FBN0IsRUFBc0NZLE9BQXRDLENBQTFCO0FBRUFxTSxRQUFBQSxpQkFBaUIsQ0FBQ2xPLElBQWxCLENBQXVCLEtBQXZCLEVBQThCLE1BQU07QUFDbEM7QUFDQTtBQUNBLGNBQUk2QixPQUFPLENBQUNtTyxpQkFBWixFQUErQjtBQUM3QixpQkFBSzlNLGdCQUFMO0FBRUEsa0JBQU1zTSxVQUFVLEdBQUcsS0FBS25XLE9BQXhCO0FBQ0EsaUJBQUtBLE9BQUwsR0FBZWEsU0FBZjtBQUNBLGlCQUFLK0YsWUFBTCxDQUFrQixLQUFLZixLQUFMLENBQVd5TSxTQUE3Qjs7QUFFQSxnQkFBSTZELFVBQVUsQ0FBQ3BOLEtBQVgsSUFBb0JvTixVQUFVLENBQUNwTixLQUFYLFlBQTRCWCxvQkFBaEQsSUFBZ0UrTixVQUFVLENBQUNwTixLQUFYLENBQWlCOEIsSUFBakIsS0FBMEIsVUFBOUYsRUFBMEc7QUFDeEdzTCxjQUFBQSxVQUFVLENBQUM5TixRQUFYLENBQW9COE4sVUFBVSxDQUFDcE4sS0FBL0I7QUFDRCxhQUZELE1BRU87QUFDTG9OLGNBQUFBLFVBQVUsQ0FBQzlOLFFBQVgsQ0FBb0IsSUFBSUQsb0JBQUosQ0FBaUIsV0FBakIsRUFBOEIsU0FBOUIsQ0FBcEI7QUFDRDtBQUNGO0FBQ0YsU0FoQkQ7QUFpQkQ7QUE5Qks7QUFYTSxHQWxmVztBQThoQjNCaEIsRUFBQUEsS0FBSyxFQUFFO0FBQ0xkLElBQUFBLElBQUksRUFBRSxPQUREO0FBRUxpRSxJQUFBQSxLQUFLLEVBQUUsWUFBVztBQUNoQixXQUFLMUMsaUJBQUwsQ0FBdUJ0SixZQUFZLENBQUNDLE1BQXBDO0FBQ0QsS0FKSTtBQUtMbU0sSUFBQUEsTUFBTSxFQUFFO0FBQ04xSSxNQUFBQSxjQUFjLEVBQUUsWUFBVyxDQUN6QjtBQUNELE9BSEs7QUFJTjJGLE1BQUFBLE9BQU8sRUFBRSxZQUFXLENBQ2xCO0FBQ0QsT0FOSztBQU9Oa0IsTUFBQUEsV0FBVyxFQUFFLFlBQVcsQ0FDdEI7QUFDRDtBQVRLO0FBTEg7QUE5aEJvQixDQUE3QiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBjcnlwdG8gZnJvbSAnY3J5cHRvJztcbmltcG9ydCBvcyBmcm9tICdvcyc7XG5pbXBvcnQgeyBTb2NrZXQgfSBmcm9tICduZXQnO1xuXG5pbXBvcnQgY29uc3RhbnRzIGZyb20gJ2NvbnN0YW50cyc7XG5pbXBvcnQgeyBjcmVhdGVTZWN1cmVDb250ZXh0LCBTZWN1cmVDb250ZXh0LCBTZWN1cmVDb250ZXh0T3B0aW9ucyB9IGZyb20gJ3Rscyc7XG5cbmltcG9ydCB7IFJlYWRhYmxlIH0gZnJvbSAnc3RyZWFtJztcblxuaW1wb3J0IHtcbiAgRGVmYXVsdEF6dXJlQ3JlZGVudGlhbCxcbiAgQ2xpZW50U2VjcmV0Q3JlZGVudGlhbCxcbiAgTWFuYWdlZElkZW50aXR5Q3JlZGVudGlhbCxcbiAgVG9rZW5DcmVkZW50aWFsLFxuICBVc2VybmFtZVBhc3N3b3JkQ3JlZGVudGlhbCxcbn0gZnJvbSAnQGF6dXJlL2lkZW50aXR5JztcblxuaW1wb3J0IEJ1bGtMb2FkLCB7IE9wdGlvbnMgYXMgQnVsa0xvYWRPcHRpb25zLCBDYWxsYmFjayBhcyBCdWxrTG9hZENhbGxiYWNrIH0gZnJvbSAnLi9idWxrLWxvYWQnO1xuaW1wb3J0IERlYnVnIGZyb20gJy4vZGVidWcnO1xuaW1wb3J0IHsgRXZlbnRFbWl0dGVyIH0gZnJvbSAnZXZlbnRzJztcbmltcG9ydCB7IEluc3RhbmNlTG9va3VwIH0gZnJvbSAnLi9pbnN0YW5jZS1sb29rdXAnO1xuaW1wb3J0IHsgVHJhbnNpZW50RXJyb3JMb29rdXAgfSBmcm9tICcuL3RyYW5zaWVudC1lcnJvci1sb29rdXAnO1xuaW1wb3J0IHsgVFlQRSB9IGZyb20gJy4vcGFja2V0JztcbmltcG9ydCBQcmVsb2dpblBheWxvYWQgZnJvbSAnLi9wcmVsb2dpbi1wYXlsb2FkJztcbmltcG9ydCBMb2dpbjdQYXlsb2FkIGZyb20gJy4vbG9naW43LXBheWxvYWQnO1xuaW1wb3J0IE5UTE1SZXNwb25zZVBheWxvYWQgZnJvbSAnLi9udGxtLXBheWxvYWQnO1xuaW1wb3J0IFJlcXVlc3QgZnJvbSAnLi9yZXF1ZXN0JztcbmltcG9ydCBScGNSZXF1ZXN0UGF5bG9hZCBmcm9tICcuL3JwY3JlcXVlc3QtcGF5bG9hZCc7XG5pbXBvcnQgU3FsQmF0Y2hQYXlsb2FkIGZyb20gJy4vc3FsYmF0Y2gtcGF5bG9hZCc7XG5pbXBvcnQgTWVzc2FnZUlPIGZyb20gJy4vbWVzc2FnZS1pbyc7XG5pbXBvcnQgeyBQYXJzZXIgYXMgVG9rZW5TdHJlYW1QYXJzZXIgfSBmcm9tICcuL3Rva2VuL3Rva2VuLXN0cmVhbS1wYXJzZXInO1xuaW1wb3J0IHsgVHJhbnNhY3Rpb24sIElTT0xBVElPTl9MRVZFTCwgYXNzZXJ0VmFsaWRJc29sYXRpb25MZXZlbCB9IGZyb20gJy4vdHJhbnNhY3Rpb24nO1xuaW1wb3J0IHsgQ29ubmVjdGlvbkVycm9yLCBSZXF1ZXN0RXJyb3IgfSBmcm9tICcuL2Vycm9ycyc7XG5pbXBvcnQgeyBDb25uZWN0b3IgfSBmcm9tICcuL2Nvbm5lY3Rvcic7XG5pbXBvcnQgeyBuYW1lIGFzIGxpYnJhcnlOYW1lIH0gZnJvbSAnLi9saWJyYXJ5JztcbmltcG9ydCB7IHZlcnNpb25zIH0gZnJvbSAnLi90ZHMtdmVyc2lvbnMnO1xuaW1wb3J0IE1lc3NhZ2UgZnJvbSAnLi9tZXNzYWdlJztcbmltcG9ydCB7IE1ldGFkYXRhIH0gZnJvbSAnLi9tZXRhZGF0YS1wYXJzZXInO1xuaW1wb3J0IHsgY3JlYXRlTlRMTVJlcXVlc3QgfSBmcm9tICcuL250bG0nO1xuaW1wb3J0IHsgQ29sdW1uRW5jcnlwdGlvbkF6dXJlS2V5VmF1bHRQcm92aWRlciB9IGZyb20gJy4vYWx3YXlzLWVuY3J5cHRlZC9rZXlzdG9yZS1wcm92aWRlci1henVyZS1rZXktdmF1bHQnO1xuXG5pbXBvcnQgeyBBYm9ydENvbnRyb2xsZXIsIEFib3J0U2lnbmFsIH0gZnJvbSAnbm9kZS1hYm9ydC1jb250cm9sbGVyJztcbmltcG9ydCB7IFBhcmFtZXRlciwgVFlQRVMgfSBmcm9tICcuL2RhdGEtdHlwZSc7XG5pbXBvcnQgeyBCdWxrTG9hZFBheWxvYWQgfSBmcm9tICcuL2J1bGstbG9hZC1wYXlsb2FkJztcbmltcG9ydCB7IENvbGxhdGlvbiB9IGZyb20gJy4vY29sbGF0aW9uJztcblxuaW1wb3J0IEFnZ3JlZ2F0ZUVycm9yIGZyb20gJ2VzLWFnZ3JlZ2F0ZS1lcnJvcic7XG5pbXBvcnQgeyB2ZXJzaW9uIH0gZnJvbSAnLi4vcGFja2FnZS5qc29uJztcbmltcG9ydCB7IFVSTCB9IGZyb20gJ3VybCc7XG5pbXBvcnQgeyBBdHRlbnRpb25Ub2tlbkhhbmRsZXIsIEluaXRpYWxTcWxUb2tlbkhhbmRsZXIsIExvZ2luN1Rva2VuSGFuZGxlciwgUmVxdWVzdFRva2VuSGFuZGxlciwgVG9rZW5IYW5kbGVyIH0gZnJvbSAnLi90b2tlbi9oYW5kbGVyJztcblxubGV0IHRydXN0U2VydmVyV2FybmluZ0VtaXR0ZWQgPSBmYWxzZTtcblxuY29uc3QgZW1pdFRydXN0U2VydmVyQ2VydGlmaWNhdGVXYXJuaW5nID0gKCkgPT4ge1xuICBpZiAoIXRydXN0U2VydmVyV2FybmluZ0VtaXR0ZWQpIHtcbiAgICB0cnVzdFNlcnZlcldhcm5pbmdFbWl0dGVkID0gdHJ1ZTtcbiAgICBwcm9jZXNzLmVtaXRXYXJuaW5nKCdgY29uZmlnLm9wdGlvbnMudHJ1c3RTZXJ2ZXJDZXJ0aWZpY2F0ZWAgd2lsbCBkZWZhdWx0IHRvIGZhbHNlIGluIHRoZSBmdXR1cmUuIFRvIHNpbGVuY2UgdGhpcyBtZXNzYWdlLCBzcGVjaWZ5IGEgdmFsdWUgZXhwbGljaXRseSBpbiB0aGUgY29uZmlnIG9wdGlvbnMnKTtcbiAgfVxufTtcblxubGV0IGRvbWFpblJlbmFtZVRvVGVuYW50SWRXYXJuaW5nRW1pdHRlZCA9IGZhbHNlO1xuY29uc3QgZW1pdERvbWFpblJlbmFtZVRvVGVuYW50SWRXYXJuaW5nID0gKCkgPT4ge1xuICBpZiAoIWRvbWFpblJlbmFtZVRvVGVuYW50SWRXYXJuaW5nRW1pdHRlZCkge1xuICAgIGRvbWFpblJlbmFtZVRvVGVuYW50SWRXYXJuaW5nRW1pdHRlZCA9IHRydWU7XG4gICAgcHJvY2Vzcy5lbWl0V2FybmluZygnYFdoZW4gdXNpbmcgYXV0aGVudGljYXRpb24gdHlwZSBgYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1wYXNzd29yZGAsJyArXG4gICAgJyBjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy5kb21haW5gIHdpbGwgYmUgcmVuYW1lZCB0byBjb25maWcuYXV0aGVudGljYXRpb25zLm9wdGlvbnMudGVuYW50SWRgJyArXG4gICAgJyBpbiB0aGUgZnV0dXJlLiBSZW5hbWUgYGRvbWFpbmAgdG8gYHRlbmFudElkYCB0byBzaWxlbmNlIHRoaXMgbWVzc2FnZS4nKTtcbiAgfVxufTtcblxudHlwZSBCZWdpblRyYW5zYWN0aW9uQ2FsbGJhY2sgPVxuICAvKipcbiAgICogVGhlIGNhbGxiYWNrIGlzIGNhbGxlZCB3aGVuIHRoZSByZXF1ZXN0IHRvIHN0YXJ0IHRoZSB0cmFuc2FjdGlvbiBoYXMgY29tcGxldGVkLFxuICAgKiBlaXRoZXIgc3VjY2Vzc2Z1bGx5IG9yIHdpdGggYW4gZXJyb3IuXG4gICAqIElmIGFuIGVycm9yIG9jY3VycmVkIHRoZW4gYGVycmAgd2lsbCBkZXNjcmliZSB0aGUgZXJyb3IuXG4gICAqXG4gICAqIEFzIG9ubHkgb25lIHJlcXVlc3QgYXQgYSB0aW1lIG1heSBiZSBleGVjdXRlZCBvbiBhIGNvbm5lY3Rpb24sIGFub3RoZXIgcmVxdWVzdCBzaG91bGQgbm90XG4gICAqIGJlIGluaXRpYXRlZCB1bnRpbCB0aGlzIGNhbGxiYWNrIGlzIGNhbGxlZC5cbiAgICpcbiAgICogQHBhcmFtIGVyciBJZiBhbiBlcnJvciBvY2N1cnJlZCwgYW4gW1tFcnJvcl1dIG9iamVjdCB3aXRoIGRldGFpbHMgb2YgdGhlIGVycm9yLlxuICAgKiBAcGFyYW0gdHJhbnNhY3Rpb25EZXNjcmlwdG9yIEEgQnVmZmVyIHRoYXQgZGVzY3JpYmUgdGhlIHRyYW5zYWN0aW9uXG4gICAqL1xuICAoZXJyOiBFcnJvciB8IG51bGwgfCB1bmRlZmluZWQsIHRyYW5zYWN0aW9uRGVzY3JpcHRvcj86IEJ1ZmZlcikgPT4gdm9pZFxuXG50eXBlIFNhdmVUcmFuc2FjdGlvbkNhbGxiYWNrID1cbiAgLyoqXG4gICAqIFRoZSBjYWxsYmFjayBpcyBjYWxsZWQgd2hlbiB0aGUgcmVxdWVzdCB0byBzZXQgYSBzYXZlcG9pbnQgd2l0aGluIHRoZVxuICAgKiB0cmFuc2FjdGlvbiBoYXMgY29tcGxldGVkLCBlaXRoZXIgc3VjY2Vzc2Z1bGx5IG9yIHdpdGggYW4gZXJyb3IuXG4gICAqIElmIGFuIGVycm9yIG9jY3VycmVkIHRoZW4gYGVycmAgd2lsbCBkZXNjcmliZSB0aGUgZXJyb3IuXG4gICAqXG4gICAqIEFzIG9ubHkgb25lIHJlcXVlc3QgYXQgYSB0aW1lIG1heSBiZSBleGVjdXRlZCBvbiBhIGNvbm5lY3Rpb24sIGFub3RoZXIgcmVxdWVzdCBzaG91bGQgbm90XG4gICAqIGJlIGluaXRpYXRlZCB1bnRpbCB0aGlzIGNhbGxiYWNrIGlzIGNhbGxlZC5cbiAgICpcbiAgICogQHBhcmFtIGVyciBJZiBhbiBlcnJvciBvY2N1cnJlZCwgYW4gW1tFcnJvcl1dIG9iamVjdCB3aXRoIGRldGFpbHMgb2YgdGhlIGVycm9yLlxuICAgKi9cbiAgKGVycjogRXJyb3IgfCBudWxsIHwgdW5kZWZpbmVkKSA9PiB2b2lkO1xuXG50eXBlIENvbW1pdFRyYW5zYWN0aW9uQ2FsbGJhY2sgPVxuICAvKipcbiAgICogVGhlIGNhbGxiYWNrIGlzIGNhbGxlZCB3aGVuIHRoZSByZXF1ZXN0IHRvIGNvbW1pdCB0aGUgdHJhbnNhY3Rpb24gaGFzIGNvbXBsZXRlZCxcbiAgICogZWl0aGVyIHN1Y2Nlc3NmdWxseSBvciB3aXRoIGFuIGVycm9yLlxuICAgKiBJZiBhbiBlcnJvciBvY2N1cnJlZCB0aGVuIGBlcnJgIHdpbGwgZGVzY3JpYmUgdGhlIGVycm9yLlxuICAgKlxuICAgKiBBcyBvbmx5IG9uZSByZXF1ZXN0IGF0IGEgdGltZSBtYXkgYmUgZXhlY3V0ZWQgb24gYSBjb25uZWN0aW9uLCBhbm90aGVyIHJlcXVlc3Qgc2hvdWxkIG5vdFxuICAgKiBiZSBpbml0aWF0ZWQgdW50aWwgdGhpcyBjYWxsYmFjayBpcyBjYWxsZWQuXG4gICAqXG4gICAqIEBwYXJhbSBlcnIgSWYgYW4gZXJyb3Igb2NjdXJyZWQsIGFuIFtbRXJyb3JdXSBvYmplY3Qgd2l0aCBkZXRhaWxzIG9mIHRoZSBlcnJvci5cbiAgICovXG4gIChlcnI6IEVycm9yIHwgbnVsbCB8IHVuZGVmaW5lZCkgPT4gdm9pZDtcblxudHlwZSBSb2xsYmFja1RyYW5zYWN0aW9uQ2FsbGJhY2sgPVxuICAvKipcbiAgICogVGhlIGNhbGxiYWNrIGlzIGNhbGxlZCB3aGVuIHRoZSByZXF1ZXN0IHRvIHJvbGxiYWNrIHRoZSB0cmFuc2FjdGlvbiBoYXNcbiAgICogY29tcGxldGVkLCBlaXRoZXIgc3VjY2Vzc2Z1bGx5IG9yIHdpdGggYW4gZXJyb3IuXG4gICAqIElmIGFuIGVycm9yIG9jY3VycmVkIHRoZW4gZXJyIHdpbGwgZGVzY3JpYmUgdGhlIGVycm9yLlxuICAgKlxuICAgKiBBcyBvbmx5IG9uZSByZXF1ZXN0IGF0IGEgdGltZSBtYXkgYmUgZXhlY3V0ZWQgb24gYSBjb25uZWN0aW9uLCBhbm90aGVyIHJlcXVlc3Qgc2hvdWxkIG5vdFxuICAgKiBiZSBpbml0aWF0ZWQgdW50aWwgdGhpcyBjYWxsYmFjayBpcyBjYWxsZWQuXG4gICAqXG4gICAqIEBwYXJhbSBlcnIgSWYgYW4gZXJyb3Igb2NjdXJyZWQsIGFuIFtbRXJyb3JdXSBvYmplY3Qgd2l0aCBkZXRhaWxzIG9mIHRoZSBlcnJvci5cbiAgICovXG4gIChlcnI6IEVycm9yIHwgbnVsbCB8IHVuZGVmaW5lZCkgPT4gdm9pZDtcblxudHlwZSBSZXNldENhbGxiYWNrID1cbiAgLyoqXG4gICAqIFRoZSBjYWxsYmFjayBpcyBjYWxsZWQgd2hlbiB0aGUgY29ubmVjdGlvbiByZXNldCBoYXMgY29tcGxldGVkLFxuICAgKiBlaXRoZXIgc3VjY2Vzc2Z1bGx5IG9yIHdpdGggYW4gZXJyb3IuXG4gICAqXG4gICAqIElmIGFuIGVycm9yIG9jY3VycmVkIHRoZW4gYGVycmAgd2lsbCBkZXNjcmliZSB0aGUgZXJyb3IuXG4gICAqXG4gICAqIEFzIG9ubHkgb25lIHJlcXVlc3QgYXQgYSB0aW1lIG1heSBiZSBleGVjdXRlZCBvbiBhIGNvbm5lY3Rpb24sIGFub3RoZXJcbiAgICogcmVxdWVzdCBzaG91bGQgbm90IGJlIGluaXRpYXRlZCB1bnRpbCB0aGlzIGNhbGxiYWNrIGlzIGNhbGxlZFxuICAgKlxuICAgKiBAcGFyYW0gZXJyIElmIGFuIGVycm9yIG9jY3VycmVkLCBhbiBbW0Vycm9yXV0gb2JqZWN0IHdpdGggZGV0YWlscyBvZiB0aGUgZXJyb3IuXG4gICAqL1xuICAoZXJyOiBFcnJvciB8IG51bGwgfCB1bmRlZmluZWQpID0+IHZvaWQ7XG5cbi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBAdHlwZXNjcmlwdC1lc2xpbnQvbm8tdW51c2VkLXZhcnNcbnR5cGUgVHJhbnNhY3Rpb25DYWxsYmFjazxUIGV4dGVuZHMgKGVycjogRXJyb3IgfCBudWxsIHwgdW5kZWZpbmVkLCAuLi5hcmdzOiBhbnlbXSkgPT4gdm9pZD4gPVxuICAvKipcbiAgICogVGhlIGNhbGxiYWNrIGlzIGNhbGxlZCB3aGVuIHRoZSByZXF1ZXN0IHRvIHN0YXJ0IGEgdHJhbnNhY3Rpb24gKG9yIGNyZWF0ZSBhIHNhdmVwb2ludCwgaW5cbiAgICogdGhlIGNhc2Ugb2YgYSBuZXN0ZWQgdHJhbnNhY3Rpb24pIGhhcyBjb21wbGV0ZWQsIGVpdGhlciBzdWNjZXNzZnVsbHkgb3Igd2l0aCBhbiBlcnJvci5cbiAgICogSWYgYW4gZXJyb3Igb2NjdXJyZWQsIHRoZW4gYGVycmAgd2lsbCBkZXNjcmliZSB0aGUgZXJyb3IuXG4gICAqIElmIG5vIGVycm9yIG9jY3VycmVkLCB0aGUgY2FsbGJhY2sgc2hvdWxkIHBlcmZvcm0gaXRzIHdvcmsgYW5kIGV2ZW50dWFsbHkgY2FsbFxuICAgKiBgZG9uZWAgd2l0aCBhbiBlcnJvciBvciBudWxsICh0byB0cmlnZ2VyIGEgdHJhbnNhY3Rpb24gcm9sbGJhY2sgb3IgYVxuICAgKiB0cmFuc2FjdGlvbiBjb21taXQpIGFuZCBhbiBhZGRpdGlvbmFsIGNvbXBsZXRpb24gY2FsbGJhY2sgdGhhdCB3aWxsIGJlIGNhbGxlZCB3aGVuIHRoZSByZXF1ZXN0XG4gICAqIHRvIHJvbGxiYWNrIG9yIGNvbW1pdCB0aGUgY3VycmVudCB0cmFuc2FjdGlvbiBoYXMgY29tcGxldGVkLCBlaXRoZXIgc3VjY2Vzc2Z1bGx5IG9yIHdpdGggYW4gZXJyb3IuXG4gICAqIEFkZGl0aW9uYWwgYXJndW1lbnRzIGdpdmVuIHRvIGBkb25lYCB3aWxsIGJlIHBhc3NlZCB0aHJvdWdoIHRvIHRoaXMgY2FsbGJhY2suXG4gICAqXG4gICAqIEFzIG9ubHkgb25lIHJlcXVlc3QgYXQgYSB0aW1lIG1heSBiZSBleGVjdXRlZCBvbiBhIGNvbm5lY3Rpb24sIGFub3RoZXIgcmVxdWVzdCBzaG91bGQgbm90XG4gICAqIGJlIGluaXRpYXRlZCB1bnRpbCB0aGUgY29tcGxldGlvbiBjYWxsYmFjayBpcyBjYWxsZWQuXG4gICAqXG4gICAqIEBwYXJhbSBlcnIgSWYgYW4gZXJyb3Igb2NjdXJyZWQsIGFuIFtbRXJyb3JdXSBvYmplY3Qgd2l0aCBkZXRhaWxzIG9mIHRoZSBlcnJvci5cbiAgICogQHBhcmFtIHR4RG9uZSBJZiBubyBlcnJvciBvY2N1cnJlZCwgYSBmdW5jdGlvbiB0byBiZSBjYWxsZWQgdG8gY29tbWl0IG9yIHJvbGxiYWNrIHRoZSB0cmFuc2FjdGlvbi5cbiAgICovXG4gIChlcnI6IEVycm9yIHwgbnVsbCB8IHVuZGVmaW5lZCwgdHhEb25lPzogVHJhbnNhY3Rpb25Eb25lPFQ+KSA9PiB2b2lkO1xuXG50eXBlIFRyYW5zYWN0aW9uRG9uZUNhbGxiYWNrID0gKGVycjogRXJyb3IgfCBudWxsIHwgdW5kZWZpbmVkLCAuLi5hcmdzOiBhbnlbXSkgPT4gdm9pZDtcbnR5cGUgQ2FsbGJhY2tQYXJhbWV0ZXJzPFQgZXh0ZW5kcyAoZXJyOiBFcnJvciB8IG51bGwgfCB1bmRlZmluZWQsIC4uLmFyZ3M6IGFueVtdKSA9PiBhbnk+ID0gVCBleHRlbmRzIChlcnI6IEVycm9yIHwgbnVsbCB8IHVuZGVmaW5lZCwgLi4uYXJnczogaW5mZXIgUCkgPT4gYW55ID8gUCA6IG5ldmVyO1xuXG50eXBlIFRyYW5zYWN0aW9uRG9uZTxUIGV4dGVuZHMgKGVycjogRXJyb3IgfCBudWxsIHwgdW5kZWZpbmVkLCAuLi5hcmdzOiBhbnlbXSkgPT4gdm9pZD4gPVxuICAvKipcbiAgICogSWYgbm8gZXJyb3Igb2NjdXJyZWQsIGEgZnVuY3Rpb24gdG8gYmUgY2FsbGVkIHRvIGNvbW1pdCBvciByb2xsYmFjayB0aGUgdHJhbnNhY3Rpb24uXG4gICAqXG4gICAqIEBwYXJhbSBlcnIgSWYgYW4gZXJyIG9jY3VycmVkLCBhIHN0cmluZyB3aXRoIGRldGFpbHMgb2YgdGhlIGVycm9yLlxuICAgKi9cbiAgKGVycjogRXJyb3IgfCBudWxsIHwgdW5kZWZpbmVkLCBkb25lOiBULCAuLi5hcmdzOiBDYWxsYmFja1BhcmFtZXRlcnM8VD4pID0+IHZvaWQ7XG5cbi8qKlxuICogQHByaXZhdGVcbiAqL1xuY29uc3QgS0VFUF9BTElWRV9JTklUSUFMX0RFTEFZID0gMzAgKiAxMDAwO1xuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5jb25zdCBERUZBVUxUX0NPTk5FQ1RfVElNRU9VVCA9IDE1ICogMTAwMDtcbi8qKlxuICogQHByaXZhdGVcbiAqL1xuY29uc3QgREVGQVVMVF9DTElFTlRfUkVRVUVTVF9USU1FT1VUID0gMTUgKiAxMDAwO1xuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5jb25zdCBERUZBVUxUX0NBTkNFTF9USU1FT1VUID0gNSAqIDEwMDA7XG4vKipcbiAqIEBwcml2YXRlXG4gKi9cbmNvbnN0IERFRkFVTFRfQ09OTkVDVF9SRVRSWV9JTlRFUlZBTCA9IDUwMDtcbi8qKlxuICogQHByaXZhdGVcbiAqL1xuY29uc3QgREVGQVVMVF9QQUNLRVRfU0laRSA9IDQgKiAxMDI0O1xuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5jb25zdCBERUZBVUxUX1RFWFRTSVpFID0gMjE0NzQ4MzY0Nztcbi8qKlxuICogQHByaXZhdGVcbiAqL1xuY29uc3QgREVGQVVMVF9EQVRFRklSU1QgPSA3O1xuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5jb25zdCBERUZBVUxUX1BPUlQgPSAxNDMzO1xuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5jb25zdCBERUZBVUxUX1REU19WRVJTSU9OID0gJzdfNCc7XG4vKipcbiAqIEBwcml2YXRlXG4gKi9cbmNvbnN0IERFRkFVTFRfTEFOR1VBR0UgPSAndXNfZW5nbGlzaCc7XG4vKipcbiAqIEBwcml2YXRlXG4gKi9cbmNvbnN0IERFRkFVTFRfREFURUZPUk1BVCA9ICdtZHknO1xuXG5pbnRlcmZhY2UgQXp1cmVBY3RpdmVEaXJlY3RvcnlNc2lBcHBTZXJ2aWNlQXV0aGVudGljYXRpb24ge1xuICB0eXBlOiAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1tc2ktYXBwLXNlcnZpY2UnO1xuICBvcHRpb25zOiB7XG4gICAgLyoqXG4gICAgICogSWYgeW91IHVzZXIgd2FudCB0byBjb25uZWN0IHRvIGFuIEF6dXJlIGFwcCBzZXJ2aWNlIHVzaW5nIGEgc3BlY2lmaWMgY2xpZW50IGFjY291bnRcbiAgICAgKiB0aGV5IG5lZWQgdG8gcHJvdmlkZSBgY2xpZW50SWRgIGFzc2NvaWF0ZSB0byB0aGVpciBjcmVhdGVkIGlkbmV0aXR5LlxuICAgICAqXG4gICAgICogVGhpcyBpcyBvcHRpb25hbCBmb3IgcmV0cmlldmUgdG9rZW4gZnJvbSBhenVyZSB3ZWIgYXBwIHNlcnZpY2VcbiAgICAgKi9cbiAgICBjbGllbnRJZD86IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIEF6dXJlQWN0aXZlRGlyZWN0b3J5TXNpVm1BdXRoZW50aWNhdGlvbiB7XG4gIHR5cGU6ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS12bSc7XG4gIG9wdGlvbnM6IHtcbiAgICAvKipcbiAgICAgKiBJZiB5b3Ugd2FudCB0byBjb25uZWN0IHVzaW5nIGEgc3BlY2lmaWMgY2xpZW50IGFjY291bnRcbiAgICAgKiB0aGV5IG5lZWQgdG8gcHJvdmlkZSBgY2xpZW50SWRgIGFzc29jaWF0ZWQgdG8gdGhlaXIgY3JlYXRlZCBpZGVudGl0eS5cbiAgICAgKlxuICAgICAqIFRoaXMgaXMgb3B0aW9uYWwgZm9yIHJldHJpZXZlIGEgdG9rZW5cbiAgICAgKi9cbiAgICBjbGllbnRJZD86IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIEF6dXJlQWN0aXZlRGlyZWN0b3J5RGVmYXVsdEF1dGhlbnRpY2F0aW9uIHtcbiAgdHlwZTogJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktZGVmYXVsdCc7XG4gIG9wdGlvbnM6IHtcbiAgICAvKipcbiAgICAgKiBJZiB5b3Ugd2FudCB0byBjb25uZWN0IHVzaW5nIGEgc3BlY2lmaWMgY2xpZW50IGFjY291bnRcbiAgICAgKiB0aGV5IG5lZWQgdG8gcHJvdmlkZSBgY2xpZW50SWRgIGFzc29jaWF0ZWQgdG8gdGhlaXIgY3JlYXRlZCBpZGVudGl0eS5cbiAgICAgKlxuICAgICAqIFRoaXMgaXMgb3B0aW9uYWwgZm9yIHJldHJpZXZpbmcgYSB0b2tlblxuICAgICAqL1xuICAgIGNsaWVudElkPzogc3RyaW5nO1xuICB9O1xufVxuXG5cbmludGVyZmFjZSBBenVyZUFjdGl2ZURpcmVjdG9yeUFjY2Vzc1Rva2VuQXV0aGVudGljYXRpb24ge1xuICB0eXBlOiAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1hY2Nlc3MtdG9rZW4nO1xuICBvcHRpb25zOiB7XG4gICAgLyoqXG4gICAgICogQSB1c2VyIG5lZWQgdG8gcHJvdmlkZSBgdG9rZW5gIHdoaWNoIHRoZXkgcmV0cml2ZWQgZWxzZSB3aGVyZVxuICAgICAqIHRvIGZvcm1pbmcgdGhlIGNvbm5lY3Rpb24uXG4gICAgICovXG4gICAgdG9rZW46IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIEF6dXJlQWN0aXZlRGlyZWN0b3J5UGFzc3dvcmRBdXRoZW50aWNhdGlvbiB7XG4gIHR5cGU6ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXBhc3N3b3JkJztcbiAgb3B0aW9uczoge1xuICAgIC8qKlxuICAgICAqIEEgdXNlciBuZWVkIHRvIHByb3ZpZGUgYHVzZXJOYW1lYCBhc3Njb2lhdGUgdG8gdGhlaXIgYWNjb3VudC5cbiAgICAgKi9cbiAgICB1c2VyTmFtZTogc3RyaW5nO1xuXG4gICAgLyoqXG4gICAgICogQSB1c2VyIG5lZWQgdG8gcHJvdmlkZSBgcGFzc3dvcmRgIGFzc2NvaWF0ZSB0byB0aGVpciBhY2NvdW50LlxuICAgICAqL1xuICAgIHBhc3N3b3JkOiBzdHJpbmc7XG5cbiAgICAvKipcbiAgICAgKiBBIGNsaWVudCBpZCB0byB1c2UuXG4gICAgICovXG4gICAgY2xpZW50SWQ6IHN0cmluZztcblxuICAgIC8qKlxuICAgICAqIE9wdGlvbmFsIHBhcmFtZXRlciBmb3Igc3BlY2lmaWMgQXp1cmUgdGVuYW50IElEXG4gICAgICovXG4gICAgdGVuYW50SWQ6IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIEF6dXJlQWN0aXZlRGlyZWN0b3J5U2VydmljZVByaW5jaXBhbFNlY3JldCB7XG4gIHR5cGU6ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXNlcnZpY2UtcHJpbmNpcGFsLXNlY3JldCc7XG4gIG9wdGlvbnM6IHtcbiAgICAvKipcbiAgICAgKiBBcHBsaWNhdGlvbiAoYGNsaWVudGApIElEIGZyb20geW91ciByZWdpc3RlcmVkIEF6dXJlIGFwcGxpY2F0aW9uXG4gICAgICovXG4gICAgY2xpZW50SWQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgY3JlYXRlZCBgY2xpZW50IHNlY3JldGAgZm9yIHRoaXMgcmVnaXN0ZXJlZCBBenVyZSBhcHBsaWNhdGlvblxuICAgICAqL1xuICAgIGNsaWVudFNlY3JldDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIERpcmVjdG9yeSAoYHRlbmFudGApIElEIGZyb20geW91ciByZWdpc3RlcmVkIEF6dXJlIGFwcGxpY2F0aW9uXG4gICAgICovXG4gICAgdGVuYW50SWQ6IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIE50bG1BdXRoZW50aWNhdGlvbiB7XG4gIHR5cGU6ICdudGxtJztcbiAgb3B0aW9uczoge1xuICAgIC8qKlxuICAgICAqIFVzZXIgbmFtZSBmcm9tIHlvdXIgd2luZG93cyBhY2NvdW50LlxuICAgICAqL1xuICAgIHVzZXJOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUGFzc3dvcmQgZnJvbSB5b3VyIHdpbmRvd3MgYWNjb3VudC5cbiAgICAgKi9cbiAgICBwYXNzd29yZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIE9uY2UgeW91IHNldCBkb21haW4gZm9yIG50bG0gYXV0aGVudGljYXRpb24gdHlwZSwgZHJpdmVyIHdpbGwgY29ubmVjdCB0byBTUUwgU2VydmVyIHVzaW5nIGRvbWFpbiBsb2dpbi5cbiAgICAgKlxuICAgICAqIFRoaXMgaXMgbmVjZXNzYXJ5IGZvciBmb3JtaW5nIGEgY29ubmVjdGlvbiB1c2luZyBudGxtIHR5cGVcbiAgICAgKi9cbiAgICBkb21haW46IHN0cmluZztcbiAgfTtcbn1cblxuaW50ZXJmYWNlIERlZmF1bHRBdXRoZW50aWNhdGlvbiB7XG4gIHR5cGU6ICdkZWZhdWx0JztcbiAgb3B0aW9uczoge1xuICAgIC8qKlxuICAgICAqIFVzZXIgbmFtZSB0byB1c2UgZm9yIHNxbCBzZXJ2ZXIgbG9naW4uXG4gICAgICovXG4gICAgdXNlck5hbWU/OiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gICAgLyoqXG4gICAgICogUGFzc3dvcmQgdG8gdXNlIGZvciBzcWwgc2VydmVyIGxvZ2luLlxuICAgICAqL1xuICAgIHBhc3N3b3JkPzogc3RyaW5nIHwgdW5kZWZpbmVkO1xuICB9O1xufVxuXG5pbnRlcmZhY2UgRXJyb3JXaXRoQ29kZSBleHRlbmRzIEVycm9yIHtcbiAgY29kZT86IHN0cmluZztcbn1cblxuaW50ZXJmYWNlIEludGVybmFsQ29ubmVjdGlvbkNvbmZpZyB7XG4gIHNlcnZlcjogc3RyaW5nO1xuICBhdXRoZW50aWNhdGlvbjogRGVmYXVsdEF1dGhlbnRpY2F0aW9uIHwgTnRsbUF1dGhlbnRpY2F0aW9uIHwgQXp1cmVBY3RpdmVEaXJlY3RvcnlQYXNzd29yZEF1dGhlbnRpY2F0aW9uIHwgQXp1cmVBY3RpdmVEaXJlY3RvcnlNc2lBcHBTZXJ2aWNlQXV0aGVudGljYXRpb24gfCBBenVyZUFjdGl2ZURpcmVjdG9yeU1zaVZtQXV0aGVudGljYXRpb24gfCBBenVyZUFjdGl2ZURpcmVjdG9yeUFjY2Vzc1Rva2VuQXV0aGVudGljYXRpb24gfCBBenVyZUFjdGl2ZURpcmVjdG9yeVNlcnZpY2VQcmluY2lwYWxTZWNyZXQgfCBBenVyZUFjdGl2ZURpcmVjdG9yeURlZmF1bHRBdXRoZW50aWNhdGlvbjtcbiAgb3B0aW9uczogSW50ZXJuYWxDb25uZWN0aW9uT3B0aW9ucztcbn1cblxuZXhwb3J0IGludGVyZmFjZSBJbnRlcm5hbENvbm5lY3Rpb25PcHRpb25zIHtcbiAgYWJvcnRUcmFuc2FjdGlvbk9uRXJyb3I6IGJvb2xlYW47XG4gIGFwcE5hbWU6IHVuZGVmaW5lZCB8IHN0cmluZztcbiAgY2FtZWxDYXNlQ29sdW1uczogYm9vbGVhbjtcbiAgY2FuY2VsVGltZW91dDogbnVtYmVyO1xuICBjb2x1bW5FbmNyeXB0aW9uS2V5Q2FjaGVUVEw6IG51bWJlcjtcbiAgY29sdW1uRW5jcnlwdGlvblNldHRpbmc6IGJvb2xlYW47XG4gIGNvbHVtbk5hbWVSZXBsYWNlcjogdW5kZWZpbmVkIHwgKChjb2xOYW1lOiBzdHJpbmcsIGluZGV4OiBudW1iZXIsIG1ldGFkYXRhOiBNZXRhZGF0YSkgPT4gc3RyaW5nKTtcbiAgY29ubmVjdGlvblJldHJ5SW50ZXJ2YWw6IG51bWJlcjtcbiAgY29ubmVjdFRpbWVvdXQ6IG51bWJlcjtcbiAgY29ubmVjdGlvbklzb2xhdGlvbkxldmVsOiB0eXBlb2YgSVNPTEFUSU9OX0xFVkVMW2tleW9mIHR5cGVvZiBJU09MQVRJT05fTEVWRUxdO1xuICBjcnlwdG9DcmVkZW50aWFsc0RldGFpbHM6IFNlY3VyZUNvbnRleHRPcHRpb25zO1xuICBkYXRhYmFzZTogdW5kZWZpbmVkIHwgc3RyaW5nO1xuICBkYXRlZmlyc3Q6IG51bWJlcjtcbiAgZGF0ZUZvcm1hdDogc3RyaW5nO1xuICBkZWJ1Zzoge1xuICAgIGRhdGE6IGJvb2xlYW47XG4gICAgcGFja2V0OiBib29sZWFuO1xuICAgIHBheWxvYWQ6IGJvb2xlYW47XG4gICAgdG9rZW46IGJvb2xlYW47XG4gIH07XG4gIGVuYWJsZUFuc2lOdWxsOiBudWxsIHwgYm9vbGVhbjtcbiAgZW5hYmxlQW5zaU51bGxEZWZhdWx0OiBudWxsIHwgYm9vbGVhbjtcbiAgZW5hYmxlQW5zaVBhZGRpbmc6IG51bGwgfCBib29sZWFuO1xuICBlbmFibGVBbnNpV2FybmluZ3M6IG51bGwgfCBib29sZWFuO1xuICBlbmFibGVBcml0aEFib3J0OiBudWxsIHwgYm9vbGVhbjtcbiAgZW5hYmxlQ29uY2F0TnVsbFlpZWxkc051bGw6IG51bGwgfCBib29sZWFuO1xuICBlbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0OiBudWxsIHwgYm9vbGVhbjtcbiAgZW5hYmxlSW1wbGljaXRUcmFuc2FjdGlvbnM6IG51bGwgfCBib29sZWFuO1xuICBlbmFibGVOdW1lcmljUm91bmRhYm9ydDogbnVsbCB8IGJvb2xlYW47XG4gIGVuYWJsZVF1b3RlZElkZW50aWZpZXI6IG51bGwgfCBib29sZWFuO1xuICBlbmNyeXB0OiBib29sZWFuO1xuICBlbmNyeXB0aW9uS2V5U3RvcmVQcm92aWRlcnM6IEtleVN0b3JlUHJvdmlkZXJNYXAgfCB1bmRlZmluZWQ7XG4gIGZhbGxiYWNrVG9EZWZhdWx0RGI6IGJvb2xlYW47XG4gIGluc3RhbmNlTmFtZTogdW5kZWZpbmVkIHwgc3RyaW5nO1xuICBpc29sYXRpb25MZXZlbDogdHlwZW9mIElTT0xBVElPTl9MRVZFTFtrZXlvZiB0eXBlb2YgSVNPTEFUSU9OX0xFVkVMXTtcbiAgbGFuZ3VhZ2U6IHN0cmluZztcbiAgbG9jYWxBZGRyZXNzOiB1bmRlZmluZWQgfCBzdHJpbmc7XG4gIG1heFJldHJpZXNPblRyYW5zaWVudEVycm9yczogbnVtYmVyO1xuICBtdWx0aVN1Ym5ldEZhaWxvdmVyOiBib29sZWFuO1xuICBwYWNrZXRTaXplOiBudW1iZXI7XG4gIHBvcnQ6IHVuZGVmaW5lZCB8IG51bWJlcjtcbiAgcmVhZE9ubHlJbnRlbnQ6IGJvb2xlYW47XG4gIHJlcXVlc3RUaW1lb3V0OiBudW1iZXI7XG4gIHJvd0NvbGxlY3Rpb25PbkRvbmU6IGJvb2xlYW47XG4gIHJvd0NvbGxlY3Rpb25PblJlcXVlc3RDb21wbGV0aW9uOiBib29sZWFuO1xuICBzZXJ2ZXJOYW1lOiB1bmRlZmluZWQgfCBzdHJpbmc7XG4gIHNlcnZlclN1cHBvcnRzQ29sdW1uRW5jcnlwdGlvbjogYm9vbGVhbjtcbiAgdGRzVmVyc2lvbjogc3RyaW5nO1xuICB0ZXh0c2l6ZTogbnVtYmVyO1xuICB0cnVzdGVkU2VydmVyTmFtZUFFOiBzdHJpbmcgfCB1bmRlZmluZWQ7XG4gIHRydXN0U2VydmVyQ2VydGlmaWNhdGU6IGJvb2xlYW47XG4gIHVzZUNvbHVtbk5hbWVzOiBib29sZWFuO1xuICB1c2VVVEM6IGJvb2xlYW47XG4gIHdvcmtzdGF0aW9uSWQ6IHVuZGVmaW5lZCB8IHN0cmluZztcbiAgbG93ZXJDYXNlR3VpZHM6IGJvb2xlYW47XG59XG5cbmludGVyZmFjZSBLZXlTdG9yZVByb3ZpZGVyTWFwIHtcbiAgW2tleTogc3RyaW5nXTogQ29sdW1uRW5jcnlwdGlvbkF6dXJlS2V5VmF1bHRQcm92aWRlcjtcbn1cblxuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5pbnRlcmZhY2UgU3RhdGUge1xuICBuYW1lOiBzdHJpbmc7XG4gIGVudGVyPyh0aGlzOiBDb25uZWN0aW9uKTogdm9pZDtcbiAgZXhpdD8odGhpczogQ29ubmVjdGlvbiwgbmV3U3RhdGU6IFN0YXRlKTogdm9pZDtcbiAgZXZlbnRzOiB7XG4gICAgc29ja2V0RXJyb3I/KHRoaXM6IENvbm5lY3Rpb24sIGVycjogRXJyb3IpOiB2b2lkO1xuICAgIGNvbm5lY3RUaW1lb3V0Pyh0aGlzOiBDb25uZWN0aW9uKTogdm9pZDtcbiAgICBtZXNzYWdlPyh0aGlzOiBDb25uZWN0aW9uLCBtZXNzYWdlOiBNZXNzYWdlKTogdm9pZDtcbiAgICByZXRyeT8odGhpczogQ29ubmVjdGlvbik6IHZvaWQ7XG4gICAgcmVjb25uZWN0Pyh0aGlzOiBDb25uZWN0aW9uKTogdm9pZDtcbiAgfTtcbn1cblxudHlwZSBBdXRoZW50aWNhdGlvbiA9IERlZmF1bHRBdXRoZW50aWNhdGlvbiB8XG4gIE50bG1BdXRoZW50aWNhdGlvbiB8XG4gIEF6dXJlQWN0aXZlRGlyZWN0b3J5UGFzc3dvcmRBdXRoZW50aWNhdGlvbiB8XG4gIEF6dXJlQWN0aXZlRGlyZWN0b3J5TXNpQXBwU2VydmljZUF1dGhlbnRpY2F0aW9uIHxcbiAgQXp1cmVBY3RpdmVEaXJlY3RvcnlNc2lWbUF1dGhlbnRpY2F0aW9uIHxcbiAgQXp1cmVBY3RpdmVEaXJlY3RvcnlBY2Nlc3NUb2tlbkF1dGhlbnRpY2F0aW9uIHxcbiAgQXp1cmVBY3RpdmVEaXJlY3RvcnlTZXJ2aWNlUHJpbmNpcGFsU2VjcmV0IHxcbiAgQXp1cmVBY3RpdmVEaXJlY3RvcnlEZWZhdWx0QXV0aGVudGljYXRpb247XG5cbnR5cGUgQXV0aGVudGljYXRpb25UeXBlID0gQXV0aGVudGljYXRpb25bJ3R5cGUnXTtcblxuZXhwb3J0IGludGVyZmFjZSBDb25uZWN0aW9uQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBIb3N0bmFtZSB0byBjb25uZWN0IHRvLlxuICAgKi9cbiAgc2VydmVyOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBDb25maWd1cmF0aW9uIG9wdGlvbnMgZm9yIGZvcm1pbmcgdGhlIGNvbm5lY3Rpb24uXG4gICAqL1xuICBvcHRpb25zPzogQ29ubmVjdGlvbk9wdGlvbnM7XG4gIC8qKlxuICAgKiBBdXRoZW50aWNhdGlvbiByZWFsdGVkIG9wdGlvbnMgZm9yIGNvbm5lY3Rpb24uXG4gICAqL1xuICBhdXRoZW50aWNhdGlvbj86IEF1dGhlbnRpY2F0aW9uT3B0aW9ucztcbn1cblxuaW50ZXJmYWNlIERlYnVnT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBBIGJvb2xlYW4sIGNvbnRyb2xsaW5nIHdoZXRoZXIgW1tkZWJ1Z11dIGV2ZW50cyB3aWxsIGJlIGVtaXR0ZWQgd2l0aCB0ZXh0IGRlc2NyaWJpbmcgcGFja2V0IGRhdGEgZGV0YWlsc1xuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYClcbiAgICovXG4gIGRhdGE6IGJvb2xlYW47XG4gIC8qKlxuICAgKiBBIGJvb2xlYW4sIGNvbnRyb2xsaW5nIHdoZXRoZXIgW1tkZWJ1Z11dIGV2ZW50cyB3aWxsIGJlIGVtaXR0ZWQgd2l0aCB0ZXh0IGRlc2NyaWJpbmcgcGFja2V0IGRldGFpbHNcbiAgICpcbiAgICogKGRlZmF1bHQ6IGBmYWxzZWApXG4gICAqL1xuICBwYWNrZXQ6IGJvb2xlYW47XG4gIC8qKlxuICAgKiBBIGJvb2xlYW4sIGNvbnRyb2xsaW5nIHdoZXRoZXIgW1tkZWJ1Z11dIGV2ZW50cyB3aWxsIGJlIGVtaXR0ZWQgd2l0aCB0ZXh0IGRlc2NyaWJpbmcgcGFja2V0IHBheWxvYWQgZGV0YWlsc1xuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYClcbiAgICovXG4gIHBheWxvYWQ6IGJvb2xlYW47XG4gIC8qKlxuICAgKiBBIGJvb2xlYW4sIGNvbnRyb2xsaW5nIHdoZXRoZXIgW1tkZWJ1Z11dIGV2ZW50cyB3aWxsIGJlIGVtaXR0ZWQgd2l0aCB0ZXh0IGRlc2NyaWJpbmcgdG9rZW4gc3RyZWFtIHRva2Vuc1xuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYClcbiAgICovXG4gIHRva2VuOiBib29sZWFuO1xufVxuXG5pbnRlcmZhY2UgQXV0aGVudGljYXRpb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIFR5cGUgb2YgdGhlIGF1dGhlbnRpY2F0aW9uIG1ldGhvZCwgdmFsaWQgdHlwZXMgYXJlIGBkZWZhdWx0YCwgYG50bG1gLFxuICAgKiBgYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1wYXNzd29yZGAsIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWFjY2Vzcy10b2tlbmAsXG4gICAqIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS12bWAsIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS1hcHAtc2VydmljZWAsXG4gICAqIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWRlZmF1bHRgXG4gICAqIG9yIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXNlcnZpY2UtcHJpbmNpcGFsLXNlY3JldGBcbiAgICovXG4gIHR5cGU/OiBBdXRoZW50aWNhdGlvblR5cGU7XG4gIC8qKlxuICAgKiBEaWZmZXJlbnQgb3B0aW9ucyBmb3IgYXV0aGVudGljYXRpb24gdHlwZXM6XG4gICAqXG4gICAqICogYGRlZmF1bHRgOiBbW0RlZmF1bHRBdXRoZW50aWNhdGlvbi5vcHRpb25zXV1cbiAgICogKiBgbnRsbWAgOltbTnRsbUF1dGhlbnRpY2F0aW9uXV1cbiAgICogKiBgYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1wYXNzd29yZGAgOiBbW0F6dXJlQWN0aXZlRGlyZWN0b3J5UGFzc3dvcmRBdXRoZW50aWNhdGlvbi5vcHRpb25zXV1cbiAgICogKiBgYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1hY2Nlc3MtdG9rZW5gIDogW1tBenVyZUFjdGl2ZURpcmVjdG9yeUFjY2Vzc1Rva2VuQXV0aGVudGljYXRpb24ub3B0aW9uc11dXG4gICAqICogYGF6dXJlLWFjdGl2ZS1kaXJlY3RvcnktbXNpLXZtYCA6IFtbQXp1cmVBY3RpdmVEaXJlY3RvcnlNc2lWbUF1dGhlbnRpY2F0aW9uLm9wdGlvbnNdXVxuICAgKiAqIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS1hcHAtc2VydmljZWAgOiBbW0F6dXJlQWN0aXZlRGlyZWN0b3J5TXNpQXBwU2VydmljZUF1dGhlbnRpY2F0aW9uLm9wdGlvbnNdXVxuICAgKiAqIGBhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXNlcnZpY2UtcHJpbmNpcGFsLXNlY3JldGAgOiBbW0F6dXJlQWN0aXZlRGlyZWN0b3J5U2VydmljZVByaW5jaXBhbFNlY3JldC5vcHRpb25zXV1cbiAgICogKiBgYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1kZWZhdWx0YCA6IFtbQXp1cmVBY3RpdmVEaXJlY3RvcnlEZWZhdWx0QXV0aGVudGljYXRpb24ub3B0aW9uc11dXG4gICAqL1xuICBvcHRpb25zPzogYW55O1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIENvbm5lY3Rpb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIEEgYm9vbGVhbiBkZXRlcm1pbmluZyB3aGV0aGVyIHRvIHJvbGxiYWNrIGEgdHJhbnNhY3Rpb24gYXV0b21hdGljYWxseSBpZiBhbnkgZXJyb3IgaXMgZW5jb3VudGVyZWRcbiAgICogZHVyaW5nIHRoZSBnaXZlbiB0cmFuc2FjdGlvbidzIGV4ZWN1dGlvbi4gVGhpcyBzZXRzIHRoZSB2YWx1ZSBmb3IgYFNFVCBYQUNUX0FCT1JUYCBkdXJpbmcgdGhlXG4gICAqIGluaXRpYWwgU1FMIHBoYXNlIG9mIGEgY29ubmVjdGlvbiBbZG9jdW1lbnRhdGlvbl0oaHR0cHM6Ly9kb2NzLm1pY3Jvc29mdC5jb20vZW4tdXMvc3FsL3Qtc3FsL3N0YXRlbWVudHMvc2V0LXhhY3QtYWJvcnQtdHJhbnNhY3Qtc3FsKS5cbiAgICovXG4gIGFib3J0VHJhbnNhY3Rpb25PbkVycm9yPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQXBwbGljYXRpb24gbmFtZSB1c2VkIGZvciBpZGVudGlmeWluZyBhIHNwZWNpZmljIGFwcGxpY2F0aW9uIGluIHByb2ZpbGluZywgbG9nZ2luZyBvciB0cmFjaW5nIHRvb2xzIG9mIFNRTFNlcnZlci5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGBUZWRpb3VzYClcbiAgICovXG4gIGFwcE5hbWU/OiBzdHJpbmcgfCB1bmRlZmluZWQ7XG5cbiAgLyoqXG4gICAqIEEgYm9vbGVhbiwgY29udHJvbGxpbmcgd2hldGhlciB0aGUgY29sdW1uIG5hbWVzIHJldHVybmVkIHdpbGwgaGF2ZSB0aGUgZmlyc3QgbGV0dGVyIGNvbnZlcnRlZCB0byBsb3dlciBjYXNlXG4gICAqIChgdHJ1ZWApIG9yIG5vdC4gVGhpcyB2YWx1ZSBpcyBpZ25vcmVkIGlmIHlvdSBwcm92aWRlIGEgW1tjb2x1bW5OYW1lUmVwbGFjZXJdXS5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGBmYWxzZWApLlxuICAgKi9cbiAgY2FtZWxDYXNlQ29sdW1ucz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgbWlsbGlzZWNvbmRzIGJlZm9yZSB0aGUgW1tSZXF1ZXN0LmNhbmNlbF1dIChhYm9ydCkgb2YgYSByZXF1ZXN0IGlzIGNvbnNpZGVyZWQgZmFpbGVkXG4gICAqXG4gICAqIChkZWZhdWx0OiBgNTAwMGApLlxuICAgKi9cbiAgY2FuY2VsVGltZW91dD86IG51bWJlcjtcblxuICAvKipcbiAgICogQSBmdW5jdGlvbiB3aXRoIHBhcmFtZXRlcnMgYChjb2x1bW5OYW1lLCBpbmRleCwgY29sdW1uTWV0YURhdGEpYCBhbmQgcmV0dXJuaW5nIGEgc3RyaW5nLiBJZiBwcm92aWRlZCxcbiAgICogdGhpcyB3aWxsIGJlIGNhbGxlZCBvbmNlIHBlciBjb2x1bW4gcGVyIHJlc3VsdC1zZXQuIFRoZSByZXR1cm5lZCB2YWx1ZSB3aWxsIGJlIHVzZWQgaW5zdGVhZCBvZiB0aGUgU1FMLXByb3ZpZGVkXG4gICAqIGNvbHVtbiBuYW1lIG9uIHJvdyBhbmQgbWV0YSBkYXRhIG9iamVjdHMuIFRoaXMgYWxsb3dzIHlvdSB0byBkeW5hbWljYWxseSBjb252ZXJ0IGJldHdlZW4gbmFtaW5nIGNvbnZlbnRpb25zLlxuICAgKlxuICAgKiAoZGVmYXVsdDogYG51bGxgKVxuICAgKi9cbiAgY29sdW1uTmFtZVJlcGxhY2VyPzogKGNvbE5hbWU6IHN0cmluZywgaW5kZXg6IG51bWJlciwgbWV0YWRhdGE6IE1ldGFkYXRhKSA9PiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE51bWJlciBvZiBtaWxsaXNlY29uZHMgYmVmb3JlIHJldHJ5aW5nIHRvIGVzdGFibGlzaCBjb25uZWN0aW9uLCBpbiBjYXNlIG9mIHRyYW5zaWVudCBmYWlsdXJlLlxuICAgKlxuICAgKiAoZGVmYXVsdDpgNTAwYClcbiAgICovXG4gIGNvbm5lY3Rpb25SZXRyeUludGVydmFsPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIG1pbGxpc2Vjb25kcyBiZWZvcmUgdGhlIGF0dGVtcHQgdG8gY29ubmVjdCBpcyBjb25zaWRlcmVkIGZhaWxlZFxuICAgKlxuICAgKiAoZGVmYXVsdDogYDE1MDAwYCkuXG4gICAqL1xuICBjb25uZWN0VGltZW91dD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGRlZmF1bHQgaXNvbGF0aW9uIGxldmVsIGZvciBuZXcgY29ubmVjdGlvbnMuIEFsbCBvdXQtb2YtdHJhbnNhY3Rpb24gcXVlcmllcyBhcmUgZXhlY3V0ZWQgd2l0aCB0aGlzIHNldHRpbmcuXG4gICAqXG4gICAqIFRoZSBpc29sYXRpb24gbGV2ZWxzIGFyZSBhdmFpbGFibGUgZnJvbSBgcmVxdWlyZSgndGVkaW91cycpLklTT0xBVElPTl9MRVZFTGAuXG4gICAqICogYFJFQURfVU5DT01NSVRURURgXG4gICAqICogYFJFQURfQ09NTUlUVEVEYFxuICAgKiAqIGBSRVBFQVRBQkxFX1JFQURgXG4gICAqICogYFNFUklBTElaQUJMRWBcbiAgICogKiBgU05BUFNIT1RgXG4gICAqXG4gICAqIChkZWZhdWx0OiBgUkVBRF9DT01NSVRFRGApLlxuICAgKi9cbiAgY29ubmVjdGlvbklzb2xhdGlvbkxldmVsPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBXaGVuIGVuY3J5cHRpb24gaXMgdXNlZCwgYW4gb2JqZWN0IG1heSBiZSBzdXBwbGllZCB0aGF0IHdpbGwgYmUgdXNlZFxuICAgKiBmb3IgdGhlIGZpcnN0IGFyZ3VtZW50IHdoZW4gY2FsbGluZyBbYHRscy5jcmVhdGVTZWN1cmVQYWlyYF0oaHR0cDovL25vZGVqcy5vcmcvZG9jcy9sYXRlc3QvYXBpL3Rscy5odG1sI3Rsc190bHNfY3JlYXRlc2VjdXJlcGFpcl9jcmVkZW50aWFsc19pc3NlcnZlcl9yZXF1ZXN0Y2VydF9yZWplY3R1bmF1dGhvcml6ZWQpXG4gICAqXG4gICAqIChkZWZhdWx0OiBge31gKVxuICAgKi9cbiAgY3J5cHRvQ3JlZGVudGlhbHNEZXRhaWxzPzogU2VjdXJlQ29udGV4dE9wdGlvbnM7XG5cbiAgLyoqXG4gICAqIERhdGFiYXNlIHRvIGNvbm5lY3QgdG8gKGRlZmF1bHQ6IGRlcGVuZGVudCBvbiBzZXJ2ZXIgY29uZmlndXJhdGlvbikuXG4gICAqL1xuICBkYXRhYmFzZT86IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICAvKipcbiAgICogU2V0cyB0aGUgZmlyc3QgZGF5IG9mIHRoZSB3ZWVrIHRvIGEgbnVtYmVyIGZyb20gMSB0aHJvdWdoIDcuXG4gICAqL1xuICBkYXRlZmlyc3Q/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIEEgc3RyaW5nIHJlcHJlc2VudGluZyBwb3NpdGlvbiBvZiBtb250aCwgZGF5IGFuZCB5ZWFyIGluIHRlbXBvcmFsIGRhdGF0eXBlcy5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGBtZHlgKVxuICAgKi9cbiAgZGF0ZUZvcm1hdD86IHN0cmluZztcblxuICBkZWJ1Zz86IERlYnVnT3B0aW9ucztcblxuICAvKipcbiAgICogQSBib29sZWFuLCBjb250cm9scyB0aGUgd2F5IG51bGwgdmFsdWVzIHNob3VsZCBiZSB1c2VkIGR1cmluZyBjb21wYXJpc29uIG9wZXJhdGlvbi5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGB0cnVlYClcbiAgICovXG4gIGVuYWJsZUFuc2lOdWxsPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSWYgdHJ1ZSwgYFNFVCBBTlNJX05VTExfREZMVF9PTiBPTmAgd2lsbCBiZSBzZXQgaW4gdGhlIGluaXRpYWwgc3FsLiBUaGlzIG1lYW5zIG5ldyBjb2x1bW5zIHdpbGwgYmVcbiAgICogbnVsbGFibGUgYnkgZGVmYXVsdC4gU2VlIHRoZSBbVC1TUUwgZG9jdW1lbnRhdGlvbl0oaHR0cHM6Ly9tc2RuLm1pY3Jvc29mdC5jb20vZW4tdXMvbGlicmFyeS9tczE4NzM3NS5hc3B4KVxuICAgKlxuICAgKiAoZGVmYXVsdDogYHRydWVgKS5cbiAgICovXG4gIGVuYWJsZUFuc2lOdWxsRGVmYXVsdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgYm9vbGVhbiwgY29udHJvbHMgaWYgcGFkZGluZyBzaG91bGQgYmUgYXBwbGllZCBmb3IgdmFsdWVzIHNob3J0ZXIgdGhhbiB0aGUgc2l6ZSBvZiBkZWZpbmVkIGNvbHVtbi5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGB0cnVlYClcbiAgICovXG4gIGVuYWJsZUFuc2lQYWRkaW5nPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSWYgdHJ1ZSwgU1FMIFNlcnZlciB3aWxsIGZvbGxvdyBJU08gc3RhbmRhcmQgYmVoYXZpb3IgZHVyaW5nIHZhcmlvdXMgZXJyb3IgY29uZGl0aW9ucy4gRm9yIGRldGFpbHMsXG4gICAqIHNlZSBbZG9jdW1lbnRhdGlvbl0oaHR0cHM6Ly9kb2NzLm1pY3Jvc29mdC5jb20vZW4tdXMvc3FsL3Qtc3FsL3N0YXRlbWVudHMvc2V0LWFuc2ktd2FybmluZ3MtdHJhbnNhY3Qtc3FsKVxuICAgKlxuICAgKiAoZGVmYXVsdDogYHRydWVgKVxuICAgKi9cbiAgZW5hYmxlQW5zaVdhcm5pbmdzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogRW5kcyBhIHF1ZXJ5IHdoZW4gYW4gb3ZlcmZsb3cgb3IgZGl2aWRlLWJ5LXplcm8gZXJyb3Igb2NjdXJzIGR1cmluZyBxdWVyeSBleGVjdXRpb24uXG4gICAqIFNlZSBbZG9jdW1lbnRhdGlvbl0oaHR0cHM6Ly9kb2NzLm1pY3Jvc29mdC5jb20vZW4tdXMvc3FsL3Qtc3FsL3N0YXRlbWVudHMvc2V0LWFyaXRoYWJvcnQtdHJhbnNhY3Qtc3FsP3ZpZXc9c3FsLXNlcnZlci0yMDE3KVxuICAgKiBmb3IgbW9yZSBkZXRhaWxzLlxuICAgKlxuICAgKiAoZGVmYXVsdDogYHRydWVgKVxuICAgKi9cbiAgZW5hYmxlQXJpdGhBYm9ydD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgYm9vbGVhbiwgZGV0ZXJtaW5lcyBpZiBjb25jYXRlbmF0aW9uIHdpdGggTlVMTCBzaG91bGQgcmVzdWx0IGluIE5VTEwgb3IgZW1wdHkgc3RyaW5nIHZhbHVlLCBtb3JlIGRldGFpbHMgaW5cbiAgICogW2RvY3VtZW50YXRpb25dKGh0dHBzOi8vZG9jcy5taWNyb3NvZnQuY29tL2VuLXVzL3NxbC90LXNxbC9zdGF0ZW1lbnRzL3NldC1jb25jYXQtbnVsbC15aWVsZHMtbnVsbC10cmFuc2FjdC1zcWwpXG4gICAqXG4gICAqIChkZWZhdWx0OiBgdHJ1ZWApXG4gICAqL1xuICBlbmFibGVDb25jYXROdWxsWWllbGRzTnVsbD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgYm9vbGVhbiwgY29udHJvbHMgd2hldGhlciBjdXJzb3Igc2hvdWxkIGJlIGNsb3NlZCwgaWYgdGhlIHRyYW5zYWN0aW9uIG9wZW5pbmcgaXQgZ2V0cyBjb21taXR0ZWQgb3Igcm9sbGVkXG4gICAqIGJhY2suXG4gICAqXG4gICAqIChkZWZhdWx0OiBgbnVsbGApXG4gICAqL1xuICBlbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0PzogYm9vbGVhbiB8IG51bGw7XG5cbiAgLyoqXG4gICAqIEEgYm9vbGVhbiwgc2V0cyB0aGUgY29ubmVjdGlvbiB0byBlaXRoZXIgaW1wbGljaXQgb3IgYXV0b2NvbW1pdCB0cmFuc2FjdGlvbiBtb2RlLlxuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYClcbiAgICovXG4gIGVuYWJsZUltcGxpY2l0VHJhbnNhY3Rpb25zPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSWYgZmFsc2UsIGVycm9yIGlzIG5vdCBnZW5lcmF0ZWQgZHVyaW5nIGxvc3Mgb2YgcHJlY2Vzc2lvbi5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGBmYWxzZWApXG4gICAqL1xuICBlbmFibGVOdW1lcmljUm91bmRhYm9ydD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIElmIHRydWUsIGNoYXJhY3RlcnMgZW5jbG9zZWQgaW4gc2luZ2xlIHF1b3RlcyBhcmUgdHJlYXRlZCBhcyBsaXRlcmFscyBhbmQgdGhvc2UgZW5jbG9zZWQgZG91YmxlIHF1b3RlcyBhcmUgdHJlYXRlZCBhcyBpZGVudGlmaWVycy5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGB0cnVlYClcbiAgICovXG4gIGVuYWJsZVF1b3RlZElkZW50aWZpZXI/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBIGJvb2xlYW4gZGV0ZXJtaW5pbmcgd2hldGhlciBvciBub3QgdGhlIGNvbm5lY3Rpb24gd2lsbCBiZSBlbmNyeXB0ZWQuIFNldCB0byBgdHJ1ZWAgaWYgeW91J3JlIG9uIFdpbmRvd3MgQXp1cmUuXG4gICAqXG4gICAqIChkZWZhdWx0OiBgZmFsc2VgKVxuICAgKi9cbiAgZW5jcnlwdD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEJ5IGRlZmF1bHQsIGlmIHRoZSBkYXRhYmFzZSByZXF1ZXN0ZWQgYnkgW1tkYXRhYmFzZV1dIGNhbm5vdCBiZSBhY2Nlc3NlZCxcbiAgICogdGhlIGNvbm5lY3Rpb24gd2lsbCBmYWlsIHdpdGggYW4gZXJyb3IuIEhvd2V2ZXIsIGlmIFtbZmFsbGJhY2tUb0RlZmF1bHREYl1dIGlzXG4gICAqIHNldCB0byBgdHJ1ZWAsIHRoZW4gdGhlIHVzZXIncyBkZWZhdWx0IGRhdGFiYXNlIHdpbGwgYmUgdXNlZCBpbnN0ZWFkXG4gICAqXG4gICAqIChkZWZhdWx0OiBgZmFsc2VgKVxuICAgKi9cbiAgZmFsbGJhY2tUb0RlZmF1bHREYj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBpbnN0YW5jZSBuYW1lIHRvIGNvbm5lY3QgdG8uXG4gICAqIFRoZSBTUUwgU2VydmVyIEJyb3dzZXIgc2VydmljZSBtdXN0IGJlIHJ1bm5pbmcgb24gdGhlIGRhdGFiYXNlIHNlcnZlcixcbiAgICogYW5kIFVEUCBwb3J0IDE0MzQgb24gdGhlIGRhdGFiYXNlIHNlcnZlciBtdXN0IGJlIHJlYWNoYWJsZS5cbiAgICpcbiAgICogKG5vIGRlZmF1bHQpXG4gICAqXG4gICAqIE11dHVhbGx5IGV4Y2x1c2l2ZSB3aXRoIFtbcG9ydF1dLlxuICAgKi9cbiAgaW5zdGFuY2VOYW1lPzogc3RyaW5nIHwgdW5kZWZpbmVkO1xuXG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCBpc29sYXRpb24gbGV2ZWwgdGhhdCB0cmFuc2FjdGlvbnMgd2lsbCBiZSBydW4gd2l0aC5cbiAgICpcbiAgICogVGhlIGlzb2xhdGlvbiBsZXZlbHMgYXJlIGF2YWlsYWJsZSBmcm9tIGByZXF1aXJlKCd0ZWRpb3VzJykuSVNPTEFUSU9OX0xFVkVMYC5cbiAgICogKiBgUkVBRF9VTkNPTU1JVFRFRGBcbiAgICogKiBgUkVBRF9DT01NSVRURURgXG4gICAqICogYFJFUEVBVEFCTEVfUkVBRGBcbiAgICogKiBgU0VSSUFMSVpBQkxFYFxuICAgKiAqIGBTTkFQU0hPVGBcbiAgICpcbiAgICogKGRlZmF1bHQ6IGBSRUFEX0NPTU1JVEVEYCkuXG4gICAqL1xuICBpc29sYXRpb25MZXZlbD86IG51bWJlcjtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBsYW5ndWFnZSBlbnZpcm9ubWVudCBmb3IgdGhlIHNlc3Npb24uIFRoZSBzZXNzaW9uIGxhbmd1YWdlIGRldGVybWluZXMgdGhlIGRhdGV0aW1lIGZvcm1hdHMgYW5kIHN5c3RlbSBtZXNzYWdlcy5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGB1c19lbmdsaXNoYCkuXG4gICAqL1xuICBsYW5ndWFnZT86IHN0cmluZztcblxuICAvKipcbiAgICogQSBzdHJpbmcgaW5kaWNhdGluZyB3aGljaCBuZXR3b3JrIGludGVyZmFjZSAoaXAgYWRkcmVzcykgdG8gdXNlIHdoZW4gY29ubmVjdGluZyB0byBTUUwgU2VydmVyLlxuICAgKi9cbiAgbG9jYWxBZGRyZXNzPzogc3RyaW5nIHwgdW5kZWZpbmVkO1xuXG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBudW1iZXIgb2YgY29ubmVjdGlvbiByZXRyaWVzIGZvciB0cmFuc2llbnQgZXJyb3JzLuOAgVxuICAgKlxuICAgKiAoZGVmYXVsdDogYDNgKS5cbiAgICovXG4gIG1heFJldHJpZXNPblRyYW5zaWVudEVycm9ycz86IG51bWJlcjtcblxuICAvKipcbiAgICogU2V0cyB0aGUgTXVsdGlTdWJuZXRGYWlsb3ZlciA9IFRydWUgcGFyYW1ldGVyLCB3aGljaCBjYW4gaGVscCBtaW5pbWl6ZSB0aGUgY2xpZW50IHJlY292ZXJ5IGxhdGVuY3kgd2hlbiBmYWlsb3ZlcnMgb2NjdXIuXG4gICAqXG4gICAqIChkZWZhdWx0OiBgZmFsc2VgKS5cbiAgICovXG4gIG11bHRpU3VibmV0RmFpbG92ZXI/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgc2l6ZSBvZiBURFMgcGFja2V0cyAoc3ViamVjdCB0byBuZWdvdGlhdGlvbiB3aXRoIHRoZSBzZXJ2ZXIpLlxuICAgKiBTaG91bGQgYmUgYSBwb3dlciBvZiAyLlxuICAgKlxuICAgKiAoZGVmYXVsdDogYDQwOTZgKS5cbiAgICovXG4gIHBhY2tldFNpemU/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFBvcnQgdG8gY29ubmVjdCB0byAoZGVmYXVsdDogYDE0MzNgKS5cbiAgICpcbiAgICogTXV0dWFsbHkgZXhjbHVzaXZlIHdpdGggW1tpbnN0YW5jZU5hbWVdXVxuICAgKi9cbiAgcG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogQSBib29sZWFuLCBkZXRlcm1pbmluZyB3aGV0aGVyIHRoZSBjb25uZWN0aW9uIHdpbGwgcmVxdWVzdCByZWFkIG9ubHkgYWNjZXNzIGZyb20gYSBTUUwgU2VydmVyIEF2YWlsYWJpbGl0eVxuICAgKiBHcm91cC4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBbaGVyZV0oaHR0cDovL21zZG4ubWljcm9zb2Z0LmNvbS9lbi11cy9saWJyYXJ5L2hoNzEwMDU0LmFzcHggXCJNaWNyb3NvZnQ6IENvbmZpZ3VyZSBSZWFkLU9ubHkgUm91dGluZyBmb3IgYW4gQXZhaWxhYmlsaXR5IEdyb3VwIChTUUwgU2VydmVyKVwiKVxuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYCkuXG4gICAqL1xuICByZWFkT25seUludGVudD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgbWlsbGlzZWNvbmRzIGJlZm9yZSBhIHJlcXVlc3QgaXMgY29uc2lkZXJlZCBmYWlsZWQsIG9yIGAwYCBmb3Igbm8gdGltZW91dFxuICAgKlxuICAgKiAoZGVmYXVsdDogYDE1MDAwYCkuXG4gICAqL1xuICByZXF1ZXN0VGltZW91dD86IG51bWJlcjtcblxuICAvKipcbiAgICogQSBib29sZWFuLCB0aGF0IHdoZW4gdHJ1ZSB3aWxsIGV4cG9zZSByZWNlaXZlZCByb3dzIGluIFJlcXVlc3RzIGRvbmUgcmVsYXRlZCBldmVudHM6XG4gICAqICogW1tSZXF1ZXN0LkV2ZW50X2RvbmVJblByb2NdXVxuICAgKiAqIFtbUmVxdWVzdC5FdmVudF9kb25lUHJvY11dXG4gICAqICogW1tSZXF1ZXN0LkV2ZW50X2RvbmVdXVxuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYClcbiAgICpcbiAgICogQ2F1dGlvbjogSWYgbWFueSByb3cgYXJlIHJlY2VpdmVkLCBlbmFibGluZyB0aGlzIG9wdGlvbiBjb3VsZCByZXN1bHQgaW5cbiAgICogZXhjZXNzaXZlIG1lbW9yeSB1c2FnZS5cbiAgICovXG4gIHJvd0NvbGxlY3Rpb25PbkRvbmU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBIGJvb2xlYW4sIHRoYXQgd2hlbiB0cnVlIHdpbGwgZXhwb3NlIHJlY2VpdmVkIHJvd3MgaW4gUmVxdWVzdHMnIGNvbXBsZXRpb24gY2FsbGJhY2suU2VlIFtbUmVxdWVzdC5jb25zdHJ1Y3Rvcl1dLlxuICAgKlxuICAgKiAoZGVmYXVsdDogYGZhbHNlYClcbiAgICpcbiAgICogQ2F1dGlvbjogSWYgbWFueSByb3cgYXJlIHJlY2VpdmVkLCBlbmFibGluZyB0aGlzIG9wdGlvbiBjb3VsZCByZXN1bHQgaW5cbiAgICogZXhjZXNzaXZlIG1lbW9yeSB1c2FnZS5cbiAgICovXG4gIHJvd0NvbGxlY3Rpb25PblJlcXVlc3RDb21wbGV0aW9uPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIHZlcnNpb24gb2YgVERTIHRvIHVzZS4gSWYgc2VydmVyIGRvZXNuJ3Qgc3VwcG9ydCBzcGVjaWZpZWQgdmVyc2lvbiwgbmVnb3RpYXRlZCB2ZXJzaW9uIGlzIHVzZWQgaW5zdGVhZC5cbiAgICpcbiAgICogVGhlIHZlcnNpb25zIGFyZSBhdmFpbGFibGUgZnJvbSBgcmVxdWlyZSgndGVkaW91cycpLlREU19WRVJTSU9OYC5cbiAgICogKiBgN18xYFxuICAgKiAqIGA3XzJgXG4gICAqICogYDdfM19BYFxuICAgKiAqIGA3XzNfQmBcbiAgICogKiBgN180YFxuICAgKlxuICAgKiAoZGVmYXVsdDogYDdfNGApXG4gICAqL1xuICB0ZHNWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIHNpemUgb2YgdmFyY2hhcihtYXgpLCBudmFyY2hhcihtYXgpLCB2YXJiaW5hcnkobWF4KSwgdGV4dCwgbnRleHQsIGFuZCBpbWFnZSBkYXRhIHJldHVybmVkIGJ5IGEgU0VMRUNUIHN0YXRlbWVudC5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGAyMTQ3NDgzNjQ3YClcbiAgICovXG4gIHRleHRzaXplPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBJZiBcInRydWVcIiwgdGhlIFNRTCBTZXJ2ZXIgU1NMIGNlcnRpZmljYXRlIGlzIGF1dG9tYXRpY2FsbHkgdHJ1c3RlZCB3aGVuIHRoZSBjb21tdW5pY2F0aW9uIGxheWVyIGlzIGVuY3J5cHRlZCB1c2luZyBTU0wuXG4gICAqXG4gICAqIElmIFwiZmFsc2VcIiwgdGhlIFNRTCBTZXJ2ZXIgdmFsaWRhdGVzIHRoZSBzZXJ2ZXIgU1NMIGNlcnRpZmljYXRlLiBJZiB0aGUgc2VydmVyIGNlcnRpZmljYXRlIHZhbGlkYXRpb24gZmFpbHMsXG4gICAqIHRoZSBkcml2ZXIgcmFpc2VzIGFuIGVycm9yIGFuZCB0ZXJtaW5hdGVzIHRoZSBjb25uZWN0aW9uLiBNYWtlIHN1cmUgdGhlIHZhbHVlIHBhc3NlZCB0byBzZXJ2ZXJOYW1lIGV4YWN0bHlcbiAgICogbWF0Y2hlcyB0aGUgQ29tbW9uIE5hbWUgKENOKSBvciBETlMgbmFtZSBpbiB0aGUgU3ViamVjdCBBbHRlcm5hdGUgTmFtZSBpbiB0aGUgc2VydmVyIGNlcnRpZmljYXRlIGZvciBhbiBTU0wgY29ubmVjdGlvbiB0byBzdWNjZWVkLlxuICAgKlxuICAgKiAoZGVmYXVsdDogYHRydWVgKVxuICAgKi9cbiAgdHJ1c3RTZXJ2ZXJDZXJ0aWZpY2F0ZT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgYm9vbGVhbiBkZXRlcm1pbmluZyB3aGV0aGVyIHRvIHJldHVybiByb3dzIGFzIGFycmF5cyBvciBrZXktdmFsdWUgY29sbGVjdGlvbnMuXG4gICAqXG4gICAqIChkZWZhdWx0OiBgZmFsc2VgKS5cbiAgICovXG4gIHVzZUNvbHVtbk5hbWVzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQSBib29sZWFuIGRldGVybWluaW5nIHdoZXRoZXIgdG8gcGFzcyB0aW1lIHZhbHVlcyBpbiBVVEMgb3IgbG9jYWwgdGltZS5cbiAgICpcbiAgICogKGRlZmF1bHQ6IGB0cnVlYCkuXG4gICAqL1xuICB1c2VVVEM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgd29ya3N0YXRpb24gSUQgKFdTSUQpIG9mIHRoZSBjbGllbnQsIGRlZmF1bHQgb3MuaG9zdG5hbWUoKS5cbiAgICogVXNlZCBmb3IgaWRlbnRpZnlpbmcgYSBzcGVjaWZpYyBjbGllbnQgaW4gcHJvZmlsaW5nLCBsb2dnaW5nIG9yXG4gICAqIHRyYWNpbmcgY2xpZW50IGFjdGl2aXR5IGluIFNRTFNlcnZlci5cbiAgICpcbiAgICogVGhlIHZhbHVlIGlzIHJlcG9ydGVkIGJ5IHRoZSBUU1FMIGZ1bmN0aW9uIEhPU1RfTkFNRSgpLlxuICAgKi9cbiAgd29ya3N0YXRpb25JZD86IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICAvKipcbiAgICogQSBib29sZWFuIGRldGVybWluaW5nIHdoZXRoZXIgdG8gcGFyc2UgdW5pcXVlIGlkZW50aWZpZXIgdHlwZSB3aXRoIGxvd2VyY2FzZSBjYXNlIGNoYXJhY3RlcnMuXG4gICAqXG4gICAqIChkZWZhdWx0OiBgZmFsc2VgKS5cbiAgICovXG4gIGxvd2VyQ2FzZUd1aWRzPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBAcHJpdmF0ZVxuICovXG5jb25zdCBDTEVBTlVQX1RZUEUgPSB7XG4gIE5PUk1BTDogMCxcbiAgUkVESVJFQ1Q6IDEsXG4gIFJFVFJZOiAyXG59O1xuXG5pbnRlcmZhY2UgUm91dGluZ0RhdGEge1xuICBzZXJ2ZXI6IHN0cmluZztcbiAgcG9ydDogbnVtYmVyO1xufVxuXG4vKipcbiAqIEEgW1tDb25uZWN0aW9uXV0gaW5zdGFuY2UgcmVwcmVzZW50cyBhIHNpbmdsZSBjb25uZWN0aW9uIHRvIGEgZGF0YWJhc2Ugc2VydmVyLlxuICpcbiAqIGBgYGpzXG4gKiB2YXIgQ29ubmVjdGlvbiA9IHJlcXVpcmUoJ3RlZGlvdXMnKS5Db25uZWN0aW9uO1xuICogdmFyIGNvbmZpZyA9IHtcbiAqICBcImF1dGhlbnRpY2F0aW9uXCI6IHtcbiAqICAgIC4uLixcbiAqICAgIFwib3B0aW9uc1wiOiB7Li4ufVxuICogIH0sXG4gKiAgXCJvcHRpb25zXCI6IHsuLi59XG4gKiB9O1xuICogdmFyIGNvbm5lY3Rpb24gPSBuZXcgQ29ubmVjdGlvbihjb25maWcpO1xuICogYGBgXG4gKlxuICogT25seSBvbmUgcmVxdWVzdCBhdCBhIHRpbWUgbWF5IGJlIGV4ZWN1dGVkIG9uIGEgY29ubmVjdGlvbi4gT25jZSBhIFtbUmVxdWVzdF1dXG4gKiBoYXMgYmVlbiBpbml0aWF0ZWQgKHdpdGggW1tDb25uZWN0aW9uLmNhbGxQcm9jZWR1cmVdXSwgW1tDb25uZWN0aW9uLmV4ZWNTcWxdXSxcbiAqIG9yIFtbQ29ubmVjdGlvbi5leGVjU3FsQmF0Y2hdXSksIGFub3RoZXIgc2hvdWxkIG5vdCBiZSBpbml0aWF0ZWQgdW50aWwgdGhlXG4gKiBbW1JlcXVlc3RdXSdzIGNvbXBsZXRpb24gY2FsbGJhY2sgaXMgY2FsbGVkLlxuICovXG5jbGFzcyBDb25uZWN0aW9uIGV4dGVuZHMgRXZlbnRFbWl0dGVyIHtcbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBmZWRBdXRoUmVxdWlyZWQ6IGJvb2xlYW47XG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY29uZmlnOiBJbnRlcm5hbENvbm5lY3Rpb25Db25maWc7XG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgc2VjdXJlQ29udGV4dDogU2VjdXJlQ29udGV4dDtcbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBpblRyYW5zYWN0aW9uOiBib29sZWFuO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHRyYW5zYWN0aW9uRGVzY3JpcHRvcnM6IEJ1ZmZlcltdO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHRyYW5zYWN0aW9uRGVwdGg6IG51bWJlcjtcbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBpc1NxbEJhdGNoOiBib29sZWFuO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGN1clRyYW5zaWVudFJldHJ5Q291bnQ6IG51bWJlcjtcbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICB0cmFuc2llbnRFcnJvckxvb2t1cDogVHJhbnNpZW50RXJyb3JMb29rdXA7XG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY2xvc2VkOiBib29sZWFuO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGxvZ2luRXJyb3I6IHVuZGVmaW5lZCB8IEFnZ3JlZ2F0ZUVycm9yIHwgQ29ubmVjdGlvbkVycm9yO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGRlYnVnOiBEZWJ1ZztcbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBudGxtcGFja2V0OiB1bmRlZmluZWQgfCBhbnk7XG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgbnRsbXBhY2tldEJ1ZmZlcjogdW5kZWZpbmVkIHwgQnVmZmVyO1xuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZGVjbGFyZSBTVEFURToge1xuICAgIElOSVRJQUxJWkVEOiBTdGF0ZTtcbiAgICBDT05ORUNUSU5HOiBTdGF0ZTtcbiAgICBTRU5UX1BSRUxPR0lOOiBTdGF0ZTtcbiAgICBSRVJPVVRJTkc6IFN0YXRlO1xuICAgIFRSQU5TSUVOVF9GQUlMVVJFX1JFVFJZOiBTdGF0ZTtcbiAgICBTRU5UX1RMU1NTTE5FR09USUFUSU9OOiBTdGF0ZTtcbiAgICBTRU5UX0xPR0lON19XSVRIX1NUQU5EQVJEX0xPR0lOOiBTdGF0ZTtcbiAgICBTRU5UX0xPR0lON19XSVRIX05UTE06IFN0YXRlO1xuICAgIFNFTlRfTE9HSU43X1dJVEhfRkVEQVVUSDogU3RhdGU7XG4gICAgTE9HR0VEX0lOX1NFTkRJTkdfSU5JVElBTF9TUUw6IFN0YXRlO1xuICAgIExPR0dFRF9JTjogU3RhdGU7XG4gICAgU0VOVF9DTElFTlRfUkVRVUVTVDogU3RhdGU7XG4gICAgU0VOVF9BVFRFTlRJT046IFN0YXRlO1xuICAgIEZJTkFMOiBTdGF0ZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcm91dGluZ0RhdGE6IHVuZGVmaW5lZCB8IFJvdXRpbmdEYXRhO1xuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgbWVzc2FnZUlvITogTWVzc2FnZUlPO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHN0YXRlOiBTdGF0ZTtcbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICByZXNldENvbm5lY3Rpb25Pbk5leHRSZXF1ZXN0OiB1bmRlZmluZWQgfCBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcmVxdWVzdDogdW5kZWZpbmVkIHwgUmVxdWVzdCB8IEJ1bGtMb2FkO1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByb2NSZXR1cm5TdGF0dXNWYWx1ZTogdW5kZWZpbmVkIHwgYW55O1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHNvY2tldDogdW5kZWZpbmVkIHwgU29ja2V0O1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIG1lc3NhZ2VCdWZmZXI6IEJ1ZmZlcjtcblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNvbm5lY3RUaW1lcjogdW5kZWZpbmVkIHwgTm9kZUpTLlRpbWVvdXQ7XG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY2FuY2VsVGltZXI6IHVuZGVmaW5lZCB8IE5vZGVKUy5UaW1lb3V0O1xuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHJlcXVlc3RUaW1lcjogdW5kZWZpbmVkIHwgTm9kZUpTLlRpbWVvdXQ7XG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcmV0cnlUaW1lcjogdW5kZWZpbmVkIHwgTm9kZUpTLlRpbWVvdXQ7XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBfY2FuY2VsQWZ0ZXJSZXF1ZXN0U2VudDogKCkgPT4gdm9pZDtcblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGRhdGFiYXNlQ29sbGF0aW9uOiBDb2xsYXRpb24gfCB1bmRlZmluZWQ7XG5cbiAgLyoqXG4gICAqIE5vdGU6IGJlIGF3YXJlIG9mIHRoZSBkaWZmZXJlbnQgb3B0aW9ucyBmaWVsZDpcbiAgICogMS4gY29uZmlnLmF1dGhlbnRpY2F0aW9uLm9wdGlvbnNcbiAgICogMi4gY29uZmlnLm9wdGlvbnNcbiAgICpcbiAgICogYGBganNcbiAgICogY29uc3QgeyBDb25uZWN0aW9uIH0gPSByZXF1aXJlKCd0ZWRpb3VzJyk7XG4gICAqXG4gICAqIGNvbnN0IGNvbmZpZyA9IHtcbiAgICogIFwiYXV0aGVudGljYXRpb25cIjoge1xuICAgKiAgICAuLi4sXG4gICAqICAgIFwib3B0aW9uc1wiOiB7Li4ufVxuICAgKiAgfSxcbiAgICogIFwib3B0aW9uc1wiOiB7Li4ufVxuICAgKiB9O1xuICAgKlxuICAgKiBjb25zdCBjb25uZWN0aW9uID0gbmV3IENvbm5lY3Rpb24oY29uZmlnKTtcbiAgICogYGBgXG4gICAqXG4gICAqIEBwYXJhbSBjb25maWdcbiAgICovXG4gIGNvbnN0cnVjdG9yKGNvbmZpZzogQ29ubmVjdGlvbkNvbmZpZ3VyYXRpb24pIHtcbiAgICBzdXBlcigpO1xuXG4gICAgaWYgKHR5cGVvZiBjb25maWcgIT09ICdvYmplY3QnIHx8IGNvbmZpZyA9PT0gbnVsbCkge1xuICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnXCIgYXJndW1lbnQgaXMgcmVxdWlyZWQgYW5kIG11c3QgYmUgb2YgdHlwZSBPYmplY3QuJyk7XG4gICAgfVxuXG4gICAgaWYgKHR5cGVvZiBjb25maWcuc2VydmVyICE9PSAnc3RyaW5nJykge1xuICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLnNlcnZlclwiIHByb3BlcnR5IGlzIHJlcXVpcmVkIGFuZCBtdXN0IGJlIG9mIHR5cGUgc3RyaW5nLicpO1xuICAgIH1cblxuICAgIHRoaXMuZmVkQXV0aFJlcXVpcmVkID0gZmFsc2U7XG5cbiAgICBsZXQgYXV0aGVudGljYXRpb246IEludGVybmFsQ29ubmVjdGlvbkNvbmZpZ1snYXV0aGVudGljYXRpb24nXTtcbiAgICBpZiAoY29uZmlnLmF1dGhlbnRpY2F0aW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGlmICh0eXBlb2YgY29uZmlnLmF1dGhlbnRpY2F0aW9uICE9PSAnb2JqZWN0JyB8fCBjb25maWcuYXV0aGVudGljYXRpb24gPT09IG51bGwpIHtcbiAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLmF1dGhlbnRpY2F0aW9uXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIE9iamVjdC4nKTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgdHlwZSA9IGNvbmZpZy5hdXRoZW50aWNhdGlvbi50eXBlO1xuICAgICAgY29uc3Qgb3B0aW9ucyA9IGNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zID09PSB1bmRlZmluZWQgPyB7fSA6IGNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zO1xuXG4gICAgICBpZiAodHlwZW9mIHR5cGUgIT09ICdzdHJpbmcnKSB7XG4gICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi50eXBlXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgIH1cblxuICAgICAgaWYgKHR5cGUgIT09ICdkZWZhdWx0JyAmJiB0eXBlICE9PSAnbnRsbScgJiYgdHlwZSAhPT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktcGFzc3dvcmQnICYmIHR5cGUgIT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWFjY2Vzcy10b2tlbicgJiYgdHlwZSAhPT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktbXNpLXZtJyAmJiB0eXBlICE9PSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1tc2ktYXBwLXNlcnZpY2UnICYmIHR5cGUgIT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXNlcnZpY2UtcHJpbmNpcGFsLXNlY3JldCcgJiYgdHlwZSAhPT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktZGVmYXVsdCcpIHtcbiAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwidHlwZVwiIHByb3BlcnR5IG11c3Qgb25lIG9mIFwiZGVmYXVsdFwiLCBcIm50bG1cIiwgXCJhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXBhc3N3b3JkXCIsIFwiYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1hY2Nlc3MtdG9rZW5cIiwgXCJhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWRlZmF1bHRcIiwgXCJhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS12bVwiIG9yIFwiYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1tc2ktYXBwLXNlcnZpY2VcIiBvciBcImF6dXJlLWFjdGl2ZS1kaXJlY3Rvcnktc2VydmljZS1wcmluY2lwYWwtc2VjcmV0XCIuJyk7XG4gICAgICB9XG5cbiAgICAgIGlmICh0eXBlb2Ygb3B0aW9ucyAhPT0gJ29iamVjdCcgfHwgb3B0aW9ucyA9PT0gbnVsbCkge1xuICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9uc1wiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBvYmplY3QuJyk7XG4gICAgICB9XG5cbiAgICAgIGlmICh0eXBlID09PSAnbnRsbScpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBvcHRpb25zLmRvbWFpbiAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy5kb21haW5cIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgc3RyaW5nLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKG9wdGlvbnMudXNlck5hbWUgIT09IHVuZGVmaW5lZCAmJiB0eXBlb2Ygb3B0aW9ucy51c2VyTmFtZSAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy51c2VyTmFtZVwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAob3B0aW9ucy5wYXNzd29yZCAhPT0gdW5kZWZpbmVkICYmIHR5cGVvZiBvcHRpb25zLnBhc3N3b3JkICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLnBhc3N3b3JkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGF1dGhlbnRpY2F0aW9uID0ge1xuICAgICAgICAgIHR5cGU6ICdudGxtJyxcbiAgICAgICAgICBvcHRpb25zOiB7XG4gICAgICAgICAgICB1c2VyTmFtZTogb3B0aW9ucy51c2VyTmFtZSxcbiAgICAgICAgICAgIHBhc3N3b3JkOiBvcHRpb25zLnBhc3N3b3JkLFxuICAgICAgICAgICAgZG9tYWluOiBvcHRpb25zLmRvbWFpbiAmJiBvcHRpb25zLmRvbWFpbi50b1VwcGVyQ2FzZSgpXG4gICAgICAgICAgfVxuICAgICAgICB9O1xuICAgICAgfSBlbHNlIGlmICh0eXBlID09PSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1wYXNzd29yZCcpIHtcbiAgICAgICAgaWYgKG9wdGlvbnMudXNlck5hbWUgIT09IHVuZGVmaW5lZCAmJiB0eXBlb2Ygb3B0aW9ucy51c2VyTmFtZSAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy51c2VyTmFtZVwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAob3B0aW9ucy5wYXNzd29yZCAhPT0gdW5kZWZpbmVkICYmIHR5cGVvZiBvcHRpb25zLnBhc3N3b3JkICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLnBhc3N3b3JkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmIChvcHRpb25zLmNsaWVudElkICE9PSB1bmRlZmluZWQgJiYgdHlwZW9mIG9wdGlvbnMuY2xpZW50SWQgIT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLmF1dGhlbnRpY2F0aW9uLm9wdGlvbnMuY2xpZW50SWRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgc3RyaW5nLicpO1xuICAgICAgICB9IGVsc2UgaWYgKG9wdGlvbnMuY2xpZW50SWQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIGVtaXRBenVyZUFEUGFzc3dvcmRDbGllbnRJZERlcHJlY2F0aW9uV2FybmluZygpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKG9wdGlvbnMuZG9tYWluICE9PSB1bmRlZmluZWQgJiYgdHlwZW9mIG9wdGlvbnMuZG9tYWluICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLmRvbWFpblwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH0gZWxzZSBpZiAob3B0aW9ucy5kb21haW4gIT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIGVtaXREb21haW5SZW5hbWVUb1RlbmFudElkV2FybmluZygpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKG9wdGlvbnMudGVuYW50SWQgIT09IHVuZGVmaW5lZCAmJiB0eXBlb2Ygb3B0aW9ucy50ZW5hbnRJZCAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy50ZW5hbnRJZFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBhdXRoZW50aWNhdGlvbiA9IHtcbiAgICAgICAgICB0eXBlOiAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1wYXNzd29yZCcsXG4gICAgICAgICAgb3B0aW9uczoge1xuICAgICAgICAgICAgdXNlck5hbWU6IG9wdGlvbnMudXNlck5hbWUsXG4gICAgICAgICAgICBwYXNzd29yZDogb3B0aW9ucy5wYXNzd29yZCxcbiAgICAgICAgICAgIHRlbmFudElkOiBvcHRpb25zLnRlbmFudElkID8/IG9wdGlvbnMuZG9tYWluLFxuICAgICAgICAgICAgY2xpZW50SWQ6IG9wdGlvbnMuY2xpZW50SWQgPz8gJzdmOThjYjA0LWNkMWUtNDBkZi05MTQwLTNiZjdlMmNlYTRkYidcbiAgICAgICAgICB9XG4gICAgICAgIH07XG4gICAgICB9IGVsc2UgaWYgKHR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWFjY2Vzcy10b2tlbicpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBvcHRpb25zLnRva2VuICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLnRva2VuXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGF1dGhlbnRpY2F0aW9uID0ge1xuICAgICAgICAgIHR5cGU6ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWFjY2Vzcy10b2tlbicsXG4gICAgICAgICAgb3B0aW9uczoge1xuICAgICAgICAgICAgdG9rZW46IG9wdGlvbnMudG9rZW5cbiAgICAgICAgICB9XG4gICAgICAgIH07XG4gICAgICB9IGVsc2UgaWYgKHR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS12bScpIHtcbiAgICAgICAgaWYgKG9wdGlvbnMuY2xpZW50SWQgIT09IHVuZGVmaW5lZCAmJiB0eXBlb2Ygb3B0aW9ucy5jbGllbnRJZCAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy5jbGllbnRJZFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBhdXRoZW50aWNhdGlvbiA9IHtcbiAgICAgICAgICB0eXBlOiAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1tc2ktdm0nLFxuICAgICAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgICAgIGNsaWVudElkOiBvcHRpb25zLmNsaWVudElkXG4gICAgICAgICAgfVxuICAgICAgICB9O1xuICAgICAgfSBlbHNlIGlmICh0eXBlID09PSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1kZWZhdWx0Jykge1xuICAgICAgICBpZiAob3B0aW9ucy5jbGllbnRJZCAhPT0gdW5kZWZpbmVkICYmIHR5cGVvZiBvcHRpb25zLmNsaWVudElkICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLmNsaWVudElkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuICAgICAgICBhdXRoZW50aWNhdGlvbiA9IHtcbiAgICAgICAgICB0eXBlOiAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1kZWZhdWx0JyxcbiAgICAgICAgICBvcHRpb25zOiB7XG4gICAgICAgICAgICBjbGllbnRJZDogb3B0aW9ucy5jbGllbnRJZFxuICAgICAgICAgIH1cbiAgICAgICAgfTtcbiAgICAgIH0gZWxzZSBpZiAodHlwZSA9PT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktbXNpLWFwcC1zZXJ2aWNlJykge1xuICAgICAgICBpZiAob3B0aW9ucy5jbGllbnRJZCAhPT0gdW5kZWZpbmVkICYmIHR5cGVvZiBvcHRpb25zLmNsaWVudElkICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLmNsaWVudElkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGF1dGhlbnRpY2F0aW9uID0ge1xuICAgICAgICAgIHR5cGU6ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS1hcHAtc2VydmljZScsXG4gICAgICAgICAgb3B0aW9uczoge1xuICAgICAgICAgICAgY2xpZW50SWQ6IG9wdGlvbnMuY2xpZW50SWRcbiAgICAgICAgICB9XG4gICAgICAgIH07XG4gICAgICB9IGVsc2UgaWYgKHR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXNlcnZpY2UtcHJpbmNpcGFsLXNlY3JldCcpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBvcHRpb25zLmNsaWVudElkICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLmNsaWVudElkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICh0eXBlb2Ygb3B0aW9ucy5jbGllbnRTZWNyZXQgIT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLmF1dGhlbnRpY2F0aW9uLm9wdGlvbnMuY2xpZW50U2VjcmV0XCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGlmICh0eXBlb2Ygb3B0aW9ucy50ZW5hbnRJZCAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy50ZW5hbnRJZFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBhdXRoZW50aWNhdGlvbiA9IHtcbiAgICAgICAgICB0eXBlOiAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1zZXJ2aWNlLXByaW5jaXBhbC1zZWNyZXQnLFxuICAgICAgICAgIG9wdGlvbnM6IHtcbiAgICAgICAgICAgIGNsaWVudElkOiBvcHRpb25zLmNsaWVudElkLFxuICAgICAgICAgICAgY2xpZW50U2VjcmV0OiBvcHRpb25zLmNsaWVudFNlY3JldCxcbiAgICAgICAgICAgIHRlbmFudElkOiBvcHRpb25zLnRlbmFudElkXG4gICAgICAgICAgfVxuICAgICAgICB9O1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgaWYgKG9wdGlvbnMudXNlck5hbWUgIT09IHVuZGVmaW5lZCAmJiB0eXBlb2Ygb3B0aW9ucy51c2VyTmFtZSAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcuYXV0aGVudGljYXRpb24ub3B0aW9ucy51c2VyTmFtZVwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAob3B0aW9ucy5wYXNzd29yZCAhPT0gdW5kZWZpbmVkICYmIHR5cGVvZiBvcHRpb25zLnBhc3N3b3JkICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5hdXRoZW50aWNhdGlvbi5vcHRpb25zLnBhc3N3b3JkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGF1dGhlbnRpY2F0aW9uID0ge1xuICAgICAgICAgIHR5cGU6ICdkZWZhdWx0JyxcbiAgICAgICAgICBvcHRpb25zOiB7XG4gICAgICAgICAgICB1c2VyTmFtZTogb3B0aW9ucy51c2VyTmFtZSxcbiAgICAgICAgICAgIHBhc3N3b3JkOiBvcHRpb25zLnBhc3N3b3JkXG4gICAgICAgICAgfVxuICAgICAgICB9O1xuICAgICAgfVxuICAgIH0gZWxzZSB7XG4gICAgICBhdXRoZW50aWNhdGlvbiA9IHtcbiAgICAgICAgdHlwZTogJ2RlZmF1bHQnLFxuICAgICAgICBvcHRpb25zOiB7XG4gICAgICAgICAgdXNlck5hbWU6IHVuZGVmaW5lZCxcbiAgICAgICAgICBwYXNzd29yZDogdW5kZWZpbmVkXG4gICAgICAgIH1cbiAgICAgIH07XG4gICAgfVxuXG4gICAgdGhpcy5jb25maWcgPSB7XG4gICAgICBzZXJ2ZXI6IGNvbmZpZy5zZXJ2ZXIsXG4gICAgICBhdXRoZW50aWNhdGlvbjogYXV0aGVudGljYXRpb24sXG4gICAgICBvcHRpb25zOiB7XG4gICAgICAgIGFib3J0VHJhbnNhY3Rpb25PbkVycm9yOiBmYWxzZSxcbiAgICAgICAgYXBwTmFtZTogdW5kZWZpbmVkLFxuICAgICAgICBjYW1lbENhc2VDb2x1bW5zOiBmYWxzZSxcbiAgICAgICAgY2FuY2VsVGltZW91dDogREVGQVVMVF9DQU5DRUxfVElNRU9VVCxcbiAgICAgICAgY29sdW1uRW5jcnlwdGlvbktleUNhY2hlVFRMOiAyICogNjAgKiA2MCAqIDEwMDAsICAvLyBVbml0czogbWlsaXNlY29uZHNcbiAgICAgICAgY29sdW1uRW5jcnlwdGlvblNldHRpbmc6IGZhbHNlLFxuICAgICAgICBjb2x1bW5OYW1lUmVwbGFjZXI6IHVuZGVmaW5lZCxcbiAgICAgICAgY29ubmVjdGlvblJldHJ5SW50ZXJ2YWw6IERFRkFVTFRfQ09OTkVDVF9SRVRSWV9JTlRFUlZBTCxcbiAgICAgICAgY29ubmVjdFRpbWVvdXQ6IERFRkFVTFRfQ09OTkVDVF9USU1FT1VULFxuICAgICAgICBjb25uZWN0aW9uSXNvbGF0aW9uTGV2ZWw6IElTT0xBVElPTl9MRVZFTC5SRUFEX0NPTU1JVFRFRCxcbiAgICAgICAgY3J5cHRvQ3JlZGVudGlhbHNEZXRhaWxzOiB7fSxcbiAgICAgICAgZGF0YWJhc2U6IHVuZGVmaW5lZCxcbiAgICAgICAgZGF0ZWZpcnN0OiBERUZBVUxUX0RBVEVGSVJTVCxcbiAgICAgICAgZGF0ZUZvcm1hdDogREVGQVVMVF9EQVRFRk9STUFULFxuICAgICAgICBkZWJ1Zzoge1xuICAgICAgICAgIGRhdGE6IGZhbHNlLFxuICAgICAgICAgIHBhY2tldDogZmFsc2UsXG4gICAgICAgICAgcGF5bG9hZDogZmFsc2UsXG4gICAgICAgICAgdG9rZW46IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIGVuYWJsZUFuc2lOdWxsOiB0cnVlLFxuICAgICAgICBlbmFibGVBbnNpTnVsbERlZmF1bHQ6IHRydWUsXG4gICAgICAgIGVuYWJsZUFuc2lQYWRkaW5nOiB0cnVlLFxuICAgICAgICBlbmFibGVBbnNpV2FybmluZ3M6IHRydWUsXG4gICAgICAgIGVuYWJsZUFyaXRoQWJvcnQ6IHRydWUsXG4gICAgICAgIGVuYWJsZUNvbmNhdE51bGxZaWVsZHNOdWxsOiB0cnVlLFxuICAgICAgICBlbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0OiBudWxsLFxuICAgICAgICBlbmFibGVJbXBsaWNpdFRyYW5zYWN0aW9uczogZmFsc2UsXG4gICAgICAgIGVuYWJsZU51bWVyaWNSb3VuZGFib3J0OiBmYWxzZSxcbiAgICAgICAgZW5hYmxlUXVvdGVkSWRlbnRpZmllcjogdHJ1ZSxcbiAgICAgICAgZW5jcnlwdDogdHJ1ZSxcbiAgICAgICAgZmFsbGJhY2tUb0RlZmF1bHREYjogZmFsc2UsXG4gICAgICAgIGVuY3J5cHRpb25LZXlTdG9yZVByb3ZpZGVyczogdW5kZWZpbmVkLFxuICAgICAgICBpbnN0YW5jZU5hbWU6IHVuZGVmaW5lZCxcbiAgICAgICAgaXNvbGF0aW9uTGV2ZWw6IElTT0xBVElPTl9MRVZFTC5SRUFEX0NPTU1JVFRFRCxcbiAgICAgICAgbGFuZ3VhZ2U6IERFRkFVTFRfTEFOR1VBR0UsXG4gICAgICAgIGxvY2FsQWRkcmVzczogdW5kZWZpbmVkLFxuICAgICAgICBtYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnM6IDMsXG4gICAgICAgIG11bHRpU3VibmV0RmFpbG92ZXI6IGZhbHNlLFxuICAgICAgICBwYWNrZXRTaXplOiBERUZBVUxUX1BBQ0tFVF9TSVpFLFxuICAgICAgICBwb3J0OiBERUZBVUxUX1BPUlQsXG4gICAgICAgIHJlYWRPbmx5SW50ZW50OiBmYWxzZSxcbiAgICAgICAgcmVxdWVzdFRpbWVvdXQ6IERFRkFVTFRfQ0xJRU5UX1JFUVVFU1RfVElNRU9VVCxcbiAgICAgICAgcm93Q29sbGVjdGlvbk9uRG9uZTogZmFsc2UsXG4gICAgICAgIHJvd0NvbGxlY3Rpb25PblJlcXVlc3RDb21wbGV0aW9uOiBmYWxzZSxcbiAgICAgICAgc2VydmVyTmFtZTogdW5kZWZpbmVkLFxuICAgICAgICBzZXJ2ZXJTdXBwb3J0c0NvbHVtbkVuY3J5cHRpb246IGZhbHNlLFxuICAgICAgICB0ZHNWZXJzaW9uOiBERUZBVUxUX1REU19WRVJTSU9OLFxuICAgICAgICB0ZXh0c2l6ZTogREVGQVVMVF9URVhUU0laRSxcbiAgICAgICAgdHJ1c3RlZFNlcnZlck5hbWVBRTogdW5kZWZpbmVkLFxuICAgICAgICB0cnVzdFNlcnZlckNlcnRpZmljYXRlOiB0cnVlLFxuICAgICAgICB1c2VDb2x1bW5OYW1lczogZmFsc2UsXG4gICAgICAgIHVzZVVUQzogdHJ1ZSxcbiAgICAgICAgd29ya3N0YXRpb25JZDogdW5kZWZpbmVkLFxuICAgICAgICBsb3dlckNhc2VHdWlkczogZmFsc2VcbiAgICAgIH1cbiAgICB9O1xuXG4gICAgaWYgKGNvbmZpZy5vcHRpb25zKSB7XG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMucG9ydCAmJiBjb25maWcub3B0aW9ucy5pbnN0YW5jZU5hbWUpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQb3J0IGFuZCBpbnN0YW5jZU5hbWUgYXJlIG11dHVhbGx5IGV4Y2x1c2l2ZSwgYnV0ICcgKyBjb25maWcub3B0aW9ucy5wb3J0ICsgJyBhbmQgJyArIGNvbmZpZy5vcHRpb25zLmluc3RhbmNlTmFtZSArICcgcHJvdmlkZWQnKTtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmFib3J0VHJhbnNhY3Rpb25PbkVycm9yICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5hYm9ydFRyYW5zYWN0aW9uT25FcnJvciAhPT0gJ2Jvb2xlYW4nICYmIGNvbmZpZy5vcHRpb25zLmFib3J0VHJhbnNhY3Rpb25PbkVycm9yICE9PSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuYWJvcnRUcmFuc2FjdGlvbk9uRXJyb3JcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgc3RyaW5nIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmFib3J0VHJhbnNhY3Rpb25PbkVycm9yID0gY29uZmlnLm9wdGlvbnMuYWJvcnRUcmFuc2FjdGlvbk9uRXJyb3I7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5hcHBOYW1lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5hcHBOYW1lICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmFwcE5hbWVcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgc3RyaW5nLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5hcHBOYW1lID0gY29uZmlnLm9wdGlvbnMuYXBwTmFtZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmNhbWVsQ2FzZUNvbHVtbnMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmNhbWVsQ2FzZUNvbHVtbnMgIT09ICdib29sZWFuJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmNhbWVsQ2FzZUNvbHVtbnNcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuY2FtZWxDYXNlQ29sdW1ucyA9IGNvbmZpZy5vcHRpb25zLmNhbWVsQ2FzZUNvbHVtbnM7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5jYW5jZWxUaW1lb3V0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5jYW5jZWxUaW1lb3V0ICE9PSAnbnVtYmVyJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmNhbmNlbFRpbWVvdXRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgbnVtYmVyLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5jYW5jZWxUaW1lb3V0ID0gY29uZmlnLm9wdGlvbnMuY2FuY2VsVGltZW91dDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmNvbHVtbk5hbWVSZXBsYWNlcikge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmNvbHVtbk5hbWVSZXBsYWNlciAhPT0gJ2Z1bmN0aW9uJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmNhbmNlbFRpbWVvdXRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgZnVuY3Rpb24uJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmNvbHVtbk5hbWVSZXBsYWNlciA9IGNvbmZpZy5vcHRpb25zLmNvbHVtbk5hbWVSZXBsYWNlcjtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmNvbm5lY3RUaW1lb3V0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5jb25uZWN0VGltZW91dCAhPT0gJ251bWJlcicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5jb25uZWN0VGltZW91dFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBudW1iZXIuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmNvbm5lY3RUaW1lb3V0ID0gY29uZmlnLm9wdGlvbnMuY29ubmVjdFRpbWVvdXQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5jb25uZWN0aW9uSXNvbGF0aW9uTGV2ZWwgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBhc3NlcnRWYWxpZElzb2xhdGlvbkxldmVsKGNvbmZpZy5vcHRpb25zLmNvbm5lY3Rpb25Jc29sYXRpb25MZXZlbCwgJ2NvbmZpZy5vcHRpb25zLmNvbm5lY3Rpb25Jc29sYXRpb25MZXZlbCcpO1xuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuY29ubmVjdGlvbklzb2xhdGlvbkxldmVsID0gY29uZmlnLm9wdGlvbnMuY29ubmVjdGlvbklzb2xhdGlvbkxldmVsO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMuY29ubmVjdFRpbWVvdXQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmNvbm5lY3RUaW1lb3V0ICE9PSAnbnVtYmVyJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmNvbm5lY3RUaW1lb3V0XCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIG51bWJlci4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuY29ubmVjdFRpbWVvdXQgPSBjb25maWcub3B0aW9ucy5jb25uZWN0VGltZW91dDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmNyeXB0b0NyZWRlbnRpYWxzRGV0YWlscyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMuY3J5cHRvQ3JlZGVudGlhbHNEZXRhaWxzICE9PSAnb2JqZWN0JyB8fCBjb25maWcub3B0aW9ucy5jcnlwdG9DcmVkZW50aWFsc0RldGFpbHMgPT09IG51bGwpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5jcnlwdG9DcmVkZW50aWFsc0RldGFpbHNcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgT2JqZWN0LicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5jcnlwdG9DcmVkZW50aWFsc0RldGFpbHMgPSBjb25maWcub3B0aW9ucy5jcnlwdG9DcmVkZW50aWFsc0RldGFpbHM7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5kYXRhYmFzZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMuZGF0YWJhc2UgIT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuZGF0YWJhc2VcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgc3RyaW5nLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5kYXRhYmFzZSA9IGNvbmZpZy5vcHRpb25zLmRhdGFiYXNlO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZGF0ZWZpcnN0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5kYXRlZmlyc3QgIT09ICdudW1iZXInICYmIGNvbmZpZy5vcHRpb25zLmRhdGVmaXJzdCAhPT0gbnVsbCkge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmRhdGVmaXJzdFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBudW1iZXIuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZGF0ZWZpcnN0ICE9PSBudWxsICYmIChjb25maWcub3B0aW9ucy5kYXRlZmlyc3QgPCAxIHx8IGNvbmZpZy5vcHRpb25zLmRhdGVmaXJzdCA+IDcpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFJhbmdlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmRhdGVmaXJzdFwiIHByb3BlcnR5IG11c3QgYmUgPj0gMSBhbmQgPD0gNycpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5kYXRlZmlyc3QgPSBjb25maWcub3B0aW9ucy5kYXRlZmlyc3Q7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5kYXRlRm9ybWF0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5kYXRlRm9ybWF0ICE9PSAnc3RyaW5nJyAmJiBjb25maWcub3B0aW9ucy5kYXRlRm9ybWF0ICE9PSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuZGF0ZUZvcm1hdFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcgb3IgbnVsbC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuZGF0ZUZvcm1hdCA9IGNvbmZpZy5vcHRpb25zLmRhdGVGb3JtYXQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5kZWJ1Zykge1xuICAgICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZGVidWcuZGF0YSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5kZWJ1Zy5kYXRhICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmRlYnVnLmRhdGFcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmRlYnVnLmRhdGEgPSBjb25maWcub3B0aW9ucy5kZWJ1Zy5kYXRhO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmRlYnVnLnBhY2tldCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5kZWJ1Zy5wYWNrZXQgIT09ICdib29sZWFuJykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuZGVidWcucGFja2V0XCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4uJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5kZWJ1Zy5wYWNrZXQgPSBjb25maWcub3B0aW9ucy5kZWJ1Zy5wYWNrZXQ7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZGVidWcucGF5bG9hZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5kZWJ1Zy5wYXlsb2FkICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmRlYnVnLnBheWxvYWRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmRlYnVnLnBheWxvYWQgPSBjb25maWcub3B0aW9ucy5kZWJ1Zy5wYXlsb2FkO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmRlYnVnLnRva2VuICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmRlYnVnLnRva2VuICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmRlYnVnLnRva2VuXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4uJyk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5kZWJ1Zy50b2tlbiA9IGNvbmZpZy5vcHRpb25zLmRlYnVnLnRva2VuO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5lbmFibGVBbnNpTnVsbCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGwgIT09ICdib29sZWFuJyAmJiBjb25maWcub3B0aW9ucy5lbmFibGVBbnNpTnVsbCAhPT0gbnVsbCkge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lOdWxsXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4gb3IgbnVsbC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGwgPSBjb25maWcub3B0aW9ucy5lbmFibGVBbnNpTnVsbDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lOdWxsRGVmYXVsdCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGxEZWZhdWx0ICE9PSAnYm9vbGVhbicgJiYgY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGxEZWZhdWx0ICE9PSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGxEZWZhdWx0XCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4gb3IgbnVsbC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGxEZWZhdWx0ID0gY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGxEZWZhdWx0O1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaVBhZGRpbmcgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lQYWRkaW5nICE9PSAnYm9vbGVhbicgJiYgY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaVBhZGRpbmcgIT09IG51bGwpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5lbmFibGVBbnNpUGFkZGluZ1wiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBib29sZWFuIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lQYWRkaW5nID0gY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaVBhZGRpbmc7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5lbmFibGVBbnNpV2FybmluZ3MgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lXYXJuaW5ncyAhPT0gJ2Jvb2xlYW4nICYmIGNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lXYXJuaW5ncyAhPT0gbnVsbCkge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lXYXJuaW5nc1wiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBib29sZWFuIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lXYXJuaW5ncyA9IGNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lXYXJuaW5ncztcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmVuYWJsZUFyaXRoQWJvcnQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmVuYWJsZUFyaXRoQWJvcnQgIT09ICdib29sZWFuJyAmJiBjb25maWcub3B0aW9ucy5lbmFibGVBcml0aEFib3J0ICE9PSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuZW5hYmxlQXJpdGhBYm9ydFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBib29sZWFuIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUFyaXRoQWJvcnQgPSBjb25maWcub3B0aW9ucy5lbmFibGVBcml0aEFib3J0O1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZW5hYmxlQ29uY2F0TnVsbFlpZWxkc051bGwgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmVuYWJsZUNvbmNhdE51bGxZaWVsZHNOdWxsICE9PSAnYm9vbGVhbicgJiYgY29uZmlnLm9wdGlvbnMuZW5hYmxlQ29uY2F0TnVsbFlpZWxkc051bGwgIT09IG51bGwpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5lbmFibGVDb25jYXROdWxsWWllbGRzTnVsbFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBib29sZWFuIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUNvbmNhdE51bGxZaWVsZHNOdWxsID0gY29uZmlnLm9wdGlvbnMuZW5hYmxlQ29uY2F0TnVsbFlpZWxkc051bGw7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5lbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5lbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0ICE9PSAnYm9vbGVhbicgJiYgY29uZmlnLm9wdGlvbnMuZW5hYmxlQ3Vyc29yQ2xvc2VPbkNvbW1pdCAhPT0gbnVsbCkge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmVuYWJsZUN1cnNvckNsb3NlT25Db21taXRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbiBvciBudWxsLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0ID0gY29uZmlnLm9wdGlvbnMuZW5hYmxlQ3Vyc29yQ2xvc2VPbkNvbW1pdDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmVuYWJsZUltcGxpY2l0VHJhbnNhY3Rpb25zICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5lbmFibGVJbXBsaWNpdFRyYW5zYWN0aW9ucyAhPT0gJ2Jvb2xlYW4nICYmIGNvbmZpZy5vcHRpb25zLmVuYWJsZUltcGxpY2l0VHJhbnNhY3Rpb25zICE9PSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuZW5hYmxlSW1wbGljaXRUcmFuc2FjdGlvbnNcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbiBvciBudWxsLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVJbXBsaWNpdFRyYW5zYWN0aW9ucyA9IGNvbmZpZy5vcHRpb25zLmVuYWJsZUltcGxpY2l0VHJhbnNhY3Rpb25zO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMuZW5hYmxlTnVtZXJpY1JvdW5kYWJvcnQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmVuYWJsZU51bWVyaWNSb3VuZGFib3J0ICE9PSAnYm9vbGVhbicgJiYgY29uZmlnLm9wdGlvbnMuZW5hYmxlTnVtZXJpY1JvdW5kYWJvcnQgIT09IG51bGwpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5lbmFibGVOdW1lcmljUm91bmRhYm9ydFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBib29sZWFuIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZU51bWVyaWNSb3VuZGFib3J0ID0gY29uZmlnLm9wdGlvbnMuZW5hYmxlTnVtZXJpY1JvdW5kYWJvcnQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5lbmFibGVRdW90ZWRJZGVudGlmaWVyICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5lbmFibGVRdW90ZWRJZGVudGlmaWVyICE9PSAnYm9vbGVhbicgJiYgY29uZmlnLm9wdGlvbnMuZW5hYmxlUXVvdGVkSWRlbnRpZmllciAhPT0gbnVsbCkge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmVuYWJsZVF1b3RlZElkZW50aWZpZXJcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbiBvciBudWxsLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVRdW90ZWRJZGVudGlmaWVyID0gY29uZmlnLm9wdGlvbnMuZW5hYmxlUXVvdGVkSWRlbnRpZmllcjtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmVuY3J5cHQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLmVuY3J5cHQgIT09ICdib29sZWFuJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmVuY3J5cHRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuZW5jcnlwdCA9IGNvbmZpZy5vcHRpb25zLmVuY3J5cHQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5mYWxsYmFja1RvRGVmYXVsdERiICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5mYWxsYmFja1RvRGVmYXVsdERiICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5mYWxsYmFja1RvRGVmYXVsdERiXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4uJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmZhbGxiYWNrVG9EZWZhdWx0RGIgPSBjb25maWcub3B0aW9ucy5mYWxsYmFja1RvRGVmYXVsdERiO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMuaW5zdGFuY2VOYW1lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5pbnN0YW5jZU5hbWUgIT09ICdzdHJpbmcnKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuaW5zdGFuY2VOYW1lXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuaW5zdGFuY2VOYW1lID0gY29uZmlnLm9wdGlvbnMuaW5zdGFuY2VOYW1lO1xuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLnBvcnQgPSB1bmRlZmluZWQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5pc29sYXRpb25MZXZlbCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGFzc2VydFZhbGlkSXNvbGF0aW9uTGV2ZWwoY29uZmlnLm9wdGlvbnMuaXNvbGF0aW9uTGV2ZWwsICdjb25maWcub3B0aW9ucy5pc29sYXRpb25MZXZlbCcpO1xuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMuaXNvbGF0aW9uTGV2ZWwgPSBjb25maWcub3B0aW9ucy5pc29sYXRpb25MZXZlbDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmxhbmd1YWdlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5sYW5ndWFnZSAhPT0gJ3N0cmluZycgJiYgY29uZmlnLm9wdGlvbnMubGFuZ3VhZ2UgIT09IG51bGwpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5sYW5ndWFnZVwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcgb3IgbnVsbC4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMubGFuZ3VhZ2UgPSBjb25maWcub3B0aW9ucy5sYW5ndWFnZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmxvY2FsQWRkcmVzcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMubG9jYWxBZGRyZXNzICE9PSAnc3RyaW5nJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmxvY2FsQWRkcmVzc1wiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBzdHJpbmcuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmxvY2FsQWRkcmVzcyA9IGNvbmZpZy5vcHRpb25zLmxvY2FsQWRkcmVzcztcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLm11bHRpU3VibmV0RmFpbG92ZXIgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLm11bHRpU3VibmV0RmFpbG92ZXIgIT09ICdib29sZWFuJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLm11bHRpU3VibmV0RmFpbG92ZXJcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMubXVsdGlTdWJuZXRGYWlsb3ZlciA9IGNvbmZpZy5vcHRpb25zLm11bHRpU3VibmV0RmFpbG92ZXI7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5wYWNrZXRTaXplICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5wYWNrZXRTaXplICE9PSAnbnVtYmVyJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLnBhY2tldFNpemVcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgbnVtYmVyLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5wYWNrZXRTaXplID0gY29uZmlnLm9wdGlvbnMucGFja2V0U2l6ZTtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLnBvcnQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLnBvcnQgIT09ICdudW1iZXInKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMucG9ydFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBudW1iZXIuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoY29uZmlnLm9wdGlvbnMucG9ydCA8PSAwIHx8IGNvbmZpZy5vcHRpb25zLnBvcnQgPj0gNjU1MzYpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgUmFuZ2VFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMucG9ydFwiIHByb3BlcnR5IG11c3QgYmUgPiAwIGFuZCA8IDY1NTM2Jyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLnBvcnQgPSBjb25maWcub3B0aW9ucy5wb3J0O1xuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmluc3RhbmNlTmFtZSA9IHVuZGVmaW5lZDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLnJlYWRPbmx5SW50ZW50ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5yZWFkT25seUludGVudCAhPT0gJ2Jvb2xlYW4nKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMucmVhZE9ubHlJbnRlbnRcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMucmVhZE9ubHlJbnRlbnQgPSBjb25maWcub3B0aW9ucy5yZWFkT25seUludGVudDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLnJlcXVlc3RUaW1lb3V0ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5yZXF1ZXN0VGltZW91dCAhPT0gJ251bWJlcicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5yZXF1ZXN0VGltZW91dFwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBudW1iZXIuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLnJlcXVlc3RUaW1lb3V0ID0gY29uZmlnLm9wdGlvbnMucmVxdWVzdFRpbWVvdXQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5tYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLm1heFJldHJpZXNPblRyYW5zaWVudEVycm9ycyAhPT0gJ251bWJlcicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5tYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnNcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgbnVtYmVyLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLm1heFJldHJpZXNPblRyYW5zaWVudEVycm9ycyA8IDApIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5tYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnNcIiBwcm9wZXJ0eSBtdXN0IGJlIGVxdWFsIG9yIGdyZWF0ZXIgdGhhbiAwLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5tYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnMgPSBjb25maWcub3B0aW9ucy5tYXhSZXRyaWVzT25UcmFuc2llbnRFcnJvcnM7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5jb25uZWN0aW9uUmV0cnlJbnRlcnZhbCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMuY29ubmVjdGlvblJldHJ5SW50ZXJ2YWwgIT09ICdudW1iZXInKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMuY29ubmVjdGlvblJldHJ5SW50ZXJ2YWxcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgbnVtYmVyLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLmNvbm5lY3Rpb25SZXRyeUludGVydmFsIDw9IDApIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5jb25uZWN0aW9uUmV0cnlJbnRlcnZhbFwiIHByb3BlcnR5IG11c3QgYmUgZ3JlYXRlciB0aGFuIDAuJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmNvbm5lY3Rpb25SZXRyeUludGVydmFsID0gY29uZmlnLm9wdGlvbnMuY29ubmVjdGlvblJldHJ5SW50ZXJ2YWw7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5yb3dDb2xsZWN0aW9uT25Eb25lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy5yb3dDb2xsZWN0aW9uT25Eb25lICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5yb3dDb2xsZWN0aW9uT25Eb25lXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4uJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLnJvd0NvbGxlY3Rpb25PbkRvbmUgPSBjb25maWcub3B0aW9ucy5yb3dDb2xsZWN0aW9uT25Eb25lO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMucm93Q29sbGVjdGlvbk9uUmVxdWVzdENvbXBsZXRpb24gIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLnJvd0NvbGxlY3Rpb25PblJlcXVlc3RDb21wbGV0aW9uICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy5yb3dDb2xsZWN0aW9uT25SZXF1ZXN0Q29tcGxldGlvblwiIHByb3BlcnR5IG11c3QgYmUgb2YgdHlwZSBib29sZWFuLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy5yb3dDb2xsZWN0aW9uT25SZXF1ZXN0Q29tcGxldGlvbiA9IGNvbmZpZy5vcHRpb25zLnJvd0NvbGxlY3Rpb25PblJlcXVlc3RDb21wbGV0aW9uO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMudGRzVmVyc2lvbiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMudGRzVmVyc2lvbiAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy50ZHNWZXJzaW9uXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMudGRzVmVyc2lvbiA9IGNvbmZpZy5vcHRpb25zLnRkc1ZlcnNpb247XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy50ZXh0c2l6ZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMudGV4dHNpemUgIT09ICdudW1iZXInICYmIGNvbmZpZy5vcHRpb25zLnRleHRzaXplICE9PSBudWxsKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMudGV4dHNpemVcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgbnVtYmVyIG9yIG51bGwuJyk7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAoY29uZmlnLm9wdGlvbnMudGV4dHNpemUgPiAyMTQ3NDgzNjQ3KSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMudGV4dHNpemVcIiBjYW5cXCd0IGJlIGdyZWF0ZXIgdGhhbiAyMTQ3NDgzNjQ3LicpO1xuICAgICAgICB9IGVsc2UgaWYgKGNvbmZpZy5vcHRpb25zLnRleHRzaXplIDwgLTEpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy50ZXh0c2l6ZVwiIGNhblxcJ3QgYmUgc21hbGxlciB0aGFuIC0xLicpO1xuICAgICAgICB9XG5cbiAgICAgICAgdGhpcy5jb25maWcub3B0aW9ucy50ZXh0c2l6ZSA9IGNvbmZpZy5vcHRpb25zLnRleHRzaXplIHwgMDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLnRydXN0U2VydmVyQ2VydGlmaWNhdGUgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBpZiAodHlwZW9mIGNvbmZpZy5vcHRpb25zLnRydXN0U2VydmVyQ2VydGlmaWNhdGUgIT09ICdib29sZWFuJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLnRydXN0U2VydmVyQ2VydGlmaWNhdGVcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMudHJ1c3RTZXJ2ZXJDZXJ0aWZpY2F0ZSA9IGNvbmZpZy5vcHRpb25zLnRydXN0U2VydmVyQ2VydGlmaWNhdGU7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBlbWl0VHJ1c3RTZXJ2ZXJDZXJ0aWZpY2F0ZVdhcm5pbmcoKTtcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLnVzZUNvbHVtbk5hbWVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgaWYgKHR5cGVvZiBjb25maWcub3B0aW9ucy51c2VDb2x1bW5OYW1lcyAhPT0gJ2Jvb2xlYW4nKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IFR5cGVFcnJvcignVGhlIFwiY29uZmlnLm9wdGlvbnMudXNlQ29sdW1uTmFtZXNcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMudXNlQ29sdW1uTmFtZXMgPSBjb25maWcub3B0aW9ucy51c2VDb2x1bW5OYW1lcztcbiAgICAgIH1cblxuICAgICAgaWYgKGNvbmZpZy5vcHRpb25zLnVzZVVUQyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMudXNlVVRDICE9PSAnYm9vbGVhbicpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy51c2VVVENcIiBwcm9wZXJ0eSBtdXN0IGJlIG9mIHR5cGUgYm9vbGVhbi4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMudXNlVVRDID0gY29uZmlnLm9wdGlvbnMudXNlVVRDO1xuICAgICAgfVxuXG4gICAgICBpZiAoY29uZmlnLm9wdGlvbnMud29ya3N0YXRpb25JZCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMud29ya3N0YXRpb25JZCAhPT0gJ3N0cmluZycpIHtcbiAgICAgICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdUaGUgXCJjb25maWcub3B0aW9ucy53b3Jrc3RhdGlvbklkXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIHN0cmluZy4nKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHRoaXMuY29uZmlnLm9wdGlvbnMud29ya3N0YXRpb25JZCA9IGNvbmZpZy5vcHRpb25zLndvcmtzdGF0aW9uSWQ7XG4gICAgICB9XG5cbiAgICAgIGlmIChjb25maWcub3B0aW9ucy5sb3dlckNhc2VHdWlkcyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh0eXBlb2YgY29uZmlnLm9wdGlvbnMubG93ZXJDYXNlR3VpZHMgIT09ICdib29sZWFuJykge1xuICAgICAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoJ1RoZSBcImNvbmZpZy5vcHRpb25zLmxvd2VyQ2FzZUd1aWRzXCIgcHJvcGVydHkgbXVzdCBiZSBvZiB0eXBlIGJvb2xlYW4uJyk7XG4gICAgICAgIH1cblxuICAgICAgICB0aGlzLmNvbmZpZy5vcHRpb25zLmxvd2VyQ2FzZUd1aWRzID0gY29uZmlnLm9wdGlvbnMubG93ZXJDYXNlR3VpZHM7XG4gICAgICB9XG4gICAgfVxuXG4gICAgbGV0IGNyZWRlbnRpYWxzRGV0YWlscyA9IHRoaXMuY29uZmlnLm9wdGlvbnMuY3J5cHRvQ3JlZGVudGlhbHNEZXRhaWxzO1xuICAgIGlmIChjcmVkZW50aWFsc0RldGFpbHMuc2VjdXJlT3B0aW9ucyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAvLyBJZiB0aGUgY2FsbGVyIGhhcyBub3Qgc3BlY2lmaWVkIHRoZWlyIG93biBgc2VjdXJlT3B0aW9uc2AsXG4gICAgICAvLyB3ZSBzZXQgYFNTTF9PUF9ET05UX0lOU0VSVF9FTVBUWV9GUkFHTUVOVFNgIGhlcmUuXG4gICAgICAvLyBPbGRlciBTUUwgU2VydmVyIGluc3RhbmNlcyBydW5uaW5nIG9uIG9sZGVyIFdpbmRvd3MgdmVyc2lvbnMgaGF2ZVxuICAgICAgLy8gdHJvdWJsZSB3aXRoIHRoZSBCRUFTVCB3b3JrYXJvdW5kIGluIE9wZW5TU0wuXG4gICAgICAvLyBBcyBCRUFTVCBpcyBhIGJyb3dzZXIgc3BlY2lmaWMgZXhwbG9pdCwgd2UgY2FuIGp1c3QgZGlzYWJsZSB0aGlzIG9wdGlvbiBoZXJlLlxuICAgICAgY3JlZGVudGlhbHNEZXRhaWxzID0gT2JqZWN0LmNyZWF0ZShjcmVkZW50aWFsc0RldGFpbHMsIHtcbiAgICAgICAgc2VjdXJlT3B0aW9uczoge1xuICAgICAgICAgIHZhbHVlOiBjb25zdGFudHMuU1NMX09QX0RPTlRfSU5TRVJUX0VNUFRZX0ZSQUdNRU5UU1xuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9XG5cbiAgICB0aGlzLnNlY3VyZUNvbnRleHQgPSBjcmVhdGVTZWN1cmVDb250ZXh0KGNyZWRlbnRpYWxzRGV0YWlscyk7XG5cbiAgICB0aGlzLmRlYnVnID0gdGhpcy5jcmVhdGVEZWJ1ZygpO1xuICAgIHRoaXMuaW5UcmFuc2FjdGlvbiA9IGZhbHNlO1xuICAgIHRoaXMudHJhbnNhY3Rpb25EZXNjcmlwdG9ycyA9IFtCdWZmZXIuZnJvbShbMCwgMCwgMCwgMCwgMCwgMCwgMCwgMF0pXTtcblxuICAgIC8vICdiZWdpblRyYW5zYWN0aW9uJywgJ2NvbW1pdFRyYW5zYWN0aW9uJyBhbmQgJ3JvbGxiYWNrVHJhbnNhY3Rpb24nXG4gICAgLy8gZXZlbnRzIGFyZSB1dGlsaXplZCB0byBtYWludGFpbiBpblRyYW5zYWN0aW9uIHByb3BlcnR5IHN0YXRlIHdoaWNoIGluXG4gICAgLy8gdHVybiBpcyB1c2VkIGluIG1hbmFnaW5nIHRyYW5zYWN0aW9ucy4gVGhlc2UgZXZlbnRzIGFyZSBvbmx5IGZpcmVkIGZvclxuICAgIC8vIFREUyB2ZXJzaW9uIDcuMiBhbmQgYmV5b25kLiBUaGUgcHJvcGVydGllcyBiZWxvdyBhcmUgdXNlZCB0byBlbXVsYXRlXG4gICAgLy8gZXF1aXZhbGVudCBiZWhhdmlvciBmb3IgVERTIHZlcnNpb25zIGJlZm9yZSA3LjIuXG4gICAgdGhpcy50cmFuc2FjdGlvbkRlcHRoID0gMDtcbiAgICB0aGlzLmlzU3FsQmF0Y2ggPSBmYWxzZTtcbiAgICB0aGlzLmNsb3NlZCA9IGZhbHNlO1xuICAgIHRoaXMubWVzc2FnZUJ1ZmZlciA9IEJ1ZmZlci5hbGxvYygwKTtcblxuICAgIHRoaXMuY3VyVHJhbnNpZW50UmV0cnlDb3VudCA9IDA7XG4gICAgdGhpcy50cmFuc2llbnRFcnJvckxvb2t1cCA9IG5ldyBUcmFuc2llbnRFcnJvckxvb2t1cCgpO1xuXG4gICAgdGhpcy5zdGF0ZSA9IHRoaXMuU1RBVEUuSU5JVElBTElaRUQ7XG5cbiAgICB0aGlzLl9jYW5jZWxBZnRlclJlcXVlc3RTZW50ID0gKCkgPT4ge1xuICAgICAgdGhpcy5tZXNzYWdlSW8uc2VuZE1lc3NhZ2UoVFlQRS5BVFRFTlRJT04pO1xuICAgICAgdGhpcy5jcmVhdGVDYW5jZWxUaW1lcigpO1xuICAgIH07XG4gIH1cblxuICBjb25uZWN0KGNvbm5lY3RMaXN0ZW5lcj86IChlcnI/OiBFcnJvcikgPT4gdm9pZCkge1xuICAgIGlmICh0aGlzLnN0YXRlICE9PSB0aGlzLlNUQVRFLklOSVRJQUxJWkVEKSB7XG4gICAgICB0aHJvdyBuZXcgQ29ubmVjdGlvbkVycm9yKCdgLmNvbm5lY3RgIGNhbiBub3QgYmUgY2FsbGVkIG9uIGEgQ29ubmVjdGlvbiBpbiBgJyArIHRoaXMuc3RhdGUubmFtZSArICdgIHN0YXRlLicpO1xuICAgIH1cblxuICAgIGlmIChjb25uZWN0TGlzdGVuZXIpIHtcbiAgICAgIGNvbnN0IG9uQ29ubmVjdCA9IChlcnI/OiBFcnJvcikgPT4ge1xuICAgICAgICB0aGlzLnJlbW92ZUxpc3RlbmVyKCdlcnJvcicsIG9uRXJyb3IpO1xuICAgICAgICBjb25uZWN0TGlzdGVuZXIoZXJyKTtcbiAgICAgIH07XG5cbiAgICAgIGNvbnN0IG9uRXJyb3IgPSAoZXJyOiBFcnJvcikgPT4ge1xuICAgICAgICB0aGlzLnJlbW92ZUxpc3RlbmVyKCdjb25uZWN0Jywgb25Db25uZWN0KTtcbiAgICAgICAgY29ubmVjdExpc3RlbmVyKGVycik7XG4gICAgICB9O1xuXG4gICAgICB0aGlzLm9uY2UoJ2Nvbm5lY3QnLCBvbkNvbm5lY3QpO1xuICAgICAgdGhpcy5vbmNlKCdlcnJvcicsIG9uRXJyb3IpO1xuICAgIH1cblxuICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuQ09OTkVDVElORyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHNlcnZlciBoYXMgcmVwb3J0ZWQgdGhhdCB0aGUgY2hhcnNldCBoYXMgY2hhbmdlZC5cbiAgICovXG4gIG9uKGV2ZW50OiAnY2hhcnNldENoYW5nZScsIGxpc3RlbmVyOiAoY2hhcnNldDogc3RyaW5nKSA9PiB2b2lkKTogdGhpc1xuXG4gIC8qKlxuICAgKiBUaGUgYXR0ZW1wdCB0byBjb25uZWN0IGFuZCB2YWxpZGF0ZSBoYXMgY29tcGxldGVkLlxuICAgKi9cbiAgb24oXG4gICAgZXZlbnQ6ICdjb25uZWN0JyxcbiAgICAvKipcbiAgICAgKiBAcGFyYW0gZXJyIElmIHN1Y2Nlc3NmdWxseSBjb25uZWN0ZWQsIHdpbGwgYmUgZmFsc2V5LiBJZiB0aGVyZSB3YXMgYVxuICAgICAqICAgcHJvYmxlbSAod2l0aCBlaXRoZXIgY29ubmVjdGluZyBvciB2YWxpZGF0aW9uKSwgd2lsbCBiZSBhbiBbW0Vycm9yXV0gb2JqZWN0LlxuICAgICAqL1xuICAgIGxpc3RlbmVyOiAoZXJyOiBFcnJvciB8IHVuZGVmaW5lZCkgPT4gdm9pZFxuICApOiB0aGlzXG5cbiAgLyoqXG4gICAqIFRoZSBzZXJ2ZXIgaGFzIHJlcG9ydGVkIHRoYXQgdGhlIGFjdGl2ZSBkYXRhYmFzZSBoYXMgY2hhbmdlZC5cbiAgICogVGhpcyBtYXkgYmUgYXMgYSByZXN1bHQgb2YgYSBzdWNjZXNzZnVsIGxvZ2luLCBvciBhIGB1c2VgIHN0YXRlbWVudC5cbiAgICovXG4gIG9uKGV2ZW50OiAnZGF0YWJhc2VDaGFuZ2UnLCBsaXN0ZW5lcjogKGRhdGFiYXNlTmFtZTogc3RyaW5nKSA9PiB2b2lkKTogdGhpc1xuXG4gIC8qKlxuICAgKiBBIGRlYnVnIG1lc3NhZ2UgaXMgYXZhaWxhYmxlLiBJdCBtYXkgYmUgbG9nZ2VkIG9yIGlnbm9yZWQuXG4gICAqL1xuICBvbihldmVudDogJ2RlYnVnJywgbGlzdGVuZXI6IChtZXNzYWdlVGV4dDogc3RyaW5nKSA9PiB2b2lkKTogdGhpc1xuXG4gIC8qKlxuICAgKiBJbnRlcm5hbCBlcnJvciBvY2N1cnMuXG4gICAqL1xuICBvbihldmVudDogJ2Vycm9yJywgbGlzdGVuZXI6IChlcnI6IEVycm9yKSA9PiB2b2lkKTogdGhpc1xuXG4gIC8qKlxuICAgKiBUaGUgc2VydmVyIGhhcyBpc3N1ZWQgYW4gZXJyb3IgbWVzc2FnZS5cbiAgICovXG4gIG9uKGV2ZW50OiAnZXJyb3JNZXNzYWdlJywgbGlzdGVuZXI6IChtZXNzYWdlOiBpbXBvcnQoJy4vdG9rZW4vdG9rZW4nKS5FcnJvck1lc3NhZ2VUb2tlbikgPT4gdm9pZCk6IHRoaXNcblxuICAvKipcbiAgICogVGhlIGNvbm5lY3Rpb24gaGFzIGVuZGVkLlxuICAgKlxuICAgKiBUaGlzIG1heSBiZSBhcyBhIHJlc3VsdCBvZiB0aGUgY2xpZW50IGNhbGxpbmcgW1tjbG9zZV1dLCB0aGUgc2VydmVyXG4gICAqIGNsb3NpbmcgdGhlIGNvbm5lY3Rpb24sIG9yIGEgbmV0d29yayBlcnJvci5cbiAgICovXG4gIG9uKGV2ZW50OiAnZW5kJywgbGlzdGVuZXI6ICgpID0+IHZvaWQpOiB0aGlzXG5cbiAgLyoqXG4gICAqIFRoZSBzZXJ2ZXIgaGFzIGlzc3VlZCBhbiBpbmZvcm1hdGlvbiBtZXNzYWdlLlxuICAgKi9cbiAgb24oZXZlbnQ6ICdpbmZvTWVzc2FnZScsIGxpc3RlbmVyOiAobWVzc2FnZTogaW1wb3J0KCcuL3Rva2VuL3Rva2VuJykuSW5mb01lc3NhZ2VUb2tlbikgPT4gdm9pZCk6IHRoaXNcblxuICAvKipcbiAgICogVGhlIHNlcnZlciBoYXMgcmVwb3J0ZWQgdGhhdCB0aGUgbGFuZ3VhZ2UgaGFzIGNoYW5nZWQuXG4gICAqL1xuICBvbihldmVudDogJ2xhbmd1YWdlQ2hhbmdlJywgbGlzdGVuZXI6IChsYW5ndWFnZU5hbWU6IHN0cmluZykgPT4gdm9pZCk6IHRoaXNcblxuICAvKipcbiAgICogVGhlIGNvbm5lY3Rpb24gd2FzIHJlc2V0LlxuICAgKi9cbiAgb24oZXZlbnQ6ICdyZXNldENvbm5lY3Rpb24nLCBsaXN0ZW5lcjogKCkgPT4gdm9pZCk6IHRoaXNcblxuICAvKipcbiAgICogQSBzZWN1cmUgY29ubmVjdGlvbiBoYXMgYmVlbiBlc3RhYmxpc2hlZC5cbiAgICovXG4gIG9uKGV2ZW50OiAnc2VjdXJlJywgbGlzdGVuZXI6IChjbGVhcnRleHQ6IGltcG9ydCgndGxzJykuVExTU29ja2V0KSA9PiB2b2lkKTogdGhpc1xuXG4gIG9uKGV2ZW50OiBzdHJpbmcgfCBzeW1ib2wsIGxpc3RlbmVyOiAoLi4uYXJnczogYW55W10pID0+IHZvaWQpIHtcbiAgICByZXR1cm4gc3VwZXIub24oZXZlbnQsIGxpc3RlbmVyKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZW1pdChldmVudDogJ2NoYXJzZXRDaGFuZ2UnLCBjaGFyc2V0OiBzdHJpbmcpOiBib29sZWFuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZW1pdChldmVudDogJ2Nvbm5lY3QnLCBlcnJvcj86IEVycm9yKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdkYXRhYmFzZUNoYW5nZScsIGRhdGFiYXNlTmFtZTogc3RyaW5nKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdkZWJ1ZycsIG1lc3NhZ2VUZXh0OiBzdHJpbmcpOiBib29sZWFuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZW1pdChldmVudDogJ2Vycm9yJywgZXJyb3I6IEVycm9yKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdlcnJvck1lc3NhZ2UnLCBtZXNzYWdlOiBpbXBvcnQoJy4vdG9rZW4vdG9rZW4nKS5FcnJvck1lc3NhZ2VUb2tlbik6IGJvb2xlYW5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBlbWl0KGV2ZW50OiAnZW5kJyk6IGJvb2xlYW5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBlbWl0KGV2ZW50OiAnaW5mb01lc3NhZ2UnLCBtZXNzYWdlOiBpbXBvcnQoJy4vdG9rZW4vdG9rZW4nKS5JbmZvTWVzc2FnZVRva2VuKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdsYW5ndWFnZUNoYW5nZScsIGxhbmd1YWdlTmFtZTogc3RyaW5nKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdzZWN1cmUnLCBjbGVhcnRleHQ6IGltcG9ydCgndGxzJykuVExTU29ja2V0KTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdyZXJvdXRpbmcnKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdyZXNldENvbm5lY3Rpb24nKTogYm9vbGVhblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGVtaXQoZXZlbnQ6ICdyZXRyeScpOiBib29sZWFuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZW1pdChldmVudDogJ3JvbGxiYWNrVHJhbnNhY3Rpb24nKTogYm9vbGVhblxuXG4gIGVtaXQoZXZlbnQ6IHN0cmluZyB8IHN5bWJvbCwgLi4uYXJnczogYW55W10pIHtcbiAgICByZXR1cm4gc3VwZXIuZW1pdChldmVudCwgLi4uYXJncyk7XG4gIH1cblxuICAvKipcbiAgICogQ2xvc2VzIHRoZSBjb25uZWN0aW9uIHRvIHRoZSBkYXRhYmFzZS5cbiAgICpcbiAgICogVGhlIFtbRXZlbnRfZW5kXV0gd2lsbCBiZSBlbWl0dGVkIG9uY2UgdGhlIGNvbm5lY3Rpb24gaGFzIGJlZW4gY2xvc2VkLlxuICAgKi9cbiAgY2xvc2UoKSB7XG4gICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGluaXRpYWxpc2VDb25uZWN0aW9uKCkge1xuICAgIGNvbnN0IHNpZ25hbCA9IHRoaXMuY3JlYXRlQ29ubmVjdFRpbWVyKCk7XG5cbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy5wb3J0KSB7XG4gICAgICByZXR1cm4gdGhpcy5jb25uZWN0T25Qb3J0KHRoaXMuY29uZmlnLm9wdGlvbnMucG9ydCwgdGhpcy5jb25maWcub3B0aW9ucy5tdWx0aVN1Ym5ldEZhaWxvdmVyLCBzaWduYWwpO1xuICAgIH0gZWxzZSB7XG4gICAgICByZXR1cm4gbmV3IEluc3RhbmNlTG9va3VwKCkuaW5zdGFuY2VMb29rdXAoe1xuICAgICAgICBzZXJ2ZXI6IHRoaXMuY29uZmlnLnNlcnZlcixcbiAgICAgICAgaW5zdGFuY2VOYW1lOiB0aGlzLmNvbmZpZy5vcHRpb25zLmluc3RhbmNlTmFtZSEsXG4gICAgICAgIHRpbWVvdXQ6IHRoaXMuY29uZmlnLm9wdGlvbnMuY29ubmVjdFRpbWVvdXQsXG4gICAgICAgIHNpZ25hbDogc2lnbmFsXG4gICAgICB9LCAoZXJyLCBwb3J0KSA9PiB7XG4gICAgICAgIGlmIChlcnIpIHtcbiAgICAgICAgICBpZiAoZXJyLm5hbWUgPT09ICdBYm9ydEVycm9yJykge1xuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIHRoaXMuZW1pdCgnY29ubmVjdCcsIG5ldyBDb25uZWN0aW9uRXJyb3IoZXJyLm1lc3NhZ2UsICdFSU5TVExPT0tVUCcpKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICB0aGlzLmNvbm5lY3RPblBvcnQocG9ydCEsIHRoaXMuY29uZmlnLm9wdGlvbnMubXVsdGlTdWJuZXRGYWlsb3Zlciwgc2lnbmFsKTtcbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBjbGVhbnVwQ29ubmVjdGlvbihjbGVhbnVwVHlwZTogdHlwZW9mIENMRUFOVVBfVFlQRVtrZXlvZiB0eXBlb2YgQ0xFQU5VUF9UWVBFXSkge1xuICAgIGlmICghdGhpcy5jbG9zZWQpIHtcbiAgICAgIHRoaXMuY2xlYXJDb25uZWN0VGltZXIoKTtcbiAgICAgIHRoaXMuY2xlYXJSZXF1ZXN0VGltZXIoKTtcbiAgICAgIHRoaXMuY2xlYXJSZXRyeVRpbWVyKCk7XG4gICAgICB0aGlzLmNsb3NlQ29ubmVjdGlvbigpO1xuICAgICAgaWYgKGNsZWFudXBUeXBlID09PSBDTEVBTlVQX1RZUEUuUkVESVJFQ1QpIHtcbiAgICAgICAgdGhpcy5lbWl0KCdyZXJvdXRpbmcnKTtcbiAgICAgIH0gZWxzZSBpZiAoY2xlYW51cFR5cGUgIT09IENMRUFOVVBfVFlQRS5SRVRSWSkge1xuICAgICAgICBwcm9jZXNzLm5leHRUaWNrKCgpID0+IHtcbiAgICAgICAgICB0aGlzLmVtaXQoJ2VuZCcpO1xuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgcmVxdWVzdCA9IHRoaXMucmVxdWVzdDtcbiAgICAgIGlmIChyZXF1ZXN0KSB7XG4gICAgICAgIGNvbnN0IGVyciA9IG5ldyBSZXF1ZXN0RXJyb3IoJ0Nvbm5lY3Rpb24gY2xvc2VkIGJlZm9yZSByZXF1ZXN0IGNvbXBsZXRlZC4nLCAnRUNMT1NFJyk7XG4gICAgICAgIHJlcXVlc3QuY2FsbGJhY2soZXJyKTtcbiAgICAgICAgdGhpcy5yZXF1ZXN0ID0gdW5kZWZpbmVkO1xuICAgICAgfVxuXG4gICAgICB0aGlzLmNsb3NlZCA9IHRydWU7XG4gICAgICB0aGlzLmxvZ2luRXJyb3IgPSB1bmRlZmluZWQ7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBjcmVhdGVEZWJ1ZygpIHtcbiAgICBjb25zdCBkZWJ1ZyA9IG5ldyBEZWJ1Zyh0aGlzLmNvbmZpZy5vcHRpb25zLmRlYnVnKTtcbiAgICBkZWJ1Zy5vbignZGVidWcnLCAobWVzc2FnZSkgPT4ge1xuICAgICAgdGhpcy5lbWl0KCdkZWJ1ZycsIG1lc3NhZ2UpO1xuICAgIH0pO1xuICAgIHJldHVybiBkZWJ1ZztcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY3JlYXRlVG9rZW5TdHJlYW1QYXJzZXIobWVzc2FnZTogTWVzc2FnZSwgaGFuZGxlcjogVG9rZW5IYW5kbGVyKSB7XG4gICAgcmV0dXJuIG5ldyBUb2tlblN0cmVhbVBhcnNlcihtZXNzYWdlLCB0aGlzLmRlYnVnLCBoYW5kbGVyLCB0aGlzLmNvbmZpZy5vcHRpb25zKTtcbiAgfVxuXG4gIGNvbm5lY3RPblBvcnQocG9ydDogbnVtYmVyLCBtdWx0aVN1Ym5ldEZhaWxvdmVyOiBib29sZWFuLCBzaWduYWw6IEFib3J0U2lnbmFsKSB7XG4gICAgY29uc3QgY29ubmVjdE9wdHMgPSB7XG4gICAgICBob3N0OiB0aGlzLnJvdXRpbmdEYXRhID8gdGhpcy5yb3V0aW5nRGF0YS5zZXJ2ZXIgOiB0aGlzLmNvbmZpZy5zZXJ2ZXIsXG4gICAgICBwb3J0OiB0aGlzLnJvdXRpbmdEYXRhID8gdGhpcy5yb3V0aW5nRGF0YS5wb3J0IDogcG9ydCxcbiAgICAgIGxvY2FsQWRkcmVzczogdGhpcy5jb25maWcub3B0aW9ucy5sb2NhbEFkZHJlc3NcbiAgICB9O1xuXG4gICAgbmV3IENvbm5lY3Rvcihjb25uZWN0T3B0cywgc2lnbmFsLCBtdWx0aVN1Ym5ldEZhaWxvdmVyKS5leGVjdXRlKChlcnIsIHNvY2tldCkgPT4ge1xuICAgICAgaWYgKGVycikge1xuICAgICAgICBpZiAoZXJyLm5hbWUgPT09ICdBYm9ydEVycm9yJykge1xuICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiB0aGlzLnNvY2tldEVycm9yKGVycik7XG4gICAgICB9XG5cbiAgICAgIHNvY2tldCA9IHNvY2tldCE7XG4gICAgICBzb2NrZXQub24oJ2Vycm9yJywgKGVycm9yKSA9PiB7IHRoaXMuc29ja2V0RXJyb3IoZXJyb3IpOyB9KTtcbiAgICAgIHNvY2tldC5vbignY2xvc2UnLCAoKSA9PiB7IHRoaXMuc29ja2V0Q2xvc2UoKTsgfSk7XG4gICAgICBzb2NrZXQub24oJ2VuZCcsICgpID0+IHsgdGhpcy5zb2NrZXRFbmQoKTsgfSk7XG4gICAgICBzb2NrZXQuc2V0S2VlcEFsaXZlKHRydWUsIEtFRVBfQUxJVkVfSU5JVElBTF9ERUxBWSk7XG5cbiAgICAgIHRoaXMubWVzc2FnZUlvID0gbmV3IE1lc3NhZ2VJTyhzb2NrZXQsIHRoaXMuY29uZmlnLm9wdGlvbnMucGFja2V0U2l6ZSwgdGhpcy5kZWJ1Zyk7XG4gICAgICB0aGlzLm1lc3NhZ2VJby5vbignc2VjdXJlJywgKGNsZWFydGV4dCkgPT4geyB0aGlzLmVtaXQoJ3NlY3VyZScsIGNsZWFydGV4dCk7IH0pO1xuXG4gICAgICB0aGlzLnNvY2tldCA9IHNvY2tldDtcblxuICAgICAgdGhpcy5jbG9zZWQgPSBmYWxzZTtcbiAgICAgIHRoaXMuZGVidWcubG9nKCdjb25uZWN0ZWQgdG8gJyArIHRoaXMuY29uZmlnLnNlcnZlciArICc6JyArIHRoaXMuY29uZmlnLm9wdGlvbnMucG9ydCk7XG5cbiAgICAgIHRoaXMuc2VuZFByZUxvZ2luKCk7XG4gICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLlNFTlRfUFJFTE9HSU4pO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBjbG9zZUNvbm5lY3Rpb24oKSB7XG4gICAgaWYgKHRoaXMuc29ja2V0KSB7XG4gICAgICB0aGlzLnNvY2tldC5kZXN0cm95KCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBjcmVhdGVDb25uZWN0VGltZXIoKSB7XG4gICAgY29uc3QgY29udHJvbGxlciA9IG5ldyBBYm9ydENvbnRyb2xsZXIoKTtcbiAgICB0aGlzLmNvbm5lY3RUaW1lciA9IHNldFRpbWVvdXQoKCkgPT4ge1xuICAgICAgY29udHJvbGxlci5hYm9ydCgpO1xuICAgICAgdGhpcy5jb25uZWN0VGltZW91dCgpO1xuICAgIH0sIHRoaXMuY29uZmlnLm9wdGlvbnMuY29ubmVjdFRpbWVvdXQpO1xuICAgIHJldHVybiBjb250cm9sbGVyLnNpZ25hbDtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY3JlYXRlQ2FuY2VsVGltZXIoKSB7XG4gICAgdGhpcy5jbGVhckNhbmNlbFRpbWVyKCk7XG4gICAgY29uc3QgdGltZW91dCA9IHRoaXMuY29uZmlnLm9wdGlvbnMuY2FuY2VsVGltZW91dDtcbiAgICBpZiAodGltZW91dCA+IDApIHtcbiAgICAgIHRoaXMuY2FuY2VsVGltZXIgPSBzZXRUaW1lb3V0KCgpID0+IHtcbiAgICAgICAgdGhpcy5jYW5jZWxUaW1lb3V0KCk7XG4gICAgICB9LCB0aW1lb3V0KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNyZWF0ZVJlcXVlc3RUaW1lcigpIHtcbiAgICB0aGlzLmNsZWFyUmVxdWVzdFRpbWVyKCk7IC8vIHJlbGVhc2Ugb2xkIHRpbWVyLCBqdXN0IHRvIGJlIHNhZmVcbiAgICBjb25zdCByZXF1ZXN0ID0gdGhpcy5yZXF1ZXN0IGFzIFJlcXVlc3Q7XG4gICAgY29uc3QgdGltZW91dCA9IChyZXF1ZXN0LnRpbWVvdXQgIT09IHVuZGVmaW5lZCkgPyByZXF1ZXN0LnRpbWVvdXQgOiB0aGlzLmNvbmZpZy5vcHRpb25zLnJlcXVlc3RUaW1lb3V0O1xuICAgIGlmICh0aW1lb3V0KSB7XG4gICAgICB0aGlzLnJlcXVlc3RUaW1lciA9IHNldFRpbWVvdXQoKCkgPT4ge1xuICAgICAgICB0aGlzLnJlcXVlc3RUaW1lb3V0KCk7XG4gICAgICB9LCB0aW1lb3V0KTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNyZWF0ZVJldHJ5VGltZXIoKSB7XG4gICAgdGhpcy5jbGVhclJldHJ5VGltZXIoKTtcbiAgICB0aGlzLnJldHJ5VGltZXIgPSBzZXRUaW1lb3V0KCgpID0+IHtcbiAgICAgIHRoaXMucmV0cnlUaW1lb3V0KCk7XG4gICAgfSwgdGhpcy5jb25maWcub3B0aW9ucy5jb25uZWN0aW9uUmV0cnlJbnRlcnZhbCk7XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNvbm5lY3RUaW1lb3V0KCkge1xuICAgIGNvbnN0IG1lc3NhZ2UgPSBgRmFpbGVkIHRvIGNvbm5lY3QgdG8gJHt0aGlzLmNvbmZpZy5zZXJ2ZXJ9JHt0aGlzLmNvbmZpZy5vcHRpb25zLnBvcnQgPyBgOiR7dGhpcy5jb25maWcub3B0aW9ucy5wb3J0fWAgOiBgXFxcXCR7dGhpcy5jb25maWcub3B0aW9ucy5pbnN0YW5jZU5hbWV9YH0gaW4gJHt0aGlzLmNvbmZpZy5vcHRpb25zLmNvbm5lY3RUaW1lb3V0fW1zYDtcbiAgICB0aGlzLmRlYnVnLmxvZyhtZXNzYWdlKTtcbiAgICB0aGlzLmVtaXQoJ2Nvbm5lY3QnLCBuZXcgQ29ubmVjdGlvbkVycm9yKG1lc3NhZ2UsICdFVElNRU9VVCcpKTtcbiAgICB0aGlzLmNvbm5lY3RUaW1lciA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLmRpc3BhdGNoRXZlbnQoJ2Nvbm5lY3RUaW1lb3V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNhbmNlbFRpbWVvdXQoKSB7XG4gICAgY29uc3QgbWVzc2FnZSA9IGBGYWlsZWQgdG8gY2FuY2VsIHJlcXVlc3QgaW4gJHt0aGlzLmNvbmZpZy5vcHRpb25zLmNhbmNlbFRpbWVvdXR9bXNgO1xuICAgIHRoaXMuZGVidWcubG9nKG1lc3NhZ2UpO1xuICAgIHRoaXMuZGlzcGF0Y2hFdmVudCgnc29ja2V0RXJyb3InLCBuZXcgQ29ubmVjdGlvbkVycm9yKG1lc3NhZ2UsICdFVElNRU9VVCcpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgcmVxdWVzdFRpbWVvdXQoKSB7XG4gICAgdGhpcy5yZXF1ZXN0VGltZXIgPSB1bmRlZmluZWQ7XG4gICAgY29uc3QgcmVxdWVzdCA9IHRoaXMucmVxdWVzdCE7XG4gICAgcmVxdWVzdC5jYW5jZWwoKTtcbiAgICBjb25zdCB0aW1lb3V0ID0gKHJlcXVlc3QudGltZW91dCAhPT0gdW5kZWZpbmVkKSA/IHJlcXVlc3QudGltZW91dCA6IHRoaXMuY29uZmlnLm9wdGlvbnMucmVxdWVzdFRpbWVvdXQ7XG4gICAgY29uc3QgbWVzc2FnZSA9ICdUaW1lb3V0OiBSZXF1ZXN0IGZhaWxlZCB0byBjb21wbGV0ZSBpbiAnICsgdGltZW91dCArICdtcyc7XG4gICAgcmVxdWVzdC5lcnJvciA9IG5ldyBSZXF1ZXN0RXJyb3IobWVzc2FnZSwgJ0VUSU1FT1VUJyk7XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHJldHJ5VGltZW91dCgpIHtcbiAgICB0aGlzLnJldHJ5VGltZXIgPSB1bmRlZmluZWQ7XG4gICAgdGhpcy5lbWl0KCdyZXRyeScpO1xuICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuQ09OTkVDVElORyk7XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNsZWFyQ29ubmVjdFRpbWVyKCkge1xuICAgIGlmICh0aGlzLmNvbm5lY3RUaW1lcikge1xuICAgICAgY2xlYXJUaW1lb3V0KHRoaXMuY29ubmVjdFRpbWVyKTtcbiAgICAgIHRoaXMuY29ubmVjdFRpbWVyID0gdW5kZWZpbmVkO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY2xlYXJDYW5jZWxUaW1lcigpIHtcbiAgICBpZiAodGhpcy5jYW5jZWxUaW1lcikge1xuICAgICAgY2xlYXJUaW1lb3V0KHRoaXMuY2FuY2VsVGltZXIpO1xuICAgICAgdGhpcy5jYW5jZWxUaW1lciA9IHVuZGVmaW5lZDtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIGNsZWFyUmVxdWVzdFRpbWVyKCkge1xuICAgIGlmICh0aGlzLnJlcXVlc3RUaW1lcikge1xuICAgICAgY2xlYXJUaW1lb3V0KHRoaXMucmVxdWVzdFRpbWVyKTtcbiAgICAgIHRoaXMucmVxdWVzdFRpbWVyID0gdW5kZWZpbmVkO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY2xlYXJSZXRyeVRpbWVyKCkge1xuICAgIGlmICh0aGlzLnJldHJ5VGltZXIpIHtcbiAgICAgIGNsZWFyVGltZW91dCh0aGlzLnJldHJ5VGltZXIpO1xuICAgICAgdGhpcy5yZXRyeVRpbWVyID0gdW5kZWZpbmVkO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgdHJhbnNpdGlvblRvKG5ld1N0YXRlOiBTdGF0ZSkge1xuICAgIGlmICh0aGlzLnN0YXRlID09PSBuZXdTdGF0ZSkge1xuICAgICAgdGhpcy5kZWJ1Zy5sb2coJ1N0YXRlIGlzIGFscmVhZHkgJyArIG5ld1N0YXRlLm5hbWUpO1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnN0YXRlICYmIHRoaXMuc3RhdGUuZXhpdCkge1xuICAgICAgdGhpcy5zdGF0ZS5leGl0LmNhbGwodGhpcywgbmV3U3RhdGUpO1xuICAgIH1cblxuICAgIHRoaXMuZGVidWcubG9nKCdTdGF0ZSBjaGFuZ2U6ICcgKyAodGhpcy5zdGF0ZSA/IHRoaXMuc3RhdGUubmFtZSA6ICd1bmRlZmluZWQnKSArICcgLT4gJyArIG5ld1N0YXRlLm5hbWUpO1xuICAgIHRoaXMuc3RhdGUgPSBuZXdTdGF0ZTtcblxuICAgIGlmICh0aGlzLnN0YXRlLmVudGVyKSB7XG4gICAgICB0aGlzLnN0YXRlLmVudGVyLmFwcGx5KHRoaXMpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZ2V0RXZlbnRIYW5kbGVyPFQgZXh0ZW5kcyBrZXlvZiBTdGF0ZVsnZXZlbnRzJ10+KGV2ZW50TmFtZTogVCk6IE5vbk51bGxhYmxlPFN0YXRlWydldmVudHMnXVtUXT4ge1xuICAgIGNvbnN0IGhhbmRsZXIgPSB0aGlzLnN0YXRlLmV2ZW50c1tldmVudE5hbWVdO1xuXG4gICAgaWYgKCFoYW5kbGVyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYE5vIGV2ZW50ICcke2V2ZW50TmFtZX0nIGluIHN0YXRlICcke3RoaXMuc3RhdGUubmFtZX0nYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIGhhbmRsZXIhO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBkaXNwYXRjaEV2ZW50PFQgZXh0ZW5kcyBrZXlvZiBTdGF0ZVsnZXZlbnRzJ10+KGV2ZW50TmFtZTogVCwgLi4uYXJnczogUGFyYW1ldGVyczxOb25OdWxsYWJsZTxTdGF0ZVsnZXZlbnRzJ11bVF0+Pikge1xuICAgIGNvbnN0IGhhbmRsZXIgPSB0aGlzLnN0YXRlLmV2ZW50c1tldmVudE5hbWVdIGFzICgodGhpczogQ29ubmVjdGlvbiwgLi4uYXJnczogYW55W10pID0+IHZvaWQpIHwgdW5kZWZpbmVkO1xuICAgIGlmIChoYW5kbGVyKSB7XG4gICAgICBoYW5kbGVyLmFwcGx5KHRoaXMsIGFyZ3MpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLmVtaXQoJ2Vycm9yJywgbmV3IEVycm9yKGBObyBldmVudCAnJHtldmVudE5hbWV9JyBpbiBzdGF0ZSAnJHt0aGlzLnN0YXRlLm5hbWV9J2ApKTtcbiAgICAgIHRoaXMuY2xvc2UoKTtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHNvY2tldEVycm9yKGVycm9yOiBFcnJvcikge1xuICAgIGlmICh0aGlzLnN0YXRlID09PSB0aGlzLlNUQVRFLkNPTk5FQ1RJTkcgfHwgdGhpcy5zdGF0ZSA9PT0gdGhpcy5TVEFURS5TRU5UX1RMU1NTTE5FR09USUFUSU9OKSB7XG4gICAgICBjb25zdCBtZXNzYWdlID0gYEZhaWxlZCB0byBjb25uZWN0IHRvICR7dGhpcy5jb25maWcuc2VydmVyfToke3RoaXMuY29uZmlnLm9wdGlvbnMucG9ydH0gLSAke2Vycm9yLm1lc3NhZ2V9YDtcbiAgICAgIHRoaXMuZGVidWcubG9nKG1lc3NhZ2UpO1xuICAgICAgdGhpcy5lbWl0KCdjb25uZWN0JywgbmV3IENvbm5lY3Rpb25FcnJvcihtZXNzYWdlLCAnRVNPQ0tFVCcpKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgbWVzc2FnZSA9IGBDb25uZWN0aW9uIGxvc3QgLSAke2Vycm9yLm1lc3NhZ2V9YDtcbiAgICAgIHRoaXMuZGVidWcubG9nKG1lc3NhZ2UpO1xuICAgICAgdGhpcy5lbWl0KCdlcnJvcicsIG5ldyBDb25uZWN0aW9uRXJyb3IobWVzc2FnZSwgJ0VTT0NLRVQnKSk7XG4gICAgfVxuICAgIHRoaXMuZGlzcGF0Y2hFdmVudCgnc29ja2V0RXJyb3InLCBlcnJvcik7XG4gIH1cblxuICAvKipcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHNvY2tldEVuZCgpIHtcbiAgICB0aGlzLmRlYnVnLmxvZygnc29ja2V0IGVuZGVkJyk7XG4gICAgaWYgKHRoaXMuc3RhdGUgIT09IHRoaXMuU1RBVEUuRklOQUwpIHtcbiAgICAgIGNvbnN0IGVycm9yOiBFcnJvcldpdGhDb2RlID0gbmV3IEVycm9yKCdzb2NrZXQgaGFuZyB1cCcpO1xuICAgICAgZXJyb3IuY29kZSA9ICdFQ09OTlJFU0VUJztcbiAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyb3IpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgc29ja2V0Q2xvc2UoKSB7XG4gICAgdGhpcy5kZWJ1Zy5sb2coJ2Nvbm5lY3Rpb24gdG8gJyArIHRoaXMuY29uZmlnLnNlcnZlciArICc6JyArIHRoaXMuY29uZmlnLm9wdGlvbnMucG9ydCArICcgY2xvc2VkJyk7XG4gICAgaWYgKHRoaXMuc3RhdGUgPT09IHRoaXMuU1RBVEUuUkVST1VUSU5HKSB7XG4gICAgICB0aGlzLmRlYnVnLmxvZygnUmVyb3V0aW5nIHRvICcgKyB0aGlzLnJvdXRpbmdEYXRhIS5zZXJ2ZXIgKyAnOicgKyB0aGlzLnJvdXRpbmdEYXRhIS5wb3J0KTtcblxuICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdyZWNvbm5lY3QnKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuc3RhdGUgPT09IHRoaXMuU1RBVEUuVFJBTlNJRU5UX0ZBSUxVUkVfUkVUUlkpIHtcbiAgICAgIGNvbnN0IHNlcnZlciA9IHRoaXMucm91dGluZ0RhdGEgPyB0aGlzLnJvdXRpbmdEYXRhLnNlcnZlciA6IHRoaXMuY29uZmlnLnNlcnZlcjtcbiAgICAgIGNvbnN0IHBvcnQgPSB0aGlzLnJvdXRpbmdEYXRhID8gdGhpcy5yb3V0aW5nRGF0YS5wb3J0IDogdGhpcy5jb25maWcub3B0aW9ucy5wb3J0O1xuICAgICAgdGhpcy5kZWJ1Zy5sb2coJ1JldHJ5IGFmdGVyIHRyYW5zaWVudCBmYWlsdXJlIGNvbm5lY3RpbmcgdG8gJyArIHNlcnZlciArICc6JyArIHBvcnQpO1xuXG4gICAgICB0aGlzLmRpc3BhdGNoRXZlbnQoJ3JldHJ5Jyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgc2VuZFByZUxvZ2luKCkge1xuICAgIGNvbnN0IFsgLCBtYWpvciwgbWlub3IsIGJ1aWxkIF0gPSAvXihcXGQrKVxcLihcXGQrKVxcLihcXGQrKS8uZXhlYyh2ZXJzaW9uKSA/PyBbICcwLjAuMCcsICcwJywgJzAnLCAnMCcgXTtcblxuICAgIGNvbnN0IHBheWxvYWQgPSBuZXcgUHJlbG9naW5QYXlsb2FkKHtcbiAgICAgIGVuY3J5cHQ6IHRoaXMuY29uZmlnLm9wdGlvbnMuZW5jcnlwdCxcbiAgICAgIHZlcnNpb246IHsgbWFqb3I6IE51bWJlcihtYWpvciksIG1pbm9yOiBOdW1iZXIobWlub3IpLCBidWlsZDogTnVtYmVyKGJ1aWxkKSwgc3ViYnVpbGQ6IDAgfVxuICAgIH0pO1xuXG4gICAgdGhpcy5tZXNzYWdlSW8uc2VuZE1lc3NhZ2UoVFlQRS5QUkVMT0dJTiwgcGF5bG9hZC5kYXRhKTtcbiAgICB0aGlzLmRlYnVnLnBheWxvYWQoZnVuY3Rpb24oKSB7XG4gICAgICByZXR1cm4gcGF5bG9hZC50b1N0cmluZygnICAnKTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZW1wdHlNZXNzYWdlQnVmZmVyKCkge1xuICAgIHRoaXMubWVzc2FnZUJ1ZmZlciA9IEJ1ZmZlci5hbGxvYygwKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgYWRkVG9NZXNzYWdlQnVmZmVyKGRhdGE6IEJ1ZmZlcikge1xuICAgIHRoaXMubWVzc2FnZUJ1ZmZlciA9IEJ1ZmZlci5jb25jYXQoW3RoaXMubWVzc2FnZUJ1ZmZlciwgZGF0YV0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBzZW5kTG9naW43UGFja2V0KCkge1xuICAgIGNvbnN0IHBheWxvYWQgPSBuZXcgTG9naW43UGF5bG9hZCh7XG4gICAgICB0ZHNWZXJzaW9uOiB2ZXJzaW9uc1t0aGlzLmNvbmZpZy5vcHRpb25zLnRkc1ZlcnNpb25dLFxuICAgICAgcGFja2V0U2l6ZTogdGhpcy5jb25maWcub3B0aW9ucy5wYWNrZXRTaXplLFxuICAgICAgY2xpZW50UHJvZ1ZlcjogMCxcbiAgICAgIGNsaWVudFBpZDogcHJvY2Vzcy5waWQsXG4gICAgICBjb25uZWN0aW9uSWQ6IDAsXG4gICAgICBjbGllbnRUaW1lWm9uZTogbmV3IERhdGUoKS5nZXRUaW1lem9uZU9mZnNldCgpLFxuICAgICAgY2xpZW50TGNpZDogMHgwMDAwMDQwOVxuICAgIH0pO1xuXG4gICAgY29uc3QgeyBhdXRoZW50aWNhdGlvbiB9ID0gdGhpcy5jb25maWc7XG4gICAgc3dpdGNoIChhdXRoZW50aWNhdGlvbi50eXBlKSB7XG4gICAgICBjYXNlICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXBhc3N3b3JkJzpcbiAgICAgICAgcGF5bG9hZC5mZWRBdXRoID0ge1xuICAgICAgICAgIHR5cGU6ICdBREFMJyxcbiAgICAgICAgICBlY2hvOiB0aGlzLmZlZEF1dGhSZXF1aXJlZCxcbiAgICAgICAgICB3b3JrZmxvdzogJ2RlZmF1bHQnXG4gICAgICAgIH07XG4gICAgICAgIGJyZWFrO1xuXG4gICAgICBjYXNlICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWFjY2Vzcy10b2tlbic6XG4gICAgICAgIHBheWxvYWQuZmVkQXV0aCA9IHtcbiAgICAgICAgICB0eXBlOiAnU0VDVVJJVFlUT0tFTicsXG4gICAgICAgICAgZWNobzogdGhpcy5mZWRBdXRoUmVxdWlyZWQsXG4gICAgICAgICAgZmVkQXV0aFRva2VuOiBhdXRoZW50aWNhdGlvbi5vcHRpb25zLnRva2VuXG4gICAgICAgIH07XG4gICAgICAgIGJyZWFrO1xuXG4gICAgICBjYXNlICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS12bSc6XG4gICAgICBjYXNlICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWRlZmF1bHQnOlxuICAgICAgY2FzZSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1tc2ktYXBwLXNlcnZpY2UnOlxuICAgICAgY2FzZSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1zZXJ2aWNlLXByaW5jaXBhbC1zZWNyZXQnOlxuICAgICAgICBwYXlsb2FkLmZlZEF1dGggPSB7XG4gICAgICAgICAgdHlwZTogJ0FEQUwnLFxuICAgICAgICAgIGVjaG86IHRoaXMuZmVkQXV0aFJlcXVpcmVkLFxuICAgICAgICAgIHdvcmtmbG93OiAnaW50ZWdyYXRlZCdcbiAgICAgICAgfTtcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgJ250bG0nOlxuICAgICAgICBwYXlsb2FkLnNzcGkgPSBjcmVhdGVOVExNUmVxdWVzdCh7IGRvbWFpbjogYXV0aGVudGljYXRpb24ub3B0aW9ucy5kb21haW4gfSk7XG4gICAgICAgIGJyZWFrO1xuXG4gICAgICBkZWZhdWx0OlxuICAgICAgICBwYXlsb2FkLnVzZXJOYW1lID0gYXV0aGVudGljYXRpb24ub3B0aW9ucy51c2VyTmFtZTtcbiAgICAgICAgcGF5bG9hZC5wYXNzd29yZCA9IGF1dGhlbnRpY2F0aW9uLm9wdGlvbnMucGFzc3dvcmQ7XG4gICAgfVxuXG4gICAgcGF5bG9hZC5ob3N0bmFtZSA9IHRoaXMuY29uZmlnLm9wdGlvbnMud29ya3N0YXRpb25JZCB8fCBvcy5ob3N0bmFtZSgpO1xuICAgIHBheWxvYWQuc2VydmVyTmFtZSA9IHRoaXMucm91dGluZ0RhdGEgPyB0aGlzLnJvdXRpbmdEYXRhLnNlcnZlciA6IHRoaXMuY29uZmlnLnNlcnZlcjtcbiAgICBwYXlsb2FkLmFwcE5hbWUgPSB0aGlzLmNvbmZpZy5vcHRpb25zLmFwcE5hbWUgfHwgJ1RlZGlvdXMnO1xuICAgIHBheWxvYWQubGlicmFyeU5hbWUgPSBsaWJyYXJ5TmFtZTtcbiAgICBwYXlsb2FkLmxhbmd1YWdlID0gdGhpcy5jb25maWcub3B0aW9ucy5sYW5ndWFnZTtcbiAgICBwYXlsb2FkLmRhdGFiYXNlID0gdGhpcy5jb25maWcub3B0aW9ucy5kYXRhYmFzZTtcbiAgICBwYXlsb2FkLmNsaWVudElkID0gQnVmZmVyLmZyb20oWzEsIDIsIDMsIDQsIDUsIDZdKTtcblxuICAgIHBheWxvYWQucmVhZE9ubHlJbnRlbnQgPSB0aGlzLmNvbmZpZy5vcHRpb25zLnJlYWRPbmx5SW50ZW50O1xuICAgIHBheWxvYWQuaW5pdERiRmF0YWwgPSAhdGhpcy5jb25maWcub3B0aW9ucy5mYWxsYmFja1RvRGVmYXVsdERiO1xuXG4gICAgdGhpcy5yb3V0aW5nRGF0YSA9IHVuZGVmaW5lZDtcbiAgICB0aGlzLm1lc3NhZ2VJby5zZW5kTWVzc2FnZShUWVBFLkxPR0lONywgcGF5bG9hZC50b0J1ZmZlcigpKTtcblxuICAgIHRoaXMuZGVidWcucGF5bG9hZChmdW5jdGlvbigpIHtcbiAgICAgIHJldHVybiBwYXlsb2FkLnRvU3RyaW5nKCcgICcpO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBzZW5kRmVkQXV0aFRva2VuTWVzc2FnZSh0b2tlbjogc3RyaW5nKSB7XG4gICAgY29uc3QgYWNjZXNzVG9rZW5MZW4gPSBCdWZmZXIuYnl0ZUxlbmd0aCh0b2tlbiwgJ3VjczInKTtcbiAgICBjb25zdCBkYXRhID0gQnVmZmVyLmFsbG9jKDggKyBhY2Nlc3NUb2tlbkxlbik7XG4gICAgbGV0IG9mZnNldCA9IDA7XG4gICAgb2Zmc2V0ID0gZGF0YS53cml0ZVVJbnQzMkxFKGFjY2Vzc1Rva2VuTGVuICsgNCwgb2Zmc2V0KTtcbiAgICBvZmZzZXQgPSBkYXRhLndyaXRlVUludDMyTEUoYWNjZXNzVG9rZW5MZW4sIG9mZnNldCk7XG4gICAgZGF0YS53cml0ZSh0b2tlbiwgb2Zmc2V0LCAndWNzMicpO1xuICAgIHRoaXMubWVzc2FnZUlvLnNlbmRNZXNzYWdlKFRZUEUuRkVEQVVUSF9UT0tFTiwgZGF0YSk7XG4gICAgLy8gc2VudCB0aGUgZmVkQXV0aCB0b2tlbiBtZXNzYWdlLCB0aGUgcmVzdCBpcyBzaW1pbGFyIHRvIHN0YW5kYXJkIGxvZ2luIDdcbiAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLlNFTlRfTE9HSU43X1dJVEhfU1RBTkRBUkRfTE9HSU4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBzZW5kSW5pdGlhbFNxbCgpIHtcbiAgICBjb25zdCBwYXlsb2FkID0gbmV3IFNxbEJhdGNoUGF5bG9hZCh0aGlzLmdldEluaXRpYWxTcWwoKSwgdGhpcy5jdXJyZW50VHJhbnNhY3Rpb25EZXNjcmlwdG9yKCksIHRoaXMuY29uZmlnLm9wdGlvbnMpO1xuXG4gICAgY29uc3QgbWVzc2FnZSA9IG5ldyBNZXNzYWdlKHsgdHlwZTogVFlQRS5TUUxfQkFUQ0ggfSk7XG4gICAgdGhpcy5tZXNzYWdlSW8ub3V0Z29pbmdNZXNzYWdlU3RyZWFtLndyaXRlKG1lc3NhZ2UpO1xuICAgIFJlYWRhYmxlLmZyb20ocGF5bG9hZCkucGlwZShtZXNzYWdlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZ2V0SW5pdGlhbFNxbCgpIHtcbiAgICBjb25zdCBvcHRpb25zID0gW107XG5cbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVBbnNpTnVsbCA9PT0gdHJ1ZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgYW5zaV9udWxscyBvbicpO1xuICAgIH0gZWxzZSBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVBbnNpTnVsbCA9PT0gZmFsc2UpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IGFuc2lfbnVsbHMgb2ZmJyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlQW5zaU51bGxEZWZhdWx0ID09PSB0cnVlKSB7XG4gICAgICBvcHRpb25zLnB1c2goJ3NldCBhbnNpX251bGxfZGZsdF9vbiBvbicpO1xuICAgIH0gZWxzZSBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVBbnNpTnVsbERlZmF1bHQgPT09IGZhbHNlKSB7XG4gICAgICBvcHRpb25zLnB1c2goJ3NldCBhbnNpX251bGxfZGZsdF9vbiBvZmYnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVBbnNpUGFkZGluZyA9PT0gdHJ1ZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgYW5zaV9wYWRkaW5nIG9uJyk7XG4gICAgfSBlbHNlIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lQYWRkaW5nID09PSBmYWxzZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgYW5zaV9wYWRkaW5nIG9mZicpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUFuc2lXYXJuaW5ncyA9PT0gdHJ1ZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgYW5zaV93YXJuaW5ncyBvbicpO1xuICAgIH0gZWxzZSBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVBbnNpV2FybmluZ3MgPT09IGZhbHNlKSB7XG4gICAgICBvcHRpb25zLnB1c2goJ3NldCBhbnNpX3dhcm5pbmdzIG9mZicpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUFyaXRoQWJvcnQgPT09IHRydWUpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IGFyaXRoYWJvcnQgb24nKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlQXJpdGhBYm9ydCA9PT0gZmFsc2UpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IGFyaXRoYWJvcnQgb2ZmJyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlQ29uY2F0TnVsbFlpZWxkc051bGwgPT09IHRydWUpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IGNvbmNhdF9udWxsX3lpZWxkc19udWxsIG9uJyk7XG4gICAgfSBlbHNlIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZUNvbmNhdE51bGxZaWVsZHNOdWxsID09PSBmYWxzZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgY29uY2F0X251bGxfeWllbGRzX251bGwgb2ZmJyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlQ3Vyc29yQ2xvc2VPbkNvbW1pdCA9PT0gdHJ1ZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgY3Vyc29yX2Nsb3NlX29uX2NvbW1pdCBvbicpO1xuICAgIH0gZWxzZSBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVDdXJzb3JDbG9zZU9uQ29tbWl0ID09PSBmYWxzZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgY3Vyc29yX2Nsb3NlX29uX2NvbW1pdCBvZmYnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy5kYXRlZmlyc3QgIT09IG51bGwpIHtcbiAgICAgIG9wdGlvbnMucHVzaChgc2V0IGRhdGVmaXJzdCAke3RoaXMuY29uZmlnLm9wdGlvbnMuZGF0ZWZpcnN0fWApO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLmRhdGVGb3JtYXQgIT09IG51bGwpIHtcbiAgICAgIG9wdGlvbnMucHVzaChgc2V0IGRhdGVmb3JtYXQgJHt0aGlzLmNvbmZpZy5vcHRpb25zLmRhdGVGb3JtYXR9YCk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlSW1wbGljaXRUcmFuc2FjdGlvbnMgPT09IHRydWUpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IGltcGxpY2l0X3RyYW5zYWN0aW9ucyBvbicpO1xuICAgIH0gZWxzZSBpZiAodGhpcy5jb25maWcub3B0aW9ucy5lbmFibGVJbXBsaWNpdFRyYW5zYWN0aW9ucyA9PT0gZmFsc2UpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IGltcGxpY2l0X3RyYW5zYWN0aW9ucyBvZmYnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy5sYW5ndWFnZSAhPT0gbnVsbCkge1xuICAgICAgb3B0aW9ucy5wdXNoKGBzZXQgbGFuZ3VhZ2UgJHt0aGlzLmNvbmZpZy5vcHRpb25zLmxhbmd1YWdlfWApO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLmVuYWJsZU51bWVyaWNSb3VuZGFib3J0ID09PSB0cnVlKSB7XG4gICAgICBvcHRpb25zLnB1c2goJ3NldCBudW1lcmljX3JvdW5kYWJvcnQgb24nKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlTnVtZXJpY1JvdW5kYWJvcnQgPT09IGZhbHNlKSB7XG4gICAgICBvcHRpb25zLnB1c2goJ3NldCBudW1lcmljX3JvdW5kYWJvcnQgb2ZmJyk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlUXVvdGVkSWRlbnRpZmllciA9PT0gdHJ1ZSkge1xuICAgICAgb3B0aW9ucy5wdXNoKCdzZXQgcXVvdGVkX2lkZW50aWZpZXIgb24nKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuZW5hYmxlUXVvdGVkSWRlbnRpZmllciA9PT0gZmFsc2UpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IHF1b3RlZF9pZGVudGlmaWVyIG9mZicpO1xuICAgIH1cblxuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLnRleHRzaXplICE9PSBudWxsKSB7XG4gICAgICBvcHRpb25zLnB1c2goYHNldCB0ZXh0c2l6ZSAke3RoaXMuY29uZmlnLm9wdGlvbnMudGV4dHNpemV9YCk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuY29ubmVjdGlvbklzb2xhdGlvbkxldmVsICE9PSBudWxsKSB7XG4gICAgICBvcHRpb25zLnB1c2goYHNldCB0cmFuc2FjdGlvbiBpc29sYXRpb24gbGV2ZWwgJHt0aGlzLmdldElzb2xhdGlvbkxldmVsVGV4dCh0aGlzLmNvbmZpZy5vcHRpb25zLmNvbm5lY3Rpb25Jc29sYXRpb25MZXZlbCl9YCk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuYWJvcnRUcmFuc2FjdGlvbk9uRXJyb3IgPT09IHRydWUpIHtcbiAgICAgIG9wdGlvbnMucHVzaCgnc2V0IHhhY3RfYWJvcnQgb24nKTtcbiAgICB9IGVsc2UgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMuYWJvcnRUcmFuc2FjdGlvbk9uRXJyb3IgPT09IGZhbHNlKSB7XG4gICAgICBvcHRpb25zLnB1c2goJ3NldCB4YWN0X2Fib3J0IG9mZicpO1xuICAgIH1cblxuICAgIHJldHVybiBvcHRpb25zLmpvaW4oJ1xcbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBwcml2YXRlXG4gICAqL1xuICBwcm9jZXNzZWRJbml0aWFsU3FsKCkge1xuICAgIHRoaXMuY2xlYXJDb25uZWN0VGltZXIoKTtcbiAgICB0aGlzLmVtaXQoJ2Nvbm5lY3QnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFeGVjdXRlIHRoZSBTUUwgYmF0Y2ggcmVwcmVzZW50ZWQgYnkgW1tSZXF1ZXN0XV0uXG4gICAqIFRoZXJlIGlzIG5vIHBhcmFtIHN1cHBvcnQsIGFuZCB1bmxpa2UgW1tSZXF1ZXN0LmV4ZWNTcWxdXSxcbiAgICogaXQgaXMgbm90IGxpa2VseSB0aGF0IFNRTCBTZXJ2ZXIgd2lsbCByZXVzZSB0aGUgZXhlY3V0aW9uIHBsYW4gaXQgZ2VuZXJhdGVzIGZvciB0aGUgU1FMLlxuICAgKlxuICAgKiBJbiBhbG1vc3QgYWxsIGNhc2VzLCBbW1JlcXVlc3QuZXhlY1NxbF1dIHdpbGwgYmUgYSBiZXR0ZXIgY2hvaWNlLlxuICAgKlxuICAgKiBAcGFyYW0gcmVxdWVzdCBBIFtbUmVxdWVzdF1dIG9iamVjdCByZXByZXNlbnRpbmcgdGhlIHJlcXVlc3QuXG4gICAqL1xuICBleGVjU3FsQmF0Y2gocmVxdWVzdDogUmVxdWVzdCkge1xuICAgIHRoaXMubWFrZVJlcXVlc3QocmVxdWVzdCwgVFlQRS5TUUxfQkFUQ0gsIG5ldyBTcWxCYXRjaFBheWxvYWQocmVxdWVzdC5zcWxUZXh0T3JQcm9jZWR1cmUhLCB0aGlzLmN1cnJlbnRUcmFuc2FjdGlvbkRlc2NyaXB0b3IoKSwgdGhpcy5jb25maWcub3B0aW9ucykpO1xuICB9XG5cbiAgLyoqXG4gICAqICBFeGVjdXRlIHRoZSBTUUwgcmVwcmVzZW50ZWQgYnkgW1tSZXF1ZXN0XV0uXG4gICAqXG4gICAqIEFzIGBzcF9leGVjdXRlc3FsYCBpcyB1c2VkIHRvIGV4ZWN1dGUgdGhlIFNRTCwgaWYgdGhlIHNhbWUgU1FMIGlzIGV4ZWN1dGVkIG11bHRpcGxlcyB0aW1lc1xuICAgKiB1c2luZyB0aGlzIGZ1bmN0aW9uLCB0aGUgU1FMIFNlcnZlciBxdWVyeSBvcHRpbWl6ZXIgaXMgbGlrZWx5IHRvIHJldXNlIHRoZSBleGVjdXRpb24gcGxhbiBpdCBnZW5lcmF0ZXNcbiAgICogZm9yIHRoZSBmaXJzdCBleGVjdXRpb24uIFRoaXMgbWF5IGFsc28gcmVzdWx0IGluIFNRTCBzZXJ2ZXIgdHJlYXRpbmcgdGhlIHJlcXVlc3QgbGlrZSBhIHN0b3JlZCBwcm9jZWR1cmVcbiAgICogd2hpY2ggY2FuIHJlc3VsdCBpbiB0aGUgW1tFdmVudF9kb25lSW5Qcm9jXV0gb3IgW1tFdmVudF9kb25lUHJvY11dIGV2ZW50cyBiZWluZyBlbWl0dGVkIGluc3RlYWQgb2YgdGhlXG4gICAqIFtbRXZlbnRfZG9uZV1dIGV2ZW50IHlvdSBtaWdodCBleHBlY3QuIFVzaW5nIFtbZXhlY1NxbEJhdGNoXV0gd2lsbCBwcmV2ZW50IHRoaXMgZnJvbSBvY2N1cnJpbmcgYnV0IG1heSBoYXZlIGEgbmVnYXRpdmUgcGVyZm9ybWFuY2UgaW1wYWN0LlxuICAgKlxuICAgKiBCZXdhcmUgb2YgdGhlIHdheSB0aGF0IHNjb3BpbmcgcnVsZXMgYXBwbHksIGFuZCBob3cgdGhleSBtYXkgW2FmZmVjdCBsb2NhbCB0ZW1wIHRhYmxlc10oaHR0cDovL3dlYmxvZ3Muc3FsdGVhbS5jb20vbWxhZGVucC9hcmNoaXZlLzIwMDYvMTEvMDMvMTcxOTcuYXNweClcbiAgICogSWYgeW91J3JlIHJ1bm5pbmcgaW4gdG8gc2NvcGluZyBpc3N1ZXMsIHRoZW4gW1tleGVjU3FsQmF0Y2hdXSBtYXkgYmUgYSBiZXR0ZXIgY2hvaWNlLlxuICAgKiBTZWUgYWxzbyBbaXNzdWUgIzI0XShodHRwczovL2dpdGh1Yi5jb20vcGVraW0vdGVkaW91cy9pc3N1ZXMvMjQpXG4gICAqXG4gICAqIEBwYXJhbSByZXF1ZXN0IEEgW1tSZXF1ZXN0XV0gb2JqZWN0IHJlcHJlc2VudGluZyB0aGUgcmVxdWVzdC5cbiAgICovXG4gIGV4ZWNTcWwocmVxdWVzdDogUmVxdWVzdCkge1xuICAgIHRyeSB7XG4gICAgICByZXF1ZXN0LnZhbGlkYXRlUGFyYW1ldGVycyh0aGlzLmRhdGFiYXNlQ29sbGF0aW9uKTtcbiAgICB9IGNhdGNoIChlcnJvcjogYW55KSB7XG4gICAgICByZXF1ZXN0LmVycm9yID0gZXJyb3I7XG5cbiAgICAgIHByb2Nlc3MubmV4dFRpY2soKCkgPT4ge1xuICAgICAgICB0aGlzLmRlYnVnLmxvZyhlcnJvci5tZXNzYWdlKTtcbiAgICAgICAgcmVxdWVzdC5jYWxsYmFjayhlcnJvcik7XG4gICAgICB9KTtcblxuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGNvbnN0IHBhcmFtZXRlcnM6IFBhcmFtZXRlcltdID0gW107XG5cbiAgICBwYXJhbWV0ZXJzLnB1c2goe1xuICAgICAgdHlwZTogVFlQRVMuTlZhckNoYXIsXG4gICAgICBuYW1lOiAnc3RhdGVtZW50JyxcbiAgICAgIHZhbHVlOiByZXF1ZXN0LnNxbFRleHRPclByb2NlZHVyZSxcbiAgICAgIG91dHB1dDogZmFsc2UsXG4gICAgICBsZW5ndGg6IHVuZGVmaW5lZCxcbiAgICAgIHByZWNpc2lvbjogdW5kZWZpbmVkLFxuICAgICAgc2NhbGU6IHVuZGVmaW5lZFxuICAgIH0pO1xuXG4gICAgaWYgKHJlcXVlc3QucGFyYW1ldGVycy5sZW5ndGgpIHtcbiAgICAgIHBhcmFtZXRlcnMucHVzaCh7XG4gICAgICAgIHR5cGU6IFRZUEVTLk5WYXJDaGFyLFxuICAgICAgICBuYW1lOiAncGFyYW1zJyxcbiAgICAgICAgdmFsdWU6IHJlcXVlc3QubWFrZVBhcmFtc1BhcmFtZXRlcihyZXF1ZXN0LnBhcmFtZXRlcnMpLFxuICAgICAgICBvdXRwdXQ6IGZhbHNlLFxuICAgICAgICBsZW5ndGg6IHVuZGVmaW5lZCxcbiAgICAgICAgcHJlY2lzaW9uOiB1bmRlZmluZWQsXG4gICAgICAgIHNjYWxlOiB1bmRlZmluZWRcbiAgICAgIH0pO1xuXG4gICAgICBwYXJhbWV0ZXJzLnB1c2goLi4ucmVxdWVzdC5wYXJhbWV0ZXJzKTtcbiAgICB9XG5cbiAgICB0aGlzLm1ha2VSZXF1ZXN0KHJlcXVlc3QsIFRZUEUuUlBDX1JFUVVFU1QsIG5ldyBScGNSZXF1ZXN0UGF5bG9hZCgnc3BfZXhlY3V0ZXNxbCcsIHBhcmFtZXRlcnMsIHRoaXMuY3VycmVudFRyYW5zYWN0aW9uRGVzY3JpcHRvcigpLCB0aGlzLmNvbmZpZy5vcHRpb25zLCB0aGlzLmRhdGFiYXNlQ29sbGF0aW9uKSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyBCdWxrTG9hZCBpbnN0YW5jZS5cbiAgICpcbiAgICogQHBhcmFtIHRhYmxlIFRoZSBuYW1lIG9mIHRoZSB0YWJsZSB0byBidWxrLWluc2VydCBpbnRvLlxuICAgKiBAcGFyYW0gb3B0aW9ucyBBIHNldCBvZiBidWxrIGxvYWQgb3B0aW9ucy5cbiAgICovXG4gIG5ld0J1bGtMb2FkKHRhYmxlOiBzdHJpbmcsIGNhbGxiYWNrOiBCdWxrTG9hZENhbGxiYWNrKTogQnVsa0xvYWRcbiAgbmV3QnVsa0xvYWQodGFibGU6IHN0cmluZywgb3B0aW9uczogQnVsa0xvYWRPcHRpb25zLCBjYWxsYmFjazogQnVsa0xvYWRDYWxsYmFjayk6IEJ1bGtMb2FkXG4gIG5ld0J1bGtMb2FkKHRhYmxlOiBzdHJpbmcsIGNhbGxiYWNrT3JPcHRpb25zOiBCdWxrTG9hZE9wdGlvbnMgfCBCdWxrTG9hZENhbGxiYWNrLCBjYWxsYmFjaz86IEJ1bGtMb2FkQ2FsbGJhY2spIHtcbiAgICBsZXQgb3B0aW9uczogQnVsa0xvYWRPcHRpb25zO1xuXG4gICAgaWYgKGNhbGxiYWNrID09PSB1bmRlZmluZWQpIHtcbiAgICAgIGNhbGxiYWNrID0gY2FsbGJhY2tPck9wdGlvbnMgYXMgQnVsa0xvYWRDYWxsYmFjaztcbiAgICAgIG9wdGlvbnMgPSB7fTtcbiAgICB9IGVsc2Uge1xuICAgICAgb3B0aW9ucyA9IGNhbGxiYWNrT3JPcHRpb25zIGFzIEJ1bGtMb2FkT3B0aW9ucztcbiAgICB9XG5cbiAgICBpZiAodHlwZW9mIG9wdGlvbnMgIT09ICdvYmplY3QnKSB7XG4gICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdcIm9wdGlvbnNcIiBhcmd1bWVudCBtdXN0IGJlIGFuIG9iamVjdCcpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IEJ1bGtMb2FkKHRhYmxlLCB0aGlzLmRhdGFiYXNlQ29sbGF0aW9uLCB0aGlzLmNvbmZpZy5vcHRpb25zLCBvcHRpb25zLCBjYWxsYmFjayk7XG4gIH1cblxuICAvKipcbiAgICogRXhlY3V0ZSBhIFtbQnVsa0xvYWRdXS5cbiAgICpcbiAgICogYGBganNcbiAgICogLy8gV2Ugd2FudCB0byBwZXJmb3JtIGEgYnVsayBsb2FkIGludG8gYSB0YWJsZSB3aXRoIHRoZSBmb2xsb3dpbmcgZm9ybWF0OlxuICAgKiAvLyBDUkVBVEUgVEFCTEUgZW1wbG95ZWVzIChmaXJzdF9uYW1lIG52YXJjaGFyKDI1NSksIGxhc3RfbmFtZSBudmFyY2hhcigyNTUpLCBkYXlfb2ZfYmlydGggZGF0ZSk7XG4gICAqXG4gICAqIGNvbnN0IGJ1bGtMb2FkID0gY29ubmVjdGlvbi5uZXdCdWxrTG9hZCgnZW1wbG95ZWVzJywgKGVyciwgcm93Q291bnQpID0+IHtcbiAgICogICAvLyAuLi5cbiAgICogfSk7XG4gICAqXG4gICAqIC8vIEZpcnN0LCB3ZSBuZWVkIHRvIHNwZWNpZnkgdGhlIGNvbHVtbnMgdGhhdCB3ZSB3YW50IHRvIHdyaXRlIHRvLFxuICAgKiAvLyBhbmQgdGhlaXIgZGVmaW5pdGlvbnMuIFRoZXNlIGRlZmluaXRpb25zIG11c3QgbWF0Y2ggdGhlIGFjdHVhbCB0YWJsZSxcbiAgICogLy8gb3RoZXJ3aXNlIHRoZSBidWxrIGxvYWQgd2lsbCBmYWlsLlxuICAgKiBidWxrTG9hZC5hZGRDb2x1bW4oJ2ZpcnN0X25hbWUnLCBUWVBFUy5OVmFyY2hhciwgeyBudWxsYWJsZTogZmFsc2UgfSk7XG4gICAqIGJ1bGtMb2FkLmFkZENvbHVtbignbGFzdF9uYW1lJywgVFlQRVMuTlZhcmNoYXIsIHsgbnVsbGFibGU6IGZhbHNlIH0pO1xuICAgKiBidWxrTG9hZC5hZGRDb2x1bW4oJ2RhdGVfb2ZfYmlydGgnLCBUWVBFUy5EYXRlLCB7IG51bGxhYmxlOiBmYWxzZSB9KTtcbiAgICpcbiAgICogLy8gRXhlY3V0ZSBhIGJ1bGsgbG9hZCB3aXRoIGEgcHJlZGVmaW5lZCBsaXN0IG9mIHJvd3MuXG4gICAqIC8vXG4gICAqIC8vIE5vdGUgdGhhdCB0aGVzZSByb3dzIGFyZSBoZWxkIGluIG1lbW9yeSB1bnRpbCB0aGVcbiAgICogLy8gYnVsayBsb2FkIHdhcyBwZXJmb3JtZWQsIHNvIGlmIHlvdSBuZWVkIHRvIHdyaXRlIGEgbGFyZ2VcbiAgICogLy8gbnVtYmVyIG9mIHJvd3MgKGUuZy4gYnkgcmVhZGluZyBmcm9tIGEgQ1NWIGZpbGUpLFxuICAgKiAvLyBwYXNzaW5nIGFuIGBBc3luY0l0ZXJhYmxlYCBpcyBhZHZpc2FibGUgdG8ga2VlcCBtZW1vcnkgdXNhZ2UgbG93LlxuICAgKiBjb25uZWN0aW9uLmV4ZWNCdWxrTG9hZChidWxrTG9hZCwgW1xuICAgKiAgIHsgJ2ZpcnN0X25hbWUnOiAnU3RldmUnLCAnbGFzdF9uYW1lJzogJ0pvYnMnLCAnZGF5X29mX2JpcnRoJzogbmV3IERhdGUoJzAyLTI0LTE5NTUnKSB9LFxuICAgKiAgIHsgJ2ZpcnN0X25hbWUnOiAnQmlsbCcsICdsYXN0X25hbWUnOiAnR2F0ZXMnLCAnZGF5X29mX2JpcnRoJzogbmV3IERhdGUoJzEwLTI4LTE5NTUnKSB9XG4gICAqIF0pO1xuICAgKiBgYGBcbiAgICpcbiAgICogQHBhcmFtIGJ1bGtMb2FkIEEgcHJldmlvdXNseSBjcmVhdGVkIFtbQnVsa0xvYWRdXS5cbiAgICogQHBhcmFtIHJvd3MgQSBbW0l0ZXJhYmxlXV0gb3IgW1tBc3luY0l0ZXJhYmxlXV0gdGhhdCBjb250YWlucyB0aGUgcm93cyB0aGF0IHNob3VsZCBiZSBidWxrIGxvYWRlZC5cbiAgICovXG4gIGV4ZWNCdWxrTG9hZChidWxrTG9hZDogQnVsa0xvYWQsIHJvd3M6IEFzeW5jSXRlcmFibGU8dW5rbm93bltdIHwgeyBbY29sdW1uTmFtZTogc3RyaW5nXTogdW5rbm93biB9PiB8IEl0ZXJhYmxlPHVua25vd25bXSB8IHsgW2NvbHVtbk5hbWU6IHN0cmluZ106IHVua25vd24gfT4pOiB2b2lkXG5cbiAgZXhlY0J1bGtMb2FkKGJ1bGtMb2FkOiBCdWxrTG9hZCwgcm93cz86IEFzeW5jSXRlcmFibGU8dW5rbm93bltdIHwgeyBbY29sdW1uTmFtZTogc3RyaW5nXTogdW5rbm93biB9PiB8IEl0ZXJhYmxlPHVua25vd25bXSB8IHsgW2NvbHVtbk5hbWU6IHN0cmluZ106IHVua25vd24gfT4pIHtcbiAgICBidWxrTG9hZC5leGVjdXRpb25TdGFydGVkID0gdHJ1ZTtcblxuICAgIGlmIChyb3dzKSB7XG4gICAgICBpZiAoYnVsa0xvYWQuc3RyZWFtaW5nTW9kZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCJDb25uZWN0aW9uLmV4ZWNCdWxrTG9hZCBjYW4ndCBiZSBjYWxsZWQgd2l0aCBhIEJ1bGtMb2FkIHRoYXQgd2FzIHB1dCBpbiBzdHJlYW1pbmcgbW9kZS5cIik7XG4gICAgICB9XG5cbiAgICAgIGlmIChidWxrTG9hZC5maXJzdFJvd1dyaXR0ZW4pIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ29ubmVjdGlvbi5leGVjQnVsa0xvYWQgY2FuJ3QgYmUgY2FsbGVkIHdpdGggYSBCdWxrTG9hZCB0aGF0IGFscmVhZHkgaGFzIHJvd3Mgd3JpdHRlbiB0byBpdC5cIik7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IHJvd1N0cmVhbSA9IFJlYWRhYmxlLmZyb20ocm93cyk7XG5cbiAgICAgIC8vIERlc3Ryb3kgdGhlIHBhY2tldCB0cmFuc2Zvcm0gaWYgYW4gZXJyb3IgaGFwcGVucyBpbiB0aGUgcm93IHN0cmVhbSxcbiAgICAgIC8vIGUuZy4gaWYgYW4gZXJyb3IgaXMgdGhyb3duIGZyb20gd2l0aGluIGEgZ2VuZXJhdG9yIG9yIHN0cmVhbS5cbiAgICAgIHJvd1N0cmVhbS5vbignZXJyb3InLCAoZXJyKSA9PiB7XG4gICAgICAgIGJ1bGtMb2FkLnJvd1RvUGFja2V0VHJhbnNmb3JtLmRlc3Ryb3koZXJyKTtcbiAgICAgIH0pO1xuXG4gICAgICAvLyBEZXN0cm95IHRoZSByb3cgc3RyZWFtIGlmIGFuIGVycm9yIGhhcHBlbnMgaW4gdGhlIHBhY2tldCB0cmFuc2Zvcm0sXG4gICAgICAvLyBlLmcuIGlmIHRoZSBidWxrIGxvYWQgaXMgY2FuY2VsbGVkLlxuICAgICAgYnVsa0xvYWQucm93VG9QYWNrZXRUcmFuc2Zvcm0ub24oJ2Vycm9yJywgKGVycikgPT4ge1xuICAgICAgICByb3dTdHJlYW0uZGVzdHJveShlcnIpO1xuICAgICAgfSk7XG5cbiAgICAgIHJvd1N0cmVhbS5waXBlKGJ1bGtMb2FkLnJvd1RvUGFja2V0VHJhbnNmb3JtKTtcbiAgICB9IGVsc2UgaWYgKCFidWxrTG9hZC5zdHJlYW1pbmdNb2RlKSB7XG4gICAgICAvLyBJZiB0aGUgYnVsa2xvYWQgd2FzIG5vdCBwdXQgaW50byBzdHJlYW1pbmcgbW9kZSBieSB0aGUgdXNlcixcbiAgICAgIC8vIHdlIGVuZCB0aGUgcm93VG9QYWNrZXRUcmFuc2Zvcm0gaGVyZSBmb3IgdGhlbS5cbiAgICAgIC8vXG4gICAgICAvLyBJZiBpdCB3YXMgcHV0IGludG8gc3RyZWFtaW5nIG1vZGUsIGl0J3MgdGhlIHVzZXIncyByZXNwb25zaWJpbGl0eVxuICAgICAgLy8gdG8gZW5kIHRoZSBzdHJlYW0uXG4gICAgICBidWxrTG9hZC5yb3dUb1BhY2tldFRyYW5zZm9ybS5lbmQoKTtcbiAgICB9XG5cbiAgICBjb25zdCBvbkNhbmNlbCA9ICgpID0+IHtcbiAgICAgIHJlcXVlc3QuY2FuY2VsKCk7XG4gICAgfTtcblxuICAgIGNvbnN0IHBheWxvYWQgPSBuZXcgQnVsa0xvYWRQYXlsb2FkKGJ1bGtMb2FkKTtcblxuICAgIGNvbnN0IHJlcXVlc3QgPSBuZXcgUmVxdWVzdChidWxrTG9hZC5nZXRCdWxrSW5zZXJ0U3FsKCksIChlcnJvcjogKEVycm9yICYgeyBjb2RlPzogc3RyaW5nIH0pIHwgbnVsbCB8IHVuZGVmaW5lZCkgPT4ge1xuICAgICAgYnVsa0xvYWQucmVtb3ZlTGlzdGVuZXIoJ2NhbmNlbCcsIG9uQ2FuY2VsKTtcblxuICAgICAgaWYgKGVycm9yKSB7XG4gICAgICAgIGlmIChlcnJvci5jb2RlID09PSAnVU5LTk9XTicpIHtcbiAgICAgICAgICBlcnJvci5tZXNzYWdlICs9ICcgVGhpcyBpcyBsaWtlbHkgYmVjYXVzZSB0aGUgc2NoZW1hIG9mIHRoZSBCdWxrTG9hZCBkb2VzIG5vdCBtYXRjaCB0aGUgc2NoZW1hIG9mIHRoZSB0YWJsZSB5b3UgYXJlIGF0dGVtcHRpbmcgdG8gaW5zZXJ0IGludG8uJztcbiAgICAgICAgfVxuICAgICAgICBidWxrTG9hZC5lcnJvciA9IGVycm9yO1xuICAgICAgICBidWxrTG9hZC5jYWxsYmFjayhlcnJvcik7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgdGhpcy5tYWtlUmVxdWVzdChidWxrTG9hZCwgVFlQRS5CVUxLX0xPQUQsIHBheWxvYWQpO1xuICAgIH0pO1xuXG4gICAgYnVsa0xvYWQub25jZSgnY2FuY2VsJywgb25DYW5jZWwpO1xuXG4gICAgdGhpcy5leGVjU3FsQmF0Y2gocmVxdWVzdCk7XG4gIH1cblxuICAvKipcbiAgICogUHJlcGFyZSB0aGUgU1FMIHJlcHJlc2VudGVkIGJ5IHRoZSByZXF1ZXN0LlxuICAgKlxuICAgKiBUaGUgcmVxdWVzdCBjYW4gdGhlbiBiZSB1c2VkIGluIHN1YnNlcXVlbnQgY2FsbHMgdG9cbiAgICogW1tleGVjdXRlXV0gYW5kIFtbdW5wcmVwYXJlXV1cbiAgICpcbiAgICogQHBhcmFtIHJlcXVlc3QgQSBbW1JlcXVlc3RdXSBvYmplY3QgcmVwcmVzZW50aW5nIHRoZSByZXF1ZXN0LlxuICAgKiAgIFBhcmFtZXRlcnMgb25seSByZXF1aXJlIGEgbmFtZSBhbmQgdHlwZS4gUGFyYW1ldGVyIHZhbHVlcyBhcmUgaWdub3JlZC5cbiAgICovXG4gIHByZXBhcmUocmVxdWVzdDogUmVxdWVzdCkge1xuICAgIGNvbnN0IHBhcmFtZXRlcnM6IFBhcmFtZXRlcltdID0gW107XG5cbiAgICBwYXJhbWV0ZXJzLnB1c2goe1xuICAgICAgdHlwZTogVFlQRVMuSW50LFxuICAgICAgbmFtZTogJ2hhbmRsZScsXG4gICAgICB2YWx1ZTogdW5kZWZpbmVkLFxuICAgICAgb3V0cHV0OiB0cnVlLFxuICAgICAgbGVuZ3RoOiB1bmRlZmluZWQsXG4gICAgICBwcmVjaXNpb246IHVuZGVmaW5lZCxcbiAgICAgIHNjYWxlOiB1bmRlZmluZWRcbiAgICB9KTtcblxuICAgIHBhcmFtZXRlcnMucHVzaCh7XG4gICAgICB0eXBlOiBUWVBFUy5OVmFyQ2hhcixcbiAgICAgIG5hbWU6ICdwYXJhbXMnLFxuICAgICAgdmFsdWU6IHJlcXVlc3QucGFyYW1ldGVycy5sZW5ndGggPyByZXF1ZXN0Lm1ha2VQYXJhbXNQYXJhbWV0ZXIocmVxdWVzdC5wYXJhbWV0ZXJzKSA6IG51bGwsXG4gICAgICBvdXRwdXQ6IGZhbHNlLFxuICAgICAgbGVuZ3RoOiB1bmRlZmluZWQsXG4gICAgICBwcmVjaXNpb246IHVuZGVmaW5lZCxcbiAgICAgIHNjYWxlOiB1bmRlZmluZWRcbiAgICB9KTtcblxuICAgIHBhcmFtZXRlcnMucHVzaCh7XG4gICAgICB0eXBlOiBUWVBFUy5OVmFyQ2hhcixcbiAgICAgIG5hbWU6ICdzdG10JyxcbiAgICAgIHZhbHVlOiByZXF1ZXN0LnNxbFRleHRPclByb2NlZHVyZSxcbiAgICAgIG91dHB1dDogZmFsc2UsXG4gICAgICBsZW5ndGg6IHVuZGVmaW5lZCxcbiAgICAgIHByZWNpc2lvbjogdW5kZWZpbmVkLFxuICAgICAgc2NhbGU6IHVuZGVmaW5lZFxuICAgIH0pO1xuXG4gICAgcmVxdWVzdC5wcmVwYXJpbmcgPSB0cnVlO1xuICAgIC8vIFRPRE86IFdlIG5lZWQgdG8gY2xlYW4gdXAgdGhpcyBldmVudCBoYW5kbGVyLCBvdGhlcndpc2UgdGhpcyBsZWFrcyBtZW1vcnlcbiAgICByZXF1ZXN0Lm9uKCdyZXR1cm5WYWx1ZScsIChuYW1lOiBzdHJpbmcsIHZhbHVlOiBhbnkpID0+IHtcbiAgICAgIGlmIChuYW1lID09PSAnaGFuZGxlJykge1xuICAgICAgICByZXF1ZXN0LmhhbmRsZSA9IHZhbHVlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgcmVxdWVzdC5lcnJvciA9IG5ldyBSZXF1ZXN0RXJyb3IoYFRlZGlvdXMgPiBVbmV4cGVjdGVkIG91dHB1dCBwYXJhbWV0ZXIgJHtuYW1lfSBmcm9tIHNwX3ByZXBhcmVgKTtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIHRoaXMubWFrZVJlcXVlc3QocmVxdWVzdCwgVFlQRS5SUENfUkVRVUVTVCwgbmV3IFJwY1JlcXVlc3RQYXlsb2FkKCdzcF9wcmVwYXJlJywgcGFyYW1ldGVycywgdGhpcy5jdXJyZW50VHJhbnNhY3Rpb25EZXNjcmlwdG9yKCksIHRoaXMuY29uZmlnLm9wdGlvbnMsIHRoaXMuZGF0YWJhc2VDb2xsYXRpb24pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZWxlYXNlIHRoZSBTUUwgU2VydmVyIHJlc291cmNlcyBhc3NvY2lhdGVkIHdpdGggYSBwcmV2aW91c2x5IHByZXBhcmVkIHJlcXVlc3QuXG4gICAqXG4gICAqIEBwYXJhbSByZXF1ZXN0IEEgW1tSZXF1ZXN0XV0gb2JqZWN0IHJlcHJlc2VudGluZyB0aGUgcmVxdWVzdC5cbiAgICogICBQYXJhbWV0ZXJzIG9ubHkgcmVxdWlyZSBhIG5hbWUgYW5kIHR5cGUuXG4gICAqICAgUGFyYW1ldGVyIHZhbHVlcyBhcmUgaWdub3JlZC5cbiAgICovXG4gIHVucHJlcGFyZShyZXF1ZXN0OiBSZXF1ZXN0KSB7XG4gICAgY29uc3QgcGFyYW1ldGVyczogUGFyYW1ldGVyW10gPSBbXTtcblxuICAgIHBhcmFtZXRlcnMucHVzaCh7XG4gICAgICB0eXBlOiBUWVBFUy5JbnQsXG4gICAgICBuYW1lOiAnaGFuZGxlJyxcbiAgICAgIC8vIFRPRE86IEFib3J0IGlmIGByZXF1ZXN0LmhhbmRsZWAgaXMgbm90IHNldFxuICAgICAgdmFsdWU6IHJlcXVlc3QuaGFuZGxlLFxuICAgICAgb3V0cHV0OiBmYWxzZSxcbiAgICAgIGxlbmd0aDogdW5kZWZpbmVkLFxuICAgICAgcHJlY2lzaW9uOiB1bmRlZmluZWQsXG4gICAgICBzY2FsZTogdW5kZWZpbmVkXG4gICAgfSk7XG5cbiAgICB0aGlzLm1ha2VSZXF1ZXN0KHJlcXVlc3QsIFRZUEUuUlBDX1JFUVVFU1QsIG5ldyBScGNSZXF1ZXN0UGF5bG9hZCgnc3BfdW5wcmVwYXJlJywgcGFyYW1ldGVycywgdGhpcy5jdXJyZW50VHJhbnNhY3Rpb25EZXNjcmlwdG9yKCksIHRoaXMuY29uZmlnLm9wdGlvbnMsIHRoaXMuZGF0YWJhc2VDb2xsYXRpb24pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFeGVjdXRlIHByZXZpb3VzbHkgcHJlcGFyZWQgU1FMLCB1c2luZyB0aGUgc3VwcGxpZWQgcGFyYW1ldGVycy5cbiAgICpcbiAgICogQHBhcmFtIHJlcXVlc3QgQSBwcmV2aW91c2x5IHByZXBhcmVkIFtbUmVxdWVzdF1dLlxuICAgKiBAcGFyYW0gcGFyYW1ldGVycyAgQW4gb2JqZWN0IHdob3NlIG5hbWVzIGNvcnJlc3BvbmQgdG8gdGhlIG5hbWVzIG9mXG4gICAqICAgcGFyYW1ldGVycyB0aGF0IHdlcmUgYWRkZWQgdG8gdGhlIFtbUmVxdWVzdF1dIGJlZm9yZSBpdCB3YXMgcHJlcGFyZWQuXG4gICAqICAgVGhlIG9iamVjdCdzIHZhbHVlcyBhcmUgcGFzc2VkIGFzIHRoZSBwYXJhbWV0ZXJzJyB2YWx1ZXMgd2hlbiB0aGVcbiAgICogICByZXF1ZXN0IGlzIGV4ZWN1dGVkLlxuICAgKi9cbiAgZXhlY3V0ZShyZXF1ZXN0OiBSZXF1ZXN0LCBwYXJhbWV0ZXJzPzogeyBba2V5OiBzdHJpbmddOiB1bmtub3duIH0pIHtcbiAgICBjb25zdCBleGVjdXRlUGFyYW1ldGVyczogUGFyYW1ldGVyW10gPSBbXTtcblxuICAgIGV4ZWN1dGVQYXJhbWV0ZXJzLnB1c2goe1xuICAgICAgdHlwZTogVFlQRVMuSW50LFxuICAgICAgbmFtZTogJ2hhbmRsZScsXG4gICAgICAvLyBUT0RPOiBBYm9ydCBpZiBgcmVxdWVzdC5oYW5kbGVgIGlzIG5vdCBzZXRcbiAgICAgIHZhbHVlOiByZXF1ZXN0LmhhbmRsZSxcbiAgICAgIG91dHB1dDogZmFsc2UsXG4gICAgICBsZW5ndGg6IHVuZGVmaW5lZCxcbiAgICAgIHByZWNpc2lvbjogdW5kZWZpbmVkLFxuICAgICAgc2NhbGU6IHVuZGVmaW5lZFxuICAgIH0pO1xuXG4gICAgdHJ5IHtcbiAgICAgIGZvciAobGV0IGkgPSAwLCBsZW4gPSByZXF1ZXN0LnBhcmFtZXRlcnMubGVuZ3RoOyBpIDwgbGVuOyBpKyspIHtcbiAgICAgICAgY29uc3QgcGFyYW1ldGVyID0gcmVxdWVzdC5wYXJhbWV0ZXJzW2ldO1xuXG4gICAgICAgIGV4ZWN1dGVQYXJhbWV0ZXJzLnB1c2goe1xuICAgICAgICAgIC4uLnBhcmFtZXRlcixcbiAgICAgICAgICB2YWx1ZTogcGFyYW1ldGVyLnR5cGUudmFsaWRhdGUocGFyYW1ldGVycyA/IHBhcmFtZXRlcnNbcGFyYW1ldGVyLm5hbWVdIDogbnVsbCwgdGhpcy5kYXRhYmFzZUNvbGxhdGlvbilcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgcmVxdWVzdC5lcnJvciA9IGVycm9yO1xuXG4gICAgICBwcm9jZXNzLm5leHRUaWNrKCgpID0+IHtcbiAgICAgICAgdGhpcy5kZWJ1Zy5sb2coZXJyb3IubWVzc2FnZSk7XG4gICAgICAgIHJlcXVlc3QuY2FsbGJhY2soZXJyb3IpO1xuICAgICAgfSk7XG5cbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLm1ha2VSZXF1ZXN0KHJlcXVlc3QsIFRZUEUuUlBDX1JFUVVFU1QsIG5ldyBScGNSZXF1ZXN0UGF5bG9hZCgnc3BfZXhlY3V0ZScsIGV4ZWN1dGVQYXJhbWV0ZXJzLCB0aGlzLmN1cnJlbnRUcmFuc2FjdGlvbkRlc2NyaXB0b3IoKSwgdGhpcy5jb25maWcub3B0aW9ucywgdGhpcy5kYXRhYmFzZUNvbGxhdGlvbikpO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGwgYSBzdG9yZWQgcHJvY2VkdXJlIHJlcHJlc2VudGVkIGJ5IFtbUmVxdWVzdF1dLlxuICAgKlxuICAgKiBAcGFyYW0gcmVxdWVzdCBBIFtbUmVxdWVzdF1dIG9iamVjdCByZXByZXNlbnRpbmcgdGhlIHJlcXVlc3QuXG4gICAqL1xuICBjYWxsUHJvY2VkdXJlKHJlcXVlc3Q6IFJlcXVlc3QpIHtcbiAgICB0cnkge1xuICAgICAgcmVxdWVzdC52YWxpZGF0ZVBhcmFtZXRlcnModGhpcy5kYXRhYmFzZUNvbGxhdGlvbik7XG4gICAgfSBjYXRjaCAoZXJyb3I6IGFueSkge1xuICAgICAgcmVxdWVzdC5lcnJvciA9IGVycm9yO1xuXG4gICAgICBwcm9jZXNzLm5leHRUaWNrKCgpID0+IHtcbiAgICAgICAgdGhpcy5kZWJ1Zy5sb2coZXJyb3IubWVzc2FnZSk7XG4gICAgICAgIHJlcXVlc3QuY2FsbGJhY2soZXJyb3IpO1xuICAgICAgfSk7XG5cbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLm1ha2VSZXF1ZXN0KHJlcXVlc3QsIFRZUEUuUlBDX1JFUVVFU1QsIG5ldyBScGNSZXF1ZXN0UGF5bG9hZChyZXF1ZXN0LnNxbFRleHRPclByb2NlZHVyZSEsIHJlcXVlc3QucGFyYW1ldGVycywgdGhpcy5jdXJyZW50VHJhbnNhY3Rpb25EZXNjcmlwdG9yKCksIHRoaXMuY29uZmlnLm9wdGlvbnMsIHRoaXMuZGF0YWJhc2VDb2xsYXRpb24pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGFydCBhIHRyYW5zYWN0aW9uLlxuICAgKlxuICAgKiBAcGFyYW0gY2FsbGJhY2tcbiAgICogQHBhcmFtIG5hbWUgQSBzdHJpbmcgcmVwcmVzZW50aW5nIGEgbmFtZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgdHJhbnNhY3Rpb24uXG4gICAqICAgT3B0aW9uYWwsIGFuZCBkZWZhdWx0cyB0byBhbiBlbXB0eSBzdHJpbmcuIFJlcXVpcmVkIHdoZW4gYGlzb2xhdGlvbkxldmVsYFxuICAgKiAgIGlzIHByZXNlbnQuXG4gICAqIEBwYXJhbSBpc29sYXRpb25MZXZlbCBUaGUgaXNvbGF0aW9uIGxldmVsIHRoYXQgdGhlIHRyYW5zYWN0aW9uIGlzIHRvIGJlIHJ1biB3aXRoLlxuICAgKlxuICAgKiAgIFRoZSBpc29sYXRpb24gbGV2ZWxzIGFyZSBhdmFpbGFibGUgZnJvbSBgcmVxdWlyZSgndGVkaW91cycpLklTT0xBVElPTl9MRVZFTGAuXG4gICAqICAgKiBgUkVBRF9VTkNPTU1JVFRFRGBcbiAgICogICAqIGBSRUFEX0NPTU1JVFRFRGBcbiAgICogICAqIGBSRVBFQVRBQkxFX1JFQURgXG4gICAqICAgKiBgU0VSSUFMSVpBQkxFYFxuICAgKiAgICogYFNOQVBTSE9UYFxuICAgKlxuICAgKiAgIE9wdGlvbmFsLCBhbmQgZGVmYXVsdHMgdG8gdGhlIENvbm5lY3Rpb24ncyBpc29sYXRpb24gbGV2ZWwuXG4gICAqL1xuICBiZWdpblRyYW5zYWN0aW9uKGNhbGxiYWNrOiBCZWdpblRyYW5zYWN0aW9uQ2FsbGJhY2ssIG5hbWUgPSAnJywgaXNvbGF0aW9uTGV2ZWwgPSB0aGlzLmNvbmZpZy5vcHRpb25zLmlzb2xhdGlvbkxldmVsKSB7XG4gICAgYXNzZXJ0VmFsaWRJc29sYXRpb25MZXZlbChpc29sYXRpb25MZXZlbCwgJ2lzb2xhdGlvbkxldmVsJyk7XG5cbiAgICBjb25zdCB0cmFuc2FjdGlvbiA9IG5ldyBUcmFuc2FjdGlvbihuYW1lLCBpc29sYXRpb25MZXZlbCk7XG5cbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy50ZHNWZXJzaW9uIDwgJzdfMicpIHtcbiAgICAgIHJldHVybiB0aGlzLmV4ZWNTcWxCYXRjaChuZXcgUmVxdWVzdCgnU0VUIFRSQU5TQUNUSU9OIElTT0xBVElPTiBMRVZFTCAnICsgKHRyYW5zYWN0aW9uLmlzb2xhdGlvbkxldmVsVG9UU1FMKCkpICsgJztCRUdJTiBUUkFOICcgKyB0cmFuc2FjdGlvbi5uYW1lLCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMudHJhbnNhY3Rpb25EZXB0aCsrO1xuICAgICAgICBpZiAodGhpcy50cmFuc2FjdGlvbkRlcHRoID09PSAxKSB7XG4gICAgICAgICAgdGhpcy5pblRyYW5zYWN0aW9uID0gdHJ1ZTtcbiAgICAgICAgfVxuICAgICAgICBjYWxsYmFjayhlcnIpO1xuICAgICAgfSkpO1xuICAgIH1cblxuICAgIGNvbnN0IHJlcXVlc3QgPSBuZXcgUmVxdWVzdCh1bmRlZmluZWQsIChlcnIpID0+IHtcbiAgICAgIHJldHVybiBjYWxsYmFjayhlcnIsIHRoaXMuY3VycmVudFRyYW5zYWN0aW9uRGVzY3JpcHRvcigpKTtcbiAgICB9KTtcbiAgICByZXR1cm4gdGhpcy5tYWtlUmVxdWVzdChyZXF1ZXN0LCBUWVBFLlRSQU5TQUNUSU9OX01BTkFHRVIsIHRyYW5zYWN0aW9uLmJlZ2luUGF5bG9hZCh0aGlzLmN1cnJlbnRUcmFuc2FjdGlvbkRlc2NyaXB0b3IoKSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbW1pdCBhIHRyYW5zYWN0aW9uLlxuICAgKlxuICAgKiBUaGVyZSBzaG91bGQgYmUgYW4gYWN0aXZlIHRyYW5zYWN0aW9uIC0gdGhhdCBpcywgW1tiZWdpblRyYW5zYWN0aW9uXV1cbiAgICogc2hvdWxkIGhhdmUgYmVlbiBwcmV2aW91c2x5IGNhbGxlZC5cbiAgICpcbiAgICogQHBhcmFtIGNhbGxiYWNrXG4gICAqIEBwYXJhbSBuYW1lIEEgc3RyaW5nIHJlcHJlc2VudGluZyBhIG5hbWUgdG8gYXNzb2NpYXRlIHdpdGggdGhlIHRyYW5zYWN0aW9uLlxuICAgKiAgIE9wdGlvbmFsLCBhbmQgZGVmYXVsdHMgdG8gYW4gZW1wdHkgc3RyaW5nLiBSZXF1aXJlZCB3aGVuIGBpc29sYXRpb25MZXZlbGBpcyBwcmVzZW50LlxuICAgKi9cbiAgY29tbWl0VHJhbnNhY3Rpb24oY2FsbGJhY2s6IENvbW1pdFRyYW5zYWN0aW9uQ2FsbGJhY2ssIG5hbWUgPSAnJykge1xuICAgIGNvbnN0IHRyYW5zYWN0aW9uID0gbmV3IFRyYW5zYWN0aW9uKG5hbWUpO1xuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLnRkc1ZlcnNpb24gPCAnN18yJykge1xuICAgICAgcmV0dXJuIHRoaXMuZXhlY1NxbEJhdGNoKG5ldyBSZXF1ZXN0KCdDT01NSVQgVFJBTiAnICsgdHJhbnNhY3Rpb24ubmFtZSwgKGVycikgPT4ge1xuICAgICAgICB0aGlzLnRyYW5zYWN0aW9uRGVwdGgtLTtcbiAgICAgICAgaWYgKHRoaXMudHJhbnNhY3Rpb25EZXB0aCA9PT0gMCkge1xuICAgICAgICAgIHRoaXMuaW5UcmFuc2FjdGlvbiA9IGZhbHNlO1xuICAgICAgICB9XG5cbiAgICAgICAgY2FsbGJhY2soZXJyKTtcbiAgICAgIH0pKTtcbiAgICB9XG4gICAgY29uc3QgcmVxdWVzdCA9IG5ldyBSZXF1ZXN0KHVuZGVmaW5lZCwgY2FsbGJhY2spO1xuICAgIHJldHVybiB0aGlzLm1ha2VSZXF1ZXN0KHJlcXVlc3QsIFRZUEUuVFJBTlNBQ1RJT05fTUFOQUdFUiwgdHJhbnNhY3Rpb24uY29tbWl0UGF5bG9hZCh0aGlzLmN1cnJlbnRUcmFuc2FjdGlvbkRlc2NyaXB0b3IoKSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJvbGxiYWNrIGEgdHJhbnNhY3Rpb24uXG4gICAqXG4gICAqIFRoZXJlIHNob3VsZCBiZSBhbiBhY3RpdmUgdHJhbnNhY3Rpb24gLSB0aGF0IGlzLCBbW2JlZ2luVHJhbnNhY3Rpb25dXVxuICAgKiBzaG91bGQgaGF2ZSBiZWVuIHByZXZpb3VzbHkgY2FsbGVkLlxuICAgKlxuICAgKiBAcGFyYW0gY2FsbGJhY2tcbiAgICogQHBhcmFtIG5hbWUgQSBzdHJpbmcgcmVwcmVzZW50aW5nIGEgbmFtZSB0byBhc3NvY2lhdGUgd2l0aCB0aGUgdHJhbnNhY3Rpb24uXG4gICAqICAgT3B0aW9uYWwsIGFuZCBkZWZhdWx0cyB0byBhbiBlbXB0eSBzdHJpbmcuXG4gICAqICAgUmVxdWlyZWQgd2hlbiBgaXNvbGF0aW9uTGV2ZWxgIGlzIHByZXNlbnQuXG4gICAqL1xuICByb2xsYmFja1RyYW5zYWN0aW9uKGNhbGxiYWNrOiBSb2xsYmFja1RyYW5zYWN0aW9uQ2FsbGJhY2ssIG5hbWUgPSAnJykge1xuICAgIGNvbnN0IHRyYW5zYWN0aW9uID0gbmV3IFRyYW5zYWN0aW9uKG5hbWUpO1xuICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLnRkc1ZlcnNpb24gPCAnN18yJykge1xuICAgICAgcmV0dXJuIHRoaXMuZXhlY1NxbEJhdGNoKG5ldyBSZXF1ZXN0KCdST0xMQkFDSyBUUkFOICcgKyB0cmFuc2FjdGlvbi5uYW1lLCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMudHJhbnNhY3Rpb25EZXB0aC0tO1xuICAgICAgICBpZiAodGhpcy50cmFuc2FjdGlvbkRlcHRoID09PSAwKSB7XG4gICAgICAgICAgdGhpcy5pblRyYW5zYWN0aW9uID0gZmFsc2U7XG4gICAgICAgIH1cbiAgICAgICAgY2FsbGJhY2soZXJyKTtcbiAgICAgIH0pKTtcbiAgICB9XG4gICAgY29uc3QgcmVxdWVzdCA9IG5ldyBSZXF1ZXN0KHVuZGVmaW5lZCwgY2FsbGJhY2spO1xuICAgIHJldHVybiB0aGlzLm1ha2VSZXF1ZXN0KHJlcXVlc3QsIFRZUEUuVFJBTlNBQ1RJT05fTUFOQUdFUiwgdHJhbnNhY3Rpb24ucm9sbGJhY2tQYXlsb2FkKHRoaXMuY3VycmVudFRyYW5zYWN0aW9uRGVzY3JpcHRvcigpKSk7XG4gIH1cblxuICAvKipcbiAgICogU2V0IGEgc2F2ZXBvaW50IHdpdGhpbiBhIHRyYW5zYWN0aW9uLlxuICAgKlxuICAgKiBUaGVyZSBzaG91bGQgYmUgYW4gYWN0aXZlIHRyYW5zYWN0aW9uIC0gdGhhdCBpcywgW1tiZWdpblRyYW5zYWN0aW9uXV1cbiAgICogc2hvdWxkIGhhdmUgYmVlbiBwcmV2aW91c2x5IGNhbGxlZC5cbiAgICpcbiAgICogQHBhcmFtIGNhbGxiYWNrXG4gICAqIEBwYXJhbSBuYW1lIEEgc3RyaW5nIHJlcHJlc2VudGluZyBhIG5hbWUgdG8gYXNzb2NpYXRlIHdpdGggdGhlIHRyYW5zYWN0aW9uLlxcXG4gICAqICAgT3B0aW9uYWwsIGFuZCBkZWZhdWx0cyB0byBhbiBlbXB0eSBzdHJpbmcuXG4gICAqICAgUmVxdWlyZWQgd2hlbiBgaXNvbGF0aW9uTGV2ZWxgIGlzIHByZXNlbnQuXG4gICAqL1xuICBzYXZlVHJhbnNhY3Rpb24oY2FsbGJhY2s6IFNhdmVUcmFuc2FjdGlvbkNhbGxiYWNrLCBuYW1lOiBzdHJpbmcpIHtcbiAgICBjb25zdCB0cmFuc2FjdGlvbiA9IG5ldyBUcmFuc2FjdGlvbihuYW1lKTtcbiAgICBpZiAodGhpcy5jb25maWcub3B0aW9ucy50ZHNWZXJzaW9uIDwgJzdfMicpIHtcbiAgICAgIHJldHVybiB0aGlzLmV4ZWNTcWxCYXRjaChuZXcgUmVxdWVzdCgnU0FWRSBUUkFOICcgKyB0cmFuc2FjdGlvbi5uYW1lLCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMudHJhbnNhY3Rpb25EZXB0aCsrO1xuICAgICAgICBjYWxsYmFjayhlcnIpO1xuICAgICAgfSkpO1xuICAgIH1cbiAgICBjb25zdCByZXF1ZXN0ID0gbmV3IFJlcXVlc3QodW5kZWZpbmVkLCBjYWxsYmFjayk7XG4gICAgcmV0dXJuIHRoaXMubWFrZVJlcXVlc3QocmVxdWVzdCwgVFlQRS5UUkFOU0FDVElPTl9NQU5BR0VSLCB0cmFuc2FjdGlvbi5zYXZlUGF5bG9hZCh0aGlzLmN1cnJlbnRUcmFuc2FjdGlvbkRlc2NyaXB0b3IoKSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJ1biB0aGUgZ2l2ZW4gY2FsbGJhY2sgYWZ0ZXIgc3RhcnRpbmcgYSB0cmFuc2FjdGlvbiwgYW5kIGNvbW1pdCBvclxuICAgKiByb2xsYmFjayB0aGUgdHJhbnNhY3Rpb24gYWZ0ZXJ3YXJkcy5cbiAgICpcbiAgICogVGhpcyBpcyBhIGhlbHBlciB0aGF0IGVtcGxveXMgW1tiZWdpblRyYW5zYWN0aW9uXV0sIFtbY29tbWl0VHJhbnNhY3Rpb25dXSxcbiAgICogW1tyb2xsYmFja1RyYW5zYWN0aW9uXV0sIGFuZCBbW3NhdmVUcmFuc2FjdGlvbl1dIHRvIGdyZWF0bHkgc2ltcGxpZnkgdGhlXG4gICAqIHVzZSBvZiBkYXRhYmFzZSB0cmFuc2FjdGlvbnMgYW5kIGF1dG9tYXRpY2FsbHkgaGFuZGxlIHRyYW5zYWN0aW9uIG5lc3RpbmcuXG4gICAqXG4gICAqIEBwYXJhbSBjYlxuICAgKiBAcGFyYW0gaXNvbGF0aW9uTGV2ZWxcbiAgICogICBUaGUgaXNvbGF0aW9uIGxldmVsIHRoYXQgdGhlIHRyYW5zYWN0aW9uIGlzIHRvIGJlIHJ1biB3aXRoLlxuICAgKlxuICAgKiAgIFRoZSBpc29sYXRpb24gbGV2ZWxzIGFyZSBhdmFpbGFibGUgZnJvbSBgcmVxdWlyZSgndGVkaW91cycpLklTT0xBVElPTl9MRVZFTGAuXG4gICAqICAgKiBgUkVBRF9VTkNPTU1JVFRFRGBcbiAgICogICAqIGBSRUFEX0NPTU1JVFRFRGBcbiAgICogICAqIGBSRVBFQVRBQkxFX1JFQURgXG4gICAqICAgKiBgU0VSSUFMSVpBQkxFYFxuICAgKiAgICogYFNOQVBTSE9UYFxuICAgKlxuICAgKiAgIE9wdGlvbmFsLCBhbmQgZGVmYXVsdHMgdG8gdGhlIENvbm5lY3Rpb24ncyBpc29sYXRpb24gbGV2ZWwuXG4gICAqL1xuICB0cmFuc2FjdGlvbihjYjogKGVycjogRXJyb3IgfCBudWxsIHwgdW5kZWZpbmVkLCB0eERvbmU/OiA8VCBleHRlbmRzIFRyYW5zYWN0aW9uRG9uZUNhbGxiYWNrPihlcnI6IEVycm9yIHwgbnVsbCB8IHVuZGVmaW5lZCwgZG9uZTogVCwgLi4uYXJnczogQ2FsbGJhY2tQYXJhbWV0ZXJzPFQ+KSA9PiB2b2lkKSA9PiB2b2lkLCBpc29sYXRpb25MZXZlbD86IHR5cGVvZiBJU09MQVRJT05fTEVWRUxba2V5b2YgdHlwZW9mIElTT0xBVElPTl9MRVZFTF0pIHtcbiAgICBpZiAodHlwZW9mIGNiICE9PSAnZnVuY3Rpb24nKSB7XG4gICAgICB0aHJvdyBuZXcgVHlwZUVycm9yKCdgY2JgIG11c3QgYmUgYSBmdW5jdGlvbicpO1xuICAgIH1cblxuICAgIGNvbnN0IHVzZVNhdmVwb2ludCA9IHRoaXMuaW5UcmFuc2FjdGlvbjtcbiAgICBjb25zdCBuYW1lID0gJ190ZWRpb3VzXycgKyAoY3J5cHRvLnJhbmRvbUJ5dGVzKDEwKS50b1N0cmluZygnaGV4JykpO1xuICAgIGNvbnN0IHR4RG9uZTogPFQgZXh0ZW5kcyBUcmFuc2FjdGlvbkRvbmVDYWxsYmFjaz4oZXJyOiBFcnJvciB8IG51bGwgfCB1bmRlZmluZWQsIGRvbmU6IFQsIC4uLmFyZ3M6IENhbGxiYWNrUGFyYW1ldGVyczxUPikgPT4gdm9pZCA9IChlcnIsIGRvbmUsIC4uLmFyZ3MpID0+IHtcbiAgICAgIGlmIChlcnIpIHtcbiAgICAgICAgaWYgKHRoaXMuaW5UcmFuc2FjdGlvbiAmJiB0aGlzLnN0YXRlID09PSB0aGlzLlNUQVRFLkxPR0dFRF9JTikge1xuICAgICAgICAgIHRoaXMucm9sbGJhY2tUcmFuc2FjdGlvbigodHhFcnIpID0+IHtcbiAgICAgICAgICAgIGRvbmUodHhFcnIgfHwgZXJyLCAuLi5hcmdzKTtcbiAgICAgICAgICB9LCBuYW1lKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICBkb25lKGVyciwgLi4uYXJncyk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSBpZiAodXNlU2F2ZXBvaW50KSB7XG4gICAgICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLnRkc1ZlcnNpb24gPCAnN18yJykge1xuICAgICAgICAgIHRoaXMudHJhbnNhY3Rpb25EZXB0aC0tO1xuICAgICAgICB9XG4gICAgICAgIGRvbmUobnVsbCwgLi4uYXJncyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aGlzLmNvbW1pdFRyYW5zYWN0aW9uKCh0eEVycikgPT4ge1xuICAgICAgICAgIGRvbmUodHhFcnIsIC4uLmFyZ3MpO1xuICAgICAgICB9LCBuYW1lKTtcbiAgICAgIH1cbiAgICB9O1xuXG4gICAgaWYgKHVzZVNhdmVwb2ludCkge1xuICAgICAgcmV0dXJuIHRoaXMuc2F2ZVRyYW5zYWN0aW9uKChlcnIpID0+IHtcbiAgICAgICAgaWYgKGVycikge1xuICAgICAgICAgIHJldHVybiBjYihlcnIpO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKGlzb2xhdGlvbkxldmVsKSB7XG4gICAgICAgICAgcmV0dXJuIHRoaXMuZXhlY1NxbEJhdGNoKG5ldyBSZXF1ZXN0KCdTRVQgdHJhbnNhY3Rpb24gaXNvbGF0aW9uIGxldmVsICcgKyB0aGlzLmdldElzb2xhdGlvbkxldmVsVGV4dChpc29sYXRpb25MZXZlbCksIChlcnIpID0+IHtcbiAgICAgICAgICAgIHJldHVybiBjYihlcnIsIHR4RG9uZSk7XG4gICAgICAgICAgfSkpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHJldHVybiBjYihudWxsLCB0eERvbmUpO1xuICAgICAgICB9XG4gICAgICB9LCBuYW1lKTtcbiAgICB9IGVsc2Uge1xuICAgICAgcmV0dXJuIHRoaXMuYmVnaW5UcmFuc2FjdGlvbigoZXJyKSA9PiB7XG4gICAgICAgIGlmIChlcnIpIHtcbiAgICAgICAgICByZXR1cm4gY2IoZXJyKTtcbiAgICAgICAgfVxuXG4gICAgICAgIHJldHVybiBjYihudWxsLCB0eERvbmUpO1xuICAgICAgfSwgbmFtZSwgaXNvbGF0aW9uTGV2ZWwpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgbWFrZVJlcXVlc3QocmVxdWVzdDogUmVxdWVzdCB8IEJ1bGtMb2FkLCBwYWNrZXRUeXBlOiBudW1iZXIsIHBheWxvYWQ6IChJdGVyYWJsZTxCdWZmZXI+IHwgQXN5bmNJdGVyYWJsZTxCdWZmZXI+KSAmIHsgdG9TdHJpbmc6IChpbmRlbnQ/OiBzdHJpbmcpID0+IHN0cmluZyB9KSB7XG4gICAgaWYgKHRoaXMuc3RhdGUgIT09IHRoaXMuU1RBVEUuTE9HR0VEX0lOKSB7XG4gICAgICBjb25zdCBtZXNzYWdlID0gJ1JlcXVlc3RzIGNhbiBvbmx5IGJlIG1hZGUgaW4gdGhlICcgKyB0aGlzLlNUQVRFLkxPR0dFRF9JTi5uYW1lICsgJyBzdGF0ZSwgbm90IHRoZSAnICsgdGhpcy5zdGF0ZS5uYW1lICsgJyBzdGF0ZSc7XG4gICAgICB0aGlzLmRlYnVnLmxvZyhtZXNzYWdlKTtcbiAgICAgIHJlcXVlc3QuY2FsbGJhY2sobmV3IFJlcXVlc3RFcnJvcihtZXNzYWdlLCAnRUlOVkFMSURTVEFURScpKTtcbiAgICB9IGVsc2UgaWYgKHJlcXVlc3QuY2FuY2VsZWQpIHtcbiAgICAgIHByb2Nlc3MubmV4dFRpY2soKCkgPT4ge1xuICAgICAgICByZXF1ZXN0LmNhbGxiYWNrKG5ldyBSZXF1ZXN0RXJyb3IoJ0NhbmNlbGVkLicsICdFQ0FOQ0VMJykpO1xuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlmIChwYWNrZXRUeXBlID09PSBUWVBFLlNRTF9CQVRDSCkge1xuICAgICAgICB0aGlzLmlzU3FsQmF0Y2ggPSB0cnVlO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy5pc1NxbEJhdGNoID0gZmFsc2U7XG4gICAgICB9XG5cbiAgICAgIHRoaXMucmVxdWVzdCA9IHJlcXVlc3Q7XG4gICAgICByZXF1ZXN0LmNvbm5lY3Rpb24hID0gdGhpcztcbiAgICAgIHJlcXVlc3Qucm93Q291bnQhID0gMDtcbiAgICAgIHJlcXVlc3Qucm93cyEgPSBbXTtcbiAgICAgIHJlcXVlc3QucnN0ISA9IFtdO1xuXG4gICAgICBjb25zdCBvbkNhbmNlbCA9ICgpID0+IHtcbiAgICAgICAgcGF5bG9hZFN0cmVhbS51bnBpcGUobWVzc2FnZSk7XG4gICAgICAgIHBheWxvYWRTdHJlYW0uZGVzdHJveShuZXcgUmVxdWVzdEVycm9yKCdDYW5jZWxlZC4nLCAnRUNBTkNFTCcpKTtcblxuICAgICAgICAvLyBzZXQgdGhlIGlnbm9yZSBiaXQgYW5kIGVuZCB0aGUgbWVzc2FnZS5cbiAgICAgICAgbWVzc2FnZS5pZ25vcmUgPSB0cnVlO1xuICAgICAgICBtZXNzYWdlLmVuZCgpO1xuXG4gICAgICAgIGlmIChyZXF1ZXN0IGluc3RhbmNlb2YgUmVxdWVzdCAmJiByZXF1ZXN0LnBhdXNlZCkge1xuICAgICAgICAgIC8vIHJlc3VtZSB0aGUgcmVxdWVzdCBpZiBpdCB3YXMgcGF1c2VkIHNvIHdlIGNhbiByZWFkIHRoZSByZW1haW5pbmcgdG9rZW5zXG4gICAgICAgICAgcmVxdWVzdC5yZXN1bWUoKTtcbiAgICAgICAgfVxuICAgICAgfTtcblxuICAgICAgcmVxdWVzdC5vbmNlKCdjYW5jZWwnLCBvbkNhbmNlbCk7XG5cbiAgICAgIHRoaXMuY3JlYXRlUmVxdWVzdFRpbWVyKCk7XG5cbiAgICAgIGNvbnN0IG1lc3NhZ2UgPSBuZXcgTWVzc2FnZSh7IHR5cGU6IHBhY2tldFR5cGUsIHJlc2V0Q29ubmVjdGlvbjogdGhpcy5yZXNldENvbm5lY3Rpb25Pbk5leHRSZXF1ZXN0IH0pO1xuICAgICAgdGhpcy5tZXNzYWdlSW8ub3V0Z29pbmdNZXNzYWdlU3RyZWFtLndyaXRlKG1lc3NhZ2UpO1xuICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5TRU5UX0NMSUVOVF9SRVFVRVNUKTtcblxuICAgICAgbWVzc2FnZS5vbmNlKCdmaW5pc2gnLCAoKSA9PiB7XG4gICAgICAgIHJlcXVlc3QucmVtb3ZlTGlzdGVuZXIoJ2NhbmNlbCcsIG9uQ2FuY2VsKTtcbiAgICAgICAgcmVxdWVzdC5vbmNlKCdjYW5jZWwnLCB0aGlzLl9jYW5jZWxBZnRlclJlcXVlc3RTZW50KTtcblxuICAgICAgICB0aGlzLnJlc2V0Q29ubmVjdGlvbk9uTmV4dFJlcXVlc3QgPSBmYWxzZTtcbiAgICAgICAgdGhpcy5kZWJ1Zy5wYXlsb2FkKGZ1bmN0aW9uKCkge1xuICAgICAgICAgIHJldHVybiBwYXlsb2FkIS50b1N0cmluZygnICAnKTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgICAgY29uc3QgcGF5bG9hZFN0cmVhbSA9IFJlYWRhYmxlLmZyb20ocGF5bG9hZCk7XG4gICAgICBwYXlsb2FkU3RyZWFtLm9uY2UoJ2Vycm9yJywgKGVycm9yKSA9PiB7XG4gICAgICAgIHBheWxvYWRTdHJlYW0udW5waXBlKG1lc3NhZ2UpO1xuXG4gICAgICAgIC8vIE9ubHkgc2V0IGEgcmVxdWVzdCBlcnJvciBpZiBubyBlcnJvciB3YXMgc2V0IHlldC5cbiAgICAgICAgcmVxdWVzdC5lcnJvciA/Pz0gZXJyb3I7XG5cbiAgICAgICAgbWVzc2FnZS5pZ25vcmUgPSB0cnVlO1xuICAgICAgICBtZXNzYWdlLmVuZCgpO1xuICAgICAgfSk7XG4gICAgICBwYXlsb2FkU3RyZWFtLnBpcGUobWVzc2FnZSk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIENhbmNlbCBjdXJyZW50bHkgZXhlY3V0ZWQgcmVxdWVzdC5cbiAgICovXG4gIGNhbmNlbCgpIHtcbiAgICBpZiAoIXRoaXMucmVxdWVzdCkge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cblxuICAgIGlmICh0aGlzLnJlcXVlc3QuY2FuY2VsZWQpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICB0aGlzLnJlcXVlc3QuY2FuY2VsKCk7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICAvKipcbiAgICogUmVzZXQgdGhlIGNvbm5lY3Rpb24gdG8gaXRzIGluaXRpYWwgc3RhdGUuXG4gICAqIENhbiBiZSB1c2VmdWwgZm9yIGNvbm5lY3Rpb24gcG9vbCBpbXBsZW1lbnRhdGlvbnMuXG4gICAqXG4gICAqIEBwYXJhbSBjYWxsYmFja1xuICAgKi9cbiAgcmVzZXQoY2FsbGJhY2s6IFJlc2V0Q2FsbGJhY2spIHtcbiAgICBjb25zdCByZXF1ZXN0ID0gbmV3IFJlcXVlc3QodGhpcy5nZXRJbml0aWFsU3FsKCksIChlcnIpID0+IHtcbiAgICAgIGlmICh0aGlzLmNvbmZpZy5vcHRpb25zLnRkc1ZlcnNpb24gPCAnN18yJykge1xuICAgICAgICB0aGlzLmluVHJhbnNhY3Rpb24gPSBmYWxzZTtcbiAgICAgIH1cbiAgICAgIGNhbGxiYWNrKGVycik7XG4gICAgfSk7XG4gICAgdGhpcy5yZXNldENvbm5lY3Rpb25Pbk5leHRSZXF1ZXN0ID0gdHJ1ZTtcbiAgICB0aGlzLmV4ZWNTcWxCYXRjaChyZXF1ZXN0KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgY3VycmVudFRyYW5zYWN0aW9uRGVzY3JpcHRvcigpIHtcbiAgICByZXR1cm4gdGhpcy50cmFuc2FjdGlvbkRlc2NyaXB0b3JzW3RoaXMudHJhbnNhY3Rpb25EZXNjcmlwdG9ycy5sZW5ndGggLSAxXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAcHJpdmF0ZVxuICAgKi9cbiAgZ2V0SXNvbGF0aW9uTGV2ZWxUZXh0KGlzb2xhdGlvbkxldmVsOiB0eXBlb2YgSVNPTEFUSU9OX0xFVkVMW2tleW9mIHR5cGVvZiBJU09MQVRJT05fTEVWRUxdKSB7XG4gICAgc3dpdGNoIChpc29sYXRpb25MZXZlbCkge1xuICAgICAgY2FzZSBJU09MQVRJT05fTEVWRUwuUkVBRF9VTkNPTU1JVFRFRDpcbiAgICAgICAgcmV0dXJuICdyZWFkIHVuY29tbWl0dGVkJztcbiAgICAgIGNhc2UgSVNPTEFUSU9OX0xFVkVMLlJFUEVBVEFCTEVfUkVBRDpcbiAgICAgICAgcmV0dXJuICdyZXBlYXRhYmxlIHJlYWQnO1xuICAgICAgY2FzZSBJU09MQVRJT05fTEVWRUwuU0VSSUFMSVpBQkxFOlxuICAgICAgICByZXR1cm4gJ3NlcmlhbGl6YWJsZSc7XG4gICAgICBjYXNlIElTT0xBVElPTl9MRVZFTC5TTkFQU0hPVDpcbiAgICAgICAgcmV0dXJuICdzbmFwc2hvdCc7XG4gICAgICBkZWZhdWx0OlxuICAgICAgICByZXR1cm4gJ3JlYWQgY29tbWl0dGVkJztcbiAgICB9XG4gIH1cbn1cblxubGV0IGF6dXJlQURQYXNzd29yZENsaWVudElkRGVwcmVjYXRpb25XYXJuaW5nRW1pdHRlZCA9IGZhbHNlO1xuZnVuY3Rpb24gZW1pdEF6dXJlQURQYXNzd29yZENsaWVudElkRGVwcmVjYXRpb25XYXJuaW5nKCkge1xuICBpZiAoYXp1cmVBRFBhc3N3b3JkQ2xpZW50SWREZXByZWNhdGlvbldhcm5pbmdFbWl0dGVkKSB7XG4gICAgcmV0dXJuO1xuICB9XG5cbiAgYXp1cmVBRFBhc3N3b3JkQ2xpZW50SWREZXByZWNhdGlvbldhcm5pbmdFbWl0dGVkID0gdHJ1ZTtcblxuICBwcm9jZXNzLmVtaXRXYXJuaW5nKFxuICAgICdXaGVuIHVzaW5nIHRoZSBgYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1wYXNzd29yZGAgYXV0aGVudGljYXRpb24gbWV0aG9kLCBwbGVhc2UgcHJvdmlkZSBhIHZhbHVlIGZvciB0aGUgYGNsaWVudElkYCBvcHRpb24uICcgK1xuICAgICdUaGlzIG9wdGlvbiB3aWxsIGJlIHJlcXVpcmVkIGluIGEgZnV0dXJlIHJlbGVhc2UuJyxcbiAgICAnRGVwcmVjYXRpb25XYXJuaW5nJyxcbiAgICBDb25uZWN0aW9uLnByb3RvdHlwZS5vblxuICApO1xufVxuXG5mdW5jdGlvbiBpc1RyYW5zaWVudEVycm9yKGVycm9yOiBBZ2dyZWdhdGVFcnJvciB8IENvbm5lY3Rpb25FcnJvcik6IGJvb2xlYW4ge1xuICBpZiAoZXJyb3IgaW5zdGFuY2VvZiBBZ2dyZWdhdGVFcnJvcikge1xuICAgIGVycm9yID0gZXJyb3IuZXJyb3JzWzBdO1xuICB9XG4gIHJldHVybiAoZXJyb3IgaW5zdGFuY2VvZiBDb25uZWN0aW9uRXJyb3IpICYmICEhZXJyb3IuaXNUcmFuc2llbnQ7XG59XG5cbmV4cG9ydCBkZWZhdWx0IENvbm5lY3Rpb247XG5tb2R1bGUuZXhwb3J0cyA9IENvbm5lY3Rpb247XG5cbkNvbm5lY3Rpb24ucHJvdG90eXBlLlNUQVRFID0ge1xuICBJTklUSUFMSVpFRDoge1xuICAgIG5hbWU6ICdJbml0aWFsaXplZCcsXG4gICAgZXZlbnRzOiB7fVxuICB9LFxuICBDT05ORUNUSU5HOiB7XG4gICAgbmFtZTogJ0Nvbm5lY3RpbmcnLFxuICAgIGVudGVyOiBmdW5jdGlvbigpIHtcbiAgICAgIHRoaXMuaW5pdGlhbGlzZUNvbm5lY3Rpb24oKTtcbiAgICB9LFxuICAgIGV2ZW50czoge1xuICAgICAgc29ja2V0RXJyb3I6IGZ1bmN0aW9uKCkge1xuICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgIH0sXG4gICAgICBjb25uZWN0VGltZW91dDogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgfVxuICAgIH1cbiAgfSxcbiAgU0VOVF9QUkVMT0dJTjoge1xuICAgIG5hbWU6ICdTZW50UHJlbG9naW4nLFxuICAgIGVudGVyOiBmdW5jdGlvbigpIHtcbiAgICAgIHRoaXMuZW1wdHlNZXNzYWdlQnVmZmVyKCk7XG5cbiAgICAgIHRoaXMubWVzc2FnZUlvLnJlYWRNZXNzYWdlKCkudGhlbigobWVzc2FnZSkgPT4ge1xuICAgICAgICB0aGlzLmRpc3BhdGNoRXZlbnQoJ21lc3NhZ2UnLCBtZXNzYWdlKTtcbiAgICAgIH0sIChlcnIpID0+IHtcbiAgICAgICAgdGhpcy5zb2NrZXRFcnJvcihlcnIpO1xuICAgICAgfSk7XG4gICAgfSxcbiAgICBldmVudHM6IHtcbiAgICAgIHNvY2tldEVycm9yOiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9LFxuICAgICAgY29ubmVjdFRpbWVvdXQ6IGZ1bmN0aW9uKCkge1xuICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgIH0sXG4gICAgICBtZXNzYWdlOiBmdW5jdGlvbihtZXNzYWdlKSB7XG4gICAgICAgIG1lc3NhZ2Uub24oJ2RhdGEnLCAoZGF0YSkgPT4ge1xuICAgICAgICAgIHRoaXMuYWRkVG9NZXNzYWdlQnVmZmVyKGRhdGEpO1xuICAgICAgICB9KTtcblxuICAgICAgICBtZXNzYWdlLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICBjb25zdCBwcmVsb2dpblBheWxvYWQgPSBuZXcgUHJlbG9naW5QYXlsb2FkKHRoaXMubWVzc2FnZUJ1ZmZlcik7XG4gICAgICAgICAgdGhpcy5kZWJ1Zy5wYXlsb2FkKGZ1bmN0aW9uKCkge1xuICAgICAgICAgICAgcmV0dXJuIHByZWxvZ2luUGF5bG9hZC50b1N0cmluZygnICAnKTtcbiAgICAgICAgICB9KTtcblxuICAgICAgICAgIGlmIChwcmVsb2dpblBheWxvYWQuZmVkQXV0aFJlcXVpcmVkID09PSAxKSB7XG4gICAgICAgICAgICB0aGlzLmZlZEF1dGhSZXF1aXJlZCA9IHRydWU7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgaWYgKHByZWxvZ2luUGF5bG9hZC5lbmNyeXB0aW9uU3RyaW5nID09PSAnT04nIHx8IHByZWxvZ2luUGF5bG9hZC5lbmNyeXB0aW9uU3RyaW5nID09PSAnUkVRJykge1xuICAgICAgICAgICAgaWYgKCF0aGlzLmNvbmZpZy5vcHRpb25zLmVuY3J5cHQpIHtcbiAgICAgICAgICAgICAgdGhpcy5lbWl0KCdjb25uZWN0JywgbmV3IENvbm5lY3Rpb25FcnJvcihcIlNlcnZlciByZXF1aXJlcyBlbmNyeXB0aW9uLCBzZXQgJ2VuY3J5cHQnIGNvbmZpZyBvcHRpb24gdG8gdHJ1ZS5cIiwgJ0VFTkNSWVBUJykpO1xuICAgICAgICAgICAgICByZXR1cm4gdGhpcy5jbG9zZSgpO1xuICAgICAgICAgICAgfVxuXG4gICAgICAgICAgICB0aGlzLm1lc3NhZ2VJby5zdGFydFRscyh0aGlzLnNlY3VyZUNvbnRleHQsIHRoaXMucm91dGluZ0RhdGE/LnNlcnZlciA/PyB0aGlzLmNvbmZpZy5zZXJ2ZXIsIHRoaXMuY29uZmlnLm9wdGlvbnMudHJ1c3RTZXJ2ZXJDZXJ0aWZpY2F0ZSk7XG4gICAgICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLlNFTlRfVExTU1NMTkVHT1RJQVRJT04pO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB0aGlzLnNlbmRMb2dpbjdQYWNrZXQoKTtcblxuICAgICAgICAgICAgY29uc3QgeyBhdXRoZW50aWNhdGlvbiB9ID0gdGhpcy5jb25maWc7XG4gICAgICAgICAgICBpZiAoYXV0aGVudGljYXRpb24udHlwZSA9PT0gJ250bG0nKSB7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuU0VOVF9MT0dJTjdfV0lUSF9OVExNKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuU0VOVF9MT0dJTjdfV0lUSF9TVEFOREFSRF9MT0dJTik7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gIH0sXG4gIFJFUk9VVElORzoge1xuICAgIG5hbWU6ICdSZVJvdXRpbmcnLFxuICAgIGVudGVyOiBmdW5jdGlvbigpIHtcbiAgICAgIHRoaXMuY2xlYW51cENvbm5lY3Rpb24oQ0xFQU5VUF9UWVBFLlJFRElSRUNUKTtcbiAgICB9LFxuICAgIGV2ZW50czoge1xuICAgICAgbWVzc2FnZTogZnVuY3Rpb24oKSB7XG4gICAgICB9LFxuICAgICAgc29ja2V0RXJyb3I6IGZ1bmN0aW9uKCkge1xuICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgIH0sXG4gICAgICBjb25uZWN0VGltZW91dDogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgfSxcbiAgICAgIHJlY29ubmVjdDogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuQ09OTkVDVElORyk7XG4gICAgICB9XG4gICAgfVxuICB9LFxuICBUUkFOU0lFTlRfRkFJTFVSRV9SRVRSWToge1xuICAgIG5hbWU6ICdUUkFOU0lFTlRfRkFJTFVSRV9SRVRSWScsXG4gICAgZW50ZXI6IGZ1bmN0aW9uKCkge1xuICAgICAgdGhpcy5jdXJUcmFuc2llbnRSZXRyeUNvdW50Kys7XG4gICAgICB0aGlzLmNsZWFudXBDb25uZWN0aW9uKENMRUFOVVBfVFlQRS5SRVRSWSk7XG4gICAgfSxcbiAgICBldmVudHM6IHtcbiAgICAgIG1lc3NhZ2U6IGZ1bmN0aW9uKCkge1xuICAgICAgfSxcbiAgICAgIHNvY2tldEVycm9yOiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9LFxuICAgICAgY29ubmVjdFRpbWVvdXQ6IGZ1bmN0aW9uKCkge1xuICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgIH0sXG4gICAgICByZXRyeTogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMuY3JlYXRlUmV0cnlUaW1lcigpO1xuICAgICAgfVxuICAgIH1cbiAgfSxcbiAgU0VOVF9UTFNTU0xORUdPVElBVElPTjoge1xuICAgIG5hbWU6ICdTZW50VExTU1NMTmVnb3RpYXRpb24nLFxuICAgIGVudGVyOiBmdW5jdGlvbigpIHtcbiAgICAgIHRoaXMubWVzc2FnZUlvLnJlYWRNZXNzYWdlKCkudGhlbigobWVzc2FnZSkgPT4ge1xuICAgICAgICB0aGlzLmRpc3BhdGNoRXZlbnQoJ21lc3NhZ2UnLCBtZXNzYWdlKTtcbiAgICAgIH0sIChlcnIpID0+IHtcbiAgICAgICAgdGhpcy5zb2NrZXRFcnJvcihlcnIpO1xuICAgICAgfSk7XG4gICAgfSxcbiAgICBldmVudHM6IHtcbiAgICAgIHNvY2tldEVycm9yOiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9LFxuICAgICAgY29ubmVjdFRpbWVvdXQ6IGZ1bmN0aW9uKCkge1xuICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgIH0sXG4gICAgICBtZXNzYWdlOiBmdW5jdGlvbihtZXNzYWdlKSB7XG4gICAgICAgIG1lc3NhZ2Uub24oJ2RhdGEnLCAoZGF0YSkgPT4ge1xuICAgICAgICAgIHRoaXMubWVzc2FnZUlvLnRsc0hhbmRzaGFrZURhdGEoZGF0YSk7XG4gICAgICAgIH0pO1xuXG4gICAgICAgIG1lc3NhZ2Uub25jZSgnZW5kJywgKCkgPT4ge1xuICAgICAgICAgIGlmICh0aGlzLm1lc3NhZ2VJby50bHNOZWdvdGlhdGlvbkNvbXBsZXRlKSB7XG4gICAgICAgICAgICB0aGlzLnNlbmRMb2dpbjdQYWNrZXQoKTtcblxuICAgICAgICAgICAgY29uc3QgeyBhdXRoZW50aWNhdGlvbiB9ID0gdGhpcy5jb25maWc7XG4gICAgICAgICAgICBpZiAoYXV0aGVudGljYXRpb24udHlwZSA9PT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktcGFzc3dvcmQnIHx8IGF1dGhlbnRpY2F0aW9uLnR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS12bScgfHwgYXV0aGVudGljYXRpb24udHlwZSA9PT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktbXNpLWFwcC1zZXJ2aWNlJyB8fCBhdXRoZW50aWNhdGlvbi50eXBlID09PSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1zZXJ2aWNlLXByaW5jaXBhbC1zZWNyZXQnIHx8IGF1dGhlbnRpY2F0aW9uLnR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWRlZmF1bHQnKSB7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuU0VOVF9MT0dJTjdfV0lUSF9GRURBVVRIKTtcbiAgICAgICAgICAgIH0gZWxzZSBpZiAoYXV0aGVudGljYXRpb24udHlwZSA9PT0gJ250bG0nKSB7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuU0VOVF9MT0dJTjdfV0lUSF9OVExNKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuU0VOVF9MT0dJTjdfV0lUSF9TVEFOREFSRF9MT0dJTik7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMubWVzc2FnZUlvLnJlYWRNZXNzYWdlKCkudGhlbigobWVzc2FnZSkgPT4ge1xuICAgICAgICAgICAgICB0aGlzLmRpc3BhdGNoRXZlbnQoJ21lc3NhZ2UnLCBtZXNzYWdlKTtcbiAgICAgICAgICAgIH0sIChlcnIpID0+IHtcbiAgICAgICAgICAgICAgdGhpcy5zb2NrZXRFcnJvcihlcnIpO1xuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gIH0sXG4gIFNFTlRfTE9HSU43X1dJVEhfU1RBTkRBUkRfTE9HSU46IHtcbiAgICBuYW1lOiAnU2VudExvZ2luN1dpdGhTdGFuZGFyZExvZ2luJyxcbiAgICBlbnRlcjogZnVuY3Rpb24oKSB7XG4gICAgICB0aGlzLm1lc3NhZ2VJby5yZWFkTWVzc2FnZSgpLnRoZW4oKG1lc3NhZ2UpID0+IHtcbiAgICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdtZXNzYWdlJywgbWVzc2FnZSk7XG4gICAgICB9LCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyKTtcbiAgICAgIH0pO1xuICAgIH0sXG4gICAgZXZlbnRzOiB7XG4gICAgICBzb2NrZXRFcnJvcjogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgfSxcbiAgICAgIGNvbm5lY3RUaW1lb3V0OiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9LFxuICAgICAgbWVzc2FnZTogZnVuY3Rpb24obWVzc2FnZSkge1xuICAgICAgICBjb25zdCBoYW5kbGVyID0gbmV3IExvZ2luN1Rva2VuSGFuZGxlcih0aGlzKTtcbiAgICAgICAgY29uc3QgdG9rZW5TdHJlYW1QYXJzZXIgPSB0aGlzLmNyZWF0ZVRva2VuU3RyZWFtUGFyc2VyKG1lc3NhZ2UsIGhhbmRsZXIpO1xuXG4gICAgICAgIHRva2VuU3RyZWFtUGFyc2VyLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICBpZiAoaGFuZGxlci5sb2dpbkFja1JlY2VpdmVkKSB7XG4gICAgICAgICAgICBpZiAoaGFuZGxlci5yb3V0aW5nRGF0YSkge1xuICAgICAgICAgICAgICB0aGlzLnJvdXRpbmdEYXRhID0gaGFuZGxlci5yb3V0aW5nRGF0YTtcbiAgICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5SRVJPVVRJTkcpO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5MT0dHRURfSU5fU0VORElOR19JTklUSUFMX1NRTCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIGlmICh0aGlzLmxvZ2luRXJyb3IpIHtcbiAgICAgICAgICAgIGlmIChpc1RyYW5zaWVudEVycm9yKHRoaXMubG9naW5FcnJvcikpIHtcbiAgICAgICAgICAgICAgdGhpcy5kZWJ1Zy5sb2coJ0luaXRpYXRpbmcgcmV0cnkgb24gdHJhbnNpZW50IGVycm9yJyk7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuVFJBTlNJRU5UX0ZBSUxVUkVfUkVUUlkpO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgdGhpcy5lbWl0KCdjb25uZWN0JywgdGhpcy5sb2dpbkVycm9yKTtcbiAgICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIHRoaXMuZW1pdCgnY29ubmVjdCcsIG5ldyBDb25uZWN0aW9uRXJyb3IoJ0xvZ2luIGZhaWxlZC4nLCAnRUxPR0lOJykpO1xuICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gIH0sXG4gIFNFTlRfTE9HSU43X1dJVEhfTlRMTToge1xuICAgIG5hbWU6ICdTZW50TG9naW43V2l0aE5UTE1Mb2dpbicsXG4gICAgZW50ZXI6IGZ1bmN0aW9uKCkge1xuICAgICAgdGhpcy5tZXNzYWdlSW8ucmVhZE1lc3NhZ2UoKS50aGVuKChtZXNzYWdlKSA9PiB7XG4gICAgICAgIHRoaXMuZGlzcGF0Y2hFdmVudCgnbWVzc2FnZScsIG1lc3NhZ2UpO1xuICAgICAgfSwgKGVycikgPT4ge1xuICAgICAgICB0aGlzLnNvY2tldEVycm9yKGVycik7XG4gICAgICB9KTtcbiAgICB9LFxuICAgIGV2ZW50czoge1xuICAgICAgc29ja2V0RXJyb3I6IGZ1bmN0aW9uKCkge1xuICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgIH0sXG4gICAgICBjb25uZWN0VGltZW91dDogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgfSxcbiAgICAgIG1lc3NhZ2U6IGZ1bmN0aW9uKG1lc3NhZ2UpIHtcbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBMb2dpbjdUb2tlbkhhbmRsZXIodGhpcyk7XG4gICAgICAgIGNvbnN0IHRva2VuU3RyZWFtUGFyc2VyID0gdGhpcy5jcmVhdGVUb2tlblN0cmVhbVBhcnNlcihtZXNzYWdlLCBoYW5kbGVyKTtcblxuICAgICAgICB0b2tlblN0cmVhbVBhcnNlci5vbmNlKCdlbmQnLCAoKSA9PiB7XG4gICAgICAgICAgaWYgKGhhbmRsZXIubG9naW5BY2tSZWNlaXZlZCkge1xuICAgICAgICAgICAgaWYgKGhhbmRsZXIucm91dGluZ0RhdGEpIHtcbiAgICAgICAgICAgICAgdGhpcy5yb3V0aW5nRGF0YSA9IGhhbmRsZXIucm91dGluZ0RhdGE7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuUkVST1VUSU5HKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuTE9HR0VEX0lOX1NFTkRJTkdfSU5JVElBTF9TUUwpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH0gZWxzZSBpZiAodGhpcy5udGxtcGFja2V0KSB7XG4gICAgICAgICAgICBjb25zdCBhdXRoZW50aWNhdGlvbiA9IHRoaXMuY29uZmlnLmF1dGhlbnRpY2F0aW9uIGFzIE50bG1BdXRoZW50aWNhdGlvbjtcblxuICAgICAgICAgICAgY29uc3QgcGF5bG9hZCA9IG5ldyBOVExNUmVzcG9uc2VQYXlsb2FkKHtcbiAgICAgICAgICAgICAgZG9tYWluOiBhdXRoZW50aWNhdGlvbi5vcHRpb25zLmRvbWFpbixcbiAgICAgICAgICAgICAgdXNlck5hbWU6IGF1dGhlbnRpY2F0aW9uLm9wdGlvbnMudXNlck5hbWUsXG4gICAgICAgICAgICAgIHBhc3N3b3JkOiBhdXRoZW50aWNhdGlvbi5vcHRpb25zLnBhc3N3b3JkLFxuICAgICAgICAgICAgICBudGxtcGFja2V0OiB0aGlzLm50bG1wYWNrZXRcbiAgICAgICAgICAgIH0pO1xuXG4gICAgICAgICAgICB0aGlzLm1lc3NhZ2VJby5zZW5kTWVzc2FnZShUWVBFLk5UTE1BVVRIX1BLVCwgcGF5bG9hZC5kYXRhKTtcbiAgICAgICAgICAgIHRoaXMuZGVidWcucGF5bG9hZChmdW5jdGlvbigpIHtcbiAgICAgICAgICAgICAgcmV0dXJuIHBheWxvYWQudG9TdHJpbmcoJyAgJyk7XG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgdGhpcy5udGxtcGFja2V0ID0gdW5kZWZpbmVkO1xuXG4gICAgICAgICAgICB0aGlzLm1lc3NhZ2VJby5yZWFkTWVzc2FnZSgpLnRoZW4oKG1lc3NhZ2UpID0+IHtcbiAgICAgICAgICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdtZXNzYWdlJywgbWVzc2FnZSk7XG4gICAgICAgICAgICB9LCAoZXJyKSA9PiB7XG4gICAgICAgICAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyKTtcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgIH0gZWxzZSBpZiAodGhpcy5sb2dpbkVycm9yKSB7XG4gICAgICAgICAgICBpZiAoaXNUcmFuc2llbnRFcnJvcih0aGlzLmxvZ2luRXJyb3IpKSB7XG4gICAgICAgICAgICAgIHRoaXMuZGVidWcubG9nKCdJbml0aWF0aW5nIHJldHJ5IG9uIHRyYW5zaWVudCBlcnJvcicpO1xuICAgICAgICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLlRSQU5TSUVOVF9GQUlMVVJFX1JFVFJZKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHRoaXMuZW1pdCgnY29ubmVjdCcsIHRoaXMubG9naW5FcnJvcik7XG4gICAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICB0aGlzLmVtaXQoJ2Nvbm5lY3QnLCBuZXcgQ29ubmVjdGlvbkVycm9yKCdMb2dpbiBmYWlsZWQuJywgJ0VMT0dJTicpKTtcbiAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgICAgIH1cbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuICB9LFxuICBTRU5UX0xPR0lON19XSVRIX0ZFREFVVEg6IHtcbiAgICBuYW1lOiAnU2VudExvZ2luN1dpdGhmZWRhdXRoJyxcbiAgICBlbnRlcjogZnVuY3Rpb24oKSB7XG4gICAgICB0aGlzLm1lc3NhZ2VJby5yZWFkTWVzc2FnZSgpLnRoZW4oKG1lc3NhZ2UpID0+IHtcbiAgICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdtZXNzYWdlJywgbWVzc2FnZSk7XG4gICAgICB9LCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyKTtcbiAgICAgIH0pO1xuICAgIH0sXG4gICAgZXZlbnRzOiB7XG4gICAgICBzb2NrZXRFcnJvcjogZnVuY3Rpb24oKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgfSxcbiAgICAgIGNvbm5lY3RUaW1lb3V0OiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9LFxuICAgICAgbWVzc2FnZTogZnVuY3Rpb24obWVzc2FnZSkge1xuICAgICAgICBjb25zdCBoYW5kbGVyID0gbmV3IExvZ2luN1Rva2VuSGFuZGxlcih0aGlzKTtcbiAgICAgICAgY29uc3QgdG9rZW5TdHJlYW1QYXJzZXIgPSB0aGlzLmNyZWF0ZVRva2VuU3RyZWFtUGFyc2VyKG1lc3NhZ2UsIGhhbmRsZXIpO1xuXG4gICAgICAgIHRva2VuU3RyZWFtUGFyc2VyLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICBpZiAoaGFuZGxlci5sb2dpbkFja1JlY2VpdmVkKSB7XG4gICAgICAgICAgICBpZiAoaGFuZGxlci5yb3V0aW5nRGF0YSkge1xuICAgICAgICAgICAgICB0aGlzLnJvdXRpbmdEYXRhID0gaGFuZGxlci5yb3V0aW5nRGF0YTtcbiAgICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5SRVJPVVRJTkcpO1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5MT0dHRURfSU5fU0VORElOR19JTklUSUFMX1NRTCk7XG4gICAgICAgICAgICB9XG5cbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBjb25zdCBmZWRBdXRoSW5mb1Rva2VuID0gaGFuZGxlci5mZWRBdXRoSW5mb1Rva2VuO1xuXG4gICAgICAgICAgaWYgKGZlZEF1dGhJbmZvVG9rZW4gJiYgZmVkQXV0aEluZm9Ub2tlbi5zdHN1cmwgJiYgZmVkQXV0aEluZm9Ub2tlbi5zcG4pIHtcbiAgICAgICAgICAgIGNvbnN0IGF1dGhlbnRpY2F0aW9uID0gdGhpcy5jb25maWcuYXV0aGVudGljYXRpb24gYXMgQXp1cmVBY3RpdmVEaXJlY3RvcnlQYXNzd29yZEF1dGhlbnRpY2F0aW9uIHwgQXp1cmVBY3RpdmVEaXJlY3RvcnlNc2lWbUF1dGhlbnRpY2F0aW9uIHwgQXp1cmVBY3RpdmVEaXJlY3RvcnlNc2lBcHBTZXJ2aWNlQXV0aGVudGljYXRpb24gfCBBenVyZUFjdGl2ZURpcmVjdG9yeVNlcnZpY2VQcmluY2lwYWxTZWNyZXQgfCBBenVyZUFjdGl2ZURpcmVjdG9yeURlZmF1bHRBdXRoZW50aWNhdGlvbjtcbiAgICAgICAgICAgIGNvbnN0IHRva2VuU2NvcGUgPSBuZXcgVVJMKCcvLmRlZmF1bHQnLCBmZWRBdXRoSW5mb1Rva2VuLnNwbikudG9TdHJpbmcoKTtcblxuICAgICAgICAgICAgY29uc3QgZ2V0VG9rZW4gPSAoY2FsbGJhY2s6IChlcnJvcjogRXJyb3IgfCBudWxsLCB0b2tlbj86IHN0cmluZykgPT4gdm9pZCkgPT4ge1xuICAgICAgICAgICAgICBjb25zdCBnZXRUb2tlbkZyb21DcmVkZW50aWFscyA9IChjcmVkZW50aWFsczogVG9rZW5DcmVkZW50aWFsKSA9PiB7XG4gICAgICAgICAgICAgICAgY3JlZGVudGlhbHMuZ2V0VG9rZW4odG9rZW5TY29wZSkudGhlbigodG9rZW5SZXNwb25zZSkgPT4ge1xuICAgICAgICAgICAgICAgICAgY2FsbGJhY2sobnVsbCwgdG9rZW5SZXNwb25zZT8udG9rZW4pO1xuICAgICAgICAgICAgICAgIH0sIGNhbGxiYWNrKTtcbiAgICAgICAgICAgICAgfTtcblxuICAgICAgICAgICAgICBpZiAoYXV0aGVudGljYXRpb24udHlwZSA9PT0gJ2F6dXJlLWFjdGl2ZS1kaXJlY3RvcnktcGFzc3dvcmQnKSB7XG4gICAgICAgICAgICAgICAgY29uc3QgY3JlZGVudGlhbHMgPSBuZXcgVXNlcm5hbWVQYXNzd29yZENyZWRlbnRpYWwoXG4gICAgICAgICAgICAgICAgICBhdXRoZW50aWNhdGlvbi5vcHRpb25zLnRlbmFudElkID8/ICdjb21tb24nLFxuICAgICAgICAgICAgICAgICAgYXV0aGVudGljYXRpb24ub3B0aW9ucy5jbGllbnRJZCxcbiAgICAgICAgICAgICAgICAgIGF1dGhlbnRpY2F0aW9uLm9wdGlvbnMudXNlck5hbWUsXG4gICAgICAgICAgICAgICAgICBhdXRoZW50aWNhdGlvbi5vcHRpb25zLnBhc3N3b3JkXG4gICAgICAgICAgICAgICAgKTtcblxuICAgICAgICAgICAgICAgIGdldFRva2VuRnJvbUNyZWRlbnRpYWxzKGNyZWRlbnRpYWxzKTtcbiAgICAgICAgICAgICAgfSBlbHNlIGlmIChhdXRoZW50aWNhdGlvbi50eXBlID09PSAnYXp1cmUtYWN0aXZlLWRpcmVjdG9yeS1tc2ktdm0nIHx8IGF1dGhlbnRpY2F0aW9uLnR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LW1zaS1hcHAtc2VydmljZScpIHtcbiAgICAgICAgICAgICAgICBjb25zdCBtc2lBcmdzID0gYXV0aGVudGljYXRpb24ub3B0aW9ucy5jbGllbnRJZCA/IFsgYXV0aGVudGljYXRpb24ub3B0aW9ucy5jbGllbnRJZCwge30gXSA6IFsge30gXTtcbiAgICAgICAgICAgICAgICBjb25zdCBjcmVkZW50aWFscyA9IG5ldyBNYW5hZ2VkSWRlbnRpdHlDcmVkZW50aWFsKC4uLm1zaUFyZ3MpO1xuXG4gICAgICAgICAgICAgICAgZ2V0VG9rZW5Gcm9tQ3JlZGVudGlhbHMoY3JlZGVudGlhbHMpO1xuICAgICAgICAgICAgICB9IGVsc2UgaWYgKGF1dGhlbnRpY2F0aW9uLnR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LWRlZmF1bHQnKSB7XG4gICAgICAgICAgICAgICAgY29uc3QgYXJncyA9IGF1dGhlbnRpY2F0aW9uLm9wdGlvbnMuY2xpZW50SWQgPyB7IG1hbmFnZWRJZGVudGl0eUNsaWVudElkOiBhdXRoZW50aWNhdGlvbi5vcHRpb25zLmNsaWVudElkIH0gOiB7fTtcbiAgICAgICAgICAgICAgICBjb25zdCBjcmVkZW50aWFscyA9IG5ldyBEZWZhdWx0QXp1cmVDcmVkZW50aWFsKGFyZ3MpO1xuXG4gICAgICAgICAgICAgICAgZ2V0VG9rZW5Gcm9tQ3JlZGVudGlhbHMoY3JlZGVudGlhbHMpO1xuICAgICAgICAgICAgICB9IGVsc2UgaWYgKGF1dGhlbnRpY2F0aW9uLnR5cGUgPT09ICdhenVyZS1hY3RpdmUtZGlyZWN0b3J5LXNlcnZpY2UtcHJpbmNpcGFsLXNlY3JldCcpIHtcbiAgICAgICAgICAgICAgICBjb25zdCBjcmVkZW50aWFscyA9IG5ldyBDbGllbnRTZWNyZXRDcmVkZW50aWFsKFxuICAgICAgICAgICAgICAgICAgYXV0aGVudGljYXRpb24ub3B0aW9ucy50ZW5hbnRJZCxcbiAgICAgICAgICAgICAgICAgIGF1dGhlbnRpY2F0aW9uLm9wdGlvbnMuY2xpZW50SWQsXG4gICAgICAgICAgICAgICAgICBhdXRoZW50aWNhdGlvbi5vcHRpb25zLmNsaWVudFNlY3JldFxuICAgICAgICAgICAgICAgICk7XG5cbiAgICAgICAgICAgICAgICBnZXRUb2tlbkZyb21DcmVkZW50aWFscyhjcmVkZW50aWFscyk7XG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIH07XG5cbiAgICAgICAgICAgIGdldFRva2VuKChlcnIsIHRva2VuKSA9PiB7XG4gICAgICAgICAgICAgIGlmIChlcnIpIHtcbiAgICAgICAgICAgICAgICB0aGlzLmxvZ2luRXJyb3IgPSBuZXcgQWdncmVnYXRlRXJyb3IoXG4gICAgICAgICAgICAgICAgICBbbmV3IENvbm5lY3Rpb25FcnJvcignU2VjdXJpdHkgdG9rZW4gY291bGQgbm90IGJlIGF1dGhlbnRpY2F0ZWQgb3IgYXV0aG9yaXplZC4nLCAnRUZFREFVVEgnKSwgZXJyXSk7XG4gICAgICAgICAgICAgICAgdGhpcy5lbWl0KCdjb25uZWN0JywgdGhpcy5sb2dpbkVycm9yKTtcbiAgICAgICAgICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgICB0aGlzLnNlbmRGZWRBdXRoVG9rZW5NZXNzYWdlKHRva2VuISk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9IGVsc2UgaWYgKHRoaXMubG9naW5FcnJvcikge1xuICAgICAgICAgICAgaWYgKGlzVHJhbnNpZW50RXJyb3IodGhpcy5sb2dpbkVycm9yKSkge1xuICAgICAgICAgICAgICB0aGlzLmRlYnVnLmxvZygnSW5pdGlhdGluZyByZXRyeSBvbiB0cmFuc2llbnQgZXJyb3InKTtcbiAgICAgICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5UUkFOU0lFTlRfRkFJTFVSRV9SRVRSWSk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICB0aGlzLmVtaXQoJ2Nvbm5lY3QnLCB0aGlzLmxvZ2luRXJyb3IpO1xuICAgICAgICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgdGhpcy5lbWl0KCdjb25uZWN0JywgbmV3IENvbm5lY3Rpb25FcnJvcignTG9naW4gZmFpbGVkLicsICdFTE9HSU4nKSk7XG4gICAgICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkZJTkFMKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgfSxcbiAgTE9HR0VEX0lOX1NFTkRJTkdfSU5JVElBTF9TUUw6IHtcbiAgICBuYW1lOiAnTG9nZ2VkSW5TZW5kaW5nSW5pdGlhbFNxbCcsXG4gICAgZW50ZXI6IGZ1bmN0aW9uKCkge1xuICAgICAgdGhpcy5zZW5kSW5pdGlhbFNxbCgpO1xuXG4gICAgICB0aGlzLm1lc3NhZ2VJby5yZWFkTWVzc2FnZSgpLnRoZW4oKG1lc3NhZ2UpID0+IHtcbiAgICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdtZXNzYWdlJywgbWVzc2FnZSk7XG4gICAgICB9LCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyKTtcbiAgICAgIH0pO1xuICAgIH0sXG4gICAgZXZlbnRzOiB7XG4gICAgICBzb2NrZXRFcnJvcjogZnVuY3Rpb24gc29ja2V0RXJyb3IoKSB7XG4gICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuRklOQUwpO1xuICAgICAgfSxcbiAgICAgIGNvbm5lY3RUaW1lb3V0OiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9LFxuICAgICAgbWVzc2FnZTogZnVuY3Rpb24obWVzc2FnZSkge1xuICAgICAgICBjb25zdCB0b2tlblN0cmVhbVBhcnNlciA9IHRoaXMuY3JlYXRlVG9rZW5TdHJlYW1QYXJzZXIobWVzc2FnZSwgbmV3IEluaXRpYWxTcWxUb2tlbkhhbmRsZXIodGhpcykpO1xuXG4gICAgICAgIHRva2VuU3RyZWFtUGFyc2VyLm9uY2UoJ2VuZCcsICgpID0+IHtcbiAgICAgICAgICB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLkxPR0dFRF9JTik7XG4gICAgICAgICAgdGhpcy5wcm9jZXNzZWRJbml0aWFsU3FsKCk7XG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgfSxcbiAgTE9HR0VEX0lOOiB7XG4gICAgbmFtZTogJ0xvZ2dlZEluJyxcbiAgICBldmVudHM6IHtcbiAgICAgIHNvY2tldEVycm9yOiBmdW5jdGlvbigpIHtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG4gICAgICB9XG4gICAgfVxuICB9LFxuICBTRU5UX0NMSUVOVF9SRVFVRVNUOiB7XG4gICAgbmFtZTogJ1NlbnRDbGllbnRSZXF1ZXN0JyxcbiAgICBlbnRlcjogZnVuY3Rpb24oKSB7XG4gICAgICB0aGlzLmVtcHR5TWVzc2FnZUJ1ZmZlcigpO1xuXG4gICAgICB0aGlzLm1lc3NhZ2VJby5yZWFkTWVzc2FnZSgpLnRoZW4oKG1lc3NhZ2UpID0+IHtcbiAgICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdtZXNzYWdlJywgbWVzc2FnZSk7XG4gICAgICB9LCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyKTtcbiAgICAgIH0pO1xuICAgIH0sXG4gICAgZXhpdDogZnVuY3Rpb24obmV4dFN0YXRlKSB7XG4gICAgICB0aGlzLmNsZWFyUmVxdWVzdFRpbWVyKCk7XG4gICAgfSxcbiAgICBldmVudHM6IHtcbiAgICAgIHNvY2tldEVycm9yOiBmdW5jdGlvbihlcnIpIHtcbiAgICAgICAgY29uc3Qgc3FsUmVxdWVzdCA9IHRoaXMucmVxdWVzdCE7XG4gICAgICAgIHRoaXMucmVxdWVzdCA9IHVuZGVmaW5lZDtcbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG5cbiAgICAgICAgc3FsUmVxdWVzdC5jYWxsYmFjayhlcnIpO1xuICAgICAgfSxcbiAgICAgIG1lc3NhZ2U6IGZ1bmN0aW9uKG1lc3NhZ2UpIHtcbiAgICAgICAgLy8gcmVxdWVzdCB0aW1lciBpcyBzdG9wcGVkIG9uIGZpcnN0IGRhdGEgcGFja2FnZVxuICAgICAgICB0aGlzLmNsZWFyUmVxdWVzdFRpbWVyKCk7XG5cbiAgICAgICAgY29uc3QgdG9rZW5TdHJlYW1QYXJzZXIgPSB0aGlzLmNyZWF0ZVRva2VuU3RyZWFtUGFyc2VyKG1lc3NhZ2UsIG5ldyBSZXF1ZXN0VG9rZW5IYW5kbGVyKHRoaXMsIHRoaXMucmVxdWVzdCEpKTtcblxuICAgICAgICAvLyBJZiB0aGUgcmVxdWVzdCB3YXMgY2FuY2VsZWQgYW5kIHdlIGhhdmUgYSBgY2FuY2VsVGltZXJgXG4gICAgICAgIC8vIGRlZmluZWQsIHdlIHNlbmQgYSBhdHRlbnRpb24gbWVzc2FnZSBhZnRlciB0aGVcbiAgICAgICAgLy8gcmVxdWVzdCBtZXNzYWdlIHdhcyBmdWxseSBzZW50IG9mZi5cbiAgICAgICAgLy9cbiAgICAgICAgLy8gV2UgYWxyZWFkeSBzdGFydGVkIGNvbnN1bWluZyB0aGUgY3VycmVudCBtZXNzYWdlXG4gICAgICAgIC8vIChidXQgYWxsIHRoZSB0b2tlbiBoYW5kbGVycyBzaG91bGQgYmUgbm8tb3BzKSwgYW5kXG4gICAgICAgIC8vIG5lZWQgdG8gZW5zdXJlIHRoZSBuZXh0IG1lc3NhZ2UgaXMgaGFuZGxlZCBieSB0aGVcbiAgICAgICAgLy8gYFNFTlRfQVRURU5USU9OYCBzdGF0ZS5cbiAgICAgICAgaWYgKHRoaXMucmVxdWVzdD8uY2FuY2VsZWQgJiYgdGhpcy5jYW5jZWxUaW1lcikge1xuICAgICAgICAgIHJldHVybiB0aGlzLnRyYW5zaXRpb25Ubyh0aGlzLlNUQVRFLlNFTlRfQVRURU5USU9OKTtcbiAgICAgICAgfVxuXG4gICAgICAgIGNvbnN0IG9uUmVzdW1lID0gKCkgPT4ge1xuICAgICAgICAgIHRva2VuU3RyZWFtUGFyc2VyLnJlc3VtZSgpO1xuICAgICAgICB9O1xuICAgICAgICBjb25zdCBvblBhdXNlID0gKCkgPT4ge1xuICAgICAgICAgIHRva2VuU3RyZWFtUGFyc2VyLnBhdXNlKCk7XG5cbiAgICAgICAgICB0aGlzLnJlcXVlc3Q/Lm9uY2UoJ3Jlc3VtZScsIG9uUmVzdW1lKTtcbiAgICAgICAgfTtcblxuICAgICAgICB0aGlzLnJlcXVlc3Q/Lm9uKCdwYXVzZScsIG9uUGF1c2UpO1xuXG4gICAgICAgIGlmICh0aGlzLnJlcXVlc3QgaW5zdGFuY2VvZiBSZXF1ZXN0ICYmIHRoaXMucmVxdWVzdC5wYXVzZWQpIHtcbiAgICAgICAgICBvblBhdXNlKCk7XG4gICAgICAgIH1cblxuICAgICAgICBjb25zdCBvbkNhbmNlbCA9ICgpID0+IHtcbiAgICAgICAgICB0b2tlblN0cmVhbVBhcnNlci5yZW1vdmVMaXN0ZW5lcignZW5kJywgb25FbmRPZk1lc3NhZ2UpO1xuXG4gICAgICAgICAgaWYgKHRoaXMucmVxdWVzdCBpbnN0YW5jZW9mIFJlcXVlc3QgJiYgdGhpcy5yZXF1ZXN0LnBhdXNlZCkge1xuICAgICAgICAgICAgLy8gcmVzdW1lIHRoZSByZXF1ZXN0IGlmIGl0IHdhcyBwYXVzZWQgc28gd2UgY2FuIHJlYWQgdGhlIHJlbWFpbmluZyB0b2tlbnNcbiAgICAgICAgICAgIHRoaXMucmVxdWVzdC5yZXN1bWUoKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICB0aGlzLnJlcXVlc3Q/LnJlbW92ZUxpc3RlbmVyKCdwYXVzZScsIG9uUGF1c2UpO1xuICAgICAgICAgIHRoaXMucmVxdWVzdD8ucmVtb3ZlTGlzdGVuZXIoJ3Jlc3VtZScsIG9uUmVzdW1lKTtcblxuICAgICAgICAgIC8vIFRoZSBgX2NhbmNlbEFmdGVyUmVxdWVzdFNlbnRgIGNhbGxiYWNrIHdpbGwgaGF2ZSBzZW50IGFcbiAgICAgICAgICAvLyBhdHRlbnRpb24gbWVzc2FnZSwgc28gbm93IHdlIG5lZWQgdG8gYWxzbyBzd2l0Y2ggdG9cbiAgICAgICAgICAvLyB0aGUgYFNFTlRfQVRURU5USU9OYCBzdGF0ZSB0byBtYWtlIHN1cmUgdGhlIGF0dGVudGlvbiBhY2tcbiAgICAgICAgICAvLyBtZXNzYWdlIGlzIHByb2Nlc3NlZCBjb3JyZWN0bHkuXG4gICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5TRU5UX0FUVEVOVElPTik7XG4gICAgICAgIH07XG5cbiAgICAgICAgY29uc3Qgb25FbmRPZk1lc3NhZ2UgPSAoKSA9PiB7XG4gICAgICAgICAgdGhpcy5yZXF1ZXN0Py5yZW1vdmVMaXN0ZW5lcignY2FuY2VsJywgdGhpcy5fY2FuY2VsQWZ0ZXJSZXF1ZXN0U2VudCk7XG4gICAgICAgICAgdGhpcy5yZXF1ZXN0Py5yZW1vdmVMaXN0ZW5lcignY2FuY2VsJywgb25DYW5jZWwpO1xuICAgICAgICAgIHRoaXMucmVxdWVzdD8ucmVtb3ZlTGlzdGVuZXIoJ3BhdXNlJywgb25QYXVzZSk7XG4gICAgICAgICAgdGhpcy5yZXF1ZXN0Py5yZW1vdmVMaXN0ZW5lcigncmVzdW1lJywgb25SZXN1bWUpO1xuXG4gICAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5MT0dHRURfSU4pO1xuICAgICAgICAgIGNvbnN0IHNxbFJlcXVlc3QgPSB0aGlzLnJlcXVlc3QgYXMgUmVxdWVzdDtcbiAgICAgICAgICB0aGlzLnJlcXVlc3QgPSB1bmRlZmluZWQ7XG4gICAgICAgICAgaWYgKHRoaXMuY29uZmlnLm9wdGlvbnMudGRzVmVyc2lvbiA8ICc3XzInICYmIHNxbFJlcXVlc3QuZXJyb3IgJiYgdGhpcy5pc1NxbEJhdGNoKSB7XG4gICAgICAgICAgICB0aGlzLmluVHJhbnNhY3Rpb24gPSBmYWxzZTtcbiAgICAgICAgICB9XG4gICAgICAgICAgc3FsUmVxdWVzdC5jYWxsYmFjayhzcWxSZXF1ZXN0LmVycm9yLCBzcWxSZXF1ZXN0LnJvd0NvdW50LCBzcWxSZXF1ZXN0LnJvd3MpO1xuICAgICAgICB9O1xuXG4gICAgICAgIHRva2VuU3RyZWFtUGFyc2VyLm9uY2UoJ2VuZCcsIG9uRW5kT2ZNZXNzYWdlKTtcbiAgICAgICAgdGhpcy5yZXF1ZXN0Py5vbmNlKCdjYW5jZWwnLCBvbkNhbmNlbCk7XG4gICAgICB9XG4gICAgfVxuICB9LFxuICBTRU5UX0FUVEVOVElPTjoge1xuICAgIG5hbWU6ICdTZW50QXR0ZW50aW9uJyxcbiAgICBlbnRlcjogZnVuY3Rpb24oKSB7XG4gICAgICB0aGlzLmVtcHR5TWVzc2FnZUJ1ZmZlcigpO1xuXG4gICAgICB0aGlzLm1lc3NhZ2VJby5yZWFkTWVzc2FnZSgpLnRoZW4oKG1lc3NhZ2UpID0+IHtcbiAgICAgICAgdGhpcy5kaXNwYXRjaEV2ZW50KCdtZXNzYWdlJywgbWVzc2FnZSk7XG4gICAgICB9LCAoZXJyKSA9PiB7XG4gICAgICAgIHRoaXMuc29ja2V0RXJyb3IoZXJyKTtcbiAgICAgIH0pO1xuICAgIH0sXG4gICAgZXZlbnRzOiB7XG4gICAgICBzb2NrZXRFcnJvcjogZnVuY3Rpb24oZXJyKSB7XG4gICAgICAgIGNvbnN0IHNxbFJlcXVlc3QgPSB0aGlzLnJlcXVlc3QhO1xuICAgICAgICB0aGlzLnJlcXVlc3QgPSB1bmRlZmluZWQ7XG5cbiAgICAgICAgdGhpcy50cmFuc2l0aW9uVG8odGhpcy5TVEFURS5GSU5BTCk7XG5cbiAgICAgICAgc3FsUmVxdWVzdC5jYWxsYmFjayhlcnIpO1xuICAgICAgfSxcbiAgICAgIG1lc3NhZ2U6IGZ1bmN0aW9uKG1lc3NhZ2UpIHtcbiAgICAgICAgY29uc3QgaGFuZGxlciA9IG5ldyBBdHRlbnRpb25Ub2tlbkhhbmRsZXIodGhpcywgdGhpcy5yZXF1ZXN0ISk7XG4gICAgICAgIGNvbnN0IHRva2VuU3RyZWFtUGFyc2VyID0gdGhpcy5jcmVhdGVUb2tlblN0cmVhbVBhcnNlcihtZXNzYWdlLCBoYW5kbGVyKTtcblxuICAgICAgICB0b2tlblN0cmVhbVBhcnNlci5vbmNlKCdlbmQnLCAoKSA9PiB7XG4gICAgICAgICAgLy8gMy4yLjUuNyBTZW50IEF0dGVudGlvbiBTdGF0ZVxuICAgICAgICAgIC8vIERpc2NhcmQgYW55IGRhdGEgY29udGFpbmVkIGluIHRoZSByZXNwb25zZSwgdW50aWwgd2UgcmVjZWl2ZSB0aGUgYXR0ZW50aW9uIHJlc3BvbnNlXG4gICAgICAgICAgaWYgKGhhbmRsZXIuYXR0ZW50aW9uUmVjZWl2ZWQpIHtcbiAgICAgICAgICAgIHRoaXMuY2xlYXJDYW5jZWxUaW1lcigpO1xuXG4gICAgICAgICAgICBjb25zdCBzcWxSZXF1ZXN0ID0gdGhpcy5yZXF1ZXN0ITtcbiAgICAgICAgICAgIHRoaXMucmVxdWVzdCA9IHVuZGVmaW5lZDtcbiAgICAgICAgICAgIHRoaXMudHJhbnNpdGlvblRvKHRoaXMuU1RBVEUuTE9HR0VEX0lOKTtcblxuICAgICAgICAgICAgaWYgKHNxbFJlcXVlc3QuZXJyb3IgJiYgc3FsUmVxdWVzdC5lcnJvciBpbnN0YW5jZW9mIFJlcXVlc3RFcnJvciAmJiBzcWxSZXF1ZXN0LmVycm9yLmNvZGUgPT09ICdFVElNRU9VVCcpIHtcbiAgICAgICAgICAgICAgc3FsUmVxdWVzdC5jYWxsYmFjayhzcWxSZXF1ZXN0LmVycm9yKTtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIHNxbFJlcXVlc3QuY2FsbGJhY2sobmV3IFJlcXVlc3RFcnJvcignQ2FuY2VsZWQuJywgJ0VDQU5DRUwnKSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH1cbiAgICB9XG4gIH0sXG4gIEZJTkFMOiB7XG4gICAgbmFtZTogJ0ZpbmFsJyxcbiAgICBlbnRlcjogZnVuY3Rpb24oKSB7XG4gICAgICB0aGlzLmNsZWFudXBDb25uZWN0aW9uKENMRUFOVVBfVFlQRS5OT1JNQUwpO1xuICAgIH0sXG4gICAgZXZlbnRzOiB7XG4gICAgICBjb25uZWN0VGltZW91dDogZnVuY3Rpb24oKSB7XG4gICAgICAgIC8vIERvIG5vdGhpbmcsIGFzIHRoZSB0aW1lciBzaG91bGQgYmUgY2xlYW5lZCB1cC5cbiAgICAgIH0sXG4gICAgICBtZXNzYWdlOiBmdW5jdGlvbigpIHtcbiAgICAgICAgLy8gRG8gbm90aGluZ1xuICAgICAgfSxcbiAgICAgIHNvY2tldEVycm9yOiBmdW5jdGlvbigpIHtcbiAgICAgICAgLy8gRG8gbm90aGluZ1xuICAgICAgfVxuICAgIH1cbiAgfVxufTtcbiJdfQ==