"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _core = require("@js-joda/core");

var _writableTrackingBuffer = _interopRequireDefault(require("../tracking-buffer/writable-tracking-buffer"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const EPOCH_DATE = _core.LocalDate.ofYearDay(1, 1);

const NULL_LENGTH = Buffer.from([0x00]);
const DateTime2 = {
  id: 0x2A,
  type: 'DATETIME2N',
  name: 'DateTime2',
  declaration: function (parameter) {
    return 'datetime2(' + this.resolveScale(parameter) + ')';
  },
  resolveScale: function (parameter) {
    if (parameter.scale != null) {
      return parameter.scale;
    } else if (parameter.value === null) {
      return 0;
    } else {
      return 7;
    }
  },

  generateTypeInfo(parameter, _options) {
    return Buffer.from([this.id, parameter.scale]);
  },

  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      return NULL_LENGTH;
    }

    switch (parameter.scale) {
      case 0:
      case 1:
      case 2:
        return Buffer.from([0x06]);

      case 3:
      case 4:
        return Buffer.from([0x07]);

      case 5:
      case 6:
      case 7:
        return Buffer.from([0x08]);

      default:
        throw new Error('invalid scale');
    }
  },

  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }

    const value = parameter.value;
    let scale = parameter.scale;
    const buffer = new _writableTrackingBuffer.default(16);
    scale = scale;
    let timestamp;

    if (options.useUTC) {
      timestamp = ((value.getUTCHours() * 60 + value.getUTCMinutes()) * 60 + value.getUTCSeconds()) * 1000 + value.getUTCMilliseconds();
    } else {
      timestamp = ((value.getHours() * 60 + value.getMinutes()) * 60 + value.getSeconds()) * 1000 + value.getMilliseconds();
    }

    timestamp = timestamp * Math.pow(10, scale - 3);
    timestamp += (value.nanosecondDelta != null ? value.nanosecondDelta : 0) * Math.pow(10, scale);
    timestamp = Math.round(timestamp);

    switch (scale) {
      case 0:
      case 1:
      case 2:
        buffer.writeUInt24LE(timestamp);
        break;

      case 3:
      case 4:
        buffer.writeUInt32LE(timestamp);
        break;

      case 5:
      case 6:
      case 7:
        buffer.writeUInt40LE(timestamp);
    }

    let date;

    if (options.useUTC) {
      date = _core.LocalDate.of(value.getUTCFullYear(), value.getUTCMonth() + 1, value.getUTCDate());
    } else {
      date = _core.LocalDate.of(value.getFullYear(), value.getMonth() + 1, value.getDate());
    }

    const days = EPOCH_DATE.until(date, _core.ChronoUnit.DAYS);
    buffer.writeUInt24LE(days);
    yield buffer.data;
  },

  validate: function (value) {
    if (value == null) {
      return null;
    }

    if (!(value instanceof Date)) {
      value = new Date(Date.parse(value));
    }

    if (isNaN(value)) {
      throw new TypeError('Invalid date.');
    }

    return value;
  }
};
var _default = DateTime2;
exports.default = _default;
module.exports = DateTime2;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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