"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InstanceLookup = void 0;

var _dns = _interopRequireDefault(require("dns"));

var _nodeAbortController = require("node-abort-controller");

var _sender = require("./sender");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const SQL_SERVER_BROWSER_PORT = 1434;
const TIMEOUT = 2 * 1000;
const RETRIES = 3; // There are three bytes at the start of the response, whose purpose is unknown.

const MYSTERY_HEADER_LENGTH = 3;

class AbortError extends Error {
  constructor() {
    super('The operation was aborted');
    this.code = void 0;
    this.code = 'ABORT_ERR';
    this.name = 'AbortError';
  }

} // Most of the functionality has been determined from from jTDS's MSSqlServerInfo class.


class InstanceLookup {
  instanceLookup(options, callback) {
    var _options$lookup, _options$port;

    const server = options.server;

    if (typeof server !== 'string') {
      throw new TypeError('Invalid arguments: "server" must be a string');
    }

    const instanceName = options.instanceName;

    if (typeof instanceName !== 'string') {
      throw new TypeError('Invalid arguments: "instanceName" must be a string');
    }

    const timeout = options.timeout === undefined ? TIMEOUT : options.timeout;

    if (typeof timeout !== 'number') {
      throw new TypeError('Invalid arguments: "timeout" must be a number');
    }

    const retries = options.retries === undefined ? RETRIES : options.retries;

    if (typeof retries !== 'number') {
      throw new TypeError('Invalid arguments: "retries" must be a number');
    }

    if (options.lookup !== undefined && typeof options.lookup !== 'function') {
      throw new TypeError('Invalid arguments: "lookup" must be a function');
    }

    const lookup = (_options$lookup = options.lookup) !== null && _options$lookup !== void 0 ? _options$lookup : _dns.default.lookup;

    if (options.port !== undefined && typeof options.port !== 'number') {
      throw new TypeError('Invalid arguments: "port" must be a number');
    }

    const port = (_options$port = options.port) !== null && _options$port !== void 0 ? _options$port : SQL_SERVER_BROWSER_PORT;
    const signal = options.signal;

    if (typeof callback !== 'function') {
      throw new TypeError('Invalid arguments: "callback" must be a function');
    }

    if (signal.aborted) {
      return process.nextTick(callback, new AbortError());
    }

    let retriesLeft = retries;

    const makeAttempt = () => {
      if (retriesLeft >= 0) {
        retriesLeft--;
        const controller = new _nodeAbortController.AbortController();

        const abortCurrentAttempt = () => {
          controller.abort();
        }; // If the overall instance lookup is aborted,
        // forward the abort to the controller of the current
        // lookup attempt.


        signal.addEventListener('abort', abortCurrentAttempt, {
          once: true
        });
        const request = Buffer.from([0x02]);
        const sender = new _sender.Sender(options.server, port, lookup, controller.signal, request);
        const timer = setTimeout(abortCurrentAttempt, timeout);
        sender.execute((err, response) => {
          clearTimeout(timer);

          if (err) {
            if ((err === null || err === void 0 ? void 0 : err.name) === 'AbortError') {
              // If the overall instance lookup was aborted,
              // do not perform any further attempts.
              if (signal.aborted) {
                return callback(new AbortError());
              }

              return makeAttempt();
            }

            return callback(new Error('Failed to lookup instance on ' + server + ' - ' + err.message));
          }

          const message = response.toString('ascii', MYSTERY_HEADER_LENGTH);
          const port = this.parseBrowserResponse(message, instanceName);

          if (port) {
            callback(undefined, port);
          } else {
            callback(new Error('Port for ' + instanceName + ' not found in ' + options.server));
          }
        });
      } else {
        callback(new Error('Failed to get response from SQL Server Browser on ' + server));
      }
    };

    makeAttempt();
  }

  parseBrowserResponse(response, instanceName) {
    let getPort;
    const instances = response.split(';;');

    for (let i = 0, len = instances.length; i < len; i++) {
      const instance = instances[i];
      const parts = instance.split(';');

      for (let p = 0, partsLen = parts.length; p < partsLen; p += 2) {
        const name = parts[p];
        const value = parts[p + 1];

        if (name === 'tcp' && getPort) {
          const port = parseInt(value, 10);
          return port;
        }

        if (name === 'InstanceName') {
          if (value.toUpperCase() === instanceName.toUpperCase()) {
            getPort = true;
          } else {
            getPort = false;
          }
        }
      }
    }
  }

}

exports.InstanceLookup = InstanceLookup;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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