"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isPacketComplete = isPacketComplete;
exports.packetLength = packetLength;
exports.Packet = exports.OFFSET = exports.TYPE = exports.HEADER_LENGTH = void 0;

var _sprintfJs = require("sprintf-js");

const HEADER_LENGTH = 8;
exports.HEADER_LENGTH = HEADER_LENGTH;
const TYPE = {
  SQL_BATCH: 0x01,
  RPC_REQUEST: 0x03,
  TABULAR_RESULT: 0x04,
  ATTENTION: 0x06,
  BULK_LOAD: 0x07,
  TRANSACTION_MANAGER: 0x0E,
  LOGIN7: 0x10,
  NTLMAUTH_PKT: 0x11,
  PRELOGIN: 0x12,
  FEDAUTH_TOKEN: 0x08
};
exports.TYPE = TYPE;
const typeByValue = {};

for (const name in TYPE) {
  typeByValue[TYPE[name]] = name;
}

const STATUS = {
  NORMAL: 0x00,
  EOM: 0x01,
  IGNORE: 0x02,
  RESETCONNECTION: 0x08,
  RESETCONNECTIONSKIPTRAN: 0x10
};
const OFFSET = {
  Type: 0,
  Status: 1,
  Length: 2,
  SPID: 4,
  PacketID: 6,
  Window: 7
};
exports.OFFSET = OFFSET;
const DEFAULT_SPID = 0;
const DEFAULT_PACKETID = 1;
const DEFAULT_WINDOW = 0;
const NL = '\n';

class Packet {
  constructor(typeOrBuffer) {
    this.buffer = void 0;

    if (typeOrBuffer instanceof Buffer) {
      this.buffer = typeOrBuffer;
    } else {
      const type = typeOrBuffer;
      this.buffer = Buffer.alloc(HEADER_LENGTH, 0);
      this.buffer.writeUInt8(type, OFFSET.Type);
      this.buffer.writeUInt8(STATUS.NORMAL, OFFSET.Status);
      this.buffer.writeUInt16BE(DEFAULT_SPID, OFFSET.SPID);
      this.buffer.writeUInt8(DEFAULT_PACKETID, OFFSET.PacketID);
      this.buffer.writeUInt8(DEFAULT_WINDOW, OFFSET.Window);
      this.setLength();
    }
  }

  setLength() {
    this.buffer.writeUInt16BE(this.buffer.length, OFFSET.Length);
  }

  length() {
    return this.buffer.readUInt16BE(OFFSET.Length);
  }

  resetConnection(reset) {
    let status = this.buffer.readUInt8(OFFSET.Status);

    if (reset) {
      status |= STATUS.RESETCONNECTION;
    } else {
      status &= 0xFF - STATUS.RESETCONNECTION;
    }

    this.buffer.writeUInt8(status, OFFSET.Status);
  }

  last(last) {
    let status = this.buffer.readUInt8(OFFSET.Status);

    if (arguments.length > 0) {
      if (last) {
        status |= STATUS.EOM;
      } else {
        status &= 0xFF - STATUS.EOM;
      }

      this.buffer.writeUInt8(status, OFFSET.Status);
    }

    return this.isLast();
  }

  ignore(last) {
    let status = this.buffer.readUInt8(OFFSET.Status);

    if (last) {
      status |= STATUS.IGNORE;
    } else {
      status &= 0xFF - STATUS.IGNORE;
    }

    this.buffer.writeUInt8(status, OFFSET.Status);
  }

  isLast() {
    return !!(this.buffer.readUInt8(OFFSET.Status) & STATUS.EOM);
  }

  packetId(packetId) {
    if (packetId) {
      this.buffer.writeUInt8(packetId % 256, OFFSET.PacketID);
    }

    return this.buffer.readUInt8(OFFSET.PacketID);
  }

  addData(data) {
    this.buffer = Buffer.concat([this.buffer, data]);
    this.setLength();
    return this;
  }

  data() {
    return this.buffer.slice(HEADER_LENGTH);
  }

  type() {
    return this.buffer.readUInt8(OFFSET.Type);
  }

  statusAsString() {
    const status = this.buffer.readUInt8(OFFSET.Status);
    const statuses = [];

    for (const name in STATUS) {
      const value = STATUS[name];

      if (status & value) {
        statuses.push(name);
      } else {
        statuses.push(undefined);
      }
    }

    return statuses.join(' ').trim();
  }

  headerToString(indent = '') {
    const text = (0, _sprintfJs.sprintf)('type:0x%02X(%s), status:0x%02X(%s), length:0x%04X, spid:0x%04X, packetId:0x%02X, window:0x%02X', this.buffer.readUInt8(OFFSET.Type), typeByValue[this.buffer.readUInt8(OFFSET.Type)], this.buffer.readUInt8(OFFSET.Status), this.statusAsString(), this.buffer.readUInt16BE(OFFSET.Length), this.buffer.readUInt16BE(OFFSET.SPID), this.buffer.readUInt8(OFFSET.PacketID), this.buffer.readUInt8(OFFSET.Window));
    return indent + text;
  }

  dataToString(indent = '') {
    const BYTES_PER_GROUP = 0x04;
    const CHARS_PER_GROUP = 0x08;
    const BYTES_PER_LINE = 0x20;
    const data = this.data();
    let dataDump = '';
    let chars = '';

    for (let offset = 0; offset < data.length; offset++) {
      if (offset % BYTES_PER_LINE === 0) {
        dataDump += indent;
        dataDump += (0, _sprintfJs.sprintf)('%04X  ', offset);
      }

      if (data[offset] < 0x20 || data[offset] > 0x7E) {
        chars += '.';

        if ((offset + 1) % CHARS_PER_GROUP === 0 && !((offset + 1) % BYTES_PER_LINE === 0)) {
          chars += ' ';
        }
      } else {
        chars += String.fromCharCode(data[offset]);
      }

      if (data[offset] != null) {
        dataDump += (0, _sprintfJs.sprintf)('%02X', data[offset]);
      }

      if ((offset + 1) % BYTES_PER_GROUP === 0 && !((offset + 1) % BYTES_PER_LINE === 0)) {
        dataDump += ' ';
      }

      if ((offset + 1) % BYTES_PER_LINE === 0) {
        dataDump += '  ' + chars;
        chars = '';

        if (offset < data.length - 1) {
          dataDump += NL;
        }
      }
    }

    if (chars.length) {
      dataDump += '  ' + chars;
    }

    return dataDump;
  }

  toString(indent = '') {
    return this.headerToString(indent) + '\n' + this.dataToString(indent + indent);
  }

  payloadString() {
    return '';
  }

}

exports.Packet = Packet;

function isPacketComplete(potentialPacketBuffer) {
  if (potentialPacketBuffer.length < HEADER_LENGTH) {
    return false;
  } else {
    return potentialPacketBuffer.length >= potentialPacketBuffer.readUInt16BE(OFFSET.Length);
  }
}

function packetLength(potentialPacketBuffer) {
  return potentialPacketBuffer.readUInt16BE(OFFSET.Length);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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