"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Sender = exports.ParallelSendStrategy = void 0;

var _dgram = _interopRequireDefault(require("dgram"));

var _net = _interopRequireDefault(require("net"));

var punycode = _interopRequireWildcard(require("punycode"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class AbortError extends Error {
  constructor() {
    super('The operation was aborted');
    this.code = void 0;
    this.code = 'ABORT_ERR';
    this.name = 'AbortError';
  }

}

class ParallelSendStrategy {
  constructor(addresses, port, signal, request) {
    this.addresses = void 0;
    this.port = void 0;
    this.request = void 0;
    this.signal = void 0;
    this.addresses = addresses;
    this.port = port;
    this.request = request;
    this.signal = signal;
  }

  send(cb) {
    const signal = this.signal;

    if (signal.aborted) {
      return cb(new AbortError());
    }

    const sockets = [];
    let errorCount = 0;

    const onError = err => {
      errorCount++;

      if (errorCount === this.addresses.length) {
        signal.removeEventListener('abort', onAbort);
        clearSockets();
        cb(err);
      }
    };

    const onMessage = message => {
      signal.removeEventListener('abort', onAbort);
      clearSockets();
      cb(null, message);
    };

    const onAbort = () => {
      clearSockets();
      cb(new AbortError());
    };

    const clearSockets = () => {
      for (const socket of sockets) {
        socket.removeListener('error', onError);
        socket.removeListener('message', onMessage);
        socket.close();
      }
    };

    signal.addEventListener('abort', onAbort, {
      once: true
    });

    for (let j = 0; j < this.addresses.length; j++) {
      const udpType = this.addresses[j].family === 6 ? 'udp6' : 'udp4';

      const socket = _dgram.default.createSocket(udpType);

      sockets.push(socket);
      socket.on('error', onError);
      socket.on('message', onMessage);
      socket.send(this.request, 0, this.request.length, this.port, this.addresses[j].address);
    }
  }

}

exports.ParallelSendStrategy = ParallelSendStrategy;

class Sender {
  constructor(host, port, lookup, signal, request) {
    this.host = void 0;
    this.port = void 0;
    this.request = void 0;
    this.lookup = void 0;
    this.signal = void 0;
    this.host = host;
    this.port = port;
    this.request = request;
    this.lookup = lookup;
    this.signal = signal;
  }

  execute(cb) {
    if (_net.default.isIP(this.host)) {
      this.executeForIP(cb);
    } else {
      this.executeForHostname(cb);
    }
  }

  executeForIP(cb) {
    this.executeForAddresses([{
      address: this.host,
      family: _net.default.isIPv6(this.host) ? 6 : 4
    }], cb);
  } // Wrapper for stubbing. Sinon does not have support for stubbing module functions.


  invokeLookupAll(host, cb) {
    this.lookup.call(null, punycode.toASCII(host), {
      all: true
    }, cb);
  }

  executeForHostname(cb) {
    this.invokeLookupAll(this.host, (err, addresses) => {
      if (err) {
        return cb(err);
      }

      this.executeForAddresses(addresses, cb);
    });
  }

  executeForAddresses(addresses, cb) {
    const parallelSendStrategy = new ParallelSendStrategy(addresses, this.port, this.signal, this.request);
    parallelSendStrategy.send(cb);
  }

}

exports.Sender = Sender;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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