"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _collation = require("../collation");

var _token = require("./token");

const types = {
  1: {
    name: 'DATABASE',
    event: 'databaseChange'
  },
  2: {
    name: 'LANGUAGE',
    event: 'languageChange'
  },
  3: {
    name: 'CHARSET',
    event: 'charsetChange'
  },
  4: {
    name: 'PACKET_SIZE',
    event: 'packetSizeChange'
  },
  7: {
    name: 'SQL_COLLATION',
    event: 'sqlCollationChange'
  },
  8: {
    name: 'BEGIN_TXN',
    event: 'beginTransaction'
  },
  9: {
    name: 'COMMIT_TXN',
    event: 'commitTransaction'
  },
  10: {
    name: 'ROLLBACK_TXN',
    event: 'rollbackTransaction'
  },
  13: {
    name: 'DATABASE_MIRRORING_PARTNER',
    event: 'partnerNode'
  },
  17: {
    name: 'TXN_ENDED'
  },
  18: {
    name: 'RESET_CONNECTION',
    event: 'resetConnection'
  },
  20: {
    name: 'ROUTING_CHANGE',
    event: 'routingChange'
  }
};

function readNewAndOldValue(parser, length, type, callback) {
  switch (type.name) {
    case 'DATABASE':
    case 'LANGUAGE':
    case 'CHARSET':
    case 'PACKET_SIZE':
    case 'DATABASE_MIRRORING_PARTNER':
      return parser.readBVarChar(newValue => {
        parser.readBVarChar(oldValue => {
          switch (type.name) {
            case 'PACKET_SIZE':
              return callback(new _token.PacketSizeEnvChangeToken(parseInt(newValue), parseInt(oldValue)));

            case 'DATABASE':
              return callback(new _token.DatabaseEnvChangeToken(newValue, oldValue));

            case 'LANGUAGE':
              return callback(new _token.LanguageEnvChangeToken(newValue, oldValue));

            case 'CHARSET':
              return callback(new _token.CharsetEnvChangeToken(newValue, oldValue));

            case 'DATABASE_MIRRORING_PARTNER':
              return callback(new _token.DatabaseMirroringPartnerEnvChangeToken(newValue, oldValue));
          }
        });
      });

    case 'SQL_COLLATION':
    case 'BEGIN_TXN':
    case 'COMMIT_TXN':
    case 'ROLLBACK_TXN':
    case 'RESET_CONNECTION':
      return parser.readBVarByte(newValue => {
        parser.readBVarByte(oldValue => {
          switch (type.name) {
            case 'SQL_COLLATION':
              {
                const newCollation = newValue.length ? _collation.Collation.fromBuffer(newValue) : undefined;
                const oldCollation = oldValue.length ? _collation.Collation.fromBuffer(oldValue) : undefined;
                return callback(new _token.CollationChangeToken(newCollation, oldCollation));
              }

            case 'BEGIN_TXN':
              return callback(new _token.BeginTransactionEnvChangeToken(newValue, oldValue));

            case 'COMMIT_TXN':
              return callback(new _token.CommitTransactionEnvChangeToken(newValue, oldValue));

            case 'ROLLBACK_TXN':
              return callback(new _token.RollbackTransactionEnvChangeToken(newValue, oldValue));

            case 'RESET_CONNECTION':
              return callback(new _token.ResetConnectionEnvChangeToken(newValue, oldValue));
          }
        });
      });

    case 'ROUTING_CHANGE':
      return parser.readUInt16LE(valueLength => {
        // Routing Change:
        // Byte 1: Protocol (must be 0)
        // Bytes 2-3 (USHORT): Port number
        // Bytes 4-5 (USHORT): Length of server data in unicode (2byte chars)
        // Bytes 6-*: Server name in unicode characters
        parser.readBuffer(valueLength, routePacket => {
          const protocol = routePacket.readUInt8(0);

          if (protocol !== 0) {
            throw new Error('Unknown protocol byte in routing change event');
          }

          const port = routePacket.readUInt16LE(1);
          const serverLen = routePacket.readUInt16LE(3); // 2 bytes per char, starting at offset 5

          const server = routePacket.toString('ucs2', 5, 5 + serverLen * 2);
          const newValue = {
            protocol: protocol,
            port: port,
            server: server
          };
          parser.readUInt16LE(oldValueLength => {
            parser.readBuffer(oldValueLength, oldValue => {
              callback(new _token.RoutingEnvChangeToken(newValue, oldValue));
            });
          });
        });
      });

    default:
      console.error('Tedious > Unsupported ENVCHANGE type ' + type.name); // skip unknown bytes

      parser.readBuffer(length - 1, () => {
        callback(undefined);
      });
  }
}

function envChangeParser(parser, _options, callback) {
  parser.readUInt16LE(length => {
    parser.readUInt8(typeNumber => {
      const type = types[typeNumber];

      if (!type) {
        console.error('Tedious > Unsupported ENVCHANGE type ' + typeNumber); // skip unknown bytes

        return parser.readBuffer(length - 1, () => {
          callback(undefined);
        });
      }

      readNewAndOldValue(parser, length, type, token => {
        callback(token);
      });
    });
  });
}

var _default = envChangeParser;
exports.default = _default;
module.exports = envChangeParser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy90b2tlbi9lbnYtY2hhbmdlLXRva2VuLXBhcnNlci50cyJdLCJuYW1lcyI6WyJ0eXBlcyIsIm5hbWUiLCJldmVudCIsInJlYWROZXdBbmRPbGRWYWx1ZSIsInBhcnNlciIsImxlbmd0aCIsInR5cGUiLCJjYWxsYmFjayIsInJlYWRCVmFyQ2hhciIsIm5ld1ZhbHVlIiwib2xkVmFsdWUiLCJQYWNrZXRTaXplRW52Q2hhbmdlVG9rZW4iLCJwYXJzZUludCIsIkRhdGFiYXNlRW52Q2hhbmdlVG9rZW4iLCJMYW5ndWFnZUVudkNoYW5nZVRva2VuIiwiQ2hhcnNldEVudkNoYW5nZVRva2VuIiwiRGF0YWJhc2VNaXJyb3JpbmdQYXJ0bmVyRW52Q2hhbmdlVG9rZW4iLCJyZWFkQlZhckJ5dGUiLCJuZXdDb2xsYXRpb24iLCJDb2xsYXRpb24iLCJmcm9tQnVmZmVyIiwidW5kZWZpbmVkIiwib2xkQ29sbGF0aW9uIiwiQ29sbGF0aW9uQ2hhbmdlVG9rZW4iLCJCZWdpblRyYW5zYWN0aW9uRW52Q2hhbmdlVG9rZW4iLCJDb21taXRUcmFuc2FjdGlvbkVudkNoYW5nZVRva2VuIiwiUm9sbGJhY2tUcmFuc2FjdGlvbkVudkNoYW5nZVRva2VuIiwiUmVzZXRDb25uZWN0aW9uRW52Q2hhbmdlVG9rZW4iLCJyZWFkVUludDE2TEUiLCJ2YWx1ZUxlbmd0aCIsInJlYWRCdWZmZXIiLCJyb3V0ZVBhY2tldCIsInByb3RvY29sIiwicmVhZFVJbnQ4IiwiRXJyb3IiLCJwb3J0Iiwic2VydmVyTGVuIiwic2VydmVyIiwidG9TdHJpbmciLCJvbGRWYWx1ZUxlbmd0aCIsIlJvdXRpbmdFbnZDaGFuZ2VUb2tlbiIsImNvbnNvbGUiLCJlcnJvciIsImVudkNoYW5nZVBhcnNlciIsIl9vcHRpb25zIiwidHlwZU51bWJlciIsInRva2VuIiwibW9kdWxlIiwiZXhwb3J0cyJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUVBOztBQUVBOztBQTJCQSxNQUFNQSxLQUF5RCxHQUFHO0FBQ2hFLEtBQUc7QUFDREMsSUFBQUEsSUFBSSxFQUFFLFVBREw7QUFFREMsSUFBQUEsS0FBSyxFQUFFO0FBRk4sR0FENkQ7QUFLaEUsS0FBRztBQUNERCxJQUFBQSxJQUFJLEVBQUUsVUFETDtBQUVEQyxJQUFBQSxLQUFLLEVBQUU7QUFGTixHQUw2RDtBQVNoRSxLQUFHO0FBQ0RELElBQUFBLElBQUksRUFBRSxTQURMO0FBRURDLElBQUFBLEtBQUssRUFBRTtBQUZOLEdBVDZEO0FBYWhFLEtBQUc7QUFDREQsSUFBQUEsSUFBSSxFQUFFLGFBREw7QUFFREMsSUFBQUEsS0FBSyxFQUFFO0FBRk4sR0FiNkQ7QUFpQmhFLEtBQUc7QUFDREQsSUFBQUEsSUFBSSxFQUFFLGVBREw7QUFFREMsSUFBQUEsS0FBSyxFQUFFO0FBRk4sR0FqQjZEO0FBcUJoRSxLQUFHO0FBQ0RELElBQUFBLElBQUksRUFBRSxXQURMO0FBRURDLElBQUFBLEtBQUssRUFBRTtBQUZOLEdBckI2RDtBQXlCaEUsS0FBRztBQUNERCxJQUFBQSxJQUFJLEVBQUUsWUFETDtBQUVEQyxJQUFBQSxLQUFLLEVBQUU7QUFGTixHQXpCNkQ7QUE2QmhFLE1BQUk7QUFDRkQsSUFBQUEsSUFBSSxFQUFFLGNBREo7QUFFRkMsSUFBQUEsS0FBSyxFQUFFO0FBRkwsR0E3QjREO0FBaUNoRSxNQUFJO0FBQ0ZELElBQUFBLElBQUksRUFBRSw0QkFESjtBQUVGQyxJQUFBQSxLQUFLLEVBQUU7QUFGTCxHQWpDNEQ7QUFxQ2hFLE1BQUk7QUFDRkQsSUFBQUEsSUFBSSxFQUFFO0FBREosR0FyQzREO0FBd0NoRSxNQUFJO0FBQ0ZBLElBQUFBLElBQUksRUFBRSxrQkFESjtBQUVGQyxJQUFBQSxLQUFLLEVBQUU7QUFGTCxHQXhDNEQ7QUE0Q2hFLE1BQUk7QUFDRkQsSUFBQUEsSUFBSSxFQUFFLGdCQURKO0FBRUZDLElBQUFBLEtBQUssRUFBRTtBQUZMO0FBNUM0RCxDQUFsRTs7QUFrREEsU0FBU0Msa0JBQVQsQ0FBNEJDLE1BQTVCLEVBQTRDQyxNQUE1QyxFQUE0REMsSUFBNUQsRUFBb0dDLFFBQXBHLEVBQTJKO0FBQ3pKLFVBQVFELElBQUksQ0FBQ0wsSUFBYjtBQUNFLFNBQUssVUFBTDtBQUNBLFNBQUssVUFBTDtBQUNBLFNBQUssU0FBTDtBQUNBLFNBQUssYUFBTDtBQUNBLFNBQUssNEJBQUw7QUFDRSxhQUFPRyxNQUFNLENBQUNJLFlBQVAsQ0FBcUJDLFFBQUQsSUFBYztBQUN2Q0wsUUFBQUEsTUFBTSxDQUFDSSxZQUFQLENBQXFCRSxRQUFELElBQWM7QUFDaEMsa0JBQVFKLElBQUksQ0FBQ0wsSUFBYjtBQUNFLGlCQUFLLGFBQUw7QUFDRSxxQkFBT00sUUFBUSxDQUFDLElBQUlJLCtCQUFKLENBQTZCQyxRQUFRLENBQUNILFFBQUQsQ0FBckMsRUFBaURHLFFBQVEsQ0FBQ0YsUUFBRCxDQUF6RCxDQUFELENBQWY7O0FBRUYsaUJBQUssVUFBTDtBQUNFLHFCQUFPSCxRQUFRLENBQUMsSUFBSU0sNkJBQUosQ0FBMkJKLFFBQTNCLEVBQXFDQyxRQUFyQyxDQUFELENBQWY7O0FBRUYsaUJBQUssVUFBTDtBQUNFLHFCQUFPSCxRQUFRLENBQUMsSUFBSU8sNkJBQUosQ0FBMkJMLFFBQTNCLEVBQXFDQyxRQUFyQyxDQUFELENBQWY7O0FBRUYsaUJBQUssU0FBTDtBQUNFLHFCQUFPSCxRQUFRLENBQUMsSUFBSVEsNEJBQUosQ0FBMEJOLFFBQTFCLEVBQW9DQyxRQUFwQyxDQUFELENBQWY7O0FBRUYsaUJBQUssNEJBQUw7QUFDRSxxQkFBT0gsUUFBUSxDQUFDLElBQUlTLDZDQUFKLENBQTJDUCxRQUEzQyxFQUFxREMsUUFBckQsQ0FBRCxDQUFmO0FBZEo7QUFnQkQsU0FqQkQ7QUFrQkQsT0FuQk0sQ0FBUDs7QUFxQkYsU0FBSyxlQUFMO0FBQ0EsU0FBSyxXQUFMO0FBQ0EsU0FBSyxZQUFMO0FBQ0EsU0FBSyxjQUFMO0FBQ0EsU0FBSyxrQkFBTDtBQUNFLGFBQU9OLE1BQU0sQ0FBQ2EsWUFBUCxDQUFxQlIsUUFBRCxJQUFjO0FBQ3ZDTCxRQUFBQSxNQUFNLENBQUNhLFlBQVAsQ0FBcUJQLFFBQUQsSUFBYztBQUNoQyxrQkFBUUosSUFBSSxDQUFDTCxJQUFiO0FBQ0UsaUJBQUssZUFBTDtBQUFzQjtBQUNwQixzQkFBTWlCLFlBQVksR0FBR1QsUUFBUSxDQUFDSixNQUFULEdBQWtCYyxxQkFBVUMsVUFBVixDQUFxQlgsUUFBckIsQ0FBbEIsR0FBbURZLFNBQXhFO0FBQ0Esc0JBQU1DLFlBQVksR0FBR1osUUFBUSxDQUFDTCxNQUFULEdBQWtCYyxxQkFBVUMsVUFBVixDQUFxQlYsUUFBckIsQ0FBbEIsR0FBbURXLFNBQXhFO0FBRUEsdUJBQU9kLFFBQVEsQ0FBQyxJQUFJZ0IsMkJBQUosQ0FBeUJMLFlBQXpCLEVBQXVDSSxZQUF2QyxDQUFELENBQWY7QUFDRDs7QUFFRCxpQkFBSyxXQUFMO0FBQ0UscUJBQU9mLFFBQVEsQ0FBQyxJQUFJaUIscUNBQUosQ0FBbUNmLFFBQW5DLEVBQTZDQyxRQUE3QyxDQUFELENBQWY7O0FBRUYsaUJBQUssWUFBTDtBQUNFLHFCQUFPSCxRQUFRLENBQUMsSUFBSWtCLHNDQUFKLENBQW9DaEIsUUFBcEMsRUFBOENDLFFBQTlDLENBQUQsQ0FBZjs7QUFFRixpQkFBSyxjQUFMO0FBQ0UscUJBQU9ILFFBQVEsQ0FBQyxJQUFJbUIsd0NBQUosQ0FBc0NqQixRQUF0QyxFQUFnREMsUUFBaEQsQ0FBRCxDQUFmOztBQUVGLGlCQUFLLGtCQUFMO0FBQ0UscUJBQU9ILFFBQVEsQ0FBQyxJQUFJb0Isb0NBQUosQ0FBa0NsQixRQUFsQyxFQUE0Q0MsUUFBNUMsQ0FBRCxDQUFmO0FBbEJKO0FBb0JELFNBckJEO0FBc0JELE9BdkJNLENBQVA7O0FBeUJGLFNBQUssZ0JBQUw7QUFDRSxhQUFPTixNQUFNLENBQUN3QixZQUFQLENBQXFCQyxXQUFELElBQWlCO0FBQzFDO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQXpCLFFBQUFBLE1BQU0sQ0FBQzBCLFVBQVAsQ0FBa0JELFdBQWxCLEVBQWdDRSxXQUFELElBQWlCO0FBQzlDLGdCQUFNQyxRQUFRLEdBQUdELFdBQVcsQ0FBQ0UsU0FBWixDQUFzQixDQUF0QixDQUFqQjs7QUFFQSxjQUFJRCxRQUFRLEtBQUssQ0FBakIsRUFBb0I7QUFDbEIsa0JBQU0sSUFBSUUsS0FBSixDQUFVLCtDQUFWLENBQU47QUFDRDs7QUFFRCxnQkFBTUMsSUFBSSxHQUFHSixXQUFXLENBQUNILFlBQVosQ0FBeUIsQ0FBekIsQ0FBYjtBQUNBLGdCQUFNUSxTQUFTLEdBQUdMLFdBQVcsQ0FBQ0gsWUFBWixDQUF5QixDQUF6QixDQUFsQixDQVI4QyxDQVM5Qzs7QUFDQSxnQkFBTVMsTUFBTSxHQUFHTixXQUFXLENBQUNPLFFBQVosQ0FBcUIsTUFBckIsRUFBNkIsQ0FBN0IsRUFBZ0MsSUFBS0YsU0FBUyxHQUFHLENBQWpELENBQWY7QUFFQSxnQkFBTTNCLFFBQVEsR0FBRztBQUNmdUIsWUFBQUEsUUFBUSxFQUFFQSxRQURLO0FBRWZHLFlBQUFBLElBQUksRUFBRUEsSUFGUztBQUdmRSxZQUFBQSxNQUFNLEVBQUVBO0FBSE8sV0FBakI7QUFNQWpDLFVBQUFBLE1BQU0sQ0FBQ3dCLFlBQVAsQ0FBcUJXLGNBQUQsSUFBb0I7QUFDdENuQyxZQUFBQSxNQUFNLENBQUMwQixVQUFQLENBQWtCUyxjQUFsQixFQUFtQzdCLFFBQUQsSUFBYztBQUM5Q0gsY0FBQUEsUUFBUSxDQUFDLElBQUlpQyw0QkFBSixDQUEwQi9CLFFBQTFCLEVBQW9DQyxRQUFwQyxDQUFELENBQVI7QUFDRCxhQUZEO0FBR0QsV0FKRDtBQUtELFNBdkJEO0FBd0JELE9BOUJNLENBQVA7O0FBZ0NGO0FBQ0UrQixNQUFBQSxPQUFPLENBQUNDLEtBQVIsQ0FBYywwQ0FBMENwQyxJQUFJLENBQUNMLElBQTdELEVBREYsQ0FFRTs7QUFDQUcsTUFBQUEsTUFBTSxDQUFDMEIsVUFBUCxDQUFrQnpCLE1BQU0sR0FBRyxDQUEzQixFQUE4QixNQUFNO0FBQ2xDRSxRQUFBQSxRQUFRLENBQUNjLFNBQUQsQ0FBUjtBQUNELE9BRkQ7QUE3Rko7QUFpR0Q7O0FBRUQsU0FBU3NCLGVBQVQsQ0FBeUJ2QyxNQUF6QixFQUF5Q3dDLFFBQXpDLEVBQThFckMsUUFBOUUsRUFBcUk7QUFDbklILEVBQUFBLE1BQU0sQ0FBQ3dCLFlBQVAsQ0FBcUJ2QixNQUFELElBQVk7QUFDOUJELElBQUFBLE1BQU0sQ0FBQzZCLFNBQVAsQ0FBa0JZLFVBQUQsSUFBZ0I7QUFDL0IsWUFBTXZDLElBQUksR0FBR04sS0FBSyxDQUFDNkMsVUFBRCxDQUFsQjs7QUFFQSxVQUFJLENBQUN2QyxJQUFMLEVBQVc7QUFDVG1DLFFBQUFBLE9BQU8sQ0FBQ0MsS0FBUixDQUFjLDBDQUEwQ0csVUFBeEQsRUFEUyxDQUVUOztBQUNBLGVBQU96QyxNQUFNLENBQUMwQixVQUFQLENBQWtCekIsTUFBTSxHQUFHLENBQTNCLEVBQThCLE1BQU07QUFDekNFLFVBQUFBLFFBQVEsQ0FBQ2MsU0FBRCxDQUFSO0FBQ0QsU0FGTSxDQUFQO0FBR0Q7O0FBRURsQixNQUFBQSxrQkFBa0IsQ0FBQ0MsTUFBRCxFQUFTQyxNQUFULEVBQWlCQyxJQUFqQixFQUF3QndDLEtBQUQsSUFBVztBQUNsRHZDLFFBQUFBLFFBQVEsQ0FBQ3VDLEtBQUQsQ0FBUjtBQUNELE9BRmlCLENBQWxCO0FBR0QsS0FkRDtBQWVELEdBaEJEO0FBaUJEOztlQUVjSCxlOztBQUNmSSxNQUFNLENBQUNDLE9BQVAsR0FBaUJMLGVBQWpCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IFBhcnNlciBmcm9tICcuL3N0cmVhbS1wYXJzZXInO1xuaW1wb3J0IHsgSW50ZXJuYWxDb25uZWN0aW9uT3B0aW9ucyB9IGZyb20gJy4uL2Nvbm5lY3Rpb24nO1xuaW1wb3J0IHsgQ29sbGF0aW9uIH0gZnJvbSAnLi4vY29sbGF0aW9uJztcblxuaW1wb3J0IHtcbiAgRGF0YWJhc2VFbnZDaGFuZ2VUb2tlbixcbiAgTGFuZ3VhZ2VFbnZDaGFuZ2VUb2tlbixcbiAgQ2hhcnNldEVudkNoYW5nZVRva2VuLFxuICBQYWNrZXRTaXplRW52Q2hhbmdlVG9rZW4sXG4gIEJlZ2luVHJhbnNhY3Rpb25FbnZDaGFuZ2VUb2tlbixcbiAgQ29tbWl0VHJhbnNhY3Rpb25FbnZDaGFuZ2VUb2tlbixcbiAgUm9sbGJhY2tUcmFuc2FjdGlvbkVudkNoYW5nZVRva2VuLFxuICBEYXRhYmFzZU1pcnJvcmluZ1BhcnRuZXJFbnZDaGFuZ2VUb2tlbixcbiAgUmVzZXRDb25uZWN0aW9uRW52Q2hhbmdlVG9rZW4sXG4gIFJvdXRpbmdFbnZDaGFuZ2VUb2tlbixcbiAgQ29sbGF0aW9uQ2hhbmdlVG9rZW5cbn0gZnJvbSAnLi90b2tlbic7XG5cbnR5cGUgRW52Q2hhbmdlVG9rZW4gPVxuICBEYXRhYmFzZUVudkNoYW5nZVRva2VuIHxcbiAgTGFuZ3VhZ2VFbnZDaGFuZ2VUb2tlbiB8XG4gIENoYXJzZXRFbnZDaGFuZ2VUb2tlbiB8XG4gIFBhY2tldFNpemVFbnZDaGFuZ2VUb2tlbiB8XG4gIEJlZ2luVHJhbnNhY3Rpb25FbnZDaGFuZ2VUb2tlbiB8XG4gIENvbW1pdFRyYW5zYWN0aW9uRW52Q2hhbmdlVG9rZW4gfFxuICBSb2xsYmFja1RyYW5zYWN0aW9uRW52Q2hhbmdlVG9rZW4gfFxuICBEYXRhYmFzZU1pcnJvcmluZ1BhcnRuZXJFbnZDaGFuZ2VUb2tlbiB8XG4gIFJlc2V0Q29ubmVjdGlvbkVudkNoYW5nZVRva2VuIHxcbiAgUm91dGluZ0VudkNoYW5nZVRva2VuIHxcbiAgQ29sbGF0aW9uQ2hhbmdlVG9rZW47XG5cbmNvbnN0IHR5cGVzOiB7IFtrZXk6IG51bWJlcl06IHsgbmFtZTogc3RyaW5nLCBldmVudD86IHN0cmluZyB9fSA9IHtcbiAgMToge1xuICAgIG5hbWU6ICdEQVRBQkFTRScsXG4gICAgZXZlbnQ6ICdkYXRhYmFzZUNoYW5nZSdcbiAgfSxcbiAgMjoge1xuICAgIG5hbWU6ICdMQU5HVUFHRScsXG4gICAgZXZlbnQ6ICdsYW5ndWFnZUNoYW5nZSdcbiAgfSxcbiAgMzoge1xuICAgIG5hbWU6ICdDSEFSU0VUJyxcbiAgICBldmVudDogJ2NoYXJzZXRDaGFuZ2UnXG4gIH0sXG4gIDQ6IHtcbiAgICBuYW1lOiAnUEFDS0VUX1NJWkUnLFxuICAgIGV2ZW50OiAncGFja2V0U2l6ZUNoYW5nZSdcbiAgfSxcbiAgNzoge1xuICAgIG5hbWU6ICdTUUxfQ09MTEFUSU9OJyxcbiAgICBldmVudDogJ3NxbENvbGxhdGlvbkNoYW5nZSdcbiAgfSxcbiAgODoge1xuICAgIG5hbWU6ICdCRUdJTl9UWE4nLFxuICAgIGV2ZW50OiAnYmVnaW5UcmFuc2FjdGlvbidcbiAgfSxcbiAgOToge1xuICAgIG5hbWU6ICdDT01NSVRfVFhOJyxcbiAgICBldmVudDogJ2NvbW1pdFRyYW5zYWN0aW9uJ1xuICB9LFxuICAxMDoge1xuICAgIG5hbWU6ICdST0xMQkFDS19UWE4nLFxuICAgIGV2ZW50OiAncm9sbGJhY2tUcmFuc2FjdGlvbidcbiAgfSxcbiAgMTM6IHtcbiAgICBuYW1lOiAnREFUQUJBU0VfTUlSUk9SSU5HX1BBUlRORVInLFxuICAgIGV2ZW50OiAncGFydG5lck5vZGUnXG4gIH0sXG4gIDE3OiB7XG4gICAgbmFtZTogJ1RYTl9FTkRFRCdcbiAgfSxcbiAgMTg6IHtcbiAgICBuYW1lOiAnUkVTRVRfQ09OTkVDVElPTicsXG4gICAgZXZlbnQ6ICdyZXNldENvbm5lY3Rpb24nXG4gIH0sXG4gIDIwOiB7XG4gICAgbmFtZTogJ1JPVVRJTkdfQ0hBTkdFJyxcbiAgICBldmVudDogJ3JvdXRpbmdDaGFuZ2UnXG4gIH1cbn07XG5cbmZ1bmN0aW9uIHJlYWROZXdBbmRPbGRWYWx1ZShwYXJzZXI6IFBhcnNlciwgbGVuZ3RoOiBudW1iZXIsIHR5cGU6IHsgbmFtZTogc3RyaW5nLCBldmVudD86IHN0cmluZyB9LCBjYWxsYmFjazogKHRva2VuOiBFbnZDaGFuZ2VUb2tlbiB8IHVuZGVmaW5lZCkgPT4gdm9pZCkge1xuICBzd2l0Y2ggKHR5cGUubmFtZSkge1xuICAgIGNhc2UgJ0RBVEFCQVNFJzpcbiAgICBjYXNlICdMQU5HVUFHRSc6XG4gICAgY2FzZSAnQ0hBUlNFVCc6XG4gICAgY2FzZSAnUEFDS0VUX1NJWkUnOlxuICAgIGNhc2UgJ0RBVEFCQVNFX01JUlJPUklOR19QQVJUTkVSJzpcbiAgICAgIHJldHVybiBwYXJzZXIucmVhZEJWYXJDaGFyKChuZXdWYWx1ZSkgPT4ge1xuICAgICAgICBwYXJzZXIucmVhZEJWYXJDaGFyKChvbGRWYWx1ZSkgPT4ge1xuICAgICAgICAgIHN3aXRjaCAodHlwZS5uYW1lKSB7XG4gICAgICAgICAgICBjYXNlICdQQUNLRVRfU0laRSc6XG4gICAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhuZXcgUGFja2V0U2l6ZUVudkNoYW5nZVRva2VuKHBhcnNlSW50KG5ld1ZhbHVlKSwgcGFyc2VJbnQob2xkVmFsdWUpKSk7XG5cbiAgICAgICAgICAgIGNhc2UgJ0RBVEFCQVNFJzpcbiAgICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG5ldyBEYXRhYmFzZUVudkNoYW5nZVRva2VuKG5ld1ZhbHVlLCBvbGRWYWx1ZSkpO1xuXG4gICAgICAgICAgICBjYXNlICdMQU5HVUFHRSc6XG4gICAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhuZXcgTGFuZ3VhZ2VFbnZDaGFuZ2VUb2tlbihuZXdWYWx1ZSwgb2xkVmFsdWUpKTtcblxuICAgICAgICAgICAgY2FzZSAnQ0hBUlNFVCc6XG4gICAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhuZXcgQ2hhcnNldEVudkNoYW5nZVRva2VuKG5ld1ZhbHVlLCBvbGRWYWx1ZSkpO1xuXG4gICAgICAgICAgICBjYXNlICdEQVRBQkFTRV9NSVJST1JJTkdfUEFSVE5FUic6XG4gICAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhuZXcgRGF0YWJhc2VNaXJyb3JpbmdQYXJ0bmVyRW52Q2hhbmdlVG9rZW4obmV3VmFsdWUsIG9sZFZhbHVlKSk7XG4gICAgICAgICAgfVxuICAgICAgICB9KTtcbiAgICAgIH0pO1xuXG4gICAgY2FzZSAnU1FMX0NPTExBVElPTic6XG4gICAgY2FzZSAnQkVHSU5fVFhOJzpcbiAgICBjYXNlICdDT01NSVRfVFhOJzpcbiAgICBjYXNlICdST0xMQkFDS19UWE4nOlxuICAgIGNhc2UgJ1JFU0VUX0NPTk5FQ1RJT04nOlxuICAgICAgcmV0dXJuIHBhcnNlci5yZWFkQlZhckJ5dGUoKG5ld1ZhbHVlKSA9PiB7XG4gICAgICAgIHBhcnNlci5yZWFkQlZhckJ5dGUoKG9sZFZhbHVlKSA9PiB7XG4gICAgICAgICAgc3dpdGNoICh0eXBlLm5hbWUpIHtcbiAgICAgICAgICAgIGNhc2UgJ1NRTF9DT0xMQVRJT04nOiB7XG4gICAgICAgICAgICAgIGNvbnN0IG5ld0NvbGxhdGlvbiA9IG5ld1ZhbHVlLmxlbmd0aCA/IENvbGxhdGlvbi5mcm9tQnVmZmVyKG5ld1ZhbHVlKSA6IHVuZGVmaW5lZDtcbiAgICAgICAgICAgICAgY29uc3Qgb2xkQ29sbGF0aW9uID0gb2xkVmFsdWUubGVuZ3RoID8gQ29sbGF0aW9uLmZyb21CdWZmZXIob2xkVmFsdWUpIDogdW5kZWZpbmVkO1xuXG4gICAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhuZXcgQ29sbGF0aW9uQ2hhbmdlVG9rZW4obmV3Q29sbGF0aW9uLCBvbGRDb2xsYXRpb24pKTtcbiAgICAgICAgICAgIH1cblxuICAgICAgICAgICAgY2FzZSAnQkVHSU5fVFhOJzpcbiAgICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG5ldyBCZWdpblRyYW5zYWN0aW9uRW52Q2hhbmdlVG9rZW4obmV3VmFsdWUsIG9sZFZhbHVlKSk7XG5cbiAgICAgICAgICAgIGNhc2UgJ0NPTU1JVF9UWE4nOlxuICAgICAgICAgICAgICByZXR1cm4gY2FsbGJhY2sobmV3IENvbW1pdFRyYW5zYWN0aW9uRW52Q2hhbmdlVG9rZW4obmV3VmFsdWUsIG9sZFZhbHVlKSk7XG5cbiAgICAgICAgICAgIGNhc2UgJ1JPTExCQUNLX1RYTic6XG4gICAgICAgICAgICAgIHJldHVybiBjYWxsYmFjayhuZXcgUm9sbGJhY2tUcmFuc2FjdGlvbkVudkNoYW5nZVRva2VuKG5ld1ZhbHVlLCBvbGRWYWx1ZSkpO1xuXG4gICAgICAgICAgICBjYXNlICdSRVNFVF9DT05ORUNUSU9OJzpcbiAgICAgICAgICAgICAgcmV0dXJuIGNhbGxiYWNrKG5ldyBSZXNldENvbm5lY3Rpb25FbnZDaGFuZ2VUb2tlbihuZXdWYWx1ZSwgb2xkVmFsdWUpKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgfSk7XG5cbiAgICBjYXNlICdST1VUSU5HX0NIQU5HRSc6XG4gICAgICByZXR1cm4gcGFyc2VyLnJlYWRVSW50MTZMRSgodmFsdWVMZW5ndGgpID0+IHtcbiAgICAgICAgLy8gUm91dGluZyBDaGFuZ2U6XG4gICAgICAgIC8vIEJ5dGUgMTogUHJvdG9jb2wgKG11c3QgYmUgMClcbiAgICAgICAgLy8gQnl0ZXMgMi0zIChVU0hPUlQpOiBQb3J0IG51bWJlclxuICAgICAgICAvLyBCeXRlcyA0LTUgKFVTSE9SVCk6IExlbmd0aCBvZiBzZXJ2ZXIgZGF0YSBpbiB1bmljb2RlICgyYnl0ZSBjaGFycylcbiAgICAgICAgLy8gQnl0ZXMgNi0qOiBTZXJ2ZXIgbmFtZSBpbiB1bmljb2RlIGNoYXJhY3RlcnNcbiAgICAgICAgcGFyc2VyLnJlYWRCdWZmZXIodmFsdWVMZW5ndGgsIChyb3V0ZVBhY2tldCkgPT4ge1xuICAgICAgICAgIGNvbnN0IHByb3RvY29sID0gcm91dGVQYWNrZXQucmVhZFVJbnQ4KDApO1xuXG4gICAgICAgICAgaWYgKHByb3RvY29sICE9PSAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Vua25vd24gcHJvdG9jb2wgYnl0ZSBpbiByb3V0aW5nIGNoYW5nZSBldmVudCcpO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGNvbnN0IHBvcnQgPSByb3V0ZVBhY2tldC5yZWFkVUludDE2TEUoMSk7XG4gICAgICAgICAgY29uc3Qgc2VydmVyTGVuID0gcm91dGVQYWNrZXQucmVhZFVJbnQxNkxFKDMpO1xuICAgICAgICAgIC8vIDIgYnl0ZXMgcGVyIGNoYXIsIHN0YXJ0aW5nIGF0IG9mZnNldCA1XG4gICAgICAgICAgY29uc3Qgc2VydmVyID0gcm91dGVQYWNrZXQudG9TdHJpbmcoJ3VjczInLCA1LCA1ICsgKHNlcnZlckxlbiAqIDIpKTtcblxuICAgICAgICAgIGNvbnN0IG5ld1ZhbHVlID0ge1xuICAgICAgICAgICAgcHJvdG9jb2w6IHByb3RvY29sLFxuICAgICAgICAgICAgcG9ydDogcG9ydCxcbiAgICAgICAgICAgIHNlcnZlcjogc2VydmVyXG4gICAgICAgICAgfTtcblxuICAgICAgICAgIHBhcnNlci5yZWFkVUludDE2TEUoKG9sZFZhbHVlTGVuZ3RoKSA9PiB7XG4gICAgICAgICAgICBwYXJzZXIucmVhZEJ1ZmZlcihvbGRWYWx1ZUxlbmd0aCwgKG9sZFZhbHVlKSA9PiB7XG4gICAgICAgICAgICAgIGNhbGxiYWNrKG5ldyBSb3V0aW5nRW52Q2hhbmdlVG9rZW4obmV3VmFsdWUsIG9sZFZhbHVlKSk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9KTtcbiAgICAgICAgfSk7XG4gICAgICB9KTtcblxuICAgIGRlZmF1bHQ6XG4gICAgICBjb25zb2xlLmVycm9yKCdUZWRpb3VzID4gVW5zdXBwb3J0ZWQgRU5WQ0hBTkdFIHR5cGUgJyArIHR5cGUubmFtZSk7XG4gICAgICAvLyBza2lwIHVua25vd24gYnl0ZXNcbiAgICAgIHBhcnNlci5yZWFkQnVmZmVyKGxlbmd0aCAtIDEsICgpID0+IHtcbiAgICAgICAgY2FsbGJhY2sodW5kZWZpbmVkKTtcbiAgICAgIH0pO1xuICB9XG59XG5cbmZ1bmN0aW9uIGVudkNoYW5nZVBhcnNlcihwYXJzZXI6IFBhcnNlciwgX29wdGlvbnM6IEludGVybmFsQ29ubmVjdGlvbk9wdGlvbnMsIGNhbGxiYWNrOiAodG9rZW46IEVudkNoYW5nZVRva2VuIHwgdW5kZWZpbmVkKSA9PiB2b2lkKSB7XG4gIHBhcnNlci5yZWFkVUludDE2TEUoKGxlbmd0aCkgPT4ge1xuICAgIHBhcnNlci5yZWFkVUludDgoKHR5cGVOdW1iZXIpID0+IHtcbiAgICAgIGNvbnN0IHR5cGUgPSB0eXBlc1t0eXBlTnVtYmVyXTtcblxuICAgICAgaWYgKCF0eXBlKSB7XG4gICAgICAgIGNvbnNvbGUuZXJyb3IoJ1RlZGlvdXMgPiBVbnN1cHBvcnRlZCBFTlZDSEFOR0UgdHlwZSAnICsgdHlwZU51bWJlcik7XG4gICAgICAgIC8vIHNraXAgdW5rbm93biBieXRlc1xuICAgICAgICByZXR1cm4gcGFyc2VyLnJlYWRCdWZmZXIobGVuZ3RoIC0gMSwgKCkgPT4ge1xuICAgICAgICAgIGNhbGxiYWNrKHVuZGVmaW5lZCk7XG4gICAgICAgIH0pO1xuICAgICAgfVxuXG4gICAgICByZWFkTmV3QW5kT2xkVmFsdWUocGFyc2VyLCBsZW5ndGgsIHR5cGUsICh0b2tlbikgPT4ge1xuICAgICAgICBjYWxsYmFjayh0b2tlbik7XG4gICAgICB9KTtcbiAgICB9KTtcbiAgfSk7XG59XG5cbmV4cG9ydCBkZWZhdWx0IGVudkNoYW5nZVBhcnNlcjtcbm1vZHVsZS5leHBvcnRzID0gZW52Q2hhbmdlUGFyc2VyO1xuIl19