﻿# This script removes traces left behind Metatogger
# Version 1.0 (2020-12-05)
# Copyright Sylvain Rougeaux, 2020-2024, www.luminescence-software.org

$ErrorActionPreference = "Stop"
$whatIf = $false

@"
Metatogger Cleanup Tool

Arguments:
  /sf  --> Delete settings files
  /rf  --> Delete recovery file
  /md  --> Delete MusicBrainz local database
  /cc  --> Delete cover cache
  /fc  --> Delete fingerprints cache database
  /sr  --> Delete scripts repository
  /aeu --> Delete all traces except user settings and scripts
  /all --> Delete ALL Metatogger traces!

"@

Function Remove([parameter(mandatory)] [string] $name, [parameter(mandatory)] [string] $path, [parameter(mandatory)] [bool] $isFile, [bool] $askUser = $true)
{
   if ($isFile) 
   {
      $type = "file"
      $exists = Test-Path -Path $path -PathType Leaf
   }
   else
   {
      $type = "folder" 
      $exists = Test-Path -Path $path -PathType Container
   }
   

   if ($exists -eq $false)
   {
      Write-Host "There is no $name $type to remove."
      Write-Host
      return
   }

   if ($askUser)
   {
      $answer = Read-Host "Do you really want to delete the $name $type ? [press 'y' to delete]"
      if ($answer -ne "y")
      {
         Write-Host "The $name $type has not been deleted."
         Write-Host
         return
      }
   }

   try
   {
      Remove-Item -LiteralPath $path -Force -Recurse -WhatIf:$whatIf
      Write-Host "The $name $type has been deleted."
   }
   catch
   {
      Write-Host $_.Exception.Message -ForegroundColor Red
   }

   Write-Host
}

$process = Get-Process -Name "Metatogger" -ErrorAction SilentlyContinue
if ($null -ne $process)
{
   Write-Host "Metatogger must be closed before running this software to clean its traces."
   exit 1
}

$localApplicationData = [Environment]::GetFolderPath("LocalApplicationData")
$rootFolder = Join-Path -Path $localApplicationData -ChildPath "Metatogger"

$musicBrainzDbPath = Join-Path -Path $rootFolder -ChildPath "MusicBrainz.db"
$recoveryPath = Join-Path -Path $rootFolder -ChildPath "SessionRecovery.jms"
$systemSettingsPath = Join-Path -Path $rootFolder -ChildPath "SystemSettings.json"
$advancedSettingsPath = Join-Path -Path $rootFolder -ChildPath "AdvancedSettings.json"
$fingerprintDbPath = Join-Path -Path $rootFolder -ChildPath "Chromaprint.db"
$coverCacheFolder = Join-Path -Path $rootFolder -ChildPath "CoverCache"
$scriptsFolder = Join-Path -Path $rootFolder -ChildPath "Scripts"

if ($args.Count -eq 0)
{
   Write-Host "No argument has been passed. Using interactive mode."
   Write-Host

   Remove "system settings" $systemSettingsPath $true
   Remove "advanced settings" $advancedSettingsPath $true
   Remove "recovery" $recoveryPath $true
   Remove "MusicBrainz local database" $musicBrainzDbPath $true
   Remove "cover cache" $coverCacheFolder $false
   Remove "fingerprints cache database" $fingerprintDbPath $true
   Remove "scripts repository" $scriptsFolder $false
}
else 
{
   Write-Host "Arguments passed: $args"
   Write-Host

   if ($args.Contains("/sf") -or $args.Contains("/all"))
   {
      Remove "system settings" $systemSettingsPath $true $false
      Remove "advanced settings" $advancedSettingsPath $true $false
   }

   if ($args.Contains("/rf") -or $args.Contains("/all") -or $args.Contains("/aeu")) 
   {
      Remove "recovery" $recoveryPath $true $false
   }

   if ($args.Contains("/md") -or $args.Contains("/all") -or $args.Contains("/aeu"))
   {
      Remove "MusicBrainz local database" $musicBrainzDbPath $true $false
   }

   if ($args.Contains("/cc") -or $args.Contains("/all") -or $args.Contains("/aeu"))
   {
      Remove "cover cache" $coverCacheFolder $false $false
   }

   if ($args.Contains("/fc") -or $args.Contains("/all") -or $args.Contains("/aeu"))
   {
      Remove "fingerprints cache database" $fingerprintDbPath $true $false
   }

   if ($args.Contains("/sr") -or $args.Contains("/all"))
   {
      Remove "scripts repository" $scriptsFolder $false $false
   }
}

if ((Test-Path -Path "$rootFolder\*") -eq $false)
{
   Remove "empty root Metatogger" $rootFolder $false $false
}
else 
{
   Write-Host "The root Metatogger folder still contains data."
}
