#!/bin/sh

TMP_DIR="/tmp/com.wetype.inputmethod"
EXPECTED_IDENTIFIER="com.tencent.inputmethod.wetype"
EXPECTED_AUTHORITY="Developer ID Application: Tencent Technology (Shenzhen) Company Limited"
timestamp=$(date +"%Y-%m-%d_%H-%M-%S")
log_file="${TMP_DIR}/update_${timestamp}.log"
UPDATE_ZIP=$1
UPDATE_METHODS_DIR=$2

log_message() {
    local message="$1"
    echo "$(date +%H:%M:%S) $message"
    echo "$(date +%H:%M:%S) $message" >> "$log_file"
}

function verify_app_signature() {
    APP_PATH="$1"
    SIGNATURE=$(codesign --display --verbose=4 "$APP_PATH" 2>&1)
    if [[ $SIGNATURE =~ Identifier=("$EXPECTED_IDENTIFIER") && $SIGNATURE =~ Authority=("$EXPECTED_AUTHORITY") ]]; then
        return 0  # success
    else
        log_message "Application signature is invalid. Original signature information:"
        log_message "$SIGNATURE"
        return 1  # failure
    fi
}

function update_normally() {
    if [ ! -d "/Library/Input Methods/WeType.app/Contents" ]; then
        log_message "target folder does not exist, moving directly"
        mv -f "$UPDATE_METHODS_DIR/WeType.app/Contents" "/Library/Input Methods/WeType.app/Contents"
    else
        log_message "target folder exists, performing renaming"
        scr="$UPDATE_METHODS_DIR/WeType.app/Contents"
        target="/Library/Input Methods/WeType.app/Contents_update"
        if ! mv -f "$scr" "$target"; then
            log_message "failed to move and rename $scr -> $target"
            return 105
        fi
        log_message "move and rename successful"
        if ! rm -rf "/Library/Input Methods/WeType.app/Contents"; then
            log_message "failed to delete original folder"
            return 106
        fi
        log_message "original folder deleted successfully"
        if ! mv "/Library/Input Methods/WeType.app/Contents_update" "/Library/Input Methods/WeType.app/Contents"; then
            log_message "failed to rename"
            return 107
        fi
        if chown -R root:staff "/Library/Input Methods/WeType.app" && chmod -R 775 "/Library/Input Methods/WeType.app"; then
            log_message "permission change successful"
        fi
        log_message "rename successful"
    fi
    return 0
}

function update_su() {
    if sudo chown -R root:staff "/Library/Input Methods/WeType.app" && sudo chmod -R 775 "/Library/Input Methods/WeType.app"; then
        log_message "permission change successful, 1"
        update_normally
        return 0
    else
        log_message "permission change failed, try another way"
        if [ -d /Library/Input\ Methods/WeType_Update.app/ ]; then
            log_message "del WeType_Update.app"
            sudo rm -rf /Library/Input\ Methods/WeType_Update.app/
        fi
        sudo rsync -av --exclude='*/' /Library/Input\ Methods/WeType.app/ /Library/Input\ Methods/WeType_Update.app/
        if [ $? -eq 1 ]; then
            log_message "failed to create WeType2"
            return 101
        fi
        if ! sudo mv -f "$UPDATE_METHODS_DIR/WeType.app/Contents" /Library/Input\ Methods/WeType_Update.app/Contents; then
            log_message "failed move to WeType_Update.app"
            return 102
        fi
        if ! sudo rm -rf  /Library/Input\ Methods/WeType.app; then
            log_message "failed del WeType.app"
            return 103
        fi
        sudo mv -f  /Library/Input\ Methods/WeType_Update.app  /Library/Input\ Methods/WeType.app
        sudo chown -R root:staff /Library/Input\ Methods/WeType.app && sudo chmod -R 775 /Library/Input\ Methods/WeType.app
        if [ $? -eq 1 ]; then
            log_message "failed to change permissions of copied folder"
            return 104
        fi
        log_message "permission change successful, 2"
        return 0
    fi
}


function clean_update_dir() {
    local error_code=$1
    if [ $error_code -eq 0 ]; then
        log_message "update success"
    else
        log_message "update error $error_code"
    fi
    log_message "clean_update_dir $UPDATE_METHODS_DIR"
    rm -rf "$UPDATE_METHODS_DIR"
    exit $error_code
}


# ===== unzip =====
log_message "$(date +%H:%M:%S) unzip package at $UPDATE_ZIP, target: $UPDATE_METHODS_DIR"
unzip -qq "$UPDATE_ZIP" -d "$UPDATE_METHODS_DIR"
if [ -d "$UPDATE_METHODS_DIR/WeType.app" ]; then
    
    verify_app_signature "$UPDATE_METHODS_DIR/WeType.app"
    if [ $? -eq 0 ]; then
        log_message "Application signature is valid, Identifier and Authority match."
            # ===== update =====
        if [ $EUID -eq 0 ]; then
            log_message "update_su"
            update_su
            clean_update_dir $?
        else
            log_message "update_normally"
            update_normally
            clean_update_dir $?
        fi
    else
        log_message "Application signature is invalid, Identifier or Authority do not match."
        clean_update_dir 100
    fi
else
    log_message "unzip err"
    clean_update_dir 100
fi
