--
-- this is a batterbowl product
-- it moves to an appliance to get backed and when it's ready
-- it can be clicked again to move to a product stock
--
-- it's the first Lua implemented product,
-- so it contains extensive comments

class 'batterbowlProduct' (Product)

--
-- additional variables are declared here
-- when you add an extra variable, don't forget it in the
-- copy constructor!
batterbowlProduct.timer = 0
batterbowlProduct.timerSuspended = true

function batterbowlProduct:__init(root) super(root)

	-- C++ product has two constructor:
	-- a const std::string& which is the baseName of the product
	-- a const Product& which is the copy constructor
	-- 
	-- in this Lua constructor, we need to do explicit type checking
	-- on the 'root' parameter (which, as said, can be either a 
	-- string or a source product), so we know when the copy 
	-- constructor gets called. if the copy constructor is called
	-- all lua variables need to be copied over as well

	if (type(root) == "userdata") then
		-- called constructor (const Product&)
		-- this is the copy constructor
		-- copy over all lua values
		self.timer = root.timer
		self.timerSuspended = root.timerSuspended
	
	end	


end

--
-- called when the product is first created
-- note that it's _not_ called when the copy constructor is used.
function batterbowlProduct:onCreate()
	
	self:setState("RAW")

end

--
-- called when the product enters an Appliance slot
function batterbowlProduct:onSlotEnter()
end

--
-- called when the product leaves an Appliance slot
function batterbowlProduct:onSlotLeave()
end

--
-- called when the product is clicked
function batterbowlProduct:onClick()

	if (self:getState() == "READY") then
		self.timerSuspended = true
		
	end	

	Product.onClick(self)

end

--
-- the state events are next
-- every product is in a state. state information can be accessed
-- through setState() and getState(). when setState() is called,
-- the product will first receive an OnStateLeave(oldState), then
-- its set to the new state and next it will receive 
-- onStateEnter(newState)

--
-- called when the product enters a new state
function batterbowlProduct:onStateEnter(state)

	if (self:getState() == "RAW") then
		self:setWorkTime(800)
		self:setWorkAnimation("WORK")
		self:animate("PROD_BUT_BATTERBOWL", 0)
	
	elseif (self:getState() == "BAKING") then
		self:setClickable(false)
		self:animate("PROD_BUT_BATTER", 0)
		self:playSound("SOUND_PANKAKE_BAKING", true)
		self.timer = 2000
		self.timerSuspended = false
	
	elseif (self:getState() == "READY") then
		self:blink()
		self:setClickable(true)
		self:animate("PROD_BUT_WAFFLE", 0)
		self:playSound("SOUND_FRYER_READY")		
		self:setWorkTime(200)
		self:setWorkAnimation("WORK")
		self.timer = 4000
		self.timerSuspended = false
		
	elseif (self:getState() == "BURNING") then
		self:setClickable(false)
		self:clearOverhead()
		self:stopSound("SOUND_PANKAKE_BAKING")
		self:playSound("SOUND_BURGER_EGG_BURNS")
		self:animate("PROD_ANIM_WAFFLE_BURNING", 1, false, 12)
		self:setY(self:getY() - 19)
		self.timer = 1500
		self.timerSuspended = false
	
	elseif (self:getState() == "FLYING") then
		self:setClickable(false)
		self:setDestroyed(true)
		self:clearOverhead()
	
	end

end

--
-- called when the product leaves a state
function batterbowlProduct:onStateLeave(state)
end

--
-- called when the product is initially received by an iProductReceiver
-- examples from iProductReceiver: 
-- * Appliance
function batterbowlProduct:onReceive()

	if (self:getState() == "RAW") then
		self:setState("BAKING")
	
	elseif (self:getState() == "READY") then
		self:setClickable(false)
	
	end

end

--
-- called when the product is completely received by an iProductReceiver
function batterbowlProduct:onReceiveComplete()

	if (self:getState() == "READY") then
		getProduct("WAFFLE"):incStock()
		self:setDestroyed(true)
		
	end

end

--
-- called when a person starts working on the product
-- if work is allowed to start it needs to return 'true'
-- otherwise it should return 'false' and the task will be aborted
function batterbowlProduct:onWorkStart(task)

	if (self:getState() == "RAW") then
		-- move the product to the iron if the iron has enough slots open
		if (getAppliance("rest0_waffle_iron"):hasSlotsOpen()) then
			task:setDuration(self:getWorkTime())
			task:getPerson():animate(self:getWorkAnimation(), -1, false, -1)
			return true
			
		end
		
	elseif (self:getState() == "READY") then
		-- move the product to the stock if there's room on the stock
		if (getProduct("WAFFLE"):getStock() < getProduct("WAFFLE"):getMaxStock()) then
			task:setDuration(self:getWorkTime())
			task:getPerson():animate(self:getWorkAnimation(), -1, false, -1)
			self:stopSound("SOUND_PANKAKE_BAKING")
			return true	
			
		else
			self.timerSuspended = false
			return false	
		
		end
	end
	
	return false

end

--
-- called each tick while a person is working on the product
-- if work is complete, it should return 'true'
-- otherwise it should return 'false'
function batterbowlProduct:onWorkUpdate(task, time)

	return Product.onWorkUpdate(self, task, time)
	
end

--
-- called when a person completed work on a product
function batterbowlProduct:onWorkComplete(task)
	
	if (self:getState() == "RAW") then
		self:moveToAppliance(getAppliance("rest0_waffle_iron"))
	
	elseif (self:getState() == "READY") then
		self:moveToAppliance(getAppliance("rest0_waffle_plate"))
		
		--
		-- changing the state of the waffle here makes sure we can't spam click
		-- it and add multiple waffles to our stack as future tasks won't start
		-- because it's not in the appropriate state
		self:setState("FLYING")		
		
	end

end

--
-- called each tick
-- useful for timers
function batterbowlProduct:onTick(time)

	if (not scene:isStartingMovie() and not scene:isPlayingMovie()) then
		if (not self.timerSuspended) then
			if (self:getState() == "BAKING") then
				self.timer = self.timer - time
				if (self.timer <= 0) then
					self:setState("READY")
				
				end
					
			elseif (self:getState() == "READY") then
				self.timer = self.timer - time
				if (self.timer <= 0) then
					self:setState("BURNING")
					
				end	
			end	
		end
	end
	
	if (not self.timerSuspended) then
		if (self:getState() == "BURNING") then
			self.timer = self.timer - time
			if (self.timer <= 0) then
				self:setDestroyed(true)
				self:setVisible(false)
				
			end	
		end
	end

end