--
-- this is a batterbowl product
-- it moves to an appliance to get backed and when it's ready
-- it can be clicked again to move to a product stock
--
-- it's the first Lua implemented product,
-- so it contains extensive comments

class 'skilletProduct' (Product)

--
-- additional variables are declared here
-- when you add an extra variable, don't forget it in the
-- copy constructor!
skilletProduct.timer = 0
skilletProduct.timerSuspended = true
skilletProduct.busy = false

function skilletProduct:__init(root) super(root)

	-- C++ product has two constructor:
	-- a const std::string& which is the baseName of the product
	-- a const Product& which is the copy constructor
	-- 
	-- in this Lua constructor, we need to do explicit type checking
	-- on the 'root' parameter (which, as said, can be either a 
	-- string or a source product), so we know when the copy 
	-- constructor gets called. if the copy constructor is called
	-- all lua variables need to be copied over as well

	if (type(root) == "userdata") then
		-- called constructor (const Product&)
		-- this is the copy constructor
		-- copy over all lua values
		self.timer = root.timer
		self.timerSuspended = root.timerSuspended
		self.busy = root.busy
	end	
end

--
-- called when the product is first created
-- note that it's _not_ called when the copy constructor is used.
function skilletProduct:onCreate()
	self:setState("RAW")
end

--
-- handle the balloon
function skilletProduct:onClick()
	
    if (self:getState() == "RAW") then
		if (self:isQueuableToTray()) then    
			Product.onClick(self)	
	        
        end
    elseif (self:getState() == "READY") then
        if (not self.busy) then
            self.timerSuspended = true
            self:queue()
            self.busy = true
	        Product.onClick(self)	    
	                
        end    
    end	
	
end

--
-- called when the product enters a new state
function skilletProduct:onStateEnter(state)
	if (self:getState() == "RAW") then
		self:setWorkAnimation("WORK_FRONT")
		--self:animate("PROD_BUT_BATTERBOWL", 0)
	
	elseif (self:getState() == "BAKING") then

		self:setClickable(false)
		--self:animate("PROD_BUT_BATTER", 0)
		self:playSound("SOUND_SKILLETING_LOOP", true)
		self.timer = 1500
		self.timerSuspended = false
	
	elseif (self:getState() == "READY") then
		self:blink()
		self:setClickable(true)
		self:animate("PROD_BUT_" .. self:getBase() .. "_BAKED", 0)
		self:playSound("SOUND_FRYER_READY")		
		self:setWorkTime(0)
		self:setWorkAnimation("WORK_FRONT")
		self.timer = 6000
		self.timerSuspended = false
		
	elseif (self:getState() == "BURNING") then
		self:setClickable(false)
		self:clearOverhead()
		self:animate("PROD_BUT_" .. self:getBase(), 0) -- Temp
		self:stopSound("SOUND_SKILLETING_LOOP")
		self:playSound("SOUND_BURGER_EGG_BURNS")
		self:animate("ANIM_" .. self:getBase() .. "_BURN", 1, 18)
		self:setY(self:getY() - 19)
		self.timer = 1000
		self.timerSuspended = false
		
	end
end

--
-- called when the product is completely received by an iProductReceiver
function skilletProduct:onReceiveComplete()
	if (self:getState() == "RAW") then
		self:setState("BAKING")
	end
end

--
-- called when the worker starts to work on this product
function skilletProduct:onWorkStart(task)

	if (self:isDestroyed()) then
		warn("skilletProduct:onWorkStart - is destroyed")
		return false
		
	elseif (self:getState() == "RAW") then
		if (getAppliance("rest3_skillet"):hasSlotsOpen()) then
			return Product.onWorkStart(self, task)
			
		else
			return false
				
		end
	elseif (self:getState() == "READY") then
		if (self:isQueuedOnTray()) then
			self:stopSound("SOUND_SKILLETING_LOOP")		
			return true
		
		else
			self.busy = false
			self.timerSuspended = false
			self:unQueue()
			return false		
		
		end
	end

	warn("skilletProduct:onWorkStart - invalid state: " .. self:getState())
	return false	
	
end

-- called when a person completed work on a product
function skilletProduct:onWorkComplete(task)
	if (self:getState() == "RAW") then
		self:moveToAppliance(getAppliance("rest3_skillet"))
		
	elseif (self:getState() == "READY") then
		if (self:isMovableToBubble()) then
			self:moveToBubble()				
			self:setVisible(false)
			self:setClickable(false)
		else
			self:unQueue()
		end	
	end
	
	self.busy = false
end

-- this event is only raised if it's an ingredient
-- of a compound product
function skilletProduct:onBubbleCancel()	

	self:playSound("SOUND_SKILLETING_LOOP", true)
	self:unQueue()
	self:setVisible(true)
	self:setClickable(true)
	self.timerSuspended = false

end

-- this event is only raised if it's an ingredient
-- of a compound product
function skilletProduct:onBubbleMerged()

	if (self:isIngredient()) then
		if (not self:isVisible()) then
			self:setDestroyed(true)
		end
	end

end

-- called each tick
-- useful for timers
function skilletProduct:onTick(time)

	if (not scene:isStartingMovie() and not scene:isPlayingMovie()) then
		if (not self.timerSuspended) then
			if (self:getState() == "BAKING") then
				self.timer = self.timer - time
				if (self.timer <= 0) then
					self:setState("READY")				
				end
					
			elseif (self:getState() == "READY") then
				self.timer = self.timer - time
				if (self.timer <= 0) then
					self:setState("BURNING")					
				end	
			end	
		end
	end
	
	if (not self.timerSuspended) then
		if (self:getState() == "BURNING") then
			self.timer = self.timer - time
			if (self.timer <= 0) then
				self:setY(self:getY() + 19)
				self:setDestroyed(true)
				self:setVisible(false)
				
			end	
		end
	end

end

function skilletProduct:onSlotEnter(slot)

	if (slot:getName() == "slot1") then
		self:setWorkOffsetX(-1)
	
	elseif (slot:getName() == "slot2") then
		self:setWorkOffsetX(2)
	
	else
		warn("skilletProduct:onSlotEnter - invalid slot name")
		
	end
	
	

end