/*
 * Copyright (c) 2020 Huawei Device Co., Ltd.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @addtogroup Registry
 * @{
 *
 * @brief Provides APIs for registering and discovering inter-process communication (IPC)
 * capabilities.
 *
 * Based on the Samgr development framework, this module helps you to develop system capabilities
 * and implement cross-process calls. \n
 * This module is used when system capabilities need to be provided across processes. \n
 *
 * @since 1.0
 * @version 1.0
 */

/**
 * @file iproxy_client.h
 *
 * @brief Provides the client proxy class.
 *
 * When you need to call system capabilities of other processes, obtain the class from Samgr. \n
 *
 * @since 1.0
 * @version 1.0
 */

#ifndef LITE_IPROXY_CLIENT_H
#define LITE_IPROXY_CLIENT_H

#include "iunknown.h"
#include "message.h"
#include "serializer.h"

#ifdef __cplusplus
#if __cplusplus
extern "C" {
#endif
#endif

#define CLIENT_PROXY_VER 0x40

/**
 * @brief Indicates the inherited macro of the client proxy.
 *
 * This constant is used when a client proxy needs to be customized or generated by a tool. \n
 *
 */
#define INHERIT_CLIENT_IPROXY \
        INHERIT_IUNKNOWN; \
        int (*Invoke)(IClientProxy *proxy, int funcId, IpcIo *request, IOwner owner, INotify notify)

typedef void *IOwner;

/**
 * @brief Called when a client request is responded.
 *
 * The client implements this <b>INotify</b> callback to receive response data from the server. \n
 * <b>owner</b> indicates the client proxy that receives the response data; <b>code</b> indicates
 * the error code of the response data from the server; <b>reply</b> indicates the response data. \n
 *
 */
typedef int (*INotify)(IOwner owner, int code, IpcIo *reply);

typedef struct IClientProxy IClientProxy;

/**
 * @brief Defines the client proxy object.
 *
 * This object is used for the IPC with the server. \n
 * If you want to use the same invocation mode as that on the server, create an object inherited
 * from {@code IClientProxy} and implement serialization.
 *
 * @since 1.0
 * @version 1.0
 */
struct IClientProxy {
    /** Inherites the <b>IUnknown</b> base class. */
    INHERIT_IUNKNOWN;

    /**
     * @brief Sends an IPC message from the client to the <b>IServerProxy</b>.
     *
     * This function is used for IPC. \n
     * The passed <b>proxy</b> is used to obtain the server information. Then, <b>request</b>
     * carries the request message to be sent to the server and processed by the function specified
     * by <b>funcId</b>. <b>notify</b> is a callback function used to process the response message. \n
     *
     * @param proxy Indicates the pointer of the client proxy object.
     * @param funcId Indicates the ID of the function implemented on the server.
     * @param request Indicates the pointer to the serialized request message.
     * @param owner Indicates the receiver (generics type) of the response message.
     * @param notify Indicates the callback function that notifies the client of the response
     * message.
     * @return Returns <b>EC_SUCCESS</b> if the IPC message is sent successfully; returns other
     * error codes if the message fails to be sent.
     * @since 1.0
     * @version 1.0
     */
    int (*Invoke)(IClientProxy *proxy, int funcId, IpcIo *request, IOwner owner, INotify notify);
};

#define INHERIT_IPROXY_ENTRY(T) INHERIT_IUNKNOWNENTRY(T)

#define CLIENT_IPROXY_BEGIN  IUNKNOWN_ENTRY_BEGIN(CLIENT_PROXY_VER)

#define IPROXY_END IUNKNOWN_ENTRY_END

/**
 * @brief Obtains the IPC address of a remote service and feature based on the service name
 * and feature name.
 *
 * This function is used when {@link IClientProxy} cannot meet your requirements for calling IPCs. \n
 * For example, if you need to receive the death notification of a remote service or feature,
 * you can call this function to obtain the address of the remote service or feature
 * and subscribe to the death notification from the IPC. \n
 *
 * @param service Indicates the pointer to the name of the remote service.
 * @param feature Indicates the pointer to the name of the remote feature.
 * @return Returns the IPC address of the remote service or feature. When the handle of the
 * obtained address structure {@link SvcIdentity} is <b>0xFFFFFFFF</b>, the address is invalid.
 * @attention This function can be called only after <b>GetFeatureApi</b> in {@link SamgrLite}
 * is successfully called. Otherwise, an invalid address is returned. When the service or feature
 * does not support IPC communication, an invalid address will be returned.
 * @since 1.0
 * @version 1.0
 */
SvcIdentity SAMGR_GetRemoteIdentity(const char *service, const char *feature);

#ifdef __cplusplus
#if __cplusplus
}
#endif
#endif
#endif // LITE_IPROXY_CLIENT_H
/** @} */
