// SPDX-License-Identifier: GPL-2.0-or-later
/*
 * Copyright (c) 2017 Fujitsu Ltd.
 * Author: Guangwen Feng <fenggw-fnst@cn.fujitsu.com>
 */

/*
 * Test for CVE-2016-7042, this regression test can crash the buggy kernel
 * when the stack-protector is enabled, and the bug was fixed in:
 *
 *  commit 03dab869b7b239c4e013ec82aea22e181e441cfc
 *  Author: David Howells <dhowells@redhat.com>
 *  Date:   Wed Oct 26 15:01:54 2016 +0100
 *
 *  KEYS: Fix short sprintf buffer in /proc/keys show function
 */

#include <errno.h>
#include <stdio.h>

#include "tst_test.h"
#include "lapi/keyctl.h"

#define PATH_KEYS	"/proc/keys"

static key_serial_t key;
static int fd;

static void do_test(void)
{
	char buf[BUFSIZ];

	key = add_key("user", "ltptestkey", "a", 1, KEY_SPEC_SESSION_KEYRING);
	if (key == -1)
		tst_brk(TBROK, "Failed to add key");

	if (keyctl(KEYCTL_UPDATE, key, "b", 1))
		tst_brk(TBROK, "Failed to update key");

	fd = SAFE_OPEN(PATH_KEYS, O_RDONLY);

	tst_res(TINFO, "Attempting to crash the system");

	SAFE_READ(0, fd, buf, BUFSIZ);

	tst_res(TPASS, "Bug not reproduced");

	SAFE_CLOSE(fd);

	if (keyctl(KEYCTL_UNLINK, key, KEY_SPEC_SESSION_KEYRING))
		tst_brk(TBROK, "Failed to unlink key");
	key = 0;
}

static void setup(void)
{
	if (access(PATH_KEYS, F_OK))
		tst_brk(TCONF, "%s does not exist", PATH_KEYS);
}

static void cleanup(void)
{
	if (key > 0 && keyctl(KEYCTL_UNLINK, key, KEY_SPEC_SESSION_KEYRING))
		tst_res(TWARN, "Failed to unlink key");

	if (fd > 0)
		SAFE_CLOSE(fd);
}

static struct tst_test test = {
	.setup = setup,
	.cleanup = cleanup,
	.test_all = do_test,
	.tags = (const struct tst_tag[]) {
		{"linux-git", "03dab869b7b2"},
		{"CVE", "2016-7042"},
		{}
	}
};
