# encoding: utf-8
from __future__ import unicode_literals

import os.path
import subprocess
import sys
import re
import time
import traceback
from ..utilsEX import debug

from .common import FileDownloader
from ..compat import (
    compat_setenv,
    compat_str,
)
from ..utilsEX import  get_top_host

from ..postprocessor.ffmpeg import FFmpegPostProcessor, EXT_TO_OUT_FORMATS
from ..utils import (
    cli_option,
    cli_valueless_option,
    cli_bool_option,
    cli_configuration_args,
    encodeFilename,
    encodeArgument,
    handle_youtubedl_headers,

    check_executable,
    is_outdated_version,
    detect_exe_version,
)
from .external import ExternalFD

def getStartInfo():
    IS_WIN32 = 'win32' in str(sys.platform).lower()
    if IS_WIN32:
        startupinfo = subprocess.STARTUPINFO()
        startupinfo.dwFlags = subprocess.CREATE_NEW_CONSOLE | subprocess.STARTF_USESHOWWINDOW
        startupinfo.wShowWindow = subprocess.SW_HIDE
        return startupinfo
    else:
        return None

# 解决日文等unicode下解码错误
def encodeArgument(s):
    return s

def encodeFilename(s, for_subprocess=False):
    return s

def get_exe_version(exe, args=['--version'],
                    version_re=None, unrecognized='present'):
    """ Returns the version of the specified executable,
    or False if the executable is not present """
    try:
        # STDIN should be redirected too. On UNIX-like systems, ffmpeg triggers
        # SIGTTOU if youtube-dl is run in the background.
        # See https://github.com/rg3/youtube-dl/issues/955#issuecomment-209789656
        out, _ = subprocess.Popen(
            [encodeArgument(exe)] + args,
            stdin=subprocess.PIPE,
            stdout=subprocess.PIPE, stderr=subprocess.STDOUT, startupinfo=getStartInfo()).communicate()
    except OSError:
        return False
    if isinstance(out, bytes):  # Python 2.x
        out = out.decode('ascii', 'ignore')
    return detect_exe_version(out, version_re, unrecognized)

class FFmpegPostProcessorEx(FFmpegPostProcessor):
    def _determine_executables(self):
        programs = ['avprobe', 'avconv', 'ffmpeg', 'ffprobe']
        prefer_ffmpeg = False

        self.basename = None
        self.probe_basename = None

        self._paths = None
        self._versions = None
        if self._downloader:
            prefer_ffmpeg = self._downloader.params.get('prefer_ffmpeg', False)
            location = self._downloader.params.get('ffmpeg_location')
            if location is not None:
                if not os.path.exists(location):
                    self._downloader.report_warning(
                        'ffmpeg-location %s does not exist! '
                        'Continuing without avconv/ffmpeg.' % (location))
                    self._versions = {}
                    return
                elif not os.path.isdir(location):
                    basename = os.path.splitext(os.path.basename(location))[0]
                    if basename not in programs:
                        self._downloader.report_warning(
                            'Cannot identify executable %s, its basename should be one of %s. '
                            'Continuing without avconv/ffmpeg.' %
                            (location, ', '.join(programs)))
                        self._versions = {}
                        return None
                    location = os.path.dirname(os.path.abspath(location))
                    if basename in ('ffmpeg', 'ffprobe'):
                        prefer_ffmpeg = True

                self._paths = dict(
                    (p, os.path.join(location, p)) for p in programs)
                self._versions = dict(
                    (p, get_exe_version(self._paths[p], args=['-version']))
                    for p in programs)
        if self._versions is None:
            self._versions = dict(
                (p, get_exe_version(p, args=['-version'])) for p in programs)
            self._paths = dict((p, p) for p in programs)

        if prefer_ffmpeg:
            prefs = ('ffmpeg', 'avconv')
        else:
            prefs = ('avconv', 'ffmpeg')
        for p in prefs:
            if self._versions[p]:
                self.basename = p
                break

        if prefer_ffmpeg:
            prefs = ('ffprobe', 'avprobe')
        else:
            prefs = ('avprobe', 'ffprobe')
        for p in prefs:
            if self._versions[p]:
                self.probe_basename = p
                break

class FFmpegFD(ExternalFD):
    @classmethod
    def supports(cls, info_dict):
        return info_dict['protocol'] in ('http', 'https', 'ftp', 'ftps', 'm3u8', 'rtsp', 'rtmp', 'mms')

    @classmethod
    def available(cls):
        return FFmpegPostProcessorEx().available

    def meger_downloaded_file(self, tmpfilename, info_dict):
        if os.path.exists(tmpfilename):
            if os.path.exists(tmpfilename + '_new'):
                tmpFloder = os.path.abspath(os.path.join(os.path.dirname(tmpfilename), ".."))

                inputtxtFile = os.path.join(os.path.dirname(tmpfilename),'filelist.txt')
                if os.path.exists(inputtxtFile):
                    os.remove(inputtxtFile)

                inputFb = open(inputtxtFile, 'w')
                strtmp = 'file ' + '\'' + tmpfilename + '\'' + '\n'
                inputFb.writelines(strtmp)
                strtmp = 'file ' + '\'' + tmpfilename + '_new' + '\''
                inputFb.writelines(strtmp)
                inputFb.close()

                outfilePath = tmpfilename + '_tmp.mp4'

                ffpp = FFmpegPostProcessorEx(downloader=self)
                args = [ffpp.executable]
                args += ['-f', 'concat']
                #Unsafe file name '/tmp/temp/Watch Naruto Shippuden Season 17
                args += ['-safe', '-1']
                args += ['-i', inputtxtFile]
                args += ['-c', 'copy']
                args += [outfilePath]
                self._debug_cmd(args)

                proc = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, startupinfo=getStartInfo())
                subStdout, subStderr = proc.communicate()
                retval = proc.wait()
                if  retval != 0:
                    subStderr = subStderr.decode('utf-8', 'replace')
                    print(subStderr)

                outputDur = self._get_Downloaded_Time(outfilePath,info_dict)
                if  outputDur is not None:
                    os.remove(tmpfilename)
                    os.remove( tmpfilename + '_new')
                    os.rename(outfilePath,tmpfilename)
                #merger failed
                else:
                    print('ffmpeg merger file failed . delete new file')
                    os.remove( tmpfilename + '_new')
                    outputDur = self._get_Downloaded_Time(tmpfilename, info_dict)
                    if outputDur is None:
                        print('can get tmpfilename Dur. delete tmpfilename')
                        os.remove(tmpfilename)
                return outputDur
            else:
                outputDur = self._get_Downloaded_Time(tmpfilename, info_dict)
                return outputDur

        return None


    def _get_Downloaded_Time(self, tmpfilename, info_dict):
        if os.path.exists(tmpfilename):
            ffpp = FFmpegPostProcessorEx(downloader=self)
            args = [ffpp.executable]
            tmpPath = '\"' + tmpfilename + '\"'
            args += ['-i', tmpfilename]
            self._debug_cmd(args)
            proc = subprocess.Popen(args, stdout=subprocess.PIPE, stderr=subprocess.PIPE, startupinfo=getStartInfo())
            subStdout, subStderr = proc.communicate()
            retval = proc.wait()
            file_duration = None
            if  retval != 0:
                subStderr = subStderr.decode('utf-8', 'replace')
                mobj = re.search(r'Duration\:\s*((?:\d\d[.:]){3}\d\d)', subStderr)
                if mobj:
                    file_duration = mobj.group(1)
                return file_duration
        return None


    def _call_downloader(self, tmpfilename, info_dict):
        url = info_dict['url']
        ffpp = FFmpegPostProcessorEx(downloader=self)
        if not ffpp.available:
            self.report_error('m3u8 download detected but ffmpeg or avconv could not be found. Please install one.')
            return False
        ffpp.check_version()

        seekTimetamp = None
        newtmpfilename = tmpfilename
        if os.path.exists(tmpfilename):
            seekTimetamp = self.meger_downloaded_file(tmpfilename, info_dict)
            if seekTimetamp is not None:
                newtmpfilename = tmpfilename + '_new'

        args = [ffpp.executable, '-y']


        if seekTimetamp is not None:
            seekable = True
            # setting -seekable prevents ffmpeg from guessing if the server
            # supports seeking(by adding the header `Range: bytes=0-`), which
            # can cause problems in some cases
            # https://github.com/rg3/youtube-dl/issues/11800#issuecomment-275037127
            # http://trac.ffmpeg.org/ticket/6125#comment:10
            if sys.platform != 'win32':
                args += ['-seek_timestamp', '1' if seekable else '0']
        resumDownloadedSize = 0
        args += self._configuration_args()
        if seekTimetamp is not None:
            args += ['-ss', seekTimetamp]
            resumDownloadedSize = os.path.getsize(tmpfilename)
            if resumDownloadedSize < 1024:
                resumDownloadedSize = 0
            print('resumDownloadedSize : %d' % resumDownloadedSize)

        args += ['-v',  '48']
        # args += ['-multiple_requests',  '1']
        # args += ['-icy',  '0']


        if info_dict['http_headers'] and re.match(r'^https?://', url):
            # Trailing \r\n after each HTTP header is important to prevent warning from ffmpeg/avconv:
            # [http @ 00000000003d2fa0] No trailing CRLF found in HTTP header.
            headers = handle_youtubedl_headers(info_dict['http_headers']).copy()
            # print ''.join('%s: %s\r\n' % (key, val) for key, val in headers.items())
            # print len(''.join('%s: %s\r\n' % (key, val) for key, val in headers.items()))
            cookieStr = ''
            if 'Cookie' in headers:
                cookieStr = headers['Cookie']
                headers.pop('Cookie')

                domain = get_top_host(url)
                args += ['-cookies', '%s; path=/; domain=%s;' % (cookieStr, domain)]

        args += ['-headers',
                ''.join('%s: %s\r\n' % (key, val) for key, val in headers.items())]


        env = None
        proxy = self.params.get('proxy')
        if proxy:
            if not re.match(r'^[\da-zA-Z]+://', proxy):
                proxy = 'http://%s' % proxy

            if proxy.startswith('socks'):
                self.report_warning(
                    '%s does not support SOCKS proxies. Downloading is likely to fail. '
                    'Consider adding --hls-prefer-native to your command.' % self.get_basename())

            # Since December 2015 ffmpeg supports -http_proxy option (see
            # http://git.videolan.org/?p=ffmpeg.git;a=commit;h=b4eb1f29ebddd60c41a2eb39f5af701e38e0d3fd)
            # We could switch to the following code if we are able to detect version properly
            # args += ['-http_proxy', proxy]
            env = os.environ.copy()
            compat_setenv('HTTP_PROXY', proxy, env=env)
            compat_setenv('http_proxy', proxy, env=env)

        protocol = info_dict.get('protocol')

        if protocol == 'rtmp':
            player_url = info_dict.get('player_url')
            page_url = info_dict.get('page_url')
            app = info_dict.get('app')
            play_path = info_dict.get('play_path')
            tc_url = info_dict.get('tc_url')
            flash_version = info_dict.get('flash_version')
            live = info_dict.get('rtmp_live', False)
            if player_url is not None:
                args += ['-rtmp_swfverify', player_url]
            if page_url is not None:
                args += ['-rtmp_pageurl', page_url]
            if app is not None:
                args += ['-rtmp_app', app]
            if play_path is not None:
                args += ['-rtmp_playpath', play_path]
            if tc_url is not None:
                args += ['-rtmp_tcurl', tc_url]
            if flash_version is not None:
                args += ['-rtmp_flashver', flash_version]
            if live:
                args += ['-rtmp_live', 'live']


        if '#' in url:
            url = url.split('#')[0]
        args += ['-i', url, '-c', 'copy']

        if self.params.get('test', False):
            args += ['-fs', compat_str(self._TEST_FILE_SIZE)]

        if protocol in ('m3u8', 'm3u8_native'):
            if self.params.get('hls_use_mpegts', False) or tmpfilename == '-':
                args += ['-f', 'mpegts']
            else:
                args += ['-f', 'mp4']
                if (ffpp.basename == 'ffmpeg' and is_outdated_version(ffpp._versions['ffmpeg'], '3.2', False)) and (not info_dict.get('acodec') or info_dict['acodec'].split('.')[0] in ('aac', 'mp4a')):
                    args += ['-bsf:a', 'aac_adtstoasc']
        elif protocol == 'rtmp':
            args += ['-f', 'flv']
        else:
            args += ['-f', EXT_TO_OUT_FORMATS.get(info_dict['ext'], info_dict['ext'])]

        args = [encodeArgument(opt) for opt in args]
        args.append(encodeFilename(ffpp._ffmpeg_filename_argument(newtmpfilename), True))

        self._debug_cmd(args)
        proc = subprocess.Popen(args, stdin=subprocess.PIPE, stderr= subprocess.PIPE, env=env, startupinfo=getStartInfo())
        totoalTime = ''
        try:
            start = time.time()
            proc_stderr_closed = False
            while not proc_stderr_closed:
                # read line from stderr
                line = ''
                while True:
                    char = proc.stderr.read(1)
                    if not char:
                        proc_stderr_closed = True
                        break
                    if char in [b'\r', b'\n']:
                        break
                    try:
                        line += char.decode('ascii', 'replace')
                    except Exception as ex:
                        continue

                if not line:
                    # proc_stderr_closed is True
                    continue

                if isinstance(line, bytes):
                    line = line.decode('utf-8')
                # print(line)
                mobj = re.search(r'Duration\:\s*((?:\d\d[.:]){3}\d\d)', line)
                if mobj:
                    totoalTime = mobj.group(1)
                mobj = re.search(r'size=\s*(\d+)kB\s*?time\=\s*((?:\d\d[.:]){3}\d\d)\s*bitrate\=\s*(.*)kbits', line)
                if mobj:
                    downloadedSize = mobj.group(1)
                    downloadedTime = mobj.group(2)
                    speed = mobj.group(3)

                    downloaded_data_len = int(float(downloadedSize) * 1024)
                    downloaded_total_data_len = downloaded_data_len + resumDownloadedSize
                    total_bytes = 1024 * 1024 * 1024 * 2
                    percent = float(downloaded_data_len / total_bytes)
                    percent = 0.9 if percent > 1 else percent
                    time_now = time.time()
                    #eta = self.calc_eta(start, time_now, 100 - resume_percent, percent - resume_percent)
                    #speed = self.calc_speed(start, time_now, downloaded_data_len - resume_downloaded_data_len)
                    data_len = None
                    if percent > 0:
                        data_len = int(downloaded_data_len * 100 / percent)


                    try:
                        self._hook_progress({
                            'status': 'downloading',
                            'downloaded_bytes': downloaded_total_data_len,
                            'total_bytes_estimate': data_len,
                            'tmpfilename': tmpfilename,
                            #'filename': filename,
                            #'eta': eta,
                            'elapsed': time_now - start,
                            'speed': self.calc_speed(start, time_now, downloaded_data_len),
                            'total_bytes': total_bytes
                        })
                    except Exception:
                        raise KeyboardInterrupt

                    cursor_in_new_line = False
                # else:
                #     # no percent for live streams
                #     mobj = re.search(r'([0-9]+\.[0-9]{3}) kB / [0-9]+\.[0-9]{2} sec', line)
                #     if mobj:
                #         downloaded_data_len = int(float(mobj.group(1)) * 1024)
                #         time_now = time.time()
                #         speed = self.calc_speed(start, time_now, downloaded_data_len)
                #         self._hook_progress({
                #             'downloaded_bytes': downloaded_data_len,
                #             'tmpfilename': tmpfilename,
                #             #'filename': filename,
                #             'status': 'downloading',
                #             'elapsed': time_now - start,
                #             'speed': speed,
                #         })
                #         cursor_in_new_line = False
                #     elif self.params.get('verbose', False):
                #         if not cursor_in_new_line:
                #             self.to_screen('')
                #         cursor_in_new_line = True
                #         self.to_screen('[rtmpdump] ' + line)
            retval = proc.wait()
            self.meger_downloaded_file(tmpfilename, info_dict)
        except KeyboardInterrupt:
            # subprocces.run would send the SIGKILL signal to ffmpeg and the
            # mp4 file couldn't be played, but if we ask ffmpeg to quit it
            # produces a file that is playable (this is mostly useful for live
            # streams). Note that Windows is not affected and produces playable
            # files (see https://github.com/rg3/youtube-dl/issues/8300).
            print('ffmpeg download cancle ...........')
            try:
                if sys.platform != 'win32':
                    proc.communicate(b'q')
                else:
                    proc.terminate()
            except:
                print('ffmpeg download cancle except...........')
                print(traceback.format_exc())

            pass
        print('ffmpeg download cancle Finish...........')
        return retval