#if UNITY_EDITOR
using UnityEngine;
using UnityEditor;
using System.IO;
using System.Collections.Generic;
using UnityEngine.SceneManagement;

public class LoadLevelToScene {
    static GameObject blockP;
    static GameObject refBlockP;
    static GameObject finishP;
    static GameObject playerP;
    static GameObject targetP;
    static GameObject wallP;
    static GameObject portalP;
    static GameObject infoP;

    static bool header;
    static string attemptOrder;
    static bool showAttemptOrder;
    static bool shedEnabled;
    static bool innerPushEnabled;
    static Stack<SceneBlock> blockStack;
    static Dictionary<int, SceneLevel> levelIDs;
    static LevelProperties levelProperties;

    public static void Load() {
        LoadFile("temp_level.txt");
    }

    public static void LoadFile(string filename) {
        StreamReader reader = new StreamReader(filename);
        string data = reader.ReadToEnd();
        reader.Close();
        Load(data);
    }

    public static void Load(string data) {
        if (SceneManager.GetActiveScene().name == "hub") {
            Debug.Log("in hub. not loading from temp file to prevent overwriting hub.");
            return;
        }

        // Destroy any existing root level
        var root = GameObject.Find("RootLevel");
        if (root != null) {
            Undo.DestroyObjectImmediate(root);
        }

        // Load prefabs
        blockP = LoadPrefab("Level/Block");
        refBlockP = LoadPrefab("Level/BlockRef");
        finishP = LoadPrefab("Level/Finish");
        playerP = LoadPrefab("Level/Player");
        targetP = LoadPrefab("Level/Target");
        wallP = LoadPrefab("Level/Wall");
        portalP = LoadPrefab("Hub/LevelSelectLevel");
        infoP = LoadPrefab("Hub/Info");

        // Initialize stuff
        header = true;
        blockStack = new Stack<SceneBlock>();
        levelIDs = new Dictionary<int, SceneLevel>();

        // Set default level properties
        if (GameObject.Find("LevelProperties") != null) {
            Undo.DestroyObjectImmediate(GameObject.Find("LevelProperties"));
        }
        levelProperties = CreatePrefab(LoadPrefab("LevelProperties")).GetComponent<LevelProperties>();

        // Read lines
        string[] lines = data.Split('\n');
        foreach (string line in lines) {
            ParseLine(line);
        }

        // Post processing stuff
        AssignLevelRefs(GameObject.Find("RootLevel"));
        SquareEditor.SnapAllBlocks();

        Debug.Log("loaded txt to scene");
    }

    static void ParseLine(string line) {
        if (line.Length == 0) return;

        if (header) {
            line = line.Replace("\t", "").Replace("\n", "").Replace("\r", "");

            if (line == "#") {
                header = false;
                return;
            }

            string[] hargs = line.Split(' ');
            if (hargs[0] == "version") {
                if (hargs[1] != "4") {
                    Debug.LogError("Unsupported level version "+hargs[1]);
                }
            }
            else if (hargs[0] == "attempt_order") {
                levelProperties.AttemptOrder = hargs[1];
            }
            else if (hargs[0] == "shed") {
                levelProperties.ShedEnabled = Util.ParseBool(hargs[1]);
            }
            else if (hargs[0] == "inner_push") {
                levelProperties.InnerPushEnabled = Util.ParseBool(hargs[1]);
            }
            else if (hargs[0] == "draw_style") {
                if (hargs[1] == "tui") {
                    levelProperties.DrawStyle = LevelProperties.DS.tui;
                }
                else if (hargs[1] == "grid") {
                    levelProperties.DrawStyle = LevelProperties.DS.grid;
                }
                else if (hargs[1] == "oldstyle") {
                    levelProperties.DrawStyle = LevelProperties.DS.oldstyle;
                }
            }
            else if (hargs[0] == "custom_level_music") {
                levelProperties.CustomLevelMusic = int.Parse(hargs[1]);
            }
            else if (hargs[0] == "custom_level_palette") {
                levelProperties.CustomLevelPalette = int.Parse(hargs[1]);
            }
            return;
        }

        // Count the number of leading tabs
        int depth = 0;
        for (int j = 0; j < line.Length; j++) {
            if (line[j] == '\t') depth++;
            else break;
        }

        // Detect when we're going down an indent level
        while (depth < blockStack.Count) {
            blockStack.Pop();
        }

        string[] args = line.Replace("\t", "").Split(' ');
        int i = 1;

        if (args[0] == "Block") {
            // Parse properties
            int x = int.Parse(args[i++]);
            int y = int.Parse(args[i++]);
            int id = int.Parse(args[i++]);

            // Level properties
            int width = int.Parse(args[i++]);
            int height = int.Parse(args[i++]);
            float hue = Util.ParseFloat(args[i++]);
            float sat = Util.ParseFloat(args[i++]);
            float val = Util.ParseFloat(args[i++]);
            float camZoomFactor = Util.ParseFloat(args[i++]);
            bool fillWithWalls = Util.ParseBool(args[i++]);

            // Block properties
            bool isPlayer = Util.ParseBool(args[i++]);
            bool possessable = Util.ParseBool(args[i++]);
            int playerOrder = int.Parse(args[i++]);
            bool flipH = Util.ParseBool(args[i++]);

            // Other block properties
            bool floatInSpace = Util.ParseBool(args[i++]);
            int specialEffect = int.Parse(args[i++]);

            // Instantiate prefab
            GameObject block;
            if (isPlayer) block = CreatePrefab(playerP);
            else block = CreatePrefab(blockP);
            var level = block.GetComponent<SceneLevel>();
            var b = block.GetComponent<SceneBlock>();

            // Set properties
            if (blockStack.Count == 0) {
                block.name = "RootLevel";
                //b.AttemptOrder = attemptOrder;
                //b.ShedEnabled = shedEnabled;
                //b.InnerPushEnabled = innerPushEnabled;
            }
            else {
                SetPosition(block, x, y);
            }

            level.width = width;
            level.height = height;
            level.CamZoomFactor = camZoomFactor;
            level.hue = hue;
            level.sat = sat;
            level.val = val;
            level.FillWithWalls = fillWithWalls;

            b.IsPlayer = isPlayer;
            b.Possessable = possessable;
            b.PlayerOrder = playerOrder;
            b.FlipH = flipH;
            b.FloatInSpace = floatInSpace;
            b.SpecialEffect = specialEffect;

            levelIDs[id] = level;
            blockStack.Push(b);
        }

        else if (args[0] == "Wall") {
            int x = int.Parse(args[i++]);
            int y = int.Parse(args[i++]);

            bool isPlayer = Util.ParseBool(args[i++]);
            bool possessable = Util.ParseBool(args[i++]);
            int playerOrder = int.Parse(args[i++]);

            //string unlockerScene = args[i++];

            var wall = CreatePrefab(wallP);
            var b = wall.GetComponent<SceneBlock>();

            SetPosition(wall, x, y);
            b.IsPlayer = isPlayer;
            b.Possessable = possessable;
            b.PlayerOrder = playerOrder;
            //b.unlockerScene = unlockerScene;
        }

        else if (args[0] == "Floor") {
            int x = int.Parse(args[i++]);
            int y = int.Parse(args[i++]);
            string floorType = args[i++];

            if (floorType == "Button") {
                var target = CreatePrefab(targetP);
                SetPosition(target, x, y);
            }
            else if (floorType == "PlayerButton") {
                var finish = CreatePrefab(finishP);
                SetPosition(finish, x, y);
            }
            else if (floorType == "LevelSelectLevel") {
                var portal = CreatePrefab(portalP).GetComponent<SceneFloor>();
                SetPosition(portal.gameObject, x, y);
                portal.SceneName = args[i++];
                portal.referenceName = args[i++].Replace("_", " ");
                SceneFloor.AssignScene(portal, portal.SceneName);
            }
            else if (floorType == "Info") {
                var info = CreatePrefab(infoP).GetComponent<SceneFloor>();
                SetPosition(info.gameObject, x, y);
                info.info = args[i++].Replace("_", " ");
            }
            else {
                Debug.LogWarning("Unrecognized floor type: "+floorType);
            }
        }

        else if (args[0] == "Ref") {
            // Parse properties
            int x = int.Parse(args[i++]);
            int y = int.Parse(args[i++]);

            // ID
            int id = int.Parse(args[i++]);

            // Inf
            bool isExitBlock = Util.ParseBool(args[i++]);
            bool isInfExitBlock = Util.ParseBool(args[i++]);
            int infExitNum = int.Parse(args[i++]);
            bool isInfEnterBlock = Util.ParseBool(args[i++]);
            int infEnterNum = int.Parse(args[i++]);
            int infEnterLevelID = int.Parse(args[i++]);

            // Block properties
            bool isPlayer = Util.ParseBool(args[i++]);
            bool possessable = Util.ParseBool(args[i++]);
            int playerOrder = int.Parse(args[i++]);
            bool flipH = Util.ParseBool(args[i++]);

            // Other block properties
            bool floatInSpace = Util.ParseBool(args[i++]);
            int specialEffect = int.Parse(args[i++]);

            // Hub
            //string areaName = args[i++];

            var refBlock = CreatePrefab(refBlockP);
            var r = refBlock.GetComponent<RefBlock>();

            SetPosition(refBlock, x, y);
            r.levelID = id;
            r.IsExitBlock = isExitBlock;
            r.IsInfExitBlock = isInfExitBlock;
            r.InfExitNum = infExitNum;
            r.IsInfEnterBlock = isInfEnterBlock;
            r.infEnterLevelID = infEnterLevelID;
            r.InfEnterNum = infEnterNum;
            r.FlipH = flipH;
            r.FloatInSpace = floatInSpace;
            r.Possessable = possessable;
            r.IsPlayer = isPlayer;
            r.PlayerOrder = playerOrder;
            r.SpecialEffect = specialEffect;
            //r.HubShortcutAreaName = areaName;
        }
    }

    static void AssignLevelRefs(GameObject obj) {
        foreach (Transform child in obj.transform) {
            var childBlock = child.GetComponent<SceneBlock>();
            var refBlock = child.GetComponent<RefBlock>();

            // Recurse into block children
            if (childBlock != null && child.GetComponent<SceneLevel>() != null) {
                AssignLevelRefs(child.gameObject);
            }
            // Assign level to ref block
            else if (refBlock != null) {
                refBlock.SubSceneLevel = levelIDs[refBlock.levelID];
                if (refBlock.IsInfEnterBlock) {
                    refBlock.InfEnterLevel = levelIDs[refBlock.infEnterLevelID];
                }
            }
        }
    }

    static GameObject LoadPrefab(string name) {
        string path = "Assets/Prefabs/" + name + ".prefab";
        return (GameObject)AssetDatabase.LoadAssetAtPath(path, typeof(GameObject));
    }

    static GameObject CreatePrefab(GameObject prefab) {
        var obj = ((GameObject)PrefabUtility.InstantiatePrefab(prefab));
        Undo.RegisterCreatedObjectUndo(obj, "LoadLevelToScene");
        return obj;
    }

    static void SetPosition(GameObject obj, int x, int y) {
        var parent = blockStack.Peek();
        obj.transform.parent = parent.transform;
        obj.transform.localPosition =
                parent.GetComponent<SceneLevel>().LocalPosition(x, y);
    }
}
#endif
