// Based on code from Brett Taylor

#if UNITY_EDITOR
using UnityEngine;
using System.Collections.Generic;
using UnityEditor;

[ExecuteAlways]
public class Snap : MonoBehaviour {
    Vector3 lastPos;
    Vector3 lastScale;
    public int depth;

    void Update() {
        if (transform.localPosition != lastPos || transform.localScale != lastScale) {
            SnapSelf();

            lastPos = transform.localPosition;
            lastScale = transform.localScale;
        }
    }

    public void SnapSelf() {
        var sprite = GetComponent<SpriteRenderer>();

        if (transform.parent == null) {
            transform.localPosition = Vector3.zero;
            transform.localScale = new Vector3(10, 10, 1);
        }
        else {
            var level = transform.parent.GetComponent<SceneLevel>();
            int x = level.LevelX(transform.localPosition.x);
            int y = level.LevelY(transform.localPosition.y);

            if (!level.InBounds(x, y)) {
                //Debug.LogWarningFormat("Object out of bounds at ({0},{1})", x, y);
            }

            transform.localPosition = level.LocalPosition(x, y);

            transform.rotation = Quaternion.identity;

            float scale = level.ChildLocalScale();

            var floor = GetComponent<SceneFloor>();
            if (floor != null && floor.type == SceneFloor.FloorType.LevelSelectLevel) {
                if (sprite.sprite != null) scale *= 100f / sprite.sprite.rect.size.x;
                name = floor.SceneName;
                if (floor.SceneName == "") name = "LevelPortal";
            }

            transform.localScale = new Vector3(scale, scale);
        }

        depth = 0;
        Transform t = transform;
        while (t.parent != null) {
            t = t.parent;
            depth++;
        }

        #if UNITY_EDITOR
        Undo.RecordObject(sprite, "asdf");
        #endif
        sprite.sortingOrder = depth;

        var b = GetComponent<SceneBlock>();
        if (b != null) {
            b.UpdateColor();
        }

        var r = GetComponent<RefBlock>();
        if (r != null) {
            r.UpdateColor();

            if (r.SubSceneLevel != null) {
                Undo.RecordObject(r, "asdf");
                PrefabUtility.RecordPrefabInstancePropertyModifications(r);
                r.HubShortcutAreaPrefix = r.SubSceneLevel.GetComponent<SceneBlock>().areaPrefix;
            }
        }
    }

    public static void SnapLevelChildren(GameObject obj, bool recurse) {
        SceneLevel level = obj.GetComponent<SceneLevel>();

        var blocks = new SceneBlock[level.width, level.height];
        var floors = new SceneFloor[level.width, level.height];
        var refs   = new RefBlock  [level.width, level.height];

        foreach (Transform child in obj.transform) {
            var block = child.GetComponent<SceneBlock>();
            var refBlock = child.GetComponent<RefBlock>();
            var floor = child.GetComponent<SceneFloor>();
            var childLevel = child.GetComponent<SceneLevel>();
            var snap = child.GetComponent<Snap>();

            int x = level.LevelX(child.localPosition.x);
            int y = level.LevelY(child.localPosition.y);

            if (!level.InBounds(x, y)) {
                Debug.LogWarningFormat("Object out of bounds at ({0},{1})", x, y);
                continue;
            }

            // Check if there's another object of this type already at that spot
            if (block != null) {
                if (blocks[x, y] != null) {
                    Debug.LogWarningFormat("Multiple blocks at ({0},{1}) {2}", x, y, obj.name);
                } else {
                    blocks[x, y] = block;
                }
            }

            if (floor != null) {
                if (floors[x, y] != null) {
                    Debug.LogWarningFormat("Multiple floors at ({0},{1}) {2}", x, y, obj.name);
                }
                else {
                    floors[x, y] = floor;
                }
            }

            if (refBlock != null) {
                if (refs[x, y] != null) {
                    Debug.LogWarningFormat("Multiple floors at ({0},{1}) {2}", x, y, obj.name);
                }
                else {
                    refs[x, y] = refBlock;
                }
            }
            // null check because this happens when undoing/redoing after creating walled block??
            // and also fixes weird behavior with blocks turning green then too.
            if (snap != null) {
                snap.SnapSelf();
            }

            if (recurse && childLevel != null) {
                SnapLevelChildren(child.gameObject, true);
            }
        }

        // Reorder walls to be at the bottom of the hierarchy
        /*
        foreach (Transform child in obj.transform) {
            var block = child.GetComponent<SceneBlock>();
            if (block != null && child.GetComponent<SceneLevel>() == null) {
                child.SetSiblingIndex(child.parent.childCount - 1);
                child.name = "Wall";
            }
        }
        */
    }
}
#endif
