﻿#if UNITY_EDITOR
using UnityEngine;
using UnityEngine.SceneManagement;
using UnityEditor;
using UnityEditor.SceneManagement;
using System.IO;
using System.Collections.Generic;

public partial class SquareEditor : Editor {
    [MenuItem("Square/Hotkeys/Snap all blocks &_a")]
    public static void SnapAllBlocks() {
        GameObject rootObj = GameObject.Find("RootLevel");
        if (rootObj == null) {
            Debug.LogWarning("Couldn't find root level");
            return;
        }

        rootObj.GetComponent<Snap>().SnapSelf();
        Snap.SnapLevelChildren(rootObj, true);
    }

    [MenuItem("Square/Hotkeys/Increment size and anchor &_i")]
    static void SnapIncreaseSize() {
        Transform parent = Selection.activeTransform;
        SceneLevel level = parent.gameObject.GetComponent<SceneLevel>();
        
        float newBlocksize = 1f / Mathf.Max(level.width + 1, level.height + 1);

        foreach (Transform child in parent) {
            Undo.RecordObject(child, "Snap increase size");
            int x = level.LevelX(child.localPosition.x);
            int y = level.LevelY(child.localPosition.y);
            child.localPosition = level.LocalPositionGeneral(x, y, level.width + 1, level.height + 1, newBlocksize);
            child.localScale = new Vector3(newBlocksize, newBlocksize);
        }

        Undo.RecordObject(level, "Snap increase size");
        level.width += 1;
        level.height += 1;
        PrefabUtility.RecordPrefabInstancePropertyModifications(level);
    }

    [MenuItem("Square/Hotkeys/Decrement size and anchor &_d")]
    static void SnapDecreaseSize() {
        Transform parent = Selection.activeTransform;
        SceneLevel level = parent.gameObject.GetComponent<SceneLevel>();
        
        float newBlocksize = 1f / Mathf.Max(level.width - 1, level.height - 1);

        foreach (Transform child in parent) {
            Undo.RecordObject(child, "Snap increase size");
            int x = level.LevelX(child.localPosition.x);
            int y = level.LevelY(child.localPosition.y);
            child.localPosition = level.LocalPositionGeneral(x, y, level.width - 1, level.height - 1, newBlocksize);
            child.localScale = new Vector3(newBlocksize, newBlocksize);
        }

        Undo.RecordObject(level, "Snap increase size");
        level.width -= 1;
        level.height -= 1;
        PrefabUtility.RecordPrefabInstancePropertyModifications(level);
    }

    [MenuItem("Square/Hotkeys/Mirror Vertical &_v")]
    static void MirrorVertical() {
        Transform parent = Selection.activeTransform;
        SceneLevel level = parent.gameObject.GetComponent<SceneLevel>();

        foreach (Transform child in parent) {
            Undo.RecordObject(child, "Mirror");
            int x = level.LevelX(child.localPosition.x);
            int y = level.LevelY(child.localPosition.y);
            child.localPosition = level.LocalPosition(x, level.height - 1 - y);
        }
    }

    [MenuItem("Square/Hotkeys/Mirror Horizontal &_h")]
    static void MirrorHorizontal() {
        Transform parent = Selection.activeTransform;
        SceneLevel level = parent.gameObject.GetComponent<SceneLevel>();

        foreach (Transform child in parent) {
            Undo.RecordObject(child, "Mirror");
            int x = level.LevelX(child.localPosition.x);
            int y = level.LevelY(child.localPosition.y);
            child.localPosition = level.LocalPosition(level.width - 1 - x, y);
        }
    }

    [MenuItem("Square/Hotkeys/Mirror across diagonal &_m")]
    static void MirrorDiagonal() {
        Transform parent = Selection.activeTransform;
        SceneLevel level = parent.gameObject.GetComponent<SceneLevel>();

        foreach (Transform child in parent) {
            Undo.RecordObject(child, "Mirror");
            int x = level.LevelX(child.localPosition.x);
            int y = level.LevelY(child.localPosition.y);
            child.localPosition = level.LocalPosition(y, x);
        }
    }

    [MenuItem("Square/Hotkeys/Enter &_n")]
    static void Enter() {
        if (Selection.gameObjects.Length == 0) return;

        var level = Selection.activeTransform.parent.gameObject.GetComponent<SceneLevel>();

        // Find the x and y position of the upper-left-most portal
        int minX = -1;
        int maxY = -1;
        GameObject handle = null;
        foreach (GameObject obj in Selection.gameObjects) {
            int x = level.LevelX(obj.transform.localPosition.x);
            int y = level.LevelY(obj.transform.localPosition.y);

            if (minX == -1) {
                minX = x;
                maxY = y;
                handle = obj;
            }
            else if (y > maxY) {
                maxY = y;
                minX = x;
                handle = obj;
            }
            else if (x < minX && y >= maxY) {
                minX = x;
                maxY = y;
                handle = obj;
            }
        }

        int handleX = minX;
        int handleY = maxY;

        // Find the other block at this spot (besides us)
        Transform toEnter = null;
        foreach (Transform t in Selection.activeTransform.parent) {
            if (t.gameObject == handle) continue;
            int x = level.LevelX(t.localPosition.x);
            int y = level.LevelY(t.localPosition.y);
            if (x == handleX && y == handleY) {
                toEnter = t;
                break;
            }
        }
        var toLevel = toEnter.GetComponent<SceneLevel>();

        // Move the portals inside that block with the appropriate offsets
        foreach (GameObject obj in Selection.gameObjects) {
            int x = level.LevelX(obj.transform.localPosition.x);
            int y = level.LevelY(obj.transform.localPosition.y);
            int toX = (toLevel.width - 1)/2 + (x - handleX);
            int toY = (toLevel.height - 1)/2 + (y - handleY);

            Undo.SetTransformParent(obj.transform, toEnter, "Enter");
            Undo.RecordObject(obj.transform, "Enter");
            obj.transform.localPosition = toLevel.LocalPosition(toX, toY);
        }
    }

    [MenuItem("Square/Hotkeys/Exit &_x")]
    static void Exit() {
        if (Selection.gameObjects.Length == 0) return;

        var level = Selection.activeTransform.parent.gameObject.GetComponent<SceneLevel>();

        // Find the x and y position of the upper-left-most portal
        int minX = -1;
        int maxY = -1;
        foreach (GameObject obj in Selection.gameObjects) {
            int x = level.LevelX(obj.transform.localPosition.x);
            int y = level.LevelY(obj.transform.localPosition.y);

            if (minX == -1) {
                minX = x;
                maxY = y;
            }
            else if (y > maxY) {
                maxY = y;
                minX = x;
            }
            else if (x < minX && y >= maxY) {
                minX = x;
                maxY = y;
            }
        }

        int handleX = minX;
        int handleY = maxY;

        var toTransform = Selection.activeTransform.parent.parent;
        var toLevel = toTransform.gameObject.GetComponent<SceneLevel>();
        int parentX = toLevel.LevelX(Selection.activeTransform.parent.localPosition.x);
        int parentY = toLevel.LevelY(Selection.activeTransform.parent.localPosition.y);

        // Move the portals inside that block with the appropriate offsets
        foreach (GameObject obj in Selection.gameObjects) {
            int x = level.LevelX(obj.transform.localPosition.x);
            int y = level.LevelY(obj.transform.localPosition.y);
            int toX = parentX + (x - handleX);
            int toY = parentY + (y - handleY);

            Undo.SetTransformParent(obj.transform, toTransform, "Enter");
            Undo.RecordObject(obj.transform, "Enter");
            obj.transform.localPosition = toLevel.LocalPosition(toX, toY);
        }
    }

    [MenuItem("Square/Hotkeys/Focus on parent &_t")]
    static void FocusParent() {
        Selection.activeTransform = Selection.activeTransform.parent;
    }

    [MenuItem("Square/Hotkeys/Move object up #_W")]
    static void MoveBlockUp() {
        MoveBlock(0, 1);
    }
    [MenuItem("Square/Hotkeys/Move object down #_S")]
    static void MoveBlockDown() {
        MoveBlock(0, -1);
    }
    [MenuItem("Square/Hotkeys/Move object left #_A")]
    static void MoveBlockLeft() {
        MoveBlock(-1, 0);
    }
    [MenuItem("Square/Hotkeys/Move object right #_D")]
    static void MoveBlockRight() {
        MoveBlock(1, 0);
    }

    static void MoveBlock(int dx, int dy) {
        // Only shift the top-most blocks
        int minDepth = -1;
        foreach (GameObject focusObj in Selection.gameObjects) {
            int depth = focusObj.GetComponent<Snap>().depth;
            if (depth < minDepth || minDepth == -1) {
                minDepth = depth;
            }
        }

        foreach (GameObject focusObj in Selection.gameObjects) {
            if (focusObj.GetComponent<Snap>().depth > minDepth) continue;

            SceneLevel level = focusObj.transform.parent.GetComponent<SceneLevel>();
            int x = level.LevelX(focusObj.transform.localPosition.x);
            int y = level.LevelY(focusObj.transform.localPosition.y);

            Undo.RecordObject(focusObj.transform, "Move");
            focusObj.transform.localPosition = level.LocalPosition(x + dx, y + dy);
        }
    }

    [MenuItem("Square/Hotkeys/Instantiate prefab to root &_e")]
    static void InstantiatePrefabToRoot() {
        Object[] selected = Selection.GetFiltered(typeof(Object), SelectionMode.DeepAssets);

        if (selected.Length > 1) {
            Debug.Log("too many selected");
            return;
        }

        if (selected.Length == 0) {
            Debug.Log("none selected");
            return;
        }

        string path = AssetDatabase.GetAssetPath(selected[0]);
        if (System.IO.Path.GetExtension(path) != ".prefab") {
            Debug.Log("Non-prefab selected");
            return;
        }
        GameObject prefab = (GameObject)AssetDatabase.LoadAssetAtPath(
                path, typeof(GameObject));

        GameObject rootObj = GameObject.Find("RootLevel");
        GameObject obj = PrefabUtility.InstantiatePrefab(prefab) as GameObject;
        Undo.RegisterCreatedObjectUndo(obj, "asdf");
        Undo.SetTransformParent(obj.transform, rootObj.transform, "InstantiatePrefabToRoot");
        Selection.activeTransform = obj.transform;

        if (path == "Assets/Prefabs/Level/Block.prefab") {
            var level = obj.GetComponent<SceneLevel>();
            Undo.RecordObject(level, "block instantiate set hsv");
            level.hue = 0.6f;
            level.sat = 0.8f;
            level.val = 1f;
            PrefabUtility.RecordPrefabInstancePropertyModifications(level);
        }

        obj.GetComponent<Snap>().SnapSelf();
    }

    [MenuItem("Square/Instantiate 3x3 walled block")]
    static void Instantiate3x3WalledBlock() {
        CreateWalledBlock(3, 3);
    }

    [MenuItem("Square/Instantiate 5x5 walled block")]
    static void Instantiate5x5WalledBlock() {
        CreateWalledBlock(5, 5);
    }

    [MenuItem("Square/Instantiate 7x7 walled block")]
    static void Instantiate7x7WalledBlock() {
        CreateWalledBlock(7, 7);
    }

    [MenuItem("Square/Instantiate 9x9 walled block")]
    static void Instantiate9x9WalledBlock() {
        CreateWalledBlock(9, 9);
    }

    static void CreateWalledBlock(int width, int height) {
        GameObject blockPrefab = (GameObject)AssetDatabase.LoadAssetAtPath(
                "Assets/Prefabs/Level/Block.prefab", typeof(GameObject));
        GameObject wallPrefab = (GameObject)AssetDatabase.LoadAssetAtPath(
                "Assets/Prefabs/Level/Wall.prefab", typeof(GameObject));

        // Create block
        GameObject rootObj = GameObject.Find("RootLevel");
        GameObject block = PrefabUtility.InstantiatePrefab(blockPrefab) as GameObject;
        Undo.RegisterCreatedObjectUndo(block, "asdf");
        Undo.SetTransformParent(block.transform, rootObj.transform, "asdf");
        Selection.activeTransform = block.transform;
        SceneLevel level = block.GetComponent<SceneLevel>();
        level.width = width;
        level.height = height;
        level.hue = 0.1f;
        level.sat = 0.8f;
        level.val = 1;

        // Create walls
        for (int x = 0; x < width; x++) {
            for (int y = 0; y < height; y++) {
                if (!(x == 0 || y == 0 || x == width-1 || y == height-1)) {
                    continue;
                }
                GameObject wall = PrefabUtility.InstantiatePrefab(wallPrefab) as GameObject;
                Undo.RegisterCreatedObjectUndo(wall, "asdf");
                Undo.SetTransformParent(wall.transform, block.transform, "InstantiatePrefabToRoot");
                Undo.RecordObject(wall, "Create walled block");
                wall.transform.localPosition = level.LocalPosition(x, y);
            }
        }

        block.GetComponent<Snap>().SnapSelf();
        Snap.SnapLevelChildren(block, true);

        PrefabUtility.RecordPrefabInstancePropertyModifications(level);
    }

    [MenuItem("Square/Hotkeys/Assign ref &_1")]
    static void AssignRef() {
        if (Selection.gameObjects.Length != 2) {
            Debug.Log("Two objects were not selected");
            return;
        }

        // Ensure obj1 is a block and obj2 is a refblock
        var obj1 = Selection.gameObjects[0];
        var obj2 = Selection.gameObjects[1];
        if (obj2.GetComponent<SceneBlock>() != null) {
            var tmp = obj1;
            obj1 = obj2;
            obj2 = tmp;
        }

        // Assign the level
        var refBlock = obj2.GetComponent<RefBlock>();
        Undo.RecordObject(refBlock, "Assign ref");
        refBlock.SubSceneLevel = obj1.GetComponent<SceneLevel>();
        refBlock.UpdateColor();
        PrefabUtility.RecordPrefabInstancePropertyModifications(refBlock);
    }

    [MenuItem("Square/Hotkeys/Assign inf enter level &_2")]
    static void AssignInfEnterLevel() {
        if (Selection.gameObjects.Length != 2) {
            Debug.Log("Two objects were not selected");
            return;
        }

        // Ensure obj1 is a block and obj2 is a refblock
        var obj1 = Selection.gameObjects[0];
        var obj2 = Selection.gameObjects[1];
        if (obj2.GetComponent<SceneBlock>() != null) {
            var tmp = obj1;
            obj1 = obj2;
            obj2 = tmp;
        }

        // Assign the inf enter level
        var refBlock = obj2.GetComponent<RefBlock>();
        Undo.RecordObject(refBlock, "Assign ref");
        refBlock.InfEnterLevel = obj1.GetComponent<SceneLevel>();
        refBlock.IsInfEnterBlock = true;
        PrefabUtility.RecordPrefabInstancePropertyModifications(refBlock);
    }

    //[MenuItem("Square/Hotkeys/Load from txt &_F7")]
    //static void LoadFromTxt() {
    //    LoadLevelToScene.Load();
    //}

}

// When saving a scene, save the txt version too
public class MyAssetModificationProcessor : UnityEditor.AssetModificationProcessor {
    public static string[] OnWillSaveAssets(string[] paths) {
        foreach (string path in paths) {
            string sceneName = Path.GetFileNameWithoutExtension(path);
            if (path.Contains(".unity")) {

                /*if (sceneName == "hub") { // || sceneName == "hub_demo") {
                    SquareEditor.HubMasterSave();
                }
                else if (sceneName == "palettes") {
                    SquareEditor.SavePalettes();
                }
                else if (path.Contains("passable")) {
                    SaveSceneAsLevel.Save("Assets/Resources/levels/"+sceneName+".txt");
                }*/

                var txtPath = Path.Combine(SquareEditor.ExportDir, SceneManager.GetActiveScene().name+".txt");
                SaveSceneAsLevel.Save(txtPath);
                Debug.Log("Saved to "+txtPath);
            }
        }
        return paths;
    }
}
#endif
