<#
    .SYNOPSIS
        Provides functions to manage the network setup for centennial converter tool.
    .DESCRIPTION
        Uses NetNat to provide Nat based networking to the isolation environments. Manages a virtual switch to provide a private subnet which is then NATed
        using NetNat over to host external adapters.
#>
#
# Imports
#
$_myLocation = (Split-Path (Get-Variable MyInvocation -Scope 0).Value.mycommand.path)
. (Join-Path $_myLocation "CWACLogger.ps1")

$defaultSubnetPrefix = '172.16.240.0/24'
#
# Functions
#

<#
    .SYNOPSIS
        Gets or creates an instance of network for centennial converter tool.
    .DESCRIPTION
        Gets or creates an instance of NATed network which is used to provide Nat based networking to the isolation environments. 
        Returns Id for the instance.
    .PARAMETER NetworkName
        This is an optional parameter. You can pass in a name for the network instance.
    .PARAMETER NatSubnetPrefix
        This is an optional parameter. Prefix value to be used for the NetNat instance. Typically you would want to change this only if your host
        machine is attached to the same subnet range as the Windows Desktop App Converter NetNat. You can query the current Windows Desktop App Converter NetNat config using
        Get-NetNat cmdlet.
    .PARAMETER Logger
        Logger object used for logging.
    .EXAMPLE
        GetOrCreateCWACNetwork
        GetOrCreateCWACNetwork -NatSubnetPrefix 192.168.100.0/24
#>
function
GetOrCreateCWACNetwork
{
    [CmdletBinding()]
    param(
        [Parameter(Mandatory=$False)]
        [string]
        [ValidateNotNullOrEmpty()]
        $NetworkName = 'Windows Desktop App Converter Network',

        [Parameter(Mandatory=$False)]
        [string]
        [AllowEmptyString()]
        $NatSubnetPrefix = $defaultSubnetPrefix,

        [Parameter(Mandatory=$True)]
        [CWACLogger]
        [ValidateNotNull()]
        $Logger
    )

    if ([string]::IsNullOrEmpty($NatSubnetPrefix))
    {
        $NatSubnetPrefix = $defaultSubnetPrefix
    }
    
    $networkId = GetCWACNetwork -NetworkName $NetworkName -NatSubnetPrefix $NatSubnetPrefix -Logger $Logger
    
    if ($networkId -eq $null)
    {
        $networkId = NewCWACNetwork -NetworkName $NetworkName -NatSubnetPrefix $NatSubnetPrefix -Logger $Logger
    }
    
    return $networkId
}

<#
    .SYNOPSIS
        Creates a new instance of NATed network
#>
function
NewCWACNetwork
{
    [CmdletBinding()]
    param(
        [Parameter(Mandatory=$True)]
        [string]
        [ValidateNotNullOrEmpty()]
        $NetworkName,

        [Parameter(Mandatory=$True)]
        [string]
        [ValidateNotNullOrEmpty()]
        $NatSubnetPrefix,

        [Parameter(Mandatory=$True)]
        [CWACLogger]
        [ValidateNotNull()]
        $Logger
    )
    
    $Logger.LogDiag("Configuring network")
    RemoveAllNatNetworks -Logger $Logger
    
    $Logger.LogDiag("Creating new virtual network: `"$($NetworkName)`"")
    $network = New-ContainerNetwork -Name $NetworkName -Mode NAT -SubnetPrefix $NatSubnetPrefix
    
    if ($network -eq $null)
    {
        throw "NewCWACNetwork: fatal error 1504: $($error)"
    }        
    
    return $network.Id
}

<#
    .SYNOPSIS
        Removed all Nat based networks since only 1 NAT instance is allowed on the machine.
#>
function
RemoveAllNatNetworks
{
    [CmdletBinding()]
    param(
        [Parameter(Mandatory=$True)]
        [CWACLogger]
        [ValidateNotNull()]
        $Logger
    )

    $allIsolatedEnvNetworks = Get-ContainerNetwork -ErrorAction stop
    
    if ($allIsolatedEnvNetworks.Count -ne 0)
    {
        foreach ($isolatedEnvNetwork in $($allIsolatedEnvNetworks |? { $_.Mode -eq "NAT" }))
        {
            $Logger.LogDiag("Removing virtual network: `"$($isolatedEnvNetwork.Name)`"")
            Remove-ContainerNetwork -Name $isolatedEnvNetwork.Name -Force -ErrorAction stop
        }
    }

    $existingNat = Get-NetNat

    if ($existingNat -ne $null)
    {
        $Logger.LogDiag("Removing all existing NetNat instances")
        Remove-NetNat -Confirm:$False
    }

}

<#
    .SYNOPSIS
        Looks up an existing instance of centennial tools network instance.
#>
function
GetCWACNetwork
{
    [CmdletBinding()]
    param(
        [Parameter(Mandatory=$True)]
        [string]
        [ValidateNotNullOrEmpty()]
        $NetworkName,

        [Parameter(Mandatory=$True)]
        [string]
        [ValidateNotNullOrEmpty()]
        $NatSubnetPrefix,

        [Parameter(Mandatory=$True)]
        [CWACLogger]
        [ValidateNotNull()]
        $Logger
    )
    $Logger.LogDiag("Checking network config...")
    
    $allIsolatedEnvNetworks = Get-ContainerNetwork -ErrorAction stop

    if ($allIsolatedEnvNetworks.Count -ne 0)
    {
        foreach ($existingNetwork in $allIsolatedEnvNetworks)
        {
            $Logger.LogDiag("Checking virtual network: `"$($existingNetwork.Name)`"")
            if ($existingNetwork.Name -eq $NetworkName -And $existingNetwork.Mode -eq "NAT" -And $existingNetwork.SubnetPrefix -eq $NatSubnetPrefix)
            {
                $Logger.LogDiag("Using virtual network: `"$($existingNetwork.Name)`"")
                return $existingNetwork.Id
            }
        }
    }
    
    $Logger.LogDiag("Necessary network config not found")
    return $null
}

#
# Entrypoint for standalone call
#
#$logger = [CWACLogger]::new("c:\temp\log.txt")
#GetOrCreateCWACNetwork -NatSubnetPrefix "172.16.103.0/24" -Logger $logger

# SIG # Begin signature block
# MIIkIwYJKoZIhvcNAQcCoIIkFDCCJBACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAaip1OVtijcSpV
# vWkIghA7pC5HQHdpBnulyj6b2jInK6CCDZIwggYQMIID+KADAgECAhMzAAAAZEeE
# lIbbQRk4AAAAAABkMA0GCSqGSIb3DQEBCwUAMH4xCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
# b3NvZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29mdCBDb2RlIFNpZ25p
# bmcgUENBIDIwMTEwHhcNMTUxMDI4MjAzMTQ2WhcNMTcwMTI4MjAzMTQ2WjCBgzEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1v
# bmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9Q
# UjEeMBwGA1UEAxMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMIIBIjANBgkqhkiG9w0B
# AQEFAAOCAQ8AMIIBCgKCAQEAky7a2OY+mNkbD2RfTahYTRQ793qE/DwRMTrvicJK
# LUGlSF3dEp7vq2YoNNV9KlV7TE2K8sDxstNSFYu2swi4i1AL3X/7agmg3GcExPHf
# vHUYIEC+eCyZVt3u9S7dPkL5Wh8wrgEUirCCtVGg4m1l/vcYCo0wbU06p8XzNi3u
# XyygkgCxHEziy/f/JCV/14/A3ZduzrIXtsccRKckyn6B5uYxuRbZXT7RaO6+zUjQ
# hiyu3A4hwcCKw+4bk1kT9sY7gHIYiFP7q78wPqB3vVKIv3rY6LCTraEbjNR+phBQ
# EL7hyBxk+ocu+8RHZhbAhHs2r1+6hURsAg8t4LAOG6I+JQIDAQABo4IBfzCCAXsw
# HwYDVR0lBBgwFgYIKwYBBQUHAwMGCisGAQQBgjdMCAEwHQYDVR0OBBYEFFhWcQTw
# vbsz9YNozOeARvdXr9IiMFEGA1UdEQRKMEikRjBEMQ0wCwYDVQQLEwRNT1BSMTMw
# MQYDVQQFEyozMTY0Mis0OWU4YzNmMy0yMzU5LTQ3ZjYtYTNiZS02YzhjNDc1MWM0
# YjYwHwYDVR0jBBgwFoAUSG5k5VAF04KqFzc3IrVtqMp1ApUwVAYDVR0fBE0wSzBJ
# oEegRYZDaHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraW9wcy9jcmwvTWljQ29k
# U2lnUENBMjAxMV8yMDExLTA3LTA4LmNybDBhBggrBgEFBQcBAQRVMFMwUQYIKwYB
# BQUHMAKGRWh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lvcHMvY2VydHMvTWlj
# Q29kU2lnUENBMjAxMV8yMDExLTA3LTA4LmNydDAMBgNVHRMBAf8EAjAAMA0GCSqG
# SIb3DQEBCwUAA4ICAQCI4gxkQx3dXK6MO4UktZ1A1r1mrFtXNdn06DrARZkQTdu0
# kOTLdlGBCfCzk0309RLkvUgnFKpvLddrg9TGp3n80yUbRsp2AogyrlBU+gP5ggHF
# i7NjGEpj5bH+FDsMw9PygLg8JelgsvBVudw1SgUt625nY7w1vrwk+cDd58TvAyJQ
# FAW1zJ+0ySgB9lu2vwg0NKetOyL7dxe3KoRLaztUcqXoYW5CkI+Mv3m8HOeqlhyf
# FTYxPB5YXyQJPKQJYh8zC9b90JXLT7raM7mQ94ygDuFmlaiZ+QSUR3XVupdEngrm
# ZgUB5jX13M+Pl2Vv7PPFU3xlo3Uhj1wtupNC81epoxGhJ0tRuLdEajD/dCZ0xIni
# esRXCKSC4HCL3BMnSwVXtIoj/QFymFYwD5+sAZuvRSgkKyD1rDA7MPcEI2i/Bh5O
# MAo9App4sR0Gp049oSkXNhvRi/au7QG6NJBTSBbNBGJG8Qp+5QThKoQUk8mj0ugr
# 4yWRsA9JTbmqVw7u9suB5OKYBMUN4hL/yI+aFVsE/KJInvnxSzXJ1YHka45ADYMK
# AMl+fLdIqm3nx6rIN0RkoDAbvTAAXGehUCsIod049A1T3IJyUJXt3OsTd3WabhIB
# XICYfxMg10naaWcyUePgW3+VwP0XLKu4O1+8ZeGyaDSi33GnzmmyYacX3BTqMDCC
# B3owggVioAMCAQICCmEOkNIAAAAAAAMwDQYJKoZIhvcNAQELBQAwgYgxCzAJBgNV
# BAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
# HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jvc29m
# dCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDExMB4XDTExMDcwODIwNTkw
# OVoXDTI2MDcwODIxMDkwOVowfjELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hp
# bmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jw
# b3JhdGlvbjEoMCYGA1UEAxMfTWljcm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EgMjAx
# MTCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAKvw+nIQHC6t2G6qghBN
# NLrytlghn0IbKmvpWlCquAY4GgRJun/DDB7dN2vGEtgL8DjCmQawyDnVARQxQtOJ
# DXlkh36UYCRsr55JnOloXtLfm1OyCizDr9mpK656Ca/XllnKYBoF6WZ26DJSJhIv
# 56sIUM+zRLdd2MQuA3WraPPLbfM6XKEW9Ea64DhkrG5kNXimoGMPLdNAk/jj3gcN
# 1Vx5pUkp5w2+oBN3vpQ97/vjK1oQH01WKKJ6cuASOrdJXtjt7UORg9l7snuGG9k+
# sYxd6IlPhBryoS9Z5JA7La4zWMW3Pv4y07MDPbGyr5I4ftKdgCz1TlaRITUlwzlu
# ZH9TupwPrRkjhMv0ugOGjfdf8NBSv4yUh7zAIXQlXxgotswnKDglmDlKNs98sZKu
# HCOnqWbsYR9q4ShJnV+I4iVd0yFLPlLEtVc/JAPw0XpbL9Uj43BdD1FGd7P4AOG8
# rAKCX9vAFbO9G9RVS+c5oQ/pI0m8GLhEfEXkwcNyeuBy5yTfv0aZxe/CHFfbg43s
# TUkwp6uO3+xbn6/83bBm4sGXgXvt1u1L50kppxMopqd9Z4DmimJ4X7IvhNdXnFy/
# dygo8e1twyiPLI9AN0/B4YVEicQJTMXUpUMvdJX3bvh4IFgsE11glZo+TzOE2rCI
# F96eTvSWsLxGoGyY0uDWiIwLAgMBAAGjggHtMIIB6TAQBgkrBgEEAYI3FQEEAwIB
# ADAdBgNVHQ4EFgQUSG5k5VAF04KqFzc3IrVtqMp1ApUwGQYJKwYBBAGCNxQCBAwe
# CgBTAHUAYgBDAEEwCwYDVR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0j
# BBgwFoAUci06AjGQQ7kUBU7h6qfHMdEjiTQwWgYDVR0fBFMwUTBPoE2gS4ZJaHR0
# cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2Vy
# QXV0MjAxMV8yMDExXzAzXzIyLmNybDBeBggrBgEFBQcBAQRSMFAwTgYIKwYBBQUH
# MAKGQmh0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2kvY2VydHMvTWljUm9vQ2Vy
# QXV0MjAxMV8yMDExXzAzXzIyLmNydDCBnwYDVR0gBIGXMIGUMIGRBgkrBgEEAYI3
# LgMwgYMwPwYIKwYBBQUHAgEWM2h0dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9wa2lv
# cHMvZG9jcy9wcmltYXJ5Y3BzLmh0bTBABggrBgEFBQcCAjA0HjIgHQBMAGUAZwBh
# AGwAXwBwAG8AbABpAGMAeQBfAHMAdABhAHQAZQBtAGUAbgB0AC4gHTANBgkqhkiG
# 9w0BAQsFAAOCAgEAZ/KGpZjgVHkaLtPYdGcimwuWEeFjkplCln3SeQyQwWVfLiw+
# +MNy0W2D/r4/6ArKO79HqaPzadtjvyI1pZddZYSQfYtGUFXYDJJ80hpLHPM8QotS
# 0LD9a+M+By4pm+Y9G6XUtR13lDni6WTJRD14eiPzE32mkHSDjfTLJgJGKsKKELuk
# qQUMm+1o+mgulaAqPyprWEljHwlpblqYluSD9MCP80Yr3vw70L01724lruWvJ+3Q
# 3fMOr5kol5hNDj0L8giJ1h/DMhji8MUtzluetEk5CsYKwsatruWy2dsViFFFWDgy
# cScaf7H0J/jeLDogaZiyWYlobm+nt3TDQAUGpgEqKD6CPxNNZgvAs0314Y9/HG8V
# fUWnduVAKmWjw11SYobDHWM2l4bf2vP48hahmifhzaWX0O5dY0HjWwechz4GdwbR
# BrF1HxS+YWG18NzGGwS+30HHDiju3mUv7Jf2oVyW2ADWoUa9WfOXpQlLSBCZgB/Q
# ACnFsZulP0V3HjXG0qKin3p6IvpIlR+r+0cjgPWe+L9rt0uX4ut1eBrs6jeZeRhL
# /9azI2h15q/6/IvrC4DqaTuv/DDtBEyO3991bWORPdGdVk5Pv4BXIqF4ETIheu9B
# CrE/+6jMpF3BoYibV3FWTkhFwELJm3ZbCoBIa/15n8G9bW1qyVJzEw16UM0xghXn
# MIIV4wIBATCBlTB+MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
# MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9u
# MSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQSAyMDExAhMzAAAA
# ZEeElIbbQRk4AAAAAABkMA0GCWCGSAFlAwQCAQUAoIHWMBkGCSqGSIb3DQEJAzEM
# BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqG
# SIb3DQEJBDEiBCAc234cipRZZQ9Un2/4hGLedGCRIuAx4hLtSKjJWz7K8TBqBgor
# BgEEAYI3AgEMMVwwWqAugCwAVwBpAG4AZABvAHcAcwAgAFAAaABvAG4AZQAgAFAA
# YQBjAGsAYQBnAGUAc6EogCZodHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vd2luZG93
# c3Bob25lLzANBgkqhkiG9w0BAQEFAASCAQAyfGWd+jIB5HHbAPYgt+1fGaiiMr15
# ZfHb1McwvDy+LOuDoL0xxDEHhWskC5msUs+O5VkVhNAnbuAT7iMmHhsqlCRzKwlE
# IYonsokucRmlewXGExwf/3z+nUTv3JgqqtcxRfsSMdbFoK7lSNpUyXY6qFqG2oCU
# SUO20wIcRDDjVoEIMViz9RRxVFTduY63hIMn72F6hHhkeCN55TjTIKBCqlVVGnXg
# 6uzX7mpIYTLO2QIR7UIBgzWF3iwg7eLqhZ3+N6KvC6MvzNYW0m5/C62+ip5DQsNb
# s10U8ks0geZiuKDfAnJiup9Sw9nZ74DVgcIMc8yk/GQbV2RNtZfcuaW9oYITSTCC
# E0UGCisGAQQBgjcDAwExghM1MIITMQYJKoZIhvcNAQcCoIITIjCCEx4CAQMxDzAN
# BglghkgBZQMEAgEFADCCATwGCyqGSIb3DQEJEAEEoIIBKwSCAScwggEjAgEBBgor
# BgEEAYRZCgMBMDEwDQYJYIZIAWUDBAIBBQAEIKvYIQD3MTxAml34222JR40YGahV
# CPMjXhWU4fueAC5aAgZW9iAkWKIYEjIwMTYwNDA2MDQxNDA5Ljk2WjAHAgEBgAIB
# 9KCBuaSBtjCBszELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAO
# BgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjEN
# MAsGA1UECxMETU9QUjEnMCUGA1UECxMebkNpcGhlciBEU0UgRVNOOjdEMkUtMzc4
# Mi1CMEY3MSUwIwYDVQQDExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloIIO
# zTCCBnEwggRZoAMCAQICCmEJgSoAAAAAAAIwDQYJKoZIhvcNAQELBQAwgYgxCzAJ
# BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
# MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xMjAwBgNVBAMTKU1pY3Jv
# c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eSAyMDEwMB4XDTEwMDcwMTIx
# MzY1NVoXDTI1MDcwMTIxNDY1NVowfDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENBIDIw
# MTAwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCpHQ28dxGKOiDs/BOX
# 9fp/aZRrdFQQ1aUKAIKF++18aEssX8XD5WHCdrc+Zitb8BVTJwQxH0EbGpUdzgkT
# jnxhMFmxMEQP8WCIhFRDDNdNuDgIs0Ldk6zWczBXJoKjRQ3Q6vVHgc2/JGAyWGBG
# 8lhHhjKEHnRhZ5FfgVSxz5NMksHEpl3RYRNuKMYa+YaAu99h/EbBJx0kZxJyGiGK
# r0tkiVBisV39dx898Fd1rL2KQk1AUdEPnAY+Z3/1ZsADlkR+79BL/W7lmsqxqPJ6
# Kgox8NpOBpG2iAg16HgcsOmZzTznL0S6p/TcZL2kAcEgCZN4zfy8wMlEXV4WnAEF
# TyJNAgMBAAGjggHmMIIB4jAQBgkrBgEEAYI3FQEEAwIBADAdBgNVHQ4EFgQU1WM6
# XIoxkPNDe3xGG8UzaFqFbVUwGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEwCwYD
# VR0PBAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wHwYDVR0jBBgwFoAU1fZWy4/oolxi
# aNE9lJBb186aGMQwVgYDVR0fBE8wTTBLoEmgR4ZFaHR0cDovL2NybC5taWNyb3Nv
# ZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvTWljUm9vQ2VyQXV0XzIwMTAtMDYtMjMu
# Y3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+aHR0cDovL3d3dy5taWNy
# b3NvZnQuY29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXRfMjAxMC0wNi0yMy5jcnQw
# gaAGA1UdIAEB/wSBlTCBkjCBjwYJKwYBBAGCNy4DMIGBMD0GCCsGAQUFBwIBFjFo
# dHRwOi8vd3d3Lm1pY3Jvc29mdC5jb20vUEtJL2RvY3MvQ1BTL2RlZmF1bHQuaHRt
# MEAGCCsGAQUFBwICMDQeMiAdAEwAZQBnAGEAbABfAFAAbwBsAGkAYwB5AF8AUwB0
# AGEAdABlAG0AZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQAH5ohRDeLG4Jg/
# gXEDPZ2joSFvs+umzPUxvs8F4qn++ldtGTCzwsVmyWrf9efweL3HqJ4l4/m87WtU
# VwgrUYJEEvu5U4zM9GASinbMQEBBm9xcF/9c+V4XNZgkVkt070IQyK+/f8Z/8jd9
# Wj8c8pl5SpFSAK84Dxf1L3mBZdmptWvkx872ynoAb0swRCQiPM/tA6WWj1kpvLb9
# BOFwnzJKJ/1Vry/+tuWOM7tiX5rbV0Dp8c6ZZpCM/2pif93FSguRJuI57BlKcWOd
# eyFtw5yjojz6f32WapB4pm3S4Zz5Hfw42JT0xqUKloakvZ4argRCg7i1gJsiOCC1
# JeVk7Pf0v35jWSUPei45V3aicaoGig+JFrphpxHLmtgOR5qAxdDNp9DvfYPw4Ttx
# Cd9ddJgiCGHasFAeb73x4QDf5zEHpJM692VHeOj4qEir995yfmFrb3epgcunCaw5
# u+zGy9iCtHLNHfS4hQEegPsbiSpUObJb2sgNVZl6h3M7COaYLeqN4DMuEin1wC9U
# JyH3yKxO2ii4sanblrKnQqLJzxlBTeCG+SqaoxFmMNO7dDJL32N79ZmKLxvHIa9Z
# ta7cRDyXUHHXodLFVeNp3lfB0d4wwP3M5k37Db9dT+mdHhk4L7zPWAUu7w2gUDXa
# 7wknHNWzfjUeCLraNtvTX4/edIhJEjCCBNowggPCoAMCAQICEzMAAAB2thDhfIcj
# PCIAAAAAAHYwDQYJKoZIhvcNAQELBQAwfDELMAkGA1UEBhMCVVMxEzARBgNVBAgT
# Cldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29m
# dCBDb3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRpbWUtU3RhbXAgUENB
# IDIwMTAwHhcNMTUxMDA3MTgxNzQwWhcNMTcwMTA3MTgxNzQwWjCBszELMAkGA1UE
# BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAc
# BgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEnMCUG
# A1UECxMebkNpcGhlciBEU0UgRVNOOjdEMkUtMzc4Mi1CMEY3MSUwIwYDVQQDExxN
# aWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNlMIIBIjANBgkqhkiG9w0BAQEFAAOC
# AQ8AMIIBCgKCAQEAowWT83TIYOEsy1BbeQ6onhnQj9IkowZhulaWtTqUAaLgZ93p
# ESP+/oJiGT8NvIcnkuZmUVCu97UR+wYOqKP3JluD78JuWOpNyFPrkXgQdcYGvoOQ
# wGm66Fl3euhadrKqYgNxz1N415Yn2Jd+t4SAv8w66FbeNA2xRfwIgDuS/Ao1yvxT
# 9QhxHypm9GBWjh0yHGQfXKFpJTmVHSpWd3CHvnDgxsVtGDmmQ/lO6hItBq5nbjHq
# jKxfsg/GexYJ02PwYesOVWP1TaoAyBnTtLpRAi2tkoFubIVmRpLzoy1TVEoJQdLj
# yyiObb5YZCnrdyh2Yv9hfM70w9PBveWdwWMhgwIDAQABo4IBGzCCARcwHQYDVR0O
# BBYEFPQUhUg89/rvrGgO0JGpx6u0AFzLMB8GA1UdIwQYMBaAFNVjOlyKMZDzQ3t8
# RhvFM2hahW1VMFYGA1UdHwRPME0wS6BJoEeGRWh0dHA6Ly9jcmwubWljcm9zb2Z0
# LmNvbS9wa2kvY3JsL3Byb2R1Y3RzL01pY1RpbVN0YVBDQV8yMDEwLTA3LTAxLmNy
# bDBaBggrBgEFBQcBAQROMEwwSgYIKwYBBQUHMAKGPmh0dHA6Ly93d3cubWljcm9z
# b2Z0LmNvbS9wa2kvY2VydHMvTWljVGltU3RhUENBXzIwMTAtMDctMDEuY3J0MAwG
# A1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwgwDQYJKoZIhvcNAQELBQAD
# ggEBAFq0RKBYfWXf+0OrepmaC87ddLBgTk+bq55jJIBI30rbqXICHIPaHRmdKJYK
# S+txqqovQJXmUoeKHg+Pcyk1k2PFjHQ4ctq0R8GIeP3a4izWSZuR8pt7WWx2ZIe5
# +FtfbcPX5OWHktYkouGLB6FiFU69vr/7BXHeQBf6GQk9V3XbEydldzMIwT+w1P2J
# uuWckVKzobi83H89NrttcUmEqoU82waAYr+Y/E+3Bd1OBUk7+it9LNWaQeALeJeV
# cUp0KPE5C8hGrAA4Qt/JLXkjfyfoV6XNTdjuhsao61e1zICA6HR95wInU9GcIXNE
# xQJHEGui5wBTFKT+F7kz2f94bMihggN2MIICXgIBATCB46GBuaSBtjCBszELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQx
# HjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3JhdGlvbjENMAsGA1UECxMETU9QUjEn
# MCUGA1UECxMebkNpcGhlciBEU0UgRVNOOjdEMkUtMzc4Mi1CMEY3MSUwIwYDVQQD
# ExxNaWNyb3NvZnQgVGltZS1TdGFtcCBTZXJ2aWNloiUKAQEwCQYFKw4DAhoFAAMV
# AN4dO70U9txQM+bRN0h9OZPgk3cvoIHCMIG/pIG8MIG5MQswCQYDVQQGEwJVUzET
# MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
# TWljcm9zb2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScwJQYDVQQLEx5u
# Q2lwaGVyIE5UUyBFU046NTdGNi1DMUUwLTU1NEMxKzApBgNVBAMTIk1pY3Jvc29m
# dCBUaW1lIFNvdXJjZSBNYXN0ZXIgQ2xvY2swDQYJKoZIhvcNAQEFBQACBQDartad
# MCIYDzIwMTYwNDA2MDAyODEzWhgPMjAxNjA0MDcwMDI4MTNaMHQwOgYKKwYBBAGE
# WQoEATEsMCowCgIFANqu1p0CAQAwBwIBAAICG4YwBwIBAAICHjEwCgIFANqwKB0C
# AQAwNgYKKwYBBAGEWQoEAjEoMCYwDAYKKwYBBAGEWQoDAaAKMAgCAQACAxbjYKEK
# MAgCAQACAwehIDANBgkqhkiG9w0BAQUFAAOCAQEAlzf39LD10SMkKmNVwayXFHOr
# E0Qjgf+cjHF9w7v26hz2XoJ3ZhiDtKUcAJG465JpQ4B46sMOvsb8j5cy47IUUVOj
# OpnCKSKxGt6R3H8C9SNh64vtQpHhKxx/i41xmiHpzlBPDEtphy0fuZvyT73FaVDJ
# yAHttHAtFNdA3+rzqMUJTeI8wLqXGFt/ZPcE7zKqBw5IR1WqrJ8rVu9LoxeatCkp
# mTrJxot+hH6tpM1IlqNLu6cbfp4cOGHLLxTYjVSMbo9JmsDbkba75k4MffSIeqDk
# MCaBb6QOD43jc0MW8RM3UfCj61pBkiIJrgEKSUASjycKnKhMmEzaAMbfKa0KrjGC
# AvUwggLxAgEBMIGTMHwxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
# MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
# b24xJjAkBgNVBAMTHU1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAA
# drYQ4XyHIzwiAAAAAAB2MA0GCWCGSAFlAwQCAQUAoIIBMjAaBgkqhkiG9w0BCQMx
# DQYLKoZIhvcNAQkQAQQwLwYJKoZIhvcNAQkEMSIEIMwOxAHv6FtCN7ofDQdcxFeR
# MEMnrhN75wk4xi5GDa7RMIHiBgsqhkiG9w0BCRACDDGB0jCBzzCBzDCBsQQU3h07
# vRT23FAz5tE3SH05k+CTdy8wgZgwgYCkfjB8MQswCQYDVQQGEwJVUzETMBEGA1UE
# CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
# b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3NvZnQgVGltZS1TdGFtcCBQ
# Q0EgMjAxMAITMwAAAHa2EOF8hyM8IgAAAAAAdjAWBBTKrTjlgEMEHWo4H+Mmva1A
# PnSqYjANBgkqhkiG9w0BAQsFAASCAQB+byGbxw93Jy/CSqiNmqN9i/XE65ZTmfos
# xRMwKt/ol3LEWX/3QLCufClMegu/pfLOzgyKtBwl2oqA2gF8KQi+/0qJbEDGlIxn
# bgc5u78RMgLqLWg1D5sQBeFbDtuqInaU026J18eQD3ARhJRbVjg2xkGErkAmWfyd
# bXtKamtie7b97iHp9N0wtBXsimh0U5Pk3iolb2rfRCHLP8JFmt9r5hS/MznpHxy0
# bhsQuhH0shqYW86jTg7tnv2ND/EQo8ITIdZLwEUjsWBTupSrSHzlzE18GwiF1SD5
# EMao3CCZkYxB/+KTKdchyFZiv5mLLKRgFl9t9tqQwShaZjh9Dn9q
# SIG # End signature block
