=begin
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
# Designed by Fredo6 - Copyright November 2008

# Permission to use this software for any purpose and without fee is hereby granted
# Distribution of this software for commercial purpose is subject to:
#  - the expressed, written consent of the author
#  - the inclusion of the present copyright notice in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name			:   bootstrap_FredoScale.rb
# Original Date	:   6 Dec 2010 - version 2.1
# Description	:   Boostrap environment for FredoScale
#-------------------------------------------------------------------------------------------------------------------------------------------------
#*************************************************************************************************
=end

module F6_FredoScale

#---------------------------------------------------------------------------------------------------------------------------
# Overall Configuration for the Tools	
#---------------------------------------------------------------------------------------------------------------------------

# SCALE tool
CONFIG_Scale = {
	'Name' => "Scaling",
	'Supported' => 'BT',
	'Color' => ['Yellow', 'DarkRed'],
	'Visible' => 'B'
	}

# TAPER tool
CONFIG_Taper = {
	'Name' => "Tapering",
	'Supported' => 'BT',
	'Color' => ['Green', 'DarkMagenta'],
	'Visible' => 'B'
	}

# PLANAR SHEAR tool
CONFIG_PlanarShear = {
	'Name' => "Planar Shearing",
	'Supported' => 'BTA',
	'Color' => ['DarkGreen', 'Magenta'],
	'Visible' => 'BA'
	}

# ROTATE tool
CONFIG_Rotate = {
	'Name' => "Rotation",
	'Supported' => 'BA',
	'Color' => ['LightBlue'],
	'Visible' => 'BA'
	}

# STRETCH tool
CONFIG_Stretch = {
	'Name' => "Stretching",
	'Supported' => 'BT',
	'Color' => ['GreenYellow', 'DarkOliveGreen'],
	'Visible' => 'B'
	}

# TWIST tool
CONFIG_Twist = {
	'Name' => "Twisting",
	'Supported' => 'B',
	'Color' => ['Red'],
	'parent' => 'Rotate',
	'Visible' => 'B'
	}

# Radial Bending tool
CONFIG_RadialBend = {
	'Name' => "Radial Bending",
	'Supported' => 'A',
	'Color' => ['LightGreen'],
	'Visible' => 'A'
	}

# Make Unique tool
CONFIG_MakeUnique = {
	'Name' => "Make Unique",
	'Supported' => 'U',
	'Color' => ['LightGreen'],
	'Tooltip' => :TIP_Make_Unique,
	'Visible' => ''
	}

# Edge Selection
CONFIG_SelectEdges = {
	'Name' => "Select Edges",
	'Supported' => 'U',
	'Color' => ['LightGreen'],
	'Tooltip' => :TIP_Select_Edges,
	'Visible' => ''
	}

# Declaring the configuration - List and setting the titles
ALL_CONFIG = [:Scale, :Taper, :PlanarShear, :Stretch, :Twist, :Rotate, :RadialBend, :MakeUnique]

ALL_CONFIG.each { |symb| eval "T6[:NAME_#{symb.to_s}] = CONFIG_#{symb.to_s}['Name']" }

MENU_PERSO = [:T_MenuTools_Fredo6Collection, "Tools", :T_MenuPlugins_Fredo6Collection, "Plugins"]

#---------------------------------------------------------------------------------------------------------------------------
#Constants for F6_FredoScale Module (do not translate here, use Translation Dialog Box instead)	
#---------------------------------------------------------------------------------------------------------------------------

# Dynamic strings for Tools menus and titles
T6[:MNU_Box] = "Box %1"
T6[:MNU_BoxTarget] = "Box %1 to Target"
T6[:MNU_Alone] = "%1 (Free)"
T6[:TIP_Box] = "%1 with orientation of scaling box"
T6[:TIP_BoxTarget] = "%1 with box by matching an origin and a target"
T6[:TIP_Angle] = "%1 by selection of plane and angle"
T6[:DEF_ColorBox] = "Box Color (%1)"

# Strings for the application
T6[:MSG_Status_Grip] = "Select a grip and Click"
T6[:MSG_Status_Drag] = "Drag grip to scale - CTRL for About Center - SHIFT for Non-Uniform scaling"
T6[:TIP_Edge] = "Click on Edge to orientate the deformation box"
T6[:TIP_Face] = "Click on Face to orientate the deformation box"
T6[:TIP_FromCenter] = "Scaling from Center"
T6[:TIP_UFromCenter] = "Uniform from Center"
T6[:TIP_Opposite] = "From Opposite point"
T6[:TIP_UOpposite] = "Uniform from Opposite point"
T6[:TIP_ClickExit] = "Click to Exit"
T6[:TIP_PostValidate] = "Click to validate deformation and reset box"
T6[:TIP_ScalingBox] = "Scaling Box"

T6[:TIP_Make_Unique] = "Make Groups and Components Unique"
T6[:TIP_Select_Edges] = "Select Edges by mouse over with CTRL"
T6[:DLG_Unique_Already] = "All components and groups are already unique"
T6[:DLG_Unique_InfoDlg] = "Make Unique Groups = %1 Components = %2"

T6[:STR_DimFromTo] = "from %1 to"

T6[:MNU_SwitchToTarget] = "Switch to Target mode (Double-Click on handle)"
T6[:MNU_SwitchToBox] = "Switch to Box mode (Double-Click on handle)"
T6[:MNU_AskDimensions] = "Enter New Dimensions (TAB)"
T6[:MNU_FreeScaling] = "%1: Free Scaling (SHIFT)"
T6[:MNU_UniformScaling] = "%1: Uniform Scaling (SHIFT)"
T6[:MNU_FromCenter] = "From Center (CTRL)"
T6[:MNU_FromOpposite] = "From Opposite point (CTRL)"
T6[:MNU_Done] = "Finish dragging"
T6[:MNU_Validate] = "Validate deformation and reset box"
T6[:MNU_Exit] = "Exit (click outside box)"
T6[:MNU_DividerDefault] = "Reset Divider to centre (Double Click)"
T6[:MNU_DividerShowHide] = "Show / Hide Divider (Stretch)"
T6[:MNU_NoDeform] = "Interactive deformation"
T6[:MNU_Wireframe] = "Wireframe simulation"
T6[:MNU_ActivateDice] = "Slicing"
T6[:MNU_DiceParam] = "Slicing Parameters (TAB)"

T6[:OPT_ShowDivider_No] = "Hidden"
T6[:OPT_ShowDivider_Grid] = "Grid"
T6[:OPT_ShowDivider_Rect] = "Rectangle"
OPT_ShowDivider = [['0', :OPT_ShowDivider_No], ['G', :OPT_ShowDivider_Grid], ['R', :OPT_ShowDivider_Rect]]

T6[:DLG_Dim_title] = "New Dimensions" 
T6[:WARNING_SelHandleFirst] = "You must select box handles first" 

#Colors and marks used by the Scale Tool
COLOR_Pk_Edge = "orange"
COLOR_Sel_Handle_Free = "white"
COLOR_Sel_Handle_FreeTarget = "lightgrey"
COLOR_Sel_Handle_Pivot = "red"
COLOR_Free_LineTarget = "purple"
MARK_Target_Origin = [16, 4, "darkgreen"]
MARK_Target_Target = [16, 4, "darkred"]
MARK_Free_Target = [4, 2, "orange"]
MARK_Divider_Origin = [7, 3, "black"]
MARK_Divider_Target = [7, 3, "darkred"]

LIMIT_InferenceScale = 0.01		#Delta for scale inference at round numbers
LIMIT_SmallScale = 0.1			#Do not go below this scale ratio when dragging

#Labels for Default Parameters
T6[:DEFAULT_SectionAll] = "Active Options at start up"
T6[:DEFAULT_SelectionExtended] = "When started, Extend selection to all connected entities"
T6[:DEFAULT_Flag_LiveDeform] = "Use Live Deformation by default (when applicable)"
T6[:DEFAULT_Key_LiveDeform] = "Toggle Live Deformation"
T6[:DEFAULT_Flag_Wireframe] = "Use Wireframe by default"
T6[:DEFAULT_Color_Wireframe] = "Color of the Wireframe"
T6[:DEFAULT_Key_Wireframe] = "Toggle Wireframe"
T6[:DEFAULT_Flag_ShowDivider] = "Stretch Divider visibility by default"
T6[:DEFAULT_Key_ShowDivider] = "Toggle Show / Hide Divider (Stretch)"
T6[:DEFAULT_Flag_ActivateDice] = "Slicing active by default"
T6[:DEFAULT_Key_ActivateDice] = "Toggle Activation Slicing"
T6[:DEFAULT_Flag_EdgeNewProp] = "Default Properties for newly created edges"
T6[:DEFAULT_Key_EdgeNewProp] = "Dialog box for Properties for newly created edges"
T6[:DEFAULT_WidthBox] = "Width of the Deformation Box (pixel)"
T6[:DEFAULT_NoSepa] = "NO Separator in the toolbar"
T6[:DEFAULT_SectionColorBox] = "Parameters for the Deformation Box"
T6[:DEFAULT_DimCollapsePixel] = "Dimension in pixel under which handles are collapsed"
T6[:DEFAULT_LimitInferenceScale] = "Minimum increment of scale factor when dragging"

Chrono = Traductor::Chrono6
Trace = Traductor::Tracer6

#---------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------
# EXTERNAL API to launch FredoScale menus
# Action code must be passed as a symbol
#   - :Scale_B --> Scale - Box mode
#   - :Scale_T --> Scale - Target mode
#   - :Taper_B --> Taper - Box mode
#   - :Taper_T --> Scale - Target mode
#   - :PlanarShear_B --> Planar Shear - Box mode
#   - :PlanarShear_T --> Planar Shear - Target mode
#   - :PlanarShear_A --> Planar Shear - Free mode
#   - :Stretch_B --> Stretch - Box mode
#   - :Stretch_T --> Stretch - Target mode
#   - :Twist_B --> Twist - Box mode
#   - :Rotate_B --> Rotate - Box mode
#   - :Rotate_A --> Rotate - Free mode
#   - :RadialBend_A --> Radial Bending - Free mode
#   - :MakeUnique_U --> Make Unique - Utility
#---------------------------------------------------------------------------------------------------------------------------------
#---------------------------------------------------------------------------------------------------------------------------------
	
def F6_FredoScale.launch_action(action_code)
	MYPLUGIN.launch_action action_code
end

#--------------------------------------------------------------------------------------------------------------
# Plugin Startup Method
#--------------------------------------------------------------------------------------------------------------			 

def F6_FredoScale.startup

	#Common Default Parameters
	MYDEFPARAM.separator :DEFAULT_SectionAll
	MYDEFPARAM.declare :DEFAULT_SelectionExtended, true, 'B'
	MYDEFPARAM.declare :DEFAULT_Flag_LiveDeform, true, 'B'
	MYDEFPARAM.declare :DEFAULT_Color_Wireframe, "Purple", 'K'
	MYDEFPARAM.declare :DEFAULT_Flag_Wireframe, true, 'B'
	MYDEFPARAM.declare :DEFAULT_Flag_ShowDivider, '0', 'H', T6[OPT_ShowDivider]
	MYDEFPARAM.declare :DEFAULT_Flag_ActivateDice, true, 'B'
	MYDEFPARAM.declare :DEFAULT_Flag_EdgeNewProp, 'S;;M', 'M', [['S', 'Soft'], ['M', 'Smooth'], ['H', 'Hidden']]
	MYDEFPARAM.declare :DEFAULT_DimCollapsePixel, 30, 'I:>=0<200'
	MYDEFPARAM.declare :DEFAULT_LimitInferenceScale, 0.05, 'F:>=0.01<0.1'
	MYDEFPARAM.declare :DEFAULT_NoSepa, false, 'B'
	
	#function keys
	fkeys = Traductor::FKeyOption.fkeys
	MYDEFPARAM.separator :T_DEFAULT_SectionFunctionKey
	MYDEFPARAM.declare :DEFAULT_Key_LiveDeform, 'F2', 'H', fkeys
	MYDEFPARAM.declare :DEFAULT_Key_Wireframe, 'F3', 'H', fkeys
	MYDEFPARAM.declare :DEFAULT_Key_ShowDivider, 'F4', 'H', fkeys
	MYDEFPARAM.declare :DEFAULT_Key_ActivateDice, 'F4', 'H', fkeys
	MYDEFPARAM.declare :DEFAULT_Key_EdgeNewProp, 'F5', 'H', fkeys
	
	MYDEFPARAM.load_file
	
	# All menus and icons
	F6_FredoScale.configure
			
	#Startup of the Plugin
	MYPLUGIN.go
end

#--------------------------------------------------------------------------------------------------------------
# Configuartion and Utilities Method at module level
#--------------------------------------------------------------------------------------------------------------			 				   

#Configuring all tools, based on naming conventions strictly followed
def F6_FredoScale.configure
	lst_visible = []
	sepa = false
	nosepa = MYDEFPARAM[:DEFAULT_NoSepa]
	
	#Default menu configuration
	MYPLUGIN.declare_topmenu nil, MENU_PERSO
	tc = Traductor::TestCondition.new() { F6_FredoScale.valid_selection? }
	submenu = MYPLUGIN.plugin_name
	MYDEFPARAM.separator :DEFAULT_SectionColorBox	
	MYDEFPARAM.declare :DEFAULT_WidthBox, 3, 'I:>=2<=9'
	
	#Configuring each tool
	ALL_CONFIG.each do |symb|
		begin
			#retrieving the hash table with configuration parameter
			code = symb.to_s
			hconfig = F6_FredoScale.const_get 'CONFIG_' + code
			stitle = 'NAME_' + code
			name = T6[stitle.intern]
			supported = hconfig['Supported']
			visible = hconfig['Visible']
			lcolors = hconfig['Color']
			if (lcolors.class == Array)
				bcolor = lcolors[0]
				tcolor = lcolors[1]
			elsif lcolors.class == String
				bcolor = lcolors
			end	
			bcolor = 'Black' unless bcolor
			tcolor = 'Red' unless tcolor
			
			#separator toolbar
			if visible
				MYPLUGIN.declare_separator if sepa
				sepa = true unless nosepa
			end	
					
			#Box tool	
			if supported =~ /B/i
				cmd = "FSC_#{code}_Command"
				scmd = cmd.intern
				menu = T6[:MNU_Box, name]
				ttip = T6[:TIP_Box, name]
				icon = code
				scol = "DEFAULT_Color_Box#{icon}"
				txcol = T6[:DEF_ColorBox, menu]
				MYDEFPARAM.declare scol.intern, bcolor, 'K', nil, txcol
				MYPLUGIN.declare_command_long(scmd, menu, ttip, icon) { F6_FredoScale.launch_action_p code }
				MYPLUGIN.declare_context_handler_long(scmd, menu, ttip, tc, nil, submenu) { F6_FredoScale.launch_action_p code }
				lst_visible.push scmd if visible =~ /B/i
			end	
		
			#Target mode
			if supported =~ /T/i
				cmd = "FSC_#{code}_TARGET_Command"
				scmd = cmd.intern
				menu = T6[:MNU_BoxTarget, name]
				ttip = T6[:TIP_BoxTarget, name]
				icon = code + "Target"
				txcol = T6[:DEF_ColorBox, menu]
				scol = "DEFAULT_Color_Box#{icon}"
				MYDEFPARAM.declare scol.intern, tcolor, 'K', nil, txcol
				MYPLUGIN.declare_command_long(scmd, menu, ttip, icon) { F6_FredoScale.launch_action_p code, 'T' }
				MYPLUGIN.declare_context_handler_long(scmd, menu, ttip, tc, nil, submenu) { F6_FredoScale.launch_action_p code, 'T' }
				lst_visible.push scmd if visible =~ /T/i
			end
			
			#Standalone mode
			if supported =~ /A/i
				cmd = "FSC_#{code}_ALONE_Command"
				scmd = cmd.intern
				menu = T6[:MNU_Alone, name]
				sttip = hconfig['Tooltip']
				ttip = (sttip) ? T6[sttip] : T6[:TIP_Angle, name]
				icon = code + "_Alone"
				MYPLUGIN.declare_command_long(scmd, menu, ttip, icon) { F6_FredoScale.launch_action_p code, 'A' }
				MYPLUGIN.declare_context_handler_long(scmd, menu, ttip, tc, nil, submenu) { F6_FredoScale.launch_action_p code, 'A' }
				lst_visible.push scmd if visible =~ /A/i
			end

			#Standalone mode
			if supported =~ /U/i
				cmd = "FSC_#{code}_UTIL_Command"
				scmd = cmd.intern
				menu = name
				sttip = hconfig['Tooltip']
				ttip = (sttip) ? T6[sttip] : T6[:TIP_Angle, name]
				icon = code + "_Util"
				MYPLUGIN.declare_command_long(scmd, menu, ttip, icon) { F6_FredoScale.launch_action_p code, 'U' }
				MYPLUGIN.declare_context_handler_long(scmd, menu, ttip, tc, nil, submenu) { F6_FredoScale.launch_action_p code, 'U' }
				lst_visible.push scmd if visible =~ /U/i
			end
			
		#End of protected section	
		rescue
		end	
	end	
	
	#declaring the items visible
	MYPLUGIN.default_icons_visible lst_visible
	MYPLUGIN.default_handlers_visible lst_visible	
end

#Private launcher of actions
def F6_FredoScale.launch_action_p(action_code, mode=nil)
	mode = 'B' unless mode
	action_code += '_' + mode
	MYPLUGIN.launch_action action_code.intern
end

# Check if the proposed selection is valid for FredoScale deformations
def F6_FredoScale.valid_selection?
	lclass = [Sketchup::Face, Sketchup::Edge, Sketchup::Group, Sketchup::ComponentInstance]
	Sketchup.active_model.selection.each { |e| return true if lclass.include?(e.class) }
	false
end
	
def F6_FredoScale.calculate_title(code, mode)
	sname = "NAME_#{code}"
	name = T6[sname.intern]
	if mode =~ /U/i
		title = name
	elsif mode =~ /A/i
		title = T6[:MNU_Alone, name]
	elsif mode =~ /T/i
		title = T6[:MNU_BoxTarget, name]
	else
		title = T6[:MNU_Box, name]
	end
	title	
end
	
# Return the title of thecurrent tool	
def F6_FredoScale.compute_title_tool(hparam)
	mode = (hparam['Mode_Target']) ? 'T' : hparam['ToolMode']
	F6_FredoScale.calculate_title hparam['ToolType'], mode
end

# Return the cursor object id for the current tool
def F6_FredoScale.get_cursor(hparam)
	mode = (hparam['Mode_Target']) ? 'T' : hparam['ToolMode']
	if mode == 'U'
		hparam["Cursor_" + hparam["OriginalToolType"] + '_Util']
	elsif mode == 'A'
		hparam["Cursor_" + hparam["OriginalToolType"] + '_Alone']
	else	
		hparam["Cursor_" + hparam["OriginalToolType"] + ((hparam["Mode_Target"]) ? 'Target' : '')]
	end	
end


end	#End module F6_FredoScale
