#!/bin/sh
#
# RemoveFlashback.sh
#   -- Flashback removal tool --
#    Based on a script by Yoshioka Tsuneo:
#      https://github.com/yoshiokatsuneo/RemoveFlashback
#
# Ref:
#   F-Secure Weblog - Mac Flashback Infections
#   http://www.f-secure.com/weblog/archives/00002345.html
#

timestamp=`date '+%Y%m%d-%H%M%S'`
infected=0
tmpfilename_base=/tmp/RemoveFlashback.$$

quar_dir=${tmpfilename_base}.quarantine

mkdir -p "$quar_dir"

# for the LaunchAgent:
#  * Check that binary start with "."
#
#    * (permission is 777)
#    * has StartInterval (4212)
#    * StandardOutput and Error is /dev/null
#
#

safari_plist_base="/Applications/Safari.app/Contents/Info"
firefox_plist_base="/Applications/Firefox.app/Contents/Info"
#safari_plist_base="`pwd`/test-Info"
macosx_environment_plist="${HOME}/.MacOSX/environment"
#macosx_environment_plist="${HOME}/work/RemoveFlashback/test-environment"

if [ "$1" = "scanonly" ]; then
    scanonly=true
    echo "------- Scanonly mode"
else
    scanonly=false
    echo "------- Quarantine mode"
fi

files_to_quarantine=/tmp/FS_to_quarantine
rm -f $files_to_quarantine

quarantine_later() {
    echo "$1" >> $files_to_quarantine
}

quarantine() {
    $scanonly && return
    if ! [ -e "$files_to_quarantine" ] ; then
        echo "Nothing to quarantine."
        return
    fi
    echo "Quarantining files:"
    sort $files_to_quarantine | uniq | \
    xargs -I % mv -v % "$quar_dir"
    rm -f $files_to_quarantine
}

zip_quarantine() {
    zipfile="${HOME}/flashback_quarantine.zip"
    zip -rq -e --password infected $zipfile ${quar_dir}
    rm -rf "${quar_dir}"
    chown $USER $zipfile
    echo "Quarantined files stored in password-protected zip file $zipfile"
}

check_launchagent()
{
    for plist in ~/Library/LaunchAgents/* ; do 
        prop=`echo $plist | sed -e 's/\.plist$//'`
        pathname=`defaults read $prop ProgramArguments 2> /dev/null | head -2 | tail -1 | sed -e 's/.*"\(.*\)".*/\1/'`
        if [ -z "$pathname" ] ; then
            continue
        fi
        filename=`basename "$pathname"`
        if echo $filename | grep -q '^\.' ; then
            infected=1
	    if $scanonly; then
	       	echo "File $filename is most likely Flashback"
	    else
	       	echo "File $filename is most likely Flashback -- removing"
           	quarantine_later "$pathname"
                quarantine_later "$plist"
	    fi
        fi
    done
}

file_is_signed() {
    codesign -R="anchor trusted" -v "$1" 2> /dev/null
}

check_libraries()
{
    plist_base="$1"
    libraries="$2"
    
    rm -f /tmp/infected
    grep -a -o '__ldpath__[ -~]*' "${libraries}" | while read line; do
        ldpath=${line/#__ldpath__/}
        if [ -f "$ldpath" ]; then
            if file_is_signed "$ldpath" ; then
                echo "Skipping signed binary $ldpath"
                continue
            fi
            if [ -x "$ldpath" ] ; then
                echo "Infected file (check 1): ${ldpath}"
                $scanonly || quarantine_later "${ldpath}"
                touch /tmp/infected
            fi
        fi
    done
    if [ -f /tmp/infected ] ; then
        infected=1
        rm /tmp/infected
    fi
    string1="slfp"
    string2="cfinh"
    
    if grep -q "${string1}" "$libraries" && grep -q "${string2}" "$libraries" ; then
        if file_is_signed "$libraries" ; then
            echo "File $libraries matches pattern (check 2) but has a valid code signature -- skipping"
            return
        fi
        echo "Infected file (check 2): $libraries"
        $scanonly || quarantine_later "$libraries"
        infected=1
    fi
    
}

check_browser_plist(){
    local plist_base=$1
    
    if ! [ -e "${plist_base}.plist" ]; then return 0; fi
    if ! defaults read "${plist_base}" LSEnvironment > "${tmpfilename_base}.plist" 2>/dev/null ]; then
        return 0
    fi
    libraries=$(defaults read "${tmpfilename_base}" "DYLD_INSERT_LIBRARIES")
    if [ $? -eq 0 ]; then
        check_libraries "${plist_base}" "$libraries"
    fi
    echo "Found DYLD_INSERT_LIBRARIES in ${plist_base}.plist LSEnvironment: $libraries"
    infected=1
    if ! $scanonly; then
        echo "Removing..."
        sudo cp -vp "${plist_base}.plist" "${quar_dir}"
        sudo defaults delete "${plist_base}" LSEnvironment
        sudo chmod 644 "${plist_base}.plist"
        if [ -f "$libraries" ] ; then
            quarantine_later "$libraries"
        fi
    fi
}

dont_do() {
    echo "NOT DOING: $@"
}


check_macosx_environment_plist()
{
    local plist_base=$1
    
    if ! [ -e "${plist_base}.plist" ]; then return 0; fi
    libraries=$(defaults read "${plist_base}" "DYLD_INSERT_LIBRARIES" 2>/dev/null)
    if [ "$?" -ne "0" ]; then
        return 0
    fi
    check_libraries "${plist_base}" "$libraries"
    echo "Found DYLD_INSERT_LIBRARIES in ${plist_base}.plist: $libraries"
    if ! $scanonly;then
        echo "Removing..."
        cp -vp "${plist_base}.plist" "${quar_dir}"
        defaults delete "${plist_base}" DYLD_INSERT_LIBRARIES
        [ -f ${plist_base}.plist ] && chown $USER ${plist_base}.plist
        if [ -f "$libraries" ] ; then
            quarantine_later "$libraries"
            infected=1
        fi
    fi
}

global_dyld=`launchctl getenv DYLD_INSERT_LIBRARIES`
if [ -n "$global_dyld" ]; then
    infected=1
    echo "Found DYLD_INSERT_LIBRARIES in launchctl environment ($global_dyld)."
    if ! $scanonly; then
        echo "Removing..."
        launchctl unsetenv DYLD_INSERT_LIBRARIES
    fi
fi

check_launchagent
check_browser_plist "${safari_plist_base}"
if ! $scanonly; then
    touch /Applications/Safari.app
fi
check_browser_plist "${firefox_plist_base}"
if ! $scanonly; then
    [ -d /Applications/Firefox.app ] && touch /Applications/Firefox.app
fi
check_macosx_environment_plist "${macosx_environment_plist}"

quarantine

if [ -f "${tmpfilename_base}.plist" ]; then
    rm "${tmpfilename_base}.plist"
fi

if [ "$infected" -eq "0" ];then
    echo "No Flashback malware found."
else
    if ! $scanonly; then
    	zip_quarantine
    	if "$0" scanonly; then
    	    echo "Your system has been cleaned."
            infected=0
    	else
    	    echo "There has been a problem and your system is still infected."
            infected=2
    	fi
    fi
fi

exit $infected


