/*------------------------------------------------------------------------------*
 * File Name: OCN_c01.h															*
 * Creation: TCZ 5/22/2001														*
 * Purpose: Origin C Header for NAG functions									*
 * Copyright (c) OriginLab Corp.	2001										*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/

#ifndef _O_NAG_C06_H
#define _O_NAG_C06_H

//#importdll "ONAG" // NAG DLL prepared by OriginLab
#pragma dll(ONAG)

#include <NAG\nag_types.h>

/*	begin proto */

/**	c06eac
		calculates the discrete Fourier transforms of a sequence of n real data values.

Example1:
	This piece of code reads in a sequence of real data values from a worksheet column, and
	prints their discrete Fourier transform (as computed by nag fft real), after expanding
	it from Hermitian form into a full complex sequence.
	Assume "Data1" Worksheet has 3 columns, the first column contain 7 data, and we want to 
	put result complex form real part in the second column and imag part in the third column.
	
	int n = 7, j, success, n2, nj;
	//Attach two Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset aa("data1",1);
	Dataset bb("data1",2);
	//Because Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, a = aa, b = bb;
	success = nag_fft_real(n, x);
	
	if(success == 0)
	{
		a[0] = x[0];
		b[0] = 0.0;
		n2 = (n-1)/2;
		for (j = 1; j<=n2; j++)
		{
			nj = n - j;
			a[j] = x[j];
			a[nj] = x[j];
			b[j] = x[nj];
			b[nj] = -x[nj];
		}
		if (n % 2==0)
		{
			a[n2+1] = x[n2+1];
			b[n2+1] = 0.0;
		}
	}
	//write the result to worksheet.
	aa = a;
	bb = b;
	
		
Example2:
	This program reads in a sequence of real data values, and prints their discrete 
	Fourier transform (as computed by nag fft real), after expanding it from 
	Hermitian form into a full complex sequence.  It then performs an inverse 
	transform using nag conjugate hermitian (c06gbc) and nag fft hermitian (c06ebc), 
	and prints the sequence so obtained alongside the original data values.
	
void test_nag_fft_real()
{

	int j, n = 7, n2, nj;
	double a[20], b[20], xx[20];
	double x[20] = {0.34907, 0.54890, 0.74776, 0.94459, 1.13850, 1.32850, 1.51370};
	int success;
	
	for (j = 0; j<n; j++)
		xx[j] = x[j];
	
	success = nag_fft_real(n, x);
	if(success == 0)
	{
		a[0] = x[0];
		b[0] = 0.0;
		n2 = (n-1)/2;
		for (j = 1; j<=n2; j++)
		{
			nj = n - j;
			a[j] = x[j];
			a[nj] = x[j];
			b[j] = x[nj];
			b[nj] = -x[nj];
		}
		if (n % 2==0)
		{
			a[n2+1] = x[n2+1];
			b[n2+1] = 0.0;
		}
		printf("\nComponents of discrete Fourier transform\n");
		printf("\n Real Imag \n\n");
		for (j = 0; j<n; j++)
			printf("%3ld %10.5f %10.5f\n", j, a[j], b[j]);
		
		nag_conjugate_hermitian(n, x);
		nag_fft_hermitian(n, x);
		printf("\nOriginal sequence as restored by inverse transform\n");
		printf("\n Original Restored\n\n");
		for (j = 0; j<n; j++)
			printf("%3ld %10.5f %10.5f\n", j, xx[j], x[j]);
	}
	else
	printf("Function c06eac has some problems.");
}

	The output is following:
	
	Components of discrete Fourier transform

		 	 Real 		 Imag
		 
	0 		 2.48361 	 0.00000
	1 		-0.26599 	 0.53090
	2 		-0.25768 	 0.20298
	3 		-0.25636 	 0.05806
	4 		-0.25636 	-0.05806
	5 		-0.25768 	-0.20298
	6 		-0.26599 	-0.53090

	Original sequence as restored by inverse transform

			Original 		Restored

	0 		0.34907 		0.34907
	1 		0.54890 		0.54890
	2 		0.74776 		0.74776
	3 		0.94459 		0.94459
	4 		1.13850 		1.13850
	5 		1.32850 		1.32850
	6 		1.51370 		1.51370	
	
Return:
	The function returns NAG error code or 0 if no error.
	
	137: At least one of the prime factors of n is greater than 19.
	138: n has more than 20 prime factors.
	12:  On entry, n must not be less than or equal to 1: n = _value_.

	successfully call of the nag_fft_real function.	
*/

	int nag_fft_real(
		int n, 	//the number of data values.
		double x[] // Input: contained in x[j], for j = 0, 1, . . . , n - 1.  Output: the discrete Fourier transform stored in Hermitian form. 
  ); // Given a sequence of n real data values xj, for j = 0, 1, . . . , n - 1, this function calculates their discrete Fourier transform.


/** c06ebc
		calculates the discrete Fourier transforms of a Hermitian sequence of n.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contain 7 data.  This piece of code reads
	in a sequence of first column data values and we want to put the result which is the discrete 
	Fourier transforms of a Hermitian sequence of nin the second column.
	
	int n = 7, success;
	//Attach two Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset aa("data1",1);
	//Because Dataset cannot be the parameter of function, but vector can be.
	vector x = xx;
	success = nag_fft_hermitian(n, x);
	//put the result, the discrete Fourier transforms to "data1" second column.
	if(success == 0)
		aa = x;
	
	
Example2:
	This program reads in a sequence of real data values which is assumed to be a Hermitian sequence of
	complex data values stored in Hermitian form. The input sequence is expanded into a full complex
	sequence and printed alongside the original sequence. The discrete Fourier transform (as computed
	by nag_fft_hermitian) is printed out.  The program then performs an inverse transform using 
	nag_fft_real (c06eac) and nag_conjugate_hermitian (c06gbc), and prints the sequence so obtained 
	alongside the original data values.

void test_nag_fft_hermitian()
{
	int j, n, n2, nj;
	double u[20], v[20], xx[20];
	double x[20] = {0.34907, 0.54890, 0.74776, 0.94459, 1.13850, 1.32850, 1.51370};
	int success;
	n  = 7;
	
	for (j = 0; j<n; j++)
		xx[j] = x[j];

	u[0] = x[0];
	v[0] = 0.0;
	n2 = (n-1)/2;
	for (j = 1; j<=n2; j++)
	{
		nj = n - j;
		u[j] = x[j];
		u[nj] = x[j];
		v[j] = x[nj];
		v[nj] = -x[nj];
	}
	if (n % 2==0)
	{
		u[n2+1] = x[n2+1];
		v[n2+1] = 0.0;
	}
	
	printf("\nOriginal and corresponding complex sequence\n");
	printf("\n Data Real Imag \n\n");
	for (j = 0; j<n; j++)
		printf("%3ld %10.5f %10.5f %10.5f\n", j, x[j], u[j], v[j]);
		
	success = nag_fft_hermitian(n, x);
	
	if(success == 0)
	{
		printf("\nComponents of discrete Fourier transform\n\n");
		for (j = 0; j<n; j++)
			printf("%3ld %10.5f\n", j, x[j]);
			
		nag_fft_real(n, x);
		
		nag_conjugate_hermitian(n, x);
		
		printf("\nOriginal sequence as restored by inverse transform\n");
		printf("\n Original Restored\n\n");
		for (j = 0; j<n; j++)
			printf("%3ld %10.5f %10.5f\n", j, xx[j], x[j]);
	}
	else
		printf("function c06ebc has some problem\n");
}	
	
	The output is following:
	
	Original and corresponding complex sequence

		Data 		Real 		 Imag

	0 	0.34907 	0.34907 	 0.00000
	1 	0.54890 	0.54890 	 1.51370
	2 	0.74776 	0.74776 	 1.32850
	3 	0.94459 	0.94459  	 1.13850
	4 	1.13850	 	0.94459 	-1.13850
	5 	1.32850 	0.74776 	-1.32850
	6 	1.51370 	0.54890 	-1.51370

	Components of discrete Fourier transform

	0	 1.82616
	1 	 1.86862
	2 	-0.01750
	3 	 0.50200
	4 	-0.59873
	5 	-0.03144
	6 	-2.62557

	Original sequence as restored by inverse transform

		Original 	Restored
		
	0 	0.34907 	0.34907
	1 	0.54890 	0.54890
	2 	0.74776 	0.74776
	3 	0.94459 	0.94459
	4 	1.13850 	1.13850
	5 	1.32850 	1.32850
	6 	1.51370 	1.51370


Return:
	The function returns NAG error code or 0 if no error.

	137: At least one of the prime factors of n is greater than 19.
	138: n has more than 20 prime factors.
	12:  On entry, n must not be less than or equal to 1: n = _value_.

	successfully call of the function nag_fft_hermitian function.
	
*/

	int nag_fft_hermitian(
		int n,	// the number of data values.
		double x[] // Input:the number of data values.  Output: the components of the discrete Fourier transforms. 
  ); // Given a Hermitian sequence of n complex data values z[j] (i.e., a sequence such that z0 is real and z(n-j) is the complex conjugate of z[j], for j = 1, 2, . . . , (n - 1), this function calculates their discrete Fourier transform.


/**	c06ecc
		calculates the discrete Fourier transforms of a sequence of n complex data values.	

Example1:
	Assume "Data1" Worksheet has 4 columns, the first two columns contain 7 data each.  The first
	column is the real part of the original complex data and second column is the imag part of 
	original complex data.  This piece of code reads in a sequence of these 7 complex data values
	and put the result which is the discrete Fourier transforms of the original complex data values
	to the third and fourth column.  The third column is the real part and the fourth column is the
	imag part.
	
	int n = 7, success;
	//Attach two Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	Dataset aa("data1",2);
	Dataset bb("data1",3);
	//Because Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, y = yy;
	success = nag_fft_complex(n, x, y);
	//put the result back to worksheet "data1", the third column and the fourth column.
	aa = x;
	bb = y;
	
	
Example2:
	This program reads in a sequence of complex data values and prints their discrete 
	Fourier transform.  It then performs an inverse transform using nag_conjugate_complex (c06gcc)
	and nag fft complex, and prints the sequence so obtained alongside the original data values.
	
void test_nag_fft_complex()
{	
	#define NMAX 20
	int j, n ;
	int success;
	double xx[20], yy[20];
	double x[20] = {0.34907, 0.54890, 0.74776, 0.94459, 1.13850, 1.32850, 1.51370};
	double y[20] = {-0.37168, -0.35669, -0.31175, -0.23702, -0.13274, 0.00074, 0.16298}; 	
 	n = 7;

	for (j = 0; j<n; ++j)
	{
		xx[j] = x[j];
		yy[j] = y[j];
	}

	success = nag_fft_complex(n, x, y);

	printf("\nComponents of discrete Fourier transforms\n\n");
	printf(" Real Imag\n\n");
	for (j = 0; j<n; ++j)
		printf("%3ld %10.5f %10.5f\n", j, x[j], y[j]);

	nag_conjugate_complex(n, y);

	nag_fft_complex(n, x, y);
	
	nag_conjugate_complex(n, y);
	
	printf("\nOriginal sequence as restored byinverse transform\n");
	printf("\n Original Restored\n");
	printf(" Real Imag Real Imag\n\n");
	for (j = 0; j<n; ++j)
		printf("%3ld %10.5f %10.5f %10.5f %10.5f\n", j, xx[j], yy[j], x[j], y[j]);	
}

	The output is following:
	
	Components of discrete Fourier transform

		 Real 			 Imag
		 
	0 	 2.48361 		-0.47100
	1 	-0.55180 	 	 0.49684
	2 	-0.36711 		 0.09756
	3 	-0.28767 		-0.05865
	4 	-0.22506 		-0.17477
	5 	-0.14825 		-0.30840
	6 	 0.01983 		-0.56496
	
	Original sequence as restored byinverse transform
		     
		      Original 				  Restored
		Real 		  Imag 		  Real 		  Imag
		
	0 	0.34907 	-0.37168 	0.34907 	-0.37168
	1 	0.54890 	-0.35669 	0.54890 	-0.35669
	2 	0.74776 	-0.31175 	0.74776 	-0.31175
	3 	0.94459 	-0.23702 	0.94459 	-0.23702
	4 	1.13850 	-0.13274 	1.13850  	-0.13274
	5 	1.32850 	 0.00074 	1.32850 	 0.00074
	6 	1.51370 	 0.16298 	1.51370 	 0.16298

Return:
	The function returns NAG error code or 0 if no error.
	
	137: At least one of the prime factors of n is greater than 19.
	138: n has more than 20 prime factors.
	
	successfully call of the nag_fft_complex function.	
*/

	int nag_fft_complex(
		int n,	//the number of data values  
		double x[],	// Input:contained in x[j] , the real part of z[j], for j = 0, 1, . . ., n . 1.  Output: the real parts a[k] of the components of the discrete Fourier transform. a[k] is contained in x[k], for k = 0, 1, . . ., n . 1.   
		double y[]	// Input: contain y[j] , the imaginary part of z[j], for j = 0, 1, . . . , n . 1.  Output: the imaginary parts b[k] of the components of the discrete Fourier transform. b[k] is contained in y[k], for k = 0, 1, . . ., n . 1.
  ); // Given a sequence of n complex data values z[j], for j = 0, 1, . . . , n . 1, this function calculates their discrete Fourier transform.

    
/**	c06ekc
		calculates the circular convolution or correlation of two real vectors of period n.
Example1:
	Assume "Data1" Worksheet has 4 columns, the first two columns contain 9 data each.  This piece 
	of code calculates the circular convolution or correlation of two real vectors of period 9 and
	put the result the circular convolution to the third column and correlation to the fourth column.

	int n = 9;
	//Attach four Datasets to these 4 columns
	Dataset dxa("data1",0);
	Dataset dya("data1",1);
	Dataset dxb("data1",2);
	Dataset dyb("data1",3);
	//Dataset cannot be the parameter of function, but vector can be.
	vector xa = dxa, ya = dya, xb = dxa, yb = dya;
	
	nag_convolution_real(NAG_Convolution, n, xa, ya);
	nag_convolution_real(Nag_Correlation, n, xb, yb);
	
	//put the result Convolution to the worksheet "data1" third column and Correlation
	//to the fourth column.
	dxb = xa;
	dyb = xb;
	
		
Example2:
	This program reads in the elements of one period of two real vectors x and y and 
	prints their discrete convolution and correlation (as computed by nag_convolution_real). 
	In realistic computations the number of data values would be much larger.
	
void test_nag_convolution_real()
{
	int j, n;
	double xb[64], yb[64];
	double xa[64] = {1.0, 1.0, 1.0, 1.0, 1.0, 0.0, 0.0, 0.0, 0.0};
	double ya[64] = {0.5, 0.5, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0, 0.0};
	n = 9;
	
	for (j = 0; j<n; ++j)
	{
		xb[j]= xa[j];
		yb[j]= ya[j];
	}
	
	nag_convolution_real(Nag_Convolution, n, xa, ya);
	
	nag_convolution_real(Nag_Correlation, n, xb, yb);
	
	printf("\n Convolution Correlation\n\n");
	for (j = 0; j<n; ++j)
		printf("%5ld %13.5f %13.5f\n", j, xa[j], xb[j]);	
}

	The output is following:
	
	 	Convolution 	Correlation
	
	0 	0.50000 		2.00000
	1 	1.00000 		1.50000
	2 	1.50000 		1.00000
	3 	2.00000 		0.50000
	4 	2.00000 		0.00000
	5 	1.50000 		0.50000
	6 	1.00000 		1.00000
	7 	0.50000 		1.50000
	8 	0.00000 		2.00000

Return:
	The function returns NAG error code or 0 if no error.
	137: At least one of the prime factors of n is greater than 19.
	138: n has more than 20 prime factors.
	12:  On entry, n must not be less than or equal to 1: n = _value_.
	70:  On entry, parameter operation has an illegal value.

	successfully call of the nag_convolution_real function.	
*/

	int nag_convolution_real(
		Nag_VectorOp operation, //the computation to be performed.
		int n, // the number of values.
		double x[],// Input:the elements of one period of the vector x. Output:the corresponding elements of the discrete convolution or correlation
		double y[]  // Input:the elements of one period of the vector y. Output:the discrete Fourier transform of the convolution 
  ); // calculates the circular convolution or correlation of two real vectors of period n.


/**	c06fpc
		computes the discrete Fourier transforms of m sequences, and each contains n real data values.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contain 18 data. This piece of code 
	computes the discrete Fourier transforms of 3 sequences and each contains 6 data values.  The
	result will be put in the second column (Discrete Fourier transforms in Hermitian format). 
	
	double trig[40];
	int m = 3, n = 6;
	//Attach two Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx;	
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_real(m, n, x, trig);
	//put the result to worksheet "data1" the second column.	
	yy = x;

Example2:
	This program reads in sequences of real data values and prints their discrete Fourier
	transforms (as computed by nag_fft_multiple real). The Fourier transforms are expanded
	into full complex form using nag_multiple_hermitian to complex (c06gsc) and printed. Inverse
	transforms are then calculated by calling nag_multiple _conjugate_hermitian (c06gqc) followed by
	nag_fft_multiple_hermitian (c06fqc) showing that the original sequences are restored.
	
void test_nag_fft_multiple_real()
{
	#define MMAX 5
	#define NMAX 20
	double trig[40];
	int i, j, m, n;
	double u[100], v[100];
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m =3;
	n =6;

	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	
	nag_fft_init_trig(n, trig); 

	nag_fft_multiple_real(m, n, x, trig);
	printf("\nDiscrete Fourier transforms in Hermitian format\n\n");
	
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n");
	}
	
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	printf("\nFourier transforms in full complex form\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", u[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}				
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", v[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}

	nag_multiple_conjugate_hermitian(m, n, x);
	nag_fft_multiple_hermitian(m, n, x, trig);
	printf ("\nOriginal data as restored by inverse transform\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 &&  i!= n-1 )
		 		printf("\n " );
		}
		printf("\n");
	}
}

	The output is following:
	
	m = 3 n = 6
	
	Original data values
	
			0.3854 	  0.6772 	 0.1138 	 0.6751 	 0.6362 	 0.1424
			0.5417 	  0.2983 	 0.1181 	 0.7255 	 0.8638 	 0.8723
			0.9172 	  0.0644 	 0.6037 	 0.6430 	 0.0428 	 0.4815
	
	Discrete Fourier transforms in Hermitian format
	
			1.0737 	 -0.1041 	 0.1126 	-0.1467 	-0.3738 	-0.0044
			1.3961 	 -0.0365 	 0.0780 	-0.1521 	-0.0607 	 0.4666
			1.1237 	  0.0914 	 0.3936 	 0.1530 	 0.3458 	-0.0508
			
	Fourier transforms in full complex form
	
	Real 	1.0737 	 -0.1041 	 0.1126 	-0.1467 	 0.1126 	-0.1041
	Imag 	0.0000 	 -0.0044 	-0.3738 	 0.0000 	 0.3738  	 0.0044
	
	Real 	1.3961 	 -0.0365 	 0.0780 	-0.1521 	 0.0780 	-0.0365
	Imag 	0.0000 	  0.4666 	-0.0607 	 0.0000 	 0.0607 	-0.4666
	
	Real 	1.1237 	  0.0914 	 0.3936 	 0.1530 	 0.3936 	 0.0914
	Imag 	0.0000 	 -0.0508 	 0.3458 	 0.0000  	-0.3458 	 0.0508
	
	Original data as restored by inverse transform
	
			0.3854 	  0.6772 	 0.1138 	 0.6751 	 0.6362 	 0.1424
			0.5417 	  0.2983 	 0.1181 	 0.7255 	 0.8638 	 0.8723
			0.9172 	  0.0644 	 0.6037 	 0.6430 	 0.0428 	 0.4815

Return:
	The function returns NAG error code or 0 if no error.
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	73:  Memory allocation failed.

	successfully call of the nag_fft_multiple_real function.	
*/
	
	int nag_fft_multiple_real(
		int m, // the number of sequences to be transformed.
		int n, // the number of real values in each sequence.
		double x[],// Input:the m data sequences must be stored in x consecutively.  Output: the m discrete Fourier transforms in Hermitian form, stored consecutively, overwrite the corresponding original sequences. 
		const double trig[]	//trigonometric coefficients are returned by a call of nag_fft_init_trig (c06gzc).
  ); // Given m sequences of n real data values x[p][j], for j = 0, 1, . . . , n - 1; p = 1, 2, . . .,m, this function simultaneously calculates the Fourier transforms of all the sequences.
 
  
/**	c06fqc
		computes the discrete Fourier transforms of m Hermitian sequences, each containing n complex data values.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contain 18 data. This piece of code 
	computes the discrete Fourier transforms of 3 sequences and each contains 6 data values.  The
	result will be put in the second column (Discrete Fourier transforms in Hermitian format). 
	
	double trig[40], u[40], v[40];
	int m = 3, n = 6;
	//Attach two Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx;
	//expand the sequences into full complex form.	
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	nag_fft_init_trig(n, trig);
	nag_fft_multiple_hermitian(m, n, x, trig);
	//put the result to worksheet "data1" the second column.
	yy = x;
		
Example2:
	This program reads in sequences of real data values which are assumed to be Hermitian sequences
	of complex data stored in Hermitian form. The sequences are expanded into full complex form
	using nag_multiple_hermitian_to_complex (c06gsc) and printed. The discrete Fourier transforms
	are then computed (using nag_fft_multiple_hermitian) and printed out. Inverse transforms are then
	calculated by calling nag_fft_multiple_real (c06fpc) followed by nag_multiple_conjugate_hermitian
	(c06gqc) showing that the original sequences are restored.
	
void test_nag_fft_multiple_hermitian()
{
	#define MMAX 5
	#define NMAX 20
	double trig[40];
	int i, j, m, n;
	double u[100], v[100];
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m = 3;
	n = 6;
	
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	printf("\nOriginal data written in full complex form\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", u[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}		
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", v[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}

	nag_fft_init_trig(n, trig);
	nag_fft_multiple_hermitian(m, n, x, trig);
	printf ("\nDiscrete Fourier transforms (real values)\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 &&  i!= n-1 )
		 		printf("\n " );
		 }
		printf("\n");
	}	
	
	nag_fft_multiple_real(m, n, x, trig);
	nag_multiple_conjugate_hermitian(m, n, x);
	printf ("\nOriginal data as restored by inverse transform\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 &&  i!= n-1 )
		 		printf("\n " );
		}
		printf("\n");
	}	
}

	The output is following:
	
	m = 3 n = 6
	
	Original data values
	
			0.3854 		0.6772 		0.1138 		0.6751 		0.6362 	 	 0.1424
			0.5417 		0.2983 		0.1181 		0.7255 		0.8638 	 	 0.8723
			0.9172 		0.0644 		0.6037 		0.6430 		0.0428 	 	 0.4815
			
	Original data written in full complex form
	
	Real 	0.3854 		0.6772 		0.1138 		0.6751		0.1138 	 	 0.6772
	Imag 	0.0000 		0.1424 		0.6362 		0.0000	   -0.6362  	-0.1424
	Real 	0.5417 		0.2983 		0.1181 		0.7255		0.1181 	 	 0.2983
	Imag 	0.0000 		0.8723 		0.8638 		0.0000	   -0.8638 		-0.8723
	Real 	0.9172 		0.0644 		0.6037 		0.6430		0.6037 	 	 0.0644
	Imag 	0.0000 		0.4815 		0.0428 		0.0000	   -0.0428 		-0.4815
	
	Discrete Fourier transforms (real values)
	
			1.0788 		0.6623 	   -0.2391 	   -0.5783 	  	0.4592 		-0.4388
			0.8573 		1.2261 		0.3533 	   -0.2222 	 	0.3413 		-1.2291
			1.1825 		0.2625 		0.6744 		0.5523  	0.0540 		-0.4790
			
	Original data as restored by inverse transform
	
			0.3854 		0.6772 		0.1138 		0.6751 		0.6362 		 0.1424
			0.5417 		0.2983 		0.1181 		0.7255 		0.8638 		 0.8723
			0.9172 		0.0644 		0.6037 		0.6430 		0.0428 		 0.4815
			
Return:
	The function returns NAG error code or 0 if no error.

	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	73:  Memory allocation failed.

	successfully call of the nag_fft_multiple_hermitian function.	
*/

	int nag_fft_multiple_hermitian(
		int m, //the number of sequences to be transformed
		int n, //the number of data values in each sequence
		double x[],	// Input: the m Hermitian sequences must be stored consecutively in x in Hermitian form.  Output: the components of the m discrete Fourier transforms, are stored consecutively.
		const double trig[]	//trigonometric coefficients are returned by a call of nag_fft_init_trig (c06gzc).
); // computes the discrete Fourier transforms of m Hermitian sequences, each contains n complex data values.


/**	c06frc
		computes the discrete Fourier transforms of m sequences, each containing n complex data values.
	
Example1:
	Assume "Data1" Worksheet has 4 columns, the first two columns contain 18 data each.  This piece 
	of code computes the discrete Fourier transforms of 3 sequences, each containing 6 complex data 
	values. and put the result to the third column (real part) and the fourth column (imag part).
	
	double trig[40];
	int m = 3, n = 6;
	//Attach four Datasets to these 4 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	Dataset dx("data1",2);
	Dataset dy("data1",3);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, y = yy;
	nag_fft_init_trig(n, trig);	
	nag_fft_multiple_complex(m, n, x, y, trig);
	//Put the result to worksheet "data1" the third and fourth column.
	dx = x;
	dy = y;
	
	
Example2:
	This program reads in sequences of complex data values and prints their discrete Fourier
	transforms (as computed by nag_fft_multiple_complex). Inverse transforms are then calculated
	using nag_conjugate_complex (c06gcc) and nag_fft_multiple_complex and printed out, showing that
	the original sequences are restored.

void test_nag_fft_multiple_complex()
{
	#define MMAX 5
	#define NMAX 20
	double trig[40];
	int i, j, m, n;
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
	 				 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815,
	 				 0.1156, 0.0685, 0.2060, 0.8630, 0.6967, 0.2792};
	
	double y[100] = {0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9089, 0.3118, 0.3465, 0.6198, 0.2668, 0.1614,
					 0.6214, 0.8681, 0.7060, 0.8652, 0.9190, 0.3355};
	m = 3;
	n = 6;
	
	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}			
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", y[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}

	nag_fft_init_trig(n, trig);
	
	nag_fft_multiple_complex(m, n, x, y, trig);
	printf("\nDiscrete Fourier transforms\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}		
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", y[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}

	nag_conjugate_complex(m*n, y);
	nag_fft_multiple_complex(m, n, x, y, trig);
	nag_conjugate_complex(m*n, y );
	
	printf ("\nOriginal data as restored by inverse transform\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}		
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", y[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}
}

	The output is following:
	
	m = 3 n = 6
	
	Original data values
	
	Real 	0.3854 	 0.6772 	0.1138 	 0.6751 	0.6362 	 0.1424
	Imag 	0.5417 	 0.2983 	0.1181 	 0.7255 	0.8638 	 0.8723
	
	Real 	0.9172 	 0.0644 	0.6037   0.6430 	0.0428 	 0.4815
	Imag 	0.9089 	 0.3118 	0.3465 	 0.6198 	0.2668 	 0.1614
	
	Real 	0.1156 	 0.0685 	0.2060 	 0.8630 	0.6967 	 0.2792
	Imag 	0.6214 	 0.8681 	0.7060 	 0.8652 	0.9190 	 0.3355
	
	Discrete Fourier transforms
	
	Real 	1.0737 	-0.5706 	0.1733 	-0.1467 	0.0518 	 0.3625
	Imag 	1.3961 	-0.0409    -0.2958 	-0.1521 	0.4517 	-0.0321
	
	Real 	1.1237 	 0.1728 	0.4185 	 0.1530 	0.3686 	 0.0101
	Imag 	1.0677 	 0.0386 	0.7481 	 0.1752 	0.0565 	 0.1403
	
	Real 	0.9100 	-0.3054 	0.4079 	-0.0785    -0.1193 	-0.5314
	Imag 	1.7617 	 0.0624    -0.0695 	 0.0725 	0.1285 	-0.4335
	
	Original data as restored by inverse transform
	
	Real 	0.3854 	 0.6772 	0.1138 	 0.6751 	0.6362 	 0.1424
	Imag 	0.5417 	 0.2983 	0.1181 	 0.7255 	0.8638 	 0.8723
	
	Real 	0.9172 	 0.0644 	0.6037 	 0.6430 	0.0428 	 0.4815
	Imag 	0.9089 	 0.3118 	0.3465 	 0.6198 	0.2668 	 0.1614
	
	Real 	0.1156 	 0.0685 	0.2060 	 0.8630 	0.6967 	 0.2792
	Imag 	0.6214 	 0.8681 	0.7060 	 0.8652 	0.9190 	 0.3355
	
Return:

	The function returns NAG error code or 0 if no error.
	
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	73:  Memory allocation failed.
		
	successfully call of the nag_fft_multiple_complex function.	
*/

	int nag_fft_multiple_complex(
		int m, //the number of sequences to be transformed.
		int n, //the number of data values in each sequence.
		double x[], // Input: the real parts of the complex data.  Output: x is overwritten by the real part of the complex transforms.
		double y[], // Input: the imaginary parts of the complex data. Output: y is overwritten by the imaginary part of the complex transforms.
		const double trig[] //trigonometric coefficients are returned by a call of nag_fft_init_trig (c06gzc).
  ); // computes the discrete Fourier transforms of m sequences, each containing n complex data values.


/**	c06fuc
		computes the two-dimensional discrete Fourier transform of a bivariate sequence of complex data values
	
Example1:
	Assume "Data1" Worksheet has 4 columns, the first two columns contain 15 data each.  This piece 
	of code computes the two-dimensional discrete Fourier transform of a bivariate sequence of complex 
	data values. and put the result to the third column (real part) and the fourth column (imag part).
	
	int m = 3, n = 5;
	double trigm[5], trign[5];
	//Attach four Datasets to these 4 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	Dataset dx("data1",2);
	Dataset dy("data1",3);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, y = yy;
	nag_fft_init_trig(m, trigm);
	nag_fft_init_trig(n, trign);
	nag_fft_2d_complex(m, n, x, y, trigm, trign);
	//Put the result to worksheet "data1" the third and fourth column.
	dx = x;
	dy = y;
	
	
Example2:
	This program reads in a bivariate sequence of complex data values and prints the two-dimensional
	Fourier transform. It then performs an inverse transform and prints the sequence so obtained,
	which may be compared to the original data values.
	
void test_nag_ftt_2d_complex()
{	
	int i, j, m, n;
	double trigm[12], trign[12];
	double x[36] = {1.000, 0.999, 0.987, 0.936, 0.802,
					0.994, 0.989, 0.963, 0.891, 0.731,
					0.903, 0.885, 0.823, 0.694, 0.467}; 	
	double y[36] = {0.000, -0.040, -0.159, -0.352, -0.597,
					-0.111, -0.151, -0.268, -0.454, -0.682,
 					-0.430, -0.466, -0.568, -0.720, -0.884};
	m = 3;
	n = 5;
	
	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}		
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", y[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}
	
	nag_fft_init_trig(m, trigm);
	nag_fft_init_trig(n, trign);
	nag_fft_2d_complex(m, n, x, y, trigm, trign);
	
	printf("\nComponents of discrete Fourier transforms\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}				
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", y[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}
	
	nag_conjugate_complex(m*n, y);
	nag_fft_2d_complex(m, n, x, y,trigm, trign);
	nag_conjugate_complex(m*n, y);
	
	printf ("\nOriginal data as restored by inverse transform\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}				
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", y[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}
}

	The output is following:
	
	m = 3 n = 5
	
	Original data values
	
	Real 	1.0000 	 0.9990   	0.9870 	 0.9360 	0.8020
	Imag 	0.0000 	-0.0400    -0.1590 	-0.3520    -0.5970
	
	Real 	0.9940 	 0.9890 	0.9630 	 0.8910 	0.7310
	Imag   -0.1110 	-0.1510    -0.2680 	-0.4540    -0.6820
	
	Real 	0.9030 	 0.8850 	0.8230 	 0.6940 	0.4670
	Imag   -0.4300 	-0.4660    -0.5680 	-0.7200    -0.8840
	
	Components of discrete Fourier transforms
	
	Real 	3.3731 	 0.4814 	0.2507 	 0.0543    -0.4194
	Imag   -1.5187 	-0.0907 	0.1776 	 0.3188 	0.4145
	
	Real 	0.4565 	 0.0549 	0.0093 	-0.0217    -0.0759
	Imag 	0.1368 	 0.0317 	0.0389 	 0.0356 	0.0045
	
	Real   -0.1705 	-0.0375    -0.0423 	-0.0377    -0.0022
	Imag 	0.4927 	 0.0584 	0.0082  -0.0255    -0.0829
	
	Original data as restored by inverse transform
	
	Real 	1.0000 	 0.9990 	0.9870 	 0.9360 	0.8020
	Imag 	0.0000 	-0.0400    -0.1590 	-0.3520    -0.5970
	
	Real 	0.9940 	 0.9890 	0.9630 	 0.8910 	0.7310
	Imag   -0.1110 	-0.1510    -0.2680 	-0.4540    -0.6820
	
	Real 	0.9030 	 0.8850 	0.8230 	 0.6940 	0.4670
	Imag   -0.4300 	-0.4660    -0.5680 	-0.7200    -0.8840


Return:
	The function returns NAG error code or 0 if no error.
	
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trign array are incompatible or trign array is not initialized.  Value of m and trigm array are incompatible or trigm array is not initialized.
	73:  Memory allocation failed.
		
	successfully call of the nag_fft_2d_complex function.	
*/

	int nag_fft_2d_complex(
		int m, //the number of rows of the bivariate data sequence.
		int n, //: the number of columns of the bivariate data sequence.
		double x[], // Input: the real parts of the complex data.  Output: the real part of the corresponding elements of the computed transform.
		double y[],	// Input: the imaginary parts of the complex data.  Output: the imaginary part of the corresponding elements of the computed transform.
		const double trigm[],  //trigonometric coefficients are returned by calls of nag_fft_init_trig (c06gzc).
		const double trign[] //trigonometric coefficients are returned by calls of nag_fft_init_trig (c06gzc). 
  ); // computes the two-dimensional discrete Fourier transform of a bivariate sequence of complex data values.


/**	c06gbc
		forms the complex conjugate of a Hermitian sequence of n data values.

Example1:
	Assume "Data1" Worksheet has 3 columns, the first column contain 7 data.  This piece of code forms
	the complex conjugate of a Hermitian sequence of 7 data values. and put the result to the second 
	column (real part) and the third column (imag part).
	
	int j, n = 7, n2, nj;
	//Attach three Datasets to these 3 columns
	Dataset xx("data1",0);
	Dataset aa("data1",1);
	Dataset bb("data1",2);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, a = aa, b = bb;
	nag_fft_real(n, x);
	nag_conjugate_hermitian(n, x);
	a[0] = x[0];
	b[0] = 0.0;
	n2 = (n-1)/2;
	for (j = 1; j<=n2; j++)
	{
		nj = n - j;
		a[j] = x[j];
		a[nj] = x[j];
		b[j] = x[nj];
		b[nj] = -x[nj];
	}
	if (n % 2==0)
	{
		a[n2+1] = x[n2+1];
		b[n2+1] = 0.0;
	}
	//Put the result to the Worksheet "data1" the second and third column.
	aa = a;
	bb = b;
	
	
Example2:
	This program reads in a sequence of real data values,cal ls nag fft real (c06eac) followed by 
	nag_conjugate_hermitian to compute their inverse discrete Fourier transform, and prints this after
	expanding it from Hermitian form into a full complex sequence.
	
void test_nag_conjugate_hermitian()
{
	#define NMAX 20

	int j, n, n2, nj;
	double a[20], b[20];
	double x[20] = {0.34907, 0.54890, 0.74776, 0.94459,
					1.13850, 1.32850, 1.51370};
	n = 7;
	
	nag_fft_real(n, x);

	nag_conjugate_hermitian(n, x);
	a[0] = x[0];
	b[0] = 0.0;
	n2 = (n-1)/2;
	for (j = 1; j<=n2; j++)
	{
		nj = n - j;
		a[j] = x[j];
		a[nj] = x[j];
		b[j] = x[nj];
		b[nj] = -x[nj];
	}
	if (n % 2==0)
	{
		a[n2+1] = x[n2+1];
		b[n2+1] = 0.0;
	}
	
	printf("\nComponents of inverse discrete Fourier transform\n");
	printf("\n Real Imag \n\n");
	for (j = 0; j<n; j++)
		printf("%3ld %10.5f %10.5f\n", j, a[j], b[j]);
}

	The output is following:
	
	Components of inverse discrete Fourier transform
	
			 Real 		 Imag
			 
	0 		 2.48361 	 0.00000
	1 		-0.26599 	-0.53090
	2 		-0.25768 	-0.20298
	3 		-0.25636 	-0.05806
	4 		-0.25636 	 0.05806
	5 		-0.25768 	 0.20298
	6 		-0.26599 	 0.53090

Return:
	The function returns NAG error code or 0 if no error.
	
	11: On entry, n must not be less than 1: n = _value_.

	successfully call of the nag_conjugate_hermitian function.	
*/

	int nag_conjugate_hermitian(
		int n, // the number of data values.
		double x[]	// Input: the Hermitian sequence in Hermitian form.  Output: the imaginary parts y[j] are negated. The real parts x[j] are not referenced.
  ); // forms the complex conjugate of a Hermitian sequence of n data values.


/**	c06gcc 
		forms the complex conjugate of a sequence of n data values.

Example1:
	Assume "Data1" Worksheet has 4 columns, the first two columns contain 7 data each.  
	This piece of code forms the complex conjugate of a sequence of 7 data values. and 
	put the result to the third column (real part) and the fourth column (imag part).
	
	int j, n = 7;
	//Attach four Datasets to these 4 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	Dataset dx("data1",2);
	Dataset dy("data1",3);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx;
	nag_conjugate_complex(n, y);	
	nag_fft_complex(n, x, y);	
	nag_conjugate_complex(n, y);
	//Put the result to the Worksheet "data1" the third and fourth column.
	dx = x;
	dy = y;
	
		
Example2:
	This program reads in a sequence of complex data values and prints their inverse discrete Fourier
	transform as computed by calling nag_conjugate_complex, followed by nag_fft_complex (c06ecc) and
	nag_conjugate_complex again.

void test_nag_conjugate_complex()
{
	#define NMAX 20
	int j, n ;
	double x[20] = {0.34907, 0.54890, 0.74776, 0.94459,
					1.13850, 1.32850, 1.51370};	
	double y[20] = {-0.37168, -0.35669, -0.31175, -0.23702,
					-0.13274, 0.00074, 0.16298};
	n = 7;
	
	nag_conjugate_complex(n, y);
	
	nag_fft_complex(n, x, y);
	
	nag_conjugate_complex(n, y);
	printf("\nComponents of inverse discrete Fourier transform\n");
	printf("\n Real Imag\n\n");
	for (j = 0; j<n; ++j)
		printf("%3ld %10.5f %10.5f\n", j, x[j], y[j]);
}

	The output is following:
	
	Components of inverse discrete Fourier transform
	
	
			 Real 		 Imag
	0 		 2.48361 	-0.47100
	1 		 0.01983 	-0.56496
	2 		-0.14825 	-0.30840
	3 		-0.22506 	-0.17477
	4 		-0.28767 	-0.05865
	5 		-0.36711 	 0.09756
	6 		-0.55180 	 0.49684


Return:
	The function returns NAG error code or 0 if no error.
	
	11: On entry, n must not be less than 1: n = _value_.
		
	successfully call of the nag_conjugate_complex function.	
*/
	
	int nag_conjugate_complex(
		int n, //the number of data values.
		double y[] //Input:the imaginary part of the  data values.  Output: these values are negated.
  ); // forms the complex conjugate of a sequence of n data values. 


/**	c06gqc
		forms the complex conjugates of m Hermitian sequences, each containing n data values.

Example1:
	Assume "Data1" Worksheet has 4 columns, the first column contains 18 data.  This piece of 
	code forms the complex conjugate of 3 Hermitian sequences, each containing 6 data values. 
	The result was put to the second column (Conjugated data values) the third column (real part) 
	and the fourth column (imag part).
	
	int m = 3, n = 6;
	//Attach four Datasets to these 4 columns
	Dataset xx("data1",0);
	Dataset yy("data1",1);
	Dataset uu("data1",2);
	Dataset vv("data1",3);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, u = uu, v = vv;
	//Original data written in full complex form, u is real part, v is imag part.
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	nag_multiple_conjugate_hermitian(m, n, x);
	//Put the Conjugated data values to Worksheet "data1" the second column.
	yy = x;
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	//Put the Conjugated data written in full complex form to the third and fourth column.
	uu = u;
	vv = v;
	
	
Example2:
	This program reads in sequences of real data values which are assumed to be Hermitian sequences
	of complex data stored in Hermitian form. The sequences are expanded into full complex form
	using nag_multiple_hermitian_to_complex (c06gsc) and printed out. The sequences are then conjugated
	(using nag_multiple_conjugate_hermitian) and the conjugated sequences are expanded into complex
	form using nag_multiple_hermitian_to_complex (c06gsc) and printed out.
	
void test_nag_multiple_conjugate_hermitian()
{
	int i, j, m, n;
	double u[100], v[100];
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m = 3;
	n =6;
	printf("\n\nm = %2ld n = %2ld\n", m, n);
	
	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}

	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	printf("\nOriginal data written in full complex form\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", u[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
				
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", v[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}

	nag_multiple_conjugate_hermitian(m, n, x);
	printf("\nConjugated data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	
	printf("\nConjugated data written in full complex form\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", u[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}				
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", v[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}
}

	The output is following:
	
	m = 3 n = 6
	
	Original data values
	
	0.3854 0.6772 0.1138 0.6751 0.6362 0.1424
	0.5417 0.2983 0.1181 0.7255 0.8638 0.8723
	0.9172 0.0644 0.6037 0.6430 0.0428 0.4815
	
	Original data written in full complex form
	
	Real 	0.3854 	  0.6772 	0.1138 	  0.6751 	 0.1138 	 0.6772
	Imag 	0.0000 	  0.1424 	0.6362 	  0.0000 	-0.6362 	-0.1424
	
	Real 	0.5417 	  0.2983 	0.1181 	  0.7255 	 0.1181 	 0.2983
	Imag 	0.0000 	  0.8723 	0.8638 	  0.0000 	-0.8638 	-0.8723
	
	Real 	0.9172 	  0.0644 	0.6037 	  0.6430 	 0.6037 	 0.0644
	Imag 	0.0000 	  0.4815 	0.0428 	  0.0000 	-0.0428 	-0.4815
	
	Conjugated data values
	
			0.3854 	  0.6772    0.1138    0.6751 	-0.6362 	-0.1424
			0.5417 	  0.2983    0.1181    0.7255 	-0.8638 	-0.8723
			0.9172    0.0644    0.6037    0.6430 	-0.0428 	-0.4815
			
	Conjugated data written in full complex form
	
	Real 	0.3854 	  0.6772 	0.1138 	  0.6751 	 0.1138 	 0.6772
	Imag 	0.0000 	 -0.1424   -0.6362    0.0000 	 0.6362 	 0.1424
	
	Real 	0.5417 	  0.2983 	0.1181    0.7255     0.1181 	 0.2983
	Imag 	0.0000 	 -0.8723   -0.8638 	  0.0000 	 0.8638 	 0.8723
	
	Real 	0.9172 	  0.0644 	0.6037 	  0.6430 	 0.6037  	 0.0644
	Imag 	0.0000 	 -0.4815   -0.0428 	  0.0000 	 0.0428 	 0.4815

Return:
	The function returns NAG error code or 0 if no error.

	11: On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.

	successfully call of the nag_multiple_conjugate_hermitian function.	
*/

	int nag_multiple_conjugate_hermitian(
		int m, // the number of Hermitian sequences to be conjugated. 
		int n, // the number of data values in each Hermitian sequence.
		double x[] //Input:the m data sequences must be stored in x consecutively.  Output:the imaginary parts are negated.
   ); // forms the complex conjugates of m Hermitian sequences, each containing n data values.

       
/**	c06gsc
		takes m Hermitian sequences, each containing n data values, and forms 
		the real and imaginary parts of the m corresponding complex sequences.
	
Example1:
	Assume "Data1" Worksheet has 3 columns, the first column contains 18 data.  This piece of 
	code forms the complex conjugate of 3 Hermitian sequences, each containing 6 data values. 
	The result was put to the second column (real part) and the third column (imag part).
	
	int m = 3, n = 6;
	//Attach 3 Datasets to these 3 columns
	Dataset xx("data1",0);
	Dataset uu("data1",1);
	Dataset vv("data1",2);
	//Dataset cannot be the parameter of function, but vector can be.
	vector x = xx, u = uu, v = vv;
	// Convert Hermitian form to full complex 
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	//Put the Conjugated data written in full complex form to the third and fourth column.
	uu = u;
	vv = v;
		
Example2:
	This program reads in sequences of real data values which are assumed to be Hermitian 
	sequences of complex data stored in Hermitian form. The sequences are then expanded 
	into full complex form using_nag_multiple_hermitian_to_complex and printed.

void test_nag_multiple_hermitian_to_complex()
{
	#define MMAX 5
	#define NMAX 20

	int i, j, m, n;
	double u[100], v[100];
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m = 3;
	n = 6;

	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		 }
		printf("\n");
	}

	// Convert Hermitian form to full complex 
	
	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	printf("\nOriginal data written in full complex form\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", u[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}				
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", v[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}	
}

	The output is following:
	
	m = 3 n = 6
	
	Original data values
		
			0.3854 	  0.6772 	0.1138 	  0.6751 	0.6362 	  0.1424
			0.5417 	  0.2983 	0.1181 	  0.7255 	0.8638 	  0.8723
			0.9172 	  0.0644 	0.6037 	  0.6430 	0.0428 	  0.4815
		
	Original data written in full complex form
	
	Real 	0.3854 	  0.6772 	0.1138 	  0.6751  	0.1138 	  0.6772
	Imag 	0.0000 	  0.1424 	0.6362 	  0.0000   -0.6362 	 -0.1424
	
	Real 	0.5417 	  0.2983 	0.1181 	  0.7255 	0.1181 	  0.2983
	Imag 	0.0000 	  0.8723 	0.8638 	  0.0000   -0.8638   -0.8723
	
	Real 	0.9172 	  0.0644 	0.6037 	  0.6430 	0.6037 	  0.0644
	Imag 	0.0000 	  0.4815 	0.0428 	  0.0000   -0.0428 	 -0.4815
	
Return:
	The function returns NAG error code or 0 if no error.
	
	11: On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	
	successfully call of the nag_multiple_hermitian_to_complex function.	
*/

	int nag_multiple_hermitian_to_complex(
		int m, //the number of Hermitian sequences to be converted into complex form.
		int n,	//the number of data values, in each sequence.
		const double x[], //the data sequences must be stored in x consecutively. 
		double u[], //the real part of the m sequences of length n are stored consecutively in u.
		double v[]	//the imaginary part of the m sequences of length n are stored consecutively v.
  ); // takes m Hermitian sequences, each containing n data values, and forms the real and imaginary parts of the m corresponding complex sequences.


/**	c06gzc
		calculates the trigonometric coefficients required for the computation of discrete Fourier transforms.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contains 18 data.  This piece of 
	code calculates the trigonometric coefficients required for the computation of discrete 
	Fourier transforms. The result was put to the second column.	
	
	double trig[40];
	int m = 3, n = 6;
	//Attach 2 Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset dx("data1",1);
	x = xx;
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_real(m, n, x, trig);
	//Put the result to the second column.
	dx = x;
	
Example2:
	The program reads in 3 real data sequences and prints their discrete Fourier transforms in Hermitian
	format as calculated by nag_fft_multiple_real (c06fpc). A call is made to nag_fft_init_trig to initialise
	the array trig prior to calling nag_fft_multiple_real (c06fpc). The transforms are then printed out
	in full complex form after a call to nag_multiple_hermitian_to_complex (c06gsc).
	
void test_nag_fft_init_trig()
{
	
	double trig[40];
	int i, j, m, n;
	double u[100], v[100];
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m = 3;
	n = 6;

	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_real(m, n, x, trig);
	printf("\nDiscrete Fourier transforms in Hermitian format\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 6 )== 5 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}

	nag_multiple_hermitian_to_complex(m, n, x, u, v);
	printf("\nFourier transforms in full complex form\n\n");
	for (j = 0; j<m; ++j)
	{
		printf("Real");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", u[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}		
		printf("\nImag");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", v[j*n + i]);
		 	if(i % 6 == 5 && i != n-1 )
		 		printf("\n " );
		}
		printf("\n\n");
	}
}	
	
	The output is following:
	
	m = 3 n = 6
	
	Original data values
	
			0.3854 	  0.6772 	0.1138    0.6751 	0.6362    0.1424
			0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723
			0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815
			
	Discrete Fourier transforms in Hermitian format
	
			1.0737 	 -0.1041  	0.1126   -0.1467   -0.3738 	 -0.0044
			1.3961   -0.0365 	0.0780   -0.1521   -0.0607 	  0.4666
			1.1237    0.0914 	0.3936 	  0.1530 	0.3458   -0.0508
			
	Fourier transforms in full complex form
	
	Real 	1.0737   -0.1041 	0.1126 	 -0.1467 	0.1126   -0.1041
	Imag 	0.0000   -0.0044   -0.3738    0.0000 	0.3738    0.0044
	
	Real 	1.3961   -0.0365 	0.0780 	 -0.1521 	0.0780   -0.0365
	Imag 	0.0000    0.4666   -0.0607    0.0000 	0.0607   -0.4666
	
	Real 	1.1237    0.0914 	0.3936    0.1530 	0.3936    0.0914
	Imag 	0.0000   -0.0508 	0.3458    0.0000   -0.3458    0.0508
		

Return:
	The function returns NAG error code or 0 if no error.

	11: On entry, n must not be less than 1: n = _value_.
		
	successfully call of the nag_fft_init_trig function.
*/

	int nag_fft_init_trig(
		int n,	//the value  in the Fourier transform function being called.
		double trig[]	//the trigonometric coefficients.
  ); // calculates the trigonometric coefficients required for the computation of discrete Fourier transforms.


/**	c06hac
		computes the discrete Fourier sine transforms of m sequences of n real data values.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contains 18 data.  This piece of 
	code computes the discrete Fourier sine transforms of 3 sequences of 6 real data values.
	The result was put to the second column.	
	
	double trig[40];
	int m = 3, n = 6;
	//Attach 2 Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset dx("data1",1);
	x = xx;
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_sine(m, n, x, trig); 
	//Put the result to the second column.
	dx = x;	
	
		
Example2:
	This program reads in sequences of real data values and prints their Fourier sine 
	transforms (as computed by nag_fft_multiple_sine). It then calls nag_fft_multiple_sine
	again and prints the results which may be compared with the original sequence.

void test_nag_fft_multiple_sine()
{
	
	double trig[40];
	int i, j, m, n, row_len;
	double x[100] = {0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m = 3;
	n = 6;	
	row_len = n - 1;
	
	printf("\nOriginal data values\n\n");
	for (i = 0; i<m; ++i)
	{
		printf(" ");
		for (j = 0; j<row_len; ++j)
		{
			printf("%10.4f", x[(i)*row_len + (j)]);
		 	if((j % 7 )== 6 && (j != row_len-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_sine(m, n, x, trig); 
	printf("\nDiscrete Fourier sine transforms\n\n");
	for (i = 0; i<m; ++i)
	{
		printf(" ");
		for (j = 0; j<row_len; ++j)
		{
			printf("%10.4f", x[(i)*row_len + (j)]);
		 	if((j % 7 )== 6 && (j != row_len-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	
	nag_fft_multiple_sine(m, n, x, trig);
	printf("\nOriginal data as restored by inverse transform\n\n");
	for (i = 0; i<m; ++i)
	{
		printf(" ");
		for (j = 0; j<row_len; ++j)
		{
			printf("%10.4f", x[(i)*row_len + (j)]);
		 	if((j % 7 )== 6 && (j != row_len-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
}	

	
	The output is following:
	
	Original data values
	
	0.6772    0.1138 	0.6751    0.6362 	0.1424
	0.2983    0.1181 	0.7255    0.8638 	0.8723
	0.0644    0.6037 	0.6430    0.0428 	0.4815
	
	Discrete Fourier sine transforms
	
	1.0014    0.0062 	0.0834    0.5286 	0.2514
	1.2477   -0.6599 	0.2570    0.0858 	0.2658
	0.8521    0.0719   -0.0561   -0.4890 	0.2056
	
	Original data as restored by inverse transform
	
	0.6772    0.1138 	0.6751    0.6362 	0.1424
	0.2983    0.1181 	0.7255    0.8638 	0.8723
	0.0644    0.6037 	0.6430    0.0428 	0.4815
		
Return:
	The function returns NAG error code or 0 if no error.
	
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	73:  Memory allocation failed.
		
	successfully call of the nag_fft_multiple_sine function.	
*/

	int	nag_fft_multiple_sine(
		int m, // the number of sequences to be transformed.
		int n, // one more than the number of real values in each sequence, i.e. the number of values in each sequence is n - 1.
		double x[],	// Input: the m data sequences stored in x consecutively.  Output: the m Fourier sine transforms stored consecutively, overwrite the corresponding original sequence.
		const double trig[]	//the trigonometric coefficients are returned by a call of nag fft init trig (c06gzc).
  ); // computes the discrete Fourier sine transforms of m sequences of real data values.


/**	c06hbc
		computes the discrete Fourier cosine transforms of m sequences of n real data values.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contains 18 data.  This piece of 
	code computes the discrete Fourier cosine transforms of 3 sequences of 6 real data values.
	The result was put to the second column.	
	
	double trig[40];
	int m = 3, n = 6;
	//Attach 2 Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset dx("data1",1);
	x = xx;
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_cosine(m, n, x, trig); 
	//Put the result to the second column.
	dx = x;	
	
			
Example2:
	This program reads in sequences of real data values and prints their Fourier cosine 
	transforms (as computed by nag_fft_multiple_cosine). It then calls nag_fft_multiple_cosine 
	again and prints the results which may be compared with the original sequence.
	
void test_nag_fft_multiple_cosine()
{	
	double trig[40];
	int i, j, m, n, row_len;
    double x[105] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,0.9562,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,0.4936,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815,0.2057};     
	m = 3;
	n = 6;
	row_len = n + 1;
	
	printf("\nOriginal data values\n\n");
	for (i = 0; i<m; ++i)
	{
		printf(" ");
		for (j = 0; j<row_len; ++j)
		{
			printf("%10.4f", x[(i)*row_len + (j)]);
		 	if((j % 7 )== 6 && (j != row_len-1 ))
		 		printf("\n " );
		 }
		printf("\n");
	}
	

	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_cosine(m, n, x, trig); 
	printf("\nDiscrete Fourier cosine transforms\n\n");
	for (i = 0; i<m; ++i)
	{
		printf(" ");
		for (j = 0; j<row_len; ++j)
		{
			printf("%10.4f", x[(i)*row_len + (j)]);
		 	if((j % 7 )== 6 && (j != row_len-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	
	nag_fft_multiple_cosine(m, n, x, trig); 
	printf("\nOriginal data as restored by inverse transform\n\n");
	for (i = 0; i<m; ++i)
	{
		printf(" ");
		for (j = 0; j<row_len; ++j)
		{
			printf("%10.4f", x[(i)*row_len + (j)]);
		 	if((j % 7 )== 6 && (j != row_len-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
}

	The output is following:
	
	Original data values
	
	0.3854    0.6772 	0.1138    0.6751 	0.6362    0.1424 	0.9562
	0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723 	0.4936
	0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815 	0.2057
	
	Discrete Fourier cosine transforms
	
	1.6833   -0.0482 	0.0176    0.1368 	0.3240   -0.5830   -0.0427
	1.9605   -0.4884   -0.0655    0.4444 	0.0964    0.0856   -0.2289
	1.3838    0.1588   -0.0761 	 -0.1184 	0.3512    0.5759  	0.0110
	
	Original data as restored by inverse transform
	
	0.3854    0.6772 	0.1138    0.6751 	0.6362    0.1424 	0.9562
	0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723 	0.4936
	0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815 	0.2057
	
Return:
	The function returns NAG error code or 0 if no error.
	
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	73:  Memory allocation failed.
	
	successfully call of the nag_fft_multiple_cosine function.	
*/

	int nag_fft_multiple_cosine(
		int m, // the number of sequences to be transformed.
		int n, // one less than the number of real values in each sequence, i.e., the number of values in each sequence is n + 1.
		double x[], // Input:the m data sequences stored in x.  Output:the m Fourier cosine transforms stored consecutively, overwrite the corresponding original sequence.
		const double trig[]	//the trigonometric coefficients as returned by a call of nag_fft_init_trig (c06gzc).
  ); // computes the discrete Fourier cosine transforms of m sequences of real data values.


/**	c06hcc
		computes the discrete quarter-wave Fourier sine transforms of m sequences of real data values.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contains 18 data.  This piece of 
	code computes the discrete quarter-wave Fourier sine transforms of 3 sequences of 6 real data values.
	The result was put to the second column.	
	
	double trig[40];
	int m = 3, n = 6;
	//Attach 2 Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset dx("data1",1);
	x = xx;
	nag_fft_init_trig(n, trig);	
	nag_fft_multiple_qtr_sine(Nag_ForwardTransform, m, n, x, trig);
	//Put the result to the second column.
	dx = x;	
	
			
Example2:
	This program reads in sequences of real data values and prints their quarter-wave sine 
	transforms as computed by nag_fft_multiple_qtr_sine with direct = Nag_ForwardTransform. 
	It then calls nag_fft_multiple_qtr_sine again with direct = Nag BackwardTransform 
	and prints the results which may be compared with the original data.

void test_nag_fft_multiple_qtr_sine()
{	
	double trig[40];
	int i, j, m, n;
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m = 3;
	n = 6;
	
	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 7 )== 6 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
	nag_fft_init_trig(n, trig);
	
	nag_fft_multiple_qtr_sine(Nag_ForwardTransform, m, n, x, trig);
	printf("\nDiscrete quarter-wave Fourier sine transforms\n\n");
	
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 7 )== 6 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}

	nag_fft_multiple_qtr_sine(Nag_BackwardTransform, m, n, x, trig);
	printf("\nOriginal data as restored by inverse transform\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 7 )== 6 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}
}

	The output is the following:
	
	Original data values
	
	0.3854    0.6772 	0.1138    0.6751 	0.6362 	  0.1424
	0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723
	0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815
	
	Discrete quarter-wave Fourier sine transforms
	
	0.7304    0.2078 	0.1150    0.2577   -0.2869 	 -0.0815
	0.9274   -0.1152 	0.2532    0.2883   -0.0026   -0.0635
	0.6268    0.3547 	0.0760    0.3078 	0.4987   -0.0507
	
	Original data as restored by inverse transform
	
	0.3854    0.6772 	0.1138    0.6751 	0.6362    0.1424
	0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723
	0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815

Return:
	The function returns NAG error code or 0 if no error.
	
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	70:  On entry, parameter direct has an illegal value.

	successfully call of the nag_fft_multiple_qtr_sine function.
*/

	int nag_fft_multiple_qtr_sine(
		Nag_TransformDirection direct, //determine which direction transform to be computed.
		int m,	// the number of sequences to be transformed.
		int n,	// the number of real values in each sequence,  
		double x[],	// Input:the m data sequences stored in x consecutively.  Output:the m quarter-wave sine transforms stored consecutively.
		const double trig[]	//the trigonometric coefficients as returned by a call of nag_fft_init_trig (c06gzc).
  );  // computes the discrete quarter-wave Fourier sine transforms of m sequences of real data values.


/**	c06hdc
		computes the discrete quarter-wave Fourier cosine transforms of m sequences of real data values.

Example1:
	Assume "Data1" Worksheet has 2 columns, the first column contains 18 data.  This piece of 
	code computes the discrete quarter-wave Fourier cosine transforms of 3 sequences of 6 real data values.
	The result was put to the second column.	
	
	double trig[40];
	int m = 3, n = 6;
	//Attach 2 Datasets to these 2 columns
	Dataset xx("data1",0);
	Dataset dx("data1",1);
	x = xx;
	nag_fft_init_trig(n, trig); 
	nag_fft_multiple_qtr_cosine(Nag_ForwardTransform, m, n, x, trig);
	//Put the result to the second column.
	dx = x;	
	
				
Example2:
	This program reads in sequences of real data values and prints their quarter-wave cosine 
	transforms as computed by nag_fft_multiple_qtr_cosine with direct = Nag_ForwardTransform. 
	It then calls nag_fft_multiple_qtr_cosine again with direct = Nag_BackwardTransform and 
	prints the results which may be compared with the original data.
	
void test_nag_fft_multiple_qtr_cosine()
{
	
	double trig[40];
	int i, j, m, n;
	double x[100] = {0.3854, 0.6772, 0.1138, 0.6751, 0.6362, 0.1424,
					 0.5417, 0.2983, 0.1181, 0.7255, 0.8638, 0.8723,
					 0.9172, 0.0644, 0.6037, 0.6430, 0.0428, 0.4815};
	m =3;
	n =6;

	printf("\nOriginal data values\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 7 )== 6 && (i != n-1 ))
		 		printf("\n " );
		}
		printf("\n");
	}	

	nag_fft_init_trig(n, trig);
	nag_fft_multiple_qtr_cosine(Nag_ForwardTransform, m, n, x, trig);
	printf("\nDiscrete quarter-wave Fourier cosine transforms\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 7 )== 6 && (i != n-1 ))
		 		printf("\n " );
		 }
		printf("\n");
	}	

	nag_fft_multiple_qtr_cosine(Nag_BackwardTransform, m, n, x, trig);
	printf("\nOriginal data as restored by inverse transform\n\n");
	for (j = 0; j<m; ++j)
	{
		printf(" ");
		for (i = 0; i<n; ++i)
		{
			printf("%10.4f", x[j*n + i]);
		 	if((i % 7 )== 6 && (i != n-1 ))
		 		printf("\n " );
		 }
		printf("\n");
	}	
}
	
	The output is following:
	
	Original data values
	
	0.3854    0.6772 	0.1138    0.6751 	0.6362    0.1424
	0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723
	0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815
	
	Discrete quarter-wave Fourier cosine transforms
	
	0.7257   -0.2216 	0.1011    0.2355   -0.1406   -0.2282
	0.7479   -0.6172 	0.4112    0.0791 	0.1331   -0.0906
	0.6713   -0.1363   -0.0064   -0.0285 	0.4758    0.1475
	
	Original data as restored by inverse transform
	
	0.3854    0.6772 	0.1138    0.6751 	0.6362    0.1424
	0.5417    0.2983 	0.1181    0.7255 	0.8638    0.8723
	0.9172    0.0644 	0.6037    0.6430 	0.0428    0.4815
			

Return:
	The function returns NAG error code or 0 if no error.
	
	11:  On entry, m must not be less than 1: m = _value_.  On entry, n must not be less than 1: n = _value_.
	139: Value of n and trig array are incompatible or trig array is not initialized.
	70:  On entry, parameter direct has an illegal value.
		
	successfully call of the nag_fft_multiple_qtr_cosine function.	
*/

	int nag_fft_multiple_qtr_cosine(
		Nag_TransformDirection direct,//determine which direction transform to be computed
		int m,	// the number of sequences to be transformed.
		int n,	// the number of real values in each sequence.
		double x[], // Input: the m data sequences stored in x consecutively.  Output: the m quarter-wave cosine transforms stored consecutively overwrite the corresponding original sequence.
 		const double trig[]	// the trigonometric coefficients as returned by by a call of nag_fft_init_trig (c06gzc).
  ); // computes the discrete quarter-wave Fourier cosine transforms of m sequences of real data values.

#endif //!_O_NAG_G01_H

