/*------------------------------------------------------------------------------*
 * File Name: OCN_g01.h															*
 * Creation: SDB 5/22/2001														*
 * Purpose: Origin C Header for NAG functions									*
 * Copyright (c) OriginLab Corp.	2001										*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 * TCZ 06/10/2001, put_some_test_results_here								    *
 *------------------------------------------------------------------------------*/

#ifndef _O_NAG_G01_H
#define _O_NAG_G01_H

//#importdll "ONAG" // NAG DLL prepared by OriginLab
#pragma dll(ONAG)	
#include <NAG\nag_types.h>


/*	begin proto */

/**	g01aac
		calculates the mean, standard deviation, coefficients of skewness
		and kurtosis, and the maximum and minimum values of a set of
		ungrouped data.  Weighting may be used.

Example1:
	Read a set of data with 10 unweighted data values from worksheet "Data1", 
	calculates the mean, standard deviation, coeffieients of skewness and kurtosis, 
	and the maximum and minimum values.
	
	double xsd,xskew, xkurt, wsum, xmean, xmax, xmin;
	double * wt;
	int sucess, weight, nvalid = 10;
	wt = NULL;
	Dataset xx("data1",0);
	//Dataset cannot be used as a parameter, but vector can.
	vector x = xx;
	sucess = nag_summary_stats_1var(nvalid, x, wt, &nvalid, &xmean, &xsd, &xskew,
			&xkurt, &xmin, &xmax, &wsum);
	 
Example2:
	In the example, there is one set of data with 24 unweighted data values.

void  test_nag_summary_stats_1var()
{
	double xsd,xskew, xkurt, wsum, xmean, xmax, xmin;
	double * wt;
	int sucess,i, weight, j, nvalid = 24;
	double x[24] = {193.0, 215.0, 112.0, 161.0, 92.0, 140.0, 38.0, 33.3, 279.0,
					249.0, 473.0, 339.0, 60.0, 130.0, 20.0, 50.0, 257.0, 284.0,
					447.0, 52.0, 67.0, 61.0, 150.0, 2220.0};	
	wt = NULL;
	
	sucess = nag_summary_stats_1var(nvalid, x, wt, &nvalid, &xmean, &xsd, &xskew,
			&xkurt, &xmin, &xmax, &wsum);
   	
   	printf("Number of cases is %d\n", nvalid);
   	printf("And there is no weight\n");
   	printf("The input is following : x=\n");
   	
   	for(i=0; i<24; i++)
	{
		printf("%10.1f", x[i]);
		if((i+1)%10 == 0)
		printf("\n");
	}
	
	printf("\nsucess is%d\n", sucess);	
	if(sucess == 0)
	{
		printf("\nsucessfully call of the nag_summary_stats_1var function\n");
		printf("the output is following:\n");
		printf("No of valid cases         %10d\n", nvalid);
		printf("mean                      %10.1f\n", xmean);
		printf("std devn                  %10.1f\n", xsd);
		printf("Skewness                  %10.1f\n", xskew);
		printf("Kurtosis                  %10.1f\n", xkurt);	
		printf("Minimun                   %10.1f\n", xmin);
		printf("Maximum                   %10.1f\n", xmax);
		printf("Sum of weights            %10.1f\n", wsum);
	}
	else
	{
		printf(" \n There are some problems.");
	}
}

	The output is following:
	
	No. of valid cases                24
	mean                           255.1
	std devn                       437.4
	Skewness                         3.9
	Kurtosis                        14.7
	Minimun                         20.0
	Maximum                       2220.0
	Sum of weights                  24.0	

Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LE(12):	On entry, n must not be less than or equal to 0: n =value.
	NE_CASES_ONE(401): The number of valid cases is one. In this case,standard deviation and coeficients of skewness and of kurtosis cannot be calculated.
	NE_CASES_ZERO(400): The number of valid cases is zero.
	NE_REAL_ARG_LT(5): On entry, wt [value] must not be less than 0.0: wt [value ]=value

	success is 0.
	successfully call of the nag_summary_stats_1var function.
	
*/

int nag_summary_stats_1var(
	int n, // the number of observations
	const double x[], // the sample of observations, x[i]. i = 0,1,2 ..., n
	const double wt[], // the weight, can be NULL is not specified 
	int* nvalid, // the number of valid observations, (n) minus number of missing values
	double *xmean, 
	double *xsd,
	double *xskew = NULL,
	double *xkurt = NULL,
	double *xmin = NULL,
	double *xmax = NULL,
	double *wsum = NULL // the sum of weights, equals (n) if no weight is specified
);	// Mean, standard deviation, skewness, kurtosis etc, one variable, from raw data.


/**	g01aec
		constructs a frequency distribution of a variable, according to either
		user-supplied, or routine-calculated class boundary values.
Example1:
	Assume we have two worksheets, "Data1"'s datatype is double, "Data2"'s datatype
	is integer.  We have two columns in "Data1", the first column contains 70 data, 
	and we have to put the result to the second column of "Data1" and first column 
	of "Data2".
	
	double  xmax, xmin;
	int success, num_class = 7, iclass = 0, n = 70;
	Nag_ClassBoundary iclass_enum;
	iclass_enum = Nag_ClassBoundaryComp;
	Dataset aa("data1",0); 
	aa.SetSize(n);
	Dataset cc("data1",1);
	cc.SetSize( 7 );
	Dataset difreq("data2",0);
	difreq.SetSize( 7 );
	vector a = aa, c = cc;
	vector<int> ifreq = difreq;
	success = nag_frequency_table(n, a, num_class, iclass_enum, c, ifreq, &xmin, &xmax );
	//write the result back to the worksheets.
	difreq = ifreq;
	cc = c;
			
Example2:
	In this example, there are 70 observations.
	
void  test_nag_frequency_table()
{
	double  xmax, xmin;
	double c[7];
	int sucess, i, num_class = 7, iclass = 0, ifreq[7], n = 70;
	Nag_ClassBoundary iclass_enum;
    double a[70] = {22.3, 21.6, 22.6, 22.4, 22.4, 22.4, 22.1, 21.9, 23.1, 23.4,
    				23.4, 22.6,	22.5, 22.5, 22.1, 22.6, 22.3, 22.4, 21.8, 22.3,
    				22.1, 23.6, 20.8, 22.2,	23.1, 21.1, 21.7, 21.4, 21.6, 22.5,
					21.2, 22.6, 22.2, 22.2, 21.4, 21.7, 23.2, 23.1, 22.3, 22.3,
					21.1, 21.4, 21.5, 21.8, 22.8, 21.4, 20.7, 21.6, 23.2, 23.6,
					22.7, 21.7, 23.0, 21.9, 22.6, 22.1, 22.2, 23.4, 21.5, 23.0,
    				22.8, 21.4, 23.2, 21.8, 21.2, 22.0, 22.4, 22.8, 23.2, 23.6};
     
    iclass_enum = Nag_ClassBoundaryComp;   

	sucess = nag_frequency_table(n, a, num_class, iclass_enum, c, ifreq, &xmin, &xmax );

		
   	printf("Number of cases is %d\n", n);
   	printf("Number of calsses, including extreme classes %d\n", num_class);
   	printf("routine -supplied class boundary\n\n\n");
   	
   	printf("the data are as following\n");
   	
   	for(i=0; i<70; i++)
	{
		printf("%10.1f", a[i]);
		if((i+1)%10 == 0)
		printf("\n");
	}
	
	printf("\nsucess is%d\n", sucess);	
	
	if(sucess == 0)
	{
		printf("\nsucessfully call of the  function nag_frequency_table\n");
		printf("the output is following:\n");
		printf("***     Frequency distribution   ***\n");
		printf("        class             frequency\n");
		printf("up to           %5.2f             %d\n", c[0],ifreq[0]);
		for (i = 2; i<(num_class - 1); i++)
			printf("%5.2f  to   %5.2f             %d\n",c[i-2], c[i-1], ifreq[i-1]);
		printf("%5.2f        and over             %d\n", c[num_class - 2],ifreq[num_class-1]);
		printf("Total frequency  = 70 \n" );
		printf("Minimun                   %10.1f\n", xmin);
		printf("Maximum                   %10.1f\n", xmax);

	}
	else
	{
		printf(" \n There are some problems.");
	}
	
}

The output is following:

***     Frequency distribution   ***
        class             frequency
up to       20.70   	       0
20.70  to   21.28       	   6
21.28  to   21.86             16
21.86  to   22.44             21
22.44  to   23.02             14
23.60  and over		           0
Total frequency  = 70 
Minimum                   20.7
Maximum                   23.6

Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LT(11):	On entry, num_class must not be less than 2: num_class = <value>.  On entry, n must not be less than 1: n  =  <value>.
	NE_BAD_PARAM(70): On entry, parameter class had an ellegal value.
	NE_NOT_STRICTLY_INCREASING(63):	The sequence cint is not strictly increasing: cint[<value>] = <value>, cint[<value>] = <value>.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function.  Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.

	success is 0
	sucessfully call of the  function nag_frequency_table
	
*/

int nag_frequency_table(
	int n, // the number of observations
	const double x[], // the sample of observations, x[i]. i = 0,1,2,...,n 
	int num_class,	// the number of classes desired in the frequency distribution
	Nag_ClassBoundary class_val,
	double cint[],
	int ifreq[],
	double *xmin,	// the smallest value in the sample
	double *xmax	// the largest value in the sample
);	// Frequency table from raw data

/**	g01alc
		calculates a five-point summary for a single sample.

Example:

void test_nag_5pt_summary_stats()
{
	double res[5];
	int i, sucess, n=12;
	double x[12] = {12.0, 9.0, 2.0, 5.0, 6.0, 8.0, 2.0, 7.0, 3.0, 1.0, 11.0, 10.0};
	
	printf("\n the number of data is 12 and as following\n");
	for( i=0; i<12; i++)
		printf("%5.1f  ", x[i]);

	sucess = nag_5pt_summary_stats(n, x, res);
	if(sucess == 0)
	{
		printf("\nThe function has been sucessfully called\n");
		printf("the results are the following\n");
		printf("Maximum			 %8.4f\n", res[4]);
		printf("Upper hinge 	 %8.4f\n", res[3]);
		printf("Median			 %8.4f\n", res[2]);
		printf("Lower hinge		 %8.4f\n", res[1]);
		printf("Minimum			 %8.4f\n", res[0]);		
	}
	else
		printf("The call is unsucessful.");
}
	 
	The output is following:

	Maximum			  12.0000
	Upper hinge 	   9.5000
	Median			   6.5000
	Lower hinge		   2.5000
	Minimum			   1.0000 
	
Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LT(11): On entry, n must not be less than 5: n = _value_.
	NE_ALLOC_FAIL(73): Memory allocation failed.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.
	If the function call is correct, please consult NAG for assistance.
	
	successfully call of the nag_5pt_summary_stats function.
*/

int nag_5pt_summary_stats(
	int n, // the number of observations
	double x[], // the sample of observations, x[i]. i = 0,1,2,...,n
	double res[]
);	// Five-point summary (median, hinges and extremes)

/**	g01bjc
		returns the lower tail, upper tail and point probabilities associated with a Binomial distribution.

Example:

void test_nag_binomial_dist()
{
	double plek, peqk, pgtk;
	int sucess, i;
	double p[4] = {0.5, 0.440, 0.750, 0.330};	
	int n[4] = {4, 19, 100, 2000};
	int k[4] = {2, 13, 67, 700};
	
	printf("    n        p    k       plek        pgtk          peqk\n");
	for(i=0; i<4; i++)
	{
		sucess = nag_binomial_dist(n[i], p[i], k[i], &plek,	&pgtk, &peqk);
		if(sucess == 0)
			printf("%4d     %4.3f  %3d    %6.5f     %6.5f       %6.5f \n",n[i], p[i], k[i], plek, pgtk, peqk);
		else	
			break;
	}
	if(sucess != 0)
		printf(" the function is not sucessfully called");	
}
	
	The output is following:
	
	    n        p    k       plek        pgtk          peqk
	   4     0.500    2    0.68750     0.31250       0.37500 
	  19     0.440   13    0.99138     0.00862       0.01939 
	 100     0.750   67    0.04460     0.95540       0.01700 
	2000     0.330  700    0.97251     0.02749       0.00312 

Return:
	The function returns NAG error code or 0 if no error.
		
	NE_2_INT_ARG_GT(19): On entry, k = _value_ while n = _value_. These parameters must satisfy k = n.
	NE_ARG_TOO_LARGE(410): On entry, n is too large to be represented exactly as a double precision number.
	NE_INT_ARG_LE(12):	On entry, p must not be less than or equal to 0.0: p = _value_.
	NE_REAL_ARG_GE(8): On entry, p must not be greater than or equal to 1.0: p = _value_.
	NE_VARIANCE_TOO_LARGE(411): On entry, the variance (= np(1 - p)) exceeds 106.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.
	If the function call is correct, please consult NAG for assistance.
	
	successfully call of the nag_binomial_dist	
*/

int nag_binomial_dist(
	int n,	// the parameter n of the binomial distribution
	double p,	// the parameter p of the binomial distribution
	int k,	// the integer k which defines the required probabilities
	double *plek,	// the lower tail probability
	double *pgtk,	// the upper tail probability
	double *peqk	// the point probability
);	// Binomial distribution function

/**	g01bkc
		returns the lower tail, upper tail and point probabilities
		associated with a Poisson distribution.
	
Example:  

void test_nag_poisson_dist()
{
	double plek, peqk, pgtk;
	int sucess, i;
	int k[4] = {3, 12, 25, 175};	
	double rlamda[4] = {0.75, 9.20, 34.00, 175.00};
    
    printf("  rlamda      k        plek        pgtk          peqk\n");
	for(i=0; i<4; i++)
	{
		sucess = nag_poisson_dist(rlamda[i], k[i], &plek, &pgtk, &peqk);
		if(sucess == 0)
			printf("%6.3f     %4d     %6.5f     %6.5f       %6.5f \n",rlamda[i], k[i], plek, pgtk, peqk);
		else	
			break;
	}
	if(sucess != 0)
		printf(" the function is not sucessfully called");		
}

	The output is following:
		
	  rlamda      k        plek        pgtk          peqk
	 0.750        3     0.99271     0.00729       0.03321 
	 9.200       12     0.86074     0.13926       0.07755 
	34.000       25     0.06736     0.93264       0.02140 
	175.000      175     0.52009     0.47991       0.03014 
	
Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LT(11): On entry, k must not be less than 0: k = _value_.
	NE_REAL_ARG_LE(6): On entry, rlamda must not be less than or equal to 0.0: rlamda = _value_.
	NE_REAL_ARG_GT(7): On entry, rlamda must not be greater than 106: rlamda = _value_.	
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.
	If the function call is correct, please consult NAG for assistance.

	successfully call of the nag_poisson_dist function
*/

int nag_poisson_dist(
	double rlamda,	// the parameter lambda of the poisson distribution
	int k,	// the integer k which defines the required probabilities
	double *plek,	// the lower tail probability
	double *pgtk,	// the upper tail probability
	double *peqk	// the point probability
);	// Poisson distribution function

/**	g01blc
		returns the lower tail, upper tail and point probabilities
		associated with a hypergeometric distribution.

Example:
	
void test_nag_hypergeom_dist()
{
	double plek, peqk, pgtk;
	int sucess, i;	
	int n[4] = {10, 40, 155, 1000};
	int l[4] = {2, 10, 35, 444};
	int m[4] = {5, 3, 122, 500};
	int k[4] = {1, 2, 22, 220};
	 	
	printf(" the input data and results\n");
	printf("   n        l        m       k       plek         pgtk        peqk\n");
	for(i=0; i<4; i++)
	{
		sucess = nag_hypergeom_dist(n[i], l[i], m[i], k[i], &plek, &pgtk, &peqk);
		if(sucess == 0)
			printf("%4d     %4d     %4d    %4d    %6.5f      %6.5f     %6.5f \n",n[i], l[i], m[i], k[i], plek, pgtk, peqk);
		else	
			break;
	}
	if(sucess != 0)
		printf(" the function is not sucessfully called");
    
}
	
	The ouput is following:
	 
	   n        l        m       k       plek         pgtk        peqk
	  10        2        5       1    0.77778      0.22222     0.55556 
	  40       10        3       2    0.98785      0.01215     0.13664 
	 155       35      122      22    0.01101      0.98899     0.00779 
	1000      444      500     220    0.42429      0.57571     0.04913	

Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LT(11): On entry, n, l, k, m must not be less than 0: n, l, k, m = _value_.
	NE_2_INT_ARG_GT(19): On entry, l = _value_ while n = _value_. These parameters must satisfy l = n.
	On entry, m = _value_ while n = _value_. These parameters must satisfy m = n.
	On entry, k = _value_ while l = _value_. These parameters must satisfy k = l.
	On entry, k = _value_ while m = _value_. These parameters must satisfy k = m.
	NE_4_INT_ARG_CONS(30): On entry, k = _value_, l = _value_, m = _value_, n = _value_. These parameters must satisfy k = l + m - n. 
	NE_REAL_ARG_LT(5): On entry, n is too large to be represented exactly as a double precision number.
	NE_VARIANCE_TOO_LARGE(411): On entry, the variance = lm(n - l)(n - m)/[(n^2)(n - 1)] exceeds 10^6.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.
	If the function call is correct, please consult NAG for assistance.

	successfully call of the nag_hypergeom_dist function	
*/

int nag_hypergeom_dist(
	int n,	// the parameter n of hypergeometric distribution
	int l,	// the parameter l of hypergeometric distribution
	int m,	// the parameter m of hypergeometric distribution
	int k,	// the integer k which defines the required probabilities
	double *plek,	// the lower tail probability
	double *pgtk,	// the upper tail probability
	double *peqk	// the point probability
);	// Hypergeometric distribution function

/**	g01cec
		returns the deviate, xp, associated with the given lower tail 
		probability, p, of the standardized Normal distribution.
		
Example
	The deviates corresponding to several lower tail probabilities from the 
	standard Normal distribution are calculated and printed.

void  test_nag_deviates_normal_dist()
{
	double x;
	int i;
	double p[5] = {0.950, 0.500, 0.995, 0.750, 0.001};
	
	printf(" the input data and results\n");
	printf("prob.         Deviate\n");
	for(i=0; i<5; i++)
	{
		x = nag_deviates_normal_dist(p[i]);
		printf("%6.5f      %6.5f\n",p[i], x);
	}
	
}

The output is following:
	
prob.         	Deviate

0.95000      	1.64485
0.50000      	0.00000
0.99500      	2.57583
0.75000      	0.67449
0.00100      	-3.09023


Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_REAL_ARG_LE(6): On entry, p must not be less than or equal to 0.0: p = _value_.
	NE_REAL_ARG_GE(8): On entry, p must not be greater than or equal to 1.0: p = _value_.
	
	successfully call of the nag_deviates_normal_dist function.	
*/

double nag_deviates_normal_dist(
	double p,	// the probablity, p, from the standardised Normal distribution
	NagError* fail=NULL // NAG error structure
);	// Deviate of Normal distribution function

/**	g01ddc
		calculates Shapiro and Wilks W statistic and its significance level
		for testing Normality.
	
Example:
	A program to test the following 2 samples (each of size 20) for Normality.

void test_nag_shapiro_wilk_test()
{
	double x[20];
	double w, swap, minvalue, pw;
	int i, j, k, sucess;
	int min;
	Boolean  calwts;
	double a[20] = {0.11, 7.87, 4.61, 10.14, 7.95, 3.14, 0.46, 4.43, 0.21, 4.75,
					0.71, 1.52, 3.24, 0.93, 0.42, 4.97, 9.53, 4.55, 0.47, 6.66};	
	double b[20] = {1.36, 1.14, 2.92, 2.55, 1.46, 1.06, 5.27, -1.11, 3.48, 1.10,
					0.88, -0.51, 1.46, 0.52, 6.20, 1.69, 0.08, 3.67, 2.81, 3.49};
	
	// Sort the list. 
	test_sort(20,a);
	printf("the input data \n");
	printf("the first set\n");
	for (i = 0; i < 20; i++)
	{
		printf("%3.2f  ",a[i]);
		if( (i+1) % 10 == 0)
			printf("\n");
	}
	
	printf("the results as following \n");
	
	calwts= true;
	sucess = nag_shapiro_wilk_test(20, a, calwts, x, &w, &pw);
	if(sucess == 0)
	{
		printf("value of W statistic = %5.4f\n",w);
		printf("Significance level is  %5.4f\n",pw);
		
	}
	else 
	{
		printf("the function call  is not sucessful");
	}
	
	// Sort the list. 	
	test_sort(20,b);
	printf("\n\n\nthe input data \n");
	printf("the second set\n");
	for (i = 0; i < 20; i++)
	{
		printf("%3.2f  ",b[i]);
		if( (i+1) % 10 == 0)
			printf("\n");
	}
	
	printf("the results as following \n");
	
	calwts= false;
	sucess = nag_shapiro_wilk_test(20, b, calwts, x, &w, &pw);
	if(sucess == 0)
	{
		printf("value of W statistic = %5.4f\n",w);
		printf("Significance level is  %5.4f\n",pw);
		
	}
	else 
	{
		printf("the function call  is not sucessful");
	}
	for(i = 0; i<20; i++)
		printf("%4.2f\n",a[i]);
	
}

//This function is used to sort a list.
void test_sort(int n, double a[])
{
	int i, j;
	int min;
	double swap, minvalue;
	
	for(i = 0; i < n; i++)
	{
		min = i;
		minvalue=a[i];
		for(j = i+1; j < n; j++)
		{
			if(minvalue > a[j])
			{
				minvalue = a[j];
				min = j;
			}	
		}
		
		if(min != i)
		{
			swap = a[i];
	    	a[i] = a[min];
			a[min] = swap;			
		}
	}
}

	The output is following:
	
	For sample number 1, value of W statistic = 0.8992
	Significance level is 0.0408
	For sample number 2, value of W statistic = 0.9583
	Significance level is 0.5171

Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LT(11): On entry, n must not be less than 3: n = _value_.
	NE_REAL_ARG_GT(7):  On entry, n must not be greater than 2000: n = _value_.
	NE_NON_MONOTONIC(83): On entry, the sequence in array x is non-monotonic. First anomaly detected at x[_value_] = _value_.
	NE_ALL_ELEMENTS_EQUAL(260): On entry, all the values in the array x must not be equal.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.
	If the function call is correct, please consult NAG for assistance.

	successfully call of the nag_shapiro_wilk_test function.
*/

int nag_shapiro_wilk_test(
	int n,	// the sample size, 3 <= n <= 2000
	const double x[],  // the ordered sample values
	Boolean calc_wts, // calculate weight or not
	double a[],	// weight
	double* w,	// the value of the statistics, W
	double* pw	// the significance level of W
);	// Shapiro and Wilk's W test for Normality

/**	g01dhc
		computes the ranks, Normal scores, an approximation to the Normal
		scores or the exponential scores as requested by the user.
	
Example:

void test_nag_ranks_and_scores()
{
	int i;
	double aa[5] = {2, 0, 2, 2, 0};
	double bb[5];
	
	nag_ranks_and_scores(Nag_SavageScores,Nag_AverageTies, 5, aa, bb);
	printf("the input data\n");
	for( i = 0; i < 5; i++)
		printf("%4.0f   ",aa[i]);
	printf("\n\nthe results as following\n");
	for(i = 0;i < 5;i++)
		printf("%5.4f   \n",bb[i]);
}		

	The output is following:
	
	1.4500   
	0.3250   
	1.4500   
	1.4500   
	0.3250   	

Return:
	The function returns NAG error code or 0 if no error.
	
	NE_INT_ARG_LT(11): On entry, n must not be less than 1: n = _value_.
	NE_BAD_PARAM(70): On entry, parameter scores has an illegal value.  On entry, parameter ties has an illegal value.
	NE_ALLOC_FAIL(73): Memory allocation failed.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.

	successfully call of the nag_ranks_and_scores function
*/
  
int nag_ranks_and_scores(
	Nag_Scores scores,
	Nag_Ties ties,
	int n, // the number of observations
	const double x[], // the sample of observations, x[i]. i = 0,1,2,...,n
	double r[]	// output scores
);	// Ranks, Normal scores, approximate Normal scores or exponential (Savage) scores


/**	g01eac
		returns a one or two tail probability for the standard distribution
	
Example:
	Four values of tail and x are inputed and the probabilities are calculated and printed.
	
void test_nag_prob_normal()
{
	double prob, x;
	int i;
	Nag_TailProbability tail;
	char tail_char;
	
	tail_char = 'L';
	tail = Nag_LowerTail;
	x = 1.96;
	prob = nag_prob_normal(tail, x);
	printf("	Tail     X       Probability\n");
	printf("	 %c     %3.2f      %5.4f\n",tail_char,x,prob);
	
	tail_char = 'U';
	tail = Nag_UpperTail;
	prob = nag_prob_normal(tail, x);
	printf("	 %c     %3.2f      %5.4f\n",tail_char,x,prob);
	tail_char = 'C';
	tail = Nag_TwoTailConfid;
	prob = nag_prob_normal(tail, x);
	printf("	 %c     %3.2f      %5.4f\n",tail_char,x,prob);
	tail_char = 'S';
	tail = Nag_TwoTailSignif;
	prob = nag_prob_normal(tail, x);
	printf("	 %c     %3.2f      %5.4f\n",tail_char,x,prob);	
}

	The output is following:
	
	Tail     X       Probability
	 L     1.96      0.9750
	 U     1.96      0.0250
	 C     1.96      0.9500
	 S     1.96      0.0500

Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail has an illegal value.

	successfully call of the nag_prob_normal function 
*/

double nag_prob_normal(
	Nag_TailProbability tail,
	double x,	// the value of the standard Normal variate
	NagError* fail=NULL // NAG error structure
);	// Probabilities for the standard Normal distribution


/**	g01ebc
		returns the lower tail,upper tail or two-tail probability for the
		Students t-distribution with real degrees of freedom.
	
	
Example:
	Values from, and degrees of freedom for Student's t-distributions are read 
	along with the required tail. The probabilities are calculated and printed 
	until the end of data is reached.

void test_nag_prob_students_t()
{
	double prob;
	int i;
	Nag_TailProbability tail[4] = {Nag_LowerTail, Nag_UpperTail, Nag_TwoTailSignif, Nag_TwoTailConfid};	
	double t[4] = {0.85, 0.85, 0.85, 0.85};
	double df[4] = {20.0, 20.0, 20.0, 20.0};

	printf("	t     df	  prob		tail\n");
	
	for( i = 0; i < 4; i++)
	{
		prob = nag_prob_students_t(tail[i], t[i], df[i]);
		if ( i == 0) 
		printf("  %4.3f   %5.3f   %5.4f     Nag_LowerTail  \n",t[i], df[i], prob); 
		if ( i == 1) 
		printf("  %4.3f   %5.3f   %5.4f     Nag_UpperTail  \n",t[i], df[i], prob);
		if ( i == 2) 
		printf("  %4.3f   %5.3f   %5.4f     Nag_TwoTailSignif  \n",t[i], df[i], prob);
		if ( i == 3) 
		printf("  %4.3f   %5.3f   %5.4f     Nag_TwoTailConfid  \n",t[i], df[i], prob);
	} 
} 


	The ouput results are following:
		
		t     df	  	prob		tail
	  0.850   20.000   	0.7973     Nag_LowerTail  
	  0.850   20.000   	0.2027     Nag_UpperTail  
	  0.850   20.000   	0.4054     Nag_TwoTailSignif  
	  0.850   20.000   	0.5946     Nag_TwoTailConfid
	  
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail has an illegal value.
	NE_REAL_ARG_LT(5):  On entry, df must not be less than 1.0: df = _value_.

	successfully call of the nag_prob_students_t function	
*/

double nag_prob_students_t(
	Nag_TailProbability tail,
	double t,	// the value of the Student s t variate
	double df,	// the degree of freedom
	NagError* fail=NULL // NAG error structure
);	// Probabilities for Student's t-distribution


/**	g01ecc
		returns the lower or upper tail probability for the  2 distribution
		with real degrees of freedom.
	

Example:
	Values from various Chi-square distributions are read, the lower-tail probabilities
	are calculated, and all these values are printed out, until the end of data is reached.

void test_nag_prob_chi_sq()
{
	double prob;
	int i;
	double df[3] = {20.0, 7.5, 45.0};
	double x[3] = {8.26, 6.2, 55.76};
	
	printf("the input data and results as following\n");
	printf("    x       df       prob\n");
	for( i = 0; i < 3; i++)
	{
		prob = nag_prob_chi_sq(Nag_LowerTail, x[i], df[i]);
		printf("  %5.3f    %5.3f      %5.4f\n", x[i], df[i], prob);
	}				
}


	The output is following:
	
	    x       df       prob
	    
	  8.260    20.000      0.0100
	  6.200    7.500       0.4279
	  55.760   45.000      0.8694
	  
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail has an illegal value.
	NE_REAL_ARG_LT(5):  On entry, x must not be less than 0.0: x = _value_.
	NE_REAL_ARG_LE(6):  On entry, df must not be less than or equal to 0.0: df = _value_.
	NE_ALG_NOT_CONV(405): The series used to calculate the gamma probabilities has failed to converge.  The result returned should represent an approximation to the solution.

	successfully call of the nag_prob_chi_sq function
*/

double nag_prob_chi_sq(
	Nag_TailProbability tail,
	double x,	// the value of the x^2 variate
	double df,	// the degree of freedom
	NagError* fail=NULL // NAG error structure
);	// Probabilities for chi^2 distribution


/**	g01edc
		returns the probability for the lower or upper tail of the F or
		variance-ratio distribution with real degrees of freedom.
	
Example:
	Values from, and degrees of freedom for F-distributions are read, the lower-tail 
	probabilities are computed, and all these values are printed, until the end of data is reached.

void test_nag_prob_f_dist()
{
	double prob;
	int i;
	double df1[3] = {1.5, 1.0, 20.25};
	double df2[3] = {25.5, 1.0, 1.0};
	double f[3] = {5.5, 39.9, 2.5};
	
	printf(" the input data and results are the following:\n");
	printf("  f      df1     df2    prob\n");
	for(i = 0; i < 3; i++)
	{
		prob = nag_prob_f_dist(Nag_LowerTail, f[i], df1[i], df2[i]);
		printf("%4.3f  %4.3f    %5.3f  %5.4f\n",f[i], df1[i], df2[i], prob);
	}
}


	The ouput is following:
		
	  f      df1     df2    prob
	
	5.500  1.500    25.500  0.9837
	39.900  1.000    1.000  0.9000
	2.500  20.250    1.000  0.5342
	
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail has an illegal value.
	NE_REAL_ARG_LT(5):  On entry, f must not be less than 0.0: f = _value_.
	NE_REAL_ARG_LE(6):  On entry, df1 must not be less than or equal to 0.0: df1 = _value_.  On entry, df2 must not be less than or equal to 0.0: df2 = _value_.

	successfully call of the nag_prob_f_dist function
*/

double nag_prob_f_dist(
	Nag_TailProbability tail,
	double f,	// the value of the F variate
	double df1,	// the degree of freedom of the numerator variance, v1
	double df2,	// the degree of freedom of the denominator variance, v2
	NagError* fail=NULL // NAG error structure
);	// Probabilities for F-distribution


/**	g01eec
		computes the upper an lower tail probabilities and the probability
		density function of the beta distribution with parameters a and b.
	
Example:
	Values for several beta distributions are read, and the probabilities 
	calculated and printed, until the end of data is reached.

void test_nag_prob_beta_dist()
{
	double p, pdf, q;
	int i, sucess;
	double a[3] = {1.0, 1.5, 2.0};
	double b[3] = {2.0, 1.5, 1.0};	
	double x[3] = {0.25, 0.75, 0.50};
	double tol[3] = {1.9, 0.0001, 1.01};
	
	printf("the input data and results\n");
	printf(" x			a			b			p			q			pdf\n");
	for(i = 0; i < 3; i++)
	{
		sucess = nag_prob_beta_dist( x[i], a[i], b[i],tol[i], &p, &q, &pdf);
		if( sucess == 0)
			printf("%5.4f  %9.4e		 %9.4e        %9.4e        %9.4e       %9.4e \n",x[i], a[i], b[i],
					p, q, pdf);
		else
			break;
	}
	if(sucess != 0)
		printf("the function call is not sucessfully");
}

	The output is following:
		
	 x				a			b						p			q					pdf
	0.2500  1.0000e+000		 2.0000e+000        4.3750e-001        5.6250e-001       1.5000e+000 
	0.7500  1.5000e+000		 1.5000e+000        8.0450e-001        1.9550e-001       1.1027e+000 
	0.5000  2.0000e+000		 1.0000e+000        2.5000e-001        7.5000e-001       1.0000e+000 

Return:
	The function returns NAG error code or 0 if no error.
	
	NE_REAL_ARG_LT(5): On entry, x must not be less than 0.0: x = _value_.
	NE_REAL_ARG_GT(7): On entry, x must not be greater than 1.0: x = _value_.  On entry, a must not be greater than 106: a = _value_.  On entry, b must not be greater than 106: b = _value_.
	NE_REAL_ARG_LE(6): On entry, a must not be less than or equal to 0.  0: a = _value_.  On entry, b must not be less than or equal to 0.0: b = _value_.
	NE_RES_NOT_ACC(404): The requested accuracy has not been achieved. Use a larger value of tol.  The values returned for p and q should be reasonable approximations.
	NE_PROBAB_CLOSE_TO_TAIL(409): The probability is too close to 0.0 or 1.0.  The result returned is either 0 or 1 as appropriate. This should be a good approximation to
	the required solution.

	successfully call of the nag_prob_beta_dist function	
*/

int nag_prob_beta_dist(
	double x,	// the value of the beta variate
	double a,	// the first parameter required by the beta distribution
	double b,	// the second parameter required by the beta distribution
	double tol,	// the required accuracy of the solution
	double *p,	// the lower tail probability
	double *q,	// the upper tail probability
	double *pdf	// the probability density function
);	// Upper and lower tail probabilities and probability density function for the beta distribution


/**	g01efc
		returns the lower or upper tail probability of the gamma distribution,
		with parameters alpha and beta.
	
Example:
	The following example reads in values for several gamma distributions, computes
	and prints the lower probabilities for each case, until the end of data is reached.

void test_nag_gamma_dist()
{
	double p;
	int i;
	double g[4] = {15.50, 0.50, 10.0, 5.0};
	double a[4] = {4.0, 4.0, 1.0, 2.0};
	double b[4] = {2.0, 1.0, 2.0, 2.0};
	
	printf("the input data and results are following\n");
	printf("Gamma deviate      Alpha     Beta      Lower tail prob.\n");
	for(i = 0; i < 4; i++)
	{
		p =  nag_gamma_dist(Nag_LowerTail, g[i], a[i], b[i]);
		printf("  %4.2f            %3.2f     %3.2f         %5.4f\n",g[i],a[i],b[i],p);		
	} 	
}

	The output is following:
		
	Gamma deviate      Alpha     Beta      Lower tail prob.
	  15.50            4.00     2.00         0.9499
	  0.50             4.00     1.00         0.0018
	  10.00            1.00     2.00         0.9933
	  5.00             2.00     2.00         0.7127

Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail has an illegal value.
	NE_REAL_ARG_LT(5):  On entry, g must not be less than 0.0: g = _value_.
	NE_REAL_ARG_LE(6):  On entry, a must not be less than or equal to 0.0: a = _value_.  On entry, b must not be less than or equal to 0.0: b = _value_.
	NE_ALG_NOT_CONV(405): The algorithm has failed to converge in _value_ iterations.  The probability returned should be a reasonable approximation to the solution.

	successfully call of the nag_gamma_dist function	
*/

double nag_gamma_dist(
	Nag_TailProbability tail,
	double g,	// the value of the gamma variate
	double a,	// the parameter alpha of the gamma distribution
	double b,	// the parameter beta of the gamme distribution
	NagError* fail=NULL // NAG error structure
);	// Probabilities for the gamma distribution


/**	g01fac
		returns the deviate associated with the given probability of the
		standard Normal distribution.
	
Example:
	Four values of tail and x are input and the probabilities calculated and printed.

void test_nag_deviates_normal()
{

	double p, dev;
	int i;
	Nag_TailProbability tail;
	char tail_char;
	
	tail_char = 'L';
	tail = Nag_LowerTail;
	p = 0.975;
	dev = nag_deviates_normal(tail, p);
	printf("	Tail     Probability        Deviate\n");
	printf("	 %c     %4.3f      %5.4f\n",tail_char, p, dev);
	
	tail_char = 'U';
	tail = Nag_UpperTail;
	p = 0.025;
	dev = nag_deviates_normal(tail, p);
	printf("	 %c     %4.3f      %5.4f\n",tail_char, p, dev);	
	
	tail_char = 'C';
	tail = Nag_TwoTailConfid;
	p = 0.95;
	dev = nag_deviates_normal(tail, p);
	printf("	 %c     %4.3f      %5.4f\n",tail_char, p, dev);
	tail_char = 'S';
	tail = Nag_TwoTailSignif;
	p = 0.05; 
	dev = nag_deviates_normal(tail, p);
	printf("	 %c     %4.3f      %5.4f\n",tail_char, p, dev);
	
}	
	
	The output is following:
	
	Tail     Probability        Deviate
	 L       0.975              1.9600
	 U       0.025              1.9600
	 C       0.950              1.9600
	 S       0.050              1.9600

Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail has an illegal value.
	NE_REAL_ARG_LE(6):  On entry, p must not be less than or equal to 0.0: p = 	value.
	NE_REAL_ARG_GE(8):  On entry, p must not be greater than or equal to 1.0: p = 	value.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function. Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.

	successfully call of the nag_deviates_normal function	
*/

double nag_deviates_normal(
	Nag_TailProbability tail,
	double p,	// the probability from the standard Normal distribution
	NagError* fail=NULL // NAG error structure
);	// Deviates for the Normal distribution


/**	g01fbc
		returns the deviate associated with the given tail probability of
		Students t-distribution with real degrees of freedom.
	
Example:
	Lower tail probabilities are read for several t-distributions, and the 
	corresponding deviates calculated and printed, until the end of data is reached.

void test_nag_deviates_students_t()
{
	double t;
	int i;
	double p[4] = {0.01, 0.85, 0.01, 0.99};
	double df[4] = {7.5, 20.0, 20.0, 45.0};
	Nag_TailProbability tail[4] = {Nag_LowerTail, Nag_UpperTail, Nag_TwoTailSignif, Nag_TwoTailConfid};

	printf("	p       df	   tail		  t\n");
	
	for( i = 0; i < 4; i++)
	{
		if(i != 1)
		{
		
			t = nag_deviates_students_t(tail[i], p[i], df[i]);
			if ( i == 0) 
				printf("  %4.3f   %5.3f  Nag_LowerTail      %5.4f\n",p[i], df[i], t); 
			if ( i == 1) 
				printf("  %4.3f   %5.3f  Nag_UpperTail      %5.4f\n",p[i], df[i], t);
			if ( i == 2) 
				printf("  %4.3f   %5.3f  Nag_TwoTailSignif  %5.4f\n",p[i], df[i], t);
			if ( i == 3) 
				printf("  %4.3f   %5.3f  Nag_TwoTailConfid  %5.4f\n",p[i], df[i], t);
		}
	} 
}
	
	The output is following:
		
		p 			df 			tail 				t
		
	0.010       7.500       Nag_LowerTail           -2.9431
    0.010       20.000      Nag_TwoTailSignif       2.8453
    0.990       45.000      Nag_TwoTailConfid       2.6896


Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_BAD_PARAM(70): On entry, parameter tail had an illegal value.
	NE_REAL_ARG_LE(6):  On entry, p must not be less than or equal to 0.0: p = _value_.
	NE_REAL_ARG_GE(8):  On entry, p must not be greater than or equal to 1.0: p = _value_.
	NE_INT_ARG_LT(11): On entry, df must not be less than 1.0: df = _value_.
	NE_SOL_CLOSE_TO_ZERO(406): The solution is too close to zero to be determined accurately.  This error will only occur when df = 1.0. The returned value of zero will be a good approximation in terms of absolute value but will have a poor relative precision.
	NE_ALG_NOT_CONV(405): The solution has failed to converge.  However, the result should be a reasonable approximation.

	successfully call of the nag_deviates_students_t function	
*/

double nag_deviates_students_t(
	Nag_TailProbability tail,
	double p,	// the probability from the standard Normal distribution
	double df,	// the degree of freedom
	NagError* fail=NULL // NAG error structure
);	// Deviates for Student's t-distribution


/**	g01fcc
		the deviate associated with the given lower tail probability of
		the distribution with real degrees of freedom.
	
Example:
	Lower tail probabilities are read for several Chi-square distributions, and 
	the corresponding deviates calculated and printed, until the end of data is reached.

void test_nag_deviates_chi_sq()
{
	double x;
	int i;
	double p[3] = {0.01, 0.4279, 0.8694};
	double df[3] = {20.0, 7.5, 45.0};
	
	printf("the input data and results are following\n");
	printf("p		df		    x\n");
	for(i = 0; i < 3; i++)
	{
		x = nag_deviates_chi_sq(p[i], df[i]);
		printf("%5.4f     %5.3f		   %5.3f\n",p[i], df[i], x);
	}
	
}
	
	The output is following:
		
		p 			df 			x
		
		0.0100 		20.000 		8.260
		0.4279 		7.500 		6.200
		0.8694 		45.000 		55.759
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	On any of the error conditions listed below except NE_ALG_NOT_CONV(405) nag_deviates_chi_sq returns 0.0.
	NE_REAL_ARG_LT(5): On entry, p must not be less than 0.0: p = _value_.
	NE_REAL_ARG_GE(8): On entry, p must not be greater than or equal to 1.0: p = _value_.
	NE_REAL_ARG_LE(6): On entry, df must not be less than or equal to 0.0: df = _value_.
	NE_PROBAB_CLOSE_TO_TAIL(409): The probability is too close to 0.0 or 1.0.
	NE_ALG_NOT_CONV(405): The algorithm has failed to converge in _value_ iterations.  The result should be a reasonable approximation.
	NE_GAM_NOT_CONV(403): The series used to calculate the gamma probabilities has failed to converge.  This is an unlikely error exit.

	successfully call of the nag_deviates_chi_sq function	
*/

double nag_deviates_chi_sq(
	double p,	// the probability p from the distribution
	double df,	// the degree of freedom
	NagError* fail=NULL // NAG error structure
);	// Deviates for the chi^2 distribution


/**	g01fdc
		the deviate associated with the given lower tail probability of
		the F or variance-ratio distribution with real degrees of freedom.

Example:
	Lower tail probabilities are read for several F-distributions, and the 
	corresponding deviates calculated and printed, until the end of data is reached.

void test_nag_deviates_f_dist()
{
	double f;
	int i;
	double df1[3] = {10.0, 1.0, 20.25};
	double df2[3] = {25.5, 1.0, 1.0};
	double p[3] = {0.9837,0.9000, 0.5342};
	
	printf(" the input data and results are the following:\n");
	printf("  p      df1     df2     f\n");
	for(i = 0; i < 3; i++)
	{
		f = nag_deviates_f_dist(p[i], df1[i], df2[i]);
		printf("%4.3f   %4.3f       %5.3f      %5.4f\n",p[i], df1[i], df2[i], f);
	}
}

	The output is following:
	
	  p      df1     	df2     	f
	
	0.984  	10.000    	25.500  	2.8366
	0.900  	1.000  	  	1.000  		39.8657
	0.534  	20.250    	1.000  		2.5004


Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	On any of the error conditions listed below except NAG_SOL_NOT_CONV(407) nag deviates f dist returns 0.0.
	NE_REAL_ARG_LT(5): On entry, p must not be less than 0.0: p = _value_.
	NE_REAL_ARG_GE(8): On entry, p must not be greater than or equal to 1.0: p = _value_.
	NE_REAL_ARG_LE(6): On entry, df1 must not be less than or equal to 0.0: df1 = _value_.  On entry, df2 must not be less than or equal to 0.0: df2 = _value_.
	NE_SOL_NOT_CONV(407): The solution has failed to converge.  However, the result should be a reasonable approximation.  Alternatively, nag deviates f dist can be used with a suitable setting of the parameter tol.
	NE_PROBAB_CLOSE_TO_TAIL(409): The probability is too close to 0.0 or 1.0.  The value of fp cannot be computed. This will only occur when the large sample approximations are used.
	
	successfully call of the nag_deviates_f_dist function	
*/

double nag_deviates_f_dist(
	double p,	// the probability from the standard Normal distribution
	double df1, // the degree of freedom of the numerator variance, v1
	double df2,	// the degree of freedom of the denominator variance, v2
	NagError* fail=NULL // NAG error structure
);	// Deviates for the F-distribution


/**	g01fec
		the deviate associated with the given lower tail probability of
		the beta distribution.
	
Example:
	Lower tail probabilities are read for several beta distributions, and the 
	corresponding deviates calculated and printed, until the end of data is reached.

void test_nag_deviates_beta()
{
	double x;
	int i;
	double a[3] = {1.0, 1.5, 20.0};	
	double b[3] = {2.0, 1.5, 10.0};
	double p[3] = {0.5, 0.99, 0.25};	
	double tol[3] = {0.0, 0.0, 0.0};
	
	printf("the input data and results\n");
	printf(" probability   A	        	 B	       deviate\n");
	for(i = 0; i < 3; i++)
	{
	   x= nag_deviates_beta(p[i], a[i], b[i], tol[i]);
	   printf("%5.4f      %5.4f		 %4.3f        %5.4f\n",p[i], a[i], b[i], x);
	}
}

	The output is following:
	
	probability   A	        	 B	       deviate
	0.5000      1.0000		 2.000        0.2929
	0.9900      1.5000		 1.500        0.9672
	0.2500      20.0000		 10.000        0.6105


Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	On any of the error conditions listed below except NE_RES_NOT_ACC(404) and NE_SOL_NOT_CONV(403) nag deviates beta returns 0.0.
	NE_REAL_ARG_LT(5): On entry, p must not be less than 0.0: p = _value_.
	NE_REAL_ARG_GT(7): On entry, p must not be greater than 1.0: p = _value_.  On entry, a must not be greater than 106: a = _value_.  On entry, b must not be greater than 106: b = _value_.
	NE_REAL_ARG_LE(6): On entry, a must not be less than or equal to 0.0: a = _value_.  On entry, b must not be less than or equal to 0.0: b = _value_.
	NE_RES_NOT_ACC(404): The requested accuracy has not been achieved. Use a larger value of tol.  There is doubt concerning the accuracy of the computed result. 100 iterations of the Newton-Raphson method have been performed without satisfying the accuracy criterion (see Section 6.1). The result should be a reasonable approximation of the solution.
	NE_GAM_NOT_CONV(403): The solution has failed to converge.  However, the result should be a reasonable approximation.  Requested accuracy not achieved when calculating beta probability. The user should try setting tol larger.  
	
	successfully call of the nag_deviates_beta function	
*/

double nag_deviates_beta(
	double p,	// the probability from the standard Normal distribution
	double a,	// the first parameter of the beta distribution
	double b,	// the second parameter of the beta distribution
	double tol=5e-6,	// the required accuracy of the solution
	NagError* fail=NULL // NAG error structure
);	// Deviates for the beta distribution


/**	g01ffc
		the deviate associated with the given lower tail probability
		of the gamma distribution.
	
Example:
	Lower tail probabilities are read for several gamma distributions, and the 
	corresponding deviates calculated and printed, until the end of data is reached.

void test_nag_deviates_gamma_dist()
{
	double g;
	int i;
	double p[3] = {0.10, 0.4279, 0.8694};
	double a[3] = {1.0, 7.5, 45.0};
	double b[3] = {20.0, 0.1, 10.0};
	
	printf("the input data and results are following\n");
	printf("p      a     b      g.\n");
	for(i = 0; i < 3; i++)
	{
		g =  nag_deviates_gamma_dist(p[i], a[i], b[i], 0.0);
		printf("%4.3f      %5.4f    %5.3f      %5.3f\n",p[i], a[i], b[i], g);		
	} 	
}

	The output is following:
		
	p      		a     		b      		g
	0.100      1.0000    20.000      2.107
	0.428      7.5000    0.100       0.670
	0.869      45.0000   10.000      525.979

Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	On any of the error conditions listed below except NE_ALG_NOT_CONV(405) nag_deviates_gamma_dist returns 0.0.
	NE_REAL_ARG_LT(5): On entry, p must not be less than 0.0: p = _value_.
	NE_REAL_ARG_GE(8): On entry, p must not be greater than or equal to 1.0: p = _value_.
	NE_REAL_ARG_LE(6): On entry, a must not be less than or equal to 0.0: a = _value_.  On entry, b must not be less than or equal to 0.0: b = _value_.
	NE_REAL_ARG_GT(7): On entry, a must not be greater than 106: a = _value_.
	NE_PROBAB_CLOSE_TO_TAIL(409): The probability is too close to 0.0 for given a result to be calculated.
	NE_ALG_NOT_CONV(405): The algorithm has failed to converge in 100 iterations.  A larger value of tol should be tried. The result may be a reasonable approximation.
	NE_GAM_NOT_CONV(403): The series used to calculate the gamma probabilities has failed to converge.  This is an unlikely error exit.
	
	successfully call of the nag_deviates_gamma_dist function
*/

	double nag_deviates_gamma_dist(
		double p,	// the probability from the required gamma distribution
		double a,	// the shape parameter of the gamma distribution
		double b,	// the scale parameter of the gamma distribution
		double tol=5e-6,	// the required accuracy of the solution
		NagError* fail=NULL // NAG error structure
	);	// Deviates for the gamma distribution


/**	g01gbc
		returns the lower tail probability for the non-central Student's
		t-distribution.
	
Example:

void test_nag_prob_non_central_students_t()
{
	double prob;
	int i;
	double delta[3] = {2.0, 1.0, 0.0};
	double df[3] = {20.0, 7.5, 45.0};
	double t[3] = {-1.528, -0.188, 1.138};
	
	printf("the input data and the results are as following\n");
	printf("	t 	df	delta	   prob\n");
	for(i = 0; i < 3; i++)
	{
		prob = nag_prob_non_central_students_t(t[i], df[i], delta[i]);
		printf("%8.3f%8.3f%8.3f%8.4f%\n", t[i],df[i],delta[i],prob);		
	}		
}

	The output is following:(with default tol and max_iter)

		t 	df	delta	   prob
	  -1.528  20.000   2.000  0.0003
	  -0.188   7.500   1.000  0.1189
	   1.138  45.000   0.000  0.8694

Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.

	NE_REAL_ARG_LT(5):  On entry, df must not be less than 1.0: df = <value>.
	NE_INT_ARG_LT(11): On entry, max_iter must not be less than 1: max_iter = <value>.
	NE_SERIES(415): One of the series has failed to converge with df = <value>.  Reconsider the requested tolerance and/or the maximum number of iterations.
	NE_PROBABILITY(416): The probability is too small to calculate accurately.
	NE_INTERNAL_ERROR(74):  An internal error has ocurred in this function.  Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.
	
	successfully call of the nag_prob_non_central_students_t function.   
*/

	double nag_prob_non_central_students_t(
		double t,	// the deviate from the Student's t-distribution, t
		double df,	// the degree of freedom
		double delta,// the non-centrality of the distribution
		double tol=5e-6, //	the absolute accuracy required by the user
		int max_iter=100,	// the maximum number of terms used in each of the summations
		NagError* fail=NULL // NAG error structure
	);  // Computes probabilities for the non-central Student's t-distribution


/**	g01gcc
		returns the lower tail probability for the non-central
		chi-square distribution.
	
Example:

void test_nag_prob_non_central_chi_sq()
{
	double prob;
	int i;
	double x[3] = {8.26, 6.20, 55.76};
	double df[3] = {20.0, 7.5, 45.0};
	double lamda[3] = {3.5, 2.0, 1.0};
	
	printf("the input data and corresponding results are as following\n");
	printf("   x       df	  lambda prob\n");
	for(i = 0; i < 3; i++)
	{
		prob = nag_prob_non_central_chi_sq(x[i], df[i],lamda[i],5e-6,50);
		printf("%8.3f%8.3f%8.3f%8.4f\n",x[i], df[i], lamda[i], prob);
	}	
}

	The output is following: (with default tol and max_iter)
	
	x		df	lambda	prob
	8.260	20.000	3.500	0.0032
	6.200	7.500	2.000	0.2699
	55.760	45.000	1.000	0.8443
	
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_REAL_ARG_LT(5):  One entry, df must not be less than 0: df = <value>.  One entry, lambda must not be less than 0: lambda = <value>.  On entry, x must not be less than 0: x = <value>.
	NE_2_REAL_ARG_CONS(31): On entry, df = <value> while lambda = <value>.  These parameters must satisfy lambda > 0 if df = 0.
	NE_INT_ARG_LT(11): On entry, max_iter must not be less than 1: max_iter = <value>
	NE_POISSON_WEIGHT(417): The initial value of the Poisson weight used in the summation of (1) was too small to be calculated.  The computed probability is likely to be zero.
	NE_CONV(420): The solution has failed to converge in <value> iterations, consider increasing max_iter or tol.
	NE_TERM_LARGE(418): The value of a term required in (2) is too large to be evaluated accurately.  The most likely cause of this error is that both x and lambda are too large.
	NE_CHI_PROB(419): The calculations for the central chi-square probability has failed to converge.  A larger value of tol should be used.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function.  Check the function call and array sized.  If the function call is correct, please consult NAG for assistance.
			
	successfully call of the nag_prob_non_central_chi_sq function
*/

	double nag_prob_non_central_chi_sq(
		double x, // the deviate from the chi-square distribution
		double df,// the degree of freedom
		double lambda, // the non-centrality parameter
		double tol=5e-6,	// the required accuracy of the solution
		int max_iter = 100, // the maximum number of iterations to be performed
		NagError *fail=NULL	// NAG error structure
	); // Computes probabilities for the non-central chi^2 distribution


/**	g01gdc
		returns the lower tail probability for the non-central F-distribution
	
Example:

void test_nag_prob_non_central_f_dist()
{	
	double prob;
	int i;
	double lambda[3] = {3.0, 2.0, 0};
	double df1[3] = {1.5, 1.0, 20.25};
	double f[3] = {5.5, 39.9, 2.5};
	double df2[3] = {25.5, 1.0, 1.0};
	
	printf("the input data and the results are as following\n");
	printf("	f 	d1f	 df2  lambda   prob\n");
	for(i = 0; i < 3; i++)
	{
		prob = nag_prob_non_central_f_dist(f[i], df1[i], df2[i], lambda[i], 5e-6, 50);
		printf("%8.3f%8.3f%8.3f%8.3f%8.4f%\n", f[i], df1[i], df2[i], lambda[i], prob);		
	}		
}

	The output is following: (with default tol and max_iter)
	
	f		df1		df2		lambda	prob
	5.500	1.5		25.5	3.000	0.8214
	39.90	1.0		1.0		2.000	0.8160
	2.500	20.25	1.0		0.000	0.5342
	
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_REAL_ARG_CONS(9): On entry, df1 = <value>.  This parameter must satisfy 0 < df1 <= 1e6.  On entry, lambda = <value>.  This parameter must satisfy 0 <= lambda <= -2 * log(X02AMC).
	NE_REAL_ARG_LE(6): On entry, df2 must not be less than or equal to 0: df2 = <value>.  On entry, f must not be less than or equal to 0: f = <value>.
	NE_INT_ARG_LT(11): On entry, max_iter must not be less than 1: max_iter = <value>.
	NE_CONV(420): The solution has failed to converge in <value> iterations, consider increasing max_iter or tol.
	NE_PROB_F(421): The required probability cannot be computed accurately.  This may happen if the result would be very close to zero or one.  Alternatively the values of df1 and f may be too large.  In the later case, the user could try using a normal approximation, see Abramowitz and Stegun (1972).
	NE_PROB_F_INIT(422): The required accuracy was not achieved when calculating the initial value of the central F or Chi-square probability.  The user should try a larger value of tol.  If the Ch-square approximation is being used then nag_prob_non_central_f_dist returns zero.  Otherwise, the value returned should be an approximation to the correct value.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function.  Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.
		
	successfully call of the nag_prob_non_central_f_dist function	
*/

	double nag_prob_non_central_f_dist(
		double f,  // the deviate from the F-distribution, F
		double df1,	// the degree of freedom of the numerator variance, v1
		double df2,	 // the degree of freedom of the denominator variance, v2
		double lambda, // the non-centrality parameter
		double tol=5e-6,	// the required accuracy of the solution
		int max_iter = 500, // the maximum number of iterations to be performed
		NagError *fail=NULL	// NAG error structure
	);	// Computes probabilities for the non-central F-distribution 


/**	g01gec
		the probability associated with the lower tail of the non-central
		beta distribution.
	
Example:

void test_nag_prob_non_central_beta_dist()
{
	double prob;
	int i;
	double a[3] = {1.0, 1.5,2.0};
	double b[3] = {2.0, 1.5, 1.0};
	double x[3] = {0.25, 0.75, 0.50};
	double lambda[3] = {1.0, 0.5, 0.0};

	printf("the input data and the results are as following\n");
	printf("	x 	a	 b      lambda   prob\n");
	for(i = 0; i < 3; i++)
	{
		prob = nag_prob_non_central_beta_dist(x[i], a[i], b[i], lambda[i], 5e-6, 50);
		printf("%8.3f%8.3f%8.3f%8.3f%8.4f%\n",x[i], a[i], b[i], lambda[i], prob);		
	}
}

	The output is following:
		
		x 	a	 b      lambda   prob
	   0.250   1.000   2.000   1.000  0.3168
	   0.750   1.500   1.500   0.500  0.7705
	   0.500   2.000   1.000   0.000  0.2500	
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_REAL_ARG_CONS(9):  On entry, x = <value>.  This parameter must satisfy 0 < x <= 1.  On entry, a = <value>.  This parameter must satisfy 0 < a <= 1e6.  On entry, b = <value>.  This parameter must satisfy 0 < b < 1e6.  On entry, lambda = <value>.  This parameter must satisfy 0 <= lambda <= -2 * log(X02AMC).
	NE_INT_ARG_LT(11): On entry, max_iter must not be less than 1: max_iter = <value>.
	NE_CONV(420):  The solution has failed to converge in <value> iterations, consider increasing max_iter or tol.
	NE_PROB_LIMIT(423): The probability is too close to 0 or 1 for the algorithm to be able to calculate the required probability. nag_prob_non_central_beta_dist will return 0 or 1 as appropriate.  This should be a reasonable approximation.
	NE_PROB_B_INIT(424):  The required accuracy was not achieved when calculating the initial value of the beta distribution.  The user should try a larger value of tol.  The returned value will be an approximation to the correct value.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function.  Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.
		
	successfully call of the nag_prob_non_central_beta_dist function	
*/

double nag_prob_non_central_beta_dist(
	const double x,	// the deviate from the beta distribution, for which the probability is to be found 
	double a,	// the first of the required beta distribution
	double b,	// the second parameter of the beta distribution
	double lambda,	// the non-centrality parameter of the beta distribution
	double tol=5e-6,	// the required accuracy of the solution
	int max_iter=100,	// the maximum number of terms used in each of the summations
	NagError* fail=NULL // NAG error structure
);	// Computes probabilities for the non-central beta distribution


/**	g01hac
		the lower tail probability for the bivariate Normal distribution.
	
Example:
	Values of x and y for a bivariate Normal distribution are read along 
	with the value of rho. The lower tail probabilities are computed.

void test_nag_bivariate_normal_dist()
{
	double prob;
	int i;
	double x[4] = {1.7, 0.0, 3.3,9.1};
	double y[4] = {23.1, 0.0, 11.1, 9.1};
	double rho[4] = {0.0, 0.1, 0.54, 0.17};
	
	printf("the input data and the results are as following\n");
	printf("	x 	y	 rho      prob\n");
	for(i = 0; i < 4; i++)
	{
		prob = nag_bivariate_normal_dist(x[i],	y[i], rho[i]);
		printf("%8.3f%8.3f%8.3f%8.4f\n",x[i],	y[i], rho[i], prob);		
	}	
}

	The output is following:
	
	x 		y	 	rho      prob
   1.700  23.100   0.000  	0.9554
   0.000   0.000   0.100  	0.2659
   3.300  11.100   0.540  	0.9995
   9.100   9.100   0.170  	1.0000
   
Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_REAL_ARG_LT(5): On entry, rho must not be less than -1.0: rho = _value_.
	NE_REAL_ARG_GT(7): On entry, rho must not be greater than 1.0: rho = _value_.
	
	successfully call of the nag_bivariate_normal_dist function
*/

double nag_bivariate_normal_dist(
	double x,	// the first arguement for which the bivariate Normal distribution funtion is to be evaluated
	double y,	// the second arguement for which the bivariate Normal distribution funtion is to be evaluated
	double rho,	// the correlation coefficent
	NagError* fail=NULL // NAG error structure
);	// Probability for the bivariate Normal distribution


/**	g01hbc
		the upper tail, lower tail or central probability associated with
		multivariate Normal distribution of up to ten dimensions.
	
Example:

void test_nag_multi_normal()
{
	double prob;
	int i;
	Nag_TailProbability  tail_enum;	
	double a[4] = {-2.0, -2.0, -2.0, -2.0};
	double b[4] = {2.0, 2.0, 2.0, 2.0};
	double sigma[16] = {1.0, 0.9, 0.9, 0.9, 0.9, 1.0, 0.9, 0.9, 
						0.9, 0.9, 1.0, 0.9, 0.9, 0.9, 0.9, 1.0};
	double mean[4] = {0.0, 0.0, 0.0, 0.0};
	
	tail_enum = Nag_Central;
	
	printf("for the inputs and corresponding results are as following\n");	
	printf("mean has the value:\n");
	for(i = 0; i < 4; i++)
		printf("%2.1f    ",mean[i]);
	printf("\na has the value:\n");
	for(i = 0; i < 4; i++)
		printf("%2.1f    ",a[i]);
	
	printf("\nb has the value:\n");
	for(i = 0; i < 4; i++)
		printf("%2.1f    ",b[i]);
	
	printf("\nsigma has the value:\n");
	for(i = 0; i < 16; i++)
	{
		printf("%2.1f    ", sigma[i]);
		if((i + 1) % 4 ==0)
			printf("\n");
	}
	
	prob = nag_multi_normal(tail_enum, 4, a, b, mean, sigma, 4, 0.0001, 2000);
	printf("multiivariate Normal probability = %5.4f\n",prob);
}

	The output is following:
	
	multiivariate Normal probability = 0.9142

Return:
	The function returns missing value (NANUM) if error occurs, 0 if no error.
	
	NE_INT_ARG_CONS(15): On entry, n = <value>.  This parameter must satisfy 1 <= n <= 10.  On entry, maxpts = <value>.  This parameter must satisfy maxpts >= 4*n, if n >= 3.
	NE_2_INT_ARG_LT(17): On entry, tdsig = <value> while n = <value>.  These parameters must satisfy tdsig >=n.
	NE_REAL_ARG_CONS(9): On entry, tol = <value>.  This parameter must satisfy tol > 0, if n > 1.
	NE_2_REAL_ARRAYS_CONS(60): On entry, a[<value>] = <value> while b[<value>] = <value>.  Constraint: if tail = Nag_Central, a[i] < b[i], for i = 0, 1, . . . , n - 1.
	NE_BAD_PARAM(70): On entry, parameter tail had an illlegal value.
	NE_ALLOC_FAIL(73): Memory allocation failed.
	NE_INTERNAL_ERROR(74): An internal error has occurred in this function.  Check the function call and array sizes.  If the function call is correct, please consult NAG for assistance.
	NE_POS_DEF(562): The matrix sigma is not positive definitely.
	NE_ACC(561): The requested accuracy cannot be achieved.  Try increasing tol.  The returned result is an approximation to the required result.
	NE_ROUND_OFF(563): Round_off error prevents the requested accuracy from being achieved.  Try increasing tol.  The returned result is an approximation to the required result.  This result will only occur if n = 3.
	
	successfully call of the nag_multi_normal function	
*/

double nag_multi_normal(
	Nag_TailProbability tail,
	int n,	// the number of dimensions
	const double a[],	// the lower bounds
	const double b[],	// the upper bounds
	const double mean[],	// the mean vector of the multivariate Normal distribution
	const double sigma[],	// the variance/covariance matrix of the multivariate Normal distribution
	int tdsig,	// the second dimension
	double tol=5e-6,	// the required accuracy of the solution
	int maxpts = 100,	// the maximum number of sub-intervals or the integrand evaluations
	NagError* fail=NULL // NAG error structure
);	// Computes probabilities for the multivariate Normal distribution

#endif //!_O_NAG_G01_H

