/*------------------------------------------------------------------------------*
 * File Name:fft_utils.h 														*
 * Creation: ER 5/15/03															*
 * Purpose: Origin's internal FFT analysis routines								*
 * Copyright (c) Originlab Corp.	2003										*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/

 
#ifndef _FFT_UTILS_H
#define _FFT_UTILS_H

/** >Analysis
		Performs circular convolution of signal with response function. 
		The convolution is performed by calling the NAG convolution function.
		The length of the signal may be truncated by a few points to make length 
		compatible with NAG call. See nag_convolution_real for more details
	Example:
		int iRet = fft_convolute(vecSignal, vecResponse, vecResult);
	Parameters:
		vecSignal: vector with Signal data
		vecResponse: vector with Response data
		vecResult: vector to return result of convolution
		bNormalize = true: normalize sum of response elements to unity, false: do not normalize
		bWrap = true: wrap response so that max value is at 0th element, false: do not wrap
	Return:
		Returns 0 for no errors, positive number for any NAG-returned error code.
*/
int fft_convolute(vector& vecSignal, vector& vecResponse, vector& vecResult, bool bNormalize = true, bool bWrap = true);


/** >Analysis
		Performs deconvolution of signal with response function. 
		The deconvolution is performed by calling NAG FFT functions.
		The length of the signal may be truncated by a few points to make length 
		compatible with NAG call. See nag_convolution_real for more details
	Example:
		int iRet = fft_deconvolute(vecSignal, vecResponse, vecResult);
	Parameters:
		vecSignal: vector with Signal data
		vecResponse: vector with Response data
		vecResult: vector to return result of deconvolution
		bNormalize = true: normalize sum of response elements to unity, false: do not normalize
		bWrap = true: wrap response so that max value is at 0th element, false: do not wrap
	Return:
		Returns 0 for no errors, positive number for any NAG-returned error code.
*/
int fft_deconvolute(vector& vecSignal, vector& vecResponse, vector& vecResult, bool bNormalize = true, bool bWrap = true);


/** >Analysis
		Performs correlation of two signals. 
		The correlation is performed by calling NAG FFT functions.
		If specified, the correlation result is normalized to be in the range of +1 to -1.
		The normalization is performed by dividing the result by the following factor:
			square root( (sum of FFT amplitudes of signal 1) * (sum of FFT amplitudes of signal 2) ).
	Example:
		int iRet = fft_correlate(vecSignal1, vecSignal2);
	Parameters:
		vecSignal1: vector with Signal1 data
		vecSignal2: vector with Signal2 data
		vecCorrelation: vector with correlation result
		bNormalize = true: normalize corelation result to +/-1 range
	Return:
		Returns 0 for no errors, positive number for any FFT-related error code.
*/
int fft_correlate(vector& vecSignal1, vector& vecSignal2, vector& vecCorrelation, bool bNormalize = true);


/** >Analysis
		Performs low pass filtering of signal.
		The filtering is performed by using NAG FFT functions.
	Example:
		int iRet = fft_lowpass(crvSignal, dFc);
	Parameters:
		crvSignal:	Input: signal data; Output: filtered signal
		dFc:		cutoff frequency
	Return:
		0:			success
		-1:			curve does not meet time resoultion criterion
		-2:			curve is invalid
		-3:			invalid cutoff frequency
		positive:	NAG error code
*/
int fft_lowpass(Curve& crvSignal, double dFc);


/** >Analysis
		Performs high pass filtering of signal.
		The filtering is performed by using NAG FFT functions.
	Example:
		int iRet = fft_highpass(crvSignal, dFc);
	Parameters:
		crvSignal:	Input: signal data; Output: filtered signal
		dFc:		cutoff frequency
		bAddOffset:	add back DC offset after filtering
	Return:
		0:			success
		-1:			curve does not meet time resoultion criterion
		-2:			curve is invalid
		-3:			invalid cutoff frequency
		positive:	NAG error code
*/
int fft_highpass(Curve& crvSignal, double dFc, bool bAddOffset = true);


/** >Analysis
		Performs band pass filtering of signal.
		The filtering is performed by using NAG FFT functions.
	Example:
		int iRet = fft_bandpass(crvSignal, dFLow, dFHigh);
	Parameters:
		crvSignal:	Input: signal data; Output: filtered signal
		dFLow:		lower cutoff frequency
		dFHigh: 	higher cutoff frequency
		bAddOffset:	add back DC offset after filtering
	Return:
		0:			success
		-1:			curve does not meet time resoultion criterion
		-2:			curve is invalid
		-3:			invalid cutoff frequency
		positive:	NAG error code
*/
int fft_bandpass(Curve& crvSignal, double dFLow, double dFHigh, bool bAddOffset = true);


/** >Analysis
		Performs band block filtering of signal.
		The filtering is performed by using NAG FFT functions.
	Example:
		int iRet = fft_bandblock(crvSignal, dFLow, dFHigh);
	Parameters:
		crvSignal:	Input: signal data; Output: filtered signal
		dFLow:		lower cutoff frequency
		dFHigh: 	higher cutoff frequency
		bAddOffset:	add back DC offset after filtering
	Return:
		0:			success
		-1:			curve does not meet time resoultion criterion
		-2:			curve is invalid
		-3:			invalid cutoff frequency
		positive:	NAG error code
*/
int fft_bandblock(Curve& crvSignal, double dFLow, double dFHigh, bool bAddOffset = true);


/** >Analysis
		Performs threshold filtering of the signal. All components below the
		specified threshold amplitude in the frequency spectrum of the signal are removed.
		The filtering is performed by using NAG FFT functions.
	Example:
		int iRet = fft_threshold_filter(crvSignal, dThreshold);
	Parameters:
		crvSignal:	Input: signal data; Output: filtered signal
		dThreshold:	threshold value; must be greater than 0
	Return:
		0:			success
		-1:			curve does not meet time resoultion criterion
		-2:			curve is invalid
		-3:			invalid threshold
		positive:	NAG error code
*/
int fft_threshold_filter(Curve& crvSignal, double dThreshold);



#endif //_FFT_UTILS_H




