/*------------------------------------------------------------------------------*
 * File Name:	Folder.h														*
 * Creation:	10/24/2001														*
 * Purpose: Origin C header	for Project Explorer - related classes and other	*
 *			related functions													*
 * Copyright (c) OriginLab Corp.2001											*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/

#ifndef __FOLDER_H__
#define __FOLDER_H__

#include <common.h> // must always include this, has printf etc
#include <string.h> // most likely you will also need strings
#include <OC_const.h> // consts used in Origin internal functions
#include <stdio.h>	// stdio function like printf
#include <wksheet.h> // graph, plot and graphic objects
#include <Page.h> // page related classes
#include <Collection.h> //Contains declaration of the template class Collection

////////////////////////////////////////////////////////////////////////////////////
// start your functions here


/** >Internal Origin Objects
		Project Explorer is a Windows Explorer like user interface inside Origin. It
		contains a folder/sub-folder structure used to organize and facilitate access to
		the graph, layout, matrix, note, and worksheet windows of an Origin project file.
		The Folder class is used to access the methods and properties of Project Explorer
		and contains collections of all Origin pages and Project Explorer folders. An
		Origin C Folder object is a wrapper object that is a reference to an internal
		Origin project explorer object. Origin C wrapper objects do not actually exist
		in Origin and merely refer to the internal Origin object. Consequently, multiple
		Origin C wrapper objects can refer to the same internal Origin object.
	Example:
		// List all page names in first level subfolders of root folder
		Folder fld, fldSub;
		fld = Application.RootFolder;
		PageBase pb;
		
		foreach(fldSub in fld.Subfolders)
		{
			// Display the name of the current subfolder
			printf( "Folder name = %s\n", fldSub.GetName() );
			foreach(pb in fldSub.Pages)
			{
				// Display the name of the page:
				printf( "\tPage name = %s\n", pb.GetName() );
			}
		}
*/
class Folder
{
public:
	/**
	Example:
		This example constructs a folder, and adds a subfolder named MyFolder.
		void run_Folder()
		{
			Folder fldRoot;
			fldRoot = Application.RootFolder;
			fldRoot.AddSubfolder("MyFolder");
		}
	*/
	Folder();

	/**
		Copy constructor.
	Example:
		This example illustrates how to copy a constructor. Folder fldRoot and fldRootpp both refer to the root.
		void run_Folder()
		{
			Folder fldRoot;
			fldRoot = Application.RootFolder;
			fldRoot.AddSubfolder("MyFolder1");
			fldRoot.AddSubfolder("MyFolder2");
				
			Folder fldRootpp(fldRoot);
			fldRootpp.RemoveSubFolder("MyFolder2");
		}
	*/
	Folder(Folder &folder);

	/**
		Construct a Folder object by pathname.
	Example:
		void run_Folder()
		{	
			Folder fldRoot("/");
			fldRoot.AddSubfolder("MyFolder");
			
			Folder fldMyFolder("/MyFolder");
			fldMyFolder.AddSubfolder("MySubFolder");
		}
	*/
	Folder(LPCSTR lpcszPathName);
	/**
		Set the project's current folder.
	Example:
		Folder fld = Application.ActiveFolder();
		Folder subfld = fld.AddSubfolder("MyFolder");
		subfld.Activate();
	*/
	BOOL Activate();

	/**
		Get the name of the folder.
	Example:
		void test_Folder_GetName()
		{
			Folder fldRoot("/");
			fldRoot.AddSubfolder("MyFolder");
			Folder fldMyFolder("/MyFolder");
			
			string strCurrFld = fldMyFolder.GetName();
			printf("The current folder is \"%s\"", strCurrFld);
		}
	*/
	string GetName();
	
	/**
		Get the name of the folder.
	Example:
		Folder fld = Application.ActiveFolder();
		string strName;
		if( fld.GetName(strName) )
			printf("Current folder is: %s", strName);
	*/
	BOOL GetName(string &strName); // return FALSE if folder is not valid

	/**
		Rename the folder.  If the new folder name is used by a sibling folder then this function will return FALSE.
	Example:
		Folder fld = Application.ActiveFolder();
		fld.Rename("Example");
		string strName = fld.GetName();
		printf("Folder renamed to %s\n", strName);
	*/
	BOOL Rename(LPCSTR strName);

	/**
		Get the full path of the folder.
	Example:
		Folder fld = Application.ActiveFolder();
		printf("The path of Current folder is: %s", fld.GetPath());
	*/
	string GetPath(); // Return full path of this folder

	/**
		Get the folder's parent folder.
	Example:
		Folder fld = Application.ActiveFolder();
		Folder parentFld = fld.GetParent();
		if( parentFld.IsValid() )
			printf("The parent of Current folder is: %s", parentFld.GetName());
	*/
	Folder GetParent();
	/**
		Can use	relative and absolute way to Get a Folder, the name is no case sensitive.
	Example:
		"/ttt", use absolute way, the project can be omitted by this way.
		"../ttt", use relative way, it find folder name is "ttt" from current folder's parents' folder.
		
		void test_Folder_GetFolder()
		{
			Folder fldRoot("/");
			fldRoot.AddSubfolder("MyFolder1");
			Folder fldMyFolder1("/MyFolder1");
			fldRoot.AddSubfolder("MyFolder2");
			Folder fldMyFolder2("/MyFolder2");
			
			Folder fldCurr = fldRoot.GetFolder("/MyFolder1");
			string strCurrName = fldCurr.GetName();
			printf("The current folder is %s\n", strCurrName);
			
			fldCurr = fldMyFolder1.GetFolder("../MyFolder2");
			strCurrName = fldCurr.GetName();
			printf("The current folder is %s", strCurrName);
		}	
	*/
	Folder	GetFolder(LPCSTR strPathName);

	/**
		Move an item from the current folder to another folder whose path name is lpcszPathName.
	Examples:
		// Assume there is a window named "data1" and susbfolders named "sub1" and "sub2" under the root folder.
		Folder root = Application.RootFolder;
		if( !root.Move("data1", "/sub1/") ) // move window "data1" to folder "sub1".
			printf("Can't move this window!");
		if( !root.Move("sub2", "/sub1/") ) // move subfolder "sub2" to folder "sub1".
			printf("Can't move this subfolder!");
	*/
	BOOL Move(LPCSTR lpcszItemName, LPCSTR lpcszPathName);

	/**
		Attach the folder object to a folder.
	Example:
		Folder fld;
		fld.Attach("/");  // attached to root folder
	*/
	Attach(LPCSTR lpcszPathName);

	/**
		Check if the folder object is valid.  Must be used after creating a new folder object.
	Example:
		void test_Folder_IsValid()
		{
			Folder fldRoot("/");
			fldRoot.AddSubfolder("MyFolder");
			Folder fldMyFolder("/MyFolder");
			
			if (fldRoot.IsValid())
				printf("The rootfolder is valid.\n");
			if (fldMyFolder.IsValid())
				printf("The subfolder is valid.\n");
		}
	SeeAlso:
		Folder::GetParent
	*/
	BOOL IsValid();

	/**
		Check if the folder is the root folder.
	Example:
		Folder fld = Application.RootFolder;
		if( fld.IsRootFolder() )
			printf("Folder is root folder.");
	*/	
	BOOL IsRootFolder();

	/**
		Remove the subfolder with the name specified in lpcszSubFolderName.
		If bFolderPrompt or bPagePrompt is TRUE then Origin will prompt the user to verify.
		Example:
		Folder fld = Application.RootFolder;
		Folder subfld;
		foreach (subfld in fld.Subfolders)
		{
			fld.RemoveSubFolder(subfld.GetName());
		}
	*/
	BOOL RemoveSubFolder(LPCSTR lpcszSubFolderName, BOOL bFolderPrompt = FALSE , BOOL bPagePrompt = FALSE);

	/**
		Create a subfolder with the name specified in lpcszSubFolderName.
		Example:
		Folder fld = Application.RootFolder;
		fld.AddSubfolder("MyFolder");
		
	*/
	Folder AddSubfolder(LPCSTR lpcszSubFolderName); 

	/**
		Get a folder by index.  First folder has index of zero.
		Example:
		Folder fld = Application.RootFolder;
		Folder subfld = fld.Subfolders(0);
		if(subfld.IsValid())
		{
			out_str("The name of the subfolder is : " + subfld.GetName());
		}	
	*/
	Folder Subfolders(int nIndex);

	/**
		Get a folder by name.
		Example:
		Folder fld = Application.RootFolder;
		Folder subfld = fld.Subfolders("MyFolder");
	*/
	Folder Subfolders(LPCSTR lpcszName);

	/**
		Get a page by index.  First page has index of zero.
		Example:
		Folder fld = Application.RootFolder;
		Page pg = fld.Pages(0);
		if(pg.IsValid())
		{
			out_str("The name of the first page is : " + pg.GetName());
		}
	*/
	PageBase Pages(int nIndex);

	/**
	Return:
		Get a page by name.
		Example:
		Folder fld = Application.RootFolder;
		Page pg = fld.Pages("Data1");
	*/
	PageBase Pages(LPCSTR lpcszName);

	/**
		Collection of Subfolders in the folder.
	Example:
		string strName;
		Folder fld = Application.ActiveFolder();
		foreach(Folder sub in fld.Subfolders)
		{
			strName = sub.GetName();
			printf("<Folder> %s\n", strName );
		} 
		foreach(PageBase page in fld.Pages)
		{
			strName = page.GetName();
			printf("%s\n", strName );
		}
	*/
	Collection<Folder> Subfolders;

	/**
		Collection of Pages in the folder.
	SeeAlso:
		Folder::Subfolders
	*/
	Collection<PageBase> Pages;
#if  _OC_VER >= 0x0750
	/**
		Save folder as project
	Return:
		TRUE if successful. FALSE otherwise
	Parameters:
       lpcszPath = specify the path
       dwCntrl = additional flags to control the saving
       may include one or more of the following bits defined in OC_const.h
   	    SFC_INCLUDE_SUBFOLDERS 		- specifies if subfolders must be saved to project
       	SFC_INCLUDE_ATTACHED_FILES  - specifies if attached OC files must be saved to project
    Example:
    	void Sample_SaveAsProject()
		{
			Folder fldSave = Application.ActiveFolder();
			string strPath = GetAppPath(1) + "OriginC\\abc";
			bool bOK = fldSave.SaveAsProject(strPath);
			out_int("bOK = ", bOK);
		}
	*/
	BOOL	SaveAsProject(LPCSTR lpcszPath, DWORD dwCntrl = SFC_INCLUDE_SUBFOLDERS);
#endif //_OC_VER >= 0x0750
};

#endif // __FOLDER_H__
