/*------------------------------------------------------------------------------*
 * File Name: graph.h															*
 * Creation: 9/5/2001														    *
 * Purpose: Origin C header	for GraphLayer class and other related functions	*
 * Copyright (c) OriginLab Corp.2001											*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/
#ifndef _GRAPH_H
#define _GRAPH_H

#include <common.h> // must always include this, has printf etc

#ifndef _STRING_H
#include <string.h> // most likely you will also need strings
#endif // _STRING_H

#include <OC_const.h> // consts used in Origin internal functions
#include <oPlotIDs.h> // IDM_PLOT_* ids and related const

#ifndef _DATA_H
#include <data.h>
#endif // _DATA_H

#include <ORgObj.h>

/** >Internal Origin Objects
		The Layer class provides methods and properties common to all internal Origin
		layers. All Origin pages (windows) except note pages contain one or more layers.
		Origin objects found "on" a page are generally contained by layers which are
		themselves contained by the page. Origin layers may contain many different graph
		objects thus the Layer class contains a collection of all the graph objects
		in the layer. An Origin C Layer object is a wrapper object that is a reference
		to an internal Origin layer object. Origin C wrapper objects do not actually
		exist in Origin and merely refer to the internal Origin object. Consequently,
		multiple Origin C wrapper objects can refer to the same internal Origin object.
		The Layer class is derived from the OriginObject class from which it inherits
		methods and properties.
	Example:
		// This example assumes Graph1 with X axis label "XB" is active window/layer in Origin
		Layer lyr;                   // Declare a Layer object named lyr
		lyr = Project.ActiveLayer(); // Attach lyr to active layer in "Graph1"
		GraphObject go ;             // Declare a GraphObject named go
		go = lyr.GraphObjects("XB"); // Attach go to X Axis Label with name "XB"
		if(go.IsValid())             // If graph object exists...
			printf("Name of GraphObject is %s\n", go.GetName()); // Display its name
*/
class Layer : public OriginObject
{
public:
	
	/**
		Default constructor which constructs an unattached Layer object.
		The member function IsValid() of the base class OriginObject
		can be used to determine if the object is valid (attached to an internal
		Origin layer object)
	Parameters:
		None
	Example:
		void	run_Layer_constructor()
		{
			Layer	lay;
			
			if (lay.IsValid())
				out_str("Object is valid");
			else
				out_str("Object is not valid");	// it should be invalid
		}
	*/
	Layer();
	
	
	/**
		Copy constructor which constructs a Layer object from an existing
		Layer object.
		The member function IsValid() of the base class OriginObject
		can be used to determine if the object is valid (attached to an internal
		Origin layer object)
	Parameters:
		layer = the source Layer object
	Example:
		void	run_Layer_constructor()
		{
			Layer	lay;
			Layer	lay2(lay);
			
			if (lay2.IsValid())
				out_str("Object is valid");
			else
				out_str("Object is not valid");	// it should be invalid
		}
	*/
	Layer(Layer & layer);

	/**
		It returns the index'th GraphObject in the layer.
		GraphObjects are text labels, rectangles, lines, arrows, etc, which
		can be drawn inside graphs, worksheets, and layouts,
	Parametes:
		index = GraphObject index
	Returns:
		the index'th GraphObject in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Layer_GraphObjects()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			GraphObject		grobj;
			
			// Get the first GraphObject in the layer:
			grobj = lay.GraphObjects(0);
			
			// Make sure that the object is valid:
			if (!grobj.IsValid())
			{
				out_str("No GraphicObject!");
				return;
			}
				
			// Display its name:
			printf("The name of the first GraphObject in layer is %s\n", grobj.GetName());
		}
	*/
	GraphObject	GraphObjects(int index);
	
	/**
		It returns the GraphObject with the given name in the layer.
		GraphObjects are text labels, rectangles, lines, arrows, etc, which
		can be drawn inside graphs, worksheets, and layouts,
	Parametes:
		lpcszName = pointer to the string representing the name of the GraphObject
	Returns:
		the GraphObject with the name lpcszName in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, in the active layer create a text label with the name "Text". You can change
		// the name of the text label GraphObject using the "Label Control" dialog box (use
		// the context menu on the text label to bring up the dialog).
		void	run_Layer_GraphObjects()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			GraphObject		grobj;
			
			// Get the GraphObject with the name "Text":
			grobj = lay.GraphObjects("Text");
			
			// Make sure that the object is valid:
			if (!grobj.IsValid())
			{
				out_str("No GraphicObject!");
				return;
			}
				
			// Display its name:
			printf("The name of the GraphObject is %s\n", grobj.GetName());
		}
	*/
	GraphObject	GraphObjects(LPCSTR lpcszName);

	/**
		It gets the parent Page object (the Page containing this layer).
	Parameters:
		None
	Returns:
		the parent Page object
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Layer_GetPage()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			Page			pg;
			
			// Get the parent page:
			pg = lay.GetPage();
			
			printf("The name of the parent page is %s\n", pg.GetName());
		}
	*/
	Page		GetPage();
	
	/**
		It detaches the internal Origin layer from this object.
	Parameters:
		None
	Returns:
		void
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Layer_Detach()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
		
			// We can check if the object lay is valid (attached to 
			// an internal Origin layer) by calling the method "IsValid)"
			if ( lay.IsValid() )
				out_str("Object is valid");			// it should valid (if "Graph1" exists)
			else
				out_str("Object is not valid");	
			
			// Now detach:
			lay.Detach();
				
			if ( lay.IsValid() )
				out_str("Object is valid");
			else
				out_str("Object is not valid");	// it should be invalid because we detached it
		}
	*/
	void		Detach();

	/**
		Removes one GraphObject from layer. The object is referred to by index.
	Parameters:
		index = the GraphObject index
	Return:
		TRUE for success, otherwise FALSE
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Layer_RemoveGraphObject()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			GraphObject		grobj;
			
			// Get the first GraphObject in the layer:
			grobj = lay.GraphObjects(0);
			
			// Make sure that the object is valid:
			if (!grobj.IsValid())
			{
				out_str("No GraphicObject!");
				return;
			}
				
			// Display its name:
			printf("The name of the first GraphObject is %s\n", grobj.GetName());
			
			// Now remove the first GraphObject:
			lay.RemoveGraphObject(0);
			
			// Get the new first GraphObject in the layer:
			grobj = lay.GraphObjects(0);
		
			// Make sure that the object is valid:
			if (!grobj.IsValid())
			{
				out_str("No GraphicObject!");
				return;
			}
			
			// Display its name:
			printf("The name of the first GraphObject is %s\n", grobj.GetName());
		}

	*/
	BOOL		RemoveGraphObject(int index);
	
	/**
		Removes one GraphObject from layer. The object is referred to by name (lpcszName).
	Parameters:
		lpcszName = the GraphObject name
	Return:
		TRUE for success, otherwise FALSE
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, in the active layer create a text label with the name "Text". You can change
		// the name of the text label GraphObject using the "Label Control" dialog box (use
		// the context menu on the text label to bring up the dialog).
		void	run_Layer_RemoveGraphObject()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			GraphObject		grobj;
			
			// Get the GraphObject named "Text" in the layer:
			grobj = lay.GraphObjects("Text");
			
			// Make sure that the object is valid:
			if (!grobj.IsValid())
			{
				out_str("No GraphicObject!");
				return;
			}
				
			// Display its name:
			printf("The name of the GraphObject is %s\n", grobj.GetName());
			
			// Now remove the GraphObject with the name "Text":
			lay.RemoveGraphObject("Text");
			
			// Get the the GraphObject named "Text" in the layer again:
			grobj = lay.GraphObjects("Text");
		
			// Make sure that the object is invalid (because we removed it above):
			if (!grobj.IsValid())
			{
				out_str("No GraphicObject!");
				return;
			}
		}
	*/
	BOOL		RemoveGraphObject(LPCSTR lpcszName);		


	/**
		The Collection property of all GraphObjects in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Layer_GraphObjects_Collection()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			GraphObject					grobj;
			Collection<GraphObject>		grobjcollection;
			
			// Get the collection object from the layer:
			grobjcollection = lay.GraphObjects;
			
			// Use foreach-loop to loop over all the GraphObjects in the collection 
			foreach (grobj in grobjcollection)
			{
				// Display the name:
				out_str(grobj.GetName());
			}
		}
	*/
	Collection<GraphObject> GraphObjects;
	
	/**
		Get the index of the Layer in the parent page, 0 offset
	Remark:
		This method returns 1-offset index in Origin 7, and in Origin 7.5 and later, it has been made consistent to return 0-offset index.
	Return:
		The index of the specific layer in its parent page
	Example:		
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the window has several layers.
		// Make layer 2 active before running the function.
		void	run_GetIndex()
		{
			// Create a Layer object and attach it to the active layer in "Graph1" graph window.
			GraphLayer		lay("Graph1");
			
			// Displaye the active index (it should be "2" if the active layer is the second layer):
			out_int("Active layer index is ", lay.GetIndex() + 1); // convert C index to LabTalk index
		}
	*/
	int			GetIndex();

	/**
		Execute labtalk script with the current layer as the active layer. This command has the combination effect of labtalk's "win -o" and "layer -o"
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the window has several layers.
		// Make layer 2 active before running the function.
		void	run_Layer_LT_execute()
		{
			GraphPage	page("Graph1");
			
			// Create a Layer object and attach it to the first layer in "Graph1" graph window.
			GraphLayer		lay = page.Layers(0);
			
			double			xx = -10.0;
			
			// we will set X scales in the layer using LabTalk script:
			string			str;
			// Construct the LabTalk script to be executed:
			str.Format("layer.x.from=%f;layer.x.to=%f;", xx, xx + 10.0);
			
			// Execute the script. After this script executes, the first layer
			// will have the x-axis running from -10 to 0.
			lay.LT_execute(str);
		}
	*/
	BOOL	LT_execute(LPCSTR lpcszstr, int wCntrl = 0);

	#if  _OC_VER >= 0x0750
	/**
		Get the layer's internal info
	Return:
		a generic value that depends on the request
	Parameters:
       nType = specify the info to obtain
       
	*/
	DWORD GetSystemParam(int nParamType);
	#endif //_OC_VER >= 0x0750
};

/** >Internal Origin Objects
		The Layout class provides methods and properties common to all internal Origin layout
		layers. Origin layout pages contain a layout layer which contains other objects on the
		layout page. An Origin C Layout object is a wrapper object that is a reference to an internal
		Origin layout object. Origin C wrapper objects do not actually exist in Origin and merely
		refer to the internal Origin object. Consequently, multiple Origin C wrapper objects
		can refer to the same internal Origin object. The Layout class is derived from the
		Layer and OriginObject classes from which it inherits methods and properties.
	Example:
		// This example assumes Layout1 window with text object named Text
		// is active window/layer in Origin
		Layout lo;                              // Declare a Layout object
		lo = (Layout) Project.ActiveLayer();    // Attach lo to active Layout (layer); must explicitly cast
		GraphObject go ;                        // Declare a GraphObject
		go = lo.GraphObjects("Text");           // Attach go to text object named Text
		if(go.IsValid())                        // If graph object exists...
			printf("Name of GraphObject is %s\n", go.GetName()); // Display its name
*/
class Layout : public Layer
{
public:
	
	/** >Internal Origin Objects
		Constructor which constructs a Layout object given the name of a layout window.
		The object will be valid if the layout window with the given name exists.
	Parameters:
		layoutname = the name of a layout window in Origin
	Example:
		// For this example to run, a layout window with the name "Layout1" must exist in the project.
		void	run_Layout_Constructor()
		{
			Layout	la("Layout1");
			
			// We can check if the object la is valid (attached to 
			// an internal Origin layout) by calling the method "IsValid)"
			if (la.IsValid())
				out_str("Object is valid");		// it should be valid
			else
				out_str("Object is not valid");	
		}
	*/
	Layout(LPCSTR layoutname);
	
	/**
		Copy constructor which constructs a Layout object from an existing
		Layer object.
		The member function IsValid() of the base class OriginObject
		can be used to determine if the object is valid (attached to an internal
		Origin layout object).
	Parameters:
		layer = the source Layer object in a layout window
	Example:
		// For this example to run, a layout window with the name "Layout1" must exist in the project.
		void	run_Layout_Constructor()
		{
			// Create an attach a source Layout object:
			Layout	lay("Layout1");
			
			Layout	la(lay);
			
			// We can check if the object la is valid (attached to 
			// an internal Origin layout) by calling the method "IsValid)"
			if (la.IsValid())
				out_str("Object is valid");
			else
				out_str("Object is not valid");	// it should be invalid
		}
	*/
	Layout(Layer & layer);	
	
};

/** >Internal Origin Objects
		The Scale class provides methods and properties common to all Origin axis scales.
		Two scale objects (one X scale and one Y scale) are contained in each graph layer
		on each graph page. An Origin C Scale object is a wrapper object that is a reference
		to an internal Origin scale object. Origin C wrapper objects do not actually exist
		in Origin and merely refer to the internal Origin object. Consequently, multiple
		Origin C wrapper objects can refer to the same internal Origin object. The Scale 
		class is derived from the OriginObject class from which it inherits methods and
		properties.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Assumes Graph1 window is active.
		GraphLayer	gl = Project.ActiveLayer(); // Get active layer in project file
		if(gl)                                  // If valid graph layer...
		{
			Scale sc(gl.X);                     // Get X scale object from graph layer
			sc.From = 0.01;                     // Assign scale from value
			sc.To = 1000;                       // assign scale to value
			sc.Type = LOG10_SPACE;              // Set axis scale as Log10
			sc.Inc = 1;                         // set axis scale increment to 1
		}
*/
class Scale : public OriginObject
{
public:
	
	/**
		Default constructor which constructs an unattached Scale object.
		The member function IsValid() of the base class OriginObject
		can be used to determine if the object is valid (attached to an internal
		Origin axis)
	Parameters:
		None
	Example:
		void	run_Scale_Constructor()
		{
			// Create an unattached Scale object
			Scale			s;
			
			if (!s.IsValid())
			{
				out_str("Invalid Scale object!");	// must be invalid
				return;
			}
		}
	*/
	Scale();
	
	/**
		Copy constructor which constructs a Scale object from another
		Scale object.
	Parameters:
		scale = the source scale object
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_Constructor()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			if (!gl.IsValid())
			{
				out_str("No GraphLayer!");
				return;
			}
			
			// Intialize a Scale object from the X-axis scale of the GraphLayer
			// (gl.X is the X-axis scale for the layer attched to "gl").
			Scale			s(gl.X);
			
			// Display the "From" property:
			out_double("From = ", s.From);
		}
	*/
	Scale(Scale & scale);
	
	/**
		The "From" property - the left limit of the axis
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_From()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			if (!gl.IsValid())
			{
				out_str("No GraphLayer!");
				return;
			}
			
			// Intialize a Scale object from the Y-axis scale of the GraphLayer
			// (gl.Y is the Y-axis scale for the layer attched to "gl").
			Scale			s(gl.Y);
			
			// Display the "From" property:
			out_double("From = ", s.From);
		}
	*/
	double 	From;
	
	/**
		The "To" property - the right limit of the axis
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_To()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			if (!gl.IsValid())
			{
				out_str("No GraphLayer!");
				return;
			}
			
			// Intialize a Scale object from the Y-axis scale of the GraphLayer
			// (gl.Y is the Y-axis scale for the layer attched to "gl").
			Scale			s(gl.Y);
			
			// Display the "To" property:
			out_double("To = ", s.To);
		}
	*/
	double 	To;
	
	/**
		The "Inc" property - the major tick increment of the axis
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_Inc()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			if (!gl.IsValid())
			{
				out_str("No GraphLayer!");
				return;
			}
			
			// Intialize a Scale object from the Y-axis scale of the GraphLayer
			// (gl.Y is the Y-axis scale for the layer attched to "gl").
			Scale			s(gl.Y);
			
			// Display the "Inc" property:
			out_double("Inc = ", s.Inc);
		}
	*/
	double 	Inc;
	
	/**
		The "Type" property - the axis type. Possible values are:
			LINEAR_SPACE			- linear
			LOG10_SPACE				- logarithm base 10
			PROB_SPACE				- probability
			PROBIT_SPACE			- probit
			RECIPROCAL_SPACE		- reciprocal
			OFFSET_RECIP_SPACE		- offset reciprocal
			LOGIT_SPACE				- logit
			LOGE_SPACE				- natural logarithm
			LOG2_SPACE				- logarithm base 2
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_Type()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			if (!gl.IsValid())
			{
				out_str("No GraphLayer!");
				return;
			}
			
			// Intialize a Scale object from the Y-axis scale of the GraphLayer
			// (gl.Y is the Y-axis scale for the layer attched to "gl").
			Scale			s(gl.Y);
			
			// Display the "Type" property:
			out_double("Type = ", s.Type);
		}
	*/
	int 	Type;
};

/** >Internal Origin Objects
		The GraphLayer class provides methods and properties common to all Origin graph
		layers. Internal Origin graph pages contain one or more graph layers thus the
		Origin C GraphPage class contains a collection of one or more GraphLayer objects.
		Origin graph layers may contain Origin data plots thus the Origin C GraphLayer
		class contains a collection of DataPlot objects. An Origin C GraphLayer object
		is a wrapper object that is a reference to an internal Origin graph layer object.
		Origin C wrapper objects do not actually exist in Origin and merely refer to the
		internal Origin object. Consequently, multiple Origin C wrapper objects can refer
		to the same internal Origin object. The GraphLayer class is derived from the Layer
		and OriginObject classes from which it inherits methods and properties.
	Example:
		// Assumes Graph1 window with one or more data plots is active in Origin
		GraphLayer	gl = Project.ActiveLayer(); // Get active layer in project file
		if(gl)                                  // If valid graph layer...
		{
			DataPlot dp = gl.DataPlots(0);      // Get first data plot in graph layer
			dp.SetColor(2);                     // Change color to green
		}
*/ 
class GraphLayer : public Layer
{
public:
	/**
		Construct and attach a GraphLayer object based on the name of the graph window
		and the layer index
	Parameters:
		lpcszWinName = the name of the graph window (page)
		nLayerNum = (optional) 0-offset index of the layer (use -1
					for the active layer)
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the window has at least two layers.
		// Make layer 2 active before running the function.
		void	run_GraphLayer_Constructor()
		{
			// Construct a GraphLayer object and attach it to the second layer of "Graph1" window.
			// (note that in the constructor the layer index is 0-offset, so the
			// first layer needs index 0)
			GraphLayer		gl("Graph1", 1);
			
			// Display the index of the layer in graph page
			// (note that the index returned from the "GetIndex" method
			// is 0-offset, so the first layer will have index 0):
			out_int("Layer index = ", gl.GetIndex() + 1);
		}
	*/
	GraphLayer(LPCSTR lpcszWinName, int nLayerNum=-1);
	
	/**
		Copy constructor which constructs a GraphLayer object from an existing
		Layer object. Note that the existing Layer object must also be in a garph page.
	Parameters:
		layer = the source Layer object
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the window has at least two layers.
		// Make layer 2 active before running the function.
		void	run_GraphLayer_Constructor()
		{
			// Construct a GraphLayer object and attach it to the second layer of "Graph1" window.
			// (note that in the constructor the layer index is 0-offset, so the
			// first layer needs index 0)
			GraphLayer		gl("Graph1", 1);
			
			GraphLayer		gl2(gl);
			
			// Display the index of the layer in graph page
			// (note that the index returned from the "GetIndex" method
			// is 1-offset, so the first layer will have index 1):
			out_int("Layer index = ", gl2.GetIndex());
		}
	*/
	GraphLayer(Layer & layer);
	
	/**
		Attached a layer in an Origin graph window to the object based on the graph name
		and the layer index.
	Parameters:
		lpcszWinName = the name of an existing Origin graph window
		nLayerNum = (optional) the 0-offset index of the layer in the graph window to attach
					or -1 for the active layer.
	Returns:
		TRUE for success, otherwise FALSE.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the window has at least two layers.
		// Make layer 2 active before running the function.
		void	run_GraphLayer_Attach()
		{
			// Construct a GraphLayer object and attach it to the second layer of "Graph1" window.
			// (note that in the constructor the layer index is 0-offset, so the
			// first layer needs index 0)
			GraphLayer		gl("Graph1", 1);
			
			// Display the index of the layer in graph page
			// (note that the index returned from the "GetIndex" method
			// is 1-offset, so the first layer will have index 1):
			out_int("Layer index = ", gl.GetIndex());
			
			// Now attach the first layer:
			gl.Attach("Graph1", 0);
		
			// Display the index of the layer in graph page.
			// (note that the index returned from the "GetIndex" method
			// is 1-offset, so the first layer will have index 1):
			out_int("Layer index = ", gl.GetIndex());
		}
	*/
	BOOL Attach(LPCSTR lpcszWinName, int nLayerNum=-1);
	
	/**
		Add one data plot to layer
	Parameters:
		cData = Data curve to add
		nPlotID = the internal Origin plot type id..
		wCntrl = (optional) not used 
	Return:
		0-offset data plot index of the dataplot added or -1 for error
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, a workseet with the name "Data1" and with columns "A" and "B" must
		// exist with some numeric data in them.
		void	run_AddPlot()
		{
			// Construct the Curve object specifying the x and y datasets to be used for
			// the dataplot
			Curve		cc("Data1_a", "Data1_b");
			
			// the graph layer needs to be initialized:
			GraphLayer	lay("Graph1");
			
			// Add the dataplot to the graph:
			int			nIndex = lay.AddPlot(cc, IDM_PLOT_SCATTER);
			
			// Display the index of the data plot just added:
			out_int("Dataplot index = ", nIndex);
		}
	*/
	int AddPlot(Curve& cData, int nPlotID = IDM_PLOT_UNKNOWN, int wCntrl = 0);
	
	/**
		The method retrieves the layer contents in form of a tree.
	*/
#if  _OC_VER > 0x0703
	/**#
	*/
	BOOL  GetLayerContents(TreeNode &tr, DWORD dwCtrl = 0);

	/**#
	*/
	int		AddPlots(TreeNode &tr, DWORD dwOptions = 0);
#endif	// _OC_VER


	

	/**#
		Add one data plot to layer by using the existing styleholder. The styleholder
		is usually used in templates to determine the type and other properties of
		a dataplot which uses the styleholder when being added to a layer.
	Parameters:
		cData = Data curve to add
		style = the style holder to use when creating the dataplot
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, a workseet with the name "Data1" and with columns "A" and "B" must
		// exist with some numeric data in them.
		void	run_AddPlot()
		{
			// Construct the Curve object specifying the x and y datasets to be used for
			// the dataplot
			Curve		cc("Data1_a", "Data1_b");
			
			// the graph layer needs to be initialized:
			GraphLayer	lay("Graph1");
			
			// Get the first style holder in the layer.
			StyleHolder style;
			style = lay.StyleHolders(0); 
			
			// Check that the style holder is valid:
			if (!style.IsValid())
			{
				out_str("No style holders found!");
				return;
			}
			
			// Add the dataplot to the graph:
			int			nIndex = lay.AddPlot(cc, style);
			
			// Display the index of the data plot just added:
			out_int("Dataplot index = ", nIndex);
		}
	*/
	int AddPlot(Curve& cData, StyleHolder& style);

	/**
		Add entire worksheet to layer.
	Parameters:
		wks = worksheet object containing all data.
		nPlotID = the internal Origin plot type id.
	Return:
		0-offset data plot index of the dataplot added or -1 for error.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, a workseet with the name "Data1" must exist with some numeric data.

		void run_AddPlot2()
		{
			GraphLayer gl("Graph1");
			Worksheet wks("Data1");
			gl.AddPlot(wks);
		}
	*/
	int AddPlot(Worksheet& wks, int nPlotID = IDM_PLOT_UNKNOWN);
	
	/**
		Adds an error bar plot to an existing dataplot.
	Parameters:
		cData = Data curve to add the error bar to
		dsErrBar = the dataset to be used as errorbar
		colErrBar = the Data curve to be used as error bar
	Returns:
		the index of the added error bar dataplot in layer, or -1 if failed.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, a workseet with the name "Data1" and with columns "A" and "B" and another column
		// (the third column) must exist with some numeric data in them. The graph
		// window should contain one dataplot (such as scatter plot) 
		// which uses the workseet's "B" column as Y-column and "A" column as X-column.
		// After the function executes, the third column of the worksheet will be used
		// for error bars (the graph may need to be refreshed to see the error bars).
		void	run_GraphLayer_AddErrBar()
		{
			GraphLayer		lay("Graph1");
			
			// a dataplot in the layer that uses Data1_b as Y and Data1_a as X
			Curve			cc("Data1_a", "Data1_b");
			
			// use the third column in the worksheet "Data1" for error bar:
			Column	colErrBar("Data1", 2);		
			int				nPlotIndex = lay.AddErrBar(cc, colErrBar);
			out_int("nPlotIndex = ", nPlotIndex);
		}
	*/
	int AddErrBar(Curve& cData, Column &colErrBar);
	/**
	Example:
		This example adds an error bar plot to an existing dataplot. 
		For this example to run, a graph window with the name "Graph1" must exist in the project.
		Also, a workseet with the name "Data1" and with columns "A" and "B" and "C" must exist 
		with some numeric data in them. The graph window should contain one dataplot which uses 
		the workseet's "B" column as Y-column and "A" column as X-column. (the graph may need 
		to be refreshed to see the error bars).
			
		void test_GraphLayer_AddErrBar()
		{
			GraphLayer glMyLayer("Graph1");
			Curve crvMyCurve("Data1_a","Data1_b");
			Dataset dsErrBar("Data1_c");
			int iPlotIndex = glMyLayer.AddErrBar(crvMyCurve, dsErrBar);
			if (iPlotIndex==-1)
				printf("ErrBar Plotting Error!");
			else
				printf("Errbar added successfully!");
		}
	*/
	int AddErrBar(Curve& cData, Dataset &dsErrBar);
	
	/**
		Adds an X error bar plot to an existing dataplot.
	Parameters:
		cData = Data curve to add the X error bar to
		dsXErrBar = the dataset to be used as X error bar
		colXErrBar = the Data curve to be used as X error bar
	Returns:
		the index of the the added X error bar dataplot in layer, or -1 if failed.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, a workseet with the name "Data1" and with columns "A" and "B" and another column
		// (the third column) must exist with some numeric data in them. The graph
		// window should contain one dataplot (such as scatter plot) 
		// which uses the workseet's "B" column as Y-column and "A" column as X-column.
		// After the function executes, the third column of the worksheet will be used
		// for X error bars (the graph may need to be refreshed to see the error bars).
		void	run_GraphLayer_AddXErrBar()
		{
			GraphLayer		lay("Graph1");
			
			// a dataplot in the layer that uses Data1_b as Y and Data1_a as X
			Curve			cc("Data1_a", "Data1_b");
			
			// use the third column in the worksheet "Data1" for X error bar:
			Column	colErrBar("Data1", 2);		
			int				nPlotIndex = lay.AddXErrBar(cc, colErrBar);
			out_int("nPlotIndex = ", nPlotIndex);
		}
	*/
	int AddXErrBar(Curve& cData, Column &colXErrBar);
	/**
	Example:
		For this example to run, a graph window with the name "Graph1" must exist in 
		the project. Also, a workseet with the name "Data1" and with columns "A" and 
		"B" and "C" must exist with some numeric data in them. The graph window should 
		contain one dataplot which uses the workseet's "B" column as Y-column and "A" 
		column as X-column. After the function executes, the graph may need to be 
		refreshed to see the error bars.
		
		void test_Graphlayer_AddXErrBar()
		{
			GraphLayer glMyLayer("Graph1");
			Curve crvMyCurve("Data1_a","Data1_b");
			Dataset dsXErrBar("Data1_c");
			int iPlotIndex = glMyLayer.AddXErrBar(crvMyCurve, dsXErrBar);
			if (iPlotIndex==-1)
				printf("X ErrBar Plotting Error!");
			else
				printf("X Errbar added successfully!");
		}
	*/
	int AddXErrBar(Curve& cData, Dataset &dsXErrBar);
	
	/**
		Adds a label plot to an existing dataplot.
	Parameters:
		cData = Data curve to add the label plot to
		dsLabel = the dataset to be used as labels
		colLabel = the Data curve to be used as label column 
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, a workseet with the name "Data1" and with columns "A" and "B" and another label column
		// (the third column) must exist. The graph
		// window should contain one dataplot (such as scatter plot) 
		// which uses the workseet's "B" column as Y-column and "A" column as X-column.
		// After the function executes, the third column of the worksheet will be used
		// for labels
		void	run_GraphLayer_AddLabelPlot()
		{
			//we need to first make sure the label column is really a Label column
			Worksheet wks("Data1");
			if(wks)
			{
				if(!wks.Columns(2).SetType(OKDATAOBJ_DESIGNATION_L))
					return;
			}
			else
				return;
			
			GraphLayer		lay("Graph1");
			
			// a dataplot in the layer that uses Data1_b as Y and Data1_a as X
			Curve			cc("Data1_a", "Data1_b");
			
			// first we plot col(B)
			int				nPlotIndex = lay.AddPlot(cc, IDM_PLOT_SCATTER);
			// use the third column in the worksheet "Data1" for labels:
			Column			colLabels("Data1", 2);		
			nPlotIndex = lay.AddLabelPlot(cc, colLabels);
			out_int("nPlotIndex = ", nPlotIndex);
			
			lay.Rescale();
		}
	*/
	int AddLabelPlot(Curve& cData, Column &colLabel);
	/**
	Example:
		For this example to run, a graph window with the name "Graph1" must exist 
		in the project. Also, a workseet with the name "Data1" and with columns "A" 
		and "B" and a label column "C" must exist. The graph window should contain 
		one dataplot which uses the workseet's "B" column as Y-column and "A" column 
		as X-column. 
		
		void test_Graphlayer_AddLabelPlot()
		{
			Worksheet wks("Data1");
			Dataset dsLables("Data1_c");
			if(wks)
			{
				if(!wks.Columns(2).SetType(OKDATAOBJ_DESIGNATION_L))
					return;
			}
			else
				return;
			
			GraphLayer glMyLayer("Graph1");
			Curve crvMyCurve("Data1_a", "Data1_b");
			
			int iPlotIndex = glMyLayer.AddLabelPlot(crvMyCurve, dsLables);
			if (iPlotIndex==-1)
				printf("Lables adding Error!");
			else
				printf("Lables added successfully!");
		}
	*/
	int AddLabelPlot(Curve& cData, Dataset &dsLabel);
	
#if  _OC_VER > 0x0703
	/**#
		Plot a Matrix into a graph layer
	Parameters:
		matObj = Matrix data object
		nPlotID = the internal Origin plot type id.
	Return:
		0-offset data plot index of the dataplot added or -1 for error
	Example:
		// important step to set the correct template to make 3D plots
		//IDM_PLOT_3D_MESH,		"mesh" // filled surface
		//IDM_PLOT_3D_MESH,		"cmap" // colormap surface
		//IDM_PLOT_3D_MESH,		"3Dbars"
		//IDM_PLOT_3D_MESH,		"wirefrm" // wireframes
		//IDM_PLOT_3D_MESH,		"wireface"  
		//IDM_PLOT_CONTOUR,		"contour"
		//IDM_PLOT_CONTOUR,		"contline"
		//IDM_PLOT_CONTOUR,		"contgray"
		//IDM_PLOT_MATRIX_IMAGE,"image"
		
		// 
		// this function plots the given matrix into a new graph
		// if nLevels is specified, will change the colormap to the specified number of levels
		// for plots not using colormap, must pass in 0
		bool plot_matrix(MatrixObject& mobj, LPCSTR lpcszTemplate, int nLevels = 0, int nPlotID = IDM_PLOT_CONTOUR)
		{
			GraphPage gp;
			string strTemp;
			gp.Create(lpcszTemplate, CREATE_HIDDEN); // use template, create as hidden to avoid unneeded drawing
			
			GraphLayer glay = gp.Layers();
			
			int nPlot = glay.AddPlot(mobj, nPlotID);
			glay.Rescale();
			if(nLevels) // to reset number of levels in color map
			{
				DataPlot dp = glay.DataPlots(0); 
				Tree tr; 
				tr = dp.Surface; // get the internal Surface branch of Format 
				tr.ColorMap.Details.Remove(); // must del actual levels so Count can be applied
				tr.ColorMap.Count.nVal = nLevels;
				dp.Surface = tr; // update surface branch from tree 
				strTemp = " Colormap is modified, levels = " + nLevels;
			}
			gp.Label = "Plot created using template: " + (string)lpcszTemplate + strTemp;
			gp.TitleShow = WIN_TITLE_SHOW_BOTH;
			gp.SetShow();// show it when all it ready
			return true;
		}
		
		void test_plot_from_matrix()
		{
			MatrixLayer mlayer = Project.ActiveLayer();
			MatrixObject mobj = mlayer.MatrixObjects();
			
			plot_matrix(mobj, "contour", 5); // contour filled color
			plot_matrix(mobj, "contour", 23);// contour lines only
			plot_matrix(mobj, "image", 0, IDM_PLOT_MATRIX_IMAGE);// grayscale contour map
			plot_matrix(mobj, "cmap", 18, IDM_PLOT_3D_MESH);// grayscale contour map
			plot_matrix(mobj, "wirefrm", 0, IDM_PLOT_3D_MESH);// grayscale contour map
		}

	*/
	int	AddPlot(MatrixObject& matObj, int nPlotID = IDM_PLOT_MATRIX_IMAGE);
#endif // _OC_VER

	/**
		It rescales the graphic layer axes to show all dataplot points that may be outside the current axis ranges.
	Return:
		TRUE if OK, otherwise FALSE.
	Parameters:
		None.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Then manually add some data to the graph and use the Axes dialog to rescale
		// the axes limits so that not all the data is visible. After the function executes,
		// all the data will be visible.
		void	run_GraphLayer_Rescale()
		{
			// the graph layer needs to be initialized:
			GraphLayer		lay("Graph1");
			
			// Rescale the layer axes if necessary:
			lay.Rescale();
		}
	*/
	BOOL	Rescale();

	
	/**
		Gets the layer's coordinate system type.
	Parameters:
		None.
	Returns:
		a value from the following enumeration indicating the type:
		enum {
			FRAME_COOR_CART = 0,			// XY Cartesian coordinate system
			FRAME_COOR_POLAR,				// polar		
			FRAME_COOR_TERNARY,				// ternary
			FRAME_COOR_SMITH_CHART			// Smith chart
		};
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_GetCoordinateType()
		{
			// the graph layer needs to be initialized:
			GraphLayer		lay("Graph1");
			
			if (lay.GetCoordinateType() == FRAME_COOR_CART)
				out_str("The coordinate system is Cartesian");
			else
				out_str("The coordinate system is not Cartesian");
		}
	*/
	UINT GetCoordinateType();

	
	/**
		Groups dataplots in layer.
	Parameters:
		nIndexStart = the index of the first dataplot to be grouped
		nIndexEnd = the index of the last dataplot to be grouped. All the dataplots from nIndexStart 
					to nIndexEnd are going to be put into the same group (if nIndexEnd is negative,
					all the dataplots from nIndexStart to the end are going to be grouped) 
	Returns:
		TRUE for success, otherwise FALSE.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least two data plots and they are not grouped.
		// After the function executes, all the dataplots in layer will be grouped in one group.
		void	run_GroupPlots()
		{
			// the graph layer needs to be initialized:
			GraphLayer		lay("Graph1");
			
			// Group all the plots in the layer into one group.
			lay.GroupPlots(0);
		}
	*/
	BOOL	GroupPlots(int nIndexStart, int nIndexEnd = -1);
	
	
	/**
		Ungroups dataplots.
	Parameters:
		nIndex = the index of any dataplot inside the group. If -1, use the first group
	Returns:
		TRUE for success, otherwise FALSE.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least two data plots and that they are grouped.
		// After the function executes, the dataplots in layer will be ungrouped.
		void	run_UngroupPlots()
		{
			// the graph layer needs to be initialized:
			GraphLayer		lay("Graph1");
			
			// Group all the plots in the layer into one group.
			lay.UngroupPlots();
		}
	*/
	BOOL	UngroupPlots(int nIndex = -1);
	
	/**
		The "X" property - the Scale object representing the X-axis in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_X()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			
			// Intialize a Scale object from the X-axis scale of the GraphLayer
			// (gl.X is the X-axis scale for the layer attached to "gl").
			Scale			s(gl.X);
			
			// Display the "From" property (the left limit of the X-axis):
			out_double("From = ", s.From);
		}
	*/
	Scale X;
	
	/**
		The "Y" property - the Scale object representing the Y-axis in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_Scale_Y()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			
			// Intialize a Scale object from the Y-axis scale of the GraphLayer
			// (gl.Y is the Y-axis scale for the layer attached to "gl").
			Scale			s(gl.Y);
			
			// Display the "From" property (the left limit of the Y-axis):
			out_double("From = ", s.From);
		}
	*/
	Scale Y;
	
	/**
		Gets an arbitrary DataPlot in layer.
	Parameters:
		index = 0-offset index of the data plot in layer.
	Returns:
		the index'th DataPlot.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot and that there are at least ten
		// data points in the plot.
		// After the function executes, the data plot will show
		// only the data points from 5 to 10.
		void	run_DataPlots()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
		
			// Change the display range of the data plot to
			// show only points from 5 to 10:
			dp.SetRange(5, 10);
		}
	*/
	DataPlot					DataPlots(int index);
	
	/**
		Gets an arbitrary DataPlot in layer.
	Parameters:
		lpcszName = the name of the dataset that the desired DataPlot should contain.
	Returns:
		the DataPlot.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot using
		// the dataset "Data1_b" and that there are at least ten
		// data points in the plot.
		// After the function executes, the data plot will show
		// only the data points from 5 to 10.
		void	run_DataPlots()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the data plot in the layer which contains dataset "Data1_b":
			DataPlot		dp = gl.DataPlots("Data1_b");
		
			// Change the display range of the data plot to
			// show only points from 5 to 10:
			dp.SetRange(5, 10);
		}
	*/
	DataPlot					DataPlots(LPCSTR lpcszName);
	
	/**
		Removes one DataPlot from layer. The object is referred to by index.
	Parameters:
		index = the DataPlot index
	Return:
		TRUE for success, otherwise FALSE
	*/
	BOOL						RemovePlot(int nIndex);
	
	/**
		Removes one DataPlot from layer. The object is referred to by refercence.
	Parameters:
		dp = the DataPlot reference
	Return:
		TRUE for success, otherwise FALSE
	*/
	BOOL						RemovePlot(DataPlot& dp);
	
	/**
		The Collection property of all DataPlots in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot there are at least ten
		// data points in every the plot.
		// After the function executes, all the data plots will show
		// only the data points from 5 to 10.
		void	run_DataPlots()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			Collection<DataPlot>	dpColl;
			
			// Get the collection of all dataplots in the layer:
			dpColl = gl.DataPlots;
		
			// Loop over all the dataplots in the collection:	
			foreach (DataPlot dp in dpColl)
			{
				// Change the display range of the data plot to
				// show only points from 5 to 10:
				dp.SetRange(5, 10);
			}
		}
	*/
 	Collection<DataPlot>	 	DataPlots;

	/**#
		Gets an arbitrary StyleHolder in layer.
	Parameters:
		index = 0-offset index of the StyleHolder in layer.
	Returns:
		the index'th StyleHolder.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_StyleHolders()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			StyleHolder		sh;
			
			// Get the first style holder:
			sh = gl.StyleHolders(0);
			
			// Display its name:
			out_str(sh.GetName());
		}
	*/
	StyleHolder	StyleHolders(int index);
	
	/**#
		Gets an arbitrary StyleHolder in layer.
	Parameters:
		lpcszName = the name of the style holder in layer (the name of a style holder
					starts with an underscore followed by a number
					indicating the plot type).
	Returns:
		the StyleHolder.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that you use the line-symbol plot template or graph, because
		// the function tries to access the style holder with the name "_202" which
		// corresponds to the line-symbol plot type.
		void	run_StyleHolders()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			StyleHolder		sh;
			
			// Get the style holder "_202":
			sh = gl.StyleHolders("_202");
			
			// Display its name:
			out_str(sh.GetName());
		}
	*/
	StyleHolder	StyleHolders(LPCSTR lpcszName);
	
	/**#
		The Collection property of all StyleHolders in the layer.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_StyleHolders()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			Collection<StyleHolder>	shColl;
			
			// Get the collection of all style holders in the layer:
			shColl = gl.StyleHolders;
		
			// Loop over all the style holders in the collection:	
			foreach (StyleHolder style in shColl)
			{
				// Display the name (the name of a style holder
				// starts with an underscore followed by a number
				// indicating the plot type):
				out_str(style.GetName());
			}
		}
	*/
	Collection<StyleHolder> StyleHolders;

#if _OC_VER > 0x0703
	/**#
	Example:
		GraphLayer  gl("Graph1");
		gl.SetDataMarkers(1,3);	
		
		int i1,i2;
		gl.GetDataMarkers(i1, i2);	
	*/
	BOOL	SetDataMarkers(int i1, int i2);
	
	/**#
	SeeAlso:
		GraphLayer::SetDataMarkers
	*/
	BOOL	GetDataMarkers(int& i1, int& i2);

	/**#
	Example:
		GraphLayer  gl("Graph1");
		Axis x = gl.XAxis;
	*/
	Axis	XAxis;

	/**#
	Example:
		GraphLayer  gl("Graph1");
		Axis y = gl.YAxis;
	*/
	Axis	YAxis;

	/**#
	Example:
		GraphLayer  gl("Graph1");
		Axis z = gl.ZAxis;
	*/
	Axis	ZAxis;

	/**#
	Example:
		GraphLayer  gl("Graph1");	//Least two plots on the graph layer
		gl.GroupPlots(0,1);			//make a group for the two plots
		
		GroupPlot gPlot;
		gPlot = gl.Groups(0);
	*/
	GroupPlot	Groups(int nIndex);
#endif //	_OC_VER > 0x0703	

};

/** >Internal Origin Objects
		The GraphObject class provides methods and properties common to all Origin graph
		objects. Origin graph objects include such things as text annotations, graphic
		annotations (e.g. rectangles, arrows, line objects, etc.), data plot style holders,
		and region of interest objects. Origin graph objects are generally contained by
		layers on an Origin page thus the Origin C GraphLayer class contains a collection
		of GraphObjects. An Origin C GraphObject is a wrapper object that is a reference to
		an internal Origin graph object. Origin C wrapper objects do not actually exist
		in Origin and merely refer to the internal Origin object. Consequently, multiple
		Origin C wrapper objects can refer to the same internal Origin object. The GraphObject
		class is derived from the OriginObject class from which it inherits methods and
		properties.
	Example:
		// Assumes Graph1 window is active in Origin
		GraphLayer gl = Project.ActiveLayer();  // Get active layer in project file
		if(gl)                                  // If valid graph layer...
		{
			GraphObject	go;                     // Declare GraphObject
			go = gl.GraphObjects("YL"); 		// Attach go to "YL" which is the the left Y-axis title
			out_str(go.Text);
		}
*/
class GraphObject : public OriginObject
{
public:
	/**
		Default constructor which creates a GraphObject which is not
		attached to an internal graphic object. It should be initialized using
		"GraphObjects" method of the Layer class because all the GraphObject's are conatined in
		layers.
	Example:
		For this example to run, a graph window with the name "Graph1" must exist in the project.
		void test_GraphObject()
		{
			GraphLayer glMyLayer("Graph1");
			GraphObject goMyObj = glMyLayer.GraphObjects("Text");
			out_str(goMyObj.Text);
		}
	*/
	GraphObject();
	/**
	Copy Constructor: The new GrpahObject object created is attached to the internal graphic object
	goOriginal was attached to. If goOriginal was invalid (not attached) so will be the new object.
	Example:
		For this example to run, a graph window with the name "Graph1" must exist in the project.
		void test_GraphObject()
		{
			GraphLayer glMyLayer("Graph1");
			GraphObject goOriginal;
			goOriginal = glMyLayer.GraphObjects("Text");
			
			GraphObject goNew(goOriginal);
			out_str(goNew.Text);
		}
	*/
	GraphObject(GraphObject &goOriginal);
	
	/**
		The X coordinate of the center of the object.
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_X()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the X coordinate:
			out_double("X = ", grobj.X);
		}
	*/
	double	X;
	
	/**
		The Y coordinate of the center of the object.
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_Y()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the Y coordinate:
			out_double("Y = ", grobj.Y);
		}
	*/
	double	Y;
	
	/**
		Width of the object in axes units 
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_DX()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the width:
			out_double("DX = ", grobj.DX);
		}
	*/
	double	DX;
	
	/**
		Height of the object in axes units. 
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_DY()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the height:
			out_double("DY = ", grobj.DY);
		}
	*/
	double	DY;
	
	/**
		Width in layer coordinate units of the object
		Read access.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_Width()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the width:
			out_int("Width = ", grobj.Width);
		}
	*/
	int		Width;
	
	/**
		Height in layer coordinate units of the object
		Read access.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_Height()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the width:
			out_int("Height = ", grobj.Height);
		}
	*/
	int		Height;
	
	/**
		The left coordinate of the object in physical coordinates 
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_Left()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the Left property:
			out_int("Left = ", grobj.Left);
		}
	*/
	int		Left;
	
	/**
		The top coordinate of the object in physical coordinates 
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Also, the active layer in the Graph should have a graphic object (such as rectangle) with
		// the name "Rect".
		void	run_GraphObject_Top()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object:
			grobj = gl.GraphObjects("Rect");
		
			// Display the Left property:
			out_int("Top = ", grobj.Top);
		}
	*/
	int		Top;
	
	/**
		The attach mode of the object. Possible values are
			0	- to Layer
			1	- to Page
			2	- to axes
		Read and Set access
	*/ 
	int		Attach;
	
	/**
		The text of a text label.
		Read and Set access
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_GraphObject_Text()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Get the graphic object "YL", which is the
			// the left Y-axis title:
			grobj = gl.GraphObjects("YL");
			
			// Display its text:
			out_str(grobj.Text);
		}
	*/
	string	Text;	
	
#if  _OC_VER < 0x0750
	/**
		The name of the graphic object. The name of each object can be seen by
		opening the "Label Control" dialog box (using the context menu on the object).
	Parameters:
		None.
	Returns:
		the name of the GraphObject. 
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_GraphObject_GetName()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Loop over all the graphic objects in the layer and for each of them
			// display its name:
			foreach (grobj in gl.GraphObjects)
			{
				out_str(grobj.GetName());
			}
		}
	*/
	string	GetName();
#endif //_OC_VER < 0x0750
	
	/**#
		It extracts into a file an enhanced metafile or device independent bitmap depending on the type of GraphObj.
		(which could have been generated, e.g,, by pasting a picture) 
	Parameters:
		lpcszPathName = the path to the file to be created which will contain the enhanced metafile
	Return:
		TRUE for success
	Example:
		GraphLayer		lay("Graph1");
		GraphObject		grEMF;
		
		grEMF = lay.GraphObjects("EMF");
		
		BOOL			bRet = grEMF.ExtractImageToFile("c:\\myfiles\\myemf.emf");
	*/
	BOOL	ExtractImageToFile(LPCSTR lpcszPathName);
		
	/**#
		Get number of nodes of a graph object in graph layer
	Parameters:
		
	Return:
		Number of Nodes
	Example:	
		GraphLayer gl("Graph1");
		foreach(GraphObject gr in gl.GraphObjects)
		{
			fpoint ptNode;
			int iNodes = gr.GetNumOfNodes();
			if (iNodes > 1 )
			{
				printf("Name = %s\n", gr.GetName());
				printf("Number of nodes = %d\n", iNodes);
				for (int ii = 0; ii < iNodes; ii ++)
				{
					gr.GetNode(ii,ptNode);
					printf("Point %d: X= %f Y = %f\n", ii, ptNode.x, ptNode.y);
				}
			}
		}		
	*/
	int		GetNumOfNodes();

#ifdef _POST_ORIGIN7_
	/**#
		Get each node's coordinate by index
	Parameters:
		nIndex = the node index
		ptNode = return point value of the coordinate
	Return:
		TRUE for success
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project
		// and have a rectangle object in it.
		void test1()
		{
			GraphLayer gl("Graph1");
			foreach(GraphObject gr in gl.GraphObjects)
			{
				fpoint ptNode;
				int iNodes = gr.GetNumOfNodes();
				if (iNodes > 1 )
				{
					printf("Name = %s\n", gr.GetName());
					printf("Number of nodes = %d\n", iNodes);
					for (int ii = 0; ii < iNodes; ii ++)
					{
						gr.GetNode(ii, ptNode);
						printf("Point %d: X= %f Y = %f\n", ii, ptNode.x, ptNode.y);
					}
				}
			}	
		}
	*/
	BOOL	GetNode(int nIndex, fpoint& ptNode);
#endif //_POST_ORIGIN7_

	/**
		The type of the graphic object as a string.
	Paramaters:
		None.
	Returns:
		the string representing the type of the graphic object.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		void	run_GraphObject_GetObjectType()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			GraphObject		grobj;
			
			// Loop over all the graphic objects in the layer and for each of them
			// display its type as string:
			foreach (grobj in gl.GraphObjects)
			{
				out_str(grobj.GetObjectType());
			}
		}
	*/
	string  GetObjectType();
	
	/**#
	*/
	int	GetRegionPoints(vectorbase& vValue);
	
	/**#
	*/
	int	GetRegionPoints(vectorbase& vValue, vector<int>& vI, vector<int>& vJ);
};

/** >Internal Origin Objects
		The StyleHolder class provides methods and properties common to all data plot
		style holders. An internal Origin data plot style holder is an object used to
		store plot type information for a data plot contained in an Origin graph layer.
		An Origin C StyleHolder is a wrapper object that is a reference to 	an internal
		Origin StyleHolder object. Origin C wrapper objects do not actually exist in
		Origin and merely refer to the internal Origin object. Consequently, multiple
		Origin C wrapper objects can refer to the same internal Origin object. The
		StyleHolder class is derived from the GraphObject and OriginObject classes from
		which it inherits methods and properties.
	Example:
		// Assumes Graph1 is active window in Origin
		GraphLayer gl = Project.ActiveLayer();        // Get active Origin layer
		if(gl)                                        // If layer is valid...
		{
			StyleHolder sh = gl.StyleHolders(0);      // Get first style holder in layer 
			out_int("Plot type is ", sh.GetPlotId()); // Get plot id 
		}
*/
class StyleHolder : public GraphObject
{
public:
	
	/**
	Example:
		// Assumes Graph1 is active window in Origin
		GraphLayer gl = Project.ActiveLayer();        // Get active Origin layer
		if(gl)                                        // If layer is valid...
		{
			StyleHolder sh = gl.StyleHolders(0);      // Get first style holder in layer 
			out_int("Plot type is ", sh.GetPlotId()); // Get plot id 
		}
	*/
	StyleHolder();
	
	/**
	Copy Constructor: The new StyleHolder object created is attached to the internal graphic object
	shOriginal was attached to. If shOriginal was invalid (not attached) so will be the new object.
	Example:
		// Assumes Graph1 exist in Origin
		void test_StyleHolder()
		{
			GraphLayer glMyLayer("Graph1");
			StyleHolder shOriginal;
			shOriginal = glMyLayer.StyleHolders(0);
			StyleHolder shNew(shOriginal);
			
			if (shNew.IsValid())
			{
				out_str(shNew.GetName());
			}
		}
	*/
	StyleHolder(StyleHolder &shOriginal);
	
	/**
	*/	
	string			GetDescription();

	/**
		get the Origin internal plot id.
	Parameter:
		bBaseID = TRUE if we only care about the graph category, such than line,scatter and line symbol plots will all return IDM_PLOT_LINESYMB	
	Example:
		GraphLayer gl = Application.ActiveLayer();
		if(gl)
		{
			StyleHolder sh = gl.StyleHolders(0);
			out_int("plot type is ", sh.GetPlotId());
		}
	*/
	int				GetPlotId(BOOL bBaseID = FALSE);

	/**
	*/
	string			GetPlotDesignations(DWORD *pdwFlags = NULL);
};


/** >Internal Origin Objects
	The DataObjectBase class provides methods and properties common to all data objects. The
	Origin C DataObjectBase class is an abstract base class used for polymorphic handling
	of DataObject and DataPlot related class types. Consequently, Origin C objects of type
	DataObjectBase can not be constructed. Derived classes such as DataObject, Column,
	MatrixObject, and DataPlot inherit DataObjectBase class methods and should be used instead.  
*/
class DataObjectBase : public OriginObject
{
	
protected:
	/**#
		Default constructor which constructs an uninitialized (unattached) DataObject.
		The object cannot be used until it is initialized.
	Parameters:
		None.
	Example:
		See examples for the methods of this class.
	*/
	DataObjectBase();

public:
	/**
		It sets the data (display) range for the object. Not that this range can be smaller than the total range
		of data of the underlying dataset if not all data values are shown. For example, in case of a data plot
		in a graph, some data points may not have been plotted. Predefined values for i1 and i2 are
		SETRANGE_UNCHANGED	- leaves corresponding value unchanged
		SETRANGE_FULL		- use full dataset range. Use this value to turn Autorange on for DataPlots.
	Parameters:
		i1 = the lower bound of the desired range, zero offset.
		i2 = the upper bound of the desired range, zero offset.
		bRepaint = TRUE to repaint the object after changing the range, FALSE not to repaint.
	Returns:
		TRUE for success, FALSE for failure.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot and that there are at least ten
		// data points in the plot.
		// After the function executes, the data plot will show
		// only the data points from 5 to 10.
		void	run_SetRange()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
		
			// Change the display range of the data plot to
			// show only points from 5 to 10: (1 offset)
			dp.SetRange(4, 9);
		}
	*/
	BOOL	SetRange(int i1 = SETRANGE_FULL, int i2 = SETRANGE_FULL, BOOL bRepaint=TRUE);// i2=-1 to indicate the full range at the end
	
	/**
		it retrieves the data (display) range for the object
	Parameter:
		i1 = lower boundary of the range passed by reference to receive the result
		i2 = upper boundary of the range passed by reference to receive the result
	Returns:
		TRUE for success, FALSE for failure.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot.
		void	run_GetRange()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
			int				i1, i2;
		
			dp.GetRange(i1, i2);
			
			printf("Data plot range: i1 = %d   i2 = %d\n", i1, i2);
		}
	*/
	BOOL	GetRange(int &i1, int &i2);
	
	
	/**
		Property: the lower data range index for the object
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot.
		void	run_DataObject_i1()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
			int				i1 = dp.i1;
		
			printf("Data plot range: i1 = %d\n", i1);
		}
	*/
	int		i1;
	/**
		Property: the upper data range index for the object
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot.
		void	run_DataObject_i2()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
			int				i2 = dp.i2;
		
			printf("Data plot range: i2 = %d\n", i2);
		}
	*/
	int		i2;

	/**
		A DataObjectBase object is always associated with a dataset inside Origin. Since Dataset is a template
		class, we use the name as the common mean of connecting to a Dataset
	Example:
		// list all the dataset names in the current graph layers
		GraphLayer gl = Project.ActiveLayer();
		if(!gl)
			return;
		int index = 1;
		foreach(DataPlot dp in gl.DataPlots)
		{
			printf("%3.0d:%s\n", index++, dp.GetDatasetName());
		}
	*/
	string	GetDatasetName();

	/**
	Remarks:
		Get the index of object in layer, 0 offset.  
	Parameters:
		None
	Return:
		index of object in layer or -1 if invalid.
	
	Example:
		Worksheet wks("data1");
		int nCol = wks.Columns("B").GetIndex();
		string strNewColName;
		if(nCol >=0)
			wks.InsertCol(nCol+1, "Result", strNewColName);
	*/
	int		GetIndex();
	
#if _OC_VER > 0x0703	
	/**
		Retrieves categorical map from DataObject
	Example:
		Column col("Data1", 0);
		if( col )
		{
			CategoricalMap cm = col.GetCategrocalMap();
			if( cm )
			{
				int nCatValue = cm.LookUp("key1");
			}
		}
	*/
	CategoricalMap	GetCategoricalMap();
#endif // #if _OC_VER > 0x0703	
};

/** >Internal Origin Objects
	The DataObject class is derived from the DataObjectBase class and is the base class of
	worksheet columns and matrix objects. Origin data objects are contained by layers on 
	an Origin page. For example, columns (data objects) are contained by a worksheet (layer)
	and are on a worksheet window (page). 
*/
class DataObject : public DataObjectBase
{
public:
	
	/**
			Default constructor for DataObject class.
	*/
	DataObject(); // Default constructor for DataObject class.
};


/** >Internal Origin Objects
		The DataPlot class provides methods and properties common to all Origin data plots.
		An internal Origin data plot object is used to store the plot characteristics of an
		Origin data plot. It is contained in a graph layer on a graph page. An Origin C
		DataPlot object is a wrapper object that is a reference to an internal Origin data
		plot object. Origin C wrapper objects do not actually exist in Origin and merely refer
		to the internal Origin object. Consequently, multiple Origin C wrapper objects can refer
		to the same internal Origin object. The DataPlot class is derived from the DataObjectBase
		and OriginObject classes from which it inherits methods and properties.
	Example:
		// Assumes Graph1 window with one or more data plots is active in Origin
		GraphLayer	gl = Project.ActiveLayer(); // Get active layer in project file
		if(gl)                                  // If valid graph layer...
		{
			DataPlot dp = gl.DataPlots(0);      // Get first data plot in graph layer
			dp.SetColor(2);                     // Change color to green
		}
*/
class DataPlot : public DataObjectBase
{
public:
	/**
		Default constructor which constructs an uninitialized (unattached) DataPlot.
		The object cannot be used until it is initialized.
	Parameters:
		None.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot.
		// The example shows how to initialize an uninitialized DataPlot object.
		void	run_DataPlot_Constructor()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Use default constructor to create an uninitialized DataPlot.
			DataPlot		dp;
			
			// Now initialize it to be the first DataPlot in the layer:
			dp = gl.DataPlots(0);
		
			// Displaye a property:
			printf("Data plot range: i2 = %d\n", dp.i2);
		}
	*/
	DataPlot();
	
	/**
		Copy constructor which constructs a DataPlot object from another
		DataPlot object.
	Parameters:
		dp = the source DataPlot object.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one data plot.
		void	run_DataPlot_Constructor()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Use copy constructor to create a DataPlot from an existing DataPlot object.
			DataPlot		dp(gl.DataPlots(0));
			
			// Displaye a property:
			printf("Data plot range: i2 = %d\n", dp.i2);
		}
	*/
	DataPlot(DataPlot &dp);
	
	
	
	/**
		set the color of the dataplot (line, symbol).
	Parameter:
		nColor = the color index in the Origin palette (typically it is 0 - black, 1 - red, etc.)
		bRepaint = TRUE to cause graph repaint, FALSE not to repaint.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one scatter data plot.
		void	run_DataPlot_SetColor()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
		
			// Set the color to red and repaint:
			dp.SetColor(1, TRUE);
		}
	*/
	BOOL	SetColor(int nColor, BOOL bRepaint = FALSE);
	
	
	
	/**
		set the color of the dataplot (line, symbol) by the intensities of red, green, and blue colors
	Parameter:
		byRed =	the intensity of red
		byGreen = the intensity of green
		byBlue = the intensity of blue
		bRepaint = TRUE to cause graph repaint, FALSE not to repaint.
	Example:
		// For this example to run, a graph window with the name "Graph1" must exist in the project.
		// Make sure that the graph has at least one scatter data plot.
		void	run_DataPlot_SetColor()
		{
			// Create and attach a graph layer from "Graph1":
			GraphLayer		gl("Graph1");
			
			// Get the first data plot in the layer:
			DataPlot		dp = gl.DataPlots(0);
		
			// Set the color to green and repaint:
			dp.SetColorRGB(0x00, 0xFF, 0x00, TRUE);
		}
	*/
	BOOL	SetColorRGB(BYTE byRed, BYTE byGreen, BYTE byBlue, BOOL bRepaint = FALSE);

	/**
		Sets the colormap properties of the dataplot (applies to dataplot with colormaps, like contour plot)
	Parameter:
		trColormap[in] : A TreeNode object with the Tree structure similar to the One obtained from GetColormap
	Return:
		The BOOL returned is TRUE for success and FALSE for failure.
	Remarks:
	The input TreeNode does not have to contain child nodes for all properties of colormap, only for those intended to be set, 
	the properties for which nodes are no found will remain un-altered unless the altered properties forces some others to be altered. 
	Example:
		// For this example to run, you need a contour graph
		void test_Set_Colormap(string str, int nCount = 0)
		{
			GraphPage gp(str);
   			GraphLayer glay = gp.Layers();
   			DataPlot dp = glay.DataPlots(0); 
   			vector vZs = {1,5.5,10};
			BOOL bRet;
			if(nCount)
			{
				Tree tr;
				tr.ColorMap.Count.nVal = nCount;
				bRet = dp.SetColormap(tr);
				
			}
			else
			{
				bRet = dp.SetColormap(vZs, false);
			}

		}

	*/
	BOOL	SetColormap(TreeNode& trColormap);
	/**
		Set the Z levels for the colormap inside the data plot (applies to dataplot with colormaps, like contour plot)
	Parameter:
		vz[in]			: A Vector of Doubles, specifying the the z-level values to be set
		bLogScale[in]	: bLogScale, TRUE will set the colormap levels scale to Logarithmic, FALSE for linear
	return:
		The BOOL returned is TRUE for success and FALSE for failure.  Returns FALSE if has negative in vz.
	Remarks:
		The 1st value in vz is the minimum Z value and the last value in vz the maximum, so vz must at least contain 2 values.
	Example:
		// For this example to run, you need a contour graph
		void test_Set_Colormap(string str, int nCount = 0)
		{
			GraphPage gp(str);
   			GraphLayer glay = gp.Layers();
   			DataPlot dp = glay.DataPlots(0); 
   			vector vZs = {1,5.5,10};
			BOOL bRet;
			if(nCount)
			{
				Tree tr;
				tr.ColorMap.Count.nVal = nCount;
				bRet = dp.SetColormap(tr);
				
			}
			else
			{
				bRet = dp.SetColormap(vZs, false);
			}

		}
	*/
	BOOL	SetColormap(const vector<double>& vz, BOOL bLogScale=FALSE);
	/**
		Gets the colormap properties of the dataplot (applies to dataplot with colormaps, like contour plot)
	Parameter:
		trColormap[Out] : A TreeNode object with the Tree structure of GetColormap
	return:
		The BOOL returned is TRUE for success and FALSE for failure.
	Remarks:
	Example:
		// For this example to run, you need a contour graph
		void test_Get_Colormap(string str)
		{
			GraphPage gp(str);
   			GraphLayer glay = gp.Layers();
   			DataPlot dp = glay.DataPlots(0); 
   			vector vZs;
   			BOOL bLogScale = FALSE;;
			BOOL bRet = dp.GetColormap(vZs, bLogScale);
			Tree tr;
			BOOL bRetT = dp.GetColormap(tr);
		}
		//The following function shows how to Show/hide contour lines 
		//Currently there is no support to  show/hide for the line above
		BOOL HideLines(int nShow)
		{
			
			GraphPage gp("graph1");
		   if(!gp.IsValid() )
			   return FALSE;
		   // Get graph layer from collection in graph page
		   GraphLayer gl = gp.Layers();
		   if(!gl.IsValid() )
			   return FALSE;
		   // Get data plot from collection in graph layer
		   DataPlot dp = gl.DataPlots(0); 
		   if( !dp.IsValid() )
			   return FALSE;
   
		   TreeNode tr;
		   dp.GetColormap(tr);
		   vector<int> vLShow;
		   vLShow = tr.Details.ShowLines.nVals;
		   vLShow = nShow;
		   tr.Details.Remove();
		   tr.Details.ShowLines.nVals  =vLShow;
			dp.SetColormap(tr);
			return TRUE;

		}

	*/
	BOOL	GetColormap(TreeNode& trColormap);
	/**
		Gets the z-levels for of colormap of the dataplot (applies to dataplot with colormaps, like contour plot)
	Parameter:
		vz[Out]			: A Vector of Doubles giving the Z-level values
		bLogScale[Out]	: bLogScale, TRUE means Logarithmic, FALSE for linear
	return:
		The BOOL returned is TRUE for success and FALSE for failure.
	Remarks:
	Example:
		// For this example to run, you need a contour graph
		void test_Get_Colormap(string str)
		{
			GraphPage gp(str);
   			GraphLayer glay = gp.Layers();
   			DataPlot dp = glay.DataPlots(0); 
   			vector vZs;
   			BOOL bLogScale = FALSE;;
			BOOL bRet = dp.GetColormap(vZs, bLogScale);
			Tree tr;
			BOOL bRetT = dp.GetColormap(tr);
		}
	*/
	BOOL	GetColormap(vector<double>& vz, BOOL& bLogScale);

	/**
	  set data plot to be the ActiveCurve
	  Parameters:
	  	  nMarkerLeft = left marker if >= 0, 
      	  nMarketRight = right marker if left was on and > left
	  Return:
	      previous ActiveCurve index in layer
	  Example:
	      GraphLayer	gl("Graph1");
		  DataPlot dp;
		  dp = gl.DataPlots(0);	
		  dp.SetActive(2,5);
	*/
	int SetActive(int nMarkerLeft = -1, int nMarkerRight = -1);
};

#ifdef _POST_ORIGIN7_

/** >Internal Origin Objects
	The AxisObject class is derived from the OriginObject class and provides access to Origin axis ticks, grids and labels.
	Origin axis objects are contained by axes on an Origin page.
*/
class	AxisObject		:	public	OriginObject
{
public:
	AxisObject();

	AxisObject(const AxisObject& ao);
};

/** >Internal Origin Objects
	The Axis class is derived from the OriginObject class and provides access to Origin axes.
	Origin axes are contained by layers on an Origin page.
*/
class	Axis		:	public OriginObject
{
public:
	Axis();

	Axis(const Axis& axis);

	AxisObject		AxisObjects(int nIndex);
};

/** >Internal Origin Objects
	The GroupPlot class is derived from the OriginObject class and provides access to Origin group plots.
	Origin GroupPlot objects are contained by layers on an Origin page.
*/
class	GroupPlot	:	public	OriginObject
{
public:
	GroupPlot();

	GroupPlot(const GroupPlot& gp);
};

/** >Internal Origin Objects
		The point class provides methods and properties common to data points located
		in a two dimensional or planar space whose (x,y) coordinates are integers.
	Example:
		point pt(4, 7);
		printf("x = %d   y = %d\n", pt.x, pt.y);
*/
class point
{
public:

	/**
		constructor which initializes a point object from x and y coordinates
	Paramaters:
		x = (optional) x coordinate
		y = (optional) y coordinate
	Example:
		void	run_point_Constructor()
		{
			point	pt(4, 7);
			
			printf("x = %d   y = %d\n", pt.x, pt.y);
		}
	*/
	point(int x = 0, int y = 0);
	
	/**
		Copy constructor which initializes the point object from another point object
	Parameters:
		pp = source point object
	Example:
		void	run_point_Constructor()
		{
			point	pt(4, 7);
			
			// Initialize pt2 from pt:
			point	pt2(pt);
			
			// Display pt2:
			printf("x = %d   y = %d\n", pt2.x, pt2.y);
		}
	*/
	point(point pp);
	
	/**
		x coordinate of the point
	Example:
		void	run_point_x_y()
		{
			point	pt(2, 3);
			
			printf("x = %d   y = %d\n", pt.x, pt.y);
		}
	*/
	int 	x;
	
	/**
		y coordinate of the point
	Example:
		void	run_point_x_y()
		{
			point	pt(2, 3);
			
			printf("x = %d   y = %d\n", pt.x, pt.y);
		}
	*/
	int 	y;
};

/** >Internal Origin Objects
		The fpoint class provides methods and properties common to data points located
		in a two dimensional or planar space whose (x,y) coordinates are doubles.
	Example:
		point fpt(1.2, 7.6);
		printf("x = %g   y = %g\n", fpt.x, fpt.y);
*/
class fpoint
{
public:

	/**
		constructor which initializes the fpoint object from x and y coordinates
	Paramaters:
		x = (optional) x coordinate
		y = (optional) y coordinate
	Example:
		void	run_fpoint_Constructor()
		{
			fpoint	pt(1.2, 7.6);
			
			printf("x = %f   y = %f\n", pt.x, pt.y);
		}
	*/
	fpoint(double x = 0, double y = 0);
	
	/**
		Copy constructor which initializes the fpoint object from another fpoint object
	Parameters:
		pp = source fpoint object
	Example:
		void	run_fpoint_Constructor()
		{
			fpoint	pt(4.4, 7.7);
			
			// Initialize pt2 from pt:
			fpoint	pt2(pt);
			
			// Display pt2:
			printf("x = %f   y = %f\n", pt2.x, pt2.y);
		}
	*/
	fpoint(fpoint pp);
	
	/**
		x coordinate of the point
	Example:
		void	run_fpoint_x_y()
		{
			fpoint	pt(1.2, 7.6);
			
			printf("x = %f   y = %f\n", pt.x, pt.y);
		}
	*/
	double 	x;
	
	/**
		y coordinate of the point
	Example:
		void	run_fpoint_x_y()
		{
			fpoint	pt(1.2, 7.6);
			
			printf("x = %f   y = %f\n", pt.x, pt.y);
		}
	*/
	double 	y;
};

/** >Internal Origin Objects
		The fpoint3d class provides methods and properties common to data points located
		in a three dimensional space whose (x,y,z) coordinates are doubles.
	Example:
		fpoint3d fpt3(2.2, 7.6, 0.5);
		printf("x = %g   y = %g   z = %g\n", fpt3.x, fpt3.y, fpt3.z);
*/
class fpoint3d
{
public:

	/**
		constructor which initializes the fpoint3d object from x, y, and z coordinates
	Paramaters:
		x = (optional) x coordinate
		y = (optional) y coordinate
		z = (optional) z coordinate
	Example:
		void	run_fpoint3d_Constructor()
		{
			fpoint3d	pt(2.2, 7.6, 0.5);
			
			printf("x = %f   y = %f   z = %f\n", pt.x, pt.y, pt.z);
		}
	*/
	fpoint3d(double x = 0, double y = 0, double z = 0);
	
	/**
		Copy constructor which initializes the fpoint3d object from another fpoint3d object
	Parameters:
		pp = source fpoint3d object
	Example:
		void	run_fpoint3d_Constructor()
		{
			fpoint3d	pt(4.4, 5.5, 7.7);
			
			// Initialize pt2 from pt:
			fpoint3d	pt2(pt);
			
			// Display pt2:
			printf("x = %f   y = %f   z = %f\n", pt2.x, pt2.y, pt2.z);
		}
	*/
	fpoint3d(fpoint3d pp);
	
	/**
		x coordinate of the point
	Example:
		void	run_fpoint3d_x_y()
		{
			fpoint3d	pt(2.2, 7.6, 0.5);
			
			printf("x = %f   y = %f   z = %f\n", pt.x, pt.y, pt.z);
		}
	*/
	double 	x;
	
	/**
		y coordinate of the point
	Example:
		void	run_fpoint3d_x_y()
		{
			fpoint3d	pt(2.2, 7.6, 0.5);
			
			printf("x = %f   y = %f   z = %f\n", pt.x, pt.y, pt.z);
		}
	*/
	double 	y;
	
	/**
		z coordinate of the point
	Example:
		void	run_fpoint3d_x_y()
		{
			fpoint3d	pt(2.2, 7.6, 0.5);
			
			printf("x = %f   y = %f   z = %f\n", pt.x, pt.y, pt.z);
		}
	*/
	double 	z;
};

#endif // _POST_ORIGIN7_
		

/** >Internal Origin Objects
		The ROIObject class provides methods and properties common to all Origin
		region of interest objects. An Origin region of interest object is used
		to identify a region of interest in an Origin matrix. An Origin C ROIObject
		is a wrapper object that is a reference to an internal Origin region of interest
		object. Origin C wrapper objects do not actually exist in Origin and merely
		refer to the internal Origin object. Consequently, multiple Origin C wrapper
		objects can refer to the same internal Origin object. The ROIObject class is
		derived from the GraphObject class from which it inherits methods and properties.
*/
class ROIObject : public GraphObject
{
public:

	/**
	*/
	ROIObject();
};


#pragma	dll(@OK)
/** >Internal Origin Objects
*/
ROIObject		GetROI();

#endif //_GRAPH_H

