/*------------------------------------------------------------------------------*
 * File Name: Wksheet.h															*
 * Creation: CPY 4/2/2001														*
 * Purpose: Origin C header	for Worksheet class and other related functions		*
 * Copyright (c) OriginLab Corp.2001											*
 * All Rights Reserved															*
 *------------------------------------------------------------------------------*/

#ifndef _WKSHEET_H
#define _WKSHEET_H

#include <common.h> // must always include this, has printf etc

#ifndef _STRING_H //string.h is big, so we should not even load if not needed 
#include <string.h> // most likely you will also need strings
#endif //!_STRING_H

#include <OC_const.h> // consts used in Origin internal functions
#include <Collection.h> //Contains declaration of the template class Collection

#ifndef _GRAPH_H // graph.h is big, avoid loading if possible
#include <graph.h> // graph, plot and graphic objects
#endif //!_GRAPH_H

/** >Internal Origin Objects
		The Datasheet class provides methods and properties common to Origin worksheet
		and matrix layers. The Datasheet class is derived from the Layer and OriginObject
		classes from which it inherits methods and properties.
*/
class Datasheet : public Layer
{
public:

/**
	Default constructor.
*/
	Datasheet();
	/**#
	*/
	Datasheet(Layer & layer);
/**
		Get the number of columns in a Datasheet including MatixLayers and Worksheets.
	Example: 
		Worksheet wks("Data1");
		int nCols = wks.GetNumCols();
	Return: 
		Returns the number of columns in the Datasheet including MatixLayers and Worksheets.
	SeeAlso:
		Datasheet::GetNumRows, Datasheet::SetNumRows
*/
	UINT	GetNumCols(); // Get the number of columns in the Datasheet.
	
/**
	Remarks: 
		Retrieves the name of the worksheet. This function is useful only for Excel
		workbooks where one workbook can have multiple sheets, each with its own name.
	Return: 
		The TRUE for succes, FALSE if failed.
	Example: 
		Page		pg("Book1");
		Layer		lay= pg.Layers();		// gets the active worksheet from Excel
		Worksheet	wks;
		wks = (Worksheet)lay;
		if (wks)
		{
			string	strName;
			wks.GetName(strName);
			out_str(strName);
		}
*/
	BOOL	GetName(string &str);

/**
		Get the number of rows in a Datasheet including MatixLayers and Worksheets.
	Example: 
		Worksheet wks("Data1");
		int nRows = wks.GetNumRows();
	Return: 
		Returns the number of rows in the Datasheet including MatixLayers and Worksheets.
	SeeAlso: 
		Datasheet::GetNumCols, Datasheet::SetNumRows
*/	
	UINT	GetNumRows(); // Get the number of rows in the Datasheet.
	
/**
		Create a new worksheet or matrix using the supplied template 
		and attach it to the object.
	Parameters: 
		lpcszTemplate = the template file name, "" or NULL to use a default template. 
						"0" to create without template
		nOption = enum {
						CREATE_TEMP = 0,	// it will be destroyed when destroying the object 
											// (when it exits the scope) and is created invisible
						CREATE_VISIBLE_SAME,// visibility is that which is stored in the template 
											// (does not apply if template not supplied)
						CREATE_VISIBLE,
						CREATE_HIDDEN,};
	Return:
		TRUE for success, otherwise FALSE.
	Example:
		Worksheet	wks;
		BOOL		bRet = wks.Create("c:\\mytemplate.otw", CREATE_VISIBLE_SAME);
		
		Worksheet	newWks;
		if(newWks.Create())
			printf("New wks(%s) is created\n",newWks.GetPage().GetName()); 
	
		Worksheet	TempWks;
		if(TempWks.Create(NULL,CREATE_HIDDEN))
			printf("New hidden wks(%s) is created\n",TempWks.GetPage().GetName());
		
		MatrixLayer mat; // change to MatrixLayer for Matrix 
		if(mat.Create(NULL,CREATE_HIDDEN))
			printf("Hidden matrix %s of %d x %d is created\n",
				mat.GetPage().GetName(), mat.GetNumCols(), mat.GetNumRows());
	SeeAlso:
		Worksheet::CreateCopy			
*/
	BOOL	Create(LPCSTR lpcszTemplate = NULL, int nOption = CREATE_VISIBLE);

/**
		Set the number of rows in a Datasheet including MatixLayers and Worksheets.
	Example: 
		MatrixLayer ml("Matrix1");
		ml.SetNumRows(50);
	Parameters:
		nRows=Input number of rows to set
	Return: 
		Returns TRUE on successful exit and FALSE on failure.
	SeeAlso: 
		Datasheet::GetNumRows, Datasheet::GetNumCols
*/
	BOOL SetNumRows(uint nRows); // Set the number of rows in a Datasheet.
	
#if _OC_VER > 0x0703
	/**#
	Reset Matrix or Worksheet
	Parameters:
		bReduceSize =  in Matrix, reduce to 2x2, in Worksheet, same as ClearWorksheet macro
	Returns:
		TRUE if reset successful
		FALSE otherwise
	Example:
		Worksheet wks("Data1");
		wks.Reset();
	*/
	BOOL	Reset( BOOL bReduceSize = TRUE );
#endif // _OC_VER > 0x0703
};

/** >Internal Origin Objects
		The Worksheet class provides methods and properties common to worksheet layers in
		Origin worksheet pages. An Origin worksheet may contain a number of worksheet columns
		thus the Worksheet class contains a collection of all the columns in the worksheet.
		An Origin C Worksheet object is a wrapper object that is a reference to an internal
		Origin worksheet object. Origin C wrapper objects do not actually exist in Origin and
		merely refer to the internal Origin object. Consequently, multiple Origin C wrapper
		objects can refer to the same internal Origin object. The Worksheet class is derived
		from the Datasheet, Layer, and OriginObject classes from which it inherits methods
		and properties.
	Example:
		Worksheet wks=Project.ActiveLayer();
		if(wks)
		{
			out_int("Num of columns =", wks.GetNumCols());
			while(wks.DeleteCol(0))	 // Remove all columns in worksheet
				;
			ASSERT(wks.GetNumCols() == 0);
		}
		else
			out_str("No active worksheet.");
*/ 
class Worksheet : public Datasheet 
{
public:
/**
	Remarks: 
			Constructor for the Worksheet class that creates a wrapper Worksheet object which is a reference 
			to an internal Origin worksheet object.If the internal origin object is not a worksheet object, 
			then a reference cannot be created. 
	Parameters:
			lpcszWksName = The name of an Origin worksheet object
	Example:
		Worksheet wks("data1");
		if(!wks)
			out_str("There is no such worksheet");
*/
	Worksheet(LPCTSTR lpcszWksName);
/**
	Remarks:
		Worksheet is one of the wrapper objects that is a reference to the actual internal Origin object.
		You can construct a new Worksheet object from another layer object. The worksheet will become invalid if 
		the layer to construct with is actually not a worksheet
	Parameter:
		
	
	Example1:
		Worksheet wks(Project.ActiveLayer());
		if(!wks)
		{
			out_str("The active layer is not a worksheet, or there is nothing in the project");
		}
		else
			printf("Worksheet %s has %d columns\n",wks.GetPage().GetName(),wks.GetNumCols());
	
	
	Example2:
		Worksheet wks1 = Project.ActiveLayer();
		Worksheet wks2(wks1);
		if(!wks2)
		{
			out_str("The active layer is not a worksheet, or there is nothing in the project");
		}
		else
			printf("Worksheet %s has %d columns\n",wks2.GetPage().GetName(),wks2.GetNumCols());
		
*/
	Worksheet(Layer & layer);
	

/**
	Remarks: 
		Attach to an Origin worksheet
	Parameters: 
		lpcszWksName = The name of an Origin worksheet object
	Return: 
		TRUE if the worksheet exists; FALSE if the worksheet does not exist
	Example: 
		Worksheet wks;
		ASSERT(wks.Attach("Data1"));
*/	
	BOOL	Attach(LPCSTR lpcszWksName);		// Attach worksheet object to a real Origin worksheet

/**
	Remarks: 
		Add a new column to the worksheet and name it if specified.
	Parameters: 
		lpcszColName = Optional, Name of column
	Return: 
		The index of the newly added column (0 offset)
	Example: 
		Worksheet wks("data1");
		int colnum = wks.AddCol(); //Add a new column
		colnum = wks.AddCol("Temp"); //Add a new column, called Temp
*/
	int		AddCol(LPCSTR lpcszColName=NULL);	// Return index of newly added column (0 offset) or -1 if add column fails

/**
	Remarks: 
		Add a new column to the worksheet with the given name and store the name in a string. 
		If the given name already exists then increase it.
	Parameters: 
		lpcszColName = desired column name
		strColNameCreated = string to store the actual column name on return
	Return: 
		The index of the newly added column (0 offset)
	Example: 
		Worksheet wks("Data1");
		string str;
		int nCol = wks.AddCol("Temp",str);
		if(nCol >= 0)
			printf("New Column created, name = %s\n",str);
		else
			out_str("Column creation failed");
*/												
	int		AddCol(LPCSTR lpcszColName, string& strColNameCreated); 
																	

/**
	Remarks: 
		This member function is similar to paste text from clipboard into the worksheet
		The data is tab delimited
	Parameters: 
		lpcszText = Tab delimited data values
		nRow = Optional row number to begin the paste
		nCol = Optional column number to begin the paste
		nRepaintMode = Window Refresh mode
	Return: 
		TRUE for success, otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		string str = "1\t2\t3";
		wks.PasteData(str); // append to end of wks from 1st column
*/
	BOOL	PasteData(LPCTSTR lpcszText, // rows of text with tab as list separator and newline of new rows
						int nRow = -1,		// beginning cell's row number, -1 means to append from the first unfilled row of the worksheet
						int nCol = 0,		// beginning cell's column number, -1 if to append from the first unfilled column ( not supported yet)
						int nRepaintMode = 0);// default will use realtime drawing, can disable repaint, or direct repaint 	
		
/**#
*/
	BOOL GetASCIMP(ASCIMP &stAscImp);  
/**#
*/
	BOOL SetASCIMP(ASCIMP &stAscImp);  

//------ CPY 9/24/02 v7.0404 QA70-2658 ASCII_IMPORT_FROM_OC
// requires Origin 7 SR3 or later
/**
    Remarks:
	    Import ASCII file into worksheet by using a ASCIMP struct that will define how to import the data
	Parameters: 
		lpcszFilename = a full path ASCII file name
		stAscImp = An ASCIMP struct that has been setup with import settings
	Return: 
		0 if success, otherwise returns error codes
	Example:
	void test_ascii_import()
	{
		ASCIMP	ascimp;
		string strFile = GetOpenBox("*.dat");
		if(AscImpReadFileStruct(strFile,&ascimp)==0)
		{
			Worksheet wks;
			wks.Create();
			wks.ImportASCII(strFile, ascimp);
		}
	}

*/
	int ImportASCII(LPCSTR lpcszFilename, ASCIMP &stAscImp);  

/**
    Remarks:
	    Import ASCII file into worksheet by using automatic procedure
	Parameters: 
		lpcszFilename = a full path ASCII file name
		bRenameWks = to rename the worksheet to the file name, might need to be fitted for worksheet naming requirement
		nMode = ASCIMP_MODE_REPLACE_DATA, ASCIMP_MODE_APPEND_COLS, ASCIMP_MODE_APPEND_ROWS
		bRenameCols = to rename the columns using labels from the file
	Return: 
		True = succesful; otherwise FALSE
	Example:
	void import_to_active_wks()
	{
		Worksheet wks = Project.ActiveLayer();
		if(wks)
		{
			string strFile = GetOpenBox("*.dat");
			wks.ImportASCII(strFile);
		}
	}

*/
	BOOL ImportASCII(LPCSTR lpcszFilename, BOOL bRenameWks=FALSE, int nMode = ASCIMP_MODE_REPLACE_DATA, BOOL bRenameCols=FALSE);  

/**
	Export worksheet to a ASCII file
	Parameters: 
		lpcszFilename = a full path file name
		dwCntrl = can be the following value(defined in OC_Const.h):
					WKS_EXPORT_HEADING	0x0001	// col name
					WKS_EXPORT_ALL		0x0002	// ignore c1c2r1r2
					WKS_EXPORT_LABELS	0x0004	// col label
					WKS_EXPORT_SELECTED 0x0008	// only selected cols
		lpcszSeparator = the separator in the data file. The default separator is "\t".
		nR1 = first row in the data range to be included with the data file.
		nC1 = first column in the data range to be included with the data file.
		nR2 = last row in the data range to be included with the data file.
		nC2 = last column in the data range to be included with the data file.
	Return: 
		On error, returns -1, otherwise returns the exported file size.
	Example:
	void test_export_ascii()
	{
		Worksheet wks = Project.ActiveLayer();
		if(wks)
		{
			string strFile = GetSaveAsBox("*.dat");
			wks.ExportASCII(strFile, WKS_EXPORT_ALL);
		}
	}

*/
	int ExportASCII(LPCSTR lpcszFilename, DWORD dwCntrl, char cSeparator = '\t', int nR1 = 0, int nC1 = 0, int nR2 = -1, int nC2 = -1);  
	
/**
	Remarks: 
		Insert an empty row in the worksheet 
	Parameters: 
		nPos = Row number with zero offset
	Return: 
		True = succesful; otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.InsertRow(10)); //Insert an empty row on 10th row
	SeeAlso: 
		Worksheet::DeleteRow, Worksheet::AppendRows	
*/	
	BOOL InsertRow(int nPos);
	
/**
	Remarks: 
		Delete a row in the worksheet 
	Parameters: 
		nPos = Row number with zero offset
	Return: 
		TRUE if Succesful; otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.DeleteRow(10)); //Delete a row from 10th row
	SeeAlso: 
		Worksheet::InsertRow, Worksheet::AppendRows	
*/		
	BOOL DeleteRow(int nPos);
	
/**
	Remarks: 
		Append rows to the worksheet 
	Parameters: 
		nRows = Number of rows to append; 1 by default
	Return: 
		True = Succesful; otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.AppendRows(10));
	SeeAlso: 
		Worksheet::InsertRow, Worksheet::DeleteRow
*/		
	BOOL AppendRows(int nRows = 1 );


	
/**
	Remarks: 
		Delete a column in the worksheet 
	Parameters: 
		nCol = Column number with zero offset
	Return: 
		True = Succesful; otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.DeleteCol(0));
	SeeAlso: 
		Worksheet::InsertCol	
*/	
	BOOL DeleteCol(int nCol);
	
/**
	Remarks: 
		Delete a range of data in the worksheet 
	Parameters: 
		nRowBegin = Beginning row number with zero offset
		nColBegin = Beginning column number with zero offset
		nRowEnd = Ending row number with zero offset
		nColEnd = Ending column number with zero offset
		bInserNotDelete = false by default; if TRUE then insert rows
	Return: 
		TRUE for success; otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.DeleteRange(1,1,3,2));
	SeeAlso: 
		Worksheet::DeleteCol
*/
	BOOL DeleteRange(int nRowBegin, int nColBegin, int nRowEnd, int nColEnd, BOOL bInsertNotDelete = false);
	
/**#
		Retrieves the current selection from the worksheet.
	Remarks:
		This function is replaced by GetSelectedRange which follows a more consistent argument list of R1C1:R2C2. So this function is provided for
		backward compatible reason only
*/
	int		GetSelection(int &c1, int &c2, int &r1, int &r2, string *pstrAddress = NULL);

/**
		Retrieves the current selection from the worksheet.
	Parameters:
		r1 = on return it receives the index of the first row in the selection, if any
		c1 = on return it receives the index of the first column in the selection, if any
		r2 = on return it receives the index of the last row in the selection, if any
		c2 = on return it receives the index of the last column in the selection, if any
		pstrAddress = optional argument that will on return receive the address of the Excel selection range as string (assuming it is an Excel worksheet)
	Return: 
		Integer indicating the type of selection. It can be a bitwise combination of one or more of the following:
		WKS_SEL_NONE  					// no selection
		WKS_SEL_EDIT					// one cell being edited
		WKS_SEL_COLUMN					// one or more entire columns selected
		WKS_SEL_ROW						// one or more entire rows selected
		WKS_SEL_RANGE					// more than one cell selected
		WKS_SEL_ONE_COL					// exactly one column selected
		WKS_SEL_DISCONTIGUOUS			// discontiguous columns selected
		WKS_SEL_ALL						// entire worksheet
	Example: 
		Worksheet wks("Data1");
		int			 r1, c1,r2, c2;
		string		strAddress;
		int			seltype = wks.GetSelectedRange(r1, c1, r2, c2, &strAddress);
		
		printf("sel = %d\tr1 = %d\tc1 = %d\tr2 = %d\tc2 = %d\n", seltype, r1, c1, r2, c2);
		out_str(strAddress);
*/
	int		GetSelectedRange(int &r1, int &c1, int &r2, int &c2, string *pstrAddress = NULL);

/**
	Remarks:
		Retrieves the 0-offset indices of the selected columns in the worksheet.
	Parameters:
		v = vector of integers which will recieve the 0-offset indices of the selected columns
	Return: 
		TRUE for success; otherwise FALSE
	Example: 
		vector<int>		v;
		Worksheet		wks("Data1");
		
		BOOL			bOK = wks.GetSelectedColumns(v);
*/
	BOOL	GetSelectedColumns(vector<int> &v);
	
/**
	Remarks:
		Updates the underlying Origin worksheet from Excel. This method is the equivalent of the combined Layer -s; and set %H -ui; LabTalk commands.
	Parameters:
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		Page			pg("Book1");
		Worksheet		wks;
		wks = (Worksheet)pg.Layers("Sheet1");
		
		BOOL			bOK = wks.UpdateOrigin();
*/
	BOOL UpdateOrigin();


	
	
/**
	Remarks: 
		Insert an empty column in the worksheet with the given name and store the name in a string. 
		If the given name already exists then increment it.
	Parameters: 
		nPos = Column number with zero offset
		lpcszColName = string to name inserted column
		strColNameCreated = string to store the col name created
	Return: 
		True for succesful; otherwise FALSE
	Example: 
		Worksheet wks("Data1");
		string ColName = "NewColoumn";
		string ColNameCreated;
		ASSERT(wks.InsertCol(1, ColName, ColNameCreated)); //Insert an empty column at the 1st column ( 0 offset)
		printf("Column Name created is %s\n", ColNameCreated);
	SeeAlso: 
		Worksheet::DeleteCol, Worksheet::InsertRow	
*/
	BOOL InsertCol(int nPos, LPCSTR lpcszColName, string &strColNameCreated);


/**
	Remarks: 
		Get the value at the location (nRow, nCol) in the worksheet 
	Parameters: 
		nRow = Row number with zero offset
		nCol = Column number with zero offset
	Return: 
		Value at the location (nRow, nCol) in the worksheet
	Example: 
		Worksheet wks("Data1");
		wks.SetCell(0, 0, 3.14);
		double cellValue = wks.Cell(0, 0);
		printf("Value at (0,0) location is %f\n", cellValue);
	SeeAlso: 
		Worksheet::GetCell, Worksheet::SetCell, Worksheet::TCell
*/
	double Cell(int nRow, int nCol);
	
	
/**
	Remarks: 
		Get the value at the location (nRow, nCol) in the worksheet as a string
	Parameters: 
		nRow = Row number with zero offset
		nCol = Column number with zero offset
		strText = string to store the value
	Return: 
		TRUE for success; otherwise FALSE.
	Example: 
		Worksheet wks("Data1");
		wks.SetCell(0, 0, 3.14);
		string strText;
		wks.GetCell(0, 0, strText);
		printf("Value at (0,0) location is %s\n", strText);
	SeeAlso: 
		Worksheet::Cell, Worksheet::SetCell, Worksheet::TCell
*/
	BOOL GetCell(int nRow, int nCol, string& strText);
	
	
/**
	Remarks: 
		Set the value at the location (nRow, nCol) in the worksheet as a string
	Parameters: 
		nRow = Row number with zero offset
		nCol = Column number with zero offset
		lpcszText = string to be set to location (nRow, nCol)
	Return: 
		TRUE for success; otherwise FALSE.
	Example: 
		Worksheet wks("Data1");
		string strText="Monday";
		wks.SetCell(0, 0, strText);
		string cellText;
		wks.GetCell(0, 0, cellText);
		printf("Value at (0,0) location is %s\n", cellText);
	SeeAlso: 
		Worksheet::Cell, Worksheet::GetCell, Worksheet::TCell
*/
	BOOL SetCell(int nRow, int nCol, LPCSTR lpcszText);
/**
	Remarks: 
		Set the value at the location (nRow, nCol) in the worksheet
	Parameters: 
		nRow = Row number with zero offset
		nCol = Column number with zero offset
		value = Value to set at the location (nRow, nCol)
	Return: 
		TRUE for success; otherwise FALSE.
	Example: 
		Worksheet wks("Data1");
		double value = 1000.05;
		wks.SetCell(0, 0, value);
		double cellValue = wks.Cell(0, 0);
		printf("Value at (0,0) location is %f\n", cellValue);
	SeeAlso: 
		Worksheet::Cell, Worksheet::GetCell, Worksheet::TCell
*/
	BOOL SetCell(int nRow, int nCol, double value);
	
	
/**
	Remarks: 
		Get the value at the location (nRow, nCol) in the worksheet as a string
	Parameters: 
		nRow = Row number with zero offset
		nCol = Column number with zero offset
	Return: 
		string at the location (nRow, nCol) in the worksheet
	Example: 
		Worksheet wks("Data1");
		string strText="Monday";
		wks.SetCell(0, 0, strText);
		string cellText = wks.TCell(0, 0);
		printf("Value at (0,0) location is %s\n", cellText);
	SeeAlso: 
		Worksheet::Cell, Worksheet::GetCell, Worksheet::SetCell
*/
	string TCell(int nRow, int nCol);

#ifdef	ORIGIN_COM_SUPPORT
/**
	Put the data from an SPC blob into the worksheet.
Parameters: 
	varBlob = the VARIANT blob containing the data
	nRow = Optional row number to begin the import at
	nCol = Optional column number to begin the import at
	nRepaintMode = Window Refresh mode
Return: 
	TRUE for success; otherwise FALSE
Example: 
	void	blob_to_wks()
	{
		Object			oSpcIO;
		string			strPathName = "E:\\myfile.spc";
		// the GRAMS object used for importing the SPC file
		oSpcIO = CreateObject("GSpcIOLib.GSpcIO");	
		oSpcIO.OpenFile(strPathName);
		
		_VARIANT		varBlob;
		varBlob = oSpcIO.SaveBlob();
		Worksheet		wks("Data1");
		BOOL			bRet = wks.PutSPCBlob(varBlob);
		
		out_int("Return value from PutSPCBlob is ", bRet);
	}
	Remarks:
		This function is available only for OriginPro versions, or with a special COM enabled license
*/
	BOOL	PutSPCBlob(_VARIANT &varBlob,
						int nRow = -1,
						int nCol = 0,
						int nRepaintMode = 0);
/**
		Put the data from a recordset into the worksheet.
	Remarks:
		This function is available only for OriginPro versions, or with a special COM enabled license
	Parameters: 
		objrs or varobjrs	= the recordset
		nRowBegin			= the starting row in the worksheet
		nNumRows			= total number of records to retrieve,or -1 to retrieve all available.
		nColBegin			= it applies only to the options LAYWKGETRECORDSET_BY_COLUMN_INDEX with or without LAYWKGETRECORDSET_BY_COLUMN_INDEX and LAYWKGETRECORDSET_SET_COLUMN_NAME bits 
							  the column in worksheet from which to begin.
		nOptions			= enum {
									LAYWKGETRECORDSET_BY_COLUMN_INDEX = 0,						// beginning with the starting column nColBegin retrieve all fields (will add columns as necessary)
									
									LAYWKGETRECORDSET_BY_COLUMN_NAME = 2,						// it retrieves only the data from those fields whose names match column names
									LAYWKGETRECORDSET_BY_COLUMN_LABEL,							// it retrieves only the data from those fields whose names match column labels 
								
									LAYWKGETRECORDSET_SET_COLUMN_NAME = 0x00010000,				// used in conjunction with LAYWKGETRECORDSET_BY_COLUMN_INDEX to set column names to field names						
									LAYWKGETRECORDSET_SET_COLUMN_LABEL = 0x00020000,				// used in conjunction with LAYWKGETRECORDSET_BY_COLUMN_INDEX to set column labels to field names						
								
									LAYWKGETRECORDSET_BY_COLUMN_INDEX_WITH_COLUMN_RENAMING = LAYWKGETRECORDSET_BY_COLUMN_INDEX | LAYWKGETRECORDSET_SET_COLUMN_NAME,		// same as LAYWKGETRECORDSET_BY_COLUMN_INDEX, except that it will also rename all columns to the field names
								};
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	recordset_to_worksheet()
		{
			Object		ocrs;
			ocrs = CreateObject("ADODB.Recordset");
			ocrs.open("select * from customers", "Provider=SQLOLEDB; Data Source=myServer; Initial Catalog=northwind; User ID=myID; Password=myPassword;");
			Worksheet		wks("Data1");
			// Put the recordset into worksheet:
			BOOL			bRet = wks.PutRecordset(ocrs, 0, -1, 0, LAYWKGETRECORDSET_BY_COLUMN_INDEX | LAYWKGETRECORDSET_SET_COLUMN_NAME);
			out_int("bRet = ", bRet);
		}
	SeeAlso: 
		Worksheet::UpdateRecordset	
	
*/
	BOOL	PutRecordset(Object &objrs,
								int nRowBegin = 0,
								int nNumRows = -1,
								int nColBegin = 0,
								int nOptions = LAYWKGETRECORDSET_BY_COLUMN_INDEX);
/**
		Put the data from a recordset into the worksheet. 
	Remarks:
		This function is available only for OriginPro versions, or with a special COM enabled license
	Parameters: 
		objrs or varobjrs	= the recordset
		nRowBegin			= the starting row in the worksheet
		nNumRows			= total number of records to retrieve,or -1 to retrieve all available.
		nColBegin			= it applies only to the options LAYWKGETRECORDSET_BY_COLUMN_INDEX and LAYWKGETRECORDSET_BY_COLUMN_INDEX_WITH_COLUMN_RENAMING
							  the column in worksheet from which to begin.
		nOptions			= enum {
									LAYWKGETRECORDSET_BY_COLUMN_INDEX = 0,						// beginning with the starting column nColBegin retrieve all fields (will add columns as necessary)
									
									LAYWKGETRECORDSET_BY_COLUMN_NAME = 2,						// it retrieves only the data from those fields whose names match column names
									LAYWKGETRECORDSET_BY_COLUMN_LABEL,							// it retrieves only the data from those fields whose names match column labels 
								
									LAYWKGETRECORDSET_SET_COLUMN_NAME = 0x00010000,				// used in conjunction with LAYWKGETRECORDSET_BY_COLUMN_INDEX to set column names to field names						
									LAYWKGETRECORDSET_SET_COLUMN_LABEL = 0x00020000,				// used in conjunction with LAYWKGETRECORDSET_BY_COLUMN_INDEX to set column labels to field names						
								
									LAYWKGETRECORDSET_BY_COLUMN_INDEX_WITH_COLUMN_RENAMING = LAYWKGETRECORDSET_BY_COLUMN_INDEX | LAYWKGETRECORDSET_SET_COLUMN_NAME,		// same as LAYWKGETRECORDSET_BY_COLUMN_INDEX, except that it will also rename all columns to the field names
								
								};
 
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	recordset_to_worksheet()
		{
			Object		ocrs;
			ocrs = CreateObject("ADODB.Recordset");
			ocrs.open("select * from customers", "Provider=SQLOLEDB; Data Source=myServer; Initial Catalog=northwind; User ID=myID; Password=myPassword;");
			Worksheet		wks("Data1");
			// Put the recordset into worksheet:
			BOOL			bRet = wks.PutRecordset(ocrs, 0, -1, 0, LAYWKGETRECORDSET_BY_COLUMN_INDEX | LAYWKGETRECORDSET_SET_COLUMN_LABEL);
			out_int("bRet = ", bRet);
		}
	SeeAlso: 
		Worksheet::UpdateRecordset			
*/
	BOOL	PutRecordset(_VARIANT &varobjrs,
								int nRowBegin = 0,
								int nNumRows = -1,
								int nColBegin = 0,
								int nOptions = LAYWKGETRECORDSET_BY_COLUMN_INDEX);
/**
		Put the data from the worksheet into the recordset. The locktype of the recordset must be appropriate for
		adding data which means that it should not be read-only. 
	Remarks:
		This function is available only for OriginPro versions, or with a special COM enabled license
	Parameters: 
		objrs or varobjrs	= the recordset
		nRowBegin			= the starting row in the worksheet
		nNumRows			= total number of rows to put,or -1 to put all rows.
		nColBegin			= the worksheet column from which to start (the first column will be used for the first field, etc.).
		nNumCols			= the number of columns to use, beginning with nColBegin, or -1 to use all columns to the right of nColBegin.
							  If there are less columns than fields, the remaining fields will be set to null.
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	worksheet_recordset()
		{
			Object		ocrs;
			ocrs = CreateObject("ADODB.Recordset");
			ocrs.open("select * from customers", "Provider=SQLOLEDB; Data Source=myServer; Initial Catalog=northwind; User ID=myID; Password=myPassword;");
			Worksheet		wks("Data1");
			// Put changes in worksheet back to the recordset:
			BOOL			bRet = wks.UpdateRecordset(ocrs);
			out_int("bRet = ", bRet);
		}
	SeeAlso:
		Worksheet::PutRecordset
	
*/
	BOOL	UpdateRecordset(Object &objrs,
								int nRowBegin = 0,
								int nNumRows = -1,
								int nColBegin = 0,
								int nNumCols = -1);
/**
		Put the data from the worksheet into the recordset. The locktype of the recordset must be appropriate for
		adding data which means that it should not be read-only.
	Remarks:
		This function is available only for OriginPro versions, or with a special COM enabled license
	Parameters: 
		objrs or varobjrs	= the recordset
		nRowBegin			= the starting row in the worksheet
		nNumRows			= total number of rows to put,or -1 to put all rows.
		nColBegin			= the worksheet column from which to start (the first column will be used for the first field, etc.).
		nNumCols			= the number of columns to use, beginning with nColBegin, or -1 to use all columns to the right of nColBegin.
							  If there are less columns than fields, the remaining fields will be set to null.
 
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	worksheet_recordset()
		{
			Object		ocrs;
			
			ocrs = CreateObject("ADODB.Recordset");
			
			ocrs.open("select * from customers", "Provider=SQLOLEDB; Data Source=myServer; Initial Catalog=northwind; User ID=myID; Password=myPassword;");
				
			Worksheet		wks("Data1");
			
			// Put changes in worksheet back to the recordset:
			BOOL			bRet = wks.UpdateRecordset(ocrs);
			out_int("bRet = ", bRet);
		}
	SeeAlso:
		Worksheet::PutRecordset
*/
	BOOL	UpdateRecordset(_VARIANT &varobjrs,
								int nRowBegin = 0,
								int nNumRows = -1,
								int nColBegin = 0,
								int nNumCols = -1);

								
#endif //#ifdef	ORIGIN_COM_SUPPORT

#if _OC_VER > 0x0703

	/**
		Create a copy of the worksheet wksSource and attaches it to the object.
	Parameters: 
		wksSource = the worksheet whose copy is made
		nOption	= enum {
						CREATE_TEMP = 0,		// it will be destroyed when destroying the object (when it exits the scope) and is created invisible
						CREATE_VISIBLE_SAME,	// visibility is that of the source worksheet
						CREATE_VISIBLE,			// create visible
						CREATE_HIDDEN,			// create hidden 
					   };
		DWORD dwCtrl = enum {
								DCTRL_COPY_DATA						= 0x00000001, // copy data from source worksheet
								DCTRL_COPY_IN_OPERATIONS			= 0x00000002, // copy operations that have worksheet as output (incoimng operations)
								DCTRL_COPY_OUT_OPERATIONS			= 0x00000004, // copy operations that have worksheet as input (outgoing operations)
								DCTRL_COPY_DEFAULT = DCTRL_COPY_DATA | DCTRL_COPY_OUT_OPERATIONS,
							}
 	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	worksheet_copy()
		{
			Worksheet	wks;
			Worksheet	wksSource("Data1");
			
			int			nOption = CREATE_VISIBLE_SAME;
			BOOL		bRet = wks.CreateCopy(wksSource, nOption);
			
			out_int("bRet = ", bRet);
		}
	SeeAlso:
		Datasheet::Create
	*/
	BOOL	CreateCopy(Worksheet &wksSource, int nOption = CREATE_VISIBLE_SAME, DWORD dwCtrl = DCTRL_COPY_DEFAULT);
#else
/**
		Create a copy of the worksheet wksSource and attaches it to the object.
	Parameters: 
		wksSource = the worksheet whose copy is made
		nOption	= enum {
						CREATE_TEMP = 0,		// it will be destroyed when destroying the object (when it exits the scope) and is created invisible
						CREATE_VISIBLE_SAME,	// visibility is that of the source worksheet
						CREATE_VISIBLE,			// create visible
						CREATE_HIDDEN,			// create hidden 
					   };






 	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	worksheet_copy()
		{
			Worksheet	wks;
			Worksheet	wksSource("Data1");
			
			int			nOption = CREATE_VISIBLE_SAME;
			BOOL		bRet = wks.CreateCopy(wksSource, nOption);
			
			out_int("bRet = ", bRet);
		}
	SeeAlso:
		Datasheet::Create
*/
	BOOL	CreateCopy(Worksheet &wksSource, int nOption = CREATE_VISIBLE_SAME);
	

#endif
/**
		Open a new worksheet from the supplied OGW file and attaches it to the object.
	Parameters: 
		lpcszTemplate = the OGW file name
		nOption	= enum {
						CREATE_TEMP = 0,	// it will be destroyed when destroying the object (when it exits the scope) and is created invisible
						CREATE_VISIBLE_SAME,	// visibility is that which is stored in the OGW file
						CREATE_VISIBLE,		// create visible
						CREATE_HIDDEN,		// create hidden
						}; 
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	worksheet_open()
		{
			Worksheet	wks;
			string		strOGW = "c:\\myfile.ogw";
			
			int			nOption = CREATE_VISIBLE_SAME;
			BOOL		bRet = wks.Open(strOGW, nOption);
			
			out_int("bRet = ", bRet);
		}
*/
	BOOL	Open(LPCSTR lpcszFileName, int nOption = CREATE_VISIBLE);
	
/**
		Show [Hide] column labels in Origin c 

	Parameters:
		bShow = TRUE (default) shows Column Labels
				FALSE will hide Column Labels
	Return:
		None.
	Example:
		void show_wks_labels()
		{
			Worksheet wks("data1");
			string strLabel = "Elapsed\n(in sec.)";
			BOOL b = wks.Columns(0).SetLabel(strLabel);
			wks.ShowLabels();
		}
	SeeAlso:
		Column::SetLabel, Column::GetLabel
*/
	void	ShowLabels(BOOL bShow = TRUE);
/**
		Dump the contents of the worksheet into the designated target window.
	Parameters: 
		dwTarget = enum {
							TYPETARGET_SCRIPT_WINDOW	= 0x00000001UL,		// to the Script window
							TYPETARGET_NAMED_WINDOW		= 0x00000002UL,		// to the particular Notes window
							TYPETARGET_OUTPUTLOG		= 0x00000004UL,		// to output log
							TYPETARGET_SHOW_ERROR		= 0x00000008UL,		// show errors
						};
			These bits cab be combined.
			Note: TYPETARGET_SHOW_ERROR is a constant which should be combined (bitwise "OR-ed") with other constants. 
			In the case it is combined with TYPETARGET_NAMED_WINDOW, it will report into Script Window (or other output) 
			when the Notes window with the supplid name does not exist.In other words, it is supposed to make the Type method 
			"report" method errors (same as type.wks() method of LabTalk works). If TYPETARGET_SHOW_ERROR is supplid alone, 
			nothing should happen.

		lpcszNotesWindow =	the name of the notes window to receive the output 
							(used only for TYPETARGET_NAMED_WINDOW).
		numCols = the number of columns in one block
		nTab = tab width (if 0, use worksheet column width as tab width)
		nRepeatCol = the 1-offset index of the column which is to appear at 
						the beginning of every block
	Return:
		TRUE for sccess otherwise FALSE.
	Example:
		void	worksheet_type()
		{
			Worksheet	wks("Data1");
			BOOL		bRet = wks.Type(TYPETARGET_OUTPUTLOG | TYPETARGET_SHOW_ERROR, NULL, 2, 0, 1);
			
			out_int("bRet = ", bRet);
		}
*/
#if _OC_VER > 0x0703	
	BOOL	Type(DWORD dwTarget, LPCTSTR lpcszNotesWindow = NULL, int numCols = 5, int nTab = 0, int nRepeatCol = 0, TreeNode &trRanges = NULL);
#else
	BOOL	Type(DWORD dwTarget, LPCTSTR lpcszNotesWindow = NULL, int numCols = 5, int nTab = 0, int nRepeatCol = 0);
#endif

/**
		Get the minimum begining and maximum ending row indices for the specified columns in this Worksheet.
	Parameters:
		iR1=Output minimum begining row index (0 based)
		iR2=Output maximum ending row index (0 based)
		iC1=Input begining column index (0 based offset, default 0 is first column)
		iC2=Input ending column index (0 based offset, default -1 to indiate last column)
		bExcludeMissingValues=TRUE (default) advances iR1 to first row with non-missing value and retards iR2 to last row with with non-missing value.
		FALSE will not modify iR1 and iR2
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		void wks_get_range()
		{
			Worksheet wks = Project.ActiveLayer();
			int nR1, nR2;
			
			wks.GetRange(nR1, nR2);
			// need to convert to LabTalk indexing by +1
			printf("%s[%d:%d]\n", wks.GetPage().GetName(), nR1+1, nR2+1);
			// show for each column and check consistency
			foreach(Column cc in wks.Columns)
			{
				Dataset aa(cc);
				printf("Col(%s)[%d:%d]\n",cc.GetName(), aa.GetLowerIndex()+1, aa.GetUpperIndex()+1);
			}
		}
*/
	BOOL GetRange( int& iR1, int& iR2, int iC1 = 0, int iC2 = -1, BOOL bExcludeMissingValues = TRUE);
/**
		Set the begining and ending row indices for the specified columns in this Worksheet.
	Parameters:
		iR1=Input begining row index (0 based)
		iR2=Input ending row index (0 based)
		iC1=Input begining column index (0 based offset, default 0 is first column)
		iC2=Input ending column index (0 based offset, default -1 is last column)
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		// usage
		// wks_set_range 1 0;// to clear worksheet
		// wks_set_range 3 200;// show rows from 3 to 200
		void wks_set_range(int nBegin, int nEnd)
		{
			Worksheet wks = Project.ActiveLayer();
			// convert LabTalk index into C index
			int nR1 = nBegin -1;
			int nR2 = nEnd -1;
			
			wks.SetRange(nR1, nR2);
		}
*/
	BOOL SetRange( int iR1, int iR2, int iC1 = 0, int iC2 = -1 ); 

/**
		Get the minimum begining and maximum ending row indices for the specified columns in this Worksheet.
	Parameters:
		iR1=Output minimum begining row index (0 based)
		iC1=Input begining column index(0 based)
		iR2=Output maximum ending row index (0 based)
		iC2=Input ending column index(0 based)
		bExcludeMissingValues=TRUE (default) advances iR1 to first row with non-missing value and retards iR2 to last row with with non-missing value.
		FALSE will not modify iR1 and iR2
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		void wks_get_bound()
		{
			Worksheet wks = Project.ActiveLayer();
			int nR1, nR2;
			
			wks.GetBounds(nR1, 0, nR2, -1);
			// need to convert to LabTalk indexing by +1
			printf("%s[%d:%d]\n", wks.GetPage().GetName(), nR1+1, nR2+1);
			// show for each column and check consistency
			foreach(Column cc in wks.Columns)
			{
				Dataset aa(cc);
				printf("Col(%s)[%d:%d]\n",cc.GetName(), aa.GetLowerIndex()+1, aa.GetUpperIndex()+1);
			}
		}

*/
	BOOL GetBounds( int& iR1, int iC1, int& iR2, int iC2, BOOL bExcludeMissingValues = TRUE);

/**
		Set the begining and ending row indices for the specified columns in this Worksheet.
	Parameters:
		iR1=Input begining row index (0 based)
		iC1=Input begining column index (0 based offset, default 0 is first column)
		iR2=Input ending row index (0 based)
		iC2=Input ending column index (0 based offset, default -1 is last column)
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		// usage
		// wks_set_bound 1 0;// to clear worksheet
		// wks_set_bound 3 200;// show rows from 3 to 200
		void wks_set_bound(int nBegin, int nEnd)
		{
			Worksheet wks = Project.ActiveLayer();
			// convert LabTalk index into C index
			int nR1 = nBegin -1;
			int nR2 = nEnd -1;
			
			wks.SetBounds(nR1, 0, nR2);
		}
*/
	 BOOL SetBounds( int iR1 = 0, int iC1 = 0, int iR2 = -1, int iC2 = -1 );

/**
		Set the ending row indices for the specified columns in this Worksheet.
		Index values are 0 based offsets so use SetUpperBound(-1) to dispaly no
		rows.
	Parameters:
		iR2=Input ending row index (0 based)
		iC1=Input begining column index (0 based offset, default 0 is first column)
		iC2=Input ending column index (0 based offset, default -1 is last column)
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		Worksheet wks = Project.ActiveLayer();
		
		int nR2 = 25;		
		wks.SetUpperBound(nR2);

*/
	BOOL SetUpperBound( int iR2, int iC1 = 0, int iC2 = -1 );
/**
		Set the begining row indices for the specified columns in this Worksheet.
	Parameters:
		iR1=Input begining row index (0 based)
		iC1=Input begining column index (0 based offset, default 0 is first column)
		iC2=Input ending column index (0 based offset, default -1 is last column)
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		Worksheet wks = Project.ActiveLayer();
		
		int nR1 = 5;		
		wks.SetLowerBound(nR1);
*/
	BOOL SetLowerBound( int iR1, int iC1 = 0, int iC2 = -1 );
	/**
		A Collection of all Columns in a Worksheet
	Example:
		Worksheet wks = Project.ActiveLayer();
		if( wks )
		{
			int ii = 1;
			foreach(Column cc in wks.Columns)
			{
				printf("Col(%d)'s name is %s\n",ii++, cc.GetName()); 
			}
		}
	*/
	Collection<Column> Columns;
/**
	Remarks: 
		Used to call Column member functions directly with a '.' after the function call	
	Parameters: 
		nCol = Column number with zero offset (-1 refers to *Active* column)
	Return: 
		A Column
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.Columns(0).IsValid());
*/	
	Column	Columns(int nCol = -1);		//Return a column - can be used to call column members directly with a '.' after function call

/**
	Remarks: 
		Used to call Column member functions directly with a '.' after the function 
		call	
	Parameters: 
		lpcszName = Column name
	Return: 
		A Column
	Example: 
		Worksheet wks("Data1");
		ASSERT(wks.Columns("A").IsValid());
*/	
	Column	Columns(LPCSTR lpcszName);	
	
	/**
	Remarks: 
		Used to apply designation pattern to entire worksheet
	
	Parameters:	
		lpcszDesignations - designation pattern 
			Possible values for designations:
			'X' - X column
			'Y' - Y column
			'Z' - Z column
			'E' - Y error column
			'L' - label column
			'M' - X error column
			'N' - ignore column
		
		bRepeat	- repeat pattern. Defines how the pattern is applied 
		if number of columns in worksheet is larger then length of pattern.
		Possible values:
			TRUE - repeat entire pattern
			FALSE - set designations for the rest of columns to be the last one deifined by pattern
		
	Example: 
		// the following code will set column designations in worksheet "Data1"		// 1st column - X
		// 2nd column - Y
		// 3rd column - label
		// 4th column - Y
		// 5rd column - label
		// pattern will repeat if worksheet contains more than 5 columns
		Worksheet wks("Data1");
		wks.SetColDesignations("XYLYL");
	
	Return:
		TRUE for success. Otherwise FALSE;
	*/
	BOOL	SetColDesignations(LPCSTR lpcszDesignations, BOOL bRepeat = TRUE);
	
#ifdef 	 _CURVE_BASE
	/**
	*/
	curvebase&	GetCurve(int nColx, int nColY);
	/**
	*/
	curvebase&	GetCurve(int nColy);
#endif// _CURVE_BASE
	/**
	Remarks: 
		Gets column designations for entire worksheet in form of string
		Possible values for designations:
			'X' - X column
			'Y' - Y column
			'Z' - Z column
			'E' - Y error column
			'L' - label column
			'M' - X error column
			'N' - ignore column
			
	Example:
		Worksheet wks("Data1");
		string strDesignations = wks.GetColDesignations();
		
	Return:
		column designations for entire worksheet in form of string
	*/
	string	GetColDesignations();
	
	/**
	/**
	Remarks: 
		Gets column designations for entire worksheet in form of string
		Possible values for designations:
			'X' - X column
			'Y' - Y column
			'Z' - Z column
			'E' - Y error column
			'L' - label column
			'M' - X error column
			'N' - ignore column
			
	Example:
		Worksheet wks("Data1");
		string strDesignations;
		wks.GetColDesignations(strDesignations);
		
	Return:
		void
	*/
	void	GetColDesignations(string& strDesignations);
	
	/**
	Remarks: 
			sets the column formats in the specified worksheet.
	Example:
		// the following code will set column format in worksheet "Data1"
		// 1st column - Numeric (0)
		// 2nd column - Text	(1)
		// 3rd column - Time	(2)
		// 4th column - Date	(3)
		// 5th column - Text and Numeric (9)
		Worksheet wks("Data1");
		string strFormats="01239";
		wks.SetColFormats(strFormats);
	*/
	BOOL	SetColFormats(LPCSTR lpcszFormats, BOOL bRepeat = TRUE);
	
	/**
	Remarks:
		  	gets the column formats of all the columns in the worksheet
	Example:
		Worksheet wks("Data1");
		string strFormats;
		strFormats=wks.GetColFormats();
	Return:
		A string containing all the column formats. 
		Example: "99"
		1st column - Text & Numeric (9)
		2nd column - Text & Numeric	(9)
	*/
	string	GetColFormats();
	
	/**
	
	Remarks:
		  	gets the column formats of all the columns in the worksheet. Same as GetColFormats() except that thi s
		  	method takes in a reference to a string as a parameter and returns nothing. 

	Example:
		Worksheet wks("Data1");
		string strFormats;
		wks.GetColFormats(strFormats);
		
	Return:
		void
	*/
	void	GetColFormats(string& strFormats);
	/**#
	Remarks:
       Sort the specified columns in a worksheet in ascending or descending order based on the
       values in the specified column number. All column and row numbers are 0 based offsets. 
   Example:
   		This example sorts column B in a worksheet in descending order, and missing values are 
   		treated as smallest. For this example to run, a worksheet with the name "Data1" should
   		exist in the project.
   		
   		void test_Worksheet_Sort()
		{
				Worksheet wksMy("Data1");
				if (wksMy.Sort(1,SORT_DESCENDING))
					out_str("Sort successful!");
				else
					out_str("Sort error!");
		}
   Parameters:
       wByColNum=Input column number containing values to sort worksheet by
       wOrder=Input sort order, possible values are SORT_ASCENDING or SORT_DESCENDING
       bMissingValuesSmall=Input flag specifies whether missing values are considered to be the smallest
                           or largest possible value
       iFromRow=Input first row number to sort
       iFromCol=Input first column number to sort
       iToRow=Input last row number to sort, default -1 sorts to last row
       iToCol=Input last column number to sort, default -1 sorts to last column
   Return:
       Returns TRUE on successful exit and FALSE on error.
   SeeAlso:
       Curve::Sort, vectorbase::Sort, vectorbase::Rank
	*/
	BOOL Sort(int wByColNum = 0,
		BOOL wOrder = SORT_ASCENDING,
        BOOL bMissingValuesSmall = TRUE,
        int iFromRow = 0,
        int iFromCol = 0,
        int iToRow = -1,
        int iToCol = -1
        ); // Sort the specified columns in a worksheet

	/**#
		   Perform a nested sort of the specified columns in a worksheet in ascending or descending
		   order based on the values in the specified column numbers. All column and row numbers are
		   0 based offsets. 
	   Example:
	   		For this example to run, a worksheet with the name "Data1" should exist in the project.
	   		
	   		void test_Worksheet_Sort() 
			{
					Worksheet wksMy("Data1");
					vector<int> vCol = {0,1,2};
					vector<int> vSort={SORT_ASCENDING, SORT_DESCENDING, SORT_DESCENDING};
					
					if (wksMy.Sort(vCol, vSort)) 
						out_str("Sort successful!");
					else
						out_str("Sort error!");
			}

	   Parameters:
		   vSortByCols=Input column numbers containing values to sort worksheet by, first element contains
					   column number of primary sort key, etc.
		   vOrder=Input sort orders, possible values are SORT_ASCENDING or SORT_DESCENDING, first element
				  contains sort order of primary sort key, etc.
		   bMissingValuesSmall=Input flag specifies whether missing values are considered to be the smallest
							   or largest possible value
		   iFromRow=Input first row number to sort
		   iFromCol=Input first column number to sort
		   iToRow=Input last row number to sort, default -1 sorts to last row
		   iToCol=Input last column number to sort, default -1 sorts to last column
	   Return:
		   Returns TRUE on successful exit and FALSE on error.
	   SeeAlso:
		   Curve::Sort, vectorbase::Sort, vectorbase::Rank
	*/
	BOOL Sort(const vector<int>& vSortByCols,
         const vector<BOOL>& vOrder,
         BOOL bMissingValuesSmall = TRUE,
         int iFromRow = 0,
         int iFromCol = 0,
         int iToRow = -1,
         int iToCol = -1
        ); // Perform a nested sort of the specified columns in a worksheet 

#if _OC_VER > 0x0703	
	/**#
		Remarks: 
			Provides categorical information for the worksheet
		Example: 
			int r1 = 0;
			int r2 = -1;
			Matrix<ushort> mEntries("Entries");
			vector<ushort> vRowMap;
			vector<ushort> vCols = { 1, 3, 4 };
			Worksheet wks("ByColumn");
			Dataset<ushort> vRM("data1_b");
			wks.MakeGroupEntriesAndRowMap( r1, r2, mEntries, vRM, vCols );
		Parameters: 
			vCols = input, an array of column indices in the worksheet (if the array is empty, do nothing and return FALSE).
			r1 = input, begining row number to be considered in the worksheet
			r2 = input, ending row number to be considered in the worksheet (r2 = -1 to consider the whole worksheet from r1 on).
			mEntries = output, a matrix with the number of columns equal to the size of vCols; each column corresponds to one column in vCols. 
				The number of rows of the matrix is equal to m_0*m_1*m_2*...*m_(n-1), 
				where n is the size of vCols, while m_i is the size of the categorical map  
				for the column vCols[i]. The values on each column of the matrix are all possible combinations of 
				indices 0, 1, ...,m_i - 1, in a nested way, the first column changes the slowest, etc.
			vRowMap = output, a vector of the size r2 - r1 (+ 1 if r2 is included), which maps every row of worksheet 
				in the range r1->r2 to rows of the matrix considered as bins 
				(so that the first row in the vector, corresponding to r1 row in the worksheet, 
				maps to the first row of the matrix, etc.).
		Return: 
			Returns TRUE on successful exit and FALSE on error.
		SeeAlso:
		   ExtractOneGroup
	*/
	BOOL MakeGroupEntriesAndRowMap(int r1, int r2, matrix<ushort> &mEntries, vector<ushort> &vRowMap, const vector<ushort> &vCols);

	
	/**#
		Remarks: 
			Vector extraction from a column with grouping information 
		Input:
			nCol = the column index
			vRowMap = The output map from the the method MakeGroupEntriesAndRowMap() (see #4139).
			r1Map = It must match r1 passed to MakeGroupEntriesAndRowMap().
			r1, and r2 = the row bounds (or r2=-1 to consider the whole column from r1 to the end);  
				r1 must be >= r1Map, and r2 must be less or equal r2 passed to MakeGroupEntriesAndRowMap().
			nGroupIndex = the values inside vRowMap to be used when creating v.
		
		Output:
			v - the vector which will receive the values from the group
		SeeAlso:
			MakeGroupEntriesAndRowMap
	*/
	BOOL ExtractOneGroup(vector &v, int nCol, const vector<ushort>& vRowMap, int r1Map, int r1, int r2, ushort nGroupIndex);

	/**#
	Input:
		nBegin = beginning of the range
		nEnd = end of the range. if nEnd is -1 then analize to last column of worksheet
	Returns:
		TRUE if there is write protected column within the range between nBegin and nEnd. FALSE otherwise
	SeeAlso:
		Column::IsWriteProtected()	
	*/
	BOOL	IsWriteProtected(int nBegin = 0, int nEnd = -1);

	/**
  			Find column in worksheet by searching for column label string
  		Parameters:
			 lpcszColLabel = string to search for
			 nColBegin = Column index (0 offset) to start searching
			 bCaseSensitive = TRUE if case sensitive
			 bFullMatch = FALSE if only match substring
		Example:
			void show_wks_labels()
			{
				Worksheet wks("data1");
				
				BOOL bOK = wks.Columns(0).SetLabel("AA");
				bOK = wks.Columns(1).SetLabel("aa");
				wks.ShowLabels();
				Column col = wks.FindCol("aa");
				string strLabel = col.GetLabel()
			}
	*/
	Column FindCol(LPCSTR lpcszColLabel, int nColBegin = 0, BOOL bCaseSensitive = TRUE, BOOL bFullMatch = TRUE);

	/**
	Parameters:
		nNumRows = number of rows in worksheet, use -1 to keep current settings
		nNumCols = number of columns, use -1 to keep current settings
		bClearData = decide if existing data in worksheet should be cleared. If True, then all data in worksheet is reset to missing values and each column is reset to upper index = 0, 
		if FALSE, then no action is taken on both the data and the upper index on each column, and newly added columns will be in default state
	Example:
		// the following code will reset the active worksheet with 1000 enumerated cols
		Worksheet wks = Project.ActiveLayer();
		if(wks)
		{
			wks.SetSize(-1,0);// del all columns
			wks.SetSize(1000,3000); //make a worksheet with 3000 cols with 1000 rows each
		}

		//a function to clear worksheet
		void clear_active_worksheet()
		{
 			Worksheet wks = Project.ActiveLayer();
			if(wks)
				wks.SetSize(-1, -1, TRUE);
		}
	Remarks:
		If nNumCols is larger then the current number of columns in the worksheet, then new columns will be added using enumerated column names which is faster then calling AddCol.
		Also, Origin does not support small number of rows in worksheet display, so setting a value of nNumRows smaller then the height of the display will be ignored.
	*/
	BOOL SetSize(int nNumRows, int nNumCols, BOOL bClearData = FALSE);
#endif //#if _OC_VER > 0x0703
};

/** >Internal Origin Objects
		The MatrixLayer class provides methods and properties common to matrix layers
		in Origin matrix pages. An Origin matrix contains a number of matrix objects
		thus the MatrixLayer class contains a collection of all the matrix objects in
		the matrix layer. An Origin C MatrixLayer object is a wrapper object that is
		a reference to an internal Origin matrix layer object. Wrapper objects do not
		actually exist in Origin and merely refer to the internal Origin object.
		Consequently, multiple Origin C wrapper objects can refer to the same internal
		Origin object. The MatrixLayer class is derived from the Datasheet, Layer, and
		OriginObject classes from which it inherits methods and properties.
	Example:
		MatrixLayer ml = Project.ActiveLayer();
		if(ml)
		{
			Matrix mm(ml);// Get the active Matrix from a MatrixLayer
			printf("The active matrix is %d x %d\n",mm.GetNumRows(), mm.GetNumCols());
		}
*/ 
class MatrixLayer : public Datasheet 
{
	
public:
	
/**
	Remarks:
		Constructor for the MatrixLayer class
	Parameters:
		MatrixName = name of the matrix
	Return:
		none
	Example:
		MatrixLayer mm("Matrix1");
*/
	MatrixLayer(const char* MatrixName);
	/**
	Remarks:
		MatrixLayer is one of the wrapper objects that is a reference to the actual internal Origin object.
		You can construct a new MatrixLayer from another layer object. The MatrixLayer will become invalid if 
		the layer to construct with is actually not an object of Matrix class in Origin.
	
	
	Parameters: layer = the worksheet that is currently selected or active	
		
	Return: void
	
	Example:
		MatrixLayer mlay(Project.ActiveLayer());
		if(!mlay)
		{
			out_str("The active layer is not a matrix, or there is nothing in the project");
		}
		else
			printf("Matrix %s has %d columns\n",mlay.GetPage().GetName(),mlay.GetNumCols());
		
	
	*/
	MatrixLayer(Layer & layer);
	
/**#	
	Sets the display format of the data in the matrix 
	Parameters:
		nDisplay = 
		numOfDigits = number of digits allowed in each element
		numofDecimals = number of decimals allowed in each element
	Return:
		TRUE for success; otherwise FALSE
	Example:
		MatrixLayer mm("Matrix1");
		printf("The matrix cell width is %d\n", mm.GetCellWidth());
	
*/
	BOOL SetDataDisplay(int nDisplay, uint numOfDigits, uint numOfDecimals);
	
/**
		Get the matrix cell width
	Parameters:
		None.
	Return:
		The matrix cell width
	Example:
		MatrixLayer mm("Matrix1");
		mm.SetDataDisplay(1,2,3);
		printf("The matrix cell width is %d\n", mm.GetCellWidth());
*/
	uint GetCellWidth();
/**
	
		Set the matrix cell width
	Parameters:
		nCellWidth = The width of the cell
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		MatrixLayer mm("Matrix1");
		mm.SetCellWidth(5);

*/
	BOOL SetCellWidth(uint nCellWidth);

/**
		Get matrix internal data type
	Parameters:
		None.
	Return:
		Internal data type of matrix
	SeeAlso:
		MatrixLayer::SetInternalData
	Example:
		MatrixLayer mm("Matrix1");
		mm.SetInternalData(FSI_BYTE);
		ASSERT( mm.GetInternalData() == FSI_BYTE );
	
*/
	UINT GetInternalData();

/**
		Set all matrix objects in layer to FSI_DOUBLE, FSI_BYTE and etc. 
	Remarks:
		If matrix is already the correct internal type, this function will do nothing. If the matrix only has image without data,
		bSaveData set to TRUE will convert the image into data so that HasData will become TRUE after this command.
	Parameters:
		iType = FSI_* constants defined in oc_const.h
		bSaveData = whether to backup the data before type change and attempt to restore afterwords. 
		Set this to FALSE if you don't need the data in the matrix to make this operation faster.
	Example:
		MatrixLayer mm("Matrix1");
		mm.SetInternalData(FSI_USHORT);	// make	Matrix1 ushort and keep its data and show as data
	SeeAlso:
		MatrixLayer::GetInternalData
*/
	BOOL SetInternalData(UINT iType, BOOL bSaveData=TRUE, BOOL bSetViewData = TRUE);

/**
		Set the number of columns in a MatixLayer.
	Example: 
		MatrixLayer ml("Matrix1");
		ml.SetNumCols(50);
	Parameters:
		nCols=Input number of columns to set
	Return: 
		Returns TRUE on successful exit and FALSE on failure.
	SeeAlso:
		Datasheet::GetNumRows, Datasheet::SetNumRows, Datasheet::GetNumCols
*/
	BOOL SetNumCols(uint nCols); // Set the number of columns in a MatixLayer.

#if _OC_VER > 0x0703
	/**
		Create a copy of the matrix window matlayerSource and attaches it to the object.
	Argument: 
		matlayerSource		= the matrix whose copy is made
		nOption				= enum {
								CREATE_TEMP = 0,		// it will be destroyed when destroying the object (when it exits the scope) and is created invisible
								CREATE_VISIBLE_SAME,	// visibility is that of the source matrix
								CREATE_VISIBLE,			// create visible
								CREATE_HIDDEN,			// create hidden
								};
		DWORD dwCtrl = enum {
								DCTRL_COPY_DATA						= 0x00000001, // copy data from source worksheet
								DCTRL_COPY_IN_OPERATIONS			= 0x00000002, // copy operations that have MatrixLayer as output (incoimng operations)
								DCTRL_COPY_OUT_OPERATIONS			= 0x00000004, // copy operations that have MatrixLayer as input (outgoing operations)
								DCTRL_COPY_DEFAULT = DCTRL_COPY_DATA | DCTRL_COPY_OUT_OPERATIONS,
							}
 	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	matrix_copy()
		{
			MatrixLayer	mat;
			MatrixLayer	matSource("Matrix1");
			
			int			nOption = CREATE_VISIBLE_SAME;
			BOOL		bRet = mat.CreateCopy(matSource, nOption);
			
			out_int("bRet = ", bRet);
		}
	SeeAlso:
		DataSheet::Create, Worksheet::CreateCopy
	*/
	BOOL	CreateCopy(MatrixLayer &matlayerSource, int nOption = CREATE_VISIBLE_SAME, DWORD dwCtrl = DCTRL_COPY_DEFAULT);
#else //_OC_VER > 0x0703
/**
		Create a copy of the matrix window matlayerSource and attaches it to the object.
	Argument: 
		matlayerSource		= the matrix whose copy is made
		nOption				= enum {
								CREATE_TEMP = 0,		// it will be destroyed when destroying the object (when it exits the scope) and is created invisible
								CREATE_VISIBLE_SAME,	// visibility is that of the source matrix
								CREATE_VISIBLE,			// create visible
								CREATE_HIDDEN,			// create hidden
								};
 
 
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	matrix_copy()
		{
			MatrixLayer	mat;
			MatrixLayer	matSource("Matrix1");
			
			int			nOption = CREATE_VISIBLE_SAME;
			BOOL		bRet = mat.CreateCopy(matSource, nOption);
			
			out_int("bRet = ", bRet);
		}
	SeeAlso:
		DataSheet::Create, Worksheet::CreateCopy
*/
	BOOL CreateCopy(MatrixLayer &matlayerSource, int nOption = CREATE_VISIBLE_SAME);
#endif //_OC_VER > 0x0703

/**
		Check if MatrixLayer has associated image to be used in View Image mode.
	Return:
		TRUE if matrix has image
	Example:
		MatrixLayer mm("Matrix1");
		if (mm.HasImage())
			printf("Matrix has image\n");
		else
			printf("Matrix has NO image\n");
	SeeAlso:
		MatrixLayer::HasData
*/
	BOOL HasImage();

	
/**
		Check if MatrixLayer has data that represent the image one sees in the  View Image mode. 
		An image can be loaded into a MatrixLayer and viewed as an image, and the associated matrix data
		is not constructed until needed, to minimize memory consumption.
	Return:
		TRUE if matrix has data, return FALSE if matrix only has image but data not yet created
	Example:
		MatrixLayer mm("Matrix1");
		if (mm.HasData())
			printf("Matrix has Data\n");
		else
			printf("Matrix has NO Data\n");
	SeeAlso:
		MatrixLayer::HasImage
*/
	BOOL HasData();

	
/**
	Parameters:
		bSetViewImage = TRUE to set matrix as View Image mode, FALSE to set matrix as View Data mode
	Return:
		TRUE if success; otherwise FALSE
	Example:
		MatrixLayer mm("Matrix1");
		Matrix MM("Matrix1");
		MM.SetSize(32,32);
		MM = 20;
		mm.SetViewImage(TRUE);
*/
	BOOL SetViewImage(BOOL bSetViewImage = TRUE);

#ifdef	ORIGIN_COM_SUPPORT
/**
		It puts the data from an SPC blob into the matrix
	Remarks:
		This function is available only for OriginPro versions, or with a special COM enabled license
	Argument: 
		varBlob = the VARIANT blob containing the data
	Return: 
		TRUE for success; otherwise FALSE
	Example: 
		void	blob_to_matrix()
		{
			Object			oSpcIO;
			string			strPathName = "E:\\myfile.spc";
			
			oSpcIO = CreateObject("GSpcIOLib.GSpcIO");	// the GRAMS object used for importing the SPC file
		
			oSpcIO.OpenFile(strPathName);
			
			_VARIANT		varBlob;
			
			varBlob = oSpcIO.SaveBlob();
			
			MatrixLayer		mm("Matrix1");
			BOOL			bRet = mm.PutSPCBlob(varBlob);
			
			out_int("Return value from PutSPCBlob is ", bRet);
		}
*/
	BOOL	PutSPCBlob(_VARIANT &varBlob);
#endif //#ifdef	ORIGIN_COM_SUPPORT

	
/**
		Opens a new matrix from the supplied OGM file and attaches it to the object.
	Parameters: 
		lpcszTemplate = the OGM file name
		nOption	= enum {
						CREATE_TEMP = 0,		// it will be destroyed when destroying the object (when it exits the scope) and is created invisible
						CREATE_VISIBLE_SAME,	// visibility is that which is stored in the OGM file
						CREATE_VISIBLE,			// create visible
						CREATE_HIDDEN,			// create hidden
						};
 
	Return:
		TRUE for success; otherwise FALSE.
	Example:
		void	matrix_open()
		{
			MatrixLayer	mat;
			string		strOGM = "c:\\c\\vc32\\junk1.ogm";
			
			int			nOption = CREATE_VISIBLE_SAME;
			BOOL		bRet = mat.Open(strOGM, nOption);
			
			out_int("bRet = ", bRet);
		}
*/
	BOOL	Open(LPCSTR lpcszFileName, int nOption = CREATE_VISIBLE);

	/**#
	Copies part of image from another matrix layer
	Argument:
		SrcLayer = source matrix layer
		left, top, right, bottom = rectangle in the source matrix layer to copy image from
	Return:
		TRUE if success; otherwise FALSE
	Example:
		void run_CopyImage()
		{
			MatrixLayer		ml("Matrix1");	//Assume Matrix1 contain data
			if( !ml )
				return;	
			
			MatrixPage 		mpCpy;	
			mpCpy.Create("origin.otm");
			MatrixLayer		mlCpy = mpCpy.Layers();		
			
			if(mlCpy)
			{		
				BOOL bOK = mlCpy.CopyImage(ml, 0, 0, ml.GetNumCols(), ml.GetNumRows() , FALSE);		
				if(mlCpy.HasImage())
				{
					mlCpy.SetViewImage(TRUE);
				}
			}	
		}
	*/
	BOOL					  CopyImage(MatrixLayer SrcLayer, int left, int top, int right, int bottom, BOOL bUndo = TRUE);
	
	/**#
	Copies part of image from another matrix layer
	Argument:
		SrcLayer = source matrix layer
		grObject = closed graphic object (polygon, rectangle or circle)
	Return:
		TRUE if success; otherwise FALSE

	Remarks:
		It copies only the part of image in source matrix layer that is within grObject
	*/
	BOOL					  CopyImage(MatrixLayer SrcLayer, GraphObject grObject, BOOL bUndo = TRUE);

	/**#
	*/
	MatrixObject			  MatrixObjects(int ObjIndex = -1);
	/**#
	*/
	Collection<MatrixObject>  MatrixObjects;

};

/** >Internal Origin Objects
		The Column class provides methods and properties common to Origin worksheet
		columns. An Origin C Worksheet object has a collection of Column objects each
		of which in turn holds a Dataset. A Column object is primarily used for style
		related control of the data in the associated Dataset. An Origin C Column object
		is a wrapper object that is a reference to an internal Origin column object.
		Wrapper objects do not actually exist in Origin and merely refer to the internal
		Origin object. Consequently, multiple Origin C wrapper objects can refer to the
		same internal Origin object. The Column class is derived from the DataObject,
		DataObjectBase, and OriginObject classes from which it inherits methods and
		properties.
	Example:
		Worksheet wks = Project.ActiveLayer();
		if(!wks)
			return;
		int ii = 1;
		foreach(Column cc in wks.Columns)
		{
			string strLabel;
			if(cc.GetLabel(strLabel) > 0)
				printf("Column(%d) has label = %s\n",ii++, strLabel);
		}

*/
class Column : public DataObject
{
public:
/**
Remarks:
		Default constructor for Column class
Parameters: 
		None
Return: 
		None
Example:
	void test_Column_Column1()
	{
			Column colMy;
			string strName;
			colMy.Attach("Data1", 0);
			if(colMy.IsValid())
			{
				colMy.GetName(strName);
				out_str(strName);
			}			
	}
*/
	Column();
/**

Remarks:
		Constructor that creates an Column object when given a column number in a worksheet.
Parameters:
		lpcszWksName = name of the worksheet 
		iColNum      = column number in the worksheet
		
Return: 
		None
		
Example:
	void test_Column_Column2()
	{
			Column colMy("Data1", 0);
			string strName;
			if(colMy.IsValid())
			{
				colMy.GetName(strName);
				out_str(strName);
			}
	}
*/
	Column(LPCTSTR lpcszWksName, UINT iColNum);	
/**
Remarks:
		Constructor that creates an Column object when given a column number in a worksheet. 
Parameters:
		wks = worksheet object
		iColNum = column number in the worksheet
		
Return: 
		None

Example:
	void test_Column_Column3()
	{
			Worksheet wksMy("Data1");
			Column colMy(wksMy, 0);
			string strName;
			if(colMy.IsValid())
			{
				colMy.GetName(strName);
				out_str(strName);
			}
	}
*/
	Column(Worksheet &wks, UINT iColNum);	
/**
	Since a Column is a wrapper of the actual internal column inside Origin, you can make copies of this wrapper and
	both the original and the copy will both be refering to the same internal object.
Parameters:
		colOriginal = column object
		
Return: 
		None

Example:
	void test_Column_Column4()
	{
			Column colOriginal("Data1", 0);
			Column colNew(colOriginal);
			string strName;
			if(colMy.IsValid())
			{
				colMy.GetName(strName);
				out_str(strName);
			}
	}
*/
	Column(DataObject& colOriginal);
	
/**
	Attach the column object to a Worksheet column by worksheet name
Parameters:
	lpcszWksName = name of the worksheet
	iColNum 	 = column number in the worksheet at which the column should be attached
Return:
	True for success; false otherwise
Example:
	void test_Column_Attach1()
	{
		Column colMy;
		if (colMy.Attach("Data1", 0))
			out_str("A column is attached to the first column of worksheet Data1.");
		else
			out_str("Attach error!");
	}
*/
	BOOL Attach(LPCSTR lpcszWksName, UINT iColNum);	
/**
	Attach the column object to a Worksheet column using a worksheet object
Parameters:	
		wks = worksheet object
		iColNum = column number in the worksheet at which the column should be attached.
Return:
	True for success; false otherwise
Example:
	void test_Column_Attach2()
	{
		Worksheet wksMy("Data1");
		Column colMy;
		if (colMy.Attach(wksMy, 0))
			out_str("A column is attached to the first column of worksheet Data1.");
		else
			out_str("Attach error!");
	}
*/
	BOOL Attach(Worksheet &wks, UINT iColNum);	
	
/**
	Remarks: 
		Gets the display mode used for numbers in a column
	Parameters: 
		None
	Return: 
		The ColDigitMode for enum {DIGITS_FREE_FORM, DIGITS_DECIMAL, DIGITS_SIGNIFICANT}
	Example: 
		Worksheet wks("data1");
		int iDigitMode = wks.Columns(0).GetDigitMode();
	SeeAlso: 
		Column::SetDigitMode, Column::GetDigits, Column::SetDigits
*/
	int GetDigitMode(); 

/**
	Remarks: 
		Sets the display mode used for numbers in a column
	Parameters: 
		iDigitMode = The ColDigitMode - enum {
			DIGITS_FREE_FORM, 
			DIGITS_DECIMAL, 
			DIGITS_SIGNIFICANT 
			}
	Return: 
		TRUE, set the digitmode successfully, FALSE, unsuccessful.
	Example: 
		Worksheet wks("data1");
		BOOL b = wks.Columns(0).SetDigitMode(DIGITS_FREE_FORM);
	SeeAlso: 
		Column::GetDigitMode, Column::GetDigits, Column::SetDigits
*/
	BOOL SetDigitMode(int iDigitMode); 

/**
	Remarks: 
		Get the digit value for numeric display mode
	Argument: 
		None
	Return: 
		The Digits value, -1 for error
	Example: 
		Worksheet wks("data1");
		int iDigits = wks.Columns(0).GetDigits();
	SeeAlso: 
		Column::GetDigitMode, Column::SetDigitMode, Column::SetDigits
*/
	int GetDigits(); 

/**
	Remarks: 
		Set the digit value for the numeric display mode
	Parameters: 
		iDigits = the number of digits
	Return: 
		TRUE, set the digits successfully, FALSE, unsuccessful.
	Example: 
		Worksheet wks("data1");
		BOOL b = wks.Columns(0).SetDigits(10);
	SeeAlso: 
		Column::GetDigitMode, Column::SetDigitMode, Column::GetDigits
*/
	BOOL SetDigits(int iDigits); 

/**
	Remarks: 
		Get the column format
	Parameters: 
		None
	Return:
		The column format, enum {
			OKCOLTYPE_NUMERIC, 
			OKCOLTYPE_TEXT, 
			OKCOLTYPE_TIME, 
			OKCOLTYPE_DATE, 
			OKCOLTYPE_MONTH, 
			OKCOLTYPE_WEEKDAY, 
			OKCOLTYPE_TEXT_NUMERIC};
	Example: 
		Worksheet wks("data1");
		int iFormat = wks.Columns(0).GetFormat();
	SeeAlso:
		Column::SetFormat
*/
	int GetFormat();

/**
	Remarks: 
		Set the column format
	Parameters: 
		iFormat = the column format,enum {
			OKCOLTYPE_NUMERIC, 
			OKCOLTYPE_TEXT, 
			OKCOLTYPE_TIME, 
			OKCOLTYPE_DATE, 
			OKCOLTYPE_MONTH, 
			OKCOLTYPE_WEEKDAY, 
			OKCOLTYPE_TEXT_NUMERIC}
	Return: 
		TRUE, set the coloumn format successfully, FALSE, unsuccessful.
	Example: 
		Worksheet wks("data1");
		BOOL b = wks.Columns(0).SetFormat(OKCOLTYPE_NUMERIC);
	SeeAlso:
		Column::GetFormat
*/
	BOOL SetFormat(int iFormat); 

/**
	Remarks: 
		Get the column justification
	Parameters: 
		None
	Return: 
		Justification for enum{ 
								COL_JUSTIFY_LEFT = 1, 
								COL_JUSTIFY_RIGHT, 
								COL_JUSTIFY_CENTER } 
	Example: 
		Worksheet wks("data1");
		int iJustify = wks.Columns(0).GetJustify();
	SeeAlso:
		Column::SetJustify
*/
	int GetJustify(); 

/**
	Remarks: 
		Set the column justification
	Parameters: 
		iJustify = enum{ 
						COL_JUSTIFY_LEFT = 1, 
						COL_JUSTIFY_RIGHT, 
						COL_JUSTIFY_CENTER }
	Return: 
		TRUE, set the column justify successfully, FALSE, unsuccessful.
	Example: 
		Worksheet wks("data1");
		BOOL bResult = wks.Columns(0).SetJustify(COL_JUSTIFY_CENTER);
	SeeAlso:
		Column::GetJustify
*/
	BOOL SetJustify(int iJustify); 

/**
	Remarks: 
		Get the column label
	Parameters: 
		strLabel = reference to the string that will receive the label
	Return: 
		length of the label, 0 if none, -1 if error
	Example:	
		Worksheet wks("data1");
		string strLabel;
		int i = wks.Columns(0).GetLabel(strLabel);
	SeeAlso:
		Column::SetLabel
*/
	int	GetLabel(string &strLabel); 
/**
	Remarks: 
		Get the column label if exist
	Example:	
		Worksheet wks("data1");
		string str = wks.Columns(0).GetLabel();
		if(str.IsEmpty())
			out_str("Column 1 does not have label");
*/
	string	GetLabel(); 

/**
	Remarks: 
		Set the column label
	Parameters: 
		lpcstrLabel = pointer to a null-terminated string to be used as the 
		column label
	Return: 
		TRUE, set the column label successfully. FALSE, unsuccessful.
		The error may simply be that Origin modified the name which is now in 
		strColLabel. 
		Any more disastrous error should set the string to null.
	Example:	
		Worksheet wks("data1");
		string strLabel = "Elapsed\n(in sec.)";
		BOOL b = wks.Columns(0).SetLabel(strLabel);
	SeeAlso:
		Column::GetLabel
*/
	BOOL SetLabel(LPCSTR lpcstrLabel); 

/**
	Remarks: 
		Get the column name
	Parameters: 
		strName = reference to the string that will receive the name
	Return: 
		length of the name, 0 if none, -1 if error
	Example:	
		Worksheet wks("data1");
		string strName;
		int i = wks.Columns(0).GetName(strName);
	SeeAlso:
		Column::SetName
*/
	int GetName(string &strName); 

#if  _OC_VER < 0x0750
/**
Example:	
	Worksheet wks("data1");
	foreach(Column cc in wks.Columns)
		printf("Column name is %s\n", cc.GetName());
*/
	string GetName();
#endif //_OC_VER < 0x0750

/**
	Remarks: 
		Set the column name
	Parameters: 
		lpcstrName = pointer to a null-terminated string to be used as the column 
		name
	Return: 
		TRUE, set the column name successfully, FALSE, unsuccessful.
		the error may simply be that Origin modified the name which is now in 
		strColName any more disastrous error should set the string to null
	Example:	
		Worksheet wks("data1");
		string strName = "A";
		BOOL b = wks.Columns(0).SetName(strName);
	SeeAlso:
		Column::GetName
*/
	BOOL SetName(LPCSTR lpcstrName); 

/**
	Remarks: 
		Get the SubFormat used by a column - the list of available subformats 
		varies for each column type. These are 'Formats' in the Worksheet Column 
		dialog
	Parameters: 
		None
	Return: 
		The subformat as an integer enumerated from 1
	Example:	
		Worksheet wks("data1");
		int iSubFormat = wks.Columns(0).GetSubFormat();
	SeeAlso:
		Column::SetSubFormat
*/
	int GetSubFormat(); 

/**
	Remarks: 
		Set the SubFormat used by a column
	Parameters: 
		iSubFormat = The integer (enumerated from one) of the subformat for the current column 
		type
	Return: 
		TRUE, set the column subformat successfully, FALSE, unsuccessful.
		an error occurs when a subformat is out of range for example, any value 
		for a Text column, a value greater than 21 for a Date column
	Example:	
		Worksheet wks("data1");
		// We will assume the first column is a Date column
		BOOL b = wks.Columns(0).SetSubFormat(13); // The 13th subformat is yyMMdd HH:mm
	SeeAlso: 
		Column::GetSubFormat
*/
	BOOL SetSubFormat(int iSubFormat); 

/**
	Remarks: 
		Get the text width of a column - this is the maximum number of characters
		that can be stored in a column. The default is 25, the maximum is 256.
		This is different than the display column width.
	Parameters: 
		None
	Return: 
		The current Text Width of the column
	Example:	
		Worksheet wks("data1");
		int iTextMaxLen = wks.Columns(0).GetTextMaxLen();
	SeeAlso:
		Column::SetTextMaxLen
*/
	int GetTextMaxLen(); 

/**
	Remarks: 
		Set the text width of a column
	Parameters: 
		iTWidth = the text width in characters
	Return: 
		TRUE, set the column width successfully, FALSE, unsuccessful.
	Example:	
		Worksheet wks("data1");
		BOOL b = wks.Columns(0).SetTextMaxLen(50);
	SeeAlso: 
		Column::GetTextMaxLen
*/
	BOOL SetTextMaxLen(int iTWidth); // Set the text width of the column

/**
	Remarks: 
		Get the type designator of a column (Y, disregard, Y Err, X, Label, Z or X Err)
	Parameters: 
		None
	Return: 
		The column type enum {OKDATAOBJ_DESIGNATION_Y = 0,
							  OKDATAOBJ_DESIGNATION_NONE,
							  OKDATAOBJ_DESIGNATION_ERROR, 
			                  OKDATAOBJ_DESIGNATION_X,  
			                  OKDATAOBJ_DESIGNATION_L,
			                  OKDATAOBJ_DESIGNATION_Z,
			                  OKDATAOBJ_DESIGNATION_X_ERROR};
	Example:	
		Worksheet wks("data1");
		int iType = wks.Columns(0).GetType();
*/
	int GetType(); 

/**
	Remarks: 
		Set the type designator of a column (Y, disregard, Y Err, X, Label, Z or X Err)
	Parameters: 
		iType = the column type enum {OKDATAOBJ_DESIGNATION_Y = 0,
									  OKDATAOBJ_DESIGNATION_NONE,
									  OKDATAOBJ_DESIGNATION_ERROR, 
									  OKDATAOBJ_DESIGNATION_X,  
									  OKDATAOBJ_DESIGNATION_L,
									  OKDATAOBJ_DESIGNATION_Z,
									  OKDATAOBJ_DESIGNATION_X_ERROR};
	Return: 
		TRUE, set the column type  successfully, FALSE, unsuccessful.
	Example:	
		Worksheet wks("data1");
		BOOL b = wks.Columns(0).SetType(OKDATAOBJ_DESIGNATION_X);
	SeeAlso: 
		Column::GetType
*/
	BOOL SetType(int iType); 

/**
	Remarks: 
		Get the display width of a column in number of characters
	Parameters: 
		None
	Return: 
		The width in characters (approximate), exact only if the font is fixed width
	Example:	
		Worksheet wks("data1");
		out_int("col(1) width=", wks.Columns(0).GetWidth());
	SeeAlso:
		Column::SetWidth
*/
	int GetWidth(); 

/**
	Remarks: 
		Set the display column width of a column in number of characters
	Parameters: 
		iWidth = the width in characters (approximate), exact only if the font is fixed width
		use -1 to set the width to the largest cell in the column
	Return: 
		TRUE, set the digitmode successfully, FALSE, unsuccessful.
	Example:	
		Worksheet wks("data1");
		int iWidth = 3;
		BOOL b = wks.Columns(0).SetWidth(iWidth);
		out_int("col(1) width set to be very small, ", wks.Columns(0).GetWidth());
		MessageBox(NULL, "Wait...");
		wks.Columns(0).SetWidth();		
		out_str("col(1) width reset to show all data");		
	SeeAlso: 
		Column::GetWidth
*/
	BOOL SetWidth(int iWidth = -1); 

/** 
	Remarks:
		Set the column internal data types (This is a characteristic of numeric 
		column format). If you are not sure the column is in numeric format, call SetFormat
		first.
	Parameters:
		iType = the internal type of a column enum{	
													FSI_DOUBLE = 0,	
													FSI_REAL, 
													FSI_SHORT, 
													FSI_LONG };
	Return:
		TRUE, set the column internal type successfully, FALSE, unsuccessful.
	Example:
		// this example set all columns in current worksheet to short integers
		Worksheet wks = Project.ActiveLayer();
		if(!wks)
			return;
		foreach(Column col in wks.Columns)
		{
			col.SetFormat(OKCOLTYPE_NUMERIC);	
			if(!col.SetInternalData(FSI_SHORT))
				out_str("set short int error");
		}
			
	SeeAlso:
		Column::GetInternalData
*/	
	BOOL SetInternalData(UINT iType);

/**
	Remarks:
		Get the column internal data types
	Parameters:
		No arguments
	Return:
		the internal type of a column:enum{	
											FSI_DOUBLE = 0,	
											FSI_REAL, 
											FSI_SHORT, 
											FSI_LONG };
	Example:
		Worksheet wks("data1");
		int iInternalData = wks.Columns("0").GetInternalData();
	SeeAlso:
		Column::SetInternalData
*/
	UINT GetInternalData();

/**
		Get the Dataset associated with a column as a vectorbase class reference
		for using in functions that expect a vectorbase class. Using vectorbase will
		allow the same function to be used on any data type
	Return:
		a reference to the vectorbase class object for the associated dataset in the column
	Example:
		static void show_sum(vectorbase& vtemp)
		{
			if(vtemp.GetSize() < 1)
				return;
			double vv;
			vtemp.Sum(vv);
			printf("sum of vector is %f\n", vv);
		}
		
		void tt()
		{
				Worksheet wks = Project.ActiveLayer();
				if( wks )
				{
					int ii = 1;
					foreach(Column cc in wks.Columns)
					{
						printf("Col(%d)'s name is %s\n",ii++, cc.GetName());
						vectorbase &aa=cc.GetDataObject();
						show_sum(aa);
					}
				}
		}
*/
	vectorbase& GetDataObject();

/**
		Set the ending row indices for this column. Index values are 0 based
		offsets so use SetUpperBound(-1) to dispaly no rows.
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		int nR2 = 25;		
		colObj.SetUpperBound(nR2);
	Parameters:
		iR2=Input ending row index (0 based)
	Return:
		Returns TRUE on success and FALSE on failure.
*/
	BOOL SetUpperBound( int iR2 );

/**
		Set the begining row indices for this column.
	Parameters:
		iR1=Input begining row index (0 based)
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		
		int nR1 = 5;		
		colObj.SetLowerBound(nR1);
*/
	BOOL SetLowerBound( int iR1 );

/**
	Set the column formula
	Parameters:
		lpcszFormula = the formula
	Return:
		Returns TRUE on success and FALSE on failure.
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		colObj.SetFormula("col(B)*2");
*/
	BOOL	SetFormula(LPCSTR lpcszFormula);

/**
	Get column formula from column
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		string strFormula = colObj.GetFormula();
*/		
	string	GetFormula();

/**
	Triggers the execution of column formula
	Parameters:
		nBegin = begining row index (0 based);
		nEnd = end row index (0 based);
		bUndo = if this parameter is TRUE method can be undone
	Remarks:
		For Origin version 8 or later:
		If nBegin and nEnd are negative formula is applied to range calculated 
		based on the upper and lower bounds of columns the formula refers to
		If column formula does not refer to other columns it is applied
		to all rows in the column
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		colObj.SetFormula("col(B)*2");
		colObj.ExecuteFormula(0, 10);
*/
	BOOL	ExecuteFormula(int nBegin = -1, int nEnd = -1, BOOL bUndo = TRUE);

/**
		Get the number of rows in a Column.
	Example: 
		Worksheet wks("Data1");
		Column col(wks,0);
		int nRows = col.GetNumRows();
	Return: 
		Returns the number of rows in a Column.
	SeeAlso: 
		Column::SetNumRows
*/	
	int GetNumRows(); // Get the number of rows in a Column.

/**
		Set the number of rows in a Column.
	Example: 
		Worksheet wks("Data1");
		Column col(wks,0);
		col.SetNumRows(50);  // set column size to 50
		Dataset ds(wks,0);   // attach it to dataset ds
		ds.SetSize(20);      // Set data range to 20;
		int ii=ds.GetSize();
		ASSERT( ii == 20);
		ii = col.GetNumRows();
		ASSERT( ii == 50);
		col.SetNumRows(10);  // Set column size to 10, as it is small than data range, so data rang changed here too
		wks.UpdateOrigin();  // update change to worksheet in origin
		ds.Update(TRUE);	 // update Origin C object ds
		ii=ds.GetSize();  
		ASSERT(ii==10);    
		
	Parameters:
		nRows=Input number of rows to set, nRows must larger than 0.
	Return:
		Returns TRUE on successful exit and FALSE on failure.
	SeeAlso: 
		Column::GetNumRows, Dataset::SetSize
*/
	BOOL SetNumRows(int nRows); // Set the number of rows in a Column.

#if _OC_VER > 0x0703	
/**#
*/
	BOOL	SetFormulaAutoUpdate(BOOL bOn = TRUE);
/**#
*/
	BOOL	SetFormulaAutoUpdate(int nBegin, int nEnd, BOOL bOn = TRUE);
/**#
*/
	BOOL	IsFormulaAutoUpdate(int* pBegin = NULL, int* pEnd = NULL);

/**#
		Get the begining row index for this column. Index values are 0 - offset.
	Parameters:
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		
		int nR1 = 5;		
		colObj.SetLowerBound(nR1);
		ASSERT( nR1 == colObj.GetLowerBound() );
	Return:
		Returns begining row index.
	SeeAlso: 
		Column::GetUpperBound, Dataset::GetUpperBound, Dataset::GetLowerBound
*/
 	int		GetLowerBound();
/**#
		Get the ending row index for this column. Index values are 0 - offset.
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		int nR2 = 25;		
		colObj.SetUpperBound(nR2);
		ASSERT( nR2 == colObj.GetUpperBound() );
	Parameters:
	Return:
		Returns ending row index.
	SeeAlso: 
		Column::GetLowerBound, Dataset::GetUpperBound, Dataset::GetLowerBound
*/
 	int		GetUpperBound();
	
/**
		Get the index of the column in the same worksheet that is farthest away
		from this column and that is dependent on this column.

		Regarding modifiers: The index returned should be such that, if a column were to be inserted
		at that location in the worksheet, the distance between any column from the same worksheet that
		is plotted in some graph and any of its modifier columns used in conjunction with the same dataplot
		in that graph should not change.
		This could be rephrased like this: if a column plotted in some graph and all its modifiers associated
		with the same dataplot are considered as one range, then the index returned
		1) should not break any such range in the worksheet, and
		2) it should be to the right of the range containing the column through which the method call is made.
	Example:
		Worksheet wks = Project.ActiveLayer();
		Column colObj(wks, 0);
		int nOffset = colObj.GetLastDependentColumnOffset(DOO_ERRORBAR);
	Parameters:
		dwCntrl is a combination of the following cotrol bits:
					typedef enum
					{
						DOO_LEFT		= 0x10000000,		// left offset
						DOO_ERRORBAR	= 0x00010000,		// errorbars
						DOO_MODIFIER	= 0x00020000,		// modifiers
						DOO_TYPE_MASK	= 0x000F0000,		// type mask
					}	DEPENDENTOBJOFFSET;
	Return:
		Returns last dependent column offset if success, -1 if error.
*/
	int		GetLastDependentColumnOffset(DWORD dwCntrl);
	
	/**
	Return:
		TRUE if column is write protected. FALSE otherwise
	See Also:
		SeeAlso:
		Worksheet::IsWriteProtected(int nBegin = 0; int nEnd = -1)	
	*/
	BOOL	IsWriteProtected();
#endif //_OC_VER > 0x0703
};



/** >Internal Origin Objects
		The MatrixObject class provides methods and properties common to internal Origin matrix
		objects. An Origin C MatrixObject is primarily used for style related control of the data in
		the internal Origin matrix while the Origin C Matrix class is used to access the data in the
		matrix. Thus, the MatrixObject class has the same relationship with the Matrix class as the Column
		class has with the Dataset class. That is, an internal Origin matrix object (accessed by the
		MatrixObject class) holds a matrix data set (accessed by the Matrix class) just as a worksheet
		column (accessed by the Column class) holds a data set (accessed by the Dataset class). The
		data values displayed in the cells of a matrix are typically referred to as Z values whose
		associated X values are linearly mapped to the columns of the matrix and whose associated Y
		values are linearly mapped to the rows of the matrix. A MatrixLayer holds a collection of
		MatrixObjects even though in most cases there is only one MatrixObject 	per MatrixLayer. An
		Origin C MatrixObject is a wrapper object that is a reference to an internal Origin matrix
		object. Wrapper objects do not actually exist in Origin and merely refer to the internal
		Origin object. Consequently, multiple Origin C wrapper objects can refer to the same internal
		Origin object. The MatrixObject class is derived from the DataObject, DataObjectBase, and
		OriginObject classes from which it inherits methods and properties.
	Example:
		MatrixLayer ml = Project.ActiveLayer();
		if(ml)
			foreach(MatrixObject mo in ml.MatrixObjects)
				printf( "%s is %d x %d\n", ml.GetPage().GetName(), mo.GetNumRows(), mo.GetNumCols());
*/
class MatrixObject : public DataObject
{
public:
/**
	Remarks:
			Default constructor.
	Parameters:
	Return: 
	Example:
		MatrixObject moMy();;

*/
	MatrixObject();
/**

	Remarks:
			This constructor initializes a user-defined matrix object with one of the objects in the ActiveLayer, 
			if the ActiveLayer is a MatrixLayer. 
	Parameters:
			mLayer = MatrixLayer object which contains one or more matrix objects
			nObjectIndex = Index of one of the matrix layers in the MatrixLayer 
						   Presently only zero is supported, as a MatrixLayer can have only one object
	Return: 
	Example:
		MatrixLayer ml = Project.ActiveLayer();
		MatrixObject moMy(ml, 0);
		if(moMy) out_str("It is a mtarix!");
		else out_str("It is not a mtarix!");

	
*/
	MatrixObject(MatrixLayer & mLayer, UINT nObjectIndex);	

/**

	Remarks:
			This constructor initializes a user-defined matrix object with one of the objects in the MatrixLayer.
	Parameters:
			MatrixName = name of the MatrixLayer object 
			nObjectIndex = Index of one of the matrix objects in the MatrixLayer object. 
						   Presently only zero is supported, as a MatrixLayer can have only one object
	Return: 
	Example:
		MatrixObject moMy("matrix1", 0);
		if(moMy) out_str("matrix1 exists");
		else out_str("matrix1 doesn't exist!");
	
*/
	MatrixObject(LPCTSTR MatrixName, UINT nObjectIndex);	
/**
	Remarks: 
			Copy constructor. 
	Parameters:
			OriginalObj = an object of DataObject which is a base class for MatrixObject
	Return:
	Example:
			MatrixObject moMy1("matrix1", 0);
			MatrixObject moMy2(moMy1);
	

*/
	MatrixObject(DataObject& OriginalObj);	// Copy constructor -- refers to the same internal object as colOriginal

	/**#
	Example:
		void run_Attach()
		{
			MatrixLayer  ml("Matrix1");	
			MatrixObject mObj;
			mObj.Attach(ml,0);
		}
	*/
	BOOL		Attach(MatrixLayer &mlayer, UINT nObjectIndex);    
	/**#
	Example:
		void run_Attach()
		{
			MatrixObject mObj;
			mObj.Attach("Matrix1",0);
		}
	*/
	BOOL		Attach(LPCTSTR MatrixName, UINT nObjectIndex);  
	/**
	Remarks: 
		Set the Matrix data format. Only OKCOLTYPE_NUMERIC is supported at present time.
	Parameters: 
		iFormat = the column format just can be OKCOLTYPE_NUMERIC 
			
	Return: 
		TRUE, set the Matrix format successfully, FALSE, unsuccessful.
	Example:
		void test_MatrixObject_SetFormat()
		{
				MatrixObject moMy("Matrix1", 0);
				if (moMy.SetFormat(OKCOLTYPE_NUMERIC))
					out_str("Format setting successfully!");
				else
					out_str("Format setting error!");
		}
	SeeAlso:
		Column::SetFormat	
	*/
	BOOL		SetFormat(int nFormat);  // only supports OKCOLTYPE_NUMERIC                        
	
	/**
		Remarks: Gets the data format of the matrix.
		
		Parameters: 
		Return:
					the integer code representing the format
		Example:
			void test_MatrixObject_GetFormat()
			{
					MatrixObject moMy("Matrix1", 0);
					if (moMy.SetFormat(OKCOLTYPE_NUMERIC))
						out_str("Format setting successfully!");
					else
						out_str("Format setting error!");
					printf("the data format in the matrix is %d", moMy.GetFormat()); 	
			}
				
	
	*/
	
	int			GetFormat();
	
	/**
		Remarks: Gets the digit mode of the matrix. There are 3 modes:
				 0 for default decimal digits
				 1 for decimal digits
				 2 for significant digits
		
		Parameters: 
					NONE
		RETURN:
					the integer code representing the format
		Example:
			void test_MatrixObject_GetDigitMode()
			{
					MatrixObject moMy("Matrix1", 0);
					printf("the data format in the matrix is %d", moMy.GetDigitMode()); 	
			}
				
	
	*/
	int			GetDigitMode();   
	
	/**
		Remarks: Sets the digit mode of the matrix. There are 3 modes:
				 0 for default decimal digits
				 1 for decimal digits
				 2 for significant digits
		
		Parameters: 
					nDigitMode = 0 for default decimal digits
								 1 for decimal digits
								 2 for significant digits
		RETURN:
					TRUE if successful; otherwise false
		Example:
			void test_MatrixObject_SetDigitMode()
			{
					MatrixObject moMy("Matrix1", 0);
					if (moMy.SetDigitMode(0))
						out_str("Digit mode is set successfully!");
					else
						out_str("Digit mode is not set successfully!");
			}
				
	
	*/
	
	BOOL		SetDigitMode(int nDigitMode);
	
	/**
		Remarks: Gets the number of digits of a particular digit mode of the matrix. 
		
		Parameters: 
					NONE
		RETURN:
				the number of digits set in a particular mode 
		Example:
			void test_MatrixObject_GetDigits()
			{
					MatrixObject moMy("Matrix1", 0);
					if (moMy.SetDigitMode(2))
						out_str("Digit mode is set successfully!");
					else
						out_str("Digit mode is not set successfully!");
					printf("the number of significant digits are %d", moMy.GetDigits());  
			}
				
	
	*/
	
	int			GetDigits();                                      
	
	/**
		Remarks:Sets the number of digits of a particular digit mode of the matrix.  
		
		Parameters: 
					nDigits = number of digits
		RETURN:
				TRUE if successful; FALSE otherwise
		Example:
			void test_MatrixObject_SetDigits()
			{
					MatrixObject moMy("Matrix1", 0);
					if (moMy.SetDigitMode(2)) // 2 -> significant digits
						out_str("Digit mode is set successfully!");
					else
						out_str("Digit mode is not set successfully!");
					moMy.SetDigits(4);
					printf("the number of significant digits are %d", moMy.GetDigits());  
			}
				
	
	*/
	
	BOOL		SetDigits(int nDigits);  
	
	
	/**
		Remarks: Gets the width of a matrix column. 
		
		Parameters: 
					NONE
		RETURN:
				width of the column
		Example:
			void test_MatrixObject_SetDigits()
			{
					MatrixObject moMy("Matrix1", 0);
					printf("the column witdh of Matrix1 is %d", moMy.GetColumnWidth());  
			}
				
		
	*/
	int			GetColumnWidth();
	
	
	/**
		Remarks: Sets the width of a matrix column. 
		
		Parameters: 
					nColWidth = user defined width of the column
		RETURN:
				TRUE if successful; FALSE otherwise
		Example:
			void test_MatrixObject_SetDigits()
			{
					MatrixObject moMy("Matrix1", 0);
					if (moMy.SetColumnWidth(5)) 
						out_str("Coluimn width is set successfully!");
					else
						out_str("Coluimn width is not set successfully!");
					printf("the column witdh of Matrix1 is %d", moMy.GetColumnWidth()); //should be 5  
			}
				
	
	*/
	
	
	BOOL		SetColumnWidth(int nColWidth);
	
	
	/** 
		Remarks:
			Set the Matrix internal data types (This is a characteristic of numeric format)  
		Parameters:
			iType = the internal type of a matrix, can be one of the followings	
					FSI_DOUBLE,
					FSI_REAL,
					FSI_SHORT,
					FSI_LONG,
					FSI_CHAR,
					FSI_BYTE,
					FSI_USHORT,
					FSI_ULONG
			bSaveData = when changing internal data type, we can keep the original data by converting them to the new type
			bSetViewData = TRUE will turn the Matrix into View Data mode after the data type change. FALSE will keep current view mode
			
		Return:
			TRUE, set the Matrix internal type successfully, FALSE, unsuccessful.
		Example:
			MatrixLayer mlay = Project.ActiveLayer();
			if(!mlay) return;
			MatrixObject mobj = mlay.MatrixObjects(0);// Most Matrix in Origin has only one MatrixObject
			if(mobj.SetInternalData(FSI_BYTE,TRUE,FALSE))
				out_str("succefully converted into BYTE matrix");
		SeeAlso:
			MatrixLayer::SetInternalData
	*/
	BOOL		SetInternalData(UINT iType, BOOL bSaveData=TRUE, BOOL bSetViewData = TRUE);
	
	
	/**
			Get internal data type
		Parameters:
		Return:
			Internal data type of matrix
		SeeAlso:
			MatrixObject::SetInternalData
		Example:
			MatrixLayer mlay = Project.ActiveLayer();
			if(!mlay) return;
			MatrixObject mobj = mlay.MatrixObjects(0);// Most MatrixLayer objects in Origin has only one MatrixObject
			if(mobj.SetInternalData(FSI_BYTE,TRUE,FALSE))
			{
				ASSERT(mobj.GetInternalData() == FSI_BYTE);
			}
		
	*/
	int			GetInternalData();
	
	/**
			Gets the number of rows in the matrix.
		Parameters:
		Return:
			number of rows in the matrix
		Example:
			MatrixLayer mlay = Project.ActiveLayer();
			MatrixObject mobj (mlay, 0);
			printf("the number of rows in the selected matrix are %d", mobj.GetNumRows());
						
	*/
	
	int			GetNumRows();  
	
	
	
	/**
		Gets the number of columns in the matrix.
	Parameters:
	Return:
		number of columns in the matrix
	Example:
		MatrixLayer mlay = Project.ActiveLayer();
		MatrixObject mobj (mlay, 0);
		printf("the number of columns in the selected matrix are %d", mobj.GetNumCols());
					
	*/
	int			GetNumCols();
	/**#
	*/
	BOOL		SetDimensions(int nRows, int nCols);
	
	
	/**
	
	   	   Puts the active Matrix window's data into a matrix data object.
 	   Parameters:
	   Return: 
	   	   A matrix containing the data of Matrix window.
	   Example:
	       void eg2_matrix()
		   {
		   	  MatrixLayer mlActive = Project.ActiveLayer();
		   	  if(mlActive == NULL)
				return;
		   	  
		   	  matrix<double> ma(mlActive.MatrixObjects(0).GetDataObject());
		   	  
		   	  matrix<double> mb;
		   	  mb = mlActive.MatrixObjects(0).GetDataObject();
			// ...
		   }
	*/
	matrixbase&  GetDataObject();
	
	/**
			GetImageMask() will get MatrixObject's image mask. The mask of an Origin image is an 
			one byte DIB image. This method will first set the Origin C matrix data object to have 
			same dimensions as the image mask, then, copy all bytes from the mask image to 
			corresponding locations in Origin C matrix data object. If the return is TRUE, the 
			Origin C matrix data object will have the mask image data.
		Parameters:
			mMask = An Origin C matrix data object to retrieve image mask. Its type is matrix<byte>.
		Return:
			TRUE/FALSE. TRUE indicates the mask has been retrieved successfully. Otherwise, return FALSE 
		SeeAlso:
			MatrixObject::SetImageMask
		Example:
			MatrixObject mObj("Matrix1", 0);  //assume Matrix1 exists 
			matrix<byte> mMask;
			if( mObj.GetImageMask(mMask) )
			{
				printf("The mask is retrieved\n");
				printf("Numder of rows are: %d, Number of Cols are: %d\n", mMask.GetNumRows(), mMask.GetNumCols()); 
				
				MatrixObject mObj2("Matrix2", 0);
				mObj2.SetImageMask(mMask);
			}
		
	*/
	BOOL		GetImageMask(matrix<byte>& mMask);
	
	/**
			SetImageMask() method will set MatrixObject's image mask. It creates an one byte DIB from 
			data of the Origin C matrix data object, then, sets that DIB image as image mask for the 
			MatrixObject.
		Parameters:
			mMask = mMask = An Origin C matrix data object to retrieve image mask. Its type is matrix<byte>.
		Return:
			TRUE/FALSE. TRUE indicates the mask has been retrieved successfully. Otherwise, return FALSE 
		SeeAlso:
			MatrixObject::GetImageMask
		Example:
			MatrixObject mObj("Matrix1", 0);  //assume Matrix1 exists 
			matrix<byte> mMask;
			if( mObj.GetImageMask(mMask) )
			{
				printf("The mask is retrieved\n");
				printf("Numder of rows are: %d, Number of Cols are: %d\n", mMask.GetNumRows(), mMask.GetNumCols()); 
				
				MatrixObject mObj2("Matrix2", 0);
				mObj2.SetImageMask(mMask);
			}
		
	*/
	BOOL		SetImageMask(matrix<byte>& mMask);
};


// Origin internal functions
// all function prototypes from here on must be implemented from within the Origin OK DLL

#pragma dll(@OK)

/** >Import Export
		Get SPC main header information from worksheet
	Returns:
		TRUE means the header information is retrieved successfully 
		FALSE indicates the failure in retrieving header information
	Parameters:
		lpcszWksName = worksheet name
		nCol = column index from 0
		SPCMainHeader = SPC main header structure
	Examples:
		string strWks = "Data1";
		Worksheet wks(strWks);
		OSPCMAINHeader spcMainHeader;
		for(int jj = 0; jj< wks.GetNumCols();jj++)
		{
			if( GetSPCMainHeader(strWks, jj, &spcMainHeader) )
			{
				//read information from the main header
				int iNumOfSubs = spcMainHeader.fnsub;
				OSPCSUBHeader spcSubHeader;
				
				for( int ii=0; ii<iNumOfSubs; ii++ )
				{
					if( GetSPCSubHeader(strWks, jj, &spcSubHeader, ii) )
						//read information from the subheader
				}			
			}	
			
			string strLog;
			strLog = GetSPCLog(strWks, jj);
		}
*/
BOOL	GetSPCMainHeader(LPCSTR lpcszWksName, int nCol, OSPCMAINHeader *SPCMainHeader);

/** >Import Export
		Get SPC subheader information from worksheet. 
		This function requires the subheader counter from the main header structure. 
		Therefore, the OSPCMAINHeader information has to be retrieved successfully first to be able to use this function.
	Returns:
		TRUE means the subheader information is retrieved successfully 
		FALSE indicates the failure in retrieving subheader information
	Parameters:
		lpcszWksName = worksheet name
		nCol = column index from 0
		SPCSubHeader = SPC sub header structure
		nIndex = the index of subheader from 0
	Examples:
		string strWks = "Data1";
		Worksheet wks(strWks);
		OSPCMAINHeader spcMainHeader;
		for(int jj = 0; jj< wks.GetNumCols();jj++)
		{
			if( GetSPCMainHeader(strWks, jj, &spcMainHeader) )
			{
				//read information from the main header
				int iNumOfSubs = spcMainHeader.fnsub;
				OSPCSUBHeader spcSubHeader;
				
				for( int ii=0; ii<iNumOfSubs; ii++ )
				{
					if( GetSPCSubHeader(strWks, jj, &spcSubHeader, ii) )
						//read information from the subheader
				}			
			}	
			
			string strLog;
			strLog = GetSPCLog(strWks, jj);
		}
*/
BOOL	GetSPCSubHeader(LPCSTR lpcszWksName, int nCol, OSPCSUBHeader *SPCSubHeader, int nIndex);

/** >Import Export
		Get SPC log information from worksheet
	Returns:
		The string contents ASCII format log
	Parameters:
		lpcszWksName = worksheet name
		nCol = column index from 0
	Example:
	 	For this example to run, a worksheet named Data1 should exist in the project.

		void run_GetSPCLog()
		{
				string strInfo;
				strInfo = GetSPCLog("Data1", 0);
				out_str(strInfo);
				out_str(strInfo);
		}
*/
string	GetSPCLog(LPCSTR lpcszWksName, int nCol);

#endif //!_WKSHEET_H

