import MySQLdb
import MySQLdb.cursors
from mantis import *




class MantisClient(object) :

    def __init__(self, host, port, login, password, db_name, charset_name) :
        self.sql_cnx = MySQLdb.connect(host = host, port = port, user = login, passwd = password,
                                       db = db_name, cursorclass = MySQLdb.cursors.DictCursor, charset = charset_name)


    def get_project_id_by_name(self, project_name) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT id, name FROM mantis_project_table")
        for row in cursor :
            if (row["name"].encode('utf8') == project_name) :
                return row["id"]

    def get_mantis_users(self) :
        cursor = self.sql_cnx.cursor()
        request = "SELECT username, realname, email FROM mantis_user_table"
        cursor.execute(request)
        result = []
        for row in cursor :
            user = MantisUser(row["username"].replace(" ", "_"))
            user.real_name=row["realname"]
            user.email = row["email"]
            result.append(user)
        return result

    def get_mantis_categories(self, project_id) :
        cursor = self.sql_cnx.cursor()
        project_ids_string = repr(self._calculate_project_ids(project_id)).replace('[','(').replace(']',')')
        request = "SELECT user_id, name FROM mantis_category_table WHERE project_id IN %s" % project_ids_string
        cursor.execute(request)
        result = []
        for row in cursor :
            category = MantisCategory(row["name"].encode('utf8'))
            user_id = row["user_id"]
            if (user_id != 0) :
                category.assignee = self.get_user_name_by_id(user_id)
            result.append(category)
        return result

    def get_mantis_versions(self, project_id) :
        cursor = self.sql_cnx.cursor()
        project_ids_string = repr(self._calculate_project_ids(project_id)).replace('[','(').replace(']',')')
        request = "SELECT version, released, obsolete FROM mantis_project_version_table WHERE project_id IN %s" % project_ids_string
        cursor.execute(request)
        result = []
        for row in cursor :
            version = MantisVersion(row["version"].encode('utf8'))
            version.is_released = (row["released"] > 0)
            version.is_obsolete = (row["obsolete"] > 0)
            result.append(version)
        return result

    def get_mantis_custom_fields(self, project_ids) :
        cursor = self.sql_cnx.cursor()
        ids = set([])
        for project_id in project_ids :
            ids = ids | set(self._calculate_project_ids(project_id))
        project_ids_string = repr(list(ids)).replace('[','(').replace(']',')')
        cf_ids_request = "SELECT DISTINCT field_id FROM mantis_custom_field_project_table WHERE project_id IN " + project_ids_string
        request = "SELECT id, name, type, possible_values, default_value FROM mantis_custom_field_table WHERE id IN (" + cf_ids_request + ")"
        cursor.execute(request)
        result = []
        for row in cursor :
            cf = MantisCustomFieldDef(row["id"])
            cf.type = row["type"]
            cf.name = row["name"].encode('utf8')
            cf.default_value = row["default_value"].encode('utf8')
            if (row["type"] in [3, 6, 7, 9, 5]) :
                # possible values
                values = row["possible_values"].split("|")
                for v in values :
                    v = v.strip()
                    if (v != "") :
                        cf.values.append(v)
            result.append(cf)
        return result

    def get_custom_fields_attached_to_project(self, project_id) :
        cursor = self.sql_cnx.cursor()
        project_ids = (repr(self._calculate_project_ids(project_id)).replace('[','(').replace(']',')'))
        request = "SELECT DISTINCT field_id FROM mantis_custom_field_project_table WHERE project_id IN " + project_ids
        cursor.execute(request)
        result = []
        for row in cursor :
            result.append(row["field_id"])
        return result


    def get_mantis_issues(self, project_id) :
        cursor = self.sql_cnx.cursor()
        project_ids = (repr(self._calculate_project_ids(project_id)).replace('[','(').replace(']',')'))
        issue_string = "SELECT id, project_id, reporter_id, handler_id, resolution, bug_text_id, "
        issue_string = issue_string + "summary, category_id, date_submitted, due_date, last_updated "
        request = issue_string + " FROM mantis_bug_table WHERE project_id IN " + project_ids
        cursor.execute(request)
        result = []
        for row in cursor :
            issue = MantisIssue(row["id"])
            reporter_id = row["reporter_id"]
            if (reporter_id != 0) :
                issue.reporter_name = self.get_user_name_by_id(reporter_id)
            handler_id = row["handler_id"]
            if (handler_id != 0) :
                issue.handler_name = self.get_user_name_by_id(handler_id)
            self._set_int_issue_parameters(issue)

            text_fields = self._get_text_fields(row["bug_text_id"])
            issue.description = text_fields["description"]
            issue.steps_to_reproduce = text_fields["steps_to_reproduce"]
            issue.additional_information = text_fields["additional_information"]

            self._set_os_issue_parameters(issue)

            issue.subproject_name = self._get_project_name_by_id(row["project_id"])
            issue.summary = row["summary"]
            self._set_version_issue_parameters(issue)
            issue.category = self._get_category_by_id(row["category_id"])
            issue.date_submitted = row["date_submitted"]
            issue.due_date = row["due_date"]
            issue.last_updated = row["last_updated"]
            #custom fields

            cf_cursor = self.sql_cnx.cursor()
            cf_cursor.execute("SELECT field_id, value FROM mantis_custom_field_string_table WHERE bug_id=%s",
                              (str(row["id"]),))
            for row in cf_cursor :
                issue_cf = self._get_cf_name_by_id(row["field_id"])
                value = row["value"]
                if (issue_cf["type"] in [3, 6, 7, 9, 5]) :
                    values = value.split("|")
                    issue.cf_values[issue_cf["name"]] = []
                    for v in values :
                        v = v.strip()
                        if (v != "") :
                            issue.cf_values[issue_cf["name"]].append(v)
                else :
                    issue.cf_values[issue_cf["name"]] = value
            issue.tags = self._get_issue_tags_by_id(issue.id)
            issue.comments = self._get_comments_by_id(issue.id)
            result.append(issue)
        return result

    def get_mantis_subprojects(self, project_id) :
        cursor = self.sql_cnx.cursor()
        project_ids = (repr(self._calculate_project_ids(project_id)).replace('[','(').replace(']',')'))
        cursor.execute("SELECT name FROM mantis_project_table WHERE id IN " + project_ids)
        result = []
        for row in cursor :
            result.append(row["name"])
        return result

    def get_issue_links(self) :
        cursor = self.sql_cnx.cursor()
        result = []
        cursor.execute("SELECT * FROM mantis_bug_relationship_table")
        for row in cursor:
            source_bug_id = row["source_bug_id"]
            target_bug_id = row["destination_bug_id"]
            link = MantisIssueLink(source_bug_id, target_bug_id, row["relationship_type"])
            link.source_project_id = self._get_project_id_by_bug_id(source_bug_id)
            link.target_project_id = self._get_project_id_by_bug_id(target_bug_id)
            result.append(link)
        return result

    def get_mantis_attachments(self, project_id):
        cursor = self.sql_cnx.cursor()
        project_ids = (repr(self._calculate_project_ids(project_id)).replace('[','(').replace(']',')'))
        issue_ids_request = "SELECT id FROM mantis_bug_table WHERE project_id IN " + project_ids
        request = "SELECT id FROM mantis_bug_file_table WHERE bug_id IN (" + issue_ids_request + ")"
        cursor.execute(request)
        result = []
        for row in cursor:
            result.append(MantisAttachment(row["id"], self.sql_cnx))
        return result


    def get_project_description(self, project_id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT description FROM mantis_project_table WHERE id=%s LIMIT 1", (str(project_id)))
        description = cursor.fetchone()["description"]
        if (description is None) :
            return "empty description"
        return description.encode('utf8')

    def get_user_name_by_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT username FROM mantis_user_table WHERE id=%s LIMIT 1", (str(id),))
        return cursor.fetchone()["username"].replace(" ", "_")

    def _calculate_project_ids(self, project_id) :
        result = self._get_child_projects_by_project_id(project_id)
        result.append(int(project_id))
        result.append(int(0))
        return result

    def _get_child_projects_by_project_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT child_id FROM mantis_project_hierarchy_table WHERE parent_id = %s", (id,))
        result = []
        for row in cursor :
            result.append(int(row["child_id"]))
            result.extend(self._get_child_projects_by_project_id(row["child_id"]))
        return result


    def _get_text_fields(self, text_id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT description, steps_to_reproduce, additional_information FROM mantis_bug_text_table WHERE id=%s LIMIT 1",
                       (str(text_id),))
        return cursor.fetchone()

    def _get_category_by_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT name FROM mantis_category_table WHERE id=%s LIMIT 1", (str(id)))
        return cursor.fetchone()["name"]

    def _get_comments_by_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT reporter_id, bugnote_text_id, date_submitted FROM mantis_bugnote_table WHERE bug_id=%s",
                       (str(id),))
        result = []
        for row in cursor :
            text_cursor = self.sql_cnx.cursor()
            text_cursor.execute("SELECT note FROM mantis_bugnote_text_table WHERE id=%s LIMIT 1", (str(row["bugnote_text_id"]),))
            comment = MantisComment()
            comment.reporter = self.get_user_name_by_id(row["reporter_id"])
            comment.date_submitted = row["date_submitted"]
            comment.text = text_cursor.fetchone()["note"]
            result.append(comment)
        return result

    def _get_project_id_by_bug_id(self, bug_id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT project_id FROM mantis_bug_table WHERE id=%s LIMIT 1", (bug_id,))
        return cursor.fetchone()["project_id"]


    def _get_cf_name_by_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT name, type  FROM mantis_custom_field_table WHERE id=%s LIMIT 1", (str(id),))
        return cursor.fetchone()

    def _set_int_issue_parameters(self, issue) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT priority, severity, reproducibility, status, resolution FROM mantis_bug_table WHERE id=%s LIMIT 1",
                       (str(issue.id),))
        row = cursor.fetchone()
        issue.priority = row["priority"]
        issue.severity = row["severity"]
        issue.reproducibility = row["reproducibility"]
        issue.status = row["status"]
        issue.resolution = row["resolution"]

    def _set_os_issue_parameters(self, issue) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT os, os_build, platform FROM mantis_bug_table WHERE id=%s LIMIT 1",
                       (str(issue.id),))
        row = cursor.fetchone()
        issue.os = row["os"]
        issue.os_build = row["os_build"]
        issue.platform = row["platform"]

    def _set_version_issue_parameters(self, issue) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT version, fixed_in_version, build, target_version FROM mantis_bug_table WHERE id=%s LIMIT 1",
                       (str(issue.id),))
        row = cursor.fetchone()
        issue.version = row["version"]
        issue.fixed_in_version = row["fixed_in_version"]
        issue.build = row["build"]
        issue.target_version = row["target_version"]

    def _get_project_name_by_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT name FROM mantis_project_table WHERE id=%s LIMIT 1", (str(id),))
        return cursor.fetchone()["name"]

    def _get_issue_tags_by_id(self, id) :
        cursor = self.sql_cnx.cursor()
        cursor.execute("SELECT name FROM mantis_tag_table WHERE id IN (SELECT tag_id FROM mantis_bug_tag_table WHERE bug_id = %s) LIMIT 1",
                       (str(id),))
        result = []
        for row in cursor :
            result.append(row["name"])
        return result