"""
YouTrack 2.0 REST API
"""

from xml.dom import Node
from xml.dom.minidom import Document
from xml.dom import minidom
import urllib
import youtrack.connection

class YouTrackException(Exception):
    def __init__(self, url, response, content):
        self.response = response
        self.content = content
        msg = 'Error for [' + url + "]: " + str(response.status)

        if response.reason is not None:
            msg += ": " + response.reason

        if response.has_key('content-type'):
            ct = response["content-type"]
            if ct is not None and ct.find('text/html') == -1:
                try:
                    xml = minidom.parseString(content)
                    self.error = YouTrackError(xml, self)
                    msg += ": " + self.error.error
                except :
                    self.error = content
                    msg += ": " + self.error

        Exception.__init__(self, msg)

class YouTrackObject(object):

    youtrack = youtrack.connection.Connection

    def __init__(self, xml=None, youtrack=None):
        self.youtrack = youtrack
        self._update(xml)

    def toXml(self):
        raise NotImplementedError

    def _update(self, xml):
        if xml is None:
            return
        if isinstance(xml, Document):
            xml = xml.documentElement

        self._updateFromAttrs(xml)
        self._updateFromChildren(xml)

    def _updateFromAttrs(self, el):
        if el.attributes is not None:
            for i in range(el.attributes.length):
                a = el.attributes.item(i)
                setattr(self, a.name, a.value)

    def _updateFromChildren(self, el):
        children = [e for e in el.childNodes if e.nodeType == Node.ELEMENT_NODE]
        if (children):
            for c in children:
                name = c.getAttribute('name')
                values = c.getElementsByTagName('value')
                if values is not None:
                    if values.length == 1:
                        setattr(self, name, self._text(values.item(0)))
                    elif values.length > 1:
                        setattr(self, name, [self._text(value) for value in values])
                    else:
                        setattr(self, name, None)

    def _text(self, el):
        return "".join([e.data for e in el.childNodes if e.nodeType == Node.TEXT_NODE])

    def __repr__(self):
        return "".join([(k + ' = ' + unicode(self.__dict__[k]) + '\n') for k in self.__dict__.iterkeys() if k != 'youtrack'])

    def __iter__(self):
        for item in self.__dict__:
            attr = self.__dict__[item]
            if isinstance(attr, str) or isinstance(attr, unicode) or isinstance(attr, list) or getattr(attr, '__iter__', False):
                yield item

    def __getitem__(self, key):
        return self.__dict__[key]

    def __setitem__(self, key, value):
        self.__dict__[key] = value

class YouTrackError(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

    def _update(self, xml):
        if xml.documentElement.tagName == 'error':
            self.error = self._text(xml.documentElement)
        else:
            self.error = xml.toxml()

class Issue(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)
        if xml is not None:
            if len(xml.getElementsByTagName('links')) > 0:
                self.links = [Link(e, youtrack) for e in xml.getElementsByTagName('issueLink')]
            else:
                self.links = None

            if len(xml.getElementsByTagName('attachments')) > 0:
                self.attachments = [Attachment(e, youtrack) for e in xml.getElementsByTagName('fileUrl')]
            else:
                self.attachments = None
            for m in ['fixedVersion', 'affectsVersion']: self._normilizeMultiple(m)

    def _normilizeMultiple(self, name):
        if hasattr(self, name):
            if not isinstance(self[name], list):
                if self[name] == '' or self[name] is None:
                    delattr(self, name)
                else:
                    self[name] = str(self[name]).split(',')

    def getReporter(self):
        return self.youtrack.getUser(self.reporterName)

    def hasAssignee(self):
        return getattr(self, 'assigneeName', None) is not None

    def getAssignee(self):
        return self.youtrack.getUser(self.assigneeName)

    def getUpdater(self):
        return self.youtrack.getUser(self.updaterName)

    def getComments(self):
        #TODO: do not make rest request if issue was initied with comments
        if hasattr(self, 'comments'):
            return self.comments
        else:
            return self.youtrack.getComments(self.id)

    def getAttachments(self):
        if getattr(self, 'attachments', None) is None:
            return self.youtrack.getAttachments(self.id)
        else:
            return self.attachments

    def getLinks(self, outwardOnly = False):
        if getattr(self, 'links', None) is None:
            return self.youtrack.getLinks(self.id, outwardOnly)
        else:
            return [l for l in self.links if l.source != self.id or not outwardOnly]

class Comment(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

    def getAuthor(self):
        return self.youtrack.getUser(self.author)

class Link(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class Attachment(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

    def getContent(self):
        return self.youtrack.getAttachmentContent(self.url)

class User(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

    def __hash__(self):
        return hash(self.login)

    def __cmp__(self, other):
        if isinstance(other, User):
            return cmp(self.login, other.login)
        else:
            return cmp(self.login, other)

class Group(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class Role(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class Project(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)
        if not hasattr(self, 'description'):
            self.description = ''

    def getSubsystems(self):
        return self.youtrack.getSubsystems(self.id)

    def createSubsystem(self, name, isDefault, defaultAssigneeLogin):
        return self.youtrack.createSubsystem(self.id, name, isDefault, defaultAssigneeLogin)

class Subsystem(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class Version(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)
        if (not hasattr(self, 'description')):
            self.description = ''    

        if (not hasattr(self, 'releaseDate')):
            self.releaseDate = None    

class Build(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class IssueLinkType(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class CustomField(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

class ProjectCustomField(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

    def _updateFromChildren(self, el):
        self.params = {}
        for c in el.getElementsByTagName('param'):
            name = c.getAttribute('name')
            value = c.getAttribute('value')
            self[name] = value
            self.params[name] = value            

class EnumBundle(YouTrackObject):
    def __init__(self, xml=None, youtrack=None):
        YouTrackObject.__init__(self, xml, youtrack)

    def _update(self, xml):
        if xml is None:
            return
        if isinstance(xml, Document):
            xml = xml.documentElement

        self.name = xml.getAttribute('name')
        values = xml.getElementsByTagName('value')
        if values is not None:
            self.values = [self._text(value) for value in values]
        else:
            self.values = []

    def toXml(self):
        res = '<enumeration name=\"' + self.name + '\">'
        res += ''.join('<value>' + v + '</value>' for v in self.values)
        res += '</enumeration>'

        return res
