// WavingFlag Shader
// 
// Copyright Outerspace Software, all rights reserved

string description = "The effect turns a picture layer into a waving flag.";

float4x4	World;
float4x4	ViewProjection;
float4x4	WorldInverseTranspose;

float4		LightPos;
float4		EyePos;
float4		UVBounds0 = float4(0,1,0,1);
float4		ImageResolution0 = float4(0,0,0,0);

float4		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;
float		Time=0;

float		PI = 3.14159;


float4 Prop0
<
string UIName="Period"; //duration animation in seconds
float UIMin=0;
float UIMax=3;
float3 UIDefault=float3(1,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(1,0,0,0);

float4 Prop1
<
string UIName="Wave Density"; //this determines the number of bumps in horizontal and vertical direction
float UIMin=0;
float UIMax=0.1;
float3 UIDefault=float3(0.03,0.02,0);
int UISliders=2;
float UIScale=100;
bool UIInteger=false;
int UIWidget=0;
> = float4(0.03,0.02,0,0);

float4 Prop2
<
string UIName="Amplitude"; //Use a higher value to simulate a higher wind speed
float UIMin=-100;
float UIMax=100;
float3 UIDefault=float3(40,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(40,0,0,0);

float4 Prop3
<
string UIName="Walk Back";
float UIMin=-1;
float UIMax=1;
float3 UIDefault=float3(0.3,0,0);
int UISliders=1;
float UIScale=100;
bool UIInteger=false;
int UIWidget=0;
> = float4(0.3,0,0,0);

float4 Prop4
<
string UIName="Hanging";
float UIMin=0;
float UIMax=100;
float3 UIDefault=float3(40,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(40,0,0,0);


texture MyTexture0;
sampler MyTextureSampler0 = sampler_state
{
   Texture = <MyTexture0>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:		TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float2 tex:		TEXCOORD0;
	float4 modelpos:	TEXCOORD1;
	float3 normal:		TEXCOORD2;	
};

struct PS_OUTPUT{
    float4 color:	COLOR;
};

float4 GetP(float4 inPos){

	float w=(125 * ImageResolution0.x) / ImageResolution0.y;

	float DistanceToPole = (inPos.x+w)/(2*w);

	float da = (Time*PI*2)/Prop0.x;

	float Alfa = da + inPos.x*Prop1.x + inPos.y*Prop1.y;

	float A=Prop2.x * DistanceToPole;

	inPos.z -= sin(Alfa)*A;

	inPos.x -= inPos.z*Prop3.x;

	inPos.y -= DistanceToPole*DistanceToPole*Prop4.x;	

	return inPos;
}

VS_OUTPUT VS(VS_INPUT input){
	VS_OUTPUT output;

	const float d=10;

	float4 p0=GetP(input.position);
	float4 p1=GetP(input.position + float4(0,d,0,0));
	float4 p2=GetP(input.position + float4(d,0,0,0));

	float3 NormalVector = cross(p1.xyz-p0.xyz,p2.xyz-p0.xyz);

	p0 = mul(p0, World);

	output.modelpos = p0;	

	p0 = mul(p0, ViewProjection);	
	
	output.normal = mul(NormalVector,WorldInverseTranspose);
	output.tex = input.tex;
	output.position = p0;
	return output;
}

PS_OUTPUT PS(VS_OUTPUT input){
	PS_OUTPUT output;

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPos - input.modelpos);
	float3 EyeVector		= normalize(EyePos   - input.modelpos);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);

	//ambient component
	output.color=AmbientCol;

	//diffuse component
	output.color += DiffuseCol * saturate(dot(NormalVector, LightVector));

	//specular component
	output.color += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	output.color.w = DiffuseCol.w;

	if(ImageResolution0.x>0){
		output.color *= tex2D(MyTextureSampler0, input.tex);
	}
	
	return output;
}

technique Shader
{
    pass P0
    {
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS();
    }
}
