// Rippling Reflective Water Phong Shader
// 
// Copyright Outerspace Software, all rights reserved

string description = "A rippling reflective water shader featuring ambient, diffuse and specular lighting.";

float4x4	WorldViewProjection;
float4x4	World;
float4x4	ViewProjection;
float4		LightPosModelSpace;
float4		EyePosModelSpace;

float4		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;

float		Time;

float		PI = 3.14159;

float4		MyTextureResolution0;

float4 Prop0
<
string UIName="Water Colour";
float UIMin=0;
float UIMax=1;
float3 UIDefault=float3(1,1,1);
int UISliders=3;
float UIScale=255;
bool UIInteger=true;
int UIWidget=2;
> = float4(1,1,1,1);

float4 Prop1
<
string UIName="Rippling Amplitude";
float UIMin=0;
float UIMax=20;
float3 UIDefault=float3(0,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(0,0,0,0);

float4 Prop2
<
string UIName="Rippling Wavelength";
float UIMin=0;
float UIMax=20;
float3 UIDefault=float3(10,10,0);
int UISliders=2;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(10,10,0,0);

float4 Prop3
<
string UIName="Rippling Period";
float UIMin=0;
float UIMax=10;
float3 UIDefault=float3(6,10,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(6,0,0,0);

float4 Prop4
<
string UIName="Transparency Reflection";
float UIMin=0;
float UIMax=1;
float3 UIDefault=float3(0.5,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(0.5,0,0,0);

float4 Prop5
<
string UIName="Transparency Gradient";
float UIMin=0;
float UIMax=0.1;
float3 UIDefault=float3(0.01,0,0);
int UISliders=1;
float UIScale=100;
bool UIInteger=false;
int UIWidget=0;
> = float4(0.01,0,0,0);

float4 Prop6
<
string UIName="Water Level";
float UIMin=-100;
float UIMax=100;
float3 UIDefault=float3(0,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(0,0,0,0);

texture MyTexture0;
sampler MyTextureSampler0 = sampler_state
{
   Texture = <MyTexture0>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Wrap;
   AddressV  = Wrap;
};

struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:			TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float2 tex:		TEXCOORD0;
	float4 worldpos:	TEXCOORD1;
	float4 modelpos:	TEXCOORD2;
	float3 normal:		TEXCOORD3;	
};

struct PS_OUTPUT{
    float4 color:		COLOR;
};

//the mirrored image
VS_OUTPUT VS1(VS_INPUT input){
	VS_OUTPUT output;      
	
	float4 pos = mul(input.position, World);
	pos.y = 2*Prop6.x - pos.y;	
	
	pos.x += Prop1.x * sin((Time*2*PI)/Prop3.x) * sin((pos.y-Prop6.x)/Prop2.y) * sin(((Time*2*PI)/Prop3.x)+pos.x/Prop2.x);
	
	output.worldpos = pos;
	output.position = mul(pos, ViewProjection);

	output.tex = input.tex;
	output.modelpos = input.position;
	output.normal = input.normal;
	return output;
}

VS_OUTPUT VS2(VS_INPUT input){
	VS_OUTPUT output;      

	float4 pos = mul(input.position, World);
	output.worldpos = pos;
	output.position = mul(pos, ViewProjection);

	output.tex = input.tex;
	output.modelpos = input.position;
	output.normal = input.normal;
	return output;
}

PS_OUTPUT PS1(VS_OUTPUT input){
	PS_OUTPUT output;

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPosModelSpace - input.modelpos);
	float3 EyeVector		= normalize(EyePosModelSpace   - input.modelpos);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);

	//ambient component
	float4 c=AmbientCol;

	//diffuse component
	c += DiffuseCol * saturate(dot(NormalVector, LightVector));

	//specular component
	c += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	c.w = DiffuseCol.w;

	c.w=1;
	if(input.worldpos.y > Prop6.x) c.w=0;
	if(MyTextureResolution0.x>0) c *= tex2D(MyTextureSampler0, input.tex);

	c.w*=(1-Prop4.x);

	float f=1-( -(input.worldpos.y-Prop6.x) * Prop5.x);
	if(f<0) f=0;
	if(f>1) f=1;
	c.w*=f;
	c.w*=DiffuseCol.w;

	c.xyz *= Prop0.xyz;

	output.color=c;
	return output;
}

PS_OUTPUT PS2(VS_OUTPUT input){
	PS_OUTPUT output;

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPosModelSpace - input.modelpos);
	float3 EyeVector		= normalize(EyePosModelSpace   - input.modelpos);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);

	//ambient component
	float4 c=AmbientCol;

	//diffuse component
	c += DiffuseCol * saturate(dot(NormalVector, LightVector));

	//specular component
	c += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	c.w = DiffuseCol.w;

	c.w=1;
	if(input.worldpos.y < Prop6.x) c.w=0;	
	if(MyTextureResolution0.x>0) c *= tex2D(MyTextureSampler0, input.tex);

	c.w*=DiffuseCol.w;
	
	output.color = c;
	return output;
}


technique Shader
{
    //mirror: alleen z-buffer schrijven
    pass P1
    {
	Alphablendenable= TRUE;
	SRCBLEND	= SRCALPHA;
	DESTBLEND	= INVSRCALPHA;
	ALPHAARG1[0]	= DIFFUSE;
	ALPHAOP[0]	= SELECTARG1;
	ALPHATESTENABLE	= TRUE;
	ALPHAREF	= 1;
	ALPHAFUNC	= GREATEREQUAL;
	CullMode	= CW;
	ZFUNC		= LESSEQUAL;

	ColorWriteEnable=0;

        VertexShader	= compile vs_1_1 VS1();
        PixelShader	= compile ps_2_0 PS1();
    }

    //mirror: RGB
    pass P0
    {
	ColorWriteEnable=15;
		
        VertexShader	= compile vs_1_1 VS1();
        PixelShader	= compile ps_2_0 PS1();
    }

    //normaal    
    pass P1
    {
	CullMode	= CCW;
        VertexShader	= compile vs_1_1 VS2();
        PixelShader	= compile ps_2_0 PS2();
    }
}



technique AlphaShader
{
    //mirror: alleen z-buffer schrijven
    pass P1
    {
	Alphablendenable= TRUE;
	SRCBLEND	= SRCALPHA;
	DESTBLEND	= INVSRCALPHA;
	ALPHAARG1[0]	= DIFFUSE;
	ALPHAOP[0]	= SELECTARG1;
	ALPHATESTENABLE	= TRUE;
	ALPHAREF	= 1;
	ALPHAFUNC	= GREATEREQUAL;
	CullMode	= CW;
	ZFUNC		= LESSEQUAL;

	ColorWriteEnable=0;

        VertexShader	= compile vs_1_1 VS1();
        PixelShader	= compile ps_2_0 PS1();
    }

    //mirror: RGB
    pass P0
    {
	ColorWriteEnable=8;
		
        VertexShader	= compile vs_1_1 VS1();
        PixelShader	= compile ps_2_0 PS1();
    }

    //normaal    
    pass P1
    {
	CullMode	= CCW;
        VertexShader	= compile vs_1_1 VS2();
        PixelShader	= compile ps_2_0 PS2();
    }
}

