// Plane2Roll Shader
//
// Copyright Outerspace Software, all rights reserved

string description = "This effect file rolls a picture up. Use the FX Morphing Stage property to animate the effect."; 

float4		ImageResolution0 	= float4(256,256,1,0);
float4		TextureResolution0 	= float4(256,256,1,0);
float4 		UVBounds0 		= float4(0,1,0,1);

float4x4	World;
float4x4	ViewProjection;
float4x4	WorldInverseTranspose;

float4		LightPos;
float4		EyePos;

float4		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;
float		Time=0;

float		PI = 3.14159;

float4 Prop0
<
string UIName="Morphing Stage";
float UIMin=0;
float UIMax=1;
float3 UIDefault=float3(0,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(0,0,0,0);

float4 Prop1
<
string UIName="Radius";
float UIMin=0;
float UIMax=40;
float3 UIDefault=float3(20,20,0);
int UISliders=2;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(20,20,0,0);

float4 Prop2
<
string UIName="Rotation";
float UIMin=-6.283;
float UIMax=6.283;
float3 UIDefault=float3(0,0,0);
int UISliders=1;
float UIScale=57.3;
bool UIInteger=false;
int UIWidget=0;
> = float4(0,0,0,0);

texture MyTexture0;
sampler MyTextureSampler0 = sampler_state
{
   Texture = <MyTexture0>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:		TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float2 tex:		TEXCOORD0;
	float4 modelpos:	TEXCOORD1;
	float3 normal:		TEXCOORD2;	
};

struct PS_OUTPUT{
	float4 color:		COLOR;
};

float4 GetPos(float2 inUV){

	float height = 126.25;
	float width  = (ImageResolution0.x * height)/ImageResolution0.y;
	
	inUV.x=(inUV.x-UVBounds0.x) / (UVBounds0.y-UVBounds0.x);
	inUV.y=(inUV.y-UVBounds0.z) / (UVBounds0.w-UVBounds0.z);

	float4 pos2;
	pos2.x = -width + inUV.x * width * 2;
	pos2.y = height - inUV.y * height * 2;
	pos2.z = 0;
	pos2.w = 1;

	float4 pos;
	pos.x = pos2.x*cos(Prop2.x) - pos2.y*sin(Prop2.x);
	pos.y = pos2.x*sin(Prop2.x) + pos2.y*cos(Prop2.x);
	pos.z = pos2.z;

	float g = -width + Prop0.x * width * 2;

	float uvy = 0.5 + (inUV.x-0.5)*sin(-Prop2.x) + (inUV.y-0.5)*cos(-Prop2.x);
	float r= Prop1.x + uvy*(Prop1.y-Prop1.x);

	if(pos.x<g){
		float a=(pos.x-g)/r;

		float s=1+(0.2*a)/(2*PI);
		if(s<0) s=0;
		
		pos.x = g + r*sin(a)*s;	
		pos.z = -r + r*cos(a)*s;
	}

	pos2.x = pos.x*cos(-Prop2.x) - pos.y*sin(-Prop2.x);
	pos2.y = pos.x*sin(-Prop2.x) + pos.y*cos(-Prop2.x);
	pos2.z = pos.z;

	return pos2;
}

VS_OUTPUT VS(VS_INPUT input){
	VS_OUTPUT output;

	const float du=0.01*(UVBounds0.y-UVBounds0.x);
	const float dv=0.01*(UVBounds0.w-UVBounds0.z);

	float4 p0=GetPos(input.tex);
	float4 p1=GetPos(input.tex + float4(0,dv,0,0));
	float4 p2=GetPos(input.tex + float4(du,0,0,0));

	float3 NormalVector = cross(p1.xyz-p0.xyz,p2.xyz-p0.xyz);

	if(input.normal.z<1) NormalVector=-NormalVector;

	p0 = mul(p0, World);

	output.modelpos = p0;

	p0 = mul(p0, ViewProjection);

	output.normal = mul(NormalVector,WorldInverseTranspose);
	output.tex = input.tex;
	output.position = p0;
	return output;
}

PS_OUTPUT PS(VS_OUTPUT input){
	PS_OUTPUT output;

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPos - input.modelpos);
	float3 EyeVector		= normalize(EyePos   - input.modelpos);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);

	//ambient component
	output.color=AmbientCol;

	//diffuse component
	output.color += DiffuseCol * saturate(dot(NormalVector, LightVector));

	//specular component
	output.color += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	output.color.w = DiffuseCol.w;

	if(ImageResolution0.x>0){
		output.color *= tex2D(MyTextureSampler0, input.tex);
	}
	return output;
}

technique Shader
{
    pass P0
    {
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS();
    }
}

