// ColourMap + ReflectionMap Shader
//
// Copyright Outerspace Software, all rights reserved

string description = "This effect combines colourmapping with reflectionmapping. It uses the first texture as the colourmap and the second texture as the reflectionmap (spheremap).";

bool optimization=false;

float4x4	WorldViewProjection;
float4		EyePosModelSpace;
float4		LightPosModelSpace;

float		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;
float4 		UVBounds1;

float4 Prop0
<
string UIName="Lighting Factor";
float UIMin=0;
float UIMax=1;
float3 UIDefault=float3(0.5,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(0.3,0,0,0);

texture MyTexture0;
sampler MyTextureSampler0 = sampler_state
{
   Texture = <MyTexture0>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Wrap;
   AddressV  = Wrap;
};

texture MyTexture1;
sampler MyTextureSampler1 = sampler_state
{
   Texture = <MyTexture1>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;   
   AddressU  = Wrap;
   AddressV  = Wrap;
};

struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:		TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float4 diffuse:		COLOR;
	float2 tex:		TEXCOORD0;
};

float2 Normal2UVBounds(float2 inUV){
	inUV.x = UVBounds1.x + inUV.x*(UVBounds1.y-UVBounds1.x);
	inUV.y = UVBounds1.z + inUV.y*(UVBounds1.w-UVBounds1.z);
	return inUV;
}

VS_OUTPUT VS1(VS_INPUT input){
	VS_OUTPUT output;      
	output.position = mul(input.position, WorldViewProjection);

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPosModelSpace - input.position);
	float3 EyeVector		= normalize(EyePosModelSpace   - input.position);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);
	
	//ambient component
	output.diffuse = AmbientCol;

	//diffuse component
	output.diffuse += DiffuseCol * saturate(dot(NormalVector, LightVector));
	
	//specular component	
	output.diffuse += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	output.diffuse.xyz *= Prop0.x;
	output.diffuse.w = DiffuseCol.w;

	output.tex = input.tex;
	return output;
}

VS_OUTPUT VS2(VS_INPUT input){
	VS_OUTPUT output;
	output.position = mul(input.position, WorldViewProjection);

	float3 NormalVector		= normalize(input.normal);
	float3 EyeVector		= normalize(EyePosModelSpace   - input.position);

	float3 ReflectionVector = reflect(EyeVector,NormalVector);
	ReflectionVector.z=abs(ReflectionVector.z);

	ReflectionVector.z+=1;
	float f=2*length(ReflectionVector);
	output.tex = float2(0.5,0.5) + ReflectionVector.xy/f;

	output.tex=Normal2UVBounds(output.tex);

	output.diffuse = float4(1-Prop0.x,1-Prop0.x,1-Prop0.x,1);
	output.diffuse*= DiffuseCol;
	
	//transparency with additive blending?
	output.diffuse.xyz *= DiffuseCol.w;
	output.diffuse.w=1;
	
	return output;
}

technique Shader
{
    //colourmap
    pass P0
    {
	ZFunc			=LESS;
	AlphaBlendEnable	=FALSE;
        Texture[0]		=(MyTexture0);

        VertexShader		=compile vs_1_1 VS1();
        PixelShader		=NULL;
    }
    
    //reflection
    pass P1
    {
	ZFunc			=LESSEQUAL;
	Alphablendenable	=TRUE;
	SrcBlend		=ONE;
	DestBlend		=ONE;
        Texture[0]		=(MyTexture1);    

        VertexShader		=compile vs_1_1 VS2();
        PixelShader		=NULL;
    }
    
}

