// ReflectionMappedDecorations
//
// Copyright Outerspace Software, all rights reserved

string description = "This effect creates shiny decorations. The first texture must be a reflection map, the second a transparent colour map.";

float4x4	WorldViewProjection;
float4		LightPosModelSpace;
float4		EyePosModelSpace;

float4		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;
float4		UVBounds0;
float4		UVBounds1;

texture MyTexture0;
sampler Samp0 = sampler_state{
    Texture   = (MyTexture0);
    AddressU  = WRAP;
    AddressV  = WRAP;
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

texture MyTexture1;
sampler Samp1 = sampler_state{
    Texture   = (MyTexture1);
    AddressU  = WRAP;
    AddressV  = WRAP;
    MipFilter = LINEAR;
    MinFilter = LINEAR;
    MagFilter = LINEAR;
};

struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:			TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float4 diffuse:		COLOR;
	float2 tex:			TEXCOORD0;
};

struct PS_OUTPUT{
    float4 color:		COLOR;
};

float2 Normal2UVBounds(float2 inUV, float4 inUVBounds){
	inUV.x = inUVBounds.x + inUV.x*(inUVBounds.y-inUVBounds.x);
	inUV.y = inUVBounds.z + inUV.y*(inUVBounds.w-inUVBounds.z);
	return inUV;
}

float2 UVBounds2Normal(float2 inUV, float4 inUVBounds){
	inUV.x = (inUV.x-inUVBounds.x)/(inUVBounds.y-inUVBounds.x);
	inUV.y = (inUV.y-inUVBounds.z)/(inUVBounds.w-inUVBounds.z);
	return inUV;
}

VS_OUTPUT VS1(VS_INPUT input){
	VS_OUTPUT output;
	output.position = mul(input.position, WorldViewProjection);

	float3 NormalVector		= normalize(input.normal);
	float3 EyeVector		= normalize(EyePosModelSpace   - input.position);

	float3 ReflectionVector = reflect(EyeVector,NormalVector);
	ReflectionVector.z=abs(ReflectionVector.z);

	ReflectionVector.z+=1;
	float f=2*length(ReflectionVector);
	output.tex = float2(0.5,0.5) + ReflectionVector.xy/f;

	output.tex=Normal2UVBounds(output.tex,UVBounds0);

	output.diffuse = float4(1,1,1,1);
	return output;
}

VS_OUTPUT VS2(VS_INPUT input){
	VS_OUTPUT output;      
	output.position = mul(input.position, WorldViewProjection);

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPosModelSpace - input.position);
	float3 EyeVector		= normalize(EyePosModelSpace   - input.position);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);
	
	//ambient component
	output.diffuse = AmbientCol;

	//diffuse component
	output.diffuse += DiffuseCol * saturate(dot(NormalVector, LightVector));
	
	//specular component	
	output.diffuse += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	output.diffuse.w = DiffuseCol.w;

	output.tex = UVBounds2Normal(input.tex, UVBounds0);
	output.tex = Normal2UVBounds(output.tex, UVBounds1);

	return output;
}

PS_OUTPUT PS2(VS_OUTPUT input){
	PS_OUTPUT output;	
	float2 t = input.tex;
	
	float4 c = input.diffuse;
	c.w = 1-tex2D(Samp1,t).w;
	
	output.color = c;	   
	return output;
}

technique Shader
{
    pass P0{
		VertexShader = compile vs_2_0 VS1();
		PixelShader  = NULL;
    }      

    pass P1{
		ZFunc			= LESSEQUAL;
		Alphablendenable	= TRUE;
		SRCBLEND		= SRCALPHA;
		DESTBLEND		= INVSRCALPHA;

		VertexShader = compile vs_2_0 VS2();
		PixelShader  = compile ps_2_0 PS2();
    }
}

