// Phong Shader
// 
// Copyright Outerspace Software, all rights reserved

string description = "A phong shader featuring ambient, diffuse and specular lighting.";

float4x4	World;
float4x4	ViewProjection;
float4x4	WorldViewProjection;
float4x4	WorldInverseTranspose;
float4		LightPos;
float4		EyePos;

float4		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;

float4		TextureResolution0;

texture MyTexture0;
sampler MyTextureSampler0 = sampler_state
{
   Texture = <MyTexture0>;
   MinFilter = Linear;
   MagFilter = Linear;
   MipFilter = Linear;
};

struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:		TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float2 tex:		TEXCOORD0;
	float4 modelpos:	TEXCOORD1;
	float3 normal:		TEXCOORD2;	
};

struct PS_OUTPUT{
    float4 color:		COLOR;
};

VS_OUTPUT VS(VS_INPUT input){
	VS_OUTPUT output;      
	
	float4 Pos=mul(input.position, World);
	output.position=mul(Pos, ViewProjection);
	
	output.tex = input.tex;
	output.modelpos = Pos;
	output.normal = mul(input.normal,WorldInverseTranspose);
	return output;
}

PS_OUTPUT PS(VS_OUTPUT input){
	PS_OUTPUT output;

	float3 NormalVector		= normalize(input.normal);
	float3 LightVector		= normalize(LightPos - input.modelpos);
	float3 EyeVector		= normalize(EyePos   - input.modelpos);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);

	//ambient component
	output.color=AmbientCol;

	//diffuse component
	output.color += DiffuseCol * saturate(dot(NormalVector, LightVector));

	//specular component
	output.color += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	output.color.w = DiffuseCol.w;

	if(TextureResolution0.x>0){
		output.color *= tex2D(MyTextureSampler0, input.tex);
	}
	
	return output;
}

technique Shader
{
    pass P0
    {
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS();
    }
}

