// ReflectionMap Lightened / SphereMap Shader
//
// Copyright Outerspace Software, all rights reserved

string description = "A reflectionmapping (spheremap) effect with lighting. It uses the first texture as a reflectionmap (spheremap). The second texture is not used.";

// a shader combining reflection mapping with lighting
// the first pass renders the object not textured with lighting
// the second pass renders the object additive with reflection mapping
// texture0: a spheremap
// texture1: not used

float4x4	World;
float4x4	ViewProjection;
float4x4	WorldInverseTranspose;
float4		LightPos;
float4		EyePos;
float4		AmbientCol;
float4		DiffuseCol;
float4		SpecularCol;
float		Power = 100;
float4		UVBounds0;

float4 Prop0
<
string UIName="Lighting Factor";
float UIMin=0;
float UIMax=1;
float3 UIDefault=float3(0.3,0,0);
int UISliders=1;
float UIScale=1;
bool UIInteger=false;
int UIWidget=0;
> = float4(0.3,0,0,0);


struct VS_INPUT{
	float4 position:	POSITION;
	float3 normal:		NORMAL;
	float3 diffuse:		COLOR;
	float2 tex:			TEXCOORD0;
};

struct VS_OUTPUT{
	float4 position:	POSITION;
	float4 diffuse:		COLOR;
	float2 tex:			TEXCOORD0;
};

float2 Normal2UVBounds(float2 inUV){
	inUV.x = UVBounds0.x + inUV.x*(UVBounds0.y-UVBounds0.x);
	inUV.y = UVBounds0.z + inUV.y*(UVBounds0.w-UVBounds0.z);
	return inUV;
}

VS_OUTPUT VS1(VS_INPUT input){
	VS_OUTPUT output;      
	
	float4 Pos=mul(input.position, World);
	output.position = mul(Pos, ViewProjection);

	float3 NormalVector		= normalize(mul(input.normal,WorldInverseTranspose));
	float3 LightVector		= normalize(LightPos - Pos);
	float3 EyeVector		= normalize(EyePos	 - Pos);
	float3 LightReflectionVector	= -reflect(LightVector,NormalVector);
	
	//ambient component
	output.diffuse = AmbientCol;

	//diffuse component
	output.diffuse += DiffuseCol * saturate(dot(NormalVector, LightVector));
	
	//specular component	
	output.diffuse += SpecularCol * pow(saturate(dot(EyeVector, LightReflectionVector)),Power);
	
	output.diffuse.xyz *= Prop0.x;
	output.diffuse.w = DiffuseCol.w;

	output.tex = input.tex;
	return output;
}

VS_OUTPUT VS2(VS_INPUT input){
	VS_OUTPUT output;
	
	float4 Pos=mul(input.position, World);
	output.position = mul(Pos, ViewProjection);

	float3 NormalVector	= normalize(mul(input.normal,WorldInverseTranspose));
	float3 EyeVector	= normalize(EyePos - Pos);

	float3 ReflectionVector = reflect(EyeVector,NormalVector);
	ReflectionVector.z=abs(ReflectionVector.z);

	ReflectionVector.z+=1;
	float f=2*length(ReflectionVector);
	output.tex = float2(0.5,0.5) + ReflectionVector.xy/f;

	output.tex = Normal2UVBounds(output.tex);

	float T=(1-Prop0.x)*DiffuseCol.w;
	output.diffuse = float4(T,T,T,1) * DiffuseCol;

	return output;
}

technique Shader
{
    pass P1
    {
	//not textured
	ColorArg1[0]		=DIFFUSE;
	ColorOp[0]		=SELECTARG1;
    
	ZFunc			=LESS;
	AlphaBlendEnable	=TRUE;
	SRCBLEND		=SRCALPHA;
	DESTBLEND		=INVSRCALPHA;

        VertexShader		=compile vs_1_1 VS1();
        PixelShader		=NULL;
    }

    pass P2
    {
	//textured
	ColorArg1[0]		=TEXTURE;
	ColorArg2[0]		=DIFFUSE;
	ColorOp[0]		=MODULATE;
    
	ZFunc			=LESSEQUAL;
	Alphablendenable	=TRUE;
	SrcBlend		=ONE;
	DestBlend		=ONE;
    
        VertexShader		=compile vs_1_1 VS2();
        PixelShader		=NULL;
    }
}