# Copyright 2004, @Last Software, Inc., modified by Didier Bur

# This software is provided as an example of using the Ruby interface
# to SketchUp.

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name        :   Rotated Rectangle Hole Tool
# Description :   A tool to create hole rectangles with the mouse.
# Menu Item   :   Plugins-> Rectangular Hole
# Context Menu:   NONE
# Usage       :   This rectangle hole tool takes three points.
#             :   The first point is the first corner of the rectangle
#             :   The second point defines the direction of one of the sides
#             :   The third point defines the opposite side.
# Date        :   11/09/2004
# Type        :   Tool
#-----------------------------------------------------------------------------

require 'sketchup.rb'

# --------------------------------------------------------------------- Opening the hole
def hole( f )
model = Sketchup.active_model
# Select all faces in the model that have the same normal but reversed
entities = model.active_entities
all_faces = []
all_norms_reverse = []
all_plans = []
j = 0
entities.each { |current_ent|
  if ( current_ent.typename == "Face" and (current_ent.normal == f.normal.reverse))
    all_faces[j] = current_ent
    all_norms_reverse[j] = current_ent.normal
    all_plans[j] = current_ent.plane
    j = j + 1
  end
}

i = 0
m = 0
faces_and_heights = []

# Face's plane equation, normal and reverse
plan = f.plane
current_norm = f.normal
current_norm_reverse = f.normal.reverse
# First vertex of face
first_point = f.vertices[0].position
#Virtual line for intersection to nearest face
current_norm_reverse.length = 1000
second_point = Geom::Point3d.new( (first_point[0] + current_norm_reverse[0]), 
                                    (first_point[1] + current_norm_reverse[1]),
                                    (first_point[2] + current_norm_reverse[2]))
v_line = [Geom::Point3d.new(first_point), second_point] 


k = 0
all_dists = []
mini = 10000
ind_face = 0
all_norms_reverse.each{ |norm|
# Calculate the distance between current_face and parallel faces
paral_face = all_faces[k]
paral_plan = paral_face.plane
inters = Geom.intersect_line_plane(v_line, paral_plan)
t = in_face( inters, paral_face )
#dist = first_point.distance_to_plane(paral_face.plane)
$dist = (plan[3].abs - all_plans[k][3].abs).abs
        if ( $dist < mini ) and ( $dist != 0 ) #and t
          mini = $dist
          ind_face = k
        end
  k = k + 1
}
i = i + 1
m = m + 2
# Pushpull
i = 0
f.pushpull -mini
i = i + 2
end # of def


def in_face( p, f )
v = f.outer_loop.vertices
i = 0
sum_angles = 0
two_pi = Math::PI * 2 #6.28318530717959
0.upto (v.length - 1) do |a|
  if(i != (v.length - 1))
    vec1 = p.vector_to v[i] 
    vec2 = p.vector_to v[i + 1] 
    else
    vec1 = p.vector_to v[i] 
    vec2 = p.vector_to v[0]
  end
  sum_angles = sum_angles + vec1.angle_between(vec2)
  i = i + 1
end

  if( (two_pi - 0.1) < sum_angles) and ( sum_angles < (two_pi + 0.1))
    return true
    else
    return false
  end
end

#=============================================================================

class RectangleHoleTool

def initialize
    @ip = Sketchup::InputPoint.new
    @ip1 = Sketchup::InputPoint.new
    reset
end

def reset
    @pts = []
    @state = 0
    @ip1.clear
    @drawn = false
    Sketchup::set_status_text "", SB_VCB_LABEL
    Sketchup::set_status_text "", SB_VCB_VALUE
    Sketchup::set_status_text "Click for start point"
    @shift_down_time = Time.now
end

def activate
    self.reset
end

def deactivate(view)
    view.invalidate if @drawn
end

def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        @pts[4] = @pts[0]
        need_draw = @ip.display? || @drawn
    when 1
        @pts[1] = @ip.position
        @width = @pts[0].distance @pts[1]
        Sketchup::set_status_text @width.to_s, SB_VCB_VALUE
    when 2
        pt1 = @ip.position
        pt2 = pt1.project_to_line @pts
        vec = pt1 - pt2
        @height = vec.length
        if( @height > 0 )
            # test for a square
            square_point = pt2.offset(vec, @width)
            if( view.pick_helper.test_point(square_point, x, y) )
                @height = @width
                @pts[2] = @pts[1].offset(vec, @height)
                @pts[3] = @pts[0].offset(vec, @height)
                view.tooltip = "Square"
            else
                @pts[2] = @pts[1].offset(vec)
                @pts[3] = @pts[0].offset(vec)
            end
        else
            @pts[2] = @pts[1]
            @pts[3] = @pts[0]
        end
        Sketchup::set_status_text @height.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end

def onMouseMove(flags, x, y, view)
    self.set_current_point(x, y, view)
end

def create_rectangle
    # check for zero height
    if( @pts[0] != @pts[3] )
        h = Sketchup.active_model.active_entities.add_face @pts
        hole(h)
    end
    self.reset
end

def increment_state
    @state += 1
    case @state
    when 1
        @ip1.copy! @ip
        Sketchup::set_status_text "Click for second point"
        Sketchup::set_status_text "Width", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 2
        @ip1.clear
        Sketchup::set_status_text "Click for third point"
        Sketchup::set_status_text "Height", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 3
        self.create_rectangle
    end
end

def onLButtonDown(flags, x, y, view)
    self.set_current_point(x, y, view)
    self.increment_state
    view.lock_inference
end

def onCancel(flag, view)
    view.invalidate if @drawn
    self.reset
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
    
    case @state
    when 1
        # update the width
        vec = @pts[1] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[1] = @pts[0].offset(vec)
            view.invalidate
            self.increment_state
        end
    when 2
        # update the height
        vec = @pts[3] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[2] = @pts[1].offset(vec)
            @pts[3] = @pts[0].offset(vec)
            self.increment_state
        end
    end
end

def getExtents
    bb = Geom::BoundingBox.new
    case @state
    when 0
        # We are getting the first point
        if( @ip.valid? && @ip.display? )
            bb.add @ip.position
        end
    when 1
        bb.add @pts[0]
        bb.add @pts[1]
    when 2
        bb.add @pts
    end
    bb
end

def draw(view)
    @drawn = false
    
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end

    # show the rectangle
    if( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
        inference_locked = view.inference_locked?
        view.line_width = 3 if inference_locked
        view.draw(GL_LINE_STRIP, @pts[0], @pts[1])
        view.line_width = 1 if inference_locked
        @drawn = true
    elsif( @state > 1 )
        # draw the curve
        view.drawing_color = "black"
        view.draw(GL_LINE_STRIP, @pts)
        @drawn = true
    end
end

def onKeyDown(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY && rpt == 1 )
        @shift_down_time = Time.now
        
        # if we already have an inference lock, then unlock it
        if( view.inference_locked? )
            view.lock_inference
        elsif( @state == 0 )
            view.lock_inference @ip
        elsif( @state == 1 )
            view.lock_inference @ip, @ip1
        end
    end
end

def onKeyUp(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY &&
        view.inference_locked? &&
        (Time.now - @shift_down_time) > 0.5 )
        view.lock_inference
    end
end

end # of class RectangleTool

#=============================================================================
#if( not file_loaded?("rectangle_hole.rb") )
#    UI.menu("Plugins").add_item("Rectangular hole") {
#        Sketchup.active_model.select_tool RectangleHoleTool.new }
#end
#---------------------------------------------------------------------------
file_loaded("rectangle_hole.rb")

